#!/usr/bin/env python3
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements sFlow configuration.  In particular, it provides:
# -  the "[no|default] sflow enable" command
# -  the "[no|default] sflow egress enable" command
# -  the "[no|default] sflow egress unmodified enable" command
# -  the "[no|default] sflow polling-interval <secs>" command
# -  the "[no|default] sflow datagram size maximum <bytes>" command
# -  the "no sflow sample" command
# -  the "sflow sample [dangerous] <rate>" command
# -  the "[no|default] sflow destination <ip-addr> [<dest-udp-port>]" command
# -  the "[no|default] sflow source <ip-addr>" command
# -  the "[no|default] sflow source address <ip-addr>" command
# -  the "[no|default] sflow agent address <ip-addr>" command
# -  the "[no|default] sflow sample rewrite dscp" command
# -  the "[no|default] sflow sample output interface" command
# -  the "[no|default] sflow sample output subinterface" command
# -  the "[no|default] sflow extension bgp" command
# -  the "[no|default] sflow extension mpls" command
# -  the "[no|default] sflow extension router" command
# -  the "[no|default] sflow extension switch" command
# -  the "[no|default] sflow extension tunnel ipv4 egress" command
# -  the "[no|default] sflow sample output portchannel ifindex" command
# -  the "[no|default] sflow sample output multicast interface" command
# -  the "[no|default] sflow run" command
# -  the "[no|default] sflow qos dscp <0-63>" command
# -  the "[no|default] sflow interface disable default" command
# -  the "[no|default] sflow interface egress enable default" command
# -  the "[no|default] sflow interface egress unmodified enable default" command
# -  the "[no|default] sflow sample include drop reason acl" command
# -  the "[no|default] sflow sample vxlan header strip" command
# -  the "[no|default] sflow sample output svi ifindex svi" command
# -  the "[no|default] sflow sample input svi ifindex svi" command
# -  the "clear sflow counters" command
# -  the "show sflow" command
# -  the "show sflow detail" command
# -  the "show sflow interfaces" command
# -  the "show sflow interfaces detail" command
#-------------------------------------------------------------------------------

import os
from collections import namedtuple

import AgentDirectory
import Tac
import CliParser
import LazyMount
import ConfigMount
import SflowConst
import CliPlugin.TechSupportCli
from CliPlugin.SflowModel import SflowStatus, SflowInterfaces, SflowTypeStatus, \
   SflowTypeStatusMap, LagMemberSflowTypeStatus, SflowInterfaceCounters, \
   IntfSampleCounterMap
from IpLibConsts import DEFAULT_VRF
import SflowUtil
import DscpCliLib
import SharedMem
import Smash
import CliExtensions
import Arnet
import Toggles.SflowLibToggleLib as ToggleSflow

dscpConfig = None
sflowHwStatusDir = None
veosStatus = None
sflowAccelCapabilities = None
sflowAccelRunnability = None
sflowAccelConfig = None
sflowAccelHwConfig = None
sflowAccelStatusDir = None
smashEntityManager = None
sflowAccelCountersTable = None
sflowConfig = None
sflowCounterConfig = None
sflowStatus = None
sflowHwConfig = None
dropExportSflowConfig = None
dropExportSflowStatus = None
em = None

AccelId = Tac.Type( "Hardware::SflowAccel::AccelId" )
MessageHelper = Tac.Type( "Hardware::SflowAccel::FeatureMessageHelper" )
FpgaCounterKey = Tac.Type( 'Hardware::SflowAccel::FpgaCounterKey' )
SflowAccelPPCountersTableType = \
      Tac.Type( "Hardware::SflowAccel::SflowAccelCounterTable" )
sampleAll = Tac.Type( 'Sflow::TrafficSampled' ).sampleAll
IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )

def sflowHwStatus():
   return SflowUtil.sflowHwStatus( sflowHwStatusDir )

def sflowHwSampleRate():
   return SflowUtil.sflowHwSampleRate( sflowHwStatusDir )

def sflowHwSamples():
   totalHwSamples = max( [ s.hwSamples for s in sflowHwStatusDir.values() ],
         default=0 )
   return max( veosStatus.hwSamples, totalHwSamples )

def sflowAccelSupported():
   return sflowAccelCapabilities.sflowAccelSupported or \
      sflowAccelCapabilities.sflowAccelPPSupported

def sflowSampleSviInputSupported():
   return ( ToggleSflow.toggleSflowSviSamplingEnabled() and
      sflowHwStatus().sviInputIfindexSupported )

def sflowSampleSviOutputSupported():
   return ( ToggleSflow.toggleSflowSviSamplingEnabled() and
      sflowHwStatus().sviOutputIfindexSupported )

def sflowSupportedGuard( mode, token ):
   if sflowHwStatus().sflowSupported:
      return None
   return CliParser.guardNotThisPlatform

def sflowAccelGuard( mode, token ):
   if sflowAccelSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowSampleIncludeDropReasonAclSupportedGuard( mode, token ):
   if sflowHwStatus().sflowSampleIncludeDropReasonAclSupported:
      return sflowSupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def sflowSampleVxlanHeaderStripSupportedGuard( mode, token ):
   if sflowHwStatus().sflowSampleVxlanHeaderStripSupported:
      return sflowSupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def sflowTunnelExtensionGuard( mode, token ):
   # Add additional tunnel extensions here as they are added
   if sflowHwStatus().tunnelIpv4EgrExtensionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowEgressSubintfGuard( mode, token ):
   if sflowHwStatus().egressSubintfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# On some switches ( currently Petra ) only few sample rates are supported
# These cli guards are used to choose the sample rate parser rule
# based on the HwStatus::sampleRangeSupported attribute published by the
# platform
def sflowSampleRangeGuard( mode, token ):
   if sflowHwStatus().sampleRangeSupported:
      return None
   return CliParser.guardNotThisPlatform

def sflowSampleSetGuard( mode, token ):
   if not sflowHwStatus().sampleRangeSupported:
      return None
   return CliParser.guardNotThisPlatform

def sflowSampleInputSviIfindexGuard( mode, token ):
   if sflowSampleSviInputSupported():
      return None
   return CliParser.guardNotThisPlatform

def sflowSampleOutputSviIfindexGuard( mode, token ):
   if sflowSampleSviOutputSupported():
      return None
   return CliParser.guardNotThisPlatform

def sflowIndependentAddressConfigGuard( mode, token ):
   if ToggleSflow.toggleSflowIndependentAddressConfigEnabled():
      return None
   return CliParser.guardNotThisEosVersion

# Takes the counters object for an accelerator and returned a python dic equivalent,
# potentially zeroing out the counters if the object timestamp is older than the last
# clear counters request.
def rectifiedSflowAccelCounters( accelCounters ):
   rectifiedCounters = {}
   fields = [
      'inPktCnt',
      'outSflowDgramCnt',
      'outFlowSampleCnt',
      'inProcPktCnt',
      'rcvPktDropCnt',
      'rcvPktTrunCnt',
      'inPktErrCnt',
      'outProcSflowDgramCnt',
      'samplePool',
   ]
   # If the last clear in smash is before the last clear request made by the Cli, it
   # means the writer agent hasn't processed the request yet, so displaying all 0s.
   needClear = accelCounters.lastClear < sflowCounterConfig.countersCleared
   for field in fields:
      value = 0 if needClear else getattr( accelCounters, field )
      rectifiedCounters[ field ] = value

   return rectifiedCounters

# Returns a python dic containing SflowAccel counters:
# {
#    sflowAccelName : {
#       counterName : counterValue,
#       ..,
#    },
#    ...
# }
def sflowAccelCounters( forAggregatedCounters=False ):
   if sflowAccelRunnability.sflowAccelPPRunnable:
      if forAggregatedCounters:
         aggregatedCountersAccelKey = FpgaCounterKey( AccelId.totalAccelId() )
         accelKeyFilter = \
               lambda accelKey: ( accelKey == aggregatedCountersAccelKey )
      else:
         accelKeyFilter = lambda accelKey: False
   else:
      accelKeyFilter = lambda accelKey: True
   accelNameFunc = lambda accelId: \
         ( 'Total' if ( accelId == AccelId.totalAccelId() )
           else AccelId( accelId ).accelName() )

   result = {}
   for accelCountersKey, accelCounters in \
         sflowAccelCountersTable.fpgaCounter.items():
      if not accelKeyFilter( accelCountersKey ):
         continue
      accelName = accelNameFunc( accelCountersKey.accelId )
      result[ accelName ] = rectifiedSflowAccelCounters( accelCounters )

   return result

# Sum of sflow accel counters for all accelerators.
def aggregatedSflowAccelCounters():
   accelCountersSum = {}
   resultCounters = sflowAccelCounters( forAggregatedCounters=True )
   for accelCounters in resultCounters.values():
      for counterName, counterValue in accelCounters.items():
         accelCountersSum[ counterName ] = ( accelCountersSum.get( counterName, 0 ) +
                                             counterValue )

   return accelCountersSum

def sflowPortChannelOutputIfindexConfigGuard( mode, token ):
   if sflowHwStatus().portChannelOutputIfindexConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def sflowSampleOutputMulticastInterfaceConfigGuard( mode, token ):
   if sflowHwStatus().egressSflowMulticastOutputIfindexConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def sflowEgressSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressModifiedSflowIntfSupported or \
      sflowHwStatus().egressUnmodifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowEgressModifiedSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressModifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def isLagIntf( intfName ):
   intf = Arnet.IntfId( intfName )
   isLag = Tac.Type( "Arnet::PortChannelIntfId" ).isPortChannelIntfId( intf )
   return isLag

def sflowIngressSubIntfGuard( mode, token ):
   if not sflowHwStatus().sflowSamplingOnSubInterfaceIngressSupported:
      return CliParser.guardNotThisPlatform
   return None

def sflowEgressSubIntfGuard( mode, token ):
   if not sflowHwStatus().samplingOnSubInterfaceEgressSupported:
      return CliParser.guardNotThisEosVersion
   else:
      return None

def sflowEgressUnmodifiedSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressUnmodifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowMaxDatagramSizeGuard( mode, token ):
   if ToggleSflow.toggleSflowMaxDatagramSizeEnabled():
      return None
   else:
      return CliParser.guardNotThisEosVersion

#-------------------------------------------------------------------------------
# the "[no] sflow run" command in "config" mode
# full syntax: {no|default} sflow run
#-------------------------------------------------------------------------------
def runSflow( mode, args ):
   sflowConfig.enabled = True

def noRunSflow( mode, args ):
   sflowConfig.enabled = False

#-------------------------------------------------------------------------------
# the "[no] sflow polling-interval <secs>" command in "config" mode
# full syntax: {no|default} sflow polling-interval <secs>
#-------------------------------------------------------------------------------
def setPollingInterval( mode, args ):
   sflowConfig.pollingInterval = args[ 'POLLING_INTERVAL' ]

def defaultPollingInterval( mode, args ):
   sflowConfig.pollingInterval = sflowConfig.pollingIntervalDefault

def noPollingInterval( mode, args ):
   # setting pollingInterval to 0 implies polling is disabled.
   sflowConfig.pollingInterval = 0

#-------------------------------------------------------------------------------
# the "[no] sflow datagram size maximum <bytes>" command in "config" mode
# full syntax: {no|default} sflow datagram size maximum <bytes>
#-------------------------------------------------------------------------------
def setMaxDatagramSize( mode, args ):
   sflowConfig.maxDatagramSize = args[ 'MAX_DATAGRAM_SIZE' ]

def defaultMaxDatagramSize( mode, args ):
   sflowConfig.maxDatagramSize = sflowConfig.maxDatagramSizeDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample <rate>" command in "config" mode
# full syntax: {default} sflow sample <rate>
#              no sflow sample
#-------------------------------------------------------------------------------
def rateRangeFn( mode, context ):
   # This function returns the range supported for the Sflow sample rate
   return ( sflowHwStatus().minSampleRate, sflowHwStatus().maxSampleRate )

def dangRateRangeFn( mode, context ):
   # This function returns the entire/dangerous Sflow sample rate range
   return ( 1, sflowHwStatus().maxSampleRate )

def setSampleRate( mode, args ):
   rate = args[ 'RATE' ] if 'RATE' in args else args[ 'DANGEROUS_RATE' ]
   sflowConfig.sampleRate = int( rate )
   sflowConfig.sampleDangerous = 'dangerous' in args

def defaultSampleRate( mode, args ):
   sflowConfig.sampleRate = sflowConfig.sampleRateDefault
   sflowConfig.sampleDangerous = False

#-------------------------------------------------------------------------------
# the "[no] sflow destination <ipv4/6-addr-or-hostname> [<dest-udp-port>]" command
# in "config" mode
# full syntax:
#   {default|no} sflow destination <ip4/6-addr-or-hostname> [<dst-udp-port>]
#-------------------------------------------------------------------------------
def setDestination( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   port = args.get( 'PORT', SflowConst.defaultPort )
   addr = str( args[ 'DESTINATION' ] )
   # BUG23291 - Reenable this when resolution is asynchronous
   # HostnameCli.resolveHostname( hostname, printWarnMsg=True )
   # BUG23291-Use hostname instead of ipAddr for an argument after the fix

   SflowUtil.addConfigCollectorInfo( sflowConfig, addr, port, vrfName )
   updateDscpRules()

def noDestination( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   port = args.get( 'PORT' )
   addr = str( args[ 'DESTINATION' ] )
   SflowUtil.removeConfigCollectorInfo( sflowConfig, addr, port, vrfName )
   updateDscpRules()

# tokenHostname = HostnameCli.IpAddrOrHostnameRule(
#   name='hostname',
#   helpname='WORD',
#   helpdesc='Hostname or IP address of Collector',
#   ipv6=False )

# BUG23291 - Reenable this when resolution is asynchronous
# BasicCli.GlobalConfigMode.addCommand(
#    ( tokenSflow, tokenDestination, tokenHostname, [ '>>port', port ],
#      setDestination ) )

# BUG23291 - Reenable this when resolution is asynchronous
# BasicCli.GlobalConfigMode.addCommand(
#    ( BasicCli.noOrDefault, tokenSflow, tokenDestination,
#       tokenHostname, [ '>>port', port ], noDestination ) )

#-------------------------------------------------------------------------------
# -  the "[no] sflow source <ip-addr>" command
# full syntax: {no|default} sflow source <ip-addr>
#-------------------------------------------------------------------------------
def setSource( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = str( args[ 'SOURCE' ] )

   if ( vrfName in sflowConfig.sflowVrfConfig and
        sflowConfig.sflowVrfConfig[ vrfName ].srcIntfName ):
      mode.addError( "Source-interface is already configured for VRF " + vrfName )
   elif( vrfName in sflowConfig.sflowVrfConfig and
         sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'independent' ):
      mode.addError( SflowConst.sflowSourceErrorMsg )
   else:
      SflowUtil.addConfigSourceIp( sflowConfig, ipAddr, vrfName, 'dependent' )

def noSource( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = args.get( 'SOURCE', None )

   if vrfName not in sflowConfig.sflowVrfConfig:
      return

   if sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'independent':
      mode.addError( SflowConst.sflowSourceErrorMsg )
      return

   if not ipAddr:
      SflowUtil.removeConfigSourceIpAll( sflowConfig, vrfName )
   else:
      SflowUtil.removeConfigSourceIp( sflowConfig, ipAddr, vrfName )

# -------------------------------------------------------------------------------
# - the "[no] sflow source-interface <intf-name>" command
# full syntax: {no|default} sflow source-interface <intf-name>
# -------------------------------------------------------------------------------
def setSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   intf = args[ 'IPINTF' ]

   cfg = sflowConfig.sflowVrfConfig.get( vrfName )
   if cfg and ( cfg.switchIpAddr or cfg.switchIp6Addr ):
      mode.addError( "Source IP address is already configured for VRF " + vrfName )
   else:
      SflowUtil.addConfigSourceIntf( sflowConfig, intf.name, vrfName )

def noSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   SflowUtil.removeConfigSourceIntf( sflowConfig, vrfName )

# -------------------------------------------------------------------------------
# - the "[no] sflow source address" command
# full syntax: {no|default} sflow source address <ip-addr>
# -------------------------------------------------------------------------------
def setSourceAddress( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = str( args[ 'SOURCE' ] )

   if vrfName in sflowConfig.sflowVrfConfig:
      if sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'dependent':
         mode.addWarning( "\"sflow source\" configuration removed." )

   SflowUtil.addConfigSourceIp( sflowConfig, ipAddr, vrfName, 'independent' )

def noSourceAddress( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = None
   if 'SOURCE' in args:
      ipAddr = str( args[ 'SOURCE' ] )

   if vrfName not in sflowConfig.sflowVrfConfig:
      return
   elif sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'dependent':
      noSource( mode, args )
      return

   sflowConfig.sflowVrfConfig[ vrfName ].switchIpAddr = IpGenAddr()
   sflowConfig.sflowVrfConfig[ vrfName ].switchIp6Addr = IpGenAddr()

   if( sflowConfig.sflowVrfConfig[ vrfName ].agentIpAddr.isAddrZero and
       sflowConfig.sflowVrfConfig[ vrfName ].agentIp6Addr.isAddrZero ):
      sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType = 'none'

      if not ipAddr:
         SflowUtil.removeConfigSourceIpAll( sflowConfig, vrfName )
      else:
         SflowUtil.removeConfigSourceIp( sflowConfig, ipAddr, vrfName )

# -------------------------------------------------------------------------------
# the "[no] sflow agent address" command
# full syntax: [no|default] sflow agent address <ip-addr>
# -------------------------------------------------------------------------------
def setAgentAddress( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = str( args[ 'ADDR' ] )

   if vrfName in sflowConfig.sflowVrfConfig:
      if sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'dependent':
         mode.addWarning( "\"sflow source\" configuration removed." )
   SflowUtil.addConfigAgentIp( sflowConfig, ipAddr, vrfName )

def noAgentAddress( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = None
   if 'ADDR' in args:
      ipAddr = args[ 'ADDR' ]

   if vrfName not in sflowConfig.sflowVrfConfig:
      return
   elif sflowConfig.sflowVrfConfig[ vrfName ].addrConfigType == 'dependent':
      mode.addError( "VRF Agent configured dependently on the source address" )
      return
   if not ipAddr:
      SflowUtil.removeConfigSourceIpAll( sflowConfig, vrfName )
   else:
      SflowUtil.removeConfigAgentIp( sflowConfig, ipAddr, vrfName )

#-------------------------------------------------------------------------------
# the "[no] sflow sample input svi ifindex (svi|member)" command
# full syntax: [no|default] sflow sample input ifindex (svi|member)"
#-------------------------------------------------------------------------------
def sflowSampleInputSviHandler( mode, args ):
   sflowConfig.inputSviIfindex = args[ 'INDEX' ] == 'svi'

def noSflowSampleInputSviHandler( mode, args ):
   sflowConfig.inputSviIfindex = sflowConfig.inputSviIfindexDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample output svi ifindex (svi|member)" command
# full syntax: [no|default] sflow sample output svi ifindex (svi|member)"
#-------------------------------------------------------------------------------
def sflowSampleOutputSviHandler( mode, args ):
   sflowConfig.outputSviIfindex = args[ 'INDEX' ] == 'svi'

def noSflowSampleOutputSviHandler( mode, args ):
   sflowConfig.outputSviIfindex = sflowConfig.outputSviIfindexDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample rewrite dscp" command in "config" mode
# full syntax: {no|default} sflow sample rewrite dscp
#-------------------------------------------------------------------------------
def rewriteDscp( mode, args ):
   sflowConfig.rewriteDscp = True

def noRewriteDscp( mode, args ):
   sflowConfig.rewriteDscp = False

#-------------------------------------------------------------------------------
# the "[no] sflow sample output interface" command in "config" mode
# full syntax: {no|default} sflow sample output interface
#
# the "[no] sflow sample output subinterface" command in "config" mode
# full syntax: {no|default} sflow sample output subinterface
#-------------------------------------------------------------------------------
def sflowSampleOutputHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = True
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = True

def sflowSampleOutputNoHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = False
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = False

def sflowSampleOutputDefaultHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = sflowConfig.outputInterfaceDefault
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = sflowConfig.egressSubintfDefault

# -------------------------------------------------------------------------------
# the "[no] sflow sample output vlan" command in "config" mode
# full syntax: {no|default} sflow sample output vlan
# -------------------------------------------------------------------------------
def sflowSampleOutputVlanHandler( mode, args ):
   sflowConfig.updateSubIntfVlan = True

def sflowSampleOutputVlanNoHandler( mode, args ):
   sflowConfig.updateSubIntfVlan = False

def sflowSampleOutputVlanDefaultHandler( mode, args ):
   sflowConfig.updateSubIntfVlan = sflowConfig.updateSubIntfVlanDefault

# -------------------------------------------------------------------------------
# the "[no] sflow sample input subinterface" command in "config" mode
# full syntax: {no|default} sflow sample input subinterface
# -------------------------------------------------------------------------------
def sflowSampleInputHandler( mode, args ):
   sflowConfig.ingressSubintf = True

def sflowSampleInputNoHandler( mode, args ):
   sflowConfig.ingressSubintf = False

def sflowSampleInputDefaultHandler( mode, args ):
   sflowConfig.ingressSubintf = sflowConfig.ingressSubintfDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample truncate size SIZE" command in "config" mode
# full syntax: {no|default} sflow sample truncate size ...
#-------------------------------------------------------------------------------
def sflowSampleTruncateSizeHandler( mode, args ):
   size = args.get( 'SIZE', sflowConfig.sampleTruncateSizeDefault )
   sflowConfig.sampleTruncateSize = size

#-------------------------------------------------------------------------------
# the "[no|default] sflow extension bgp" command in global "config" mode
# the "[no|default] sflow extension mpls" command in global "config" mode
# the "[no|default] sflow extension router" command in global "config" mode
# the "[no|default] sflow extension switch" command in global "config" mode
# the "[no|default] sflow extension tunnel ipv4 egress" command in global "config"
# mode
#-------------------------------------------------------------------------------
def setExtension( attrName, value ):
   # None means to use the default value of this extension.
   effectiveValue = getattr( sflowConfig, attrName + 'Default' ) \
                    if value is None else value
   setattr( sflowConfig, attrName, effectiveValue )

#-------------------------------------------------------------------------------
# the "sflow sample output portchannel ifindex" command in "config" mode
# full syntax:
# {default} sflow sample output portchannel ifindex { member | portchannel }
#-------------------------------------------------------------------------------
def setPortChannelIfindexMode( mode, args ):
   sflowConfig.portChannelIfindex = args[ 'IFINDEX_MODE' ]

def defaultPortChannelIfindex( mode, args ):
   sflowConfig.portChannelIfindex = 'platformDefault'

#-------------------------------------------------------------------------------
# the "sflow sample output multicast interface" command in "config" mode
# full syntax: [ no | default ] sflow sample output multicast interface
#-------------------------------------------------------------------------------
def outputMuticastInterface( mode, args ):
   sflowConfig.egressSflowMulticastOutputIfindex = True

def noOutputMuticastInterface( mode, args ):
   sflowConfig.egressSflowMulticastOutputIfindex = False

#-------------------------------------------------------------------------------
# the "[no] sflow enable" command in "config-if" mode
# full syntax: [no] sflow enable
#-------------------------------------------------------------------------------
def intfIngressSflowEnableHandler( mode, args ):
   if mode.intf.name in sflowConfig.disabledIngressIntf:
      del sflowConfig.disabledIngressIntf[ mode.intf.name ]
   sflowConfig.enabledIngressIntf[ mode.intf.name ] = True

def intfIngressSflowEnableNoHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledIngressIntf:
      del sflowConfig.enabledIngressIntf[ mode.intf.name ]
   sflowConfig.disabledIngressIntf[ mode.intf.name ] = True

def intfIngressSflowEnableDefaultHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledIngressIntf:
      del sflowConfig.enabledIngressIntf[ mode.intf.name ]
   if mode.intf.name in sflowConfig.disabledIngressIntf:
      del sflowConfig.disabledIngressIntf[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow egress enable" command in "config-if" mode
# full syntax: [no] sflow egress enable
#-------------------------------------------------------------------------------
def intfEgressSflowEnableHandler( mode, args ):
   if mode.intf.name in sflowConfig.disabledEgressIntf:
      del sflowConfig.disabledEgressIntf[ mode.intf.name ]
   sflowConfig.enabledEgressIntf[ mode.intf.name ] = True

def intfEgressSflowEnableNoHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledEgressIntf:
      del sflowConfig.enabledEgressIntf[ mode.intf.name ]
   sflowConfig.disabledEgressIntf[ mode.intf.name ] = True

def intfEgressSflowEnableDefaultHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledEgressIntf:
      del sflowConfig.enabledEgressIntf[ mode.intf.name ]
   if mode.intf.name in sflowConfig.disabledEgressIntf:
      del sflowConfig.disabledEgressIntf[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow id ID" command in "config-if" mode
# full syntax: [no|default] sflow id ...
#-------------------------------------------------------------------------------
def intfIdHandler( mode, args ):
   configuredId = args.get( 'ID' )
   sflowConfig.intfConfig.newMember( mode.intf.name ).id = configuredId

def intfIdNoDefaultHandler( mode, args ):
   del sflowConfig.intfConfig[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow interface disable default" command in "config" mode
# full syntax: {no|default} sflow interface disable default
#-------------------------------------------------------------------------------
def disableIngressInterfacesByDefault( mode, args ):
   sflowConfig.ingressIntfsEnabledDefault = False

def enableIngressInterfacesByDefault( mode, args ):
   sflowConfig.ingressIntfsEnabledDefault = True

#-------------------------------------------------------------------------------
# the "[no] sflow interface egress enable default" command in "config" mode
# full syntax: {no|default} sflow interface egress enable default
#-------------------------------------------------------------------------------
def disableEgressInterfacesByDefault( mode, args ):
   sflowConfig.egressIntfsEnabledDefault = False

def enableEgressInterfacesByDefault( mode, args ):
   sflowConfig.egressIntfsEnabledDefault = True

#-------------------------------------------------------------------------------
# the "[no] sflow sample include drop reason acl" command in "config" mode
# full syntax: {no|default} sflow sample include drop reason acl
#-------------------------------------------------------------------------------
def includeDropReasonAcl( mode, args ):
   sflowConfig.sampleIncludeDropReasonAcl = True

def noIncludeDropReasonAcl( mode, args ):
   sflowConfig.sampleIncludeDropReasonAcl = False

#-------------------------------------------------------------------------------
# the "[no] sflow sample vxlan header strip" command in "config" mode
# full syntax: {no|default} sflow sample vxlan header strip
#-------------------------------------------------------------------------------
def vxlanHeaderStrip( mode, args ):
   sflowConfig.sampleVxlanHeaderStrip = True

def noVxlanHeaderStrip( mode, args ):
   sflowConfig.sampleVxlanHeaderStrip = False

#-------------------------------------------------------------------------------
# the "clear sflow counters" command in "config" mode
# full syntax: clear sflow counters
#-------------------------------------------------------------------------------
def setCountersCleared( mode, args ):
   sflowCounterConfig.countersCleared = sflowCounterConfig.countersCleared + 1

#-------------------------------------------------------------------------------
# the "[no|default] sflow qos dscp <0-63>" command in global "config" mode
#-------------------------------------------------------------------------------
def updateDscpRules():
   dscpValue = sflowConfig.dscpValue

   if not dscpValue:
      del dscpConfig.protoConfig[ 'sflow' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'sflow' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for vrfConfig in sflowConfig.sflowVrfConfig.values():
      for hostAndPort in vrfConfig.collectorHostAndPort.values():
         DscpCliLib.addDscpRule( ruleColl, hostAndPort.hostname, hostAndPort.port,
                                 False, vrfConfig.vrfName, 'udp', dscpValue )
         DscpCliLib.addDscpRule( ruleColl, hostAndPort.hostname, hostAndPort.port,
                                 False, vrfConfig.vrfName, 'udp', dscpValue,
                                 v6=True )

def setDscp( mode, args ):
   sflowConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   sflowConfig.dscpValue = sflowConfig.dscpValueDefault
   updateDscpRules()

#-------------------------------------------------------------------------------
# the "show sflow" and "show sflow detail" commands
#-------------------------------------------------------------------------------

def updateCounters( mode ):
   if os.environ.get( 'SIMULATION_VMID' ):
      # Bypass this in btests
      return

   sflowCounterConfig.counterUpdateRequestTime = Tac.now() # monotonic time

   def countersUpdated():
      sflowCountersUpdated = True
      dropExportCountersUpdated = True
      if sflowConfig.enabled:
         sflowCountersUpdated = ( sflowStatus.counterUpdateTime >=
                                  sflowCounterConfig.counterUpdateRequestTime )
      if dropExportSflowConfig.enabled:
         dropExportCountersUpdated = ( dropExportSflowStatus.counterUpdateTime >=
                                       sflowCounterConfig.counterUpdateRequestTime )
      return sflowCountersUpdated and dropExportCountersUpdated
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def getDestIpAddresses( config, status ):
   ipv4Destinations = []
   ipv6Destinations = []
   for vrfName in sorted( status.sflowVrfStatus ):
      if vrfName not in config.sflowVrfConfig:
         continue
      collectorHostAndPort = \
         config.sflowVrfConfig[ vrfName ].collectorHostAndPort
      sflowVrfStatus = status.sflowVrfStatus[ vrfName ]
      for hostname in sflowVrfStatus.collectorIpAndPort:
         ipv4Destination = SflowStatus.Ipv4Destination()
         ip = sflowVrfStatus.collectorIpAndPort[ hostname ].ip
         if hostname in collectorHostAndPort:
            ipv4Destination.hostname = hostname
            ipv4Destination.ipv4Address = ip
            ipv4Destination.port = collectorHostAndPort[ hostname ].port
            ipv4Destination.vrfName = vrfName
            ipv4Destination.datagramsCount = sflowVrfStatus.datagramsCount.get(
                                             hostname, 0 )
            ipv4Destinations.append( ipv4Destination )

      for hostname in sflowVrfStatus.collectorIp6AndPort:
         ipv6Destination = SflowStatus.Ipv6Destination()
         ip = sflowVrfStatus.collectorIp6AndPort[ hostname ].ip
         if hostname in collectorHostAndPort:
            ipv6Destination.hostname = hostname
            ipv6Destination.ipv6Address = ip
            ipv6Destination.port = collectorHostAndPort[ hostname ].port
            ipv6Destination.vrfName = vrfName
            ipv6Destination.datagramsCount = sflowVrfStatus.datagramsCount.get(
                                             hostname, 0 )
            ipv6Destinations.append( ipv6Destination )

   return ipv4Destinations, ipv6Destinations

def getSourceIpAddresses( config, status ):
   ipv4Sources = []
   ipv6Sources = []
   for vrfName in sorted( config.sflowVrfConfig ):
      ipv4Source = SflowStatus.Ipv4Source()
      ipv6Source = SflowStatus.Ipv6Source()

      sflowVrfConfig = config.sflowVrfConfig[ vrfName ]
      if sflowVrfConfig.srcIntfName:
         sflowVrfStatus = status.sflowVrfStatus.get( vrfName )
         ipv4Source.ipv4Address = \
            sflowVrfStatus.srcIpAddr if sflowVrfStatus else SflowConst.defaultIp
         ipv4Source.sourceInterface = sflowVrfConfig.srcIntfName
         ipv6Source.ipv6Address = \
            sflowVrfStatus.srcIp6Addr if sflowVrfStatus else SflowConst.defaultIp6
         ipv6Source.sourceInterface = sflowVrfConfig.srcIntfName
      else:
         ipv4Source.ipv4Address = sflowVrfConfig.switchIpAddr.v4Addr
         ipv4Source.sourceInterface = None
         ipv6Source.ipv6Address = sflowVrfConfig.switchIp6Addr.v6Addr.stringValue
         ipv6Source.sourceInterface = None

      ipv4Source.vrfName = vrfName
      ipv6Source.vrfName = vrfName
      ipv4Sources.append( ipv4Source )
      ipv6Sources.append( ipv6Source )

   if not ipv4Sources:
      ipv4Source = SflowStatus.Ipv4Source()
      ipv4Source.ipv4Address = SflowConst.defaultIp
      ipv4Source.sourceInterface = None
      ipv4Source.vrfName = DEFAULT_VRF
      ipv4Sources.append( ipv4Source )

   if not ipv6Sources:
      ipv6Source = SflowStatus.Ipv6Source()
      ipv6Source.ipv6Address = SflowConst.defaultIp6
      ipv6Source.sourceInterface = None
      ipv6Source.vrfName = DEFAULT_VRF
      ipv6Sources.append( ipv6Source )

   return ipv4Sources, ipv6Sources

def getAgentIpAddresses( config, status ):
   ipv4Agents = []
   ipv6Agents = []
   for vrfName in sorted( config.sflowVrfConfig ):
      ipv4Source = SflowStatus.Ipv4Source()
      ipv6Source = SflowStatus.Ipv6Source()

      sflowVrfConfig = config.sflowVrfConfig[ vrfName ]
      if sflowVrfConfig.srcIntfName:
         sflowVrfStatus = status.sflowVrfStatus.get( vrfName )
         ipv4Source.ipv4Address = \
               sflowVrfStatus.agentIpAddr if sflowVrfStatus else SflowConst.defaultIp
         ipv4Source.sourceInterface = ( sflowVrfConfig.srcIntfName if
               sflowVrfConfig.agentIpAddr.v4Addr == SflowConst.defaultIp else None )
         ipv6Source.ipv6Address = ( sflowVrfStatus.agentIp6Addr
                     if sflowVrfStatus else SflowConst.defaultIp6 )
         ipv6Source.sourceInterface = ( sflowVrfConfig.srcIntfName if
               sflowVrfConfig.agentIp6Addr.v6Addr ==
                              SflowConst.defaultIp6 else None )
      else:
         ipv4Source.ipv4Address = sflowVrfConfig.agentIpAddr.v4Addr
         ipv4Source.sourceInterface = None
         ipv6Source.ipv6Address = sflowVrfConfig.agentIp6Addr.v6Addr.stringValue
         ipv6Source.sourceInterface = None

      ipv4Source.vrfName = vrfName
      ipv6Source.vrfName = vrfName
      ipv4Agents.append( ipv4Source )
      ipv6Agents.append( ipv6Source )

   if not ipv4Agents:
      ipv4Source = SflowStatus.Ipv4Source()
      ipv4Source.ipv4Address = SflowConst.defaultIp
      ipv4Source.sourceInterface = None
      ipv4Source.vrfName = DEFAULT_VRF
      ipv4Agents.append( ipv4Source )

   if not ipv6Agents:
      ipv6Source = SflowStatus.Ipv6Source()
      ipv6Source.ipv6Address = SflowConst.defaultIp6
      ipv6Source.sourceInterface = None
      ipv6Source.vrfName = DEFAULT_VRF
      ipv6Agents.append( ipv6Source )

   return ipv4Agents, ipv6Agents


def getSendingDatagrams( config, status ):
   sendingDatagrams = []
   for vrfName in sorted( config.sflowVrfConfig ):
      if vrfName in status.sflowVrfStatus:
         sendingDatagram = SflowStatus.SendingDatagram()
         sflowVrfStatus = status.sflowVrfStatus[ vrfName ]
         sendingDatagram.sending = sflowVrfStatus.sendingSflowDatagrams
         sendingDatagram.reason = sflowVrfStatus.sendingSflowDatagramsReason
         sendingDatagram.vrfName = vrfName
         sendingDatagrams.append( sendingDatagram )

   if not sendingDatagrams:
      sendingDatagram = SflowStatus.SendingDatagram()
      sendingDatagram.sending = status.sendingSflowDatagrams
      sendingDatagram.reason = status.sendingSflowDatagramsReason
      sendingDatagram.vrfName = DEFAULT_VRF
      sendingDatagrams.append( sendingDatagram )

   return sendingDatagrams

def getBgpExports():
   bgpExports = []
   for vrfName in sorted( sflowConfig.sflowVrfConfig ):
      bgpExport = SflowStatus.BgpExport()
      # Bgp extension is currently not per vrf
      bgpExport.export = sflowConfig.bgpExtension
      bgpExport.vrfName = vrfName
      bgpExports.append( bgpExport )

   if not bgpExports:
      bgpExport = SflowStatus.BgpExport()
      bgpExport.export = sflowConfig.bgpExtension
      bgpExport.vrfName = DEFAULT_VRF
      bgpExports.append( bgpExport )

   return bgpExports

def sflowEncodingFormat():
   LazyMount.force( sflowAccelStatusDir )
   statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                    sflowAccelStatusDir )
   sflowAccelStatus = statusHelper.sflowAccelStatus()
   hardwareAccelerated = sflowAccelSupported() and sflowAccelStatus.running

   ingressSubintfSupported = (
         sflowAccelCapabilities.advancedCapabilities.get( "ingressSubintf", False )
         if hardwareAccelerated else True )
   egressSubintfSupported = (
         sflowAccelCapabilities.advancedCapabilities.get( "egressSubintf", False )
         if hardwareAccelerated else True )

   if ( ( sflowConfig.ingressSubintf and ingressSubintfSupported ) or
         ( sflowConfig.egressSubintf and egressSubintfSupported ) ):
      return "expanded"
   else:
      return "compact"

def getSflowDetails():
   details = SflowStatus.Details()
   details.hardwareSamplingEnabled = sflowHwConfig.enabled
   details.samplesDiscarded = sflowStatus.samplesDiscarded
   details.sampleOutputInterface = sflowConfig.outputInterface
   details.sampleMplsExtension = sflowConfig.mplsExtension
   details.sampleVplsExtension = sflowConfig.vplsExtension
   details.sampleEvpnMplsExtension = sflowConfig.evpnMplsExtension
   details.sampleVxlanExtension = sflowConfig.vxlanExtension
   details.sampleSwitchExtension = sflowConfig.switchExtension
   details.sampleRouterExtension = sflowConfig.routerExtension
   details.sampleTunnelIpv4EgrExtension = sflowConfig.tunnelIpv4EgrExtension
   details.sampleVxlanHeaderStrip = sflowConfig.sampleVxlanHeaderStrip
   details.portChannelOutputIfIndex = sflowHwStatus().portChannelOutputIfindex
   details.sviIfindexOutput = sflowConfig.outputSviIfindex
   details.sviIfindexInput = sflowConfig.inputSviIfindex
   details.sampleIngressSubintf = sflowConfig.ingressSubintf
   details.sampleEgressSubintf = sflowConfig.egressSubintf
   if ToggleSflow.toggleSflowL2SubIntfVlanEnabled():
      details.sampleUpdateSubIntfVlan = sflowConfig.updateSubIntfVlan
   details.sampleEncodingFormat = sflowEncodingFormat()
   LazyMount.force( sflowAccelStatusDir )
   statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                   sflowAccelStatusDir )
   sflowAccelStatus = statusHelper.sflowAccelStatus()
   # No point displaying SflowAccel status if not enabled
   if sflowAccelSupported() and sflowAccelStatus.running and \
      sflowAccelStatus.inactiveFeature:
      for feature in sflowAccelStatus.inactiveFeature.values():
         descriptionStr = MessageHelper.featureDescription( feature )
         details.accelUnsupportedExtensions.append( descriptionStr )

   return details

def doPrepShowSflow( mode, args ):
   if sflowConfig.enabled or dropExportSflowConfig.enabled:
      updateCounters( mode )
   else:
      mode.addWarning( "Displaying counters that may be stale" )

def doShowSflow( mode, args ):
   detail = 'detail' in args
   dropPacketSamplingEnabled = dropExportSflowConfig.enabled
   ret = SflowStatus()
   ret.enabled = sflowConfig.enabled
   ret.dropPacketSamplingEnabled = dropPacketSamplingEnabled
   ret.samplingEnabled = sflowConfig.samplingEnabled
   ret.sendingDatagrams = getSendingDatagrams( sflowConfig, sflowStatus )
   # Indicates whether sflow Agent is running
   ret.running = sflowStatus.active
   # Indicates whether drop sample export is running in sFlow Agent
   ret.dropPacketExportRunning = dropExportSflowStatus.active
   # These are optional attributes that we set only if drop sampling is enabled.
   # i.e mirror-on-drop is configured and sFlow export is enabled.
   # There is no point in filling in the sendingDatagrams attribute when
   # drop export sampling is not running ( active ).
   if dropPacketSamplingEnabled and ret.dropPacketExportRunning:
      ret.sendingDatagramsForDropSamples = getSendingDatagrams(
         dropExportSflowConfig, dropExportSflowStatus )
   ret.bgpExports = getBgpExports()
   ret.polling = sflowStatus.pollingEnabled
   ret.pollingInterval = sflowConfig.pollingInterval
   ret.sampleRate = sflowConfig.sampleRate
   ret.sampleTruncateSize = sflowConfig.sampleTruncateSize
   if ToggleSflow.toggleSflowMaxDatagramSizeEnabled():
      ret.maxDatagramSize = sflowConfig.maxDatagramSize
   ret.rewriteDscp = sflowConfig.rewriteDscp
   ret.dscpValue = sflowConfig.dscpValue
   ret.hardwareSampleRate = sflowHwSampleRate()
   ret.ipv4Destinations, ret.ipv6Destinations = getDestIpAddresses( sflowConfig,
                                                                    sflowStatus )
   # These are optional attributes that we set only if drop sampling is enabled.
   # i.e mirror-on-drop is configured and sFlow export is enabled.
   if dropPacketSamplingEnabled:
      ret.ipv4DestinationsForDropSamples, ret.ipv6DestinationsForDropSamples = \
         getDestIpAddresses( dropExportSflowConfig, dropExportSflowStatus )
   ret.ipv4Sources, ret.ipv6Sources = getSourceIpAddresses( sflowConfig,
                                                            sflowStatus )
   ret.ipv4Agents, ret.ipv6Agents = getAgentIpAddresses( sflowConfig,
                                                         sflowStatus )
   # These are optional attributes that we set only if drop sampling is enabled.
   # i.e mirror-on-drop is configured and sFlow export is enabled.
   if dropPacketSamplingEnabled:
      ret.ipv4SourcesForDropSamples, ret.ipv6SourcesForDropSamples = \
         getSourceIpAddresses( dropExportSflowConfig, dropExportSflowStatus )

   ret.totalPackets = sflowStatus.totalPackets
   ret.softwareSamples = sflowStatus.swSamples
   ret.samplePool = sflowStatus.samplePool
   ret.hardwareSamples = sflowHwSamples()
   ret.datagrams = sflowStatus.datagrams
   # These are optional attributes that we set only if drop sampling is enabled.
   # i.e mirror-on-drop is configured and sFlow export is enabled.
   ret.dropSampleDatagrams = dropExportSflowStatus.datagrams
   ret.dropSamplesReceived = dropExportSflowStatus.totalDiscardSamplesReceived
   ret.dropSamplesSent = dropExportSflowStatus.totalSamplesSent
   if detail:
      ret.details = getSflowDetails()

   if sflowAccelSupported():
      LazyMount.force( sflowAccelStatusDir )
      statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                      sflowAccelStatusDir )
      sflowAccelStatus = statusHelper.sflowAccelStatus()
      ret.accelSupported = True
      ret.accelEnabled = sflowAccelStatus.running
      if sflowAccelConfig.sampleRate == 0:
         # 0 means use the SW sflow rate
         ret.accelSampleRate = sflowConfig.sampleRate
      else:
         ret.accelSampleRate = sflowAccelConfig.sampleRate
      # No point displaying SflowAccel status if not enabled
      if sflowAccelStatus.running:
         ret.hardwareAccelSampleRate = sflowAccelHwConfig.sampleRate
         ret.hwSampleTruncateSize = sflowHwStatus().hwSampleTruncateSize

      ret.dynamicUdpSourcePort = sflowAccelConfig.dynamicUdpSourcePort
      accelCtrs = aggregatedSflowAccelCounters()
      # Don't need to update ret.totalPackets, as it's the sum of all packets on all
      # interfaces, it is only updated by PollerSms when running counters polling.

      # softwareSamples is actually showed as "Number of Samples" in the Cli output,
      # for SW Sflow it actually represents the packets processed by the Sflow
      # agent. The closest equivalent for SflowAccel is inProcPktCnt which represents
      # the packets that made it to the accelerator and were successfuly processed.
      ret.softwareSamples += accelCtrs.get( 'inProcPktCnt', 0 )

      # Sample pool as given by Sflow accelerators.
      ret.samplePool += accelCtrs.get( 'samplePool', 0 )

      # hardwareSamples is actually showed as "Hardware Trigger" in the Cli output,
      # for SW Sflow it is updated using sand-dma counter. The closest equivalent for
      # SflowAccel is inPktCnt which represents all the packets that made it to the
      # accelerator, they may then be processed or dropped.
      ret.hardwareSamples += accelCtrs.get( 'inPktCnt', 0 )

      # Number of datagrams generated by the Aclow accelerators.
      ret.datagrams += accelCtrs.get( 'outSflowDgramCnt', 0 )

      if detail:
         # Use the sum of packets that arrived with errors to the accelerator + the
         # packets that had to be dropped due to bandwidth limitations of the
         # accelerator.
         ret.details.samplesDiscarded += ( accelCtrs.get( 'rcvPktDropCnt', 0 ) +
                                           accelCtrs.get( 'inPktErrCnt', 0 ) )

   return ret

#-------------------------------------------------------------------------------
# the "show sflow interfaces" command
#-------------------------------------------------------------------------------
# the hook should return the egress runIntfs disabled by platform, currently provided
# by SandFap and SandDanz only
egressIntfDisabledHook = CliExtensions.CliHook()

# egressLagMemberDisabledHook extensions should return a dictionary where the key
# a lag interface and the value is a LagMemberStatus namedtuple. Each field in the
# LagMemberStatus tuple should be a list of interfaces. This is required to handle
# the case where some but not all lag members are programmed in hardware. Currently,
# this case only applies to egress sFlow lag interfaces on Jericho2, so
# the egressLagMemberStatusHook is only provided by SandDanz.
lagMemberStatusFields = ( 'activeIntfs', 'inactiveIntfs' )
LagMemberStatus = namedtuple( 'LagMemberStatus', lagMemberStatusFields )
egressLagMemberStatusHook = CliExtensions.CliHook()

def doShowSflowIntfs( mode, args ):
   detail = 'detail' in args
   ret = SflowInterfaces( _detail=detail )
   ret.enabled = sflowConfig.enabled
   ret.ingressInterfacesEnabledDefault = sflowConfig.ingressIntfsEnabledDefault
   ret.egressInterfacesEnabledDefault = sflowConfig.egressIntfsEnabledDefault
   egressSamplingIntfs = set()
   disabledEgressIntfs = set()
   egressLagMemberStatus = {}

   # This is a temporary workaround for BUG173418. RunIntfs should be cleared by
   # ConfigSm by reacting to pollingEnabled and samplingEnabled
   if sflowConfig.samplingEnabled:
      # Intfs that have sflow-action traffic-policy applied will also be present in
      # sflowHwConfig.runIntf with value 'sampleTargeted'. Only show intfs with
      # 'sampleAll' (intfs that have ingress sflow enabled).
      samplingIntfs = { k for k, v in sflowHwConfig.runIntf.items()
                        if v == sampleAll }
      egressSamplingIntfs = set( sflowHwConfig.runEgressIntf )
   else:
      samplingIntfs = set()

   if sflowStatus.pollingEnabled:
      pollingIntfs = set( sflowStatus.runIntfForPolling )
   else:
      pollingIntfs = set()

   for hook in egressIntfDisabledHook.extensions():
      disabledEgressIntfs.update( hook( mode ) )

   for hook in egressLagMemberStatusHook.extensions():
      egressLagMemberStatus.update( hook( mode ) )

   for intfName in pollingIntfs | samplingIntfs | egressSamplingIntfs:
      intfStatusMap = SflowTypeStatusMap()
      if ( intfName in sflowHwStatus().intfOperStatus and
           not sflowHwStatus().intfOperStatus[ intfName ].running ):
         reason = sflowHwStatus().intfOperStatus[ intfName ].reason
         intfStatusMap.intfOperStatusReason = reason
      else:
         if intfName in samplingIntfs:
            sflowType = 'ingress'
            typeStatus = SflowTypeStatus( status='running' )
            intfStatusMap.addSflowStatus( sflowType, typeStatus )
         if intfName in egressSamplingIntfs:
            sflowType = 'egress'
            if intfName in egressLagMemberStatus:
               lagMemberStatus = egressLagMemberStatus[ intfName ]
               if lagMemberStatus.activeIntfs:
                  status = 'partial' if lagMemberStatus.inactiveIntfs else 'running'
               else:
                  status = 'inactive'
            else:
               # No lag status in egressLagMemberStatus; either not a lag or on a
               # platform where egress sFlow lags cannot be partially programmed.
               status = 'inactive' if intfName in disabledEgressIntfs else 'running'
            typeStatus = SflowTypeStatus( status=status )
            intfStatusMap.addSflowStatus( sflowType, typeStatus )
         if intfName in pollingIntfs:
            sflowType = 'counter'
            typeStatus = SflowTypeStatus( status='running' )
            intfStatusMap.addSflowStatus( sflowType, typeStatus )
      ret.interfaces[ intfName ] = intfStatusMap

   if detail:
      for lagIntf, memberStatus in egressLagMemberStatus.items():
         if lagIntf not in ret.interfaces:
            continue
         lagIntfSflows = ret.interfaces[ lagIntf ].sFlows

         # No need to populate egress lag status information if all lag members are
         # either inactive or running, since printing the lag members won't provide
         # any new information (all members will have the same status)
         egressStatus = lagIntfSflows.get( 'egress', None )
         if egressStatus != SflowTypeStatus( status='partial' ):
            continue

         for intf in memberStatus.activeIntfs + memberStatus.inactiveIntfs:
            lagMemberSflowTypeStatus = LagMemberSflowTypeStatus()

            # Populate mapping between lag member and lag interface name
            lagMemberSflowTypeStatus.lagInterface = lagIntf

            # Add egress sFlow status mapping for each lag member
            lagMemberStatusMap = SflowTypeStatusMap()
            status = 'running' if intf in memberStatus.activeIntfs else 'inactive'
            typeStatus = SflowTypeStatus( status=status )
            lagMemberStatusMap.addSflowStatus( 'egress', typeStatus )

            # Partially programmed lags only apply to egress sFlow; for ingress sFlow
            # and polling just use the same status as the lagIntf
            if 'ingress' in lagIntfSflows:
               lagMemberStatusMap.addSflowStatus( 'ingress',
                                                  lagIntfSflows[ 'ingress' ] )
            if 'counter' in lagIntfSflows:
               lagMemberStatusMap.addSflowStatus( 'counter',
                                                  lagIntfSflows[ 'counter' ] )
            lagMemberSflowTypeStatus.lagSflowTypeStatus = lagMemberStatusMap
            ret.lagInterfaces[ intf ] = lagMemberSflowTypeStatus
   return ret

# -------------------------------------------------------------------------------
# the "show sflow interfaces counters" command
# -------------------------------------------------------------------------------
def doShowSflowIntfCounters( mode, args ):
   ret = SflowInterfaceCounters()
   ret.enabled = sflowConfig.enabled
   samplingIntfs = set()
   egressSamplingIntfs = set()
   isSubIntfId = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId

   # Workaround for BUG173418 as done in doShowSflowIntfs
   if sflowConfig.samplingEnabled:
      samplingIntfs = { k for k, v in sflowHwConfig.runIntf.items()
                        if v == sampleAll }
      egressSamplingIntfs = set( sflowHwConfig.runEgressIntf )

   for intf in samplingIntfs | egressSamplingIntfs:
      if isSubIntfId( intf ):
         continue
      intfMap = IntfSampleCounterMap()
      if intf in samplingIntfs:
         intfMap.ingress = sflowStatus.ingressCount.get( intf, 0 )
      if intf in egressSamplingIntfs:
         intfMap.egress = sflowStatus.egressCount.get( intf, 0 )
      ret.interfaces[ intf ] = intfMap

   return ret

#################################################################################
#
# PLEASE KEEP 'show tech support' registration AT BOTTOM of this file, and INSERT
# new commands ABOVE here.
#
# Register interesting 'show platform' commands with 'show tech support'.
#
#################################################################################

def _showTechGuard():
   return AgentDirectory.agent( em.sysname(), 'Sflow' )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2024-03-18 15:22:48',
   cmds=[ 'show sflow detail',
          'show sflow interfaces detail', ],
   cmdsGuard=_showTechGuard )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global sflowConfig, sflowCounterConfig, sflowStatus, sflowHwConfig
   global dropExportSflowConfig, dropExportSflowStatus
   global sflowHwStatusDir
   global dscpConfig, veosStatus
   global sflowAccelCapabilities
   global sflowAccelRunnability
   global sflowAccelConfig
   global sflowAccelHwConfig
   global sflowAccelStatusDir
   global smashEntityManager
   global sflowAccelCountersTable
   global em

   sflowConfig = ConfigMount.mount(
      entityManager, 'sflow/config', 'Sflow::Config', 'w' )
   dropExportSflowConfig = LazyMount.mount(
      entityManager, 'sflow/dropExportConfig', 'Sflow::Config', 'r' )
   sflowCounterConfig = LazyMount.mount(
      entityManager, 'sflow/counterConfig', 'Sflow::CounterConfig', 'w' )
   sflowStatus = LazyMount.mount(
      entityManager, 'sflow/status', 'Sflow::Status', 'rO' )
   dropExportSflowStatus = LazyMount.mount(
      entityManager, 'sflow/dropExportStatus', 'Sflow::Status', 'rO' )
   sflowHwConfig = LazyMount.mount(
      entityManager, 'sflow/hwconfigdir/sflow', 'Sflow::HwConfig', 'rO' )
   sflowHwStatusDir = LazyMount.mount(
      entityManager, 'sflow/hwstatus', 'Tac::Dir', 'ri' )
   veosStatus = LazyMount.mount(
      entityManager, 'sflow/veosstatus', 'Sflow::HwStatus', 'rO' )
   dscpConfig = ConfigMount.mount( entityManager, 'mgmt/dscp/config',
                                   'Mgmt::Dscp::Config', 'w' )

   # SflowAccel
   sflowAccelCapabilities = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/capabilities',
      'Hardware::SflowAccel::Capabilities', 'r' )
   sflowAccelRunnability = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/runnability',
      'Hardware::SflowAccel::Runnability', 'r' )
   sflowAccelConfig = ConfigMount.mount(
      entityManager, 'hardware/sflowAccel/config',
      'Hardware::SflowAccel::Config', 'w' )
   sflowAccelHwConfig = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/hwConfig',
      'Hardware::SflowAccel::HwConfig', 'r' )
   sflowAccelStatusDir = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/status',
      'Tac::Dir', 'ri' )

   em = entityManager
   smashEntityManager = SharedMem.entityManager( sysdbEm=entityManager )
   counterMountInfo = Smash.mountInfo( 'reader' )
   sflowAccelCountersTable = smashEntityManager.doMount(
      "hardware/sflowAccel/counters",
      "Hardware::SflowAccel::Counters", counterMountInfo )
