#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from collections import defaultdict
from collections import OrderedDict
import Arnet
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from IntfModels import Interface
import SflowConst
import TableOutput
import Tac
from Toggles.SflowLibToggleLib import (
      toggleSflowMaxDatagramSizeEnabled,
      toggleSflowIndependentAddressConfigEnabled,
      toggleSflowL2SubIntfVlanEnabled, )

SampleTruncateSizeType = Tac.Type( "Sflow::SampleTruncateSizeType" )

_sendingReasonMap = {
   'noError' : '',
   'notRunning' : ' (Sflow not running)',
   'invalidSource' : ' (Invalid source IP address)',
   'invalidDestination' : ' (Invalid destination IP address)',
   'invalidSourceDestination' : ' (Invalid combination of IPv4 '
   'and IPv6 addresses)',
   'invalidVrf' : ' (Invalid VRF)',
   'bindingInProgress' : ' (Binding to source IP in progress)',
}

# Make sure the order of sFlow types goes Ingress,Egress,Counter or
# Flow,Egress,Counter to be consistent with legacy "show sflow interfaces" output
_sflowTypeStringMap = OrderedDict( {
   'flow' : 'Flow',
   'ingress' : 'Ingress',
   'egress' : 'Egress',
   'counter' : 'Counter',
} )

# Order in which sFlow statuses should appear in "show sflow interfaces" output
_statusValues = [ 'running', 'partial', 'inactive' ]

def _convertToYesNo( booleanToConvert ):
   return 'Yes' if booleanToConvert else 'No'

def _convertToYesNoSmallCase( booleanToConvert ):
   return 'yes' if booleanToConvert else 'no'

class SflowStatus( Model ):
   class Details( Model ):
      hardwareSamplingEnabled = Bool( help='Hardware sampling enabled' )
      samplesDiscarded = Int( help='Number of discarded samples' )
      sampleOutputInterface = Bool( help='Include output interface in '
                                    'packet sample' )
      sampleMplsExtension = Bool( help='Include MPLS extension header in '
                                  'packet sample', optional=True )
      sampleEvpnMplsExtension = Bool(
         help='Include MPLS VC extension header for EVPN packet sample',
         optional=True )
      sampleVplsExtension = Bool( help='Include MPLS VC extension header for VPLS '
                                    'packet sample', optional=True )
      sampleVxlanExtension = Bool( help='Include VXLAN extension headers in '
                                    'packet sample', optional=True )
      sampleSwitchExtension = Bool( help='Include switch extension header in '
                                    'packet sample', optional=True )
      sampleRouterExtension = Bool( help='Include router extension header in '
                                    'packet sample', optional=True )
      sampleTunnelIpv4EgrExtension = Bool( help='Include tunnel IPv4 egress '
                                           'extension header in packet sample',
                                           optional=True )
      sampleVxlanHeaderStrip = Bool( help='Include VXLAN tunnel headers of VXLAN '
                                     'encapsulated packets in packet sample',
                                     optional=True )
      sampleIngressSubintf = Bool( help='Include input subinterface information '
                                       'in packet sample; sample sent '
                                       'in expanded format', optional=True )
      sampleEgressSubintf = Bool( help='Include output subinterface information '
                                       'in packet sample; sample sent '
                                       'in expanded format', optional=True )
      sampleUpdateSubIntfVlan = Bool( help='Report VLAN ID changes on L2 '
                                       'subinterfaces', optional=True )
      portChannelOutputIfIndex = Enum( help='Output interface index for '
                                       'port channels',
                                       values=( 'member', 'portchannel' ),
                                       optional=True )
      sviIfindexOutput = Bool( help='Output interface index of SVI reported in '
                                    'sample', optional=True )
      sviIfindexInput = Bool( help='Input interface index of SVI reported in '
                                    'sample', optional=True )
      accelUnsupportedExtensions = List( help='Extensions that are not supported for'
                                         ' hardware acceleration',
                                         valueType=str, optional=True )
      sampleEncodingFormat = Enum( help='Encoding format for samples',
                                   values=( 'compact', 'expanded' ),
                                   optional=True )

   class Destination( Model ):
      port = Int( help='Port of sFlow collector' )
      hostname = Str( help='Hostname of sFlow collector' )
      datagramsCount = Int( help='Number of datagrams sent to sFlow collector' )

   class Ipv4Destination( Destination ):
      ipv4Address = Ip4Address( help='IPv4 address of sFlow collector' )
      vrfName = Str( help='Associated VRF for sFlow collector' )

      def render( self ):
         if self.hostname == str( self.ipv4Address ):
            print( '  %s:%s (VRF: %s)' % ( self.ipv4Address, self.port,
                                           self.vrfName ) )
         else:
            print( '  %s:%s (%s) (VRF: %s)' % ( self.ipv4Address, self.port,
                                                self.hostname, self.vrfName ) )

   class Ipv6Destination( Destination ):
      ipv6Address = Ip6Address( help='IPv6 address of sFlow collector' )
      vrfName = Str( help='Associated VRF for sFlow collector' )

      def render( self ):
         if self.hostname == str( self.ipv6Address ):
            print( '  [%s]:%s (VRF: %s)' % ( self.ipv6Address, self.port,
                                             self.vrfName ) )
         else:
            print( '  [%s]:%s (%s) (VRF: %s)' % ( self.ipv6Address, self.port,
                                                  self.hostname, self.vrfName ) )

   class Ipv4Source( Model ):
      ipv4Address = Ip4Address( help='Source IPv4 address' )
      sourceInterface = Interface( help='Source Interface', optional=True )
      vrfName = Str( help='Associated VRF for sFlow source' )

      def render( self ):
         finalIpAddr = SflowConst.valueWithDefaultInd( str( self.ipv4Address ),
                                                       SflowConst.defaultIp )
         if self.sourceInterface:
            interfaceName = str( self.sourceInterface ).replace( '\'', '' )
            print( '  %s from %s (VRF: %s)' % ( finalIpAddr, interfaceName,
                                                self.vrfName ) )
         else:
            print( '  %s (VRF: %s)' % ( finalIpAddr, self.vrfName ) )

   class Ipv6Source( Model ):
      ipv6Address = Ip6Address( help='Source IPv6 address' )
      sourceInterface = Interface( help='Source Interface', optional=True )
      vrfName = Str( help='Associated VRF for sFlow source' )

      def render( self ):
         finalIpAddr = SflowConst.valueWithDefaultInd( str( self.ipv6Address ),
                                                       SflowConst.defaultIp6 )
         if self.sourceInterface:
            interfaceName = str( self.sourceInterface ).replace( '\'', '' )
            print( '  %s from %s (VRF: %s)' % ( finalIpAddr, interfaceName,
                                                self.vrfName ) )
         else:
            print( '  %s (VRF: %s)' % ( finalIpAddr, self.vrfName ) )

   class SendingDatagram( Model ):
      sending = Bool( help='sFlow samples being sent' )
      reason = Enum( values=list( _sendingReasonMap ),
                     help='Reason for sendingDatagrams state' )
      vrfName = Str( help='VRF name' )

      def _getSendingReason( self, sendingDatagram, sendingDatagramReason ):
         if sendingDatagram:
            return ''
         else:
            return _sendingReasonMap[ sendingDatagramReason ]

      def render( self ):
         print( '  %s%s (VRF: %s)' % ( _convertToYesNo( self.sending ),
                                       self._getSendingReason( self.sending,
                                                               self.reason ),
                                       self.vrfName ) )

   class BgpExport( Model ):
      export = Bool( help='Export BGP information' )
      vrfName = Str( help='VRF name' )

      def render( self ):
         print( '  %s (VRF: %s)' % ( _convertToYesNo( self.export ),
      self.vrfName ) )

   # Configuration
   enabled = Bool( help='sFlow globally enabled' )
   dropPacketSamplingEnabled = Bool( help='sFlow export enabled for dropped '
                                     'packets', optional=True )
   ipv4Destinations = List( help='List of IPv4 destinations',
                            valueType=Ipv4Destination )
   ipv6Destinations = List( help='List of IPv6 destinations',
                            valueType=Ipv6Destination )
   ipv4DestinationsForDropSamples = List( help='List of IPv4 destinations for drop '
                                          'samples', valueType=Ipv4Destination,
                                          optional=True )
   ipv6DestinationsForDropSamples = List( help='List of IPv6 destinations for drop '
                                          'samples', valueType=Ipv6Destination,
                                          optional=True )
   ipv4Sources = List( help='List of Source IPv4 addresses', valueType=Ipv4Source )
   ipv6Sources = List( help='List of Source IPv6 addresses', valueType=Ipv6Source )
   ipv4SourcesForDropSamples = List( help='List of Source IPv4 addresses for '
                                     'drop samples', valueType=Ipv4Source,
                                     optional=True )
   ipv6SourcesForDropSamples = List( help='List of Source IPv6 addresses for '
                                     'drop samples', valueType=Ipv6Source,
                                     optional=True )
   ipv4Agents = List( help='List of Agent IPv4 addresses', valueType=Ipv4Source )
   ipv6Agents = List( help='List of Agent IPv6 addresses', valueType=Ipv6Source )

   dynamicUdpSourcePort = Bool( help='Use dynamic UDP source port for sFlow entropy',
                                optional=True )
   sampleRate = Int( help='Sample rate, 1 sample per N packets' )
   sampleTruncateSize = Int( help='Sample truncate size limit in bytes' )
   if toggleSflowMaxDatagramSizeEnabled():
      maxDatagramSize = Int( help='Max datagram size limit in bytes' )
   hwSampleTruncateSize = Int( help='HW Sample truncate size limit in bytes',
                               optional=True )
   accelSampleRate = Int( help='Sample rate for hardware acceleration, 1 sample per'
                          ' N packets', optional=True )
   pollingInterval = Float( help='Polling interval in seconds' )
   rewriteDscp = Bool( help='Rewrite DSCP value in sFlow samples',
                              optional=True )
   dscpValue = Int( help='DSCP value in sFlow datagram' )

   # Status
   running = Bool( help='sFlow sampling is running' )
   dropPacketExportRunning = Bool( help='sFlow export is running for drop '
                                   'samples', optional=True )
   accelEnabled = Bool( help='sFlow hardware acceleration running', optional=True )
   polling = Bool( help='sFlow polling' )
   samplingEnabled = Bool( help='sFlow sampling enabled' )
   bgpExports = List( help='List of BGP Exports',
                      valueType=BgpExport, optional=True )
   sendingDatagrams = List( help='List of Sending datagrams',
                            valueType=SendingDatagram )
   sendingDatagramsForDropSamples = List( help='List of sending datagrams for '
                                          'drop samples', valueType=SendingDatagram,
                                          optional=True )
   hardwareSampleRate = Int( help='Hardware sample rate for software Sflow, 1 sample'
                             ' per N packets' )
   hardwareAccelSampleRate = Int( help='Hardware sample rate for hardware'
                                  ' accelerated Sflow, 1 sample per N packets',
                                  optional=True )

   accelSupported = Bool( help='Is hardware acceleration supported', optional=True,
                          default=False )

   # Statistics
   # The following counters contain SflowAccel counters too.
   totalPackets = Int( help='Total packets' )
   softwareSamples = Int( help='Number of samples' )
   samplePool = Int( help='Sample pool' )
   hardwareSamples = Int( help='Number of hardware samples' )
   datagrams = Int( help='Number of datagrams sent for regular sFlow samples' )
   dropSampleDatagrams = Int( help='Number of datagrams sent for drop samples',
                              optional=True )
   dropSamplesReceived = Int( help='Number of drop samples received at '
                                 'sFlow Agent', optional=True )
   dropSamplesSent = Int( help='Number of drop samples sent to external collectors '
                          'from sFlow Agent', optional=True )

   # Details
   details = Submodel( valueType=Details, help='Detailed sFlow information',
                       optional=True )

   def render( self ):
      print( 'sFlow Configuration' )
      print( '-------------------' )
      print( 'Enabled: %s' % _convertToYesNoSmallCase( self.enabled ) )
      self._renderDestinations( self.ipv4Destinations, self.ipv6Destinations,
                                SflowConst.sflowDestDefaultMsg,
                                SflowConst.sflowDestMsg )
      self._renderSrcIpAddrs( self.ipv4Sources, self.ipv6Sources,
                              SflowConst.sflowSourceDefaultMsg,
                              SflowConst.sflowSourceMsg )
      # Render agent ID addresses
      if toggleSflowIndependentAddressConfigEnabled():
         self._renderSrcIpAddrs( self.ipv4Agents, self.ipv6Agents,
                                 SflowConst.sflowAgentDefaultMsg,
                                 SflowConst.sflowAgentMsg )

      if self.dynamicUdpSourcePort:
         print( 'UDP source port for HW sFlow: dynamic' )

      print( 'Hardware sample rate for SW sFlow: %s' % self._getSampleRate() )
      if self.accelSupported:
         print( 'Hardware sample rate for HW sFlow: {}'.format(
            self._getAccelSampleRate() ) )
      print( 'Polling interval (sec): %s' % self._getPollingInterval() )
      print( 'Rewrite DSCP value: %s' %
             _convertToYesNoSmallCase( self._isRewriteDscpOn() ) )
      print( 'Datagram DSCP value: %s' % self._dscpValue() )

      print( '' )

      print( 'Configuration For Drop Packet Samples' )
      print( '-------------------------------------' )
      print( 'Enabled: %s' %
             _convertToYesNoSmallCase( self.dropPacketSamplingEnabled ) )
      self._renderDestinations( self.ipv4DestinationsForDropSamples,
                                self.ipv6DestinationsForDropSamples,
                                SflowConst.sflowDestDefaultMsg,
                                SflowConst.sflowDestMsg )
      self._renderSrcIpAddrs( self.ipv4SourcesForDropSamples,
                              self.ipv6SourcesForDropSamples,
                              SflowConst.sflowSourceDefaultMsg,
                              SflowConst.sflowSourceMsg )
      print( '' )

      print( 'Status' )
      print( '------' )
      print( 'Running: %s' % _convertToYesNoSmallCase( self.enabled ) )
      if self.accelSupported:
         print( 'Hardware acceleration running: {}'.format(
            _convertToYesNoSmallCase( self.accelEnabled ) ) )
      print( 'Polling on: %s' % ( 'yes (default)' if self.polling else 'no' ) )
      print( 'Sampling on: %s' % ( 'yes (default)' if self._isSamplingOn()
                                   else 'no' ) )
      print( 'Sample truncation size for SW sFlow: %s' %
             ( self._getSampleTruncateSize() ) )
      if self.accelSupported and self.accelEnabled:
         print( 'Sample truncation size for HW sFlow: %s' %
                ( self._getHwSampleTruncateSize() ) )
      if toggleSflowMaxDatagramSizeEnabled():
         print( 'Datagram maximum size: %s' %
               ( self._getMaxDatagramSize() ) )
      self._renderSendingDatagrams()
      self._renderBgpExports()
      print( 'Hardware sample rate for SW sFlow: %s' %
             self._getHardwareSampleRate() )
      if self.accelSupported and self.accelEnabled:
         print( 'Hardware sample rate for HW sFlow: {}'.format(
            self._getHardwareAccelSampleRate() ) )
      if self.details:
         print( 'Hardware sampling on: %s' % _convertToYesNoSmallCase(
            self.details.hardwareSamplingEnabled ) )
         sampleOutputText = _convertToYesNoSmallCase(
            self.details.sampleOutputInterface )
         print( "Sample output interface: %s" % sampleOutputText )
         sampleMplsExtensionText = _convertToYesNoSmallCase(
            self.details.sampleMplsExtension )
         print( "Sample MPLS extension: %s" % sampleMplsExtensionText )
         sampleEvpnMplsExtensionText = _convertToYesNoSmallCase(
            self.details.sampleEvpnMplsExtension )
         print( "Sample MPLS VC extension for EVPN traffic: "
                f"{sampleEvpnMplsExtensionText}" )
         sampleVplsExtensionText = _convertToYesNoSmallCase(
            self.details.sampleVplsExtension )
         print( "Sample MPLS VC extension for VPLS traffic: "
                f"{sampleVplsExtensionText}" )
         sampleVxlanExtensionText = _convertToYesNoSmallCase(
            self.details.sampleVxlanExtension )
         print( "Sample VXLAN extension: "
                f"{sampleVxlanExtensionText}" )
         sampleSwitchExtensionText = _convertToYesNoSmallCase(
            self.details.sampleSwitchExtension )
         print( "Sample switch extension: %s" % sampleSwitchExtensionText )
         sampleRouterExtensionText = _convertToYesNoSmallCase(
            self.details.sampleRouterExtension )
         print( "Sample router extension: %s" % sampleRouterExtensionText )
         sampleTunnelIpv4EgrExtensionText = _convertToYesNoSmallCase(
            self.details.sampleTunnelIpv4EgrExtension )
         print( "Sample tunnel IPv4 egress extension: %s" %
                sampleTunnelIpv4EgrExtensionText )
         sampleIngressSubintfText = _convertToYesNoSmallCase(
            self.details.sampleIngressSubintf )
         print( "Sample input subinterface: %s" %
                sampleIngressSubintfText )
         sampleEgressSubintfText = _convertToYesNoSmallCase(
            self.details.sampleEgressSubintf )
         print( "Sample output subinterface: %s" %
                sampleEgressSubintfText )
         if toggleSflowL2SubIntfVlanEnabled():
            sampleUpdateSubIntfVlanText = _convertToYesNoSmallCase(
               self.details.sampleUpdateSubIntfVlan )
            print( "Sample output VLAN: %s" %
                  sampleUpdateSubIntfVlanText )
         sampleVxlanHeaderStripText = _convertToYesNoSmallCase(
             self.details.sampleVxlanHeaderStrip )
         print( "VXLAN header strip: %s" % sampleVxlanHeaderStripText )
         portchOutputIfIndexText = self.details.portChannelOutputIfIndex
         print( "Port channel output interface index: %s" %
                portchOutputIfIndexText )
         sviInputIfindex = self.details.sviIfindexInput
         sviOutputIfindex = self.details.sviIfindexOutput
         self._renderSviIfindex( sviInputIfindex, sviOutputIfindex )

         if self.accelSupported and self.accelEnabled:
            if self.details.accelUnsupportedExtensions:
               print( 'Hardware acceleration unsupported features:' )
            for unsupportedFeature in self.details.accelUnsupportedExtensions:
               print( f'  {unsupportedFeature}' )
         sampleEncodingFormatText = self.details.sampleEncodingFormat
         print( "Sample encoding format: %s" % sampleEncodingFormatText )
      print( '' )

      print( 'Status For Drop Packet Samples' )
      print( '------------------------------' )
      print( 'Running: %s' % _convertToYesNo( self.dropPacketExportRunning ) )
      if self.dropPacketExportRunning:
         self._renderSendingDatagramsForDropSamples()
      print( '' )

      print( 'Statistics' )
      print( '----------' )
      print( 'Total packets: %d' % self.totalPackets )
      print( 'Number of samples: %d' % self.softwareSamples )
      print( 'Sample pool: %d' % self.samplePool )
      print( 'Hardware trigger: %d' % self.hardwareSamples )
      print( 'Number of datagrams: %d' % self.datagrams )
      if self.details:
         print( 'Number of samples discarded: %d' % self.details.samplesDiscarded )

      # === dropSamplingToggleEnabled ===
      # This section pertains specifically to the 'dropSamplingToggleEnabled' feature
      # Please refrain from adding any unrelated output beyond this point
      print( '' )
      print( 'Drop Packet Sample Counters:' )
      print( '  Datagrams sent: %d' % self.dropSampleDatagrams )
      print( '  Samples received: %d' % self.dropSamplesReceived )
      print( '  Samples sent: %d' % self.dropSamplesSent )

      if ( self.ipv4Destinations or self.ipv6Destinations or
           self.ipv4DestinationsForDropSamples or
           self.ipv6DestinationsForDropSamples ):
         print( '' )
         print( 'Collector Counters' )
         print( '------------------' )
         print( '' )
         tableHeadings = ( 'VRF', 'Collector', 'Datagrams' )
         if self.ipv4Destinations or self.ipv6Destinations:
            self._renderPerCollectorDatagramCounters( tableHeadings,
                                                      self.ipv4Destinations,
                                                      self.ipv6Destinations )
         if ( self.ipv4DestinationsForDropSamples or
              self.ipv6DestinationsForDropSamples ):
            self._renderPerCollectorDatagramCounters(
                                             tableHeadings,
                                             self.ipv4DestinationsForDropSamples,
                                             self.ipv6DestinationsForDropSamples,
                                             dropSamples=True )

   def _getPollingInterval( self ):
      return SflowConst.valueWithDefaultInd( self.pollingInterval,
                                             SflowConst.defaultPollingInterval )

   def _isSamplingOn( self ):
      return self.samplingEnabled and self.enabled

   def _isRewriteDscpOn( self ):
      return self.rewriteDscp

   def _dscpValue( self ):
      return self.dscpValue

   def _getSampleRate( self ):
      if not self.samplingEnabled:
         return 'none'
      else:
         return SflowConst.valueWithDefaultInd( self.sampleRate,
                                                SflowConst.defaultSampleRate )

   def _getSampleTruncateSize( self ):
      return SflowConst.valueWithDefaultInd( self.sampleTruncateSize,
                                             SampleTruncateSizeType.min )

   def _getMaxDatagramSize( self ):
      return SflowConst.valueWithDefaultInd( self.maxDatagramSize,
                                             SflowConst.defaultMaxDatagramSize )

   def _getHwSampleTruncateSize( self ):
      return self.hwSampleTruncateSize

   def _getAccelSampleRate( self ):
      if not self.samplingEnabled:
         return 'none'
      else:
         return SflowConst.valueWithDefaultInd( self.accelSampleRate,
                                                SflowConst.defaultSampleRate )

   def _getHardwareSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.hardwareSampleRate,
                                                SflowConst.defaultSampleRate )

   def _getHardwareAccelSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.hardwareAccelSampleRate,
                                                SflowConst.defaultSampleRate )

   def _renderSrcIpAddrs( self, ipv4Sources, ipv6Sources,
                          defaultHeaderMsg, headerMsg ):
      if not ipv4Sources and not ipv6Sources:
         print( defaultHeaderMsg )
      else:
         print( headerMsg )

         allIpv4Sources = {}
         for source in ipv4Sources:
            allIpv4Sources[ source.vrfName ] = source

         allIpv6Sources = {}
         for source in ipv6Sources:
            allIpv6Sources[ source.vrfName ] = source

         for vrfName in sorted( set( list( allIpv4Sources ) +
                                     list( allIpv6Sources ) ) ):
            if vrfName in allIpv4Sources:
               allIpv4Sources[ vrfName ].render()
            if vrfName in allIpv6Sources:
               allIpv6Sources[ vrfName ].render()

   def _getDestinationsSortedByVrf( self, ipv4Destinations, ipv6Destinations ):
      allIpv4Destinations = {}
      for dest in ipv4Destinations:
         allIpv4Destinations.setdefault( dest.vrfName, [] ).append( dest )

      allIpv6Destinations = {}
      for dest in ipv6Destinations:
         allIpv6Destinations.setdefault( dest.vrfName, [] ).append( dest )

      return allIpv4Destinations, allIpv6Destinations

   def _renderDestinations( self, ipv4Destinations, ipv6Destinations,
                            defaultHeaderMsg, headerMsg ):
      if not ipv4Destinations and not ipv6Destinations:
         print( defaultHeaderMsg )
      else:
         print( headerMsg )

      allIpv4Destinations, allIpv6Destinations = self._getDestinationsSortedByVrf(
                                                 ipv4Destinations, ipv6Destinations )
      for vrfName in sorted( set( list( allIpv4Destinations ) +
                                  list( allIpv6Destinations ) ) ):
         if vrfName in allIpv4Destinations:
            for dest in allIpv4Destinations[ vrfName ]:
               dest.render()
         if vrfName in allIpv6Destinations:
            for dest in allIpv6Destinations[ vrfName ]:
               dest.render()

   def _renderSendingDatagrams( self ):
      print( 'Send datagrams:' )
      for send in self.sendingDatagrams:
         send.render()

   def _renderSendingDatagramsForDropSamples( self ):
      print( 'Send datagrams:' )
      for send in self.sendingDatagramsForDropSamples:
         send.render()

   def _renderBgpExports( self ):
      print( 'BGP export:' )
      for export in self.bgpExports:
         export.render()

   def _renderPerCollectorDatagramCounters( self, tableHeadings, ipv4Destinations,
                                            ipv6Destinations, dropSamples=False ):
      if dropSamples:
         print( 'Drop Samples' )
      else:
         print( 'Flow and Counter Samples' )
      # Adding a new line before the table is displayed to make it more readable
      print( '' )
      table = TableOutput.createTable( tableHeadings )
      left = TableOutput.Format( justify='left' )
      right = TableOutput.Format( justify='right' )
      for fmt in [ left, right ]:
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )
      table.formatColumns( left, left, right )

      allIpv4Destinations, allIpv6Destinations = self._getDestinationsSortedByVrf(
                                                      ipv4Destinations,
                                                      ipv6Destinations )
      for vrfName in sorted( set( list( allIpv4Destinations ) +
                                  list( allIpv6Destinations ) ) ):
         if vrfName in allIpv4Destinations:
            for dest in allIpv4Destinations[ vrfName ]:
               args = [ vrfName ]
               if dest.hostname != str( dest.ipv4Address ):
                  args.append( f"{dest.ipv4Address} ({dest.hostname})" )
               else:
                  args.append( dest.ipv4Address )
               args.append( str( dest.datagramsCount ) )
               table.newRow( *args )
         if vrfName in allIpv6Destinations:
            for dest in allIpv6Destinations[ vrfName ]:
               args = [ vrfName ]
               if dest.hostname != str( dest.ipv6Address ):
                  args.append( f"{dest.ipv6Address} ({dest.hostname})" )
               else:
                  args.append( dest.ipv6Address )
               args.append( str( dest.datagramsCount ) )
               table.newRow( *args )
      print( table.output() )

   def _renderSviIfindex( self, sviInputMode, sviOutputMode ):
      inputMode = "svi" if sviInputMode else "member"
      outputMode = "svi" if sviOutputMode else "member"
      print( "SVI input interface index: %s" % inputMode )
      print( "SVI output interface index: %s" % outputMode )

class SflowTypeStatus( Model ):
   status = Enum( values=_statusValues,
                  help='sFlow status for an interface' )

class SflowTypeStatusMap( Model ):
   sFlows = Dict( keyType=str, valueType=SflowTypeStatus,
                  help='A mapping of sFlow type to its status' )
   intfOperStatusReason = Str( help='Operational status reason', optional=True )

   def addSflowStatus( self, sflowType, status ):
      assert sflowType in _sflowTypeStringMap
      self.sFlows[ sflowType ] = status

class LagMemberSflowTypeStatus( Model ):
   lagInterface = Interface( help='LAG interface for LAG member' )
   lagSflowTypeStatus = Submodel( valueType=SflowTypeStatusMap,
                                  help='sFlow type status map for LAG member' )

class SflowInterfaces( Model ):
   __revision__ = 3
   enabled = Bool( help='sFlow globally enabled' )
   ingressInterfacesEnabledDefault = Bool(
                           help='Global ingress sFlow configuration for interfaces' )
   egressInterfacesEnabledDefault = Bool(
                           help='Global egress sFlow configuration for interfaces' )
   interfaces = Dict( keyType=Interface, valueType=SflowTypeStatusMap,
                      help='Mapping between an interface and its status' )
   _detail = Bool( help='Detailed sFlow interfaces information' )
   lagInterfaces = Dict( keyType=Interface, valueType=LagMemberSflowTypeStatus,
                         help='Mapping between a LAG member and its status' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'interfacesEnabledDefault' ] = \
               dictRepr[ 'ingressInterfacesEnabledDefault' ]
         del dictRepr[ 'ingressInterfacesEnabledDefault' ]
         del dictRepr[ 'egressInterfacesEnabledDefault' ]
      interfaces = {}
      for intf in dictRepr[ 'interfaces' ]:
         interfaces[ intf ] = self._typeStatusStr( self.interfaces[ intf ] )
      dictRepr[ 'interfaces' ] = interfaces
      return dictRepr

   def _typeStatusStr( self, statusList ):
      if statusList.intfOperStatusReason:
         return statusList.intfOperStatusReason
      statusToSflowType = defaultdict( list )
      for sflowType, status in statusList.sFlows.items():
         statusToSflowType[ status.status ].append( sflowType )

      def sflowTypeStr( sflowTypes ):
         return ','.join( _sflowTypeStringMap[ sflowType ]
                          for sflowType in _sflowTypeStringMap
                          if sflowType in sflowTypes )

      statusStr = ', '.join( status + '(' +
                             sflowTypeStr( statusToSflowType[ status ] ) + ')'
                             for status in _statusValues
                             if status in statusToSflowType and
                             statusToSflowType[ status ] )

      return statusStr

   def _renderDetailTable( self ):

      def statusStr( typeStatusMap, sflowType ):
         if sflowType in typeStatusMap:
            return typeStatusMap[ sflowType ].status
         else:
            return 'unconfigured'

      def addTableRow( table, intf, typeStatus, member='' ):
         table.newRow( intf,
                       member,
                       statusStr( typeStatus, 'ingress' ),
                       statusStr( typeStatus, 'egress' ),
                       statusStr( typeStatus, 'counter' ) )

      tableHeadings = ( 'Interface', 'Member', 'Ingress', 'Egress', 'Counter' )
      table = TableOutput.createTable( tableHeadings )
      fmt = TableOutput.Format( justify='left' )
      table.formatColumns( *( fmt for _ in tableHeadings ) )
      if not self.enabled:
         print( table.output() )
         return
      for intf in Arnet.sortIntf( self.interfaces ):
         typeStatus = self.interfaces[ intf ].sFlows
         addTableRow( table, intf, typeStatus )
         for member in Arnet.sortIntf( self.lagInterfaces ):
            lagIntfName = str( self.lagInterfaces[ member ].lagInterface ).replace(
               '\'', '' )
            if lagIntfName == intf:
               lagTypeStatus = self.lagInterfaces[ member ].lagSflowTypeStatus.sFlows
               addTableRow( table, intf, lagTypeStatus, member )
      print( table.output() )

   def render( self ):
      if self._detail:
         self._renderDetailTable()
         return

      ingressIntfConfigStr = "Default ingress sFlow configuration for an interface: "
      if self.ingressInterfacesEnabledDefault:
         print( ingressIntfConfigStr + 'Enabled' )
      else:
         print( ingressIntfConfigStr + 'Disabled' )

      egressIntfConfigStr = "Default egress sFlow configuration for an interface: "
      if self.egressInterfacesEnabledDefault:
         print( egressIntfConfigStr + 'Enabled' )
      else:
         print( egressIntfConfigStr + 'Disabled' )
      print( "sFlow Interface (s):" )
      print( "--------------------" )
      if not self.enabled:
         print( "sFlow is not running" )
      elif len( self.interfaces ) == 0:
         print( "No interfaces have been configured for sFlow" )
      else:
         for interface in Arnet.sortIntf( self.interfaces ):
            print( "%s - %s" % ( interface,
                                 self._typeStatusStr(
                                    self.interfaces[ interface ] ) ) )

class IntfSampleCounterMap( Model ):
   ingress = Int( help='Interface ingress sample count', optional=True )
   egress = Int( help='Interface egress sample count', optional=True )

class SflowInterfaceCounters( Model ):
   enabled = Bool( help='sFlow globally enabled' )
   interfaces = Dict( keyType=Interface, valueType=IntfSampleCounterMap,
                      help='Mapping between an interface and its sample counters' )

   def render( self ):
      tableHeadings = [ 'Interface' ]
      renderIngressCol = renderEgressCol = False
      for intf in self.interfaces:
         if self.interfaces[ intf ].ingress is not None:
            renderIngressCol = True
         if self.interfaces[ intf ].egress is not None:
            renderEgressCol = True
      if renderIngressCol:
         tableHeadings.append( 'Ingress Samples' )
      if renderEgressCol:
         tableHeadings.append( 'Egress Samples' )
      table = TableOutput.createTable( tuple( tableHeadings ) )
      for intf in Arnet.sortIntf( self.interfaces ):
         args = [ intf ]
         if renderIngressCol:
            if self.interfaces[ intf ].ingress is not None:
               args.append( self.interfaces[ intf ].ingress )
            else:
               args.append( 'n/a' )
         if renderEgressCol:
            if self.interfaces[ intf ].egress is not None:
               args.append( self.interfaces[ intf ].egress )
            else:
               args.append( 'n/a' )
         table.newRow( *args )
      print( table.output() )
      if not self.enabled:
         print( "sFlow is not running" )
      elif len( self.interfaces ) == 0:
         print( "No interfaces have been configured for sFlow" )
