# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for SampledFlowTracker configuration commands

import BasicCli
import Cell
import ConfigMount
import CliCommand
import CliParser
from CliPlugin import EthIntfCli, LagIntfCli, VirtualIntfRule
from CliPlugin.FlowTrackingCliLib import (
      flowTableKw,
      sampledClearKw,
      trackerKw,
      trackerNameMatcher,
      trackingClearKw,
)
from CliPlugin.SftCliLib import (
      sampledHardwareKw,
      tokenDstIp,
      tokenDstPort,
      tokenInterface,
      tokenIPGen,
      tokenPort,
      tokenProtocol,
      tokenProtocolValue,
      tokenSrcIp,
      tokenSrcPort,
      tokenVlan,
      tokenVlanValue,
      tokenVrf,
      tokenVrfValue,
      tokenState,
      tokenStateValue,
)
from CliToken.Clear import clearKwNode
from CliToken.Flow import flowMatcherForClear
import LazyMount
import SmashLazyMount
from SftCliUtil import HoFlowStateEnum
import Tac
from TypeFuture import TacLazyType

ClearHoFlowConfigReq = TacLazyType( 'SampledFlowTracker::ClearHoFlowConfigReq' )
IpProtoType = Tac.Type( 'Arnet::IpProtocolNumber' )
HoFlowState = TacLazyType( 'FlowTracking::HoFlowState' )
VlanId = TacLazyType( 'Bridging::VlanId' )
VrfId = Tac.Value( 'Vrf::VrfIdMap::VrfId' )
sftConfigReq = None
arpVrfIdMap = None
vrfNameStatus = None

def protoEnumValue( protoStr ):
   if not protoStr.startswith( 'ipProto' ):
      protoStr = 'ipProto' + protoStr
   for p in IpProtoType.attributes:
      if p.lower() == protoStr.lower():
         return Tac.enumValue( IpProtoType, p )
   return Tac.enumValue( IpProtoType, 'ipProtoUnknown' )

hoFlowStateEnumValue = {
   'unknown' : HoFlowStateEnum.hoFlowStateUnknown,
   'active' : HoFlowStateEnum.inHardware,
   'in-progress' : HoFlowStateEnum.inProgress,
   'hw-limit' : HoFlowStateEnum.outOfResources,
}

#-------------------------------------------------------------------------------
# clear flow tracking sampled flow-table hardware
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ tracker <tracker-name> ] [ interface <interface-range> ]
#-------------------------------------------------------------------------------

def clearHoFlow( mode, args ):
   tracker = args.get( 'TRACKER_NAME' )

   srcIpAddr = args.get( 'SRC_IP' )
   dstIpAddr = args.get( 'DST_IP' )

   if ( srcIpAddr and dstIpAddr and srcIpAddr.af != dstIpAddr.af ):
      # Both a v4 and v6 address were specified
      raise CliParser.InvalidInputError()

   srcPort = args.get( 'SRC_PORT' )
   dstPort = args.get( 'DST_PORT' )
   protocol = args.get( 'PROTOCOL' )
   vrfName = args.get( 'VRF' )
   vlanId = args.get( 'VLAN' )
   state = args.get( 'STATE' )
   intf = args.get( 'INTERFACE' )

   clearHoConfig = Tac.Value( "SampledFlowTracker::ClearHoFlowConfigReq" )

   if tracker is not None:
      clearHoConfig.trackerName = tracker
   if srcIpAddr is not None:
      clearHoConfig.srcAddr = srcIpAddr
   if dstIpAddr is not None:
      clearHoConfig.dstAddr = dstIpAddr
   if srcPort is not None:
      clearHoConfig.srcPort = srcPort
   if dstPort is not None:
      clearHoConfig.dstPort = dstPort
   if protocol is not None:
      clearHoConfig.ipProtocol = protoEnumValue( protocol )
   if vrfName is not None:
      clearHoConfig.vrfId = vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName,
         VrfId.null )
      if clearHoConfig.vrfId == VrfId.null:
         raise CliParser.InvalidInputError( 'VRF %s not found' % vrfName )
   if state is not None:
      clearHoConfig.state = hoFlowStateEnumValue[ state ]
   if intf is not None:
      clearHoConfig.intfId = intf.status().intfId
   clearHoConfig.vlanId = vlanId if vlanId is not None else VlanId.invalid

   clearHoConfig.reqTime = Tac.now()
   sftConfigReq.clearHoConfig = clearHoConfig

intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfRule = CliCommand.Node( intfMatcher, maxMatches=1 )

class ClearHoTableFilter( CliCommand.CliExpression ):
   expression = ( '''[ { ( tracker TRACKER_NAME )
                       | ( vlan VLAN )
                       | ( vrf VRF )
                       | ( src-ip SRC_IP )
                       | ( dst-ip DST_IP )
                       | ( src-port SRC_PORT )
                       | ( dst-port DST_PORT )
                       | ( protocol PROTOCOL )
                       | ( state STATE )
                       | ( interface INTERFACE ) } ]''' )
   data = {
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IP' : tokenIPGen,
      'dst-ip' : tokenDstIp,
      'DST_IP' : tokenIPGen,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'state' : tokenState,
      'STATE' : tokenStateValue,
      'interface' : tokenInterface,
      'INTERFACE' : intfRule
   }

class SftClearHocCmd( CliCommand.CliCommandClass ):
   syntax = 'clear flow tracking sampled flow-table hardware [ FILTER ]'
   data = {
      'clear' : clearKwNode,
      'flow' : flowMatcherForClear,
      'tracking' : trackingClearKw,
      'sampled' : sampledClearKw,
      'flow-table' : flowTableKw,
      'hardware' : sampledHardwareKw,
      'FILTER' : ClearHoTableFilter
   }
   handler = clearHoFlow

BasicCli.EnableMode.addCommandClass( SftClearHocCmd )

def clearSampledDebugCounter( mode, args ):
   clearCounters = Tac.Value( "SampledFlowTracker::ClearCountersDebug" )
   clearCounters.reqTime = Tac.now()
   sftConfigReq.clearCountersDebug = clearCounters

class SftClearSampledDebugCounter( CliCommand.CliCommandClass ):
   syntax = 'clear flow tracking sampled counters debug'
   data = {
      'clear' : clearKwNode,
      'flow' : flowMatcherForClear,
      'tracking' : trackingClearKw,
      'sampled' : sampledClearKw,
      'counters' : 'Clear flow tracking counters',
      'debug' : 'Clear debug counters',
   }
   handler = clearSampledDebugCounter

BasicCli.EnableMode.addCommandClass( SftClearSampledDebugCounter )

#--------------------------
def Plugin( em ):
   global sftConfigReq
   global arpVrfIdMap
   global vrfNameStatus

   sftConfigReq = ConfigMount.mount( em, "flowtracking/sampled/configReq",
                                     "SampledFlowTracker::ConfigReq", 'w' )
   arpVrfIdMap = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
   vrfNameStatus = LazyMount.mount( em, Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
