# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple

from Ark import switchTimeToUtc
from Arnet import IpGenAddr
from BasicCli import addShowCommandClass
import CliCommand
import CliMatcher
from CliModel import cliPrinted
from CliPlugin.FlowTrackingCliLib import (
   IP6_GROUP,
   EGRESS_IP6_GROUP,
   IP_GROUP,
   EGRESS_IP_GROUP,
   FtConsts,
   allGroupNameMatcher,
   exporterKw,
   exporterNameMatcher,
   flowTableKw,
   getFlowGroups,
   getFlowGroupsAndIds,
   getFtrTypeFromArgs,
   getTrackerNamesAndIds,
   groupKw,
   isSftAgentRunning,
   sampledShowKw,
   trackerKw,
   trackerNameMatcher,
   trackingShowKw,
)
from CliPlugin.FlowTrackingCounterCliLib import (
   CollectorInfo,
   CollectorStatisticsKey,
   FlowCounterEntry,
   FlowCounterKey,
   FlowGroupCounterEntry,
   FlowGroupCounterKey,
   TemplateIdType,
   addExporters,
)
from CliPlugin.SftCliLib import (
   AllowedPacketDirection,
   getFlowTable,
   getTrackerName,
   hoFlowStateShowStr,
   intfRangeRule,
   lfibSourceToTopLabelType,
   sampledHardwareKw,
   tokenDirectionValue,
   tokenDstIp,
   tokenDstPort,
   tokenInterface,
   tokenIPv4,
   tokenIPv6,
   tokenPort,
   tokenProtocol,
   tokenProtocolValue,
   tokenSrcIp,
   tokenSrcPort,
   tokenState,
   tokenStateValue,
   tokenVlan,
   tokenVlanValue,
   tokenVrf,
   tokenVrfValue,
)
from CliPlugin import SftModel
from CliToken.Flow import flowMatcherForShow
from FlowTrackerCliUtil import (
   ftrTypeSampled,
   ipStr,
   protocolStr,
   tcpFlagStr,
)
from IpLibConsts import ALL_VRF_NAME
import LazyMount
from SftCliUtil import (
   HoFlowConfig,
   HoFlowCounter,
   HoFlowStateEnum,
   HoFlowStatus,
   HosStateCounter,
   HosStateCounterKey,
)
from SftLib import (
   ifindexToIntf,
   vniOrUnknown,
)
import SharedMem
import ShowCommand
import Smash
import SmashLazyMount
import sys
import Tac
from Toggles.FlowTrackerToggleLib import toggleSftLimitHoFlowConfigsEnabled
import Tracing
from TypeFuture import TacLazyType

HoIpFlowKey = TacLazyType( "FlowTracking::HoIpFlowKey" )
HoIp6FlowKey = TacLazyType( "FlowTracking::HoIp6FlowKey" )
AddressFamily = TacLazyType( "Arnet::AddressFamily" )
FtConstants = TacLazyType( 'FlowTracking::Constants' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
IpfixConst = TacLazyType( 'Arnet::IpfixConst' )
MplsLFibSource = TacLazyType( 'Mpls::LFib::Source' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
SmashCollectorInfo = TacLazyType( 'Smash::Ipfix::CollectorInfo' )
TemplateIdEnum = TacLazyType( 'FlowTracking::StaticTemplateId' )
dirIngress = Tac.Type( 'Bridging::AllowedTrafficDirection' ).ingress
dirEgress = Tac.Type( 'Bridging::AllowedTrafficDirection' ).egress

hoIpv4FgId = FtConsts.reservedTemplateId( 'dataTemplateIpv4' )
hoEgressIpv4FgId = FtConsts.reservedTemplateId( 'dataTemplateEgressIpv4' )
hoIpv6FgId = FtConsts.reservedTemplateId( 'dataTemplateIpv6' )
hoEgressIpv6FgId = FtConsts.reservedTemplateId( 'dataTemplateEgressIpv6' )

zeroIpv4GenAddr = IpGenAddr( "0.0.0.0" )
zeroIpv6GenAddr = IpGenAddr( "::" )

allFgIds = ( hoIpv4FgId, hoIpv6FgId, hoEgressIpv4FgId, hoEgressIpv6FgId, )

traceHandle = Tracing.Handle( 'SftCliShow' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

activeAgentDir = None
arpVrfIdMap = None
entityManager = None
ethIntfStatusDir = None
FgCount = namedtuple( 'FgCount', [ 'flows', 'activeFlows', 'packets' ] )
hosStateCounter = None
ipfixStats = {}
interfaceCounters = {}
sflowHwStatusDir = None
ftConfig = {}
ftCounters = {}
sftHoCounters = None
shmemEntityManager = None
sftPacketCounters = None
simRespErrorCounters = None
tunnelIntfStatusDir = None

def ipv4TemplateIdSet():
   return frozenset( (
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateIpv4 ),
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateEgressIpv4 ),
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateMplsIpv4 ),
   ) )

def isIpv4TemplateId( templateId ):
   return templateId in ipv4TemplateIdSet()

def ipv6TemplateIdSet():
   return frozenset( (
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateIpv6 ),
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateEgressIpv6 ),
      FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateMplsIpv6 ),
   ) )

def isIpv6TemplateId( templateId ):
   return templateId in ipv6TemplateIdSet()

def isAfTemplateId( af, templateId ):
   if af == AddressFamily.ipv4:
      return isIpv4TemplateId( templateId )
   elif af == AddressFamily.ipv6:
      return isIpv6TemplateId( templateId )
   else:
      return None

def getVrfNameByVrfIdFromArpVrfIdMap( vrfId ):
   vrf = arpVrfIdMap.vrfIdToName.get( vrfId )
   return vrf.vrfName if vrf else "unknown"

def flowKeyMatch( key, vrfName, vlanId,
                  srcIp, dstIp, srcPort, dstPort, protocol, direction ):
   if srcIp is not None and srcIp != key.srcAddr:
      return False

   if dstIp is not None and dstIp != key.dstAddr:
      return False

   if srcPort is not None and srcPort != key.srcPort:
      return False

   if dstPort is not None and dstPort != key.dstPort:
      return False

   if vlanId is not None and vlanId != key.vlanId:
      return False

   if direction is not None and direction != key.direction:
      return False

   vrfNameStr = getVrfNameByVrfIdFromArpVrfIdMap( key.vrfId )
   if vrfName is not None and vrfName != vrfNameStr and vrfName != ALL_VRF_NAME:
      return False

   if ( protocol is not None and
         ( protocol.lower() != protocolStr( key.ipProtocol,
               key.ipProtocolNumber ).lower() ) and
         protocol != str( key.ipProtocolNumber ) ):
      return False

   return True

def getHoFlowConfig( shmemEm ):
   smashHoFlowConfig = shmemEm.doMount( HoFlowConfig.mountPath( ftrTypeSampled ),
                                        'FlowTracking::HoFlowConfig',
                                        Smash.mountInfo( 'reader' ) )
   return smashHoFlowConfig

def getHoFlowStatus( shmemEm ):
   smashHoFlowStatus = shmemEm.doMount( HoFlowStatus.mountPath( ftrTypeSampled ),
                                        'FlowTracking::HoFlowStatus',
                                        Smash.mountInfo( 'reader' ) )
   return smashHoFlowStatus

def getHoFlowCounters( shmemEm ):
   smashHoFlowCounter = \
      shmemEm.doMount( HoFlowCounter.mountPath( ftrTypeSampled, "hfc" ),
                       'FlowTracking::HoFlowCounter',
                       Smash.mountInfo( 'reader' ) )
   return smashHoFlowCounter

def getTrackerNameForHocKey( hoFlowConfig, key, af ):
   config = None
   if af == AddressFamily.ipv4:
      config = hoFlowConfig.hoIpFlowConfig.get( key )
   elif af == AddressFamily.ipv6:
      config = hoFlowConfig.hoIp6FlowConfig.get( key )
   cfgIntf = key.intfId
   if config and config.cachedIntfId and key.intfId != config.cachedIntfId:
      cfgIntf = config.cachedIntfId
   tracker = getTrackerName( ftConfig[ ftrTypeSampled ], cfgIntf,
                             key.direction )
   return tracker, config

def getExpiredFlows( entry ):
   return entry.totalFlowsCreated - entry.activeFlows

# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking sampled flow-table [detail]
# [ tracker <tracker-name> ] [ group <group-name> ]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ interface <interface-range> ] [ ingress|egress ]
#------------------------------------------------------------
class ShowFlowTable:

   def showFlowDetail( self, flowDetailModel, groupAf, entry, bgp, mpls,
                       ifIndexMap ):
      flowDetailModel.srcEthAddr = entry.srcEthAddr
      flowDetailModel.dstEthAddr = entry.dstEthAddr
      flowDetailModel.tcpFlags = tcpFlagStr( entry.tcpFlags )
      flowDetailModel.lastPktTime = entry.lastPktTime
      flowDetailModel.tos = entry.tos
      flowDetailModel.ingressVlanId = entry.ingressVlanId()
      flowDetailModel.ingressIntf = ifindexToIntf( ifIndexMap, entry.inIfIndex ) if \
         entry.inIfIndex else ''
      flowDetailModel.egressVlanId = entry.egressVlanId()
      flowDetailModel.egressIntf = ifIndexMap.get( entry.outIfIndex, 'unknown' )
      flowDetailModel.vni = vniOrUnknown( entry.vni )
      if entry.dot1qVlanTag.hasValidVid():
         flowDetailModel.dot1qVlanId = entry.dot1qVlanTag.vid
         if entry.dot1qCustomerVlanTag.hasValidVid():
            flowDetailModel.dot1qCustomerVlanId = entry.dot1qCustomerVlanTag.vid
      if bgp:
         flowDetailModel.srcAs = bgp.srcAs
         flowDetailModel.dstAs = bgp.dstAs
         flowDetailModel.nextHopIp = IpGenAddr( ipStr( bgp.nextHopIp ) )
         flowDetailModel.bgpNextHopIp = IpGenAddr( ipStr( bgp.bgpNextHopIp ) )
         flowDetailModel.srcPrefixLen = bgp.srcPrefixLen
         flowDetailModel.dstPrefixLen = bgp.dstPrefixLen
      else:
         flowDetailModel.srcAs = 0
         flowDetailModel.dstAs = 0
         zeroAddr = ( zeroIpv4GenAddr if groupAf == AddressFamily.ipv4
               else zeroIpv6GenAddr )
         flowDetailModel.nextHopIp = zeroAddr
         flowDetailModel.bgpNextHopIp = zeroAddr
         flowDetailModel.srcPrefixLen = 0
         flowDetailModel.dstPrefixLen = 0

      if mpls:
         mplsDetail = SftModel.MplsFlowDetail()
         if mpls.topLabel != MplsLabel.null:
            mplsDetail.topLabel = mpls.topLabel
            mplsDetail.topLabelType = lfibSourceToTopLabelType.get( mpls.source )
         mplsDetail.labelStack = [
            mpls.label.get( i ) for i in range( mpls.labelStackSize )
         ]
         mplsDetail.labelStackTruncated = mpls.labelStackTruncated
         if mpls.getRawAttribute( 'fec' ) != IpGenPrefix():
            mplsDetail.fec = mpls.fec
         t1( "MPLS flow entry", mpls.key.str(), "topLabelType:",
             repr( mplsDetail.topLabelType ), "source:", repr( mpls.source ) )
         flowDetailModel.mpls = mplsDetail

      flowDetailModel.sampledPktsReceived = entry.pkts
      flowDetailModel.sampledBytesReceived = entry.bytes
      flowDetailModel.hwPktsReceived = entry.hwPkts
      flowDetailModel.hwBytesReceived = entry.hwBytes

      if groupAf == AddressFamily.ipv6:
         flowDetailModel.flowLabel = entry.flowLabel

   def isEntryMatchingFilter( self, key, entry, ifIndexMap=None, srcIpAddr=None,
         dstIpAddr=None, srcPort=None, dstPort=None, vrfName=None, vlanId=None,
         protocol=None, intfs=None, direction=None ):

      if not flowKeyMatch( key, vrfName, vlanId,
            srcIpAddr, dstIpAddr, srcPort, dstPort, protocol, direction ):
         return False

      # If there are no interfaces on which to filter, consider the flow a match.
      if not intfs:
         return True

      # It is possible a flow entry's ingress interface name cannot be located via
      # the index. In this case, filter the entry.
      if entry.inIfIndex not in ifIndexMap:
         return False

      # This mapping contains special interfaces e.g. 0 maps to "unknown" while
      # 0x3fffffff maps to "CPU". These are to cover possible logical interface
      # terminations for a flow for which there is no corresponding Arnet::IntfId. In
      # practice, these will never match a provided interface filter.
      entryInIntfName = ifIndexMap[ entry.inIfIndex ]
      return entryInIntfName in intfs

   def populateFlowModel( self, key, entry ):
      flowModel = SftModel.FlowModel()
      flowModel.bytesReceived = entry.bytes
      flowModel.pktsReceived = entry.pkts
      flowModel.pktsReceived += entry.hwPkts
      flowModel.bytesReceived += entry.hwBytes
      flowModel.startTime = entry.startTime

      flowKeyModel = SftModel.FlowKeyModel()
      flowKeyModel.vrfName = getVrfNameByVrfIdFromArpVrfIdMap( key.vrfId )
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowKeyModel.direction = key.direction

      flowModel.key = flowKeyModel

      return flowModel

   def showFlowEntry( self,
                      flowTable,
                      groupAf,
                      key,
                      entry,
                      bgp=None,
                      mpls=None,
                      ifIndexMap=None,
                      displayType=None ):

      flowModel = self.populateFlowModel( key, entry )

      if displayType == "detail":
         flowDetailModel = SftModel.FlowDetailModel()
         self.showFlowDetail( flowDetailModel, groupAf, entry, bgp, mpls,
                              ifIndexMap )
         flowModel.flowDetail = flowDetailModel

      return flowModel

   def fetchTrackingModel( self ):
      trackingModel = SftModel.TrackingModel()
      trackingModel.running = isSftAgentRunning( entityManager, activeAgentDir )
      trackingModel.ftrType = ftrTypeSampled
      return trackingModel

   def getFlowTable( self, sMount, trackerName ):
      return getFlowTable( sMount, trackerName )

   def showFlowTable( self, mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      srcIpAddr = None
      dstIpAddr = None
      groupName = args.get( 'GROUP_NAME' )
      trackerName = args.get( 'TRACKER_NAME' )
      if 'SRC_IPV4' in args:
         srcIpAddr = args.get( 'SRC_IPV4' )
      else:
         srcIpAddr = args.get( 'SRC_IPV6' )
      if srcIpAddr is not None:
         srcIpAddr = IpGenAddr( str( srcIpAddr ) )
      if 'DST_IPV4' in args:
         dstIpAddr = args.get( 'DST_IPV4' )
      else:
         dstIpAddr = args.get( 'DST_IPV6' )
      if dstIpAddr is not None:
         dstIpAddr = IpGenAddr( str( dstIpAddr ) )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      if protocol is not None:
         if protocol.isalpha():
            try:
               protocol = Tac.enumValue( 'Arnet::IpProtocolNumber',
                                         'ipProto' + protocol.capitalize() )
            except AttributeError:
               # unassigned protocol number will not have ipProtocol Enum
               # assign ipProtoUnknown and filter it out in ShowCliHelper
               protocol = Tac.enumValue( 'Arnet::IpProtocolNumber',
                                         'ipProtoUnknown' )
         else:
            protocol = int( protocol )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      intfs = args.get( 'INTFS', [] )
      direction = args.get( 'DIRECTION' )

      interfaceFilter = Tac.newInstance( 'FlowTracking::InterfaceFilter' )
      for intfName in intfs:
         interfaceFilter.intfFilter.add( intfName )

      trackingModel = self.fetchTrackingModel()
      if not trackingModel.running:
         return trackingModel

      trNames = set()
      for trName in ftConfig[ ftrType ].hwFtConfig:
         if trackerName and trName != trackerName:
            continue
         trNames.add( trName )
      trNames = sorted( trNames )

      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      # get flow tables
      flowtableHelper = Tac.newInstance( 'FlowTracking::FlowtableHelper' )
      for trName in trNames:
         flowTable = self.getFlowTable( shmemEntityManager, trName )
         if not flowTable:
            continue
         for grName in getFlowGroups( ftrType, trName ):
            if groupName and grName != groupName:
               continue
         flowtableHelper.flowTable[ trName ] = flowTable
      helper = Tac.newInstance( 'FlowTracking::ShowCliHelper',
                                 flowtableHelper,
                                 'detail' in args,
                                 groupName,
                                 srcIpAddr,
                                 dstIpAddr,
                                 srcPort,
                                 dstPort,
                                 protocol,
                                 vrfName,
                                 vlanId,
                                 direction,
                                 interfaceFilter,
                                 ethIntfStatusDir,
                                 arpVrfIdMap,
                                 ftrType,
                                 tunnelIntfStatusDir
                                 )
      with Tac.ActivityLockHolder():
         helper.render( fd, fmt )
      return cliPrinted( SftModel.TrackingModel )

def showFlowTable( mode, args ):
   t = ShowFlowTable()
   return t.showFlowTable( mode, args )

tokenDetail = CliCommand.Node(
                        CliMatcher.KeywordMatcher( 'detail',
                                                       "Detailed flow information" ),
                         storeSharedResult=True, maxMatches=1 )

def generateShowTrackingFilter():
   _expression = '''[ { ( tracker TRACKER_NAME )
                      | ( group GROUP_NAME )
                      | ( src-ip ( SRC_IPV4 | SRC_IPV6 ) )
                      | ( vlan VLAN )
                      | ( dst-ip ( DST_IPV4 | DST_IPV6 ) )
                      | ( vrf VRF )
                      | ( src-port SRC_PORT )
                      | ( dst-port DST_PORT )
                      | ( detail )
                      | ( protocol PROTOCOL )
                      | ( interface INTFS )
                      | ( DIRECTION ) } ]'''
   _data = {
      'detail' : tokenDetail,
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'group' : CliCommand.Node( groupKw, maxMatches=1 ),
      'GROUP_NAME' : allGroupNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'SRC_IPV6' : tokenIPv6,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'DST_IPV6' : tokenIPv6,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'interface' : tokenInterface,
      'INTFS' : intfRangeRule,
      'DIRECTION' : tokenDirectionValue,
   }

   class ShowTrackingFilter( CliCommand.CliExpression ):
      expression = _expression
      data = _data

   return ShowTrackingFilter

class ShowFlowTrackingFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled flow-table [ FILTER ] '
   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'flow-table' : flowTableKw,
      'FILTER' : generateShowTrackingFilter()
   }

   handler = showFlowTable
   cliModel = SftModel.TrackingModel

addShowCommandClass( ShowFlowTrackingFlowTable )

#------------------------------------------------------------
# show flow tracking sampled flow-table hardware [detail]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ] [ state <state> ]
# [ tracker <tracker-name> ] [ interface <interface-range> ] [DIRECTION]
#------------------------------------------------------------
class HoFlowTable:

   def isEntryMatchingFilter( self, key, entry, srcIpAddr=None, dstIpAddr=None,
         srcPort=None, dstPort=None, vrfName=None, vlanId=None,
         protocol=None, state=None, intfs=None, direction=None ):

      if not flowKeyMatch( key, vrfName, vlanId,
            srcIpAddr, dstIpAddr, srcPort, dstPort, protocol, direction ):
         return False

      if state is not None and state != hoFlowStateShowStr[ entry.hoFlowState ]:
         return False

      return not intfs or key.intfId in intfs

   def hoFlowEntry( self, key, entry ):
      flowEntryModel = SftModel.HoFlowEntryModel()

      flowEntryModel.state = entry.hoFlowState

      flowKeyModel = SftModel.FlowKeyModel()

      flowKeyModel.vrfName = getVrfNameByVrfIdFromArpVrfIdMap( key.vrfId )
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowKeyModel.direction = key.direction
      if key.direction == AllowedPacketDirection.ingress:
         flowEntryModel.ingressIntf = key.intfId
      else:
         flowEntryModel.egressIntf = key.intfId

      flowEntryModel.key = flowKeyModel
      flowEntryModel.hwCreateTime = entry.createTime

      return flowEntryModel

   def hoFlowDetail( self, key, config, status ):
      hoFlowDetailModel = SftModel.HoFlowDetailModel()
      hoFlowDetailModel.hwUpdateTime = status.updateTime
      if config:
         hoFlowDetailModel.swCreateTime = config.createTime
         hoFlowDetailModel.srcEthAddr = config.srcEthAddr
         hoFlowDetailModel.dstEthAddr = config.dstEthAddr

      return hoFlowDetailModel

   def hoFlowEntries( self, hoFlowStatus, hoFlowTableModel, detail,
         srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol, state,
         intfs, direction, trackerName ):
      hoFlowConfig = getHoFlowConfig( shmemEntityManager )

      def _processEntry( key, entry, af ):
         if not self.isEntryMatchingFilter( key, entry, srcIpAddr, dstIpAddr,
               srcPort, dstPort, vrfName, vlanId, protocol, state, intfs,
               direction ):
            return
         tracker, config = getTrackerNameForHocKey( hoFlowConfig, key, af )
         if trackerName is not None and tracker != trackerName:
            return

         trackerModel = None
         groupModel = None
         groupName = 'Unknown'
         if tracker != 'Unknown':
            for fgName, fgId in getFlowGroupsAndIds( ftrTypeSampled, tracker ):
               if isAfTemplateId( af, fgId ):
                  groupName = fgName
                  break
         if tracker not in hoFlowTableModel.trackers:
            trackerModel = SftModel.HoTrackerModel()
            hoFlowTableModel.trackers[ tracker ] = trackerModel
            trackerModel.numFlows = 0
         else:
            trackerModel = hoFlowTableModel.trackers[ tracker ]

         if groupName not in trackerModel.groups:
            trackerModel.groups[ groupName ] = SftModel.HoGroupModel()

         groupModel = trackerModel.groups[ groupName ]

         flowModel = self.hoFlowEntry( key, entry )
         if detail:
            hoFlowDetailModel = SftModel.HoFlowDetailModel()

            hoFlowDetailModel = self.hoFlowDetail( key, config, entry )
            flowModel.flowDetail = hoFlowDetailModel
         groupModel.flows.append( flowModel )
         trackerModel.numFlows += 1

      for key, entry in hoFlowStatus.hoIpFlowStatus.items():
         _processEntry( key, entry, AddressFamily.ipv4 )
      for key, entry in hoFlowStatus.hoIp6FlowStatus.items():
         _processEntry( key, entry, AddressFamily.ipv6 )

   def hoFlowTable( self, mode, args ):
      detail = 'detail' in args
      srcIpAddr = args.get( 'SRC_IPV4' )
      dstIpAddr = args.get( 'DST_IPV4' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      trackerName = args.get( 'TRACKER_NAME' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      state = args.get( 'STATE' )
      intfs = set( args.get( 'INTFS', () ) )
      direction = args.get( 'DIRECTION' )

      hoFlowTableModel = SftModel.HoFlowTableModel()
      hoFlowTableModel.running = isSftAgentRunning( entityManager, activeAgentDir )
      hoFlowTableModel.ftrType = ftrTypeSampled
      if not hoFlowTableModel.running:
         return hoFlowTableModel

      hoFlowStatus = getHoFlowStatus( shmemEntityManager )
      if not hoFlowStatus:
         return hoFlowTableModel

      self.hoFlowEntries( hoFlowStatus, hoFlowTableModel, detail,
            srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol,
            state, intfs, direction, trackerName )
      return hoFlowTableModel

def doShowHoFlowTable( mode, args ):
   t = HoFlowTable()
   return t.hoFlowTable( mode, args )

class HoTableFilter( CliCommand.CliExpression ):
   expression = ( '''[ { ( tracker TRACKER_NAME )
                       | ( vlan VLAN )
                       | ( vrf VRF )
                       | ( src-ip SRC_IPV4 )
                       | ( dst-ip DST_IPV4 )
                       | ( src-port SRC_PORT )
                       | ( dst-port DST_PORT )
                       | ( protocol PROTOCOL )
                       | ( state STATE )
                       | ( interface INTFS )
                       | ( DIRECTION )
                       | ( detail ) } ]''' )
   data = {
      'detail' : tokenDetail,
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'state' : tokenState,
      'STATE' : tokenStateValue,
      'interface' : tokenInterface,
      'INTFS' : intfRangeRule,
      'DIRECTION' : tokenDirectionValue,
   }

class ShowFlowTrackingHoFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled flow-table hardware [ FILTER ] '
   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'flow-table' : flowTableKw,
      'hardware' : sampledHardwareKw,
      'FILTER' : HoTableFilter
   }

   handler = doShowHoFlowTable
   cliModel = SftModel.HoFlowTableModel

addShowCommandClass( ShowFlowTrackingHoFlowTable )

#------------------------------------------------------------
# show flow tracking sampled flow-table hardware counters
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ tracker <tracker-name> ] [ interface <interface-range> ]
# [ ingress|egress ]
#------------------------------------------------------------
class HoFlowCounters:

   def isEntryMatchingFilter( self, key, entry, srcIpAddr=None, dstIpAddr=None,
                              srcPort=None, dstPort=None, vrfName=None, vlanId=None,
                              protocol=None, intfs=None, direction=None ):

      if not flowKeyMatch( key, vrfName, vlanId,
            srcIpAddr, dstIpAddr, srcPort, dstPort, protocol, direction ):
         return False

      return not intfs or entry.intfId in intfs

   def hoFlowCounterEntry( self, key, entry ):
      flowCounterEntryModel = SftModel.HoFlowCounterEntryModel()

      flowKeyModel = SftModel.FlowKeyModel()
      flowKeyModel.vrfName = getVrfNameByVrfIdFromArpVrfIdMap( key.vrfId )
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowKeyModel.direction = key.direction

      if key.direction == AllowedPacketDirection.ingress:
         flowCounterEntryModel.ingressIntf = entry.intfId
      else:
         flowCounterEntryModel.egressIntf = entry.intfId

      flowCounterEntryModel.key = flowKeyModel
      flowCounterEntryModel.pktsReceived = entry.packets
      flowCounterEntryModel.bytesReceived = entry.bytes
      flowCounterEntryModel.createTime = entry.createTime
      flowCounterEntryModel.updateTime = entry.updateTime

      return flowCounterEntryModel

   def hoFlowEntries( self, hoFlowCounters, hoFlowCountersModel, srcIpAddr,
                      dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol,
                      intfs, direction, trackerName ):
      hoFlowConfig = getHoFlowConfig( shmemEntityManager )
      def _processEntry( key, entry, af ):
         if not self.isEntryMatchingFilter( key, entry, srcIpAddr, dstIpAddr,
                           srcPort, dstPort, vrfName, vlanId, protocol, intfs,
                           direction ):
            return
         if af == AddressFamily.ipv4:
            hoKey = HoIpFlowKey( key, entry.intfId )
         elif af == AddressFamily.ipv6:
            hoKey = HoIp6FlowKey( key, entry.intfId )
         tracker, _ = getTrackerNameForHocKey( hoFlowConfig, hoKey, af )
         if trackerName is not None and tracker != trackerName:
            return

         trackerModel = None
         groupModel = None
         groupName = 'Unknown'
         if tracker != 'Unknown':
            for fgName, fgId in getFlowGroupsAndIds( ftrTypeSampled, tracker ):
               if isAfTemplateId( af, fgId ):
                  groupName = fgName
                  break
         if tracker not in hoFlowCountersModel.trackers:
            trackerModel = SftModel.HoTrackerCounterModel()
            hoFlowCountersModel.trackers[ tracker ] = trackerModel
            trackerModel.numFlows = 0

         trackerModel = hoFlowCountersModel.trackers[ tracker ]

         if groupName not in trackerModel.groups:
            trackerModel.groups[ groupName ] = SftModel.HoGroupCounterModel()

         groupModel = trackerModel.groups[ groupName ]

         flowCounterModel = self.hoFlowCounterEntry( key, entry )
         groupModel.flows.append( flowCounterModel )
         trackerModel.numFlows += 1

      for key, entry in hoFlowCounters.hoIpFlowCounter.items():
         _processEntry( key, entry, AddressFamily.ipv4 )
      for key, entry in hoFlowCounters.hoIp6FlowCounter.items():
         _processEntry( key, entry, AddressFamily.ipv6 )

   def hoFlowCounters( self, mode, args ):
      srcIpAddr = None
      dstIpAddr = None
      intfs = None
      srcIpAddr = args.get( 'SRC_IPV4' )
      dstIpAddr = args.get( 'DST_IPV4' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      trackerName = args.get( 'TRACKER_NAME' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      intfs = args.get( 'INTFS' )
      direction = args.get( 'DIRECTION' )

      hoFlowCountersModel = SftModel.HoFlowCountersModel()
      hoFlowCountersModel.running = isSftAgentRunning( entityManager,
                                                       activeAgentDir )
      hoFlowCountersModel.ftrType = ftrTypeSampled
      if not hoFlowCountersModel.running:
         return hoFlowCountersModel

      hoFlowCounters = getHoFlowCounters( shmemEntityManager )
      if not hoFlowCounters:
         return hoFlowCountersModel

      self.hoFlowEntries( hoFlowCounters, hoFlowCountersModel, srcIpAddr, dstIpAddr,
                          srcPort, dstPort, vrfName, vlanId, protocol, intfs,
                          direction, trackerName )
      return hoFlowCountersModel

def doShowHoFlowCounters( mode, args ):
   t = HoFlowCounters()
   return t.hoFlowCounters( mode, args )

class HoCountersFilter( CliCommand.CliExpression ):
   expression = ( '''[ { ( tracker TRACKER_NAME )
                       | ( vlan VLAN )
                       | ( vrf VRF )
                       | ( src-ip SRC_IPV4 )
                       | ( dst-ip DST_IPV4 )
                       | ( src-port SRC_PORT )
                       | ( dst-port DST_PORT )
                       | ( protocol PROTOCOL )
                       | ( interface INTFS )
                       | ( DIRECTION ) } ]''' )
   data = {
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'interface' : tokenInterface,
      'INTFS' : intfRangeRule,
      'DIRECTION' : tokenDirectionValue,
   }

class ShowFlowTrackingHoFlowCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled flow-table hardware counters [ FILTER ] '
   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'flow-table' : flowTableKw,
      'hardware' : sampledHardwareKw,
      'counters' : 'Show flow tracking sampled hardware counters',
      'FILTER' : HoCountersFilter
   }

   handler = doShowHoFlowCounters
   cliModel = SftModel.HoFlowCountersModel

addShowCommandClass( ShowFlowTrackingHoFlowCounters )

#--------------------------
class ShowCounters:
   def addExporters( self, ftrType, trModel, trName, expFilter ):
      ftCfg = ftConfig[ ftrType ]
      ipFxStats = ipfixStats[ ftrType ]
      addExporters( trModel, trName, expFilter, ftCfg, ipFxStats,
                    ftCfg.swExport, counterModel=SftModel )

   def getHoModel( self, model ):
      if not model.hardwareOffload:
         hoModel = SftModel.HardwareOffloadCounters()
         hoModel.flowsCreated = 0
         hoModel.flowsDeleted = 0
         hoModel.pktsReceived = 0
         hoModel.pktsHardwareFailed = 0
         hoModel.pktsHardwareMiss = 0
         hoModel.pktsDiscarded = 0
         model.hardwareOffload = hoModel
      return model.hardwareOffload

   def populateHoCounters( self, hoModel, hoCounts ):
      hoModel.flowsCreated += hoCounts.hoFlowsCreated
      hoModel.flowsDeleted += hoCounts.hoFlowsDeleted
      hoModel.pktsReceived += hoCounts.hoPackets
      hoModel.pktsDiscarded += hoCounts.hoFlowTrackerDisabled + \
                                 hoCounts.hoInvalidIfIndex + \
                                 hoCounts.hoFlowConfigMiss
      hoModel.pktsHardwareMiss += hoCounts.hoFlowOffloadMiss
      hoModel.pktsHardwareFailed += hoCounts.hoFlowConfigLimit

   def addFgHoCounters( self, fgModel, ftId, fgIds=None ):
      if sftHoCounters:
         if not fgIds:
            fgIds = allFgIds

         for fgId in fgIds:
            counterKey = FlowGroupCounterKey( ftId, fgId )
            hoCounts = sftHoCounters.hoCounter.get( counterKey )
            if hoCounts:
               hoModel = self.getHoModel( fgModel )
               self.populateHoCounters( hoModel, hoCounts )
               t1( 'addFgHoCounters', hoCounts, 'for fgId', fgId )

   def addHoCounters( self, ftrType, model ):
      if sftHoCounters:
         for _, ftId in getTrackerNamesAndIds( model.ftrType ):
            self.addFgHoCounters( model, ftId )
         self.addFgHoCounters( model, 0 )

      hoModel = self.getHoModel( model )
      if hosStateCounter:
         flowsActive = 0
         flowsPending = 0
         inHardware = HosStateCounterKey( HoFlowStateEnum.inHardware )
         for key, flow in hosStateCounter.hosStateIpCounter.items():
            if key == inHardware:
               flowsActive += flow.hoFlows
            else:
               flowsPending += flow.hoFlows
            t1( 'hosStateIpCounter', key, flow )
         for key, flow in hosStateCounter.hosStateIp6Counter.items():
            if key == inHardware:
               flowsActive += flow.hoFlows
            else:
               flowsPending += flow.hoFlows
            t1( 'hosStateIp6Counter', key, flow )
         hoModel.flowsActive = flowsActive
         hoModel.flowsPending = flowsPending

         hoFlowConfig = getHoFlowConfig( shmemEntityManager )

         sampledFtConfig = ftConfig.get( ftrTypeSampled )
         if toggleSftLimitHoFlowConfigsEnabled():
            if hoFlowConfig and sampledFtConfig:
               if sampledFtConfig.hwOffloadIpv4:
                  hoModel.hoLimitIpv4 = hoFlowConfig.maxHoIpFlows
               if sampledFtConfig.hwOffloadIpv6:
                  hoModel.hoLimitIpv6 = hoFlowConfig.maxHoIp6Flows

   def addFlowGroups( self, ftrType, ftrCounters, trModel, trName,
                      ftId ):
      anyGroups = False
      flows = 0
      activeFlows = 0
      packets = 0
      for fgName, fgId in getFlowGroupsAndIds( ftrType, trName ):
         anyGroups = True
         t1( 'process flow group', fgName )
         fgModel = SftModel.FlowGroupCounters()
         counterKey = FlowGroupCounterKey( ftId, fgId )
         fgCounts = ftrCounters.flowGroupCounter.get( counterKey )
         if not fgCounts:
            t0( 'No counters for', counterKey.smashString() )
            fgCounts = FlowGroupCounterEntry()
         fgModel.activeFlows = fgCounts.flowEntry.activeFlows
         if fgName == IP_GROUP:
            self.addFgHoCounters( fgModel, ftId, [ hoIpv4FgId ] )
         elif fgName == IP6_GROUP:
            self.addFgHoCounters( fgModel, ftId, [ hoIpv6FgId ] )
         elif fgName == EGRESS_IP_GROUP:
            self.addFgHoCounters( fgModel, ftId, [ hoEgressIpv4FgId ] )
         elif fgName == EGRESS_IP6_GROUP:
            self.addFgHoCounters( fgModel, ftId, [ hoEgressIpv6FgId ] )
         else:
            assert fgName == "UnknownGroup"

         flows += fgCounts.flowEntry.totalFlowsCreated
         activeFlows += fgCounts.flowEntry.activeFlows
         fgModel.flows = fgCounts.flowEntry.totalFlowsCreated
         fgExpFlows = getExpiredFlows( fgCounts.flowEntry )
         fgModel.expiredFlows = fgExpFlows
         packets += fgCounts.flowEntry.packets
         fgModel.packets = fgCounts.flowEntry.packets

         trModel.flowGroups[ fgName ] = fgModel

      clearTimeKey = FlowGroupCounterKey( ftId, TemplateIdType.maxTemplateId )
      clearTime = ftrCounters.flowGroupCounter.get( clearTimeKey )
      if clearTime and clearTime.key == clearTimeKey:
         # clearTime should be most recent lastClearedTime from either
         # ftCounters or ipfixStats
         lastClearedTime = switchTimeToUtc( clearTime.flowEntry.lastClearedTime )
         if trModel.clearTime is None or lastClearedTime > trModel.clearTime:
            trModel.clearTime = lastClearedTime

      if not anyGroups:
         t0( 'WARNING: no flow groups for tracker', trName )

      return FgCount( flows=flows, activeFlows=activeFlows, packets=packets )

   def addTrackers( self, model, ftrCounters, trFilter, expFilter ):
      allTrackerFlows = 0
      allActiveFlows = 0
      allPkts = 0
      for trName, ftId in getTrackerNamesAndIds( model.ftrType ):
         if trFilter and trFilter != trName:
            t1( 'tracker', trName, 'did not match filter' )
            continue
         t1( 'process tracker', trName )
         flowTable = self.getFlowTable( shmemEntityManager, trName )
         if not flowTable:
            continue
         trModel = SftModel.TrackerCounters()
         clearTimeKey = CollectorStatisticsKey( trName, "", CollectorInfo() )
         ipfxStats = ipfixStats[ model.ftrType ]
         clearTime = ipfxStats.stats.get( clearTimeKey )
         if clearTime and clearTime.key == clearTimeKey:
            trModel.clearTime = switchTimeToUtc(
               clearTime.lastClearedTime )
         counts = self.addFlowGroups( model.ftrType, ftrCounters, trModel,
                                      trName, ftId )
         trModel.flows = counts.flows
         trModel.activeFlows = counts.activeFlows
         trModel.expiredFlows = counts.flows - counts.activeFlows

         allTrackerFlows += counts.flows
         allActiveFlows += trModel.activeFlows
         trModel.packets = counts.packets
         allPkts += counts.packets
         self.addExporters( model.ftrType, trModel, trName, expFilter )
         model.trackers[ trName ] = trModel
      ftrKey = FlowCounterKey( 0 )
      ftrCounts = ftrCounters.flowsCounter.get( ftrKey )
      if not ftrCounts:
         ftrCounts = FlowCounterEntry()
      if ftrCounts.flowEntry.lastClearedTime != 0:
         lastClearedTime = switchTimeToUtc( ftrCounts.flowEntry.lastClearedTime )
         model.clearTime = lastClearedTime
      model.activeFlows = allActiveFlows
      model.flows = allTrackerFlows
      model.expiredFlows = model.flows - model.activeFlows
      model.packets = allPkts

   def getFlowTable( self, sMount, trackerName ):
      return getFlowTable( sMount, trackerName )

   def addSampleCounters( self, ftrType, model ):
      model.received = 0
      model.poolSize = 0
      model.discarded = 0
      counter = interfaceCounters[ ftrType ]
      # Global counters use a null intfId.
      nullIntf = Tac.Value( 'Arnet::IntfId' )
      initialHwSamples = 0
      if counter:
         for direction in [ dirIngress, dirEgress ]:
            model.received += counter.sampledPkts( direction )
            model.poolSize += counter.samplePool( direction )
            model.discarded += counter.undersize( nullIntf, direction )
            model.discarded += counter.otherError( nullIntf, direction )
            model.discarded += counter.unknownEtherType( nullIntf, direction )
         # BUG796326 initialHwSamples is only supported for ingress by DMA driver.
         # ignore egress lookup intentionally
         initialHwSamples = counter.initialHwSamples( dirIngress )
      hwSamples = [ s.hwSamples for s in sflowHwStatusDir.values() ]
      hwSamples = max( hwSamples ) if hwSamples else 0
      if hwSamples >= initialHwSamples:
         # In case hwSamples went backward report hwSamples from sflow hwStatus
         hwSamples -= initialHwSamples
      model.hardwareReceived = hwSamples

   def showCounters( self, mode, args, model, renderHwOffload=False ):
      if model.running:
         model.sampleCounters = SftModel.SampleCounters()
         self.addSampleCounters( model.ftrType, model.sampleCounters )
         trFilter = args.get( 'TRACKER_NAME' )
         expFilter = args.get( 'EXPORTER_NAME' )
         self.addTrackers( model, ftCounters[ model.ftrType ], trFilter, expFilter )
         if renderHwOffload:
            self.addHoCounters( model.ftrType, model )

def showSftFlowCounters( mode, args ):
   t = ShowCounters()
   model = SftModel.FtrCounters()
   model.ftrType = ftrTypeSampled
   model.running = isSftAgentRunning( entityManager, activeAgentDir )
   t.showCounters( mode, args, model, renderHwOffload=True )
   return model

class ShowSampledCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled counters ' + \
            '[ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'counters' : 'Show flow tracking sampled counters',
      'tracker' : trackerKw,
      'TRACKER_NAME' : trackerNameMatcher,
      'exporter' : exporterKw,
      'EXPORTER_NAME' : exporterNameMatcher,
   }

   handler = showSftFlowCounters
   cliModel = SftModel.FtrCounters

addShowCommandClass( ShowSampledCounters )

#--------------------------
class ShowCountersDebug:
   def addPacketCounters( self, ftrType, myModel ):
      for trName, ftId in getTrackerNamesAndIds( ftrType ):
         for _, fgId in getFlowGroupsAndIds( ftrType, trName ):
            counterKey = FlowGroupCounterKey( ftId, fgId )

            # If an entry exists for these
            # flow group and tracker ids, set the incremental change
            # to the its values
            if sftPacketCounters:
               pktErrorEntry = sftPacketCounters.packetErrorCounter.get( counterKey )
               if pktErrorEntry:
                  myModel.flowTableLimit += pktErrorEntry.rxFlowTableLimit
                  myModel.ftInputQueueFull += pktErrorEntry.rxFtInputQueueFull
                  myModel.fttSimReqQueueFull += pktErrorEntry.rxFwdSimReqQueueFull
                  myModel.routeInfoQueueFull += pktErrorEntry.rxRoutingInfoQueueFull
            if simRespErrorCounters:
               smrEntry = simRespErrorCounters.simRespErrorCounter.get( counterKey )
               if smrEntry:
                  myModel.fttSimRespQueueFull += smrEntry.rxFwdSimRespQueueFull

   def showCountersDebug( self, mode, args, model ):
      if model.running:
         model.packetErrorCounters = SftModel.PacketErrorCounter()
         model.packetErrorCounters.flowTableLimit = 0
         model.packetErrorCounters.ftInputQueueFull = 0
         model.packetErrorCounters.fttSimReqQueueFull = 0
         model.packetErrorCounters.fttSimRespQueueFull = 0
         model.packetErrorCounters.routeInfoQueueFull = 0
         self.addPacketCounters( model.ftrType, model.packetErrorCounters )

def showSftCounterDebug( mode, args ):
   t = ShowCountersDebug()
   model = SftModel.FtrCountersDebug()
   model.ftrType = ftrTypeSampled
   model.running = isSftAgentRunning( entityManager, activeAgentDir )
   t.showCountersDebug( mode, args, model )
   return model

class ShowSampledCountersDebug( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled counters debug'

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'counters' : 'Show flow tracking sampled counters',
      'debug' : 'Provide visibility into discarded samples',
   }

   handler = showSftCounterDebug
   cliModel = SftModel.FtrCountersDebug

addShowCommandClass( ShowSampledCountersDebug )

def getIntfCounter( intf, direction ):
   model = SftModel.PerInterfaceFlowErrCounters()
   counters = interfaceCounters[ ftrTypeSampled ]
   if counters:
      model.parseFail = counters.undersize( intf, direction )
      model.unknownProtocol = counters.unknownEtherType( intf, direction )
      model.other = ( counters.otherError( intf, direction ) +
                      counters.disabledIntf( intf, direction ) )
   else:
      model.parseFail = 0
      model.unknownProtocol = 0
      model.other = 0
   return model

def showSftCountersInterfaceError( mode, args ):
   intfs = args.get( 'INTFS' )
   model = SftModel.InterfaceCountersFlowErr()
   model.running = isSftAgentRunning( entityManager, activeAgentDir )
   model.ftrType = ftrTypeSampled
   if not intfs:
      # Populate intf with all interfaces that have flow tracking enabled
      sampled = ftConfig[ ftrTypeSampled ]
      intfs = set()
      intfs.update( sampled.hwIntfFtConfig )
      intfs.update( sampled.hwEgressIntfFtConfig )
   ingressCounters = model.ingressCounters
   egressCounters = model.egressCounters
   for intf in intfs:
      if intf in ftConfig[ ftrTypeSampled ].hwIntfFtConfig:
         ingressCounters[ intf ] = getIntfCounter( intf, dirIngress )
      if intf in ftConfig[ ftrTypeSampled ].hwEgressIntfFtConfig:
         egressCounters[ intf ] = getIntfCounter( intf, dirEgress )

   return model

class ShowSampledCountersInterface( ShowCommand.ShowCliCommandClass ):
   syntax = '''show flow tracking sampled counters
      interface [ INTFS ] errors'''

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'counters' : 'Show flow tracking sampled counters',
      'interface' : tokenInterface,
      'INTFS' : intfRangeRule,
      'errors' : "Packet parse errors"
   }

   handler = showSftCountersInterfaceError
   cliModel = SftModel.InterfaceCountersFlowErr

addShowCommandClass( ShowSampledCountersInterface )


#--------------------------
def Plugin( em ):
   global activeAgentDir
   global arpVrfIdMap
   global ethIntfStatusDir
   global entityManager
   global sflowHwStatusDir
   global shmemEntityManager
   global sftHoCounters
   global hosStateCounter
   global sftPacketCounters
   global simRespErrorCounters
   global tunnelIntfStatusDir

   entityManager = em
   shmemEntityManager = SharedMem.entityManager( sysdbEm=em )

   ftrType = ftrTypeSampled
   interfaceCounters[ ftrType ] = SmashLazyMount.mount( em,
                              'flowtracking/%s/interfaceCounters' % ftrType,
                              'Smash::SampledFlowTracker::InterfaceCounters',
                              SmashLazyMount.mountInfo( 'reader' ) )
   ftConfig[ ftrType ] = LazyMount.mount( em,
                                'hardware/flowtracking/config/%s' % ftrType,
                                'HwFlowTracking::Config', 'r' )
   ftCounters[ ftrType ] = SmashLazyMount.mount( em,
                                       'flowtracking/%s/counters' % ftrType,
                                       'Smash::FlowTracker::FtCounters',
                                       SmashLazyMount.mountInfo( 'reader' ) )
   ipfixStats[ ftrType ] = SmashLazyMount.mount( em,
                                   'flowtracking/%s/ipfix/statistics' % ftrType,
                                   'Smash::Ipfix::CollectorStatistics',
                                   SmashLazyMount.mountInfo( 'reader' ) )

   sflowHwStatusDir = LazyMount.mount( em, 'sflow/hwstatus', 'Tac::Dir', 'ri' )

   hosStateCounter = SmashLazyMount.mount( em,
                                    HosStateCounter.mountPath( ftrTypeSampled ),
                                    'FlowTracking::HosStateCounter',
                                    SmashLazyMount.mountInfo( 'reader' ) )
   sftHoCounters = SmashLazyMount.mount( em,
                                    'flowtracking/sampled/hoCounters',
                                    'Smash::SampledFlowTracker::HoCounters',
                                    SmashLazyMount.mountInfo( 'reader' ) )
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
   arpVrfIdMap = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
   ethIntfStatusDir = LazyMount.mount( em, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   sftPacketCounters = SmashLazyMount.mount( em,
                                    'flowtracking/sampled/pktcounters',
                                    'Smash::SampledFlowTracker::PacketCounters',
                                    SmashLazyMount.mountInfo( 'reader' ) )
   simRespErrorCounters = SmashLazyMount.mount( em,
                                 'flowtracking/sampled/srespcounters',
                                 'Smash::SampledFlowTracker::SimRespPacketCounters',
                                 SmashLazyMount.mountInfo( 'reader' ) )
   tunnelIntfStatusDir = LazyMount.mount( em, "interface/status/tunnel/intf",
                                          "Interface::TunnelIntfStatusDir", "r" )
