#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpAddr
import BasicCli
from CliCommand import (
   CliExpression,
   singleKeyword,
   singleNode,
)
import CliMatcher
from CliPlugin.BgpVpnCli import (
   showExtCommunity,
)
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher
from CliPlugin.RoutingBgpShowCli import (
   summaryVrfModel,
)
from CliToken.RoutingBgpShowCliTokens import (
   advertised,
   bgpAfterShow,
   communityExact,
   detail,
   neighbors,
   received,
   receivedAll,
   summary,
)
from PseudowireLib import vplsNameRegex
from RouteMapLib import getExtCommValue
import ShowCommand
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled

matcherRouteType = CliMatcher.KeywordMatcher( 'route-type',
      helpdesc='Filter by NLRI route type' )
# matcher for different route types
matcherAutoDiscovery = CliMatcher.KeywordMatcher( 'auto-discovery',
      helpdesc='Filter by BGP auto-discovery route' )
matcherSignaling = CliMatcher.KeywordMatcher( 'signaling',
      helpdesc='Filter by BGP signaling route' )

matcherVeId = CliMatcher.IntegerMatcher( 0, 0xffff, helpdesc="VPLS Edge identifier" )

vpls = CliMatcher.KeywordMatcher( "vpls", helpdesc="Virtual private LAN service" )

vplsInstance = CliMatcher.KeywordMatcher( "instance", helpdesc="VPLS instance" )
matcherVplsInstance = CliMatcher.PatternMatcher( pattern=vplsNameRegex,
                                                 helpname="WORD",
                                                 helpdesc="VPLS instance name" )

vplsGroup = CliMatcher.KeywordMatcher( "group", helpdesc="VPLS group" )
matcherVplsGroup = CliMatcher.PatternMatcher( pattern=vplsNameRegex,
                                              helpname="WORD",
                                              helpdesc="VPLS group name" )

class VplsRouteTypeExpression( CliExpression ):
   expression = '''route-type ( auto-discovery | signaling )'''

   data = {
      'route-type': singleNode( matcher=matcherRouteType, maxMatches=1 ),
      'auto-discovery': matcherAutoDiscovery,
      'signaling': matcherSignaling
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'route-type', None ):
         return
      nlriType = None
      if args.pop( 'auto-discovery', None ):
         nlriType = 'vplsVpnAd'
      if args.pop( 'signaling', None ):
         nlriType = 'vplsVpnVe'
      if nlriType:
         args[ 'nlriType' ] = nlriType

# -------------------------------------------------------------------------------
# "show bgp vpls summary [ route-type auto-discovery | signaling ]"
# -------------------------------------------------------------------------------
# Show BGP VPLS peers and optionally filter by route-type.
# When route-type is not specified, summary statistics reflect the number of routes
# received and accepted for all VPLS route-types.
# When a route-type is specified, summary statistics reflect the number of routes
# received and accepted for the specified route-type.
class BgpVplsSummaryCommand( ShowCommand.ShowCliCommandClass ):
   if toggleVplsBgpSignalingEnabled():
      syntax = "show bgp vpls summary [ROUTE_TYPE]"
   else:
      syntax = "show bgp vpls summary"
   data = {
      "bgp": bgpAfterShow,
      "vpls": vpls,
      "summary": summary,
      "ROUTE_TYPE": VplsRouteTypeExpression,
   }
   cliModel = summaryVrfModel
   handler = "ShowBgpVplsHandler.BgpVplsSummaryCommand_handler"

BasicCli.addShowCommandClass( BgpVplsSummaryCommand )

# -------------------------------------------------------------------------------
# "show bgp vpls auto-discovery"
# -------------------------------------------------------------------------------
class BgpVplsAdCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpls auto-discovery [ instance INSTANCE [ group GROUP ] ]"
   data = {
      "bgp": bgpAfterShow,
      "vpls": vpls,
      "auto-discovery": "Auto discovery",
      "instance": vplsInstance,
      "INSTANCE": matcherVplsInstance,
      "group": vplsGroup,
      "GROUP": matcherVplsGroup,
   }
   cliModel = "ShowBgpVplsModels.BgpVplsAdInstances"
   handler = "ShowBgpVplsHandler.BgpVplsAdCommand_handler"

BasicCli.addShowCommandClass( BgpVplsAdCommand )

class BgpVplsExtCommValuesAndExactExpr( CliExpression ):
   expression = "{ ( rt RT_VAL ) | ( l2-vpn-id L2_VPN_ID ) } [ exact ]"
   data = {
      "rt": "Route target",
      "RT_VAL": RtSooExtCommCliMatcher( "Route Target" ),
      "l2-vpn-id": singleKeyword( "l2-vpn-id", "L2 VPN identifier" ),
      "L2_VPN_ID": singleNode(
         matcher=RtSooExtCommCliMatcher( "L2 VPN identifier",
                                         acceptLongAsn=False ) ),
      "exact": communityExact,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      extCommList = []
      rts = args.pop( "RT_VAL", None )
      if rts:
         extCommList.extend(
            getExtCommValue( "rt " + rt ).extendedCommunity() for rt in rts )
      l2VpnId = args.pop( "L2_VPN_ID", None )
      if l2VpnId:
         extCommList.append(
            getExtCommValue( "l2VpnId " + l2VpnId ).extendedCommunity() )
      if extCommList:
         # extCommValues will be ExtCommU64|ExtCommU64|ExtCommU64|...
         args[ "extCommValues" ] = "|".join( str( c ) for c in extCommList )
         args[ "extendedCommunitiesExactMatch" ] = args.pop( "exact", None )

class BgpVplsOptions( CliExpression ):
   vplsBgpSignalingFilters_ = " ( ROUTE_TYPE ) | ( ve-id VE_ID ) | " \
      if toggleVplsBgpSignalingEnabled() else ""
   expression = """{
      ( rd RD ) |
      ( pe-addr ( PE_ADDR | PE_ADDR_NUM ) ) |
      ( next-hop NEXTHOP ) |
      ( extcommunity EXT_COMM ) | %s
      ( detail )
   } """ % vplsBgpSignalingFilters_

   data = {
      "rd": singleKeyword( "rd", helpdesc="Filter by BGP route distinguisher" ),
      "RD": singleNode(
         matcher=RdDistinguisherMatcher( helpdesc="BGP route distinguisher" ) ),
      "pe-addr": singleKeyword( "pe-addr",
         helpdesc="Filter by PE address (auto discovery routes only)" ),
      "PE_ADDR": singleNode( matcher=IpAddrMatcher( helpdesc="PE address" ) ),
      "PE_ADDR_NUM": singleNode(
         matcher=CliMatcher.IntegerMatcher( 0, 0xffffffff, helpdesc="PE address" ) ),
      "next-hop": singleKeyword( "next-hop",
         helpdesc="Filter by route next hop IP address" ),
      "NEXTHOP": singleNode( matcher=IpGenAddrMatcher(
                                        "IPv4 or IPv6 address",
                                        helpdesc4="IPv4 address",
                                        helpdesc6="IPv6 address" ) ),
      "extcommunity": singleNode( matcher=showExtCommunity ),
      "EXT_COMM": BgpVplsExtCommValuesAndExactExpr,
      "detail": singleNode( matcher=detail ),
      "ROUTE_TYPE": VplsRouteTypeExpression,
      "ve-id": singleKeyword( "ve-id",
         helpdesc="Filter by VPLS Edge identifier (signaling routes only)" ),
      "VE_ID": singleNode( matcher=matcherVeId ),
   }

   @staticmethod
   def adapter( mode, args, argList ):
      rd = args.pop( "RD", None )
      if rd:
         args[ "rdValue" ] = rd
      peAddr = args.pop( "PE_ADDR", None )
      if peAddr:
         args[ "peAddrValue" ] = IpAddr( peAddr ).value
      else:
         peAddr = args.pop( "PE_ADDR_NUM", None )
         if peAddr is not None:
            args[ "peAddrValue" ] = peAddr
      nh = args.pop( "NEXTHOP", None )
      if nh:
         args[ "nexthopValue" ] = nh
      veId = args.pop( "VE_ID", None )
      if veId is not None:
         args[ "veIdValue" ] = veId

# -------------------------------------------------------------------------------
# "show bgp vpls
# -------------------------------------------------------------------------------
class BgpVplsCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpls [ BGP_VPLS_OPTIONS ]"
   data = {
      "bgp": bgpAfterShow,
      "vpls": vpls,
      "BGP_VPLS_OPTIONS": BgpVplsOptions,
   }
   cliModel = "ShowBgpVplsModels.BgpVplsRoutes"
   handler = "ShowBgpVplsHandler.BgpVplsCommand_handler"

BasicCli.addShowCommandClass( BgpVplsCommand )

# -------------------------------------------------------------------------------
# "show bgp neighbors <> vpls
# -------------------------------------------------------------------------------
class BgpVplsNeighborsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp neighbors PEER vpls
      ( ADV_ROUTES | RCVD_ROUTES | ROUTES ) [ BGP_VPLS_OPTIONS ]"""
   data = {
      "bgp": bgpAfterShow,
      "neighbors": neighbors,
      "PEER": IpGenAddrMatcher( "Neighbor IPv4 or IPv6 address",
                                helpdesc4="Neighbor IPv4 address",
                                helpdesc6="Neighbor IPv6 address" ),
      "vpls": vpls,
      "ADV_ROUTES": advertised,
      "RCVD_ROUTES": receivedAll,
      "ROUTES": received,
      "BGP_VPLS_OPTIONS": BgpVplsOptions,
   }
   cliModel = "ShowBgpVplsModels.BgpVplsRoutes"
   handler = "ShowBgpVplsHandler.BgpVplsNeighborsCmd_handler"

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( "ADV_ROUTES", None ):
         args[ "routeType" ] = "advertised-routes"
      if args.pop( "RCVD_ROUTES", None ):
         args[ "routeType" ] = "received-routes"
      if args.pop( "ROUTES", None ):
         args[ "routeType" ] = "routes"
      args[ "peerAddr" ] = args.pop( "PEER" )

BasicCli.addShowCommandClass( BgpVplsNeighborsCmd )

# -------------------------------------------------------------------------------
# "show bgp vpls signaling" (AID10962)
# -------------------------------------------------------------------------------
class BgpVplsVeCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpls signaling [ instance INSTANCE [ group GROUP ] ]"
   data = {
      "bgp": bgpAfterShow,
      "vpls": vpls,
      "signaling": "BGP signaling",
      "instance": vplsInstance,
      "INSTANCE": matcherVplsInstance,
      "group": vplsGroup,
      "GROUP": matcherVplsGroup,
   }
   cliModel = "ShowBgpVplsModels.BgpVplsVeInstances"
   handler = "ShowBgpVplsHandler.BgpVplsVeCommand_handler"

if toggleVplsBgpSignalingEnabled():
   BasicCli.addShowCommandClass( BgpVplsVeCommand )
