# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BasicCli
import CliCommand
import CliMatcher
import CliToken.Hardware
import CliToken.Resource
import LazyMount
import Plugins
import SharedMem
import ShowCommand
import TableOutput
import Tac
import AncillaryCoreMemory
import Tracing
import natsort
import CliExtensions
from CliModel import Model, List, Str, Dict, cliPrinted
from CliPlugin import ResourceMgrCliLib
from TypeFuture import TacLazyType
import sys

__defaultTraceHandle__ = Tracing.Handle( "AsicResourceMgrCli" )
t4 = Tracing.trace4

IndexTableType = TacLazyType( 'Asic::IndexTable' )

allConfigPathDir = None
em = None
registry = None
dmaSmaEm = None
shmemEm = None
sysdbEm = None
invalidUnitIdValue = 65536
pythonImplTables = CliExtensions.CliHook()
pythonImplTableSet = set()

# Class to register helper plugins
class PluginRegistry:
   def __init__( self ):
      self.pluginRegistry = {}

   def addPlugin( self, plugin ):
      self.pluginRegistry[ plugin.name ] = plugin

   def getPlugin( self, name ):
      return self.pluginRegistry.get( name, None )

# Context information for loading helper plugin
class PluginContext:
   def __init__( self, entityManager, pluginRegistry ):
      self.entityManager = entityManager
      self.pluginRegistry = pluginRegistry

# Cli matcher function for available table options
def getHwTable( mode=None ):
   setOfTables = set()
   hwTableMountInfos = ResourceMgrCliLib.collectMountInfos(
         allConfigPathDir )
   for table in hwTableMountInfos:
      setOfTables.add( table.split( '::' )[ -1 ] )
      setOfTables.add( table )
   setOfTables.add( '*' )
   filterRegistry = registry.pluginRegistry
   for tblFilter in filterRegistry.values():
      setOfTables.update( tblFilter.getCustomTable() )

   return { table: "Show table info for " + table
            for table in setOfTables }

def getCustomHandlers():
   '''Update the set of custom table handlers from AsicResourceFilterPlugin.
   This is platfrom specific'''
   customHandlers = {}
   filterRegistry = registry.pluginRegistry
   for tblFilter in filterRegistry.values():
      customHandlers.update( tblFilter.getCustomHandler() )

   for customTableName, customHandler in customHandlers.items():
      customHandlers[ customTableName ] = customHandler
   return customHandlers

def getAgentName( mode, context ):
   setOfAgents = set()
   if context.sharedResult:
      if context.sharedResult[ 'RESOURCE' ] == '*':
         setOfAgents = { writerAgent
                         for ownerAgent in allConfigPathDir.values()
                         for table in ownerAgent.values()
                         for writerAgent in table
                       }
      else:
         table = context.sharedResult[ 'RESOURCE' ]
         for ownerAgent in allConfigPathDir.values():
            for hwTable in ownerAgent:
               if table in hwTable:
                  setOfAgents.update( ownerAgent[ hwTable ] )
         filterRegistry = registry.pluginRegistry
         for tblFilter in filterRegistry.values():
            ownerAgent = tblFilter.getOwnerAgent( table )
            if ownerAgent:
               setOfAgents.update( ownerAgent )

   setOfAgents.add( '*' )
   return { agent: "Writer agent"
            for agent in setOfAgents }

nodeTable = CliCommand.Node(
      matcher=CliMatcher.DynamicKeywordMatcher( getHwTable ),
      storeSharedResult=True )
nodeAgent = CliCommand.Node(
      matcher=CliMatcher.DynamicKeywordMatcher( getAgentName, passContext=True ),
      storeSharedResult=True )

# cliModel for hardware table entry
class TableEntry( Model ):
   keyFields = Dict( valueType=str,
                     help="A mapping of entry key field attr to field val" )
   dataFields = Dict( valueType=str,
                      help="A mapping of entry data field attr to field val" )

# cliModel for hardware table
class Table( Model ):
   entries = List( valueType=TableEntry, help="Hardware table entries" )

class View( Model ):
   views = Dict( valueType=Table,
         help="A mapping of view to its hardware table entries" )

class Unit( Model ):
   units = Dict( valueType=View, help="A mapping of unit to its views" )

class AgentName( Model ):
   agents = Dict( valueType=Unit, help="A mapping of agent name to its units" )

def getNonZeroFieldNames( hwTbl, non0Attr, non0Key ):
   for entry in hwTbl.entries:
      if entry.keyFields:
         non0Key.update( list( entry.keyFields ) )
      non0Attr.update( list( entry.dataFields ) )

def getNonZeroFieldNamesEntries( entries, non0Attr, non0Key ):
   for entry in entries:
      if entry.keyFields:
         non0Key.update( list( entry.keyFields ) )
      non0Attr.update( list( entry.dataFields ) )

def mapIdentifierToTable( hwTbls, tables ):
   for table in tables:
      agents = tables[ table ].agents
      for agent in agents:
         units = agents[ agent ].units
         for unit in units:
            views = units[ unit ].views
            for view in views:
               tblIdentifier = '/'.join( [ table, agent, unit, view ] )
               hwTbls[ tblIdentifier ] = views[ view ]

def orderEntryFields( keyFields, dataFields ):
   """This function orders table fields being rendered."""
   def key( field ):
      return(
              field != 'hwIndex', # hwIndex goes first
              field != 'entryIndex', # entryIndex goes first if not hwIndex
              field != 'dataIndex', # dataIndex if it exists, goes after entryIndex.
              field == 'status', # status goes last.
              'ipv' not in field, # IPvX columns come first.
              field )

   dataAttr = natsort.natsorted( dataFields, key=key )
   keyAttr = natsort.natsorted( keyFields, key=key )
   return keyAttr, dataAttr

# cliModel for `show hardware resource`
class HardwareResource( Model ):
   tables = Dict( valueType=AgentName, help="A mapping of table name to its agents" )
   _outputFormat = Str( help="Output format" )
   _fieldOrder = List( valueType=str, help="Column order for fields" )
   _tableFormat = List( valueType=str, help="Column formatting" )
   _customColWidth = List( valueType=int, help="Equal length columns" )

   def render( self ):
      # move to a function
      hwTbls = {}
      mapIdentifierToTable( hwTbls, self.tables )

      # display each hardware table info
      for identifier, resourceInfo in hwTbls.items():
         resource, agent, unitId, view = identifier.split( "/" )
         hwTbl = resourceInfo
         if not hwTbl.entries:
            continue

         print( 'Resource:', resource )
         print( 'Feature agent:', agent )
         print( 'Unit id:', unitId )
         print( 'View:', view )

         fmts = []
         # get tableformats and create array for TableOutput
         if self._tableFormat:
            if self._customColWidth:
               customMinWidth, customMaxWidth = self._customColWidth
            else:
               customMinWidth = None
               customMaxWidth = None
            # the first column format should always be small and right aligned
            # as it's entryIndx/hwIndex the others could be wider
            justify = TableOutput.Format( justify='right' )
            justify.padLimitIs( True )
            fmts += [ justify ]
            for colFormat in self._tableFormat[ 1 : ]:
               justify = TableOutput.Format( justify=f"{ colFormat }",
                                             minWidth=customMinWidth,
                                             maxWidth=customMaxWidth, wrap=True )
               justify.padLimitIs( True )
               fmts += [ justify ]

         # default print format( tabular format )
         if not self._outputFormat:
            if self._fieldOrder:
               tableAttr = self._fieldOrder
            else:
               non0Attr = set()
               non0Key = set()
               getNonZeroFieldNames( hwTbl, non0Attr, non0Key )
               # Get the ordered field names
               non0Key, non0Attr = orderEntryFields( list( non0Key ),
                  list( non0Attr ) )
               tableAttr = non0Key + non0Attr
            table = TableOutput.createTable( tableAttr )
            table.formatColumns( *fmts )
            for entry in hwTbl.entries:
               # Map entry field names to entry field values
               attrList = ResourceMgrCliLib.getEntryFieldValue( tableAttr,
                  entry.keyFields, entry.dataFields )
               table.newRow( *attrList )
            print( table.output() )

         # user-chosen print format( verbose or csv format )
         else:
            for entry in hwTbl.entries:
               verbose = self._outputFormat == 'verbose'
               # Get the ordered field names
               keyAttr, dataAttr = orderEntryFields(
                                    entry.keyFields, entry.dataFields )
               tableAttr = keyAttr + dataAttr
               # Map entry field names to entry field values
               attrList = ResourceMgrCliLib.getEntryFieldValue( tableAttr,
                     entry.keyFields, entry.dataFields )
               for attrs in list( zip( tableAttr, attrList ) ):
                  if verbose:
                     print( f'{attrs[ 0 ]}: {attrs[ 1 ]}' )
                  else:
                     print( f'{attrs[ 0 ]}={attrs[ 1 ]}', end=',' )
               # prevents extra space in verbose output
               if verbose:
                  print( "\n", end='' )
               else:
                  print( "\n" )

def checkEntryView( entry, subIdx=None ):
   entryViewT = getattr( entry, 'viewT', None )
   if entryViewT is None:
      return True
   if subIdx is not None:
      return entry.isViewTValid( subIdx )
   return entry.isViewTValid()

def getResponseFields( entryStatusCol, entryKey, indexedTable,
      dataFieldsList, tableFormat ):
   """Adds hwIndex, status attributes and corresponding alignments"""
   if entryStatusCol and entryStatusCol.get( entryKey ):
      entryStatus = entryStatusCol.get( entryKey )
      if not indexedTable:
         tableFormat[ 'hwIndex' ] = 'right'
      tableFormat[ 'status' ] = 'left'
      for dataFields in dataFieldsList:
         if not indexedTable:
            dataFields[ 'hwIndex' ] = str( entryStatus.hwEntryIndex )
         dataFields[ 'status' ] = entryStatus.entryState

def doShowEntry( indexedTable, outputFormat, entryDict,
      entryStatusCol, status, entryIdx ):
   verbose = outputFormat == 'verbose'
   tableEntries = []
   tableFormat = []

   # get type of entry to be used to get attrs/attrType
   # for alignments to print
   if entryIdx is None:
      entryTacType = entryDict.tacAttr.memberType.fullTypeName
      entry = Tac.Type( entryTacType )
   else:
      entry = Tac.Type( entryDict[ entryIdx ].tacType.fullTypeName )

   numSubEntries = entry.numSubEntries()
   tableFormat = ResourceMgrCliLib.getFilteredEntryAlignments( entry,
   numSubEntries, verbose, indexedTable, attrPrefix='' )

   iterColl = entryDict
   if entryStatusCol:
      iterColl = entryStatusCol

   # entryStatusCol is keyed by the entryKey or entryIdx
   for entryKey in iterColl:
      entry = entryDict.get( entryKey, None )
      # No entry exists at that index in the corresponding
      # writeResponse collection.
      if entry is None:
         continue
      numSubEntries = entry.numSubEntries()
      for subIdx in range( numSubEntries ):
         if numSubEntries > 1:
            # check if viewT value for the different sub-entries
            # matches the view currently being iterated
            if not checkEntryView( entry, subIdx ):
               continue
            if not entry.isSubEntryValid( subIdx ) and not verbose:
               continue
         else:
            if not checkEntryView( entry ):
               continue
            if not entry.isValid() and not verbose:
               continue
         keyFieldsList, dataFieldsList = ResourceMgrCliLib.getEntryFields(
               entry, subIdx, numSubEntries, verbose, indexedTable, entryKey )
         # get writeResponse attributes, if present
         if status:
            getResponseFields( entryStatusCol, entryKey, indexedTable,
               dataFieldsList, tableFormat )
         for idx, dataFieldsDict in enumerate( dataFieldsList ):
            if dataFieldsDict:
               if keyFieldsList:
                  tableEntries.append(
                     TableEntry( keyFields=keyFieldsList[ idx ],
                        dataFields=dataFieldsDict ) )
               else:
                  tableEntries.append(
                     TableEntry( keyFields={}, dataFields=dataFieldsDict ) )
   return tableEntries, tableFormat


def sortTableEntryList( tableEntries ):
   # Sort tableEntries based on:
   # 1) entryIndex : for index tables
   # 2) hwIndex : for hash tables with status enabled
   # If tableEntry contains neither of the above-mentioned
   # fields, return the list as is
   tableEntry = tableEntries[ 0 ]
   sortKey = None
   # check if the tableEntry contains any fields to sort by
   for attr in [ 'hwIndex', 'entryIndex' ]:
      if tableEntry.dataFields.get( attr ):
         sortKey = attr

   def key( tableEntry ):
      return int( tableEntry.dataFields.get( sortKey ) )

   if sortKey:
      tableEntries = sorted( tableEntries, key=key )
   return tableEntries

def displayTableNames():
   displayedTables = set()
   resource = {}

   def updateDisplayedTables( tableName ):
      if tableName not in displayedTables:
         print( tableName )
         resource[ tableName ] = AgentName()
         displayedTables.add( tableName )

   hwTableMountInfos = ResourceMgrCliLib.collectMountInfos(
         allConfigPathDir )
   for tableName in hwTableMountInfos:
      updateDisplayedTables( tableName )

   # Add custom hardware table names to the table list
   customTableSet = set()
   filterRegistry = registry.pluginRegistry
   for tblFilter in filterRegistry.values():
      customTableSet.update( tblFilter.getCustomTable() )
   for table in customTableSet:
      print( table )
      resource[ table ] = AgentName()
   return resource

class ShHwResrcHelper:
   def __init__( self, mode, args, pythonHelperNeeded=False ):
      self.hwTbl = args.get( "RESOURCE" )
      self.fA = args.get( "AGENT" )
      self.unitId = args.get( "UNIT_ID" )
      self.entryIdx = args.get( "IDX" )
      self.status = "status" in args
      self.outputFormat = args.get( "FORMAT", "" )
      # this pythonHelperNeeded bool will protect against instantiating
      # two cliPrintHelpers as the CPP Helper inherits from this Helper
      if pythonHelperNeeded:
         if mode.session_.outputFormat() == 2:
            self.printJSON = True
         else:
            self.printJSON = False
         self.cliPrintHelper = Tac.newInstance(
               "ShowHwResource::PythonPrintHelper", self.status,
               self.outputFormat, self.hwTbl, self.printJSON )
         cliprintOutputFormat = mode.session_.outputFormat()
         fileDesc = sys.stdout.fileno()
         self.cliPrintHelper.initHelper( fileDesc, cliprintOutputFormat )
      self.mode = mode
      self.resource = {}
      self.tableFormat = {}

   def doShowBase( self, resourceConfig, tableFormat, entryIdx,
                   outputFormat, mode, tableName, agentName, unitId,
                   entryStatusCol=None, status=False ):
      tableTacType = resourceConfig.tacType
      entryTacType = tableTacType.attr( 'entry0' ).memberType
      views = []
      # if the entryTacType has viewT attribute, iterate through
      # every tacAttr, check if it's a collection and its entries
      # are derived from the entry tac type, thus it's one of the view
      # collections
      if entryTacType.attr( 'viewT' ):
         for tableAttr in tableTacType.attributeQ:
            if tableAttr.isCollection and tableAttr.memberType is not entryTacType:
               if tableAttr.memberType.isDerived( entryTacType ):
                  views.append( tableAttr.name )

      # Iterate over the entry collection when no views are supported
      if not views:
         views.append( 'entry0' )
      if self.printJSON:
         self.cliPrintHelper.startAgentJSON( agentName, unitId )
      for view in views:
         entryDict = getattr( resourceConfig, view )
         # index table has entryindex for each entry, while
         # keyed table doesn't. We need entryindex to order
         # the entries for index table which is why we need
         # to differentiate between index and keyed tables.
         indexedTable = tableTacType.isDerived( IndexTableType.tacType )
         # it is only valid to specify entry index for index table
         if entryIdx is not None and indexedTable:
            entryDict = ResourceMgrCliLib.getEntryAtIdx( entryDict, entryIdx, mode )
            if not entryDict:
               continue
            if entryStatusCol:
               entryStatusCol = ResourceMgrCliLib.getEntryAtIdx( entryStatusCol,
                     entryIdx, mode )
         if entryIdx and not indexedTable:
            return
         tableEntries, filteredFormats = doShowEntry( indexedTable, outputFormat,
               entryDict, entryStatusCol, status, entryIdx )
         if tableEntries:
            # Don't sort entries if JSON output
            if self.cliPrintHelper.printJSON:
               entries = tableEntries
            else:
               entries = sortTableEntryList( tableEntries )
            self.printToCpp( tableName, agentName, view,
                             unitId, entries, filteredFormats )
      if self.printJSON:
         self.cliPrintHelper.endAgentJSON()

   def renderJSONEntry( self, entry ):
      """Render every entry for JSON output"""
      def output( attrs ):
         for field, val in attrs.items():
            self.cliPrintHelper.renderEntryJSON( field, val )

      self.cliPrintHelper.startListEntryJSON()

      self.cliPrintHelper.startDictJSON( "keyFields" )
      if hasattr( entry, "keyFields" ):
         output( entry.keyFields )
      self.cliPrintHelper.endDictJSON()

      self.cliPrintHelper.startDictJSON( "dataFields" )
      if hasattr( entry, "dataFields" ):
         output( entry.dataFields )
      self.cliPrintHelper.endDictJSON()

      self.cliPrintHelper.endListEntryJSON()

   def printToCpp( self, tableName, agentName, view,
         unitId, entries, filteredFormats ):
      """Prints to Cli using CliPrinter"""
      # create sets of all non-zero fields names to render
      non0Attr = set()
      non0Key = set()
      getNonZeroFieldNamesEntries( entries, non0Attr, non0Key )
      if self.printJSON:
         # if printing JSON then start cliprint JSON view dict
         self.cliPrintHelper.startEntryJSON( view )
         for entry in entries:
            self.renderJSONEntry( entry )
         self.cliPrintHelper.endEntryJSON()
         return

      # print the table header information if non-JSON
      tableDetails = Tac.newInstance(
            "ShowHwResource::TableDetails", tableName,
            agentName, unitId, view )
      self.cliPrintHelper.printTblDetails( tableDetails )

      # Get the ordered field names
      non0Key, non0Attr = orderEntryFields( list( non0Key ),
         list( non0Attr ) )
      tableAttr = non0Key + non0Attr

      # print regular tabular output using cliprint
      if not self.outputFormat:
         # get tableformat for the current view and assign them to print helper
         if filteredFormats:
            filteredFormats = {
                  k: v for k, v in filteredFormats.items() if
                  k in tableAttr }

         attrDetail = Tac.nonConst( self.cliPrintHelper.attrDetail )

         for attr, colFormat in filteredFormats.items():
            attrDetail.attrFormat[ attr ] = colFormat
         for index, attr in enumerate( tableAttr ):
            attrDetail.attrName[ index ] = attr

         self.cliPrintHelper.attrDetail = attrDetail

         self.cliPrintHelper.initTblRenderer()
         # print every entry after fetching value for each attribute
         for entry in entries:
            attrList = ResourceMgrCliLib.getEntryFieldValue( tableAttr,
               entry.keyFields, entry.dataFields )
            for index, attrVal in enumerate( attrList ):
               attrDetail.attrVal[ index ] = attrVal
               self.cliPrintHelper.attrDetail = attrDetail
            self.cliPrintHelper.printTblRndrAttrVal( self.cliPrintHelper.attrDetail )

      # user-chosen print format( verbose or csv format )
      else:
         for entry in entries:
            lastAttr = False
            verbose = self.outputFormat == 'verbose'
            # Get the ordered field names
            keyAttr, dataAttr = orderEntryFields(
                                 entry.keyFields, entry.dataFields )
            tableAttr = keyAttr + dataAttr
            # Map entry field names to entry field values
            attrList = ResourceMgrCliLib.getEntryFieldValue( tableAttr,
                  entry.keyFields, entry.dataFields )
            attrsValues = list( zip( tableAttr, attrList ) )
            for index, attrs in enumerate( attrsValues ):
               if index == len( attrsValues ) - 1:
                  lastAttr = True
               self.cliPrintHelper.printFmtAttrs( attrs[ 0 ], attrs[ 1 ],
                                                     verbose, lastAttr )

   def doShowSmash( self, tableMountInfo, tableName,
                    featureName, unitId ):
      resourceConfig, resourceStatus = ResourceMgrCliLib.mountTables(
             shmemEm, unitId, tableName, featureName, tableMountInfo=tableMountInfo )
      entryStatusCol = resourceStatus.writeResponse if resourceStatus else None
      self.doShowBase( resourceConfig, self.tableFormat, self.entryIdx,
                       self.outputFormat, self.mode, tableName, featureName,
                       str( unitId ), entryStatusCol, self.status )

   def doShowDmaMem( self, tableMountInfo, unitId,
                     tableName, featureName, isHardwareAccess=False ):
      # mount dmamem table
      dmaTable, mountInfo = ResourceMgrCliLib.mountTables( dmaSmaEm, unitId,
            tableName, featureName, isHardwareAccess=isHardwareAccess )
      self.doShowBase( dmaTable, self.tableFormat, self.entryIdx,
            self.outputFormat, self.mode, tableName, featureName, str( unitId ) )
      dmaSmaEm.unmount( mountInfo.path )
      del dmaTable

class ShHwResrcCliPrintHelper( ShHwResrcHelper ):
   def __init__( self, mode, args ):
      ShHwResrcHelper.__init__( self, mode, args, pythonHelperNeeded=False )
      self.entryIdxPresent = self.entryIdx is not None
      self.entryIdx = args.get( "IDX", 0 )
      self.cliPrintHelper = Tac.newInstance(
            "ShowHwResource::HwResourceHelper", self.status, self.outputFormat )
      cliprintOutputFormat = mode.session_.outputFormat()
      fileDesc = sys.stdout.fileno()
      self.cliPrintHelper.initHelper( fileDesc, cliprintOutputFormat )

   def doShowDmaMem( self, tableMountInfo, unitId, tableName,
                     featureName, isHardwareAccess=False ):
      dmaTable, mountInfo = ResourceMgrCliLib.mountTables( dmaSmaEm, unitId,
            tableName, featureName, isHardwareAccess=isHardwareAccess )
      # BUG#####: account for the writeResponse collection for dmaMemTables
      self.cliPrintHelper.doShowTableEntries( tableName, str( unitId ),
            featureName, self.entryIdx, self.entryIdxPresent, dmaTable, None )
      dmaSmaEm.unmount( mountInfo.path )
      del dmaTable

   def doShowSmash( self, tableMountInfo, tableName,
                    featureName, unitId ):
      resourceConfig, resourceStatus = ResourceMgrCliLib.mountTables( shmemEm,
            unitId, tableName, featureName, tableMountInfo=tableMountInfo )
      self.cliPrintHelper.doShowTableEntries( tableName, str( unitId ),
            featureName, self.entryIdx, self.entryIdxPresent,
            resourceConfig, resourceStatus )

class CustomShHwResrcHelper( ShHwResrcHelper ):
   """Helper class to call the custom handlers"""
   def __init__( self, mode, args, customHandler, hwTableName, entryIdx, extension ):
      ShHwResrcHelper.__init__( self, mode, args, pythonHelperNeeded=True )
      self.customHandler = customHandler
      self.hwTableName = hwTableName
      self.entryIdx = entryIdx
      self.resource = None
      self.fieldOrder = None
      self.tableFormat = None
      self.extension = extension

   def showCustomHwResrc( self ):
      self.resource, self.fieldOrder, *self.tableFormat = self.customHandler(
         allConfigPathDir, dmaSmaEm,
         self.hwTableName, em,
         shmemEm, self.mode, self.entryIdx,
         cliPrintHelper=self.cliPrintHelper,
         extension=self.extension )

class PrintCppOrJSON:
   """Anchor used to print field via Cpp (csv/verbose/tabular) or JSON.
   This is currently used for Custom Handlers, which are all datafields,
   So keyfields for JSON are ignored"""
   def __init__( self, cliPrintHelper, fields ):
      self.cliPrintHelper = cliPrintHelper
      self.fields = fields
      self.attrDetail = cliPrintHelper.attrDetail
      self.printJSON = self.cliPrintHelper.printJSON

   def __enter__( self ):
      return self

   def print( self ):
      # Render every entry for JSON output
      if self.printJSON:
         def output( attrs ):
            for field, val in attrs.items():
               self.cliPrintHelper.renderEntryJSON( field, val )

         self.cliPrintHelper.startListEntryJSON()

         self.cliPrintHelper.startDictJSON( "keyFields" )
         self.cliPrintHelper.endDictJSON()

         self.cliPrintHelper.startDictJSON( "dataFields" )
         output( self.fields )
         self.cliPrintHelper.endDictJSON()

      elif self.cliPrintHelper.outputFormatArg == 'verbose':
         # Render every entry for verbose output
         lastAttr = False
         for index, ( field, val ) in enumerate( self.fields.items() ):
            if index == len( self.fields ) - 1:
               lastAttr = True
            self.cliPrintHelper.printFmtAttrs( field, val, True, lastAttr )

      elif self.cliPrintHelper.outputFormatArg == 'csv':
         # Render every entry for csv output
         lastAttr = False
         for index, ( field, val ) in enumerate( self.fields.items() ):
            if index == len( self.fields ) - 1:
               lastAttr = True
            self.cliPrintHelper.printFmtAttrs( field, val, False, lastAttr )

      else:
         # Render every entry for tabular output with TableRenderer
         for index, attrName in enumerate( list(
               self.cliPrintHelper.attrDetail.attrName.values() ) ):
            attrDetail = Tac.nonConst( self.attrDetail )
            attrDetail.attrVal[ index ] = self.fields.get( attrName, '0' )
            self.attrDetail = attrDetail
         self.cliPrintHelper.printTblRndrAttrVal( self.attrDetail )

   def __exit__( self, exc_type, exc_value, traceback ):
      if self.printJSON:
         self.cliPrintHelper.endListEntryJSON()

class StartTableJSONAnchor:
   """Anchor for starting and ending the table for JSON"""
   def __init__( self, cliPrintHelper ):
      self.cliPrintHelper = cliPrintHelper
      self.printJSON = self.cliPrintHelper.printJSON

   def __enter__( self ):
      if self.printJSON:
         self.cliPrintHelper.startTableJSON()

   def __exit__( self, exc_type, exc_value, traceback ):
      if self.printJSON:
         self.cliPrintHelper.endTableJSON()

      self.cliPrintHelper.destroyHelper()

class TableFieldJSONAnchor:
   """Anchor for starting and ending the table, agents, and entry list/dicts"""
   def __init__( self, cliPrintHelper, hwTableParam, agentName,
         unitId, fieldOrder, customTableRenderer ):
      self.cliPrintHelper = cliPrintHelper
      self.printJSON = self.cliPrintHelper.printJSON
      self.hwTableParam = hwTableParam
      self.agentName = agentName
      self.unitId = unitId
      self.fieldOrder = fieldOrder
      self.customTableRenderer = customTableRenderer

   # Prints the table datails if the appropriate terminating level has entries.
   # Also prints the table renderer in tabular, csv, and verbose.
   # There is a custom customRenderHandler for each custom ALPM handler
   # as they need different custom alignments for the attributes present on each
   # layer
   def printTableDetailsAndRenderer( self, cliPrintHelper, fieldOrder,
         customTableRenderer, agentName, unitId ):
      if cliPrintHelper.outputFormatArg in [ '', 'csv', 'verbose' ]:
         tableDetails = Tac.newInstance(
               "ShowHwResource::TableDetails", cliPrintHelper.hwTableParam,
               agentName, str( unitId ), 'entry0' )
         cliPrintHelper.printTblDetails( tableDetails )
         if cliPrintHelper.outputFormatArg == '':
            customTableRenderer( cliPrintHelper, fieldOrder )

   def __enter__( self ):
      if self.printJSON:
         self.cliPrintHelper.startTableFieldJSON( self.hwTableParam )
         self.cliPrintHelper.startAgentJSON( self.agentName, str( self.unitId ) )
         self.cliPrintHelper.startEntryJSON( 'entry0' )

      else:
         self.printTableDetailsAndRenderer( self.cliPrintHelper, self.fieldOrder,
               self.customTableRenderer, self.agentName, self.unitId )

   def __exit__( self, exc_type, exc_value, traceback ):
      if self.printJSON:
         self.cliPrintHelper.endEntryJSON()
         self.cliPrintHelper.endAgentJSON()
         self.cliPrintHelper.endTableFieldJSON()

def showHwResrc( showHwResourceHelper ):
   hwTableMountInfos = ResourceMgrCliLib.collectMountInfos(
         allConfigPathDir, showHwResourceHelper.hwTbl,
         showHwResourceHelper.fA, showHwResourceHelper.unitId )
   showHwResourceHelper.cliPrintHelper.startTableJSON()
   for tableName, hwTableMountInfo in hwTableMountInfos.items():
      if not showHwResourceHelper.fA:
         # With no fA specified, the CLI displays the aggregated view of the
         # hardware table if supported. Aggregagted hardware tables are either
         # direct index tables or local smash tables.
         if not hwTableMountInfo.checkAggregationSupported():
            showHwResourceHelper.cliPrintHelper.endTableJSON()
            showHwResourceHelper.cliPrintHelper.destroyHelper()
            print( "aggregated resource view not supported yet." )
            return None
         # For aggregated tables, we dont need to iterate
         # through the per feature unitMountInfos
         showHwResourceHelper.cliPrintHelper.startTableFieldJSON( tableName )
         for unitId in hwTableMountInfo.unitIdList:
            featureName = hwTableMountInfo.hamOwner
            if hwTableMountInfo.storage == 'dmamem':
               showHwResourceHelper.doShowDmaMem( hwTableMountInfo,
                                                  unitId,
                                                  tableName,
                                                  featureName,
                                                  isHardwareAccess=True )
      else:
         # iterate through the per feature agent unitMountInfo collection
         # to display the smash table information
         showHwResourceHelper.cliPrintHelper.startTableFieldJSON( tableName )
         for featureName, unitMountInfos in hwTableMountInfo.mountInfoDict.items():
            for unitId, tableMountInfo in unitMountInfos.items():
               if hwTableMountInfo.isDirectIndexTable():
                  # mount and display dmamem Table information
                  showHwResourceHelper.doShowDmaMem( hwTableMountInfo,
                                                     unitId,
                                                     tableName,
                                                     featureName )
               else:
                  showHwResourceHelper.doShowSmash( tableMountInfo,
                                                    tableName,
                                                    featureName,
                                                    unitId )
      showHwResourceHelper.cliPrintHelper.endTableFieldJSON()
   showHwResourceHelper.cliPrintHelper.endTableJSON()
   showHwResourceHelper.cliPrintHelper.destroyHelper()
   return showHwResourceHelper.resource

def updatePythonTableSet():
   for hook in pythonImplTables.extensions():
      pythonImplTableSet.update( hook() )

def doShowHardwareResource( mode, args ):
   hwTableName = args.get( "RESOURCE" )
   outputFormat = args.get( "FORMAT", "" )
   entryIdx = args.get( "IDX" )
   if hwTableName is None:
      resource = displayTableNames()
      return HardwareResource( tables=resource, _outputFormat=outputFormat )

   tableName = hwTableName.split( '::' )[ -1 ]
   customHandlers = getCustomHandlers()
   if tableName in customHandlers:
      customHandler, *extension = customHandlers[ tableName ]
      extension = extension[ 0 ] if extension else None
      helper = CustomShHwResrcHelper( mode, args, customHandler,
                  hwTableName, entryIdx, extension )
      helper.showCustomHwResrc()
      if helper.resource is not None:
         # if tableFormats is return from the customHandler unpack the
         # column alignments and equal column bool if it exists
         if helper.tableFormat:
            tableFormat, *customColWidth = helper.tableFormat
            customColWidth = customColWidth[ 0 ] if customColWidth else None
         else:
            tableFormat, customColWidth = None, None
         return HardwareResource( tables=helper.resource,
               _fieldOrder=helper.fieldOrder, _outputFormat=outputFormat,
               _tableFormat=tableFormat, _customColWidth=customColWidth )
      else:
         return cliPrinted( HardwareResource )

   updatePythonTableSet()
   # For tables in pythonImplTableSet, hybrid python->cpp implementation of the
   # cli is used, for all other tables cpp implentation of the cli is used
   if tableName not in pythonImplTableSet:
      retModel = showHwResrc( ShHwResrcCliPrintHelper( mode, args ) )
      if retModel is None:
         # aggregated view of hardware table not supported
         return HardwareResource( tables={}, _outputFormat=outputFormat )
      else:
         return cliPrinted( HardwareResource )
   else:
      resource = showHwResrc( ShHwResrcHelper( mode, args,
         pythonHelperNeeded=True ) )
      if resource is None:
         return HardwareResource( tables={}, _outputFormat=outputFormat )
      else:
         return cliPrinted( HardwareResource )

# -----------------------------------------------------------------------------------
# The "show hardware resource [ RESOURCE [ agent ( AGENT | * ) ] [ unit
#      ( UNIT_ID ) ] [ index IDX ] [ status | response ] [ FORMAT ] ]" command
# -----------------------------------------------------------------------------------
class ShowHardwareResourceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show hardware resource [ RESOURCE
               [ agent AGENT ]
               [ unit UNIT_ID ]
               [ index IDX ]
               [ status ]
               [ FORMAT ] ]'''
   data = {
         'hardware': CliToken.Hardware.hardwareForShowMatcher,
         'resource': CliToken.Resource.resourceMatcherForShow,
         'RESOURCE': nodeTable,
         'agent': 'Specify the feature agent name',
         'AGENT': nodeAgent,
         'unit': 'Specify the unit ID',
         'UNIT_ID': CliMatcher.IntegerMatcher( 0, 65535,
                                               helpdesc='Unit id' ),
         'index': 'Specify the entry index to display',
         'IDX': CliMatcher.IntegerMatcher( 0, 0xffffffff,
                                             helpdesc='Show entry at index' ),
         'status': 'Display hardware table entry status',
         'FORMAT': CliMatcher.EnumMatcher( { 'verbose': 'Raw dump of resource data',
                                             'csv': 'CSV format', } ),
         '*': CliCommand.Node(
            CliMatcher.KeywordMatcher( '*', helpdesc='Comprehensive view' ),
            maxMatches=1 ),
   }

   handler = doShowHardwareResource
   cliModel = HardwareResource

BasicCli.addShowCommandClass( ShowHardwareResourceCmd )

def Plugin( entityManager ):
   global em, allConfigPathDir
   global registry
   global dmaSmaEm
   global shmemEm
   global sysdbEm
   registry = PluginRegistry()
   em = entityManager
   context = PluginContext( em, registry )
   Plugins.loadPlugins( 'AsicResourceFilterPlugin', context )
   allConfigPathDir = LazyMount.mount(
         entityManager, 'hardware/resource', 'Tac::Dir', 'ri' )
   dmaSmaEm = AncillaryCoreMemory.entityManager( em.sysname() )
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   sysdbEm = em.cEntityManager()
