# Copyright (c) 2009-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# 'string statement has no effect'
# pylint: disable-msg=W0105

""" Implements cli commands for displaying cpu utilization.
Supported in unpriviledged mode:
    - show processes
    - show processes top
    - show uptime
"""

from ArPyUtils.AsanHelper import disableSubprocessAsanPreload
import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import CliPlugin.TechSupportCli
from CliPlugin import ShowProcessesModel
from PyWrappers.Procps import topName
import ShowCommand
import ShowProcessesLib
import Tac
import os
import sys

processesKwMatcher = CliMatcher.KeywordMatcher(
   'processes',
   helpdesc='Show CPU and memory usage of running processes' )

def doShowUptime( mode, args=None ):
   output = ''
   try:
      output = Tac.run( [ 'uptime' ], stdout=Tac.CAPTURE, stderr=sys.stderr )
   except Tac.SystemCommandError as e:
      print( e )
      return ShowProcessesModel.SystemTimeInfo()
   
   return ShowProcessesLib.parseShowUptime( output )

def doShowProcesses( mode, args ):
   model = ShowProcessesModel.SystemProcesses()
   output = ''
   try:
      # Capture the output, then parse it
      output = Tac.run( [ 'ps', '-e', '-o', 
                          'pid,pcpu,pmem,tt,stat,lstart,start,time,cmd', 
                          '--sort=-pcpu', '-ww' ], 
                        stdout=Tac.CAPTURE, stderr=sys.stderr )
   except Tac.SystemCommandError as e:
      print( e )
      return model 
   
   model.timeInfo = doShowUptime( mode )
   model.processes = ShowProcessesLib.parseShowProcesses( output )
   return model

@BasicCliUtil.EapiIncompatible()
def doShowTop( mode, memory=False ):
   # Run top in secure mode (-s) to disallow (k)ill, (r)enice, etc.
   cmd = [ 'top', '-s' ]
   if memory:
      cmd += [ '-o', '%MEM' ]
   # top hands if run using Asan
   with disableSubprocessAsanPreload():
      Tac.run( cmd, stdin=sys.stdin, stdout=sys.stdout, stderr=sys.stderr,
               env=os.environ, ignoreReturnCode=True )

def doShowTopOnce( mode, args ):
   memory = 'memory' in args
   # top hands if run using Asan
   with disableSubprocessAsanPreload():
      env = dict( os.environ )
      if not memory:
         env[ 'CPULOOP' ] = '1'    # delay to get one cycle of CPU stats
      try:
         # Run top in batch mode, one iteration, and with max width (we have no
         # terminal with Tac.CAPTURE for stdout, so top defaults to a narrow
         # 80-column mode otherwise.)
         cmd = [ topName(), '-s', '-b', '-n', '1', '-w', '512' ]
         if memory:
            cmd += [ '-o', '%MEM' ]
         output = Tac.run( cmd, input='', stdout=Tac.CAPTURE,
                           stderr=sys.stderr, env=env )
      except Tac.SystemCommandError as e:
         print( e )
         return ShowProcessesModel.SystemTopInfo()

      return ShowProcessesLib.parseShowTopOnce( output )

# Add 'show processes top' directly to UnprivMode and EnableMode since
# we do not want the pipes.
class ShowProcessesTop( CliCommand.CliCommandClass ):
   syntax = "show processes top [ memory ]"
   data = { 'show' : ShowCommand.showKwMatcher,
            'processes' : processesKwMatcher,
            'top' : ( "Show a dynamic view of the running system. "
                      "Quit with 'q' or Ctrl-C" ),
            'memory' : "Sort processes by memory usage" }
   @staticmethod
   def handler( mode, args ):
      doShowTop( mode, 'memory' in args )

BasicCli.ExecMode.addCommandClass( ShowProcessesTop )

#-----------------------------------------------------------------------------------
# show uptime
#-----------------------------------------------------------------------------------
class ShowUptime( ShowCommand.ShowCliCommandClass ):
   syntax = 'show uptime'
   data = {
            'uptime': 'Show how long the system has been running',
          }
   cliModel = ShowProcessesModel.SystemTimeInfo
   handler = doShowUptime

BasicCli.addShowCommandClass( ShowUptime )

#-----------------------------------------------------------------------------------
# show processes
#-----------------------------------------------------------------------------------
class ShowProcesses( ShowCommand.ShowCliCommandClass ):
   syntax = 'show processes'
   data = {
            'processes': processesKwMatcher,
          }
   cliModel = ShowProcessesModel.SystemProcesses
   handler = doShowProcesses

BasicCli.addShowCommandClass( ShowProcesses )

#-----------------------------------------------------------------------------------
# show processes top [ memory ] once
#-----------------------------------------------------------------------------------
class ShowProcessesTopOnce( ShowCommand.ShowCliCommandClass ):
   syntax = 'show processes top [ memory ] once'
   data = {
            'processes': processesKwMatcher,
            'top': ( 'Show a dynamic view of the running system. '
                     'Quit with \'q\' or Ctrl-C' ),
            'memory': 'Sort processes by memory usage',
            'once': ( 'Only run one iteration of \'top\' '
                      '(after gathering enough info)' ),
          }
   cliModel = CliPlugin.ShowProcessesModel.SystemTopInfo
   handler = doShowTopOnce

BasicCli.addShowCommandClass( ShowProcessesTopOnce )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   # keep the time stamp earlier than gated's as it's possible for gated to crash
   # while executing some commands in show tech' BUG162829
   '2010-05-29 23:59:49',
   cmds=[ 'show processes top once',
          'show processes top memory once' ],
   summaryCmds=[ 'show processes top once',
                 'show processes top memory once' ] )
