#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Ark
import TableOutput
from CliModel import Model, Submodel
from CliModel import Bool, Dict, Str, Float, Enum

fl = TableOutput.Format( justify='left' )
fr = TableOutput.Format( justify='right' )

#-----------------------------------------------------------------------------
# CAPI models for 'show rib ready [ vrf <vrfName | all> ]' command
#-----------------------------------------------------------------------------

# Models specific to multi-agent implementation of the command
class ProtoReadyModel( Model ):
   """Protocols participating in the NSF process and their config/status"""
   configPending = Bool( help="Whether processing of protocol configuration "
                         "is pending or not" )
   configured = Bool( help="Whether the protocol is configured" )
   ready = Bool( help="Whether the protocol has converged" )
   readyTime = Float( optional=True, help="UTC timestamp when this protocol became "
                      "ready" )
   timeTaken = Float( optional=True, help="Time elapsed from start until this "
                      "protocol becomes ready " )

   def renderEntry( self, protocol, table ):
      table.newRow( protocol, self.configPending,
                    self.configured, self.ready, self.timeTaken )

class RibReadyStateModel( Model ):
   """Various milestones in the process of supporting non stop forwarding. In order,
   Igp ready-> Rib ready-> RoutesRecursivelyResolved-> Fib ready
   """
   igpReady = Bool( help="Whether IGPs have converged" )
   igpReadyTime = Float( optional=True,
                         help="UTC time at which IGPs converged" )
   timeTakenForIgpReady = Float( optional=True, help="Time elapsed in seconds from "
                                 "start until IGPs converged" )
   ribReady = Bool( help="Whether all protocols have converged" )
   ribReadyTime = Float( optional=True,
                         help="UTC time at which all protocols converged" )
   timeTakenForRibReady = Float( optional=True, help="Time elapsed in seconds from "
                                 "start until all protocols converged" )
   recursiveResolution = Bool( help="Whether recursive resolution is complete" )
   recursiveResolutionTime = Float( optional=True,
                                    help="UTC time at which recursive route "
                                    "resolution was complete" )
   timeTakenForRecursiveResolution = Float( optional=True, help="Time elapsed in "
                                            "seconds from start until recursive "
                                            "resolution was complete" )
   fibReady = Bool( help="Whether the routes have been synced to FIB" )
   fibReadyTime = Float( optional=True,
                         help="UTC time at which all routes were synced to FIB" )
   timeTakenForFibReady = Float( optional=True, help="Time elapsed in seconds from "
                                 "start until FIB became ready" )

   def renderEntry( self, table ):
      table.newRow( "IGP ready", self.igpReady, self.timeTakenForIgpReady )
      table.newRow( "RIB ready", self.ribReady, self.timeTakenForRibReady )
      table.newRow( "Routes recursively resolved", self.recursiveResolution,
                    self.timeTakenForRecursiveResolution )
      table.newRow( "FIB ready", self.fibReady, self.timeTakenForFibReady )

def timeToDisplay( timeTaken ):
   if timeTaken is None:
      # Protocols may not yet be ready and time is -inf which is the initial
      # value. Return NA instead of "-inf" so as to not confuse users.
      return "NA"
   else:
      return "{:.3f}".format(
         round( timeTaken, 2 ) )

# Models specific to ribd implementation of the command
class RouteReady( Model ):
   """Route types involved in the NSF process and their config/status"""
   configured = Bool( help="Whether the route type is configured" )
   configPending = Bool( help="Whether processing of route configuration "
                         "is pending" )
   ready = Bool( help="Whether the route type has converged" )
   timeTaken = Float( optional=True, help="Time elapsed from start until this "
                      "route type became ready " )

   def renderEntry( self, routeType, table ):
      table.newRow( routeType, self.configured, self.configPending,
                    self.ready, timeToDisplay( self.timeTaken ) )

class RibReadyFlags( Model ):
   """Various milestones in the process of supporting non stop forwarding in gated.
   Rib ready-> RoutesRecursivelyResolved-> V4RoutesSync -> V6RoutesSync -> Fib ready
   """
   fibReady = Bool( help="Whether the routes have been synced to FIB" )
   timeTakenForFibReady = Float( optional=True, help="Time elapsed in seconds from "
                                 "start until FIB became ready" )
   ribReady = Bool( help="Whether all protocols have converged" )
   timeTakenForRibReady = Float( optional=True, help="Time elapsed in seconds from "
                                 "start until all protocols converged" )
   recursiveResolution = Bool( help="Whether the static routes have "
                               "been recursively resolved" )
   timeTakenForRecursiveResolution = Float( optional=True, help="Time elapsed in "
                                            "seconds from start until static routes "
                                            "were recursively resolved" )
   v4RouteSync = Bool( help="Whether v4Routes have been synced to Fib" )
   timeTakenForV4RouteSync = Float( optional=True, help="Time elapsed in seconds "
                                    "from start until v4 routes were synced" )
   v6RouteSync = Bool( help="Whether v6 routes have been synced to Fib" )
   timeTakenForV6RouteSync = Float( optional=True, help="Time elapsed in seconds "
                                    "from start until v4 routes were synced" )

   def renderEntry( self, table ):
      table.newRow( "Fib ready", self.fibReady,
                    timeToDisplay( self.timeTakenForFibReady ) )
      table.newRow( "Rib ready", self.ribReady,
                    timeToDisplay( self.timeTakenForRibReady ) )
      table.newRow( "StaticRtRecursivelyResolved",
                    self.recursiveResolution,
                    timeToDisplay( self.timeTakenForRecursiveResolution ) )
      table.newRow( "V4RoutesSynced", self.v4RouteSync,
                    timeToDisplay( self.timeTakenForV4RouteSync ) )
      table.newRow( "V6RoutesSynced", self.v6RouteSync,
                    timeToDisplay( self.timeTakenForV6RouteSync ) )

# Model to hold info for 'show rib ready' command in both ribd and multi-agent mode
# The CLI registration for this command is present in IpRib/CliPlugin. We also have
# a CLI hook to which both these commands have added an extension and on the basis of
# protocol model in use( 'l3/config' ), we decide which implementation to call.
# Since the rendered output of the show commands is different for ribd and
# multi-agent world, we render the model for appropriate protocol in use based on the
# _protocolModel field in this class.
class RibReadyModel( Model ):
   """Information about the NSF process. Contains config/status of the protocols
   participating in the NSF process and the progress of the process in terms of
   states reached
   """
   _vrf = Str( help="VRF name" )
   _protocolModel = Enum( values=( "ribd", "multi-agent" ),
                          help="Protocol model in use" )
   gracefullyRestarting = Bool( help="The vrf is gracefully restarting or not" )
   startTime = Float( help="UTC time at which Rib ready verification was started" )
   states = Submodel( valueType=RibReadyStateModel, optional=True,
                      help="Various states that occur during the Rib ready "
                      "verification in the multi-agent protocol model" )
   protocols = Dict( keyType=str, valueType=ProtoReadyModel, optional=True,
                     help="Current state of the protocols taking part in the "
                     "Rib ready process in multi-agent protocol model, "
                     "key being protocol name" )
   flags = Submodel( valueType=RibReadyFlags, optional=True,
                     help="Various checkpoints along the path to declare Fib ready"
                     "in ribd protocol model" )
   routes = Dict( keyType=str, valueType=RouteReady, optional=True,
                  help="Current state of the route types considered part of the "
                  "Rib ready process in ribd protocol model, key being route type" )

   def render( self ):
      print( "=======================" )
      print( "VRF %s" % ( self._vrf ) )
      print( "=======================" )

      if self._protocolModel == "ribd":
         print( "Start time : %s" %
                Ark.timestampToStr( self.startTime ) )
         print( "Grestart Init : %d" % ( 1 if self.gracefullyRestarting else 0 ) )
         print( "" )

         flagsTable = TableOutput.createTable( ( "Flags",
                                                 "Value",
                                                 "TimeTaken( in sec )" ) )
         flagsTable.formatColumns( fl, fl, fr )
         routesTable = TableOutput.createTable( ( "RouteType",
                                                  "Configured",
                                                  "Config Pending",
                                                  "Ready",
                                                  "TimeTaken( in sec )" ) )
         routesTable.formatColumns( fl, fl, fl, fl, fr )

         self.flags.renderEntry( flagsTable )

         for rtType in self.routes:
            self.routes[ rtType ].renderEntry( rtType, routesTable )

         print( flagsTable.output() )
         print( "" )
         print( routesTable.output() )

      else:
         print( "Start time : %s( %s )" %
            ( Ark.timestampToStr( self.startTime, relative=False, now=Tac.utcNow() ),
              Ark.timestampToStr( self.startTime, now=Tac.utcNow() ) ) )
         print( "Grestart Init : %d" % ( 1 if self.gracefullyRestarting else 0 ) )
         print( "" )

         stateTable = TableOutput.createTable( ( "State",
                                                 "State Reached",
                                                 "Time Taken( in sec )" ) )
         stateTable.formatColumns( fl, fl, fr )
         protocolReadyTable = TableOutput.createTable( ( "Protocol",
                                                         "Config Pending",
                                                         "Configured",
                                                         "Ready",
                                                         "Time Taken( in sec )" ) )
         protocolReadyTable.formatColumns( fl, fl, fl, fl, fr )

         self.states.renderEntry( stateTable )

         for protocol in self.protocols:
            self.protocols[ protocol ].renderEntry( protocol, protocolReadyTable )

         print( stateTable.output() )
         print( "" )
         print( protocolReadyTable.output() )
