#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from TableOutput import createTable, TableFormatter, Format, FormattedCell
from CliModel import Bool, Int, List, Str, Dict, Model, Float, Enum

tableWidth = 200
maxNameLen = 100

class SlabAllocatorStatsPerThreadModel( Model ):
   threadName = Str( help='The name of the thread the statistics pertains to' )
   blocksInUse = Int( help='Number of blocks in use' )
   blocksFree = Int( help='Number of blocks free' )
   pages = Int( help='Number of pages used by this allocator in this thread' )
   pagesHighWaterMark = Int( help='Largest number of pages used by this allocator '
                             'in this thread' )
   fragmentationScore = Float( help="Fragmentation score: pages*free/used" )
   pagesWithBlocksFreeCount = \
      Dict( keyType=int, valueType=int,
            help='Key is number of blocks in a page that is free, value is the ' \
                 'number of pages with that number of blocks free' )

class SlabAllocatorStatsModel( Model ):
   name = Str( help='The name of the allocator, "normal", "transient" or custom ' \
                    'name assigned' )
   blockSize = Int( help='The size of blocks allocated from this allocator' )
   maxBlocksPerPage = \
      Int( help='The maximum number of blocks we can allocate from a single page' )
   # The per thread state is a list, as we cannot guarantee the threadName is unique.
   perThread = List( valueType=SlabAllocatorStatsPerThreadModel,
                     help='Per thread statistics for this allocator' )

class SlabAllocatorEmptyPagesPerThreadModel( Model ):
   threadName = Str( help='The name of the thread the statistics pertains to' )
   emptyPages = Int( help='Number of empty pages' )

class SlabAllocatorAgentModel( Model ):
   allocators = \
      Dict( keyType=str, valueType=SlabAllocatorStatsModel,
            help='Per allocator statistics, keyed by unique name of the allocator' )
   pageSize = Int( help='Size of the internal pages in bytes' )
   mmapPages = Int( help='Number of pages of virtual address mmap\'d by '
                         'the slab allocator' )
   touchedPages = Int( help='Number of pages touched by the slab allocator' )
   allocatedPages = Int( help='Number of pages allocated to individual '
                              'allocators or held in per thread free caches' )
   emptyPages = Int( help='Number of pages that are empty, but dirty' )
   releasedPages = Int( help='Number of pages that were used in the past, but '
                             'released back to the kernel, these do not '
                             'consume physical memory' )
   emptyPagesPerThread = List( valueType=SlabAllocatorEmptyPagesPerThreadModel,
                                   help='Number of empty pages in the per thread '
                                        'empty page cache' )

   @staticmethod
   def humanSize( value ):
      if value < 1024:
         return f'{value}B '
      value = float( value )
      value /= 1024
      if value < 1024:
         return f'{value:.2f}KB'
      value /= 1024
      if value < 1024:
         return f'{value:.2f}MB'
      value /= 1024
      return f'{value:.2f}GB'

   def pagesHumanSize( self, pageCount ):
      value = pageCount * int( self.pageSize / 1024 )
      if value < 1024:
         return f'{value}KB'
      value = float( value )
      value /= 1024
      if value < 1024:
         return f'{value:.2f}MB'
      value /= 1024
      return f'{value:.2f}GB'

   @staticmethod
   def shortPagesWithBlocksFreeCount( perThread ):
      if not perThread.pagesWithBlocksFreeCount:
         # Display as blank
         return ( ( '', '' ), )
      return sorted( perThread.pagesWithBlocksFreeCount.items(),
                     key=lambda tup: tup[ 0 ] )

   @staticmethod
   def percent( value, total ):
      if value == '':
         # Special case from shortPagesWithBlocksFreeCount() when we have no data.
         return ''
      assert isinstance( value, int )
      assert isinstance( total, int )
      if total == 0:
         return ''
      return f'{float( value * 100 ) / total:0.1f}%'

   def renderAllocatorPerThread( self, renderFragmentation, sortByFragScore,
                                 table, allocator, perThread,
                                 col0, col1, col2 ):
      table.newRow( col0, col1, col2,
                    perThread.blocksInUse,
                    perThread.blocksFree,
                    self.percent( perThread.blocksFree,
                                  perThread.blocksFree +
                                  perThread.blocksInUse ),
                    self.humanSize( allocator.blockSize *
                                    perThread.blocksInUse ),
                    self.humanSize( allocator.blockSize *
                                    perThread.blocksFree ),
                    self.humanSize( perThread.pages * self.pageSize -
                                    perThread.blocksInUse * allocator.blockSize -
                                    perThread.blocksFree * allocator.blockSize ),
                    self.pagesHumanSize( perThread.pages ),
                    perThread.pages,
                    perThread.pagesHighWaterMark )
      if not renderFragmentation and not sortByFragScore:
         return
      if renderFragmentation:
         pagesWithBlocksFreeCount = \
            self.shortPagesWithBlocksFreeCount( perThread )
         table.newCell( pagesWithBlocksFreeCount[ 0 ][ 0 ] )
         table.newCell( self.percent( pagesWithBlocksFreeCount[ 0 ][ 0 ],
                                      allocator.maxBlocksPerPage ) )
         table.newCell( pagesWithBlocksFreeCount[ 0 ][ 1 ] )
      if sortByFragScore:
         # pylint: disable-next=consider-using-f-string
         table.newCell( "%6.1f" % perThread.fragmentationScore )
      if renderFragmentation:
         for blocksFree, pageCount in pagesWithBlocksFreeCount[ 1 : ]:
            table.newRow( FormattedCell( content='', nCols=12, format=None ),
                          blocksFree,
                          self.percent( blocksFree, allocator.maxBlocksPerPage ),
                          pageCount )

   def doRender( self, renderFragmentation, sortByFragScore ):
      fl = Format( justify="left", wrap=True )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify="right", wrap=True )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      fPad = Format( justify='center', pad='.' )

      # We can have multiple self.emptyPagesPerThread when using
      # ArSlab::SlabAllocatorThreadUnsafe as each unique named
      # SlabAllocatorThreadUnsafe creates its own "per thread" page
      # manager. Thus don't look at how many entries we have in
      # self.emptyPagesPerThread to determine if we are multi
      # threaded, instead see if any single allocator has more than
      # one per thread entry.
      multiThreaded = any( len( allocator.perThread ) > 1
                           for allocator in self.allocators.values() )
     
      # Model is stuctured allocatorSizes/threadNames and displayed in that order
      # (loop in loop) but when sorting by fragmentation score, we need to flatten
      # it to sort it (inner loop becomes degenerated), which also means adding the
      # thread-name as a key. We also add the thread name into the allocator name
      # and set multiThreaded to False (no need to use 2 lines: allocator summary and
      # thread details).
      def deThread( allocators  ):
         flatAllocators = {}
         for allocator in allocators.values():
            for pt in allocator.perThread:
               allocatorModel = SlabAllocatorStatsModel()
               allocatorModel.blockSize = allocator.blockSize
               allocatorModel.maxBlocksPerPage = allocator.maxBlocksPerPage
               allocatorModel.perThread.append( pt )
               if not pt.threadName:
                  pt.threadName = "noName"
               allocatorModel.name = allocator.name + "/" + pt.threadName
               key = f"{allocator.name}-{str( allocator.blockSize )}"
               flatAllocators[ key ] = allocatorModel
         self.allocators = flatAllocators
      if sortByFragScore and multiThreaded:
         deThread( self.allocators )
         multiThreaded = False

      table = createTable( ( '', 'Pages', 'Memory' ) )
      table.formatColumns( fl, fr )
      table.newRow( 'mmap\'d', self.mmapPages,
                    self.pagesHumanSize( self.mmapPages ) )
      table.newRow( 'Touched', self.touchedPages,
                    self.pagesHumanSize( self.touchedPages ) )
      table.newRow( 'Allocated', self.allocatedPages,
                    self.pagesHumanSize( self.allocatedPages ) )
      table.newRow( 'Empty madvise\'d', self.releasedPages,
                    self.pagesHumanSize( self.releasedPages ) )
      table.newRow( 'Empty dirty', self.emptyPages,
                    self.pagesHumanSize( self.emptyPages ) )
      if len( self.emptyPagesPerThread ) > 1:
         for perThread in self.emptyPagesPerThread:
            table.newRow( 'Thread empty cache: ' + perThread.threadName,
                          perThread.emptyPages,
                          self.pagesHumanSize( perThread.emptyPages ) )
      elif self.emptyPagesPerThread:
         pageCount = self.emptyPagesPerThread[ 0 ].emptyPages
         table.newRow( 'Empty cache', pageCount,
                       self.pagesHumanSize( pageCount ) )
      print( table.output(), '\n' )

      # Don't use createTable, we want to use pad='.' for the multi
      # column headings to make it more readable.
      table = TableFormatter( indent=0, tableWidth=tableWidth )
      table.formatColumns( fl,
                           *[ fr ] * ( 13 if renderFragmentation else 10 ) )

      table.startRow( Format( isHeading=True, border=False ) )
      if multiThreaded:
         table.newCell( 'Allocator Name' )
      else:
         table.newCell( '' )
      table.newCell( 'Block' )
      table.newFormattedCell( 'Blocks', nCols=4, format=fPad )
      table.newFormattedCell( 'Memory', nCols=4, format=fPad )
      table.newCell( 'Total' )
      table.newCell( 'Peak' )
      if renderFragmentation or sortByFragScore:
         if sortByFragScore and renderFragmentation:
            table.newFormattedCell( 'Fragmentation', nCols=4, format=fPad )
         elif renderFragmentation:
            table.newFormattedCell( 'Fragmentation', nCols=3, format=fPad )
         else:
            table.newFormattedCell( 'Fragmentation', nCols=1, format=fPad )

      table.startRow( Format( isHeading=True, border=True ) )
      if multiThreaded:
         table.newCell( '  Thread Name' )
      else:
         table.newCell( 'Allocator Name' )
      table.newCell( 'Size' )
      table.newCell( 'Per Page' )
      table.newCell( 'In Use' )
      table.newCell( 'Free' )
      table.newCell( '%Free' )
      table.newCell( 'In Use' )
      table.newCell( 'Free' )
      table.newCell( 'Overhead' )
      table.newCell( 'Total' )
      table.newCell( 'Pages' )
      table.newCell( 'Pages' )
      if renderFragmentation:
         table.newCell( 'Free Blocks' )
         table.newCell( '%Free' )
         table.newCell( 'Pages' )
      if sortByFragScore:
         table.newCell( 'Score' )

      def getSortedAllocators( allocators, sortByFragScore ):
         if sortByFragScore:
            return sorted( self.allocators.values(),
                           key=lambda a: ( -a.perThread[ 0 ].fragmentationScore,
                                           a.perThread[ 0 ].blocksFree / (
                                           a.perThread[ 0 ].blocksInUse + 0.01 ) ) )
         # Sort to display custom at the end.
         return sorted( self.allocators.values(),
                  key=lambda a: ( 0 if a.name in ( 'normal', 'transient' ) else 1,
                                  a.name,
                                  a.blockSize ) )

      for allocator in getSortedAllocators(self.allocators, sortByFragScore):
         if not multiThreaded:
            self.renderAllocatorPerThread( renderFragmentation, sortByFragScore,
                                           table,
                                           allocator,
                                           allocator.perThread[ 0 ],
                                           allocator.name[ : maxNameLen ],
                                           allocator.blockSize,
                                           allocator.maxBlocksPerPage )
         else:
            table.newRow( allocator.name[ : maxNameLen ],
                          allocator.blockSize,
                          allocator.maxBlocksPerPage )
            if renderFragmentation:
               table.newFormattedCell( '', nCols=3, format=fl )
            for perThread in allocator.perThread:
               self.renderAllocatorPerThread( renderFragmentation, sortByFragScore,
                                              table,
                                              allocator,
                                              perThread,
                                              '  ' + perThread.threadName,
                                              '',
                                              '' )

      threads = self.getPerThreadTotals()
      for threadName, thread in sorted( threads.items(), key=lambda tup: tup[ 0 ] ):
         table.newRow( 'Total ' + threadName if len( threads ) > 1 else 'Total',
                       '', # Block size
                       '', # Blocks per page
                       thread.blocksInUse,
                       thread.blocksFree,
                       self.percent( thread.blocksFree,
                                     thread.blocksInUse + thread.blocksFree ),
                       self.humanSize( thread.memoryInUse ),
                       self.humanSize( thread.memoryFree ),
                       self.humanSize( thread.pages * self.pageSize -
                                       ( thread.memoryInUse + thread.memoryFree ) ),
                       self.pagesHumanSize( thread.pages ),
                       thread.pages )
         if renderFragmentation:
            table.newFormattedCell( '', nCols=3, format=fl )

      print( table.output(), '\n' )

   def getPerThreadTotals( self ):
      class PerThreadTotal:
         def __init__( self, blocksInUse, blocksFree, memoryInUse, memoryFree,
                       pages ):
            self.blocksInUse = blocksInUse
            self.blocksFree = blocksFree
            self.memoryInUse = memoryInUse
            self.memoryFree = memoryFree
            self.pages = pages

         @staticmethod
         def create( blockSize, model ):
            return PerThreadTotal( model.blocksInUse,
                                   model.blocksFree,
                                   model.blocksInUse * blockSize,
                                   model.blocksFree * blockSize,
                                   model.pages )

         def __add__( self, rhs ):
            return PerThreadTotal( self.blocksInUse + rhs.blocksInUse,
                                   self.blocksFree + rhs.blocksFree,
                                   self.memoryInUse + rhs.memoryInUse,
                                   self.memoryFree + rhs.memoryFree,
                                   self.pages + rhs.pages )

      threads = dict() # pylint: disable=use-dict-literal
      for allocator in self.allocators.values():
         for perThread in allocator.perThread:
            pt = PerThreadTotal.create( allocator.blockSize, perThread )
            thread = threads.get( perThread.threadName )
            if thread:
               threads[ perThread.threadName ] = thread + pt
            else:
               threads[ perThread.threadName ] = pt
      return threads

class SlabAllocatorModel( Model ):
   agents = Dict( keyType=str, valueType=SlabAllocatorAgentModel,
                 help='Per agent statistics, keyed by agent name' )
   errorAgents = Dict( keyType=str, valueType=str,
                      help='Errors encountered, keyed by agent name' )
   _renderDetail = Bool( help='If True the text render includes per allocator '
                              'information, otherwise it just includes the total' )
   _renderFragmentation = Bool( help='If True the text render includes detailed '
                                     'statistics regarding page fragmentation' )
   _sortByFragScore = Bool( help='If True sort by fragmentation score' )
   _renderSort = Str( help='Sort instructions for text render' )

   def renderDetail( self ):
      # pylint: disable-msg=protected-access
      return self._renderDetail

   def renderFragmentation( self ):
      # pylint: disable-msg=protected-access
      return self._renderFragmentation

   def sortByFragScore( self ):
      # pylint: disable-msg=protected-access
      return self._sortByFragScore

   def renderSort( self ):
      # pylint: disable-msg=protected-access
      return self._renderSort

   def render( self ):
      if self.errorAgents:
         print( 'Error getting statistics from:' )
         table = createTable( ( 'Agent Name', 'Error' ) )
         table.formatColumns( * [ Format( justify="left", wrap=True ) ] * 2 )
         for agent in sorted( self.errorAgents ):
            table.newRow( agent, self.errorAgents[ agent ] )
         print( table.output(), '\n' )

      if self.renderDetail():
         for agent in sorted( self.agents ):
            print( agent )
            self.agents[ agent ].doRender( self.renderFragmentation(),
                                           self.sortByFragScore() )
         return

      self.doRenderSummary()

   def doRenderSummary( self ):
      if not self.agents:
         return

      fl = Format( justify="left", wrap=True )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify="right", wrap=True )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      fPad = Format( justify='center', pad='.' )

      # Don't use createTable, we want to use pad='.' for the multi
      # column headings to make it more readable.
      table = TableFormatter( indent=0, tableWidth=tableWidth )
      table.formatColumns( fl, *[ fr ] * 15 )

      table.startRow( Format( isHeading=True, border=False ) )
      table.newCell( '' )
      table.newFormattedCell( 'Blocks', nCols=3, format=fPad )
      table.newFormattedCell( 'Memory', nCols=8, format=fPad )

      table.startRow( Format( isHeading=True, border=True ) )
      table.newCell( 'Agent' )
      table.newCell( 'In Use' )
      table.newCell( 'Free' )
      table.newCell( '%Free' )
      table.newCell( 'In Use' )
      table.newCell( 'Free' )
      table.newCell( '%Free' )
      table.newCell( 'Overhead' )
      table.newCell( 'Total' )
      table.newCell( 'mmap\'d' )
      table.newCell( 'madvise\'d' )
      table.newCell( 'Empty Dirty' )

      totalBlocksInUse = 0
      totalBlocksFree = 0
      totalMemoryInUse = 0
      totalMemoryFree = 0
      totalPages = 0
      totalMmapPages = 0
      totalReleasedPages = 0
      totalEmptyPages = 0

      pageSize = None

      lines = list() # pylint: disable=use-list-literal
      for agentName, agentModel in self.agents.items():
         if pageSize is None:
            pageSize = agentModel.pageSize
         else:
            assert pageSize == agentModel.pageSize
         perThreadTotals = list( agentModel.getPerThreadTotals().values() )
         if not perThreadTotals:
            continue
         # We cannot use sum(), it results in
         # TypeError: unsupported operand type(s) for +: 'int' and 'PerThreadTotal'
         total = perThreadTotals[ 0 ]
         for t in perThreadTotals[ 1 : ]:
            total += t
         emptyPages = ( agentModel.emptyPages +
                        sum(  perThread.emptyPages
                               for perThread in
                               agentModel.emptyPagesPerThread  ) )
         totalBlocksInUse += total.blocksInUse
         totalBlocksFree += total.blocksFree
         totalMemoryInUse += total.memoryInUse
         totalMemoryFree += total.memoryFree
         totalPages += total.pages
         totalMmapPages += agentModel.mmapPages
         totalReleasedPages += agentModel.releasedPages
         totalEmptyPages += emptyPages

         def calcPercent( value, total ):
            if total == 0:
               return 0
            return float( 100 * value ) / total

         lines.append( ( agentModel,
                         agentName,
                         total.blocksInUse,
                         total.blocksFree,
                         calcPercent( total.blocksFree,
                                      total.blocksInUse + total.blocksFree ),
                         total.memoryInUse,
                         total.memoryFree,
                         calcPercent( total.memoryFree,
                                      total.memoryInUse + total.memoryFree ),
                         ( total.pages * agentModel.pageSize -
                           total.memoryInUse - total.memoryFree ),
                         total.pages,
                         agentModel.mmapPages,
                         agentModel.releasedPages,
                         emptyPages ) )
      renderSort = self.renderSort()
      sortIdx = None
      sortReverse = True
      if not renderSort:
         sortIdx = 1 # agentName
         sortReverse = False
      elif renderSort == 'blocks-in-use':
         sortIdx = 2
      elif renderSort == 'blocks-free':
         sortIdx = 3
      elif renderSort == 'blocks-percent-free':
         sortIdx = 4
      elif renderSort == 'memory-in-use':
         sortIdx = 5
      elif renderSort == 'memory-free':
         sortIdx = 6
      elif renderSort == 'memory-percent-free':
         sortIdx = 7
      elif renderSort == 'memory-overhead':
         sortIdx = 8
      elif renderSort == 'memory-total':
         sortIdx = 9
      elif renderSort == 'memory-mmap':
         sortIdx = 10
      elif renderSort == 'memory-madvise':
         sortIdx = 11
      elif renderSort == 'memory-dirty':
         sortIdx = 12
      else:
         # pylint: disable-next=consider-using-f-string
         assert False, 'unexpected renderSort %s' % renderSort

      for tup in sorted( lines,
                         key=lambda tup: tup[ sortIdx ],
                         reverse=sortReverse ):
         agentModel = tup[ 0 ]
         table.newRow( tup[ 1 ], # agentName
                       tup[ 2 ], # blocksInUse
                       tup[ 3 ], # blocksFree
                       agentModel.percent( tup[ 3 ], tup[ 2 ] + tup[ 3 ] ),
                       agentModel.humanSize( tup[ 5 ] ), # memoryInUse
                       agentModel.humanSize( tup[ 6 ] ), # memoryFree
                       agentModel.percent( tup[ 6 ], tup[ 5 ] + tup[ 6 ] ),
                       agentModel.humanSize( tup[ 9 ] * agentModel.pageSize -
                                             tup[ 5 ] - tup[ 6 ] ),
                       agentModel.pagesHumanSize( tup[ 9 ] ), # pages
                       agentModel.pagesHumanSize( tup[ 10 ] ), # mmap
                       agentModel.pagesHumanSize( tup[ 11 ] ), # madvise
                       agentModel.pagesHumanSize( tup[ 12 ] )  # dirty
                      )
      table.newRow( 'Total',
                    totalBlocksInUse,
                    totalBlocksFree,
                    agentModel.percent( totalBlocksFree,
                                        totalBlocksInUse + totalBlocksFree ),
                    agentModel.humanSize( totalMemoryInUse ),
                    agentModel.humanSize( totalMemoryFree ),
                    agentModel.percent( totalMemoryFree,
                                        totalMemoryInUse + totalMemoryFree ),
                    agentModel.humanSize( totalPages * agentModel.pageSize -
                                          totalMemoryInUse -
                                          totalMemoryFree ),
                    agentModel.pagesHumanSize( totalPages ),
                    agentModel.pagesHumanSize( totalMmapPages ),
                    agentModel.pagesHumanSize( totalReleasedPages ),
                    agentModel.pagesHumanSize( totalEmptyPages ) )

      print( table.output(), '\n' )

class SlabAllocatorBlockModel( Model ):
   data = Int( help='One pointers worth of data from the block, for blocks in use '
                    'it is from the start of the block, for free blocks it starts '
                    '8 bytes into the block. For blocks not large enough this is '
                    'always zero' )
   onFreeList = Bool( help='Is the block free' )
   count = Int( help='Number of blocks seen with this data & onFreeList' )
   dataSymbolName = Str( help='The symbol name corresponding to key.data' )

class SlabAllocatorContentPerThreadModel( Model ):
   threadName = Str( help='The name of the thread the statistics pertains to' )
   blocks = List( valueType=SlabAllocatorBlockModel,
                  help='Blocks in partially filled pages' )

class SlabAllocatorContentModel( Model ):
   name = Str( help='The name of the allocator, "normal", "transient" or custom '
                    'name assigned' )
   uniqueName = Str( help='Unique name of the allocator' )
   blockSize = Int( help='The size of blocks allocated from this allocator' )
   maxBlocksPerPage = \
      Int( help='The maximum number of blocks we can allocate from a single page' )
   # The per thread state is a list, as we cannot guarantee the threadName is unique.
   perThread = List( valueType=SlabAllocatorContentPerThreadModel,
                     help='Per thread statistics for this allocator' )
   _renderDetail = Bool( help='If True the text render includes block data' )

   def renderDetail( self ):
      # pylint: disable-msg=protected-access
      return self._renderDetail

   def render( self ):
      # if self.renderDetail():
      #    self.doRenderDetail
      #    return

      fl = Format( justify="left", wrap=True )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify="right", wrap=True )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      fPad = Format( justify='center', pad='.' )
      displayThreads = ( len( self.perThread ) != 1 )
      align = ( fl, fr, fr )
      if self.renderDetail():
         align = ( fl, ) + align
      if displayThreads:
         align = ( fl, ) + align
      table = TableFormatter( indent=0, tableWidth=tableWidth )
      table.formatColumns( *align )
      table.startRow( Format( isHeading=True, border=False ) )
      if displayThreads:
         table.newCell( '' )
      if self.renderDetail():
         table.newFormattedCell( 'Symbol', nCols=2, format=fPad )
      else:
         table.newCell( '' )
      table.newFormattedCell( 'Blocks', nCols=2, format=fPad )
      table.startRow( Format( isHeading=True, border=True ) )
      if displayThreads:
         table.newCell( 'Thread' )
      if self.renderDetail():
         table.newFormattedCell( 'Address', nCols=1, format=fl )
      table.newFormattedCell( 'Name', nCols=1, format=fl )
      table.newFormattedCell( 'Free', nCols=1, format=fl )
      table.newFormattedCell( 'In Use', nCols=1, format=fl )

      for perThread in sorted( self.perThread, key=lambda pt: pt.threadName ):
         threadRow = ( perThread.threadName, ) if displayThreads else tuple()
         if self.renderDetail():
            for block in sorted( perThread.blocks, key=lambda b: -1 * b.count ):
               row = threadRow + ( f'0x{block.data:016x}',
                                   block.dataSymbolName,
                                   block.count if block.onFreeList else 0,
                                   0 if block.onFreeList else block.count )
               table.newRow( *row )
               threadRow = ( '', ) if displayThreads else tuple()
         else:
            d = dict() # pylint: disable=use-dict-literal
            for block in perThread.blocks:
               name = block.dataSymbolName
               name = name.replace( 'vtable for ', '' )
               name = name.replace( 'typeinfo for ', '' )
               freeCount = block.count if block.onFreeList else 0
               inUseCount = 0 if block.onFreeList else block.count
               tup = d.get( name )
               if tup is None:
                  d[ name ] = ( freeCount, inUseCount )
               else:
                  d[ name ] = ( tup[ 0 ] + freeCount,
                                tup[ 1 ] + inUseCount )
            for name, ( freeCount, inUseCount ) in \
                sorted( d.items(),
                        key=lambda tup: -( tup[ 1 ][ 0 ] + tup[ 1 ][ 1 ] ) ):
               row = threadRow + ( name, freeCount, inUseCount )
               table.newRow( *row )
               threadRow = ( '', ) if displayThreads else tuple()

      print( table.output(), '\n' )

   def doRenderDetail( self ):
      fl = Format( justify="left", wrap=True )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify="right", wrap=True )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      fPad = Format( justify='center', pad='.' )
      displayThreads = ( len( self.perThread ) != 1 )
      align = ( fr, fl, fr, fr )
      if displayThreads:
         align = ( fl, ) + align
      table = TableFormatter( indent=0, tableWidth=tableWidth )
      table.formatColumns( *align )
      table.startRow( Format( isHeading=True, border=False ) )
      if displayThreads:
         table.newCell( '' )
      table.newFormattedCell( 'Symbol', nCols=2, format=fPad )
      table.newFormattedCell( 'Blocks', nCols=2, format=fPad )
      table.startRow( Format( isHeading=True, border=True ) )
      if displayThreads:
         table.newCell( 'Thread' )
      table.newFormattedCell( 'Address', nCols=1, format=fl )
      table.newFormattedCell( 'Name', nCols=1, format=fl )
      table.newFormattedCell( 'Free', nCols=1, format=fl )
      table.newFormattedCell( 'In Use', nCols=1, format=fl )

      for perThread in sorted( self.perThread, key=lambda pt: pt.threadName ):
         threadRow = ( perThread.threadName, ) if displayThreads else tuple()
         for block in sorted( perThread.blocks, key=lambda b: -1 * b.count ):
            row = threadRow + ( f'0x{block.data:016x}',
                                block.dataSymbolName,
                                block.count if block.onFreeList else 0,
                                0 if block.onFreeList else block.count )
            table.newRow( *row )
            threadRow = ( '', ) if displayThreads else tuple()
      print( table.output(), '\n' )

class SlabAllocatorMappedRangeModel( Model ):
   rangeStart = Int( help='Memory address of the start of the mapped range' )
   rangeEnd = Int( help='Memory address of the end of the mapped range' )

   def render( self ):
      print( f'{self.rangeStart:#016x}-{self.rangeEnd:#016x}' )

class SlabAllocatorPageModel( Model ):
   pageStart = Int( help='Memory address of the start of the page' )
   pageState = Enum( values=( 'empty',
                              'madvised',
                              'emptyThreadCache1',
                              'emptyThreadCache2',
                              'full',
                              'partiallyFull' ),
                     help='The state of the page' )
   name = Str( help='Name of the thread and/or allocator the page belongs to, '
                    'not populated for empty and madvised',
               optional=True )
   blockSize = Int( help='Size of blocks allocated from this page. '
                         'Only populated in full and partiallyFull pages',
                    optional=True )
   blocksPerPage = Int( help='The number of blocks that fit in one page. '
                             'Only populated in full and partiallyFull pages',
                        optional=True )
   freeBlockCount = Int( help='The number of blocks that are free in the page. '
                              'Only populated in full and partiallyFull pages',
                         optional=True )

   def addRow( self, table ):
      table.newRow( f'{self.pageStart:#016x}',
                    self.pageState,
                    self.name if self.name else '',
                    self.blockSize if self.blockSize else '',
                    self.blocksPerPage if self.blocksPerPage else '',
                    self.freeBlockCount if self.freeBlockCount else '',
                    f'{self.freeBlockCount/self.blocksPerPage:.1%}'
                    if self.freeBlockCount else '' )

class SlabAllocatorPagesStatsModel( Model ):
   mappedRanges = List( valueType=SlabAllocatorMappedRangeModel,
                        help='List of all mapped memory ranges used by the '
                             'SlabAllocator' )
   mappedRangesCoalesced = List( valueType=SlabAllocatorMappedRangeModel,
                                 help='Coalesced list of all mapped memory ranges '
                                      'used by the SlabAllocator' )
   pages = List( valueType=SlabAllocatorPageModel,
                 help='List of all pages touched by the SlabAllocator' )

   def render( self ):
      print( 'Mapped ranges:' )
      for mappedRange in self.mappedRanges:
         mappedRange.render()

      print( 'Mapped ranges, coalesced:' )
      for mappedRange in self.mappedRangesCoalesced:
         mappedRange.render()

      print( '\nPages:' )
      table = TableFormatter( indent=0 )
      fl = Format( justify="left", wrap=True )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify="right", wrap=True )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      fPad = Format( justify='center', pad='.' )
      table.formatColumns( fl, fl, fl, fr, fr, fr, fr )
      table.startRow( Format( isHeading=True, border=False ) )
      table.newCell( '' )
      table.newCell( '' )
      table.newCell( '' )
      table.newFormattedCell( 'Blocks', nCols=3, format=fPad )
      table.newCell( '' )
      table.startRow( Format( isHeading=True, border=True ) )
      table.newCell( 'Page address' )
      table.newCell( 'State' )
      table.newCell( 'Name' )
      table.newCell( 'Size' )
      table.newCell( 'Per page' )
      table.newCell( 'Free count' )
      table.newCell( 'Free %' )
      for page in self.pages:
         page.addRow( table )
      print( table.output(), '\n' )
