# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliParser import InvalidInputError, AlreadyHandledError
from CliModel import Model, Dict, Str, Int, Float, Submodel, Enum, List, Bool

priorities = (
      'critical',
      'absolute',
      'high',
      'normal',
      'low',
      'scavenge',
      )

class PriorityStats( Model ):
   runnableTasks = Int( help="Runnable tasks with this priority" )
   idleTasks = Int( help="Idle tasks with this priority" )
   completedTasks = Int( help="Completed tasks with this priority" )
   totalRunCount = Int( help="Total run count" )

class WorkCount( Model ):
   total = Int( help="Total work done" )
   decayingAvg = Float( help="Work count decaying average" )
   highwater = Int( help="Work count high water mark" )

class WorkPending( Model ):
   pending = Int( help="Pending work" )
   highwater = Int( help="Pending work high water mark" )

class Task( Model ):
   name = Str( help="Task name" )
   address = Str( help="Memory address of live task", optional=True )
   priority = Enum( values=priorities, help="Task priority" )
   subScheduler = Str( optional=True,
                       help="If the task is a nested hierarchical scheduler, "
                       "this is identifier in the TaskScheduler.subSchedulers "
                       "collection" )
   onRunQueue = Bool( help="Task is on run queue" )
   currentlyRunning = Bool( help="Task is currently running" )
   suspended = Bool( help="Task is suspended" )
   schedulerSuspended = Bool( help="Direct or indirect parent scheduler is "
                                   "suspended",
                              optional=True )
   hasTimer = Bool( help="Task has an associated timer" )
   timerRunning = Bool( help="Task's associated timer is running" )
   runCount = Int( help="Number of times the Task has run" )
   totalRunTime = Float( help="Task total run time, in seconds" )
   firstTaskRunTimestamp = Int( help="Timestamp of the first time the task ran, "
                                "in nanoseconds since system boot. If the task "
                                "has not run it will be zero." )
   lastTaskRunTimestamp = Int( help="Timestamp of the most recent time the task ran"
                               ", in nanoseconds since system boot. If the task "
                               " as not run it will be zero." )
   lastSchedulingLatency = Float( help="Tasks's last scheduling latency, "
                                  "in seconds" )
   schedulingLatencyHighWaterMark = Float( help="Tasks's scheduling latency "
                                                "high water mark, in seconds" )
   qv200 = Int( help="Number of times task used 200% of quantum" )
   qv500 = Int( help="Number of times task used 500% of quantum" )
   maxMs = Int( help="Maximum task run time (in ms)" )
   # completedCount is only present in completedTasks
   completedCount = Int( optional=True, help="The number of tasks by this name "
                                             "that have completed" )
   workCount = Submodel( WorkCount, optional=True, help="Work count" )
   workPending = Submodel( WorkPending, optional=True, help="Pending work" )

   def statusStr( self ):
      # The 2-letter task status string:
      # first: T/t for active/idle timers, * for running task, blank or
      #        S for suspended tasks, X for tasks under suspended scheduler
      # second: priority and runnable status, upper case for runnable
      if self.currentlyRunning:
         firstChar = '*'
      elif self.suspended:
         firstChar = 'S'
      elif self.schedulerSuspended:
         firstChar = 'X'
      elif self.hasTimer:
         if self.timerRunning:
            firstChar = 'T'
         else:
            firstChar = 't'
      else:
         firstChar = ' '
      secondChar = str( self.priority )[ 0 ]
      if self.onRunQueue:
         secondChar = secondChar.upper()
      return firstChar + secondChar

class TaskList( Model ):
   tasks = List( valueType=Task, help="List of tasks" )
   skipped = Int( help="Skipped tasks with no runs" )

   def printTasks( self, title, displayAddr=False, completedCount=False ):
      addressLen = 0
      if displayAddr and self.tasks and self.tasks[ 0 ].address:
         addressLen = len( self.tasks[ 0 ].address )
      maxTaskNameLen = max(
         [ len( title ) ] +
         [ len( task.name ) + addressLen for task in self.tasks ] )
      hdrFmt = "%-" + str( maxTaskNameLen ) + "s %2s %10s %13s %14s %4s %4s %5s"
      hdrArgs = [ title, "SP", "RunCount", "TotalSec", "Last/Max-Delay",
                         "200%", "500%", "MaxMs" ]
      fmt = "%-" + str( maxTaskNameLen ) + "s %2s %10u %13.3f %6.3f %7.3f " \
            "%4u %4u %5u"
      if completedCount:
         hdrFmt += " %4s"
         hdrArgs += [ "Done" ]
         fmt += " %4u"
      hdrFmt += " %26s %20s"
      hdrArgs += [ "Total/DecayA/Max-WorkCount", "Last/Max-WorkPending" ]
      fmt += " %10s %8s %6s %9s %10s"

      print( hdrFmt % tuple( hdrArgs ) )
      for task in sorted( self.tasks, key=lambda t: ( -t.totalRunTime, t.name ) ):
         title = task.name
         if displayAddr and task.address:
            title += task.address
         args = [ title, task.statusStr(), task.runCount, task.totalRunTime,
               task.lastSchedulingLatency, task.schedulingLatencyHighWaterMark,
               task.qv200, task.qv500, task.maxMs ]
         if completedCount:
            args += [ task.completedCount ]
         if task.workCount:
            args += [ task.workCount.total, "%6.1f" % task.workCount.decayingAvg,
                  task.workCount.highwater ]
         else:
            args += [ "-", "-", "-" ]
         if task.workPending:
            args += [ task.workPending.pending, task.workPending.highwater ]
         else:
            args += [ "-", "-" ]
         print( fmt % tuple( args ) )
      if self.skipped:
         # pylint: disable-next=bad-string-format-type
         print( "Skipped %u task%s with no runs" % ( self.skipped,
            "s" if self.skipped > 1 else "" ) )
      print()

class HistoryEvent( Model ):
   nsSinceBoot = Int( help="Event time, in nanoseconds since system boot" )
   eventType = Str( help="Which type of event" )
   eventId = Str( optional=True, help="The object associated with the event" )

class History( Model ):
   history = List( valueType=HistoryEvent, help="History events" )
   historyEntryCount = Int( help="Max number of entries in circular history" )
   historyEventCount = Int( help="The number of events in circular history" )
   def render( self ):
      head = "Last %d scheduler events, absolute timestamp and " \
             "relative to last event" % ( self.historyEventCount )
      print( head )
      print( "-" * len( head ) )
      firstEventNs = None
      for entry in self.history:
         if firstEventNs is None:
            firstEventNs = entry.nsSinceBoot
         if entry.eventId:
            typeAndId = "%-12s %s" % ( entry.eventType, entry.eventId )
         else:
            typeAndId = entry.eventType
         ago = ( firstEventNs - entry.nsSinceBoot + 500 ) / 1000
         print( "%22d ns since boot %10d us %s" % ( entry.nsSinceBoot,
                                                    ago, typeAndId ) )

class TaskScheduler( Model ):
   schedulerName = Str( help="Name of the scheduler" )
   generalError = Str( help="Error encountered running show command",
                       optional=True )
   taskNameRegex = Str( help="Regular expression used to filter task names",
                        optional=True )
   taskNameRegexError = Str( help="If 'taskNameRegex' is supplied and is not a "
                             "valid regular expression, this will contain an error "
                             "message for diagnostic purposes.",
                        optional=True )
   displayOrder = Int( help="Rendered text output sorts multiple schedulers in "
                       "ascending order of ( displayOrder, schdulerName )",
                       optional=True )
   isRootScheduler = Bool( help="This is a root scheduler" )
   priorityStats = Dict( keyType=str, valueType=PriorityStats,
                         help="Task counts and statistics indexed by priority" )
   maxSchedulerSlip = Float( help="Max. scheduler slip, in seconds" )
   tasks = Submodel( valueType=TaskList, help="List of active tasks" )
   completedTasks = Submodel( optional=True, valueType=TaskList,
                              help="List of completed tasks" )
   # Hack, we cannot reference TaskScheduler yet, so claim it is of
   # type 'Model', we fix that after the class definition.
   subSchedulers = Dict( keyType=str, valueType=Model,
                         help="Nested hierarchical subschedulers indexed by "
                         "a unique key" )
   history = Submodel( optional=True, valueType=History,
                       help="History of task actions" )

   def flattenSubSchedulers( self, source ):
      for prio, prioStats in source.priorityStats.items():
         if prio not in self.priorityStats:
            self.priorityStats[ prio ] = prioStats
         else:
            self.priorityStats[ prio ].runnableTasks += prioStats.runnableTasks
            self.priorityStats[ prio ].idleTasks += prioStats.idleTasks
            self.priorityStats[ prio ].completedTasks += prioStats.completedTasks
            self.priorityStats[ prio ].totalRunCount += prioStats.totalRunCount
      for t in ( 'tasks', 'completedTasks' ):
         sourceT = getattr( source, t )
         if not sourceT:
            continue
         selfT = getattr( self, t )
         if selfT:
            selfT.tasks += sourceT.tasks
            selfT.skipped += sourceT.skipped
         else:
            setattr( self, t, sourceT )
      for subScheduler in source.subSchedulers.values():
         self.flattenSubSchedulers( subScheduler )

   def displayScheduler( self, displayAddr, flatten ):
      if self.taskNameRegexError:
         raise InvalidInputError( 'Task name regular expression \'' +
                                  self.taskNameRegex + '\' is invalid, error: ' +
                                  self.taskNameRegexError )
      if self.generalError:
         raise AlreadyHandledError( 'An error was encountered: ' + self.generalError,
                                    msgType=AlreadyHandledError.TYPE_ERROR )
      if flatten:
         for subScheduler in self.subSchedulers.values():
            self.flattenSubSchedulers( subScheduler )
      else:
         raise NotImplementedError( 'Non-flattened output not yet implemented' )

      print( "TaskScheduler stats for", self.schedulerName )
      if self.taskNameRegex:
         print( 'Filtered to include only tasks matching \'' + self.taskNameRegex +
                '\'' )
      print()
      # First, the priority stats.
      print( "%-8s %13s %13s %9s %15s" % ( "Priority",
         "RunnableCount", "IdleCount", "DoneCount", "TotalRunCount" ) )
      print( "-" * 62 )
      for prio in priorities:
         priorityStat = self.priorityStats.get( prio )
         if not priorityStat:
            continue
         print( "%-8s %13u %13u %9u %15u" % (
            prio, priorityStat.runnableTasks,
            priorityStat.idleTasks,
            priorityStat.completedTasks,
            priorityStat.totalRunCount ) )
      print( "Maximum scheduler slip %.6f seconds" % ( self.maxSchedulerSlip ) )
      print()
      TASK_STAT_LEGEND_FMT = "%-17s %s"
      print( TASK_STAT_LEGEND_FMT % ( "SP",
         "= Task status and priority % two characters:" ) )
      print( TASK_STAT_LEGEND_FMT % ( "   first",
         "  T/t for active/idle timers, * for running task, blank or" ) )
      print( TASK_STAT_LEGEND_FMT % ( "",
         "  S for suspended tasks, X for tasks under suspended scheduler" ) )
      print( TASK_STAT_LEGEND_FMT % ( "   second",
         "  priority and runnable status, upper case for runnable" ) )
      print( TASK_STAT_LEGEND_FMT % ( "RunCount",
         "= Number of times task has run" ) )
      print( TASK_STAT_LEGEND_FMT % ( "TotalSec",
         "= Total run time in seconds" ) )
      print( TASK_STAT_LEGEND_FMT % ( "LastDelay",
         "= Scheduling delay last time task ran, in seconds" ) )
      print( TASK_STAT_LEGEND_FMT % ( "MaxDelay",
         "= Maximum scheduling delay encountered, in seconds" ) )
      print( TASK_STAT_LEGEND_FMT % ( "200% 500%",
         "= Number of times task used this percentage of quantum" ) )
      print( TASK_STAT_LEGEND_FMT % ( "MaxMs",
         "= Maximum single run time in milliseconds" ) )
      print( TASK_STAT_LEGEND_FMT % ( "Done",
         "= Number of times a task with this name has completed" ) )
      print( TASK_STAT_LEGEND_FMT % ( "TotalWorkCount",
         "= Counter of total work units processed" ) )
      print( TASK_STAT_LEGEND_FMT % ( "DecayAvgWorkCount",
         "=    exponential decaying average of work units processed" ) )
      print( TASK_STAT_LEGEND_FMT % ( "MaxWorkCount",
         "=    high watermark of work units processed in one run" ) )
      print( TASK_STAT_LEGEND_FMT % ( "LastWorkPending",
         "= Work units pending processing" ) )
      print( TASK_STAT_LEGEND_FMT % ( "MaxWorkPending",
         "=    high watermark of work units pending processing" ) )
      print()

      self.tasks.printTasks( "TaskName", displayAddr=displayAddr )
      if self.completedTasks and self.completedTasks.tasks:
         # Completed tasks are no longer running, and has no address,
         # so pass displayAddr as False unconditionally.
         print()
         self.completedTasks.printTasks( "TaskName (completed)",
                                         displayAddr=False,
                                         completedCount=True )

      if self.history:
         self.history.render()
         print()

# Hack, fix the valueType of the subSchedulers to be TaskScheduler now
# that the type is fully defined and we can reference it.
TaskScheduler.subSchedulers.valueType = TaskScheduler

class Overall( Model ):
   schedulers = Dict( keyType=str, valueType=TaskScheduler,
                      help="Root schedulers indexed by name" )
   _debug = Bool( help="The 'debug' keyword was used at the cli" )
   _flatten = Bool( help="Flatten sub-schedulers into root scheduler for display" )

   def render( self ):
      def sortFn( args ):
         schedulerName, scheduler = args[ 0 ], args[ 1 ]
         # Sort first on the displayOrder attribute, and secondly on
         # schedulerName if multiple schedulers have the same
         # displayOrder
         if scheduler.displayOrder is None:
            return ( 0, schedulerName )
         return ( scheduler.displayOrder, schedulerName )
      for _, scheduler in sorted( self.schedulers.items(), key=sortFn ):
         scheduler.displayScheduler( displayAddr=self._debug, flatten=self._flatten )
