# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable-msg=anomalous-backslash-in-string

import io
import os
import re
import socket
from socket import IPPROTO_UDP, IPPROTO_TCP
import sys

import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
import CliPlugin.TechSupportCli
from CliPlugin import AgentCli, AgentLibShowCommands
from CliPlugin.AclCli import getAclConfig
from CliPlugin.SnmpModel import ShowSnmpHosts, SnmpInterface, SnmpChassis, \
      ShowSnmpUsers, Contact, SnmpLocation, SnmpBuffer, \
      SnmpCounters, SnmpLogging, SnmpV3UserParams, UserCollectionModel, \
      SnmpLoggingHost, SnmpVrfs, SnmpMiscConfig, HostModel, SnmpV3HostParams, \
      SnmpV1V2cHostParams, UserModel, SnmpCommunityDetailModel, SnmpCommunityModel, \
      SnmpEngineIdModel, SnmpRemoteEngineIdHostToPortModel, \
      SnmpRemoteEngineIdPortToEngIdModel, SnmpNotificationModel, SnmpNotification, \
      SnmpView, SnmpViewModel, SnmpGroupModel, SnmpGroupVerToDetailModel, \
      SnmpGroupDetailModel, SnmpTotalModel, SnmpMibTranslateModel, \
      SnmpRegisteredOidsModel, SnmpOidsModel, SnmpTableIteratorContextCounterModel
import AclCliLib
import Cell
import CliMatcher
import CmdExtension
import ConfigMount
import DscpCliLib
import HostnameCli
from IpLibConsts import DEFAULT_VRF, DEFAULT_VRF_OLD
import LazyMount
from PyWrappers.NetSnmpUtils import snmpbulkwalk
import SnmpCliUtil
import TableOutput
import Tracing
import Tac
import json
import AgentCommandRequest
import CliModel

warningEngIdStr = """System engineID {} differs from that of {} user(s) and the
localized passphrases may be incorrect as a result.
To re-localize the user passphrases with the system engineID,
enter the "snmp-server user" command with plaintext passwords
without specifying an engineID."""

__defaultTraceHandle__ = Tracing.Handle( 'SnmpCli' )

SnmpConfig = Tac.Type( 'Snmp::Config' )

snmpConfig = None
snmpStatus = None
snmpCounters = None
snmpInternalCounters = None
entityMibStatus = None
allVrfStatusLocal = None
nsConfig = None
ipStatus = None
ip6Status = None
dscpConfig = None
snmpTestMib = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None

# TODO: figure out the correct regex for an SNMP community string
communityStringRe = r'[^\s]+'
communityStringLengthMax = 32 # this is the industry-standard limit

viewNameRe = r'[^\s]+'
viewNameLengthMax = 80  # this is arbitrary
oidTreeRe = r'[A-Za-z0-9\.-]+'

# Since 'snmp-server user foo bar v2c' actually maps to a community, it seems
# reasonable to have the same restrictions for users as communities.
userNameRe = communityStringRe
userNameLengthMax = communityStringLengthMax

groupNameRe = r'[^\s]+'
groupNameLengthMax = 32

contextNameRe = r'[^\s]+'
contextNameLengthMax = 80 # this is arbitrary

authPassPhraseRe = r'[^\s]+'
authPassPhraseReStrict = '^([0-9A-Fa-f]){%d}$'
authPassPhraseLengthMax = 64

privPassPhraseRe = r'[^\s]+'
privPassPhraseReStrict = authPassPhraseReStrict
privPassPhraseLengthMax = 64

contactLengthMax = 255
locationLengthMax = 255

# For 'legacy-style' engineId handling where automatic padding of odd-length
# hexstrings is done and the CliPlugin code does the error checking instead
# of letting the CLI parser and/or Python regex lib handle the errors.
engineIdRe = '[0-9A-Fa-f]+'
engineIdLengthMin = 10
engineIdLengthMax = 64

# For newer strict engineId handling where the CLI parser and/or Python
# regex lib perform error checking and the user must supply valid octet
# strings.
engineIdOctetsMin = engineIdLengthMin // 2
engineIdOctetsMax = engineIdLengthMax // 2
engineIdReStrict = '^([0-9A-Fa-f]{2}){%d,%d}$' % (
   engineIdOctetsMin, engineIdOctetsMax )
engineIdStrictHelpDesc = 'engine ID octet string (%d to %d pairs of hex digits)' % (
   engineIdOctetsMin, engineIdOctetsMax )

contextPattern = '.*'
# We don't try to match an OID (e.g., '([a-zA-Z0-9-]+::)?[a-z0-9][a-zA-Z0-9.]*')
# with this pattern, since that just results in an 'Invalid input' error.
# We delegate bad OID handling to the command-line tools.
oidPattern = '[^|>].*'

# TODO: these should be moved to a CLI Token file
snmpServerKwMatcher = CliMatcher.KeywordMatcher(
   'snmp-server',
   helpdesc='Modify SNMP engine parameters' )
snmpKwMatcher = CliMatcher.KeywordMatcher( 'snmp',
      helpdesc='Details on SNMP operation' )
mibKwMatcherForShow = CliMatcher.KeywordMatcher( 'mib',
                                       helpdesc='Show SNMP MIB contents' )

def checkLength( mode, string, maxLen, typ ):
   if len( string ) > maxLen:
      mode.addError( "%s too long: must be no more than %d characters"
                     % ( typ, maxLen ) )
      return False
   return True

# While it may seem like a good idea to keep an instance of EngineIdGenerator around,
# if we keep just a single global pointer to it, then its snmpConfig pointer may end
# up being stale if the generator was instantiated in a config session; otherwise we
# would have to have one generator per-config-session. In order to avoid the bug in
# the former scenario and the complexity in the latter, we accept the overhead of
# creating it every time we need it."
def engineIdGenerator():
   """Instantiates EngineIdGenerator on demand."""
   Tracing.trace0( "Instantiating EngineIdGenerator" )

   class EngineIdReturnValue:
      def __init__( self, entMibStatus, config ):
         generator = Tac.newInstance(
            "Snmp::EngineIdGenerator", entMibStatus.force(), config.force() )
         self.engineId = generator.engineId
         self.defaultEngineId = generator.defaultEngineId

   return EngineIdReturnValue( entityMibStatus, snmpConfig )

def checkEngineId( mode, engineId ):
   if len( engineId ) > engineIdLengthMax:
      mode.addError( "engineID too long: must be no more than %d hex digits" % (
         engineIdLengthMax ) )
      return False
   if len( engineId ) < engineIdLengthMin:
      mode.addError( "engineID too short: must be no fewer than %d hex "
                     "digits" % ( engineIdLengthMin ) )
      return False
   return True

def snmpCommunity( mode, communityStr, create=False ):
   config = snmpConfig
   communities = config.community
   if communityStr in communities:
      community = communities[ communityStr ]
   elif create:
      community = communities.newMember( communityStr )
      if community is None:
         Tracing.trace0( "Unable to create Community" )
   else:
      community = None
   if community is not None:
      assert community.name == communityStr
   return community

def snmpView( mode, viewName, create=False ):
   config = snmpConfig
   views = config.view
   if viewName in views:
      view = views[ viewName ]
   elif create:
      view = views.newMember( viewName )
      if view is None:
         Tracing.trace0( "Unable to create View" )
   else:
      view = None
   if view is not None:
      assert view.name == viewName
   return view

def snmpUser( mode, userName, version, create=False ):
   config = snmpConfig
   users = config.user
   userspec = Tac.Value( "Snmp::UserSpec", name=userName, protocolVersion=version )
   if userspec in users:
      user = users[ userspec ]
   elif create:
      user = users.newMember( userspec )
      if user is None:
         Tracing.trace0( "Unable to create User" )
   else:
      user = None
   if user is not None:
      assert user.userName == userName
      assert user.protocolVersion == version
   return user

def snmpRemoteUser( mode, hostname, port, userName, create=False ):
   config = snmpConfig
   remoteUsers = config.remoteUser
   spec = Tac.Value(
      "Snmp::RemoteUserSpec", hostname=hostname, port=port, userName=userName )
   if spec in remoteUsers:
      remote = remoteUsers[ spec ]
   elif create:
      remote = remoteUsers.newMember( spec )
      if remote is None:
         Tracing.trace0( "Unable to create RemoteUser" )
   else:
      remote = None
   if remote is not None:
      assert remote.userName == userName
   return remote

def snmpGroup( mode, groupName, version, create=False ):
   config = snmpConfig
   groups = config.group
   spec = Tac.Value( "Snmp::GroupSpec", name=groupName, protocolVersion=version )
   if spec in groups:
      group = groups[ spec ]
   elif create:
      group = groups.newMember( spec )
      if group is None:
         Tracing.trace0( "Unable to create Group: name", groupName,
            "protoColVersion", version )
   else:
      group = None
   if group is not None:
      assert group.groupName == groupName
      assert group.protocolVersion == version
   return group

def snmpNotificationSink( mode, hostname, vrf, port, securityName, create=False ):
   config = snmpConfig
   sinks = config.notificationSink
   spec = Tac.Value(
      "Snmp::NotificationSinkSpec", hostname=hostname, vrf=vrf, port=port,
      securityName=securityName )
   if spec in sinks:
      sink = sinks[ spec ]
   elif create:
      sink = sinks.newMember( spec )
      if sink is None:
         Tracing.trace0( "Unable to create NotificationSink: name", hostname )
   else:
      sink = None
   if sink is not None:
      assert sink.hostname == hostname
      assert sink.vrf == vrf
      assert sink.port == port
      assert sink.securityName == securityName
   return sink

def snmpRemoteEngineId( mode, hostname, port ):
   config = snmpConfig
   spec = Tac.Value( "Snmp::RemoteEngineIdSpec", hostname=hostname, port=port )
   return config.remoteEngineId.get( spec )

def updateServiceEnabled( mode ):
   config = snmpConfig
   if ( not config.community and not config.user ) or \
      not config.vrf:
      config.serviceEnabled = False
   else:
      config.serviceEnabled = True

def showSnmpAcrJson( mode, args, model, cmd, cmdArgs=None ):
   cmdArgs = cmdArgs or []
   cmdArgsStr = ""

   for arg, val in cmdArgs:
      cmdArgsStr += f"{arg}={val}\t"

   output = io.StringIO()
   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, "Snmp", cmd, cmdArgsStr, mode, stringBuff=output,
      forceOutputFormat="json" )
   jsonTxt = output.getvalue()
   output.close()

   if model is None:
      print( jsonTxt )
      return None

   j = json.loads( jsonTxt )

   errors = j.pop( "errors", None )
   if errors:
      for error in errors:
         mode.addError( error )
      return model()

   ret = CliModel.unmarshalModel( model, j )

   return ret

#------------------------------------------------------------------------------
# The "show snmp" command
#------------------------------------------------------------------------------
def snmpEnabled( mode, verbose=True ):
   config = snmpConfig
   if verbose and not config.serviceEnabled:
      mode.addError( "SNMP agent not enabled" )
   return config.serviceEnabled

def showSnmpContact( mode, args=None ):
   return Contact( contact=snmpConfig.contact )

def showSnmpLocation( mode, args=None ):
   return SnmpLocation( location=snmpConfig.location )

def showSnmpSourceInterface( mode, args=None ):
   result = SnmpInterface()
   for vrf, srcIntf in snmpConfig.notificationSourceIntf.items():
      ipIntfStatus = ipStatus.ipIntfStatus.get( srcIntf )
      ip6IntfStatus = ip6Status.intf.get( srcIntf )
      result.sourceInterfaces[ vrf ] = srcIntf
      if vrf not in snmpConfig.vrf:
         mode.addWarning( "SNMP agent is not enabled in VRF %s" % vrf )
      if ipIntfStatus and ipIntfStatus.vrf != vrf:
         mode.addWarning( "%s is in VRF %s" % ( srcIntf, ipIntfStatus.vrf ) )
      elif ip6IntfStatus and ip6IntfStatus.vrf != vrf:
         mode.addWarning( "%s is in VRF %s" % ( srcIntf, ip6IntfStatus.vrf ) )
   return result

def showSnmpCounters( mode, args=None ):
   snmpCountersModel = SnmpCounters()
   snmpCountersModel.serviceEnabled = snmpConfig.serviceEnabled
   if not snmpEnabled( mode, verbose=False ):
      return snmpCountersModel
   s = snmpCounters
   i = snmpInternalCounters
   snmpCountersModel.inPkts = s.inPkts
   snmpCountersModel.inVersionErrs = s.inVersionErrs
   snmpCountersModel.inBadCommunityNames = s.inBadCommunityNames
   snmpCountersModel.inBadCommunityUses = s.inBadCommunityUses
   snmpCountersModel.inParseErrs = s.inParseErrs
   snmpCountersModel.inRequestVars = s.inRequestVars
   snmpCountersModel.inSetVars = s.inSetVars
   snmpCountersModel.inGetPdus = s.inGetPdus
   snmpCountersModel.inGetNextPdus = s.inGetNextPdus
   snmpCountersModel.inSetPdus = s.inSetPdus
   snmpCountersModel.outPkts = s.outPkts
   snmpCountersModel.outTooBigErrs = s.outTooBigErrs
   snmpCountersModel.outNoSuchNameErrs = s.outNoSuchNameErrs
   snmpCountersModel.outBadValueErrs = s.outBadValueErrs
   snmpCountersModel.outGeneralErrs = s.outGeneralErrs
   snmpCountersModel.outGetResponsePdus = s.outGetResponsePdus
   snmpCountersModel.outTrapPdus = s.outTrapPdus
   snmpCountersModel.outTrapDrops = i.trapDrops
   snmpCountersModel.groupLength = len( snmpConfig.group )
   snmpCountersModel.numberOfGroups = len( snmpConfig.user )
   snmpCountersModel.numberOfViews = len( snmpConfig.view )
   return snmpCountersModel

# This method checks to see if any users have undefined groups attached
# or if any groups have undefined views attached
def showSnmpWarnings( mode ):
   warningStr = ' is not configured'
   warningStrings = []
   for g in snmpConfig.group.values():
      if g.readView and g.readView not in snmpConfig.view:
         warningStrings.append( 'Read view "' + g.readView +
                              '" of group "' + g.groupName +
                              '"' + warningStr )
      if g.writeView and g.writeView not in snmpConfig.view:
         warningStrings.append( 'Write view "' + g.writeView +
                              '" of group "' + g.groupName +
                              '"' + warningStr )
      if g.notifyView and g.notifyView not in snmpConfig.view:
         warningStrings.append( 'Notify view "' + g.notifyView +
                              '" of group "' + g.groupName +
                              '"' + warningStr )
   for u in snmpConfig.user.values():
      groupspec = Tac.Value( "Snmp::GroupSpec", u.group, u.protocolVersion )
      if groupspec not in snmpConfig.group:
         warningStrings.append( 'Group "' + u.group +
                              '" of user "' + u.userName +
                             '"' + warningStr )
   for err in warningStrings:
      mode.addWarning( err )

# This is just a very abbreviated version of what you get with
# "show snmp notification host" for inclusion in the basic "show snmp" output.
def showSnmpLogging( mode, args=None ):
   result = SnmpLogging()
   if not snmpEnabled( mode, verbose=False ):
      return result
   hosts = snmpConfig.notificationSink
   result._snmpEnabled = True # pylint: disable-msg=protected-access
   result.loggingEnabled = bool( len( hosts ) )

   for h in hosts.values():
      result.hosts[ h.hostname ] = SnmpLoggingHost( port=h.port, vrf=h.vrf )

   return result

def showSnmpVrf( mode, args=None ):
   return SnmpVrfs( snmpVrfs=sorted( snmpConfig.vrf ) )

def showSnmpMisc( mode, args=None ):
   result = SnmpMiscConfig()
   result.transmitMsgMaxSize = snmpConfig.transmitMsgSize
   return result

def showSnmpBasic( mode, args ):
   model = SnmpTotalModel()
   model.chassis = showSnmpChassis( mode )
   model.contact = showSnmpContact( mode )
   model.location = showSnmpLocation( mode )
   # counters is an optional attribute of SnmpTotalModel, however,
   # the attributes of SnmpCounters are not optional.
   # Therefore if the contents of counters are empty
   # don't set counters to avoid missing attribute warnings
   counters = showSnmpCounters( mode )
   # The attributes of SnmpCounters have been set if counters.serviceEnabled is True
   if counters.serviceEnabled:
      model.counters = counters
   model.logging = showSnmpLogging( mode )
   model.vrfs = showSnmpVrf( mode )
   model.srcIntf = showSnmpSourceInterface( mode )
   model.misc = showSnmpMisc( mode )
   showSnmpWarnings( mode )
   model.enabled = snmpEnabled( mode, verbose=False )
   return model
#------------------------------------------------------------------------------
# The "show snmp community" command
#------------------------------------------------------------------------------
def showSnmpCommunity( mode, args ):
   model = SnmpCommunityModel()
   config = snmpConfig
   communities = config.community
   for c in sorted( communities.values() ):
      community = SnmpCommunityDetailModel()
      model.communities[ c.communityString ] = community
      community.access = "readOnly" if c.access == 'ro' else 'readWrite'
      community.view = c.view
      community.defaultContext = c.context
      if c.view:
         if c.view not in snmpConfig.view:
            community.viewStatus = 'notConfigured'

      def _getAcl( acl, af ):
         status = None
         if acl:
            aclCfg = getAclConfig( af ).get( acl )
            if not aclCfg:
               status = 'nonExistent'
            elif not aclCfg.standard:
               status = "ignoredNotAStandardAcl"
         return acl, status

      community.ipV4AccessList, status = _getAcl( c.acl, 'ip' )
      community.ipV4AccessListStatus = status
      community.ipV6AccessList, status = _getAcl( c.acl6, 'ipv6' )
      community.ipV6AccessListStatus = status
   return model

#------------------------------------------------------------------------------
# The "show snmp notification" command
#
# legacy:
# The "show snmp trap" command (internally 'notification')
#------------------------------------------------------------------------------
def showSnmpNotificationHelper( model, parentName, parentValue, parentPath ):
   for notifName in parentValue.notification:
      notifStatus = parentValue.notificationStatus( notifName )
      notifName = SnmpCliUtil.trapToToken( notifName, parentValue.strip )
      notif = SnmpNotification()
      notif.component = ' '.join( parentPath )
      notif.name = notifName
      notif.enabled = notifStatus.enabled
      notif.reason = notifStatus.reason
      model.notifications.append( notif )

   for subName, subValue in parentValue.subtype.items():
      showSnmpNotificationHelper( model, subName, subValue,
                                  parentPath + [ subName ] )

def showSnmpNotification( mode, args ):
   model = SnmpNotificationModel()
   for configName, configValue in snmpConfig.notificationConfig.items():
      showSnmpNotificationHelper( model, configName, configValue, [ configName ] )
   return model

#------------------------------------------------------------------------------
# The "show snmp engineID" command
#------------------------------------------------------------------------------
def showSnmpEngineId( mode, args ):
   model = SnmpEngineIdModel()
   model.localEngineId = engineIdGenerator().engineId
   config = snmpConfig
   hostToPort = SnmpRemoteEngineIdHostToPortModel()
   model.remoteEngineIds = hostToPort
   if config.remoteEngineId:
      for spec, remote in config.remoteEngineId.items():
         if spec.hostname in hostToPort.hosts:
            portToEngId = hostToPort.hosts[ spec.hostname ]
         else:
            portToEngId = SnmpRemoteEngineIdPortToEngIdModel()
            hostToPort.hosts[ spec.hostname ] = portToEngId
         portToEngId.ports[ spec.port ] = remote.engineId
   return model

#------------------------------------------------------------------------------
# The "show snmp group" command
#------------------------------------------------------------------------------
def _secModel( protoVersion, authLevel ):
   authLevelStrMap = { 'levelNoAuth': 'v3NoAuth', 'levelAuth': 'v3Auth',
      'levelAuthAndPriv': 'v3Priv' }
   if protoVersion == "v3":
      secModel = authLevelStrMap[ authLevel ]
   else:
      secModel = protoVersion
   return secModel

def showSnmpGroup( mode, args ):
   model = SnmpGroupModel()
   groupName = args.get( 'GROUP' )
   config = snmpConfig
   groups = config.group

   groupValues = list( groups.values() )
   if groupName is not None:
      groupValues = [ g for g in groupValues if g.groupName == groupName ]
      if groupValues == []:
         print( 'SNMP Group %s not found' % groupName )
         return model
   for g in sorted( groupValues ):
      if not model.groups or g.spec.name not in model.groups:
         model.groups[ g.spec.name ] = SnmpGroupVerToDetailModel()
      groupVer = model.groups[ g.spec.name ]

      group = SnmpGroupDetailModel()
      groupVer.versions[ g.spec.protocolVersion ] = group

      group.readView = g.readView
      group.writeView = g.writeView
      group.notifyView = g.notifyView

      def _getViewConfig( view ):
         viewConfig = None
         if view:
            viewConfig = view in config.view
         return viewConfig

      group.readViewConfig = _getViewConfig( g.readView )
      group.writeViewConfig = _getViewConfig( g.writeView )
      group.notifyViewConfig = _getViewConfig( g.notifyView )
      group.secModel = _secModel( g.protocolVersion, g.authLevel )
   return model
#------------------------------------------------------------------------------
# The "show snmp notification host" command
#
# legacy:
# The "show snmp host" command
#------------------------------------------------------------------------------
def showSnmpHost( mode, args ):
   authLevelStrMap = { 'levelNoAuth': 'noAuthNoPriv', 'levelAuth': 'authNoPriv',
      'levelAuthAndPriv': 'authPriv' }
   config = snmpConfig
   hosts = config.notificationSink
   ret = ShowSnmpHosts()
   for k in sorted( hosts ):
      h = hosts[ k ]
      snmpHost = HostModel()
      snmpHost.hostname = h.hostname
      snmpHost.port = h.port
      snmpHost.vrf = h.vrf
      snmpHost.protocolVersion = h.protocolVersion
      snmpHost.notificationType = h.notificationType
      snmpHost.refresh = h.refresh
      if h.protocolVersion == 'v3':
         v3Params = SnmpV3HostParams()
         v3Params.user = h.securityName
         v3Params.securityLevel = authLevelStrMap[ h.authLevel ]
         snmpHost.v3Params = v3Params
      else:
         v1v2cParams = SnmpV1V2cHostParams()
         v1v2cParams.communityString = h.securityName
         snmpHost.v1v2cParams = v1v2cParams
      ret.hosts.append( snmpHost )
   return ret

#------------------------------------------------------------------------------
# The "show snmp user" command
#------------------------------------------------------------------------------
def showSnmpUser( mode, args ):
   userName = args.get( 'USER' )
   authTypeStrMap = { 'authMd5': 'MD5', 'authSha': 'SHA',
         'authSha224': 'SHA-224', 'authSha256': 'SHA-256',
         'authSha384': 'SHA-384', 'authSha512': 'SHA-512' }
   privTypeStrMap = { 'privacyAes': 'AES-128', 'privacyDes': 'DES',
         'privacy3Des': '3DES', 'privacyAes192': 'AES-192',
         'privacyAes256': 'AES-256' }

   config = snmpConfig
   users = config.user
   groups = config.group
   ret = ShowSnmpUsers()
   userValues = list( users.values() )
   nMismatchEngIdUsrs = 0
   if userName is not None:
      userValues = [ u for u in userValues if u.userName == userName ]
   for u in userValues:
      if u.protocolVersion not in ret.usersByVersion:
         ret.usersByVersion[ u.protocolVersion ] = UserCollectionModel()
      users = ret.usersByVersion[ u.protocolVersion ].users
      groupspec = Tac.Value( "Snmp::GroupSpec", u.group,
                              u.protocolVersion )
      userModel = UserModel()
      userModel.groupName = u.group
      userModel.groupConfigured = groupspec in groups
      if u.protocolVersion == 'v3':
         currEngId = engineIdGenerator().engineId
         v3Params = SnmpV3UserParams()
         v3Params.engineId = u.engineId
         v3Params.engineIdMatch = u.engineId == currEngId
         nMismatchEngIdUsrs += not v3Params.engineIdMatch
         if u.authType in authTypeStrMap:
            v3Params.authType = authTypeStrMap[ u.authType ]
         if u.privacyType in privTypeStrMap:
            v3Params.privType = privTypeStrMap[ u.privacyType ]
         userModel.v3Params = v3Params
      users[ u.userName ] = userModel

   if nMismatchEngIdUsrs:
      mode.addWarning( warningEngIdStr.format( currEngId,
                       nMismatchEngIdUsrs ) )
   return ret

#------------------------------------------------------------------------------
# The "show snmp view" command
#------------------------------------------------------------------------------
def showSnmpView( mode, args ):
   model = SnmpViewModel()
   viewName = args.get( 'VIEW' )
   config = snmpConfig
   views = config.view

   if viewName is not None:
      v = config.view.get( viewName )
      if v is None:
         view = SnmpView()
         view.name = viewName
         model.views.append( view )
         return model
      viewValues = [ v ]
   else:
      viewValues = list( views.values() )
   for v in viewValues:
      view = SnmpView()
      view.name = v.viewName
      for root in v.subtree:
         subTree = v.subtree[ root ]
         view.viewType[ root ] = subTree.viewType == 'included'
      model.views.append( view )
   return model

#------------------------------------------------------------------------------
# The "show snmp mib" command
#------------------------------------------------------------------------------
def showSnmpMibRegistry( mode, args ):
   registeredOids = SnmpRegisteredOidsModel()
   if not snmpEnabled( mode ):
      return registeredOids
   # We walk the NET-SNMP-AGENT-MIB::nsModuleTable,
   # parse out the context and registration point
   # from the INDEX, and then sort them.
   # We work completely on numeric values, and then
   # use the bulk mode of snmptranslate to translate
   # them to object names.

   walkOid = 'NET-SNMP-AGENT-MIB::nsModuleTimeout'
   try:
      objects = Tac.run( [ snmpbulkwalk(), '-Cr30', '-Obq',
                           os.environ.get( 'SNMPHOST', 'localhost' ),
                           walkOid ],
                         stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      mode.addError( str( e ) )
      if e.output:
         for line in e.output.strip().split( "\n" ):
            mode.addError( line )
      return registeredOids
   else:
      if not objects:
         return registeredOids

   registry = {}
   for o in objects.rstrip().split( "\n" ):
      # Eg: NET-SNMP-AGENT-MIB::nsModuleTimeout.0.12.1.3.6.1.6.3.16.1.5.2.1.3.127 -1
      Tracing.trace2( "Analyzing %s output:" % snmpbulkwalk(), o or "<empty line>" )
      oid = []
      if not o.startswith( walkOid ):
         # Probably some unrelated shell output got interleaved to stdout while
         # snmpbulkwalk was printing output. Just ignore it.
         continue
      parts = o.split( " " )
      try:
         oid = parts[ 0 ].split( "." )
         val = parts[ 1 ]
         assert oid.pop( 0 ) == walkOid and val.lstrip( "+-" ).isdigit()
         assert oid and all( x.isdigit() for x in oid )
      except ( IndexError, AssertionError ):
         # Shouldn't happen. Trace for debugging and try the next line.
         Tracing.trace0( "Bad %s response:" % snmpbulkwalk(), o )
         continue
      oids = [ int( o ) for o in oid ]
      contextLen = oids.pop( 0 )
      # Turn [ 65, 66, 67, <bunch of other numbers> ] into 'ABC'
      context = bytearray( oids[ : contextLen ] ).decode( 'utf-8' )
      oids = oids[ contextLen : ]
      objectLen = oids.pop( 0 )
      if objectLen == 1:
         # Internal top-level registration.  Skip it.
         continue
      obj = tuple( oids[ : objectLen ] )
      # We ignore the priority.  All we care about is that there is some
      # registration for this object.
      registry.setdefault( context, {} )[ obj ] = True
   for context in sorted( registry ):
      oidsModel = SnmpOidsModel()
      key = context if context else "default"
      registeredOids.contexts[ key ] = oidsModel
      oids = sorted( registry[ context ] )
      text = "\n".join( ".".join( str( oo ) for oo in o ) for o in oids )
      try:
         mibs = Tac.run( [ "snmptranslate", "-m", "ALL", "-" ], input=text,
                         stdout=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         mode.addError( str( e ) )
         if e.output:
            for line in e.output.strip().split( "\n" ):
               mode.addError( line )
         registeredOids.contexts.clear()
         return registeredOids
      else:
         for mib in mibs.split( "\n" ):
            if mib:
               oidsModel.oids.append( mib )

   return registeredOids

#------------------------------------------------------------------------------
# The "show snmp mib [context <context>] [numeric {index|all}]
#                                        [get|get-next|walk|table]" command
#------------------------------------------------------------------------------
def showSnmpMib( mode, args ):
   if not snmpEnabled( mode ):
      return
   if 'OID' in args:
      oid = args[ 'OID' ]
   elif 'WALK_OID' in args:
      oid = [ args[ 'WALK_OID' ] ]
   xtra = []
   if 'get' in args:
      cmd = 'get'
   elif 'get-next' in args:
      cmd = 'get-next'
   elif 'walk' in args:
      cmd = 'walk'
   else:
      assert 'table' in args
      cmd = 'table'

   if cmd == 'table':
      width = TableOutput.terminalWidth()
      xtra = [ '-Ci', '-Cw', '%d' % ( width - 2 ) ]
   elif cmd == 'get-next':
      cmd = 'getnext'
   # -OX: default: "program-like" index output, e.g.,
   #   Q-BRIDGE-MIB::dot1qTpFdbPort[172][STRING: 0:2a:81:7:59:4]
   # -On: numeric all: fully numeric OID, e.g.,
   #   .1.3.6.1.2.1.17.7.1.2.2.1.2.172.0.42.129.7.89.4
   # -Ob: numeric index: textual object name, but numeric instance, e.g.,
   #   Q-BRIDGE-MIB::dot1qTpFdbPort.172.0.42.129.7.89.4
   outputFmt = '-OX'
   if 'all' in args:
      outputFmt = '-On'
   elif 'index' in args:
      outputFmt = '-Ob'
   # When a value is an octet-string with no display-hint, net-snmp
   # applies a heuristic: if it's all ASCII, then print it as ASCII,
   # otherwise as hex.  However, if it's an IP address, for example,
   # then 192.168.0.1 will be printed as hex, but 72.105.104.105 will
   # be printed as "Hihi".  By default, we add the combination
   # of "-Ox" to disable this heuristic and always print hex strings, and
   # "-OT" to add printing the ASCII value additionally, so 72.105.104.105
   # will be printed as HEX-String: 48 69 68 69 [Hihi].  If hex is
   # requested, then we only add "-Ox"; if ascii is requested, then we
   # only add "-Oa".
   if 'hex' in args:
      xtra += [ '-Ox' ]
   elif 'ascii' in args:
      xtra += [ '-Oa' ]
   else:
      xtra += [ '-Ox', '-OT' ]
   shellCmd = [ 'snmp%s' % cmd, '-Le', outputFmt ] + xtra + \
              [ os.environ.get( 'SNMPHOST', 'localhost' ) ]
   if 'CONTEXT' in args:
      shellCmd += [ '-n', args[ 'CONTEXT' ] ]
   shellCmd += oid
   vrfs = list( snmpConfig.vrf )
   if DEFAULT_VRF not in vrfs:
      enabledVrfs = [ vrf for vrf in vrfs if vrf in allVrfStatusLocal.vrf ]
      if not enabledVrfs:
         return
      vrfName = enabledVrfs[ 0 ]
      cliCmdExt = CmdExtension.getCmdExtender()
      # Pass sys.stdout explicitly so that filters work.
      # Tac.INHERIT bypasses the cli's filters.
      err = cliCmdExt.runCmd( shellCmd, mode.session, vrfName=vrfName,
                              stdout=sys.stdout, stderr=Tac.CAPTURE,
                              asRoot=True, ignoreReturnCode=True )
   else:
      err = Tac.run( shellCmd, stdout=sys.stdout, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True )
   if err:
      for line in err.strip().split( "\n" ):
         mode.addError( line )

def chassisId():
   if snmpConfig and snmpConfig.chassisId:
      return snmpConfig.chassisId
   if entityMibStatus and entityMibStatus.root:
      return entityMibStatus.root.serialNum
   return ''

def showSnmpChassis( mode, args=None ):
   return SnmpChassis( chassisId=chassisId() )

#------------------------------------------------------------------------------
# The "show snmp mib translate" command
#------------------------------------------------------------------------------
def showSnmpMibTranslate( mode, args ):
   model = SnmpMibTranslateModel()
   oid = args.get( 'OID' )
   if re.match( r'[\d.]+$', oid ):
      # numeric -> text.
      model.numToSymIs( True )
      model.numeric = oid
      shellCmd = [ 'snmptranslate', '-OX', oid ]
      # This will never fail, if the oid isn't found the oid
      # is returned as the symbolic name.
      # .i.e. show snmp mib trans .177777.3.6.1.2.1.1.1.0
      # returns .177777.3.6.1.2.1.1.1.0
      model.symbolic = Tac.run( shellCmd, stdout=Tac.CAPTURE,
                                stderr=Tac.CAPTURE ).strip( "\n" )
   else:
      model.numToSymIs( False )
      model.symbolic = oid
      shellCmd = [ 'snmptranslate', '-IR', '-On', oid ]
      try:
         model.numeric = Tac.run( shellCmd, stdout=Tac.CAPTURE,
                                  stderr=Tac.CAPTURE ).strip( "\n" )
      except Tac.SystemCommandError as e:
         model.numToSymIs( None )
         for line in e.output.strip().split( "\n" ):
            mode.addError( line )
   return model

#------------------------------------------------------------------------------
# The "show snmp net-snmp buffer" command
#------------------------------------------------------------------------------
def showSnmpBuffer( mode, args ):
   return SnmpBuffer( clientRecvBuf=nsConfig.clientRecvBuf,
                      clientSendBuf=nsConfig.clientSendBuf,
                      serverRecvBuf=nsConfig.serverRecvBuf,
                      serverSendBuf=nsConfig.serverSendBuf )

#------------------------------------------------------------------------------
# "show snmp scheduler [verbose] [internal] [history] [reset]"
#------------------------------------------------------------------------------
def doShowSnmpScheduler( mode, args ):
   if 'reset' in args:
      AgentCli.runTaskSchedulerSocketCommand( 'Snmp', mode, 'clear' )
      return None
   # Translate options to those used in AgentCli
   translation = { 'verbose': 'detail',
                   'internal': 'debug',
                   'history': 'history' }
   agentArgs = { 'AGENT': 'Snmp',
                 # pylint: disable-next=consider-using-dict-items
                 'SCHEDULER_OPTS': [ translation[ i ] for i in translation
                                                      if i in args ],
                 }
   return AgentLibShowCommands.showTaskSchedulerHandler( mode, agentArgs )

#------------------------------------------------------------------------------
# The "no snmp-server" command in config mode.
#------------------------------------------------------------------------------
def noSnmpServer( mode, args ):
   config = snmpConfig
   config.serviceEnabled = False
   # If we wanted to completely mimic the industry standard behavior, we would
   # stop right here.  The industry standard doesn't clear out the rest of the
   # SNMP configuration when "no snmp-server" runs: it all disappears from the
   # running config, but just add a single community and all of the rest of the
   # configuration reappears!  That behavior seems very unintuitive and broken,
   # and it's highly unlikely that any customer would rely on this.
   config.chassisId = ""
   config.contact = ""
   config.location = ""
   config.engineId = ""
   config.view.clear()
   config.user.clear()
   config.remoteUser.clear()
   config.group.clear()
   config.community.clear()
   config.notificationSink.clear()
   config.remoteEngineId.clear()

   # Revert VRF config to default
   config.vrf.clear()
   config.vrf[ DEFAULT_VRF ] = True

   # Set dscp value to its default and
   # clear all dscp config rules.
   noDscp( mode=None )

   noNotificationLogEntryLimit( mode=None )

#-------------------------------------------------------------------------------
# The "[no] snmp-server community <community-name> [view <view-name>]
#    [ro | rw] [ipv6 <ipv6-acl-name>] [<acl-name>]"
# command, in "config" mode.
#-------------------------------------------------------------------------------
def setCommunity( mode, args ):
   communityStr = args[ 'COMMUNITY' ]
   view = args.get( 'VIEW' )
   access = 'rw' if 'rw' in args else 'ro'
   aclName = args.get( 'IP_ACL' )
   ip6AclName = args.get( 'IP6_ACL' )
   contextName = args.get( 'CONTEXT', '' )
   if contextName == DEFAULT_VRF:
      mode.addError( '"default" is not a valid context.' )
      return

   if len( communityStr ) > communityStringLengthMax:
      # truncate communityStr to max allowed length
      communityStr = communityStr[ : communityStringLengthMax ]

   def _checkAcl( acl, ipv6=False ):
      if acl:
         aclCfg = getAclConfig( 'ip%s' % ( 'v6' if ipv6 else '' ) ).get( acl )
         if aclCfg and not aclCfg.standard:
            mode.addError( '%s is not a standard acl' % acl )
            return False
      return True

   if not _checkAcl( aclName ) or \
      not _checkAcl( ip6AclName, ipv6=True ):
      return

   community = snmpCommunity( mode, communityStr, create=True )
   community.access = access if access else "ro"
   community.view = view if view else ""
   community.acl = aclName if aclName else ""
   community.acl6 = ip6AclName if ip6AclName else ""
   community.context = contextName

   # Enabling the SNMP service is a side-effect of updating a community or
   # user.  All praise to the wonderful industry standard!
   updateServiceEnabled( mode )

def noCommunity( mode, args ):
   communityStr = args[ 'COMMUNITY' ]
   if len( communityStr ) > communityStringLengthMax:
      # truncate communityStr to max allowed length
      communityStr = communityStr[ : communityStringLengthMax ]
   config = snmpConfig
   if communityStr in config.community:
      del config.community[ communityStr ]
      # Disabling the SNMP service is a side-effect of deleting the last
      # community or user.  All praise to the wonderful industry standard!
      updateServiceEnabled( mode )

#-------------------------------------------------------------------------------
# The "[no] snmp-server chassis-id <chassis-id>" command, in "config" mode.
#-------------------------------------------------------------------------------
def setChassis( mode, args ):
   chassis = args.get( 'CHASSIS_ID' )
   config = snmpConfig
   if len( chassis ) > config.chassisLengthMax:
      chassis = chassis[ : config.chassisLengthMax ]
   config.chassisId = chassis

def noChassis( mode, args ):
   config = snmpConfig
   config.chassisId = ""

#-------------------------------------------------------------------------------
# "[no|default] snmp-server qos dscp <0-63>" command, in "config" mode.
#-------------------------------------------------------------------------------
def updateDscpRules():
   dscpValue = snmpConfig.dscpValue

   if not dscpValue:
      del dscpConfig.protoConfig[ 'snmp' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'snmp' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for vrf in snmpConfig.vrf:
      # Outgoing SNMP responses.
      DscpCliLib.addDscpRule( ruleColl, '0.0.0.0',
                              161, True, vrf, 'udp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, '0.0.0.0',
                              161, True, vrf, 'tcp', dscpValue )

      DscpCliLib.addDscpRule( ruleColl, '::', 161, True, vrf, 'udp', dscpValue,
                              v6=True )
      DscpCliLib.addDscpRule( ruleColl, '::', 161, True, vrf, 'tcp', dscpValue,
                              v6=True )

   for spec in snmpConfig.notificationSink:
      # Traffic to notification recipient.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'udp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'tcp', dscpValue )

      DscpCliLib.addDscpRule( ruleColl, spec.hostname, spec.port, False, spec.vrf,
                              'udp', dscpValue, v6=True )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname, spec.port, False, spec.vrf,
                              'tcp', dscpValue, v6=True )

def setDscp( mode, args ):
   snmpConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args=None ):
   snmpConfig.dscpValue = snmpConfig.dscpValueDefault
   updateDscpRules()

#-------------------------------------------------------------------------------
# The "[no] snmp-server contact <contact>" command, in "config" mode.
#-------------------------------------------------------------------------------
def setContact( mode, args ):
   contact = args.get( 'CONTACT' )
   if len( contact ) > contactLengthMax:
      contact = contact[ : contactLengthMax ]
   config = snmpConfig
   config.contact = contact

def noContact( mode, args ):
   config = snmpConfig
   config.contact = ""

#-------------------------------------------------------------------------------
# The "[no|default] snmp-server enable traps [<trap-type>] [<trap>|<subtype>]+"
# command, in "config" mode.
#-------------------------------------------------------------------------------
def enableNotifications( mode, args ):
   if 'TRAP_TYPE' in args:
      _doEnableNotif( mode, args[ 'TRAP_TYPE' ], args.get( 'SUBTYPE', [] ),
                      args.get( 'TRAP', [] ), 'notifEnabled' )
   else:
      snmpConfig.notificationsGlobalEnabled = 'notifEnabled'

def disableNotifications( mode, args ):
   if 'TRAP_TYPE' in args:
      _doEnableNotif( mode, args[ 'TRAP_TYPE' ], args.get( 'SUBTYPE', [] ),
                      args.get( 'TRAP', [] ), 'notifDisabled' )
   else:
      snmpConfig.notificationsGlobalEnabled = 'notifDisabled'

def defaultNotifications( mode, args ):
   if 'TRAP_TYPE' in args:
      _doEnableNotif( mode, args[ 'TRAP_TYPE' ], args.get( 'SUBTYPE', [] ),
                      args.get( 'TRAP', [] ), 'notifDefault' )
   else:
      snmpConfig.notificationsGlobalEnabled = 'notifDefault'

def _doEnableNotif( mode, notifType, subtypeList, notifList, enabled ):

   def getTokens( typeConfig ):
      notifTokens = {}
      for notifName, notifConfig in typeConfig.notification.items():
         notifToken = SnmpCliUtil.trapToToken( notifName, typeConfig.strip )
         notifTokens[ notifToken ] = notifConfig
      return notifTokens

   def enableNotifs( typeConfig, notifList, enabled ):
      typeNotifTokens = getTokens( typeConfig )
      for notif in notifList:
         notifConfig = typeNotifTokens.get( notif )
         if notifConfig:
            notifConfig.enabled = enabled

   typeConfig = snmpConfig.notificationConfig[ notifType ]
   enableNotifs( typeConfig, notifList, enabled )
   for subtype in subtypeList:
      typeConfig = typeConfig.subtype[ subtype ]
      enableNotifs( typeConfig, notifList, enabled )

   notifsOfLastType = getTokens( typeConfig )
   # Case when the last token is a trap type
   if not notifList or notifList[ -1 ] not in notifsOfLastType:
      typeConfig.enabled = enabled

def getNotificationTypes( mode ):
   a = {}
   for notifType, config in snmpConfig.notificationConfig.items():
      a[ notifType ] = config.description
   return a

def getParentConfig( context ):
   parentName = context.sharedResult[ 'TRAP_TYPE' ]
   parentConfig = snmpConfig.notificationConfig[ parentName ]
   prevSubtypes = context.sharedResult.get( 'SUBTYPE', [] )
   for subtypeName in prevSubtypes:
      parentConfig = parentConfig.subtype[ subtypeName ]
   return parentConfig

def getNotificationSubtypes( mode, context ):
   b = {}
   parentConfig = getParentConfig( context )
   for subtypeName, subtypeConfig in parentConfig.subtype.items():
      b[ subtypeName ] = subtypeConfig.description
   return b

def getSpecificNotifications( mode, context ):
   c = {}
   parentConfig = getParentConfig( context )
   for notifName, notifConfig in parentConfig.notification.items():
      notifToken = SnmpCliUtil.trapToToken( notifName, parentConfig.strip )
      if notifToken not in context.sharedResult.get( 'TRAP', [] ):
         c[ notifToken ] = notifConfig.description
   return c

#-------------------------------------------------------------------------------
# The "[no|default] snmp-server objects <objects-name> disabled" command,
# in "config" mode.
#-------------------------------------------------------------------------------
def disableObjects( mode, args ):
   snmpConfig.disableObjectsConfig[ args[ 'OBJECTS' ] ].enabled = False

def noDisableObjects( mode, args ):
   snmpConfig.disableObjectsConfig[ args[ 'OBJECTS' ] ].enabled = True

def _getObjectsTypes( mode ):
   a = {}
   for objectType in sorted( snmpConfig.disableObjectsConfig.keys() ):
      a[ objectType ] = snmpConfig.disableObjectsConfig[ objectType ].description
   return a

#-------------------------------------------------------------------------------
# The "[no] snmp-server engineId {local | remote <host> [udp-port <port>]}
#    <engine-id>" command, in "config" mode.
#-------------------------------------------------------------------------------
_remoteEnginePortDefault = 162

def setEngineId( mode, args ):
   engineId = args.get( 'ENGINE_ID' )
   if not checkEngineId( mode, engineId ):
      return
   if len( engineId ) % 2 == 1:
      # EngineID has odd number of characters, prepending a 0 to the last digit
      engineId = engineId[ : -1 ] + "0" + engineId[ -1 ]

   config = snmpConfig
   if "remote" in args:
      host = args[ "REMOTE" ]
      port = args.get( "UDP_PORT", _remoteEnginePortDefault )
      if port is None:
         port = _remoteEnginePortDefault
      spec = Tac.Value( "Snmp::RemoteEngineIdSpec", hostname=host, port=port )
      if spec in config.remoteEngineId:
         del config.remoteEngineId[ spec ]
      remoteEngineId = Tac.Value(
         "Snmp::RemoteEngineId", spec=spec, engineId=engineId )
      config.remoteEngineId.addMember( remoteEngineId )
   else:
      assert "local" in args
      if engineId != engineIdGenerator().defaultEngineId:
         config.engineId = engineId
      else:
         config.engineId = ""

def noEngineId( mode, args ):
   config = snmpConfig
   if "remote" in args:
      host = args[ "REMOTE" ]
      port = args.get( "UDP_PORT", _remoteEnginePortDefault )
      if port is None:
         port = _remoteEnginePortDefault
      spec = Tac.Value( "Snmp::RemoteEngineIdSpec", hostname=host, port=port )
      if spec in config.remoteEngineId:
         del config.remoteEngineId[ spec ]
   else:
      assert "local" in args
      config.engineId = ""

#-------------------------------------------------------------------------------
# The "[no] snmp-server location <location>" command, in "config" mode.
#-------------------------------------------------------------------------------
def setLocation( mode, args ):
   location = args[ 'LOCATION' ]
   if len( location ) > locationLengthMax:
      location = location[ : locationLengthMax ]
   config = snmpConfig
   config.location = location

def noLocation( mode, args ):
   config = snmpConfig
   config.location = ""

#-------------------------------------------------------------------------------
# snmp-server view VIEW OID_TREE ( included | excluded )
# command in "config" mode.
#-------------------------------------------------------------------------------
def setView( mode, args ):
   viewName = args[ 'VIEW' ]
   oidTree = args[ 'OID_TREE' ]
   viewType = 'included' if 'included' in args else 'excluded'
   assert viewType in args

   if len( viewName ) > viewNameLengthMax:
      mode.addError( "View name too long: must be no longer than %d "
                     "characters" % ( viewNameLengthMax ) )
      return
   view = snmpView( mode, viewName, create=True )
   if oidTree in view.subtree:
      # Snmp::ViewSubtree is a value type, so changes to a ViewSubtree instance
      # will not obviously not notify, therefore we need to delete the existing
      # subtree and re-add it so that the change will be propagated to Sysdb.
      del view.subtree[ oidTree ]
   view.subtree.addMember( Tac.Value(
      "Snmp::ViewSubtree", root=oidTree, viewType=viewType ) )

def noView( mode, args ):
   viewName = args[ 'VIEW' ]
   oidTree = args.get( 'OID_TREE' )
   if len( viewName ) > viewNameLengthMax:
      mode.addError( "View name too long: must be no longer than %d "
                     "characters" % ( viewNameLengthMax ) )
      return
   config = snmpConfig
   if viewName in config.view:
      view = config.view[ viewName ]
      # if we get an oidTree, we delete the specified subtree, if it exists,
      # otherwise we delete all subtrees.
      if oidTree is not None:
         if oidTree in view.subtree:
            del view.subtree[ oidTree ]
      else:
         # should call view.subtree.clear() here, but get NotImplementedError
         view.subtree.clear()
      # delete the whole view if no subtrees remain
      if not view.subtree:
         del config.view[ viewName ]

#-------------------------------------------------------------------------------
# The "[no] snmp-server user <user-name> <group-name> [remote <host>
#    [udp-port <port>]] {v1 | v2c | v3 [localized <engineId>]
#    [auth {md5 | sha} <auth-passphrase>] [priv {des | aes} <priv-passphrase>]}"
# command in config mode.
#-------------------------------------------------------------------------------
authTypeMap = { None: 'authNone', 'md5': 'authMd5', 'sha': 'authSha',
   'sha224': 'authSha224', 'sha256': 'authSha256', 'sha384': 'authSha384',
   'sha512': 'authSha512' }
authLevelMap = { None: 'levelNoAuth', 'noauth': 'levelNoAuth',
   'auth': 'levelAuth', 'priv': 'levelAuthAndPriv' }
privTypeMap = { None: 'privacyNone', 'des': 'privacyDes',
   'aes': 'privacyAes', '3des': 'privacy3Des',
   'aes192': 'privacyAes192', 'aes256': 'privacyAes256' }

class VersionDataException ( Exception ):
   pass

def extractVersionData( mode, args, engineId ):
   """Takes the arguments from an snmp-server user command and pulls
   out all the authentication and privacy parameters, returning a dict
   containing the information."""
   localizedAuthKey = ""
   localizedPrivKey = ""
   authType = None
   authPhrase = None
   privType = None
   privPhrase = None
   if 'v3' in args:
      v = 'v3'
      authType = args.get( 'AUTH_TYPE' )
      authPhrase = args.get( 'AUTH_PHRASE' )
      privType = args.get( 'PRIV_TYPE' )
      privPhrase = args.get( 'PRIV_PHRASE' )
      if 'localized' in args:
         engineId = args[ 'ENGINE_ID' ]
         # The phrase strings must contain hexadecimal values of length
         # determined by the HMAC type: 128 bits for MD5 (32 hex digits), 160
         # bits for SHA-1 (40 hex digits), and longer for SHA-2.
         # Encryption algorithms have their own length requirements;
         # we keep them in the same dict because there is no overlap
         # on algorithm names.

         def _checkKey( authType, phrase, privType=None ):
            def _reparseKey( phrase, regexp ):
               return re.match( regexp, phrase )
            if privType is not None:
               keyRe = privPassPhraseReStrict
               relevantType = privType
            else:
               keyRe = authPassPhraseReStrict
               relevantType = authType
            reqLen = SnmpCliUtil.userKeyLen( relevantType )
            if _reparseKey( phrase, keyRe % reqLen ):
               return True
            if len( phrase ) > reqLen and \
               ( privType is not None and
                 _reparseKey( phrase, keyRe % SnmpCliUtil.userKeyLen( authType ) ) ):
               # Earlier EOS cli would save more bytes than
               # needed, e.g., for auth sha + priv des.
               # net-snmp will truncate internally, so this
               # is OK.
               return True
            mode.addError( "%s localized key must contain %d hexadecimal "
                           "digits" % ( relevantType, reqLen ) )
            return False
         if authType:
            if not _checkKey( authType, authPhrase ):
               raise VersionDataException
            localizedAuthKey = authPhrase
            if privType:
               if not _checkKey( authType, privPhrase, privType=privType ):
                  raise VersionDataException
               localizedPrivKey = privPhrase
      else:
         if authPhrase and not checkLength( mode, authPhrase,
                                            authPassPhraseLengthMax,
                                            "authentication password" ):
            raise VersionDataException
         if privPhrase and not checkLength( mode, privPhrase,
                                            privPassPhraseLengthMax,
                                            "privacy password" ):
            raise VersionDataException
         # The engineId must be set before a v3 user can be created.
         # If it is empty, I'm probably running in a test context
         # where someone needs to explicitly set the engineId.
         if engineId == "":
            mode.addError( "Cannot create SNMPv3 user when no engineID is set" )
            raise VersionDataException
         if authType:
            assert authPhrase != ""
            localizedAuthKey = SnmpCliUtil.localizePassphrase( authPhrase,
                                 authType, engineId )
         if privType:
            assert privPhrase != ""
            localizedPrivKey = SnmpCliUtil.localizePassphrase( privPhrase,
                                 authType, engineId, privType )
   elif 'v2c' in args:
      v = 'v2c'
   else:
      assert 'v1' in args
      v = 'v1'

   return { "version": v, "engineId": engineId,
      "authType": authType, "localizedAuthKey": localizedAuthKey,
      "privType": privType, "localizedPrivKey": localizedPrivKey }

def applyVersionDataToUser( vd, user ):
   user.engineId = vd[ "engineId" ]
   user.authType = authTypeMap[ vd[ "authType" ] ]
   user.authLocalizedKey = vd[ "localizedAuthKey" ]
   user.privacyType = privTypeMap[ vd[ "privType" ] ]
   user.privacyLocalizedKey = vd[ "localizedPrivKey" ]

def setRemoteUser( mode, args, userName, groupName ):
   hostname = args[ 'REMOTE' ]
   hoststr = hostname
   if ( HostnameCli.resolveHostname( mode, hostname, doWarn=True ) ==
        HostnameCli.Resolution.RESOLVED ):
      ( fam, _, _, _, _ ) = socket.getaddrinfo( hostname, None )[ 0 ]
      if fam == socket.AF_INET6:
         hoststr = ( 'udp6:[%s]' if ':' in hostname else 'udp6:%s' ) % hostname
   port = args.get( "UDP_PORT", _remoteEnginePortDefault )
   engineId = args.get( 'ENGINE_ID' )
   if engineId is None:
      remoteEngineId = snmpRemoteEngineId( mode, hostname, port )
      if remoteEngineId:
         engineId = remoteEngineId.engineId
   if engineId is None:
      # pylint: disable-msg=c-extension-no-member
      import SnmpEngineIdProbe # pylint: disable=import-outside-toplevel
      try:
         engineId = SnmpEngineIdProbe.probe( hoststr, port )
      except Exception as e:                     # pylint: disable-msg=W0703
         mode.addError( "Error discovering engineID for SNMP entity at %s:%d (%s)" %
                        ( hoststr, port, str( e ) ) )
         return
      spec = Tac.Value( "Snmp::RemoteEngineIdSpec", hostname=hostname, port=port )
      remoteEngineId = Tac.Value(
         "Snmp::RemoteEngineId", spec=spec, engineId=engineId )
      # Add remote engineId to collection so it will be remembered in the
      # saved config and we won't have to probe the next time a user is added
      # for this host/port, such as when processing the startup-config command
      # that will create this user.
      config = snmpConfig
      config.remoteEngineId.addMember( remoteEngineId )

   assert isinstance( engineId, str )
   try:
      vd = extractVersionData( mode, args, engineId )
   except VersionDataException:
      return

   remoteUser = snmpRemoteUser( mode, hostname, port, userName, create=True )
   remoteUser.group = groupName
   applyVersionDataToUser( vd, remoteUser )

def noRemoteUser( mode, userName, args ):
   hostname = args[ 'REMOTE' ]
   port = args.get( "UDP_PORT", _remoteEnginePortDefault )
   config = snmpConfig
   remoteUsers = config.remoteUser
   spec = Tac.Value(
      "Snmp::RemoteUserSpec", hostname=hostname, port=port, userName=userName )
   if spec in remoteUsers:
      del remoteUsers[ spec ]

def setUser( mode, args ):
   userName = args[ 'USER' ]
   groupName = args[ 'GROUP_NAME' ]

   if not checkLength( mode, userName, userNameLengthMax, "user name" ):
      return
   if not checkLength( mode, groupName, groupNameLengthMax, "group name" ):
      return
   if 'remote' in args:
      setRemoteUser( mode, args, userName, groupName )
      return
   try:
      engineId = engineIdGenerator().engineId
      vd = extractVersionData( mode, args, engineId )
   except VersionDataException:
      return

   user = snmpUser( mode, userName, vd[ "version" ], create=True )
   user.group = groupName
   applyVersionDataToUser( vd, user )
   # Enabling the SNMP service is a side-effect of updating a user or
   # community.  All praise to the wonderful industry standard!
   updateServiceEnabled( mode )

def noUser( mode, args ):
   userName = args[ 'USER' ]
   groupName = args[ 'GROUP_NAME' ]

   if not checkLength( mode, userName, userNameLengthMax, "user name" ):
      return
   if not checkLength( mode, groupName, groupNameLengthMax, "group name" ):
      return
   if 'remote' in args:
      noRemoteUser( mode, userName, args )
      return

   if 'v1' in args:
      v = 'v1'
   elif 'v2c' in args:
      v = 'v2c'
   else:
      assert 'v3' in args
      v = 'v3'
   config = snmpConfig
   users = config.user
   userspec = Tac.Value( "Snmp::UserSpec", name=userName, protocolVersion=v )
   if userspec in users:
      del users[ userspec ]
   # Disabling the SNMP service is a side-effect of removing the last user or
   # community.  All praise to the wonderful industry standard!
   updateServiceEnabled( mode )

#-------------------------------------------------------------------------------
# The "[no] snmp-server group <group-name> {v1 | v2c | v3 {auth | noauth | priv}}
#    [context <context-name>] [read <read-view>] [write <write-view>]
#    [notify <notify-view>]" command in
# config mode.
#-------------------------------------------------------------------------------
def setGroup( mode, args ):
   Tracing.trace2( "setGroup: args", args )
   groupName = args[ 'GROUP_NAME' ]
   contextName = args.get( 'CONTEXT' )
   if not checkLength( mode, groupName, groupNameLengthMax, "group name" ):
      return
   if contextName and not checkLength( mode, contextName, contextNameLengthMax,
                                   "context name" ):
      return
   if 'v3' in args:
      v = 'v3'
   elif 'v2c' in args:
      v = 'v2c'
   else:
      v = 'v1'
   assert v in args

   group = snmpGroup( mode, groupName, v, create=True )
   group.context = contextName or ""

   group.readView = args[ 'READ_VIEW' ][ 0 ] if 'READ_VIEW' in args else ''
   group.writeView = args[ 'WRITE_VIEW' ][ 0 ] if 'WRITE_VIEW' in args else ''
   group.notifyView = args[ 'NOTIFY_VIEW' ][ 0 ] if 'NOTIFY_VIEW' in args else ''

   group.authLevel = authLevelMap[ args.get( 'AUTH_LEVEL' ) ]

def noGroup( mode, args ):
   Tracing.trace2( "noGroup: args", args )
   groupName = args[ 'GROUP_NAME' ]
   if not checkLength( mode, groupName, groupNameLengthMax, "group name" ):
      return

   if 'v3' in args:
      v = 'v3'
   elif 'v2c' in args:
      v = 'v2c'
   else:
      v = 'v1'
   assert v in args

   config = snmpConfig
   groups = config.group
   groupspec = Tac.Value( "Snmp::GroupSpec", name=groupName, protocolVersion=v )
   if groupspec in groups:
      del groups[ groupspec ]

#-------------------------------------------------------------------------------
# The "[no] snmp-server host <host-addr> [vrf <vrf-name>] [traps | informs]
#    [refresh] [version {1 | 2c | 3 [auth | noauth | priv]}] <community-string>
#    [udp-port <port>]" command in config mode.
#
# NOTE: The industry-standard version of this command allows the user to
# provide a list of the type of traps allowed to be sent to the host.  The
# allowed values are not specific trap names, but rather names of groupings.
# I believe that the grouping names are just aliases for some OID prefix.
# For example, the linkUp/linkDown traps are in the "snmp" grouping, which
# probably corresponds to { iso(1) org(3) dod(6) internet(1) snmpV2(6)
# snmpModules(3) snmpMIB(1) snmpMIBObjects(1) snmpTraps(5) }.  For us
# to support something similar we would require the SnmpPlugins to register
# the traps that they provide, and perhaps a mapping of config name -> OID
# prefix, and then set up views to enforce the restriction.
#-------------------------------------------------------------------------------
_notificationSinkPortDefault = 162

def setHost( mode, args ):
   Tracing.trace2( "setHost: args", args )
   hostname = args[ 'HOST' ]
   securityName = args[ 'SECURITY_NAME' ]
   notificationType = 'inform' if 'informs' in args else 'trap'
   refresh = 'refresh' in args
   vrfName = args.get( 'VRF' )
   udpPort = args.get( 'UDP_PORT' )

   if not checkLength( mode, securityName, communityStringLengthMax,
                       "user/community name" ):
      return
   # Use defaults for optional arguments
   if notificationType is None:
      notificationType = 'trap'
   if udpPort is None:
      udpPort = _notificationSinkPortDefault
   if not vrfName or vrfName in ( DEFAULT_VRF, DEFAULT_VRF_OLD ):
      vrfName = ''

   # version is either going to be a string (v1, v2c) or a dict containing
   # the v3 auth level.
   authLevel = None
   if '3' in args:
      v = 'v3'
      authLevel = args.get( 'AUTH_LEVEL' )
   elif '1' in args:
      v = 'v1'
   else:
      v = 'v2c'
   if notificationType == "inform" and v == 'v1':
      mode.addError( "SNMPv1 does not support informs" )
      return

   HostnameCli.resolveHostname( mode, hostname, doWarn=True )

   sink = snmpNotificationSink( mode, hostname, vrfName, udpPort, securityName,
                                create=True )
   sink.notificationType = notificationType
   sink.protocolVersion = v
   sink.authLevel = authLevelMap[ authLevel ]
   sink.refresh = refresh
   updateDscpRules()

def noHost( mode, args ):
   Tracing.trace2( "noHost: args", args )
   hostname = args[ 'HOST' ]
   securityName = args[ 'SECURITY_NAME' ]
   vrfName = args.get( 'VRF' )
   udpPort = args.get( 'UDP_PORT', _notificationSinkPortDefault )
   if not checkLength( mode, securityName, communityStringLengthMax,
                       "user/community name" ):
      return

   if not vrfName or vrfName in ( DEFAULT_VRF, DEFAULT_VRF_OLD ):
      vrfName = ''

   config = snmpConfig
   spec = Tac.Value( "Snmp::NotificationSinkSpec", hostname=hostname,
                     vrf=vrfName, port=udpPort, securityName=securityName )
   sinks = config.notificationSink
   if spec in sinks:
      del sinks[ spec ]
   updateDscpRules()

#-------------------------------------------------------------------------------
# The "[no] snmp-server [vrf <vrfName>] local-interface <intfName>" command
# in config mode.
#
# legacy:
# The "[no] snmp-server [vrf <vrfName>] source-interface <intfName>" command
#-------------------------------------------------------------------------------
def setSrcIntf( mode, args ):
   vrfName = args.get( 'VRF' )
   intf = args[ 'IP_INTF' ]
   srcIntfs = snmpConfig.notificationSourceIntf
   if not vrfName:
      vrfName = DEFAULT_VRF
   srcIntfs[ vrfName ] = intf.name

def noSrcIntf( mode, args ):
   vrfName = args.get( 'VRF' )
   srcIntfs = snmpConfig.notificationSourceIntf
   if vrfName is None:
      vrfName = DEFAULT_VRF
   del srcIntfs[ vrfName ]

#-------------------------------------------------------------------------------
# The "[no] snmp-server extension <oid> <handler> [oneshot]" command in config mode.
#-------------------------------------------------------------------------------
def setExtension( mode, args ):
   oid = args[ 'OID' ]
   handler = args[ 'HANDLER' ]
   oneShot = 'one-shot' in args
   if not oid.startswith( "." ):
      oid = "." + oid
   if not re.match( '(.[0-9]+)+$', oid ):
      mode.addError( 'Invalid OID %s' % oid )
   extension = Tac.Value( "Snmp::Extension", oid=oid,
                          handler=handler.localFilename(), oneShot=oneShot )
   if oid in snmpConfig.extension:
      # Just replace it.
      del snmpConfig.extension[ oid ]
   snmpConfig.extension.addMember( extension )

def noExtension( mode, args ):
   oid = args[ 'OID' ]
   del snmpConfig.extension[ oid ]

#-------------------------------------------------------------------------------
# The "[no|default] snmp-server vrf <vrfName>" command in config mode.
#-------------------------------------------------------------------------------
def setVrf( mode, args ):
   vrfName = args[ 'VRF' ]
   if vrfName == DEFAULT_VRF_OLD:
      vrfName = DEFAULT_VRF
   snmpConfig.vrf[ vrfName ] = True
   updateServiceEnabled( mode )
   updateDscpRules()

def noVrf( mode, args ):
   vrfName = args[ 'VRF' ]
   if vrfName == DEFAULT_VRF_OLD:
      vrfName = DEFAULT_VRF
   del snmpConfig.vrf[ vrfName ]
   updateServiceEnabled( mode )
   updateDscpRules()

def defaultVrf( mode, args ):
   # VRF 'default' is on by default.
   if args[ 'VRF' ] in ( DEFAULT_VRF, DEFAULT_VRF_OLD ):
      setVrf( mode, args )
   else:
      noVrf( mode, args )

#-------------------------------------------------------------------------------
# The "test snmp notification [ COMMENT ]" command in enable mode
#-------------------------------------------------------------------------------
def sendTestNotification( mode, args ):
   comment = args.get( 'COMMENT', '' )
   comment = comment[ : 255 ]
   snmpTestMib.notificationComment = comment
   snmpTestMib.notificationCounter += 1

#-------------------------------------------------------------------------------
# Config mode hidden cli:
#     [no] snmp-server net-snmp buffer {client | server} {send | receive} <size>
#-------------------------------------------------------------------------------
def setBuffer( mode, args ):
   size = args[ 'SIZE' ]
   clientOrServer = 'client' if 'client' in args else 'server'
   recvOrSend = 'receive' if 'receive' in args else 'send'

   assert clientOrServer in args
   assert recvOrSend in args
   try:
      bufferSize = int( size )
      # On 64-bit systems int can be 64-bit. We must NOT accept values >
      # 0xffffffff
      if bufferSize > 0xffffffff:
         return
   except ValueError:
      return
   if not isinstance( bufferSize, int ):
      return
   if clientOrServer == 'client':
      if recvOrSend == 'receive':
         nsConfig.clientRecvBuf = bufferSize
      else:
         nsConfig.clientSendBuf = bufferSize
   else:
      if recvOrSend == 'receive':
         nsConfig.serverRecvBuf = bufferSize
      else:
         nsConfig.serverSendBuf = bufferSize

def noBuffer( mode, args ):
   clientOrServer = 'client' if 'client' in args else 'server'
   recvOrSend = 'receive' if 'receive' in args else 'send'

   assert clientOrServer in args
   assert recvOrSend in args
   if clientOrServer == 'client':
      if recvOrSend == 'receive':
         nsConfig.clientRecvBuf = 0
      else:
         nsConfig.clientSendBuf = 0
   else:
      if recvOrSend == 'receive':
         nsConfig.serverRecvBuf = 0
      else:
         nsConfig.serverSendBuf = 0

#-------------------------------------------------------------------------------
# The "[no] snmp-server transport tcp" command, in "config" mode.
#-------------------------------------------------------------------------------
def setTcpTransport( mode, args ):
   snmpConfig.tcpTransport = True

def noTcpTransport( mode, args ):
   snmpConfig.tcpTransport = False

#-------------------------------------------------------------------------------
# The "[no|default] snmp-server transmit max-size <size>" command, in "config"
# mode.
#-------------------------------------------------------------------------------
def setTransmitMsgSize( mode, args ):
   snmpConfig.transmitMsgSize = args[ 'SIZE' ]

def noDefaultTransmitMsgSize( mode, args ):
   snmpConfig.transmitMsgSize = snmpConfig.transmitMsgSizeDefault

#-------------------------------------------------------------------------------
# The "[no|default] snmp-server notification log entry limit <limit>" command
# in config mode.
#-------------------------------------------------------------------------------
def setNotificationLogEntryLimit( mode, args ):
   snmpConfig.notificationLogEntryLimit = args[ 'LIMIT' ]

def noNotificationLogEntryLimit( mode, args=None ):
   default = snmpConfig.notificationLogEntryLimitDefault
   snmpConfig.notificationLogEntryLimit = default

#-----------------------------------------------------------------------------
# [ no | default ] snmp-server ( ( ipv4 access-list IP_ACL ) |
#                                ( ipv6 access-list IP6_ACL ) ) [ vrf VRF ]
#-----------------------------------------------------------------------------
def setIpAcl( mode, args ):
   aclName = args[ 'IP_ACL' ] if 'ipv4' in args else args[ 'IP6_ACL' ]
   vrfName = args.get( 'VRF' )
   aclType = 'ip' if 'ipv4' in args else 'ipv6'
   AclCliLib.setServiceAcl( mode, 'snmp-tcp', IPPROTO_TCP,
                            aclConfig, aclCpConfig,
                            aclName, aclType, vrfName, port=[ 161 ] )
   AclCliLib.setServiceAcl( mode, 'snmp-udp', IPPROTO_UDP,
                            aclConfig, aclCpConfig,
                            aclName, aclType, vrfName, port=[ 161 ] )

def noIpAcl( mode, args=None ):
   aclType = 'ip' if 'ipv4' in args else 'ipv6'
   aclName = args.get( 'ACL_NAME' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   AclCliLib.noServiceAcl( mode, 'snmp-tcp', aclConfig,
                           aclCpConfig, aclName, aclType, vrfName )
   AclCliLib.noServiceAcl( mode, 'snmp-udp', aclConfig,
                           aclCpConfig, aclName, aclType, vrfName )

# ---------------------------------------------------------------------------
# no snmp-server objects override
# ---------------------------------------------------------------------------
def noObjectsOverride( mode, args ):
   snmpConfig.contextOverride.clear()

# ---------------------------------------------------------------------------
# [no] context FOO in snmp-server objects override mode
# ---------------------------------------------------------------------------
def ensureContextOverride( mode, context ):
   snmpConfig.newContextOverride( context )

def deleteContextOverride( mode, context ):
   del snmpConfig.contextOverride[ context ]

def setOverrideOid( mode, args ):
   context = mode.param_
   oid = args[ 'OID' ]
   existed = oid in snmpConfig.contextOverride[ context ].objectOverride
   obj = snmpConfig.contextOverride[ context ].newObjectOverride( oid )
   tableInstance = 'table-instance' in args
   if existed and tableInstance != obj.tableInstance:
      mode.addError( 'Delete and recreate in order to change table-instance value' )
      return
   obj.tableInstance = tableInstance
   if 'integer' in args:
      obj.asnType = 'asnInteger'
      obj.integerValue = args[ 'INTEGER' ]
   elif 'octet-string' in args:
      obj.asnType = 'asnOctetStr'
      stringValue = args[ 'STRING' ]
      if stringValue.startswith( '0x' ):
         shouldBeHex = stringValue[ 2: ]
         if not re.match( r'([0-9a-fA-F]{2})+$', shouldBeHex ):
            mode.addError( 'Invalid hex string %s' % shouldBeHex )
            return
         obj.octetStringValue = bytes.fromhex( shouldBeHex )
         obj.octetStringHex = True
      else:
         obj.octetStringValue = stringValue.encode( 'utf-8' )
         obj.octetStringHex = False
   elif 'object-id' in args:
      obj.asnType = 'asnObjectId'
      obj.oidValue = args[ 'OID2' ]
   elif 'counter' in args:
      obj.asnType = 'asnCounter'
      obj.unsignedValue = args[ 'UNSIGNED' ]
   elif 'unsigned' in args:
      obj.asnType = 'asnUnsigned'
      obj.unsignedValue = args[ 'UNSIGNED' ]
   elif 'timeticks' in args:
      obj.asnType = 'asnTimeTicks'
      obj.unsignedValue = args[ 'UNSIGNED' ]
   else:
      mode.addError( 'Unknown type (internal error)' )

def delOverrideOid( mode, args ):
   context = mode.param_
   oid = args[ 'OID' ]
   del snmpConfig.contextOverride[ context ].objectOverride[ oid ]

#---------------------------------------------------------------------------
# show snmp  ( ipv4 access-list [IP4ACL] ) | ( ipv6 access-list [ IP6ACL ] )
#---------------------------------------------------------------------------
def showSnmpAccessLists( mode, args=None ):

   # Since the ACLs are always the same for tcp and udp
   # only tcp is being displayed
   aclType = 'ip' if 'ipv4' in args else 'ipv6'
   aclName = args[ '<aclNameExpr>' ]
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 aclName,
                                 serviceName='snmp-tcp' )

# ---------------------------------------------------------------------------
# show snmp debug counters table
# ---------------------------------------------------------------------------
def showSnmpDebugCounterTableIterator( mode, args ):
   cmd = Tac.Type( "Snmp::SnmpRegVoidCountersAcr" ).command
   cmdArgs = []
   tableName = args.get( 'TABLE' )
   if not tableName:
      assert args.get( 'all' )
      tableName = 'all'
   cmdArgs.append( ( 'table', tableName ) )

   context = args.get( 'CONTEXT', DEFAULT_VRF )
   cmdArgs.append( ( 'context', context ) )

   return showSnmpAcrJson(
      mode, args, SnmpTableIteratorContextCounterModel, cmd, cmdArgs=cmdArgs )

# ---------------------------------------------------------------------------
# show snmp debug registry
# ---------------------------------------------------------------------------
def showSnmpDebugRegistry( mode, args ):
   cmd = Tac.Type( "Snmp::SnmpRegistryDumpAcr" ).command
   return showSnmpAcrJson( mode, args, None, cmd )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-06-24 15:05:50',
   cmds=[ 'show snmp' ],
   summaryCmds=[ 'show snmp' ] )

def Plugin( entityManager ):
   global snmpConfig, snmpStatus, snmpCounters, snmpTestMib, allVrfStatusLocal
   global nsConfig, ipStatus, ip6Status, dscpConfig
   global entityMibStatus
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global snmpInternalCounters
   snmpConfig = ConfigMount.mount( entityManager, "snmp/config",
                                   "Snmp::Config", "w" )
   snmpStatus = LazyMount.mount( entityManager, "snmp/status",
                                 "Snmp::Status", "r" )
   snmpCounters = LazyMount.mount( entityManager, "snmp/counters",
                                   "Snmp::Counters", "r" )
   snmpInternalCounters = LazyMount.mount( entityManager,
                                           "snmp/internalCounters",
                                           "Snmp::InternalCounters", "r" )
   snmpTestMib = LazyMount.mount( entityManager, 'snmp/testMib',
                                  "Snmp::TestMib", "w" )
   entityMibStatus = LazyMount.mount( entityManager, "hardware/entmib",
                                      "EntityMib::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( 'ip/vrf/status/local' ),
                                        'Ip::AllVrfStatusLocal', 'r' )
   nsConfig = ConfigMount.mount( entityManager, "snmp/netSnmpConfig",
                                 "Snmp::NetSnmpConfig", "w" )
   ipStatus = LazyMount.mount( entityManager, "ip/status",
                               "Ip::Status", "r" )
   ip6Status = LazyMount.mount( entityManager, "ip6/status",
                               "Ip6::Status", "r" )
   dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                'Acl::Status', 'r' )
   aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                    'Acl::CheckpointStatus', 'r' )
