# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, CliToken.Hardware, CliParser, Tac
import BasicCliUtil, ConfigMount, LazyMount
import CliCommand
import CliExtensions
import CliMatcher
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.SpeedGroupModel as SpeedGroupModel
# pylint: disable-next=consider-using-from-import
import CliPlugin.SpeedGroupRule as SpeedGroupRule
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from IntfRangePlugin.SpeedGroup import SpeedGroupType
import ProductAttributes
import ShowCommand
from Toggles.EthIntfToggleLib import toggleL1PolicyPllHookEnabled
from TypeFuture import TacLazyType

MountConstants = TacLazyType( 'L1::MountConstants' )
L1PolicyVersion = TacLazyType( 'L1::PolicyVersion' )

capsDefaultRootDir = None
productConfig = None
speedGroupConfigSliceDir = None
speedGroupStatusSliceDir = None
ethPhyIntfDefaultConfigDir = None

canPromptForAbortHook = CliExtensions.CliHook()

def getSpeedGroupSliceSubdomain( groupNumber ):
   for sliceId, capsDefaultSliceDir in capsDefaultRootDir.items():
      for subdomain, capsDefaultDir in capsDefaultSliceDir.items():
         if groupNumber in capsDefaultDir.pllCapabilities:
            return ( sliceId, subdomain )
   return ( None, None )

def getPolicyVersion( sliceId, subdomain ):
   if not sliceId or not subdomain:
      return L1PolicyVersion()

   prodCardSlotConfig = productConfig.cardSlotConfig.get( sliceId )
   if not prodCardSlotConfig:
      return L1PolicyVersion()

   prodSubdomainConfig = prodCardSlotConfig.subdomainConfig.get( subdomain )
   if not prodSubdomainConfig:
      return L1PolicyVersion()

   return prodSubdomainConfig.policyVersion

def getSpeedGroupConfigDir( name ):
   pa = ProductAttributes.productAttributes().productAttributes
   if pa.hasSwitchcard and ( pa.isSingleChip or pa.platform == "sand" ):
      sliceName = "Switchcard1"
   elif '/' in name:
      # Modular system
      sliceName = "Linecard" + name.split( '/' )[ 0 ]
   else:
      sliceName = "FixedSystem"
   return speedGroupConfigSliceDir[ sliceName ]

def nonL1PolicyCheckSpeedGroupChangeCancel( mode,
                                            group,
                                            compatibility,
                                            onlyPolicyV3 ):
   if onlyPolicyV3:
      # exit if this is not the correct hook
      return False

   promptTemplate = "Changing the speed group setting of {0} may cause member " \
                    "interfaces in speed group to flap."
   warningPrompt = ""
   if mode.session.commandConfirmation():
      changedGroup = []
      for name in IntfRange.intfListFromCanonical( [ group ] ):
         nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         speedGroupConfig = speedGroupConfigDir.group.get( nameWithNoType )
         if speedGroupConfig:
            curConfig = set( speedGroupConfig.setting.keys() )
         else:
            curConfig = set()
         if compatibility:
            newConfig = set( compatibility )
         else:
            newConfig = set()
         if newConfig != curConfig:
            changedGroup.append( nameWithNoType )
      if changedGroup:
         warningPrompt = promptTemplate.format( group )

   if warningPrompt:
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         # pylint: disable-next=consider-using-f-string
         abortMsg = "Command aborted by user for %s" % group
         mode.addMessage( abortMsg )
         return True

   return False

canPromptForAbortHook.addExtension( nonL1PolicyCheckSpeedGroupChangeCancel )

def configSpeedGroup( mode, group, compatibility=None ):
   onlyPolicyV3 = True
   # If all speedgroups being configured are on a subdomain using policyV3,
   # execute the new hook. Else if there is even one sg on a policy < 3
   # or without a policy, execute the cancel function.
   for name in IntfRange.intfListFromCanonical( [ group ] ):
      nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
      sliceId, subdomain = getSpeedGroupSliceSubdomain( nameWithNoType )
      sgPolicy = getPolicyVersion( sliceId, subdomain )
      if sgPolicy < L1PolicyVersion( 3, 0 ):
         onlyPolicyV3 = False

   # only run the old hook while this toggle has not been enabled
   if not toggleL1PolicyPllHookEnabled():
      onlyPolicyV3 = False

   for hook in canPromptForAbortHook.extensions():
      if hook( mode, group, compatibility, onlyPolicyV3 ):
         return

   for name in IntfRange.intfListFromCanonical( [ group ] ):
      nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
      if compatibility:
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         speedGroupConfig = speedGroupConfigDir.group.get( nameWithNoType )
         if speedGroupConfig is None:
            speedGroupConfig = speedGroupConfigDir.newGroup( nameWithNoType )
         curConfig = set( speedGroupConfig.setting.keys() )
         curGenId = speedGroupConfig.settingGeneration.id
         # Delete unconfigured compatibility settings
         for setting in speedGroupConfig.setting:
            if setting not in compatibility:
               del speedGroupConfig.setting[ setting ]
         # Add configured compatibility settings
         for attr in compatibility:
            speedGroupConfig.setting[ attr ] = True
         newConfig = set( speedGroupConfig.setting.keys() )
         if newConfig != curConfig:
            speedGroupConfig.settingGeneration = Tac.Value( "Ark::Generation",
                                                            curGenId + 1, True )
      else:
         # no/default command delete the entity
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         del speedGroupConfigDir.group[ nameWithNoType ]

# Add a hidden tokenRule to parse a speed-group when the group in
# SpeedGroupStatusDir has not been created.
speedGroupNameNode = CliCommand.Node(
   matcher=SpeedGroupRule.SpeedGroupMatcher( priority=CliParser.PRIO_LOW ),
   storeSharedResult=True,
   hidden=True )

speedGroupNameRangeNode = CliCommand.Node(
   matcher=IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType, ),
         dollarCompHelpText='Group list end' ),
   storeSharedResult=True )

def ethLinkModeSetToHelpDesc( elms ):
   compatibilityList = []
   if elms.mode1GbpsFull:
      compatibilityList.append( '1' )
   if elms.mode10GbpsFull:
      compatibilityList.append( '10' )
   if elms.mode25GbpsFull:
      compatibilityList.append( '25' )
   if elms.mode40GbpsFull:
      compatibilityList.append( '40' )
   if elms.mode50GbpsFull:
      compatibilityList.append( '50-2' )
   if elms.mode50GbpsFull1Lane:
      compatibilityList.append( '50-1' )
   if elms.mode100GbpsFull:
      compatibilityList.append( '100-4' )
   if elms.mode100GbpsFull2Lane:
      compatibilityList.append( '100-2' )
   if elms.mode100GbpsFull1Lane:
      compatibilityList.append( '100-1' )
   if elms.mode200GbpsFull4Lane:
      compatibilityList.append( '200-4' )
   if elms.mode200GbpsFull2Lane:
      compatibilityList.append( '200-2' )
   if elms.mode400GbpsFull8Lane:
      compatibilityList.append( '400-8' )
   if elms.mode400GbpsFull4Lane:
      compatibilityList.append( '400-4' )
   if elms.mode800GbpsFull8Lane:
      compatibilityList.append( '800-8' )

   # pylint: disable-next=consider-using-f-string
   return '%sGbE' % '/'.join( compatibilityList )

def getKeyword( context, speed ):
   intfRange = ( context.sharedResult.get( 'SPEED-GROUP' ) or
                 context.sharedResult[ 'INTF' ] )
   speedGroups = []
   for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
      speedGroups.append( IntfRange.intfTypeFromName( name )[ 1 ] )
   helpDescList = []
   statuses = getSpeedGroupStatusesAll( speedGroups )
   if statuses:
      for status in statuses.values():
         compatibility = status.supportedModes.get( tokenToCompatEnum[ speed ] )
         if compatibility:
            helpDesc = ethLinkModeSetToHelpDesc( compatibility )
            if helpDesc not in helpDescList:
               helpDescList.append( helpDesc )
      if not helpDescList:
         # return an empty dictionary to guard this token
         return {}
      helpDescList.sort( key=len )
   else:
      # andreyk_TODO: Fallback hardcoded helpDesc if there are no speedGroupStatuses
      # Add 800g only when we have a product which supports it
      helpDesc = { '10g' : '1/10/40GbE',
                   '25g' : '25/50-2/100-4GbE',
                   '50g' : '50-1/100-2/200-4/400-8GbE',
                   '100g': '100-1/200-2/400-4GbE' } [ speed ]
      helpDescList.append( helpDesc )
   return { speed : 'Allow ' + ' or '.join( helpDescList ) }

def speedGroup10g( mode, context ):
   return getKeyword( context, '10g' )

def speedGroup25g( mode, context ):
   return getKeyword( context, '25g' )

def speedGroup50g( mode, context ):
   return getKeyword( context, '50g' )

def speedGroup100g( mode, context ):
   return getKeyword( context, '100g' )

matcher10g = CliMatcher.DynamicKeywordMatcher( speedGroup10g, passContext=True )
matcher25g = CliMatcher.DynamicKeywordMatcher( speedGroup25g, passContext=True )
matcher50g = CliMatcher.DynamicKeywordMatcher( speedGroup50g, passContext=True )
matcher100g = CliMatcher.DynamicKeywordMatcher( speedGroup100g, passContext=True )

speedSet = {
   '10g' : CliCommand.Node( matcher=matcher10g ),
   '25g' : CliCommand.Node( matcher=matcher25g ),
   '50g' : CliCommand.Node( matcher=matcher50g ),
   '100g': CliCommand.Node( matcher=matcher100g ),
   }

compatEnumToToken = { 'compatibility10g'  : '10g',
                      'compatibility25g'  : '25g',
                      'compatibility50g'  : '50g',
                      'compatibility100g' : '100g' }
tokenToCompatEnum = { y: x for x, y in compatEnumToToken.items() }

class HardwareSpeedGroupCommand( CliCommand.CliCommandClass ):
   syntax = "hardware SPEED-GROUP | INTF serdes COMPAT-SPEEDS"
   noOrDefaultSyntax = "hardware SPEED-GROUP | INTF serdes ..."
   data = {
      "hardware": CliToken.Hardware.hardwareForConfigMatcher,
      "SPEED-GROUP": speedGroupNameNode,
      "INTF": speedGroupNameRangeNode,
      'serdes': 'serdes speed compatibility',
      'COMPAT-SPEEDS': CliCommand.setCliExpression( speedSet, optional=False,
                                                    name='compatibility' )
      }
   @staticmethod
   def handler( mode, args ):
      group = args.get( 'SPEED-GROUP' ) or args[ 'INTF' ]
      compatibility = [ tokenToCompatEnum[ x ] for x in args[ 'compatibility' ] ]
      configSpeedGroup( mode, group, compatibility=compatibility )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      group = args.get( 'SPEED-GROUP' ) or args[ 'INTF' ]
      configSpeedGroup( mode, group, compatibility=None )

BasicCli.GlobalConfigMode.addCommandClass( HardwareSpeedGroupCommand )

def getSpeedGroupStatusesAll( speedGroup ):
   # Returns a map of speedGroup name to speedGroupsStatus. When speedGroup is
   # an empty list, every group in every speedGroupStatusDir is added to the map.
   speedGroupStatuses = {}
   for linecard in speedGroupStatusSliceDir.values():
      for speedGroupStatusDir in linecard.values():
         if not speedGroup:
            speedGroupStatuses.update( speedGroupStatusDir.group )
         else:
            for group in speedGroup:
               status = speedGroupStatusDir.group.get( group )
               if status is not None:
                  speedGroupStatuses[ group ] = status

   return speedGroupStatuses

def getSpeedGroupStatuses( speedGroup ):
   statuses = getSpeedGroupStatusesAll( speedGroup )
   groupStatuses = SpeedGroupModel.SpeedGroupStatuses()
   for name, status in statuses.items():
      groupStatus = SpeedGroupModel.SpeedGroupStatus()
      config = getSpeedGroupConfigDir( name ).group.get( name )
      groupStatuses.groups[ name ] = groupStatus.toModel( status, config,
            ethPhyIntfDefaultConfigDir )

   return groupStatuses

class ShowHardwareSpeedGroupStatusCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show hardware speed-group | SPEED-GROUP status"
   data = {
      "hardware": CliToken.Hardware.hardwareForShowMatcher,
      "speed-group": "Hardware speed group",
      "SPEED-GROUP": IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType, ),
         dollarCompHelpText='Group list end' ),
      "status": "Show speed group status"
      }
   cliModel = SpeedGroupModel.SpeedGroupStatuses
   @staticmethod
   def handler( mode, args ):
      speedGroup = []
      intfRange = args.get( 'SPEED-GROUP' )
      if intfRange:
         for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
            speedGroup.append( IntfRange.intfTypeFromName( name )[ 1 ] )

      return getSpeedGroupStatuses( speedGroup )

BasicCli.addShowCommandClass( ShowHardwareSpeedGroupStatusCommand )

def getSpeedGroupCaps( speedGroup ):
   statuses = getSpeedGroupStatusesAll( speedGroup )
   groupCaps = SpeedGroupModel.SpeedGroupCaps()
   for name, status in statuses.items():
      groupCap = SpeedGroupModel.SpeedGroupCap()
      groupCaps.groups[ name ] = groupCap.toModel( status )

   return groupCaps

class ShowHardwareSpeedGroupConfigCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show hardware speed-group | SPEED-GROUP configurations"
   data = {
      "hardware": CliToken.Hardware.hardwareForShowMatcher,
      "speed-group": "Hardware speed group",
      "SPEED-GROUP": IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType, ),
         dollarCompHelpText='Group list end' ),
      "configurations": "Show available speed configurations"
      }
   cliModel = SpeedGroupModel.SpeedGroupCaps

   @staticmethod
   def handler( mode, args ):
      speedGroup = []
      intfRange = args.get( 'SPEED-GROUP' )
      if intfRange:
         for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
            speedGroup.append( IntfRange.intfTypeFromName( name )[ 1 ] )

      return getSpeedGroupCaps( speedGroup )

BasicCli.addShowCommandClass( ShowHardwareSpeedGroupConfigCommand )

def _updateSpeedGroupHelpDescForModular():
   """ Updates the hardware speed group type to have the correct help description
   for slot/group numbers. """
   SpeedGroupType.helpDesc = [ 'Slot number', 'Group number' ]

FruCli.registerModularSystemCallback( _updateSpeedGroupHelpDescForModular )

def checkSpeedGroupAutoChangeCancel( mode ):
   promptTemplate = "This command may implicitly change the speed group settings " \
                    "causing interfaces to flap or link down."

   if mode.session.commandConfirmation():
      mode.addWarning( promptTemplate )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         mode.addMessage( "Command aborted by user" )
         return True

   return False

def configSpeedGroupAuto( mode, autoMode ):
   if ethPhyIntfDefaultConfigDir.speedGroupAutoMode == autoMode:
      return

   if checkSpeedGroupAutoChangeCancel( mode ):
      return

   ethPhyIntfDefaultConfigDir.speedGroupAutoMode = autoMode

class HardwareSpeedGroupAutoCommand( CliCommand.CliCommandClass ):
   syntax = "hardware speed-group default single-slot auto"
   noOrDefaultSyntax = syntax
   data = {
      "hardware": CliToken.Hardware.hardwareForConfigMatcher,
      "speed-group": "Hardware speed group",
      "default": "default configuration",
      "single-slot": "Single transceiver slot",
      "auto": "Auto-configuration",
      }

   @staticmethod
   def handler( mode, args ):
      configSpeedGroupAuto( mode, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      configSpeedGroupAuto( mode, False )

BasicCli.GlobalConfigMode.addCommandClass( HardwareSpeedGroupAutoCommand )

def Plugin( em ):
   global capsDefaultRootDir
   global productConfig
   global speedGroupConfigSliceDir
   global speedGroupStatusSliceDir
   global ethPhyIntfDefaultConfigDir
   capsDefaultRootDir = LazyMount.mount(
      em, MountConstants.l1PolicyDefaultCapsRootDirPath(), 'Tac::Dir', 'ri' )
   productConfig = LazyMount.mount(
      em, MountConstants.productConfigRootDirPath(), 'L1::ProductConfig', 'ri' )
   speedGroupConfigSliceDir = ConfigMount.mount( em,
         "interface/archer/config/eth/phy/speedgroup/slice", "Tac::Dir", "wi" )
   speedGroupStatusSliceDir = LazyMount.mount( em,
         "interface/archer/status/eth/phy/speedgroup/slice", "Tac::Dir", "ri" )
   ethPhyIntfDefaultConfigDir = ConfigMount.mount(
                         em,
                         "interface/config/eth/phy/globalDefault",
                         "Interface::EthPhyIntfGlobalDefaultConfigDir", "w" )
