# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool, Model, Enum, List, Dict
from EthIntfLib import (
   linkModeCapabilitiesToList,
   pllSpeedGroupSettingStrMap,
)
from EthIntfLib.Types import COMPAT_AUTO
from Intf.IntfRange import intfListToCanonical
from CliPlugin.EthIntfModel import CapabilitiesMixIn
from IntfModels import Interface
import Tac
from TypeFuture import TacLazyType

EthTypesApi = TacLazyType( 'Interface::EthTypesApi' )

class SerdesSpeed( Model ):
   speed = Enum( values=pllSpeedGroupSettingStrMap, help='SerDes speed' )

class SpeedGroupStatus( Model ):
   interfaces = List( valueType=Interface, help='Group member interfaces' )
   masterInterface = Interface( help='Master interface of the group' )
   supportedSerdesSpeeds = List( valueType=SerdesSpeed,
                                 help='Supported serdes speeds' )
   defaultSerdesSpeeds = List( valueType=SerdesSpeed, help='Default serdes speeds' )
   configuredSerdesSpeeds = List( valueType=SerdesSpeed,
                                  help='Configured serdes speeds' )
   activeSerdesSpeeds = List( valueType=SerdesSpeed, help='Active serdes speeds' )
   speedGroupAutoMode = Bool( optional=True,
                              help='Auto-configure speed-group setting' )

   def toModel( self, status, config, globalConfig ):
      self.interfaces = list( status.memberIntf )
      self.masterInterface = status.masterIntf
      self.defaultSerdesSpeeds = self._enumsToSerdesSpeeds(
            list( status.settingPriority.values() )[ : status.settingCount ] )
      self.configuredSerdesSpeeds = self._enumsToSerdesSpeeds(
            list( config.setting ) if config else [] )
      self.activeSerdesSpeeds = self._enumsToSerdesSpeeds(
            list( status.setting ) )

      if COMPAT_AUTO in set( status.settingPriority.values() ):
         self.supportedSerdesSpeeds = self._enumsToSerdesSpeeds(
               list( status.supportedModes ) )
         self.speedGroupAutoMode = True
      else:
         self.supportedSerdesSpeeds = self._enumsToSerdesSpeeds(
               list( status.settingPriority.values() ) )
         if globalConfig.speedGroupAutoMode and status.speedGroupAutoSupported:
            self.speedGroupAutoMode = True

      return self

   def _enumsToSerdesSpeeds( self, enums ):
      return [ SerdesSpeed(
         speed=EthTypesApi.speedGroupSettingToStr( e ) ) for e in enums ]

   def _serdesSpeedsString( self, serdesSpeeds ):
      return ','.join( sorted( [ s.speed for s in serdesSpeeds ] ) )

   def supportedSerdesSpeedsString( self ):
      return self._serdesSpeedsString( self.supportedSerdesSpeeds )

   def defaultSerdesSpeedsString( self ):
      if self.masterInterface:
         # pylint: disable-next=consider-using-f-string
         return "Per %s" % self.masterInterface.shortName
      elif self.speedGroupAutoMode:
         return "auto"
      else:
         return self._serdesSpeedsString( self.defaultSerdesSpeeds )

   def configuredSerdesSpeedsString( self ):
      if self.configuredSerdesSpeeds:
         return self._serdesSpeedsString( self.configuredSerdesSpeeds )
      else:
         return 'default'

   def activeSerdesSpeedsString( self ):
      if self.activeSerdesSpeeds:
         return self._serdesSpeedsString( self.activeSerdesSpeeds )
      elif self.masterInterface:
         # pylint: disable-next=consider-using-f-string
         return "Per %s" % self.masterInterface.shortName
      else:
         return 'unknown'

def firstIntfKey( item ):
   '''Key function for sorting by first interface.
   '''
   _, value = item
   return Arnet.intfNameKey( value.interfaces[ 0 ] )

class SpeedGroupStatuses( Model ):
   groups = Dict( keyType=str, valueType=SpeedGroupStatus,
         help="Mapping between a speed group's name and its status" )

   def render( self ):
      if not self.groups:
         return

      colWidths = ( 5, 17, 11, 11, 11, 11 )
      # pylint: disable-next=consider-using-f-string
      hdrFmt = ' '.join( '%%-%ds' % w for w in colWidths )
      print( hdrFmt % ( 'Group', 'Interfaces', 'Supported',
                        'Default', 'Configured', 'Active' ) )
      print( hdrFmt % tuple( '-' * w for w in colWidths ) )

      for group, data in sorted( self.groups.items(), key=firstIntfKey ):
         print( hdrFmt % ( group,
                           ','.join( intfListToCanonical( data.interfaces,
                                                          noHoleRange=True ) ),
                           data.supportedSerdesSpeedsString(),
                           data.defaultSerdesSpeedsString(),
                           data.configuredSerdesSpeedsString(),
                           data.activeSerdesSpeedsString() ) )

class SpeedGroupCap( Model ):
   interfaces = List( valueType=Interface, help='Group member interfaces' )
   availableSpeedConfigurations = Dict( keyType=str, valueType=str,
         help='Mapping between a SerDes speed and available speed configurations' )

   def showLanesInStr( self, status ):
      if COMPAT_AUTO in status.settingPriority.values():
         settings = sorted( status.supportedModes )
      else:
         settings = status.settingPriority.values()

      for setting in settings:
         caps = status.supportedModes[ setting ]
         if( caps.mode50GbpsFull1Lane or caps.mode100GbpsFull1Lane or
             caps.mode100GbpsFull2Lane or caps.mode200GbpsFull2Lane or
             caps.mode200GbpsFull4Lane or caps.mode200GbpsFull8Lane or
             caps.mode400GbpsFull4Lane or caps.mode400GbpsFull8Lane or
             caps.mode800GbpsFull8Lane ):
            return True
      return False

   def toModel( self, status ):
      self.interfaces = list( status.memberIntf )
      showLanesInStr = self.showLanesInStr( status )

      if COMPAT_AUTO in list( status.settingPriority.values() ):
         settings = sorted( status.supportedModes )
      else:
         settings = status.settingPriority.values()

      for setting in settings:
         caps = status.supportedModes[ setting ]
         lmCapList = linkModeCapabilitiesToList( caps, showLanes=showLanesInStr )
         ethModes = CapabilitiesMixIn.linkModeCapsToEthSpeedModel(
            lmCapList, False )[ 0 ]
         lmCapStrList = []
         for ethMode in ethModes:
            tmpStr = CapabilitiesMixIn.formatSpeedLanesDuplex(
               ethMode.speed, ethMode.laneCount, ethMode.showLanes(), ethMode.duplex,
               False, False )
            lmCapStrList.append( tmpStr )
         lmCapStr = ','.join( lmCapStrList )
         self.availableSpeedConfigurations[
            setting.lower().removeprefix( 'compatibility' ) ] = lmCapStr
      return self

class SpeedGroupCaps( Model ):
   groups = Dict( keyType=str, valueType=SpeedGroupCap,
         help="Mapping between a speed group's name and its capabilities" )

   def render( self ):
      if not self.groups:
         return

      colWidths = ( 5, 17, 12, 46 )
      # pylint: disable-next=consider-using-f-string
      hdrFmt = ' '.join( '%%-%ds' % w for w in colWidths )
      print( hdrFmt % ( 'Group', 'Interfaces', 'Select Value',
                        'Available Speed Configurations' ) )
      print( hdrFmt % tuple( '-' * w for w in colWidths ) )
      for group, data in sorted( self.groups.items(), key=firstIntfKey ):
         availSpeedConfigsDict = data.availableSpeedConfigurations
         for setting, string in sorted( availSpeedConfigsDict.items() ):
            print( hdrFmt % ( group,
                              ','.join( intfListToCanonical( data.interfaces,
                                                             noHoleRange=True ) ),
                              setting,
                              string ) )
