# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
import ShowCommand
import SharedMem
import Smash
import Tac

from Arnet import IpGenAddr
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.SrTePolicyLibCli import getSegmentListVias
from CliPlugin.SrP2mpModel import ( ReplicationSegmentSharedStatModel,
                                    ReplicationSegmentStateModel,
                                    ReplicationSegmentRoot,
                                    ReplicationSegmentTree,
                                    ReplicationSegmentPath,
                                    ReplicationSegmentStatisticsModel,
                                    ReplicationSegmentSummaryModel,
                                    ReplicationSegmentSummaryVrfModel,
                                    SrP2mpCandidatesVrfModel,
                                    SrP2mpCandidatesModel,
                                    SrP2mpCandidates,
                                    SrP2mpRepSegSegmentList,
                                    SrP2mpReplicationSid,
                                    SrP2mpSegment )
from CliPlugin.SrTePolicyCli import ( SrTeMode,
                                      maxLabelStackSize )
from CliPlugin.TunnelCli import ( TunnelTableIdentifier,
                                  readMountTunnelTable )
from CliMode.SrTePolicy import ReplicationSegmentModeBase
from CliToken.SegmentRoutingToken import matcherSegmentRoutingForShow
from IpLibConsts import DEFAULT_VRF
from SrTePolicyLib import ( MplsLabel,
                            MplsLabelOperation,
                            tacProtocol,
                            tacSegmentListId,
                            policySrcToString )
from SrTePolicy import ( tacIndex,
                         Index )
from SrTePolicyCommonLib import ( ReplicationSegmentKey,
                                  tacDownstreamNode )

from Toggles.SrTePolicyToggleLib import toggleReplicationSegmentEnabled

repSegConfig = None
mplsConfig = None
repSegAll = None
allRepSeg = None
allSl = None
pcepInpRepSeg = None
staticInpRepSeg = None

slStatus = None
slTunnelTable = None
ID_MAX = 4294967295

matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Traffic Engineering related information' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Show policies configured statically' )

# ReplicationMode
class ReplicationSegmentMode( ReplicationSegmentModeBase, BasicCli.ConfigModeBase ):
   name = "Replication Segment Configuration"

   def __init__( self, parent, session, repSeg ):
      ReplicationSegmentModeBase.__init__( self, repSeg )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.repSeg = repSeg

#--------------------------------------------------------
# [ no | default ] replication-segment  <replication-id>
#--------------------------------------------------------
class CfgReplicationSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'replication-segment REPID'
   noOrDefaultSyntax = syntax
   data = {
      'replication-segment': 'Replication Segment',
      'REPID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                  helpdesc='Replication segment identifier' ),
   }

   @staticmethod
   def handler( mode, args ):
      repId = args[ 'REPID' ]
      if repId not in repSegConfig.staticReplicationSegment:
         repSeg = repSegConfig.staticReplicationSegment.newMember( repId )
         repSeg.repSegKey = ReplicationSegmentKey( repId )
      else:
         repSeg = repSegConfig.staticReplicationSegment[ repId ]

      childMode = mode.childMode( ReplicationSegmentMode, repSeg=repSeg )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      repId = args[ 'REPID' ]
      del repSegConfig.staticReplicationSegment[ repId ]

if toggleReplicationSegmentEnabled():
   SrTeMode.addCommandClass( CfgReplicationSegmentCmd )

# --------------------------------------------------------
# root ROOT tree-id TID instance-id INSTID
# --------------------------------------------------------
class CfgReplicationSegmentRootTreeInstCmd( CliCommand.CliCommandClass ):
   syntax = 'root ROOT tree-id TID instance-id INSTID'
   noOrDefaultSyntax = syntax
   data = {
      'root': 'Root-ID of the policy mapped with this replication segment'
      ' as an IPv4/IPv6 address',
      'ROOT': IpGenAddrMatcher( 'Root-ID of the policy mapped with '
            'this replication segment as an IP address' ),
      'tree-id': 'Tree-ID of the P2MP policy mapped',
      'TID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                           helpdesc='Tree-ID of the policy mapped with '
                                 'this replication segment ' ),
      'instance-id': 'ID of the path-instance mapped with '
                     'this replication segment',
      'INSTID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                           helpdesc='ID of the path-instance mapped with '
                                 'this replication segment' ),
   }

   @staticmethod
   def _getRepSegKey( args, repId ):
      root = args[ 'ROOT' ]
      treeId = args[ 'TID' ]
      instanceId = args[ 'INSTID' ]
      return ReplicationSegmentKey( repId=repId, root=str( root ), treeId=treeId,
            instanceId=instanceId )

   @staticmethod
   def handler( mode, args ):
      replicationId = mode.repSeg.replicationId
      key = CfgReplicationSegmentRootTreeInstCmd._getRepSegKey( args, replicationId )
      for repId in repSegConfig.staticReplicationSegment:
         repSegKey = repSegConfig.staticReplicationSegment[ repId ].repSegKey
         # if same root, treeId, instanceId is already configured with some other
         # replication segment, don't allow this config
         if ( key.rootAddress == repSegKey.rootAddress and
         key.treeId == repSegKey.treeId and
         key.instanceId == repSegKey.instanceId and
         key.replicationId != repSegKey.replicationId ):
            mode.addError( 'Root, TreeId, InstanceId is already used' )
            return
      repSegConfig.staticReplicationSegment[ replicationId ].repSegKey = key
      mode.repSeg.repSegKey = key

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      replicationId = mode.repSeg.replicationId
      key = ReplicationSegmentKey( repId=replicationId )
      mode.repSeg.repSegKey = key

ReplicationSegmentMode.addCommandClass( CfgReplicationSegmentRootTreeInstCmd )

# --------------------------------------------------------
# name NAME
# --------------------------------------------------------
class CfgReplicationSegmentNameCmd( CliCommand.CliCommandClass ):
   syntax = 'name NAME'
   noOrDefaultSyntax = syntax
   data = {
      'name': 'Symbolic name of the replication segment',
      'NAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9+/=]+',
                                         helpdesc='Symbolic name of the'
                                         ' replication segment',
                                         helpname='WORD' ),
      }

   @staticmethod
   def handler( mode, args ):
      mode.repSeg.replicationSegmentName = args[ 'NAME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.repSeg.replicationSegmentName = ''

ReplicationSegmentMode.addCommandClass( CfgReplicationSegmentNameCmd )

#--------------------------------------------------------
# desc DESC
#--------------------------------------------------------
class CfgReplicationSegmentDescCmd( CliCommand.CliCommandClass ):
   syntax = 'desc DESC'
   noOrDefaultSyntax = syntax
   data = {
      'desc': 'Description of the replication segment',
      'DESC': CliMatcher.StringMatcher(
         helpname='LINE',
         helpdesc='Description of the replication segment' )
      }

   @staticmethod
   def handler( mode, args ):
      mode.repSeg.desc = args[ 'DESC' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.repSeg.desc = ''

ReplicationSegmentMode.addCommandClass( CfgReplicationSegmentDescCmd )

def srlbRangeFunc( mode, context ):
   '''
   Return a tuple
   ( srlbRange.base,  srlbRange.base + srlbRange.size )
   '''
   srlbRange = mplsConfig.labelRange[ 'srlb' ]
   return ( srlbRange.base, srlbRange.base + srlbRange.size )

#--------------------------------------------------------
# replication-sid  <sid>
#--------------------------------------------------------
class CfgReplicationSid( CliCommand.CliCommandClass ):
   syntax = 'replication-sid RSID '
   noOrDefaultSyntax = 'replication-sid ...'
   data = {
         'replication-sid': 'Replication SID',
         'RSID': CliMatcher.DynamicIntegerMatcher(
            srlbRangeFunc,
            helpdesc='MPLS Label for this replication SID in SRLB range' )
         }

   @staticmethod
   def handler( mode, args ):
      mode.repSeg.replicationSid = MplsLabel( args[ 'RSID' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.repSeg.replicationSid == MplsLabel():
         return
      mode.repSeg.replicationSid = MplsLabel()

ReplicationSegmentMode.addCommandClass( CfgReplicationSid )

# ------------------------------------------------------------
# [ no | default ] segment-list index INDEX
# ( downstream-rsid RSID   next-hop NHOP ) |
# ( label-stack TOP_LABEL RSID ) [ desc DESC ]
# ------------------------------------------------------------
class CfgReplicationSegmentList( CliCommand.CliCommandClass ):
   syntax = '''segment-list index INDEX
         ( ( downstream-rsid RSID next-hop NHOP ) |
         ( label-stack TOP_LABEL RSID ) ) [ desc DESC ]'''
   noOrDefaultSyntax = '''segment-list index INDEX
      [ ( downstream-rsid RSID next-hop NHOP ) |
      ( label-stack TOP_LABEL RSID ) ] [ desc DESC ]'''
   data = {
         'segment-list': 'Segment list',
         'index': 'Index of Segment list',
         'INDEX': CliMatcher.IntegerMatcher( tacIndex.min + 1, tacIndex.max,
                                             helpdesc='' ),
         'downstream-rsid': "Downstream node's RSID",
         'RSID': CliCommand.Node(
                     matcher=CliMatcher.IntegerMatcher( MplsLabel.min,
                                                        MplsLabel.max,
                                                        helpdesc='RSID' ),
                     maxMatches=maxLabelStackSize() ),
         'next-hop': 'Next hop IP address',
         'NHOP': IpGenAddrMatcher( 'Next hop address in IP format' ),
         'label-stack': 'Label stack',
         'TOP_LABEL': CliCommand.Node(
            matcher=CliMatcher.IntegerMatcher( MplsLabel.min, MplsLabel.max,
               helpdesc='The label on top of the stack' ),
            maxMatches=maxLabelStackSize() ),
         'desc': 'Description',
         'DESC': CliMatcher.StringMatcher( helpname='LINE',
                              helpdesc='Description of the segment list' )
         }

   @staticmethod
   def _segListToLabelStack( *segments ):
      newSegList = MplsLabelOperation()
      index = 0
      for index, segment in enumerate( segments ):
         newSegList.labelStackIs( index, MplsLabel( int( segment ) ) )
      newSegList.stackSize = index + 1
      return newSegList

   @staticmethod
   def _segStackFromArgs( args ):
      if 'label-stack' in args:
         labelStack = CfgReplicationSegmentList._segListToLabelStack(
               args[ 'TOP_LABEL' ], args[ 'RSID' ] )
      else:
         labelStack = CfgReplicationSegmentList._segListToLabelStack(
               args[ 'RSID' ] )
      if 'next-hop' in args:
         nexthop = args[ 'NHOP' ]
         segStack = tacDownstreamNode( IpGenAddr( str( nexthop ) ), labelStack )
         return segStack
      segStack = tacDownstreamNode( IpGenAddr(), labelStack )
      return segStack

   @staticmethod
   def handler( mode, args ):
      segStack = CfgReplicationSegmentList._segStackFromArgs( args )
      segList = mode.repSeg.staticSegmentList.get( segStack )
      index = args.get( 'INDEX' )
      if segList is not None:
         if segList.index != index:
            # Same downstream rsid and nexthop has been configured for a different
            # segment list, so this config won't be allowed
            # pylint: disable-next=consider-using-f-string
            mode.addError( 'Index mismatch, expected index %d' % segList.index )
         return

      for st in mode.repSeg.staticSegmentList:
         sl = mode.repSeg.staticSegmentList[ st ]
         if st != segStack and index == sl.index:
            # Overriding the existing SL
            del mode.repSeg.staticSegmentList[ st ]
      segList = Tac.Value(
            'SrTePolicy::SrP2mp::StaticSegmentList', Index( index ) )
      desc = args.get( 'DESC' )
      if desc:
         segList.desc = args[ 'DESC' ]
      mode.repSeg.staticSegmentList[ segStack ] = segList

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      index = args.get( 'INDEX' )
      for st in mode.repSeg.staticSegmentList:
         sl = mode.repSeg.staticSegmentList[ st ]
         if index == sl.index:
            del mode.repSeg.staticSegmentList[ st ]

ReplicationSegmentMode.addCommandClass( CfgReplicationSegmentList )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing replication-segment summary
# -----------------------------------------------------------------------
def populateSrcStat():
   sourceDict = { tacProtocol.staticProtocol: 0,
                  tacProtocol.pcepProtocol: 0 }

   activeRepSeg = { tacProtocol.staticProtocol: 0,
                    tacProtocol.pcepProtocol: 0,
                    tacProtocol.invalidProtocol:0 }
   validRepSeg = { tacProtocol.staticProtocol: 0,
                   tacProtocol.pcepProtocol: 0,
                   tacProtocol.invalidProtocol: 0 }
   invalidRepSeg = { tacProtocol.staticProtocol: 0,
                     tacProtocol.pcepProtocol: 0,
                     tacProtocol.invalidProtocol:0 }

   repSegSrcCount = ReplicationSegmentStatisticsModel()
   totalRepSeg = ReplicationSegmentStateModel()
   repSegStateDict = { 'Active': activeRepSeg,
                       'ConflictingRsid': validRepSeg,
                       'ConflictingKey': validRepSeg,
                       'Invalid': invalidRepSeg }
   for repSegKey, state in repSegAll.statusEntry.items():
      repSegStateDict[ state.status ][ repSegKey.protocol ] += 1

   for source in sourceDict:
      repSegSrcEntry = ReplicationSegmentStateModel()
      # ACTIVE
      repSegSrcEntry.active = activeRepSeg[ source ]
      # VALID
      repSegSrcEntry.valid = validRepSeg[ source ]
      # INVALID
      repSegSrcEntry.invalid = invalidRepSeg[ source ]
      # TOTAL per Source
      totalRepSeg.active += repSegSrcEntry.active
      totalRepSeg.valid += repSegSrcEntry.valid
      totalRepSeg.invalid += repSegSrcEntry.invalid
      repSegSrcCount.sources[ policySrcToString( source ) ] = repSegSrcEntry
   repSegSrcCount.totalStat = totalRepSeg
   return repSegSrcCount

def populateSharedStat():
   repSegSharedCnt = ReplicationSegmentSharedStatModel()
   repSegSharedDict = repSegSharedCnt.replicationIds
   for repSegKey, state in repSegAll.statusEntry.items():
      repSegStatus = state.status
      repSegEntry = ReplicationSegmentStateModel()
      repId = repSegKey.replicationId
      repSegStatusDict = { 'Active': 0,
                           'Invalid': 0,
                           'ConflictingRsid': 0,
                           'ConflictingKey': 0
                          }
      # Shared replication-segment has zero ip address
      if repSegKey.rootAddress.isAddrZero:
         repSegStatusDict[ repSegStatus ] = 1
         repSegEntry.active = repSegStatusDict[ 'Active' ]
         repSegEntry.valid = repSegStatusDict[ 'ConflictingKey' ] + \
                             repSegStatusDict[ 'ConflictingRsid' ]
         repSegEntry.invalid = repSegStatusDict[ 'Invalid' ]
         if repId not in repSegSharedDict:
            repSegSharedDict[ repId ] = repSegEntry
         else:
            repSegSharedDict[ repId ].active += repSegEntry.active
            repSegSharedDict[ repId ].valid += repSegEntry.valid
            repSegSharedDict[ repId ].invalid += repSegEntry.invalid
   return repSegSharedCnt

def populateNonSharedStat():
   rootCliModel = ReplicationSegmentRoot()
   rootsDict = rootCliModel.roots
   for repSegKey, state in repSegAll.statusEntry.items():
      statusCliModel = ReplicationSegmentStateModel()
      repSegStatusDict = { 'Active': 0,
                           'Invalid': 0,
                           'ConflictingKey': 0,
                           'ConflictingRsid': 0
                          }
      rootAddr = repSegKey.rootAddress.stringValue
      treeId = repSegKey.treeId
      instanceId = repSegKey.instanceId
      # Non-shared replication-segment has rootAddress non-zero ip address
      if not repSegKey.rootAddress.isAddrZero:
         repSegStatusDict[ state.status ] = 1
         statusCliModel.active = repSegStatusDict[ 'Active' ]
         statusCliModel.valid = repSegStatusDict[ 'ConflictingKey' ] + \
                                repSegStatusDict[ 'ConflictingRsid' ]
         statusCliModel.invalid = repSegStatusDict[ 'Invalid' ]
         if rootAddr not in rootsDict:
            rootsDict[ rootAddr ] = ReplicationSegmentTree()
         if treeId not in rootsDict[ rootAddr ].trees:
            rootsDict[ rootAddr ].trees[ treeId ] = ReplicationSegmentPath()
         if instanceId not in rootsDict[ rootAddr ].trees[ treeId ].paths:
            rootsDict[ rootAddr ].trees[ treeId ].paths[ instanceId ] = \
                                                                      statusCliModel
         else:
            statusDict = rootsDict[ rootAddr ].trees[ treeId ].paths[ instanceId ]
            statusDict.active += statusCliModel.active
            statusDict.valid += statusCliModel.valid
            statusDict.invalid += statusCliModel.invalid
   return rootCliModel

def countRepSegRsidConflictCount():
   return sum( repSegState.status == 'ConflictingRsid'
               for repSegState in repSegAll.statusEntry.values() )

class ShowReplicationSegmentSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing replication-segment summary'
   data = {
       'traffic-engineering': matcherTrafficEngineering,
       'segment-routing': matcherSegmentRoutingForShow,
       'replication-segment': 'Replication Segment',
       'summary': 'Show summary of Segment Routing Replication Segments',
   }
   cliModel = ReplicationSegmentSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      vrfModel = ReplicationSegmentSummaryVrfModel()
      summary = ReplicationSegmentSummaryModel()
      perSourceRepSegStat = populateSrcStat()
      nonSharedRepSegStat = populateNonSharedStat()
      sharedRepSegStat = populateSharedStat()
      summary.protocolStats = perSourceRepSegStat
      summary.nonSharedStatistics = nonSharedRepSegStat
      summary.sharedStatistics = sharedRepSegStat
      summary.replicationSegmentsRsidConflictCount = \
                         countRepSegRsidConflictCount()
      vrfModel.vrfs[ DEFAULT_VRF ] = summary
      return vrfModel

if toggleReplicationSegmentEnabled():
   BasicCli.addShowCommandClass( ShowReplicationSegmentSummaryCmd )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing replication-segment *
# -----------------------------------------------------------------------
def getStateFilter( args ):
   state = args.get( 'STATUS' )
   if state == 'valid':
      return [ 'ConflictingKey', 'ConflictingRsid' ]
   elif state == 'active':
      return [ 'Active' ]
   elif state == 'invalid':
      return [ 'Invalid' ]
   return [ 'ConflictingKey', 'Invalid', 'Active', 'ConflictingRsid' ]

def getSourceFilter( args ):
   source = args.get( 'SRC' )
   if source == 'static':
      return [ tacProtocol.staticProtocol ]
   elif source == 'pcep':
      return [ tacProtocol.pcepProtocol ]
   else:
      return [ tacProtocol.staticProtocol, tacProtocol.pcepProtocol ]

def populateSegmentLists( wsl ):
   sl = SrP2mpRepSegSegmentList()
   sl.segmentListId = wsl.segmentListId
   sl.vias = []
   sl.backupVias = []
   slId = sl.segmentListId
   updateReason = True

   tunnelId = tacSegmentListId( slId ).getTunnelId()
   segList = allSl.segmentList.get( slId )
   if segList is not None:
      for i in range( segList.size ):
         if segList.segment[ i ].type ==\
               Tac.Type( "SrTePolicy::SegmentType" ).ipNexthop:
            sl.segments.append( SrP2mpSegment(
             nexthopAddr=segList.segment[ i ].nexthop ) )
         else:
            sl.segments.append( SrP2mpSegment(
               mplsLabelSid=segList.segment[ i ].mplsLabel ) )

   if segList is None or segList.size == 0:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "segmentListMissing"
      return sl

   segListStatus = slStatus.status.get( slId, None )
   if segListStatus is not None:
      segListStatus = segListStatus.resolveError
   if segListStatus != 'success':
      sl.segmentListValid = False
      if updateReason:
         if segListStatus is None:
            sl.invalidReason = 'unresolvedTopLabel'
         elif segListStatus == 'zeroLabels':
            sl.invalidReason = 'noResolvedLabels'
         elif segListStatus == 'platformLabelStackDepthExceeded':
            sl.invalidReason = 'resolvedLabelsExceedPlatformMsd'
      return sl

   tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
   if tunnelEntry is None:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "unresolvedTopLabel"
      return sl

   sl.segmentListValid = True
   vias, backupVias = getSegmentListVias( tunnelEntry )
   sl.vias = vias if vias else []
   sl.backupVias = backupVias if backupVias else []
   return sl

def repSegInputTableForProtocol( protocol ):
   if protocol == 'staticProtocol':
      return staticInpRepSeg
   elif protocol == 'pcepProtocol':
      return pcepInpRepSeg
   return None

def populateReplicationSegments( sourceFilter, stateFilter, repId, root,
                                 treeId, instId, sharedStatus ):
   keys = []
   if repId:
      for key in repSegAll.statusEntry:
         if key.replicationId == repId:
            keys.append( key )
   elif root:
      for key in repSegAll.statusEntry:
         croot = key.rootAddress.stringValue
         ctreeId = key.treeId
         cinstanceId = key.instanceId
         strRoot = root.stringValue
         if treeId  and instId:
            if croot == strRoot and ctreeId == treeId and cinstanceId == instId:
               keys.append( key )
         elif treeId:
            if croot == strRoot and ctreeId == treeId:
               keys.append( key )
         else:
            if croot == strRoot:
               keys.append( key )
   elif sharedStatus:
      if sharedStatus == 'shared':
         for key in repSegAll.statusEntry:
            if key.rootAddress.isAddrZero:
               keys.append( key )
      else:
         for key in repSegAll.statusEntry:
            if not key.rootAddress.isAddrZero:
               keys.append( key )
   else:
      # pylint: disable-next=unnecessary-comprehension
      keys = [ key for key in repSegAll.statusEntry ]

   for key in sorted( keys ):
      repSegStatus =  repSegAll.statusEntry.get( key )
      if repSegStatus.status in stateFilter and key.protocol in sourceFilter:
         repSeg = SrP2mpCandidates()
         repKeySelEntry = allRepSeg.replicationSegmentKeySelectionEntry.get( key )
         SysSource = key.protocol
         table = repSegInputTableForProtocol( SysSource )
         if table is None:
            continue
         repSegInputEntry = table.replicationSegmentInputEntry.get( key )
         if repSegInputEntry is None:
            continue
         SysState = repSegStatus.status
         protocol = 'static'
         state = None
         if SysSource == 'pcepProtocol':
            protocol = 'pcep'
         # pylint: disable-next=consider-using-in
         if SysState == 'ConflictingKey' or SysState == 'ConflictingRsid':
            state = 'valid'
         elif SysState == 'Invalid':
            state = 'invalid'
         else:
            state = 'active'
         repSeg.source = protocol
         repSeg.state = state
         repSeg.replicationSid = SrP2mpReplicationSid( mplsLabelSid =
                                                    repSegInputEntry.replicationSid )
         repSeg.root = key.rootAddress
         repSeg.treeId = key.treeId
         repSeg.pathInstance = key.instanceId
         repSeg.nodeAddress = key.nodeAddress
         if repKeySelEntry:
            for index in sorted( repKeySelEntry.wSegmentList ):
               wsl = repKeySelEntry.wSegmentList.get( index )
               if wsl is not None:
                  repSeg.segmentLists.append(
                        populateSegmentLists( wsl )
                    )
         yield key.replicationId, repSeg

class ShowSrP2mpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing replication-segment
            [ ( SHARED [ STATUS ] [ SRC ] )
            | ( [ STATUS ] [ SRC ]
              [ ( replication-id REP_ID )
              | ( root ROOT [ tree-id TREE_ID [ path-instance INSTANCE_ID ] ] ) ] ) ]
              '''
   data = {
        'traffic-engineering': matcherTrafficEngineering,
        'segment-routing': matcherSegmentRoutingForShow,
        'replication-segment': 'Replication Segment',
        'SHARED': CliMatcher.EnumMatcher( {
             'shared' : 'Show all replication segments that are shared',
             'non-shared' : 'Show all replication segments that are non-shared' } ),
        'SRC': CliMatcher.EnumMatcher( {
             'pcep': 'Show replication segment learnt via pcep',
             'static': 'Show replication segment configured via cli' } ),
        'STATUS':  CliMatcher.EnumMatcher( {
             'active': 'Show active replication segments',
             'valid' : 'Show valid replication segments',
             'invalid': 'Show invalid replication segments'} ),
        'root': 'Root-ID of the policy mapped with this replication segment'
                ' as an IPv4/Ipv6 address',
        'ROOT': IpGenAddrMatcher( 'Root-ID of the policy mapped with '
                'this replication segment as an IP address' ),
        'tree-id': 'Tree-ID of the P2MP policy mapped',
        'TREE_ID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                   helpdesc='Tree-ID of the policy mapped with '
                            'this replication segment ' ),
        'path-instance': 'ID of the path-instance mapped with '
                         'this replication segment',
        'INSTANCE_ID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                       helpdesc='ID of the path-instance mapped with '
                                'this replication segment' ),
        'replication-id': 'Replication segment identifier',
        'REP_ID': CliMatcher.IntegerMatcher( 0, ID_MAX,
                helpdesc='Replication segment identifier' ),
   }
   cliModel = SrP2mpCandidatesVrfModel

   @staticmethod
   def handler( mode, args ):
      root = args.get( 'ROOT' )
      sharedStatus = args.get( 'SHARED' )
      treeId = args.get( 'TREE_ID' )
      instanceId = args.get( 'INSTANCE_ID' )
      repId = args.get( 'REP_ID' )
      stateFilter = getStateFilter( args )
      sourceFilter = getSourceFilter( args )
      vrfModel = SrP2mpCandidatesVrfModel()
      candidates = SrP2mpCandidatesModel()
      candidates.replicationSegments = populateReplicationSegments( sourceFilter,
                       stateFilter, repId, root, treeId, instanceId, sharedStatus )
      vrfModel.vrfs[ DEFAULT_VRF ] = candidates
      return vrfModel

if toggleReplicationSegmentEnabled():
   BasicCli.addShowCommandClass( ShowSrP2mpCmd )

def Plugin( entityManager ):
   global repSegConfig
   global mplsConfig
   global repSegAll
   global allRepSeg
   global staticInpRepSeg
   global pcepInpRepSeg
   global allSl
   global slStatus
   global slTunnelTable

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( "reader" )
   repSegAll = smashEm.doMount(
                       "te/segmentrouting/replicationsegment/reportBackStatus",
                       "SrTePolicy::SrP2mp::ReportBackStatus", readerInfo )
   repSegConfig = ConfigMount.mount( entityManager,
                                 "te/segmentrouting/replicationsegment/config",
                                 "SrTePolicy::SrP2mp::Config", "w" )
   mplsConfig = LazyMount.mount( entityManager,
                                 "routing/mpls/config",
                                 "Mpls::Config", "r" )
   allRepSeg = LazyMount.mount( entityManager,
                              "te/segmentrouting/replicationsegment/allRepSeg",
                              "SrTePolicy::SrP2mp::ReplicationSegmentSelectionInput",
                              "r" )
   pcepInpRepSeg = smashEm.doMount( "replicationSegment/Pcep",
                                    "SrTePolicy::SrP2mp::ReplicationSegmentInput",
                                    readerInfo )
   staticInpRepSeg = smashEm.doMount( "replicationSegment/Static",
                                      "SrTePolicy::SrP2mp::ReplicationSegmentInput",
                                      readerInfo )
   allSl = LazyMount.mount( entityManager,
                            "te/segmentrouting/srtepolicy/allsegmentlist",
                            "SrTePolicy::Export::SegmentListCollection",
                            "r" )
   slStatus = LazyMount.mount( entityManager,
                               "te/segmentrouting/srtepolicy/segmentliststatus",
                               "SrTePolicy::SegmentListStatus",
                               "r" )
   slTunnelTable = readMountTunnelTable(
                   TunnelTableIdentifier.srTeSegmentListTunnelTable, entityManager )
