#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Arnet
from CliModel import Bool, Enum, Model, Submodel
from ArnetModel import IpGenericPrefix
import Tac

class Srv6SidFlavorFlags( Model ):
   nextCsid = Bool( optional=True, help="NEXT-CSID flavor" )
   nextOnlyCsid = Bool( optional=True, help="NEXT-ONLY-CSID flavor" )
   psp = Bool( optional=True,
               help="Penultimate Segment Pop (PSP) of the SRH flavor" )
   usp = Bool( optional=True,
               help="Ultimate Segment Pop (USP) of the SRH flavor" )
   usd = Bool( optional=True,
               help="Ultimate Segment Decapsulation (USD) flavor" )

class Srv6Sid( Model ):
   '''Representation of an SRv6 SID'''
   sid = IpGenericPrefix( help='SID value (IPv6 prefix)' )
   behavior = Enum( values=( 'unknown', 'end', 'endX', 'endDx4', 'endDx6',
                             'endDt4', 'endDt6' ),
                    help='SID behavior' )
   flavors = Submodel( optional=True, valueType=Srv6SidFlavorFlags,
                       help="SID flavors" )

   def formattedString( self ):
      tacGenSid = Tac.Value( "Srv6::GenSid", Arnet.Ip6Prefix( str( self.sid ) ) )
      tacGenSid.behavior = self.behavior
      if self.flavors:
         tacFlavor = Tac.Value( "Srv6::EndBehaviorFlavors" )
         for attr in Srv6SidFlavorFlags.__attributes__:
            if getattr( self.flavors, attr ):
               setattr( tacFlavor, attr, True )
         tacGenSid.flavors = tacFlavor
      return tacGenSid.stringValue()
