# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import Tac
import DefaultSslLib
import DefaultSslProfile
import CliCommand
import CliParser
from CliPlugin import Security
from CliPlugin import SslModel
from CliPlugin import ConfigMgmtMode
from CliPlugin import SslCliLib
from CliPlugin import ConfigConvert
import ConfigMount
import LazyMount
import CliMode.Ssl as SslProfileMode
import CliToken.Reset
import os
import os.path
import re
import ShowCommand
from Toggles import MgmtSecurityToggleLib, AutoCertMgmtLibToggleLib
import CliMatcher
from TypeFuture import TacLazyType

config = None
status = None
allAutoCertProfileStatus = None
CertLocation = TacLazyType( "Mgmt::Security::Ssl::CertLocation" )
KeyLocation = TacLazyType( "Mgmt::Security::Ssl::KeyLocation" )
CertKeyPair = Tac.Type( "Mgmt::Security::Ssl::CertKeyPair" )
Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
NamedDhparams = Tac.Type( "Mgmt::Security::Ssl::NamedDhparams" )

def _listdir( path ):
   if not os.path.isdir( path ):
      return []
   return os.listdir( path )

sslMatcher = CliMatcher.KeywordMatcher(
   'ssl',
   helpdesc='Configure SSL related options' )
profileMatcher = CliMatcher.KeywordMatcher(
   'profile',
   helpdesc='Configure SSL profile' )
certificateKwMatcher = CliMatcher.KeywordMatcher(
   'certificate',
   helpdesc='Configure certificate for self authentication' )
verifyKwMatcher = CliMatcher.KeywordMatcher(
   'verify',
   helpdesc='Verify specified field in certificate' )
trustKwMatcher = CliMatcher.KeywordMatcher(
   'trust',
   helpdesc='Configure trusted certificate' )
chainKwMatcher = CliMatcher.KeywordMatcher(
   'chain',
   helpdesc='Configure chained certificate' )
revocationKwMatcher = CliMatcher.KeywordMatcher(
   'revocation',
   helpdesc='Configure checking the revocation status of presented certificates'
)
requirementKwMatcher = CliMatcher.KeywordMatcher(
   'requirement',
   helpdesc='Add a check to the certificate for validity' )
basicConstraintKwMatcher = CliMatcher.KeywordMatcher(
   'basic-constraint',
   helpdesc='Certificate basic constraint extension',
   alternates=[ 'basic-constraints' ] )
monitorKwMatcher = CliMatcher.KeywordMatcher(
   'monitor',
   helpdesc='Configure SSL monitoring options for all certificates' )
expiryKwMatcher = CliMatcher.KeywordMatcher(
   'expiry',
   helpdesc='Log SSL monitoring warnings for certificates approaching expiration' )
peerKwMatcher = CliMatcher.KeywordMatcher(
   'peer',
   helpdesc='Configure peer related options' )
monitorExpiryTimeUnits = {
   "hours": "Hour(s)",
   "days": "Day(s)"
}

# Show tokens
sslShowMatcher = CliMatcher.KeywordMatcher(
   'ssl',
   helpdesc='Show SSL status' )

profileNameMatcher = CliMatcher.DynamicNameMatcher(
                                             lambda mode: config.profileConfig,
                                             'SSL profile name' )
ocspProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: config.ocspProfileConfig,
   'OCSP profile name',
   pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
statusProfileNameMatcher = CliMatcher.DynamicNameMatcher(
                                             lambda mode: status.profileStatus,
                                             'SSL profile name' )
certificateNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: _listdir( Constants.certsDirPath() ),
                                 'Certificate name',
                                 pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
crlFileNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: _listdir( Constants.certsDirPath() ),
                                 'CRL file name',
                                 pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
crlNameMatcher = CliMatcher.DynamicNameMatcher( 
                        lambda mode: config.profileConfig[ mode.profileName ].crl,
                        'CRL name', 
                        pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
keyNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: _listdir( Constants.keysDirPath() ),
                                 'Key name',
                                 pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
autoCertificateNameMatcher = CliMatcher.DynamicNameMatcher(
                        lambda mode: _listdir( Constants.autoCertsDirPath() ),
                        'Certificate name',
                        pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
autoCertProfileNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: allAutoCertProfileStatus.profileStatus,
                                 'Auto certificate profile name' )

def _listCiphers( isTlsv1_3, ciphers ):
   argKey = 'cipherSuiteStr' if isTlsv1_3 else 'cipherListStr'
   kargs = { argKey: ciphers }
   return SslCliLib.listCipherSuiteNames( **kargs )

class ProfileConfigModeBase( BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, profileName, profileType ):
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.profileName = profileName

      if self.profileName not in config.profileConfig:
         self.profileConfig_ = config.newProfileConfig( self.profileName )
         self.profileConfig_.profileType = profileType
      else:
         self.profileConfig_ = config.profileConfig[ self.profileName ]
         if self.profileConfig_.profileType != profileType:
            return

   def enableTlsVersion( self, mask, add=False ):
      if add:
         mask |= self.profileConfig_.tlsVersion
      self.profileConfig_.tlsVersion = mask

   def disableTlsVersion( self, mask ):
      version = self.profileConfig_.tlsVersion & ~mask
      if not version:
         self.addError( "Cannot disable all TLS versions" )
         return
      self.profileConfig_.tlsVersion = version

   def setDhparam( self, dhparam ):
      self.profileConfig_.dhparam = dhparam

   def disableDhparam( self ):
      self.profileConfig_.dhparam = NamedDhparams.generated

   def enableFipsMode( self ):
      self.profileConfig_.fipsMode = True

   def disableFipsMode( self ):
      self.profileConfig_.fipsMode = False

   def _validateCiphersOrStop( self, version, ciphers ):
      def __addWarningAndStop( msg ):
         self.addWarning( msg )
         raise CliParser.AlreadyHandledError()

      isTlsv1_3 = version == 'v1.3'
      if 'Error' in _listCiphers( isTlsv1_3, ciphers ):
         if 'Error' in _listCiphers( not isTlsv1_3, ciphers ):
            # Cipher syntax error
            self.addErrorAndStop( 'Invalid cipher list.' )
         else:
            # Cipher TLS version mismatch error
            if isTlsv1_3:
               warningMsg = ( "Cannot configure TLSv1.2 and below cipher lists "
                              "in command 'cipher v1.3 CIPHERS' that is used "
                              "for TLSv1.3 only. Please configure the cipher "
                              "lists using command 'cipher v1.0 CIPHERS'." )
            else:
               warningMsg = ( "Cannot configure TLSv1.3 ciphersuites in command "
                              "'cipher-list CIPHER' or 'cipher v1.0 CIPHERS' "
                              "that is used for TLSv1.2 and below only. Please "
                              "configure the ciphersuites using command "
                              "'cipher v1.3 CIPHERS'." )
            __addWarningAndStop( warningMsg )
 
      # Check mixed cipher configuration error
      # For example: 'SHA256:TLS_AES_256_GCM_SHA384'
      # https://www.openssl.org/docs/man3.0/man1/openssl-ciphers.html
      # Both ciphersuite(TLSv1.3) and cipherlist(TLSv1.2 and below) are
      # separated by ":"
      mixCiphers = []
      for cipher in ciphers.split( ":" ):
         if ( cipher in DefaultSslLib.CipherSuitesTlsV1_3 ) is not isTlsv1_3:
            mixCiphers.append( cipher )

      if mixCiphers:
         __addWarningAndStop( "Found mixed ciphers configured for both TLSv1.3 and "
                              f"TLSv1.2 and below: '{':'.join( mixCiphers )}'. "
                              "Please separate the ciphers based on TLS versions "
                              "and use corresponding tokens in CLI command "
                              "'cipher ( v1.0 | v1.3 ) CIPHERS'." )

   def setCipherForTlsVersion( self, version, cipher ):
      setattr( self.profileConfig_,
               'cipherSuiteV1_3' if version == 'v1.3' else 'cipherSuite',
               cipher )

   def enableCipherList( self, cipherList ):
      self._validateCiphersOrStop( 'v1.0', cipherList )
      self.setCipherForTlsVersion( 'v1.0', cipherList )

   def disableCipherList( self ):
      self.setCipherForTlsVersion( 'v1.0', Constants.defaultCipherSuite() )

   def enableCipher( self, version, cipher ):
      self._validateCiphersOrStop( version, cipher )
      self.setCipherForTlsVersion( version, cipher )
   
   def disableCipher( self, version ):
      defaultCipher = ( Constants.defaultCipherSuiteV1_3() if version == 'v1.3'
                        else Constants.defaultCipherSuite() )
      self.setCipherForTlsVersion( version, defaultCipher )

   def setCertKey( self, certificate, key,
                         certLocation=CertLocation.certs,
                         keyLocation=KeyLocation.keys ):
      certKeyPair = CertKeyPair( certificate, key )
      certKeyPair.certLocation = certLocation
      certKeyPair.keyLocation = keyLocation
      self.profileConfig_.certKeyPair = certKeyPair

   def noCertKey( self ):
      self.profileConfig_.certKeyPair = CertKeyPair( "", "" )

   def addTrustedCert( self, trustedCert ):
      if not trustedCert in self.profileConfig_.trustedCert:
         self.profileConfig_.trustedCert[ trustedCert ] = 1

   def noTrustedCert( self, trustedCert ):
      if trustedCert in self.profileConfig_.trustedCert:
         del self.profileConfig_.trustedCert[ trustedCert ]

   def addChainedCert( self, chainedCert ):
      if not chainedCert in self.profileConfig_.chainedCert:
         self.profileConfig_.chainedCert[ chainedCert ] = 1

   def noChainedCert( self, chainedCert ):
      if chainedCert in self.profileConfig_.chainedCert:
         del self.profileConfig_.chainedCert[ chainedCert ]

   def addCrl( self, crl ):
      if not crl in self.profileConfig_.crl:
         self.profileConfig_.crl[ crl ] = 1

   def noCrl( self, crl ):
      if crl in self.profileConfig_.crl:
         del self.profileConfig_.crl[ crl ]

   def setOcspProfile( self, args ):
      self.profileConfig_.ocspProfileName = args[ 'OCSP_PROFILE' ]

   def noOcspProfile( self, args ):
      self.profileConfig_.ocspProfileName = ""

   def addOIDChecks( self, args ):
      updatedOIDs = {}
      oids = args.get( "OIDS" )
      if oids:
         vals = args.get( "OID_VALUES" )
         updatedOIDs.update( zip( oids, vals ) )

      oids = args.get( "OIDS_HIDDEN" )
      if oids:
         vals = args.get( "OID_VALUES_HIDDEN" )
         updatedOIDs.update( zip( oids, vals ) )

      for oid in [ "ip", "dns", "uri", "issuer", "subject" ]:
         vals = args.get( f"{oid.upper()}_VALUES" )
         if not vals:
            continue
         joinedVals = " ".join( vals )
         updatedOIDs[ oid ] = joinedVals

      for oid in self.profileConfig_.oidCheck:
         if oid not in updatedOIDs:
            del self.profileConfig_.oidCheck[ oid ]

      self.profileConfig_.oidCheck.update( updatedOIDs )

   def delOIDChecks( self, args ):
      self.profileConfig_.oidCheck.clear()

   def enableExtendedParameters( self ):
      self.profileConfig_.verifyExtendedParameters = True

   def disableExtendedParameters( self ):
      self.profileConfig_.verifyExtendedParameters = False

   def enableVerifyBasicConstraintTrust( self ):
      self.profileConfig_.verifyBasicConstraintTrust = True

   def disableVerifyBasicConstraintTrust( self ):
      self.profileConfig_.verifyBasicConstraintTrust = False

   def enableVerifyBasicConstraintChain( self ):
      self.profileConfig_.verifyBasicConstraintChain = True

   def disableVerifyBasicConstraintChain( self ):
      self.profileConfig_.verifyBasicConstraintChain = False

   def enableVerifyExpiryDateEndCert( self ):
      self.profileConfig_.verifyExpiryDateEndCert = True

   def disableVerifyExpiryDateEndCert( self ):
      self.profileConfig_.verifyExpiryDateEndCert = False

   def enableVerifyExpiryDateTrustCert( self ):
      self.profileConfig_.verifyExpiryDateTrustCert = True

   def disableVerifyExpiryDateTrustCert( self ):
      self.profileConfig_.verifyExpiryDateTrustCert = False

   def enableVerifyExpiryDateCrl( self ):
      self.profileConfig_.verifyExpiryDateCrl = True

   def disableVerifyExpiryDateCrl( self ):
      self.profileConfig_.verifyExpiryDateCrl = False

   def setVerifyChainHasRootCA( self, setting ):
      self.profileConfig_.verifyChainHasRootCA = setting

   def setCommonNameRegex( self, regex ):
      regex = regex.pop()
      for exp in regex.split():
         try:
            assert re.compile( exp )
         except Exception:  # pylint: disable-msg=W0703
            self.addError( 'Invalid regular expression:' + exp )
            continue
         self.profileConfig_.commonNameRegex.add( exp )

   def noCommonNameRegex( self ):
      self.profileConfig_.commonNameRegex.clear()

   def enableVerifyHosenameMatch( self ):
      self.profileConfig_.verifyHostnameMatch = True

   def disableVerifyHosenameMatch( self ):
      self.profileConfig_.verifyHostnameMatch = False

   def enableVerifyTrustHostnameFqdn( self ):
      self.profileConfig_.verifyTrustHostnameFqdn = True

   def disableVerifyTrustHostnameFqdn( self ):
      self.profileConfig_.verifyTrustHostnameFqdn = False

class SslProfileConfigMode( SslProfileMode.SslProfileMode, ProfileConfigModeBase ):
   name = "SSL profile configuration"

   def __init__( self, parent, session, profileName ):
      SslProfileMode.SslProfileMode.__init__( self, profileName )
      ProfileConfigModeBase.__init__( self, parent, session, profileName,
                                      "profileTypeSsl" )

def noSecurityConfig():
   for profileName in config.profileConfig:
      if profileName != DefaultSslProfile.ARISTA_PROFILE:
         del config.profileConfig[ profileName ]
   config.monitorConfig.clear()

Security.registerSecurityCleanupCallback( noSecurityConfig )

   
class GotoSslProfileModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
            'ssl': sslMatcher,
            'profile': profileMatcher,
            'PROFILE_NAME': profileNameMatcher
          }
   handler = "SslHandler.GotoSslProfileModeCmd_handler"
   noOrDefaultHandler = "SslHandler.GotoSslProfileModeCmd_noOrDefaultHandler"

Security.SecurityConfigMode.addCommandClass( GotoSslProfileModeCmd )

#--------------------------------------------------
# [no|default] ssl monitor expiry TIME_UNTIL_EXPIRY TIME_UNIT
#              log interval REPEAT_INTERVAL REPEAT_INTERVAL_TIME_UNIT
# --------------------------------------------------
class SslMonitorExpiry( CliCommand.CliCommandClass ):
   syntax = 'ssl monitor expiry TIME_UNTIL_EXPIRY '\
            'TIME_UNIT log interval REPEAT_INTERVAL REPEAT_INTERVAL_TIME_UNIT'
   noOrDefaultSyntax = 'ssl monitor expiry [ TIME_UNTIL_EXPIRY TIME_UNIT '\
                       '[ log interval REPEAT_INTERVAL REPEAT_INTERVAL_TIME_UNIT ] ]'
   data = {
      'ssl': sslMatcher,
      'monitor': monitorKwMatcher,
      'expiry': expiryKwMatcher,
      'TIME_UNTIL_EXPIRY': CliMatcher.IntegerMatcher( 1, 999,
                           helpdesc='Amount of time prior to '
                                    'certificate expiration' ),
      'TIME_UNIT': CliMatcher.EnumMatcher( monitorExpiryTimeUnits ),
      'log': CliMatcher.KeywordMatcher(
                           'log',
                           helpdesc='Log warnings to syslog' ),
      'interval': CliMatcher.KeywordMatcher(
                           'interval',
                           helpdesc='Log warnings periodically' ),
      'REPEAT_INTERVAL': CliMatcher.IntegerMatcher( 1, 999,
                           helpdesc='Amount of time for repeat interval' ),
      'REPEAT_INTERVAL_TIME_UNIT': CliMatcher.EnumMatcher( monitorExpiryTimeUnits )
   }
   handler = "SslHandler.SslMonitorExpiry_handler"
   noOrDefaultHandler = "SslHandler.SslMonitorExpiry_noOrDefaultHandler"

Security.SecurityConfigMode.addCommandClass( SslMonitorExpiry )

# --------------------------------------------------
# [no|default] certificate requirement hostname match
#--------------------------------------------------
class VerifyHostnameActionCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate requirement hostname match'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'hostname': 'Verify hostname for the certificate',
            'match': 'Hostname in certificate must match this device'
          }
   handler = "SslHandler.VerifyHostnameActionCmd_handler"
   noOrDefaultHandler = "SslHandler.VerifyHostnameActionCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( VerifyHostnameActionCmd )

#--------------------------------------------------
# [no|default] certificate requirement extended-key-usage
#--------------------------------------------------
class VerifyExtendedParametersCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate requirement extended-key-usage'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'extended-key-usage': 'Certificate extended key usage extension'
          }
   handler = "SslHandler.VerifyExtendedParametersCmd_handler"
   noOrDefaultHandler = "SslHandler.VerifyExtendedParametersCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( VerifyExtendedParametersCmd )

#--------------------------------------------------
# [no|default] [trust|chain] certificate requirement basic-constraint ca true
#--------------------------------------------------
class VerifyChainBasicConstraintTrustCmd( CliCommand.CliCommandClass ):
   syntax = ( '(trust|chain) certificate requirement '
              'basic-constraint ca true' )
   noOrDefaultSyntax = syntax
   data = {
            'trust': trustKwMatcher,
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'requirement': requirementKwMatcher,
            'basic-constraint': basicConstraintKwMatcher,
            'ca': 'Certificate authority attribute must satisfy the requirement',
            'true': 'Attribute must be set to true',
          }
   handler = "SslHandler.VerifyChainBasicConstraintTrustCmd_handler"
   noOrDefaultHandler = \
      "SslHandler.VerifyChainBasicConstraintTrustCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( VerifyChainBasicConstraintTrustCmd )

#---------------------------------------------------
# [no|default] certificate policy expiry-date ignore
#---------------------------------------------------
class IgnoreExpiryDateCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate policy expiry-date ignore'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'policy': 'Policy parameters',
            'expiry-date': 'Certificate expiry date',
            'ignore': 'Ignore a requirement',
          }
   handler = "SslHandler.IgnoreExpiryDateCmd_handler"
   noOrDefaultHandler = "SslHandler.IgnoreExpiryDateCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( IgnoreExpiryDateCmd )

#---------------------------------------------------------
# [no|default] trust certificate policy expiry-date ignore
#---------------------------------------------------------
class IgnoreExpiryDateTrustCmd( CliCommand.CliCommandClass ):
   syntax = 'trust certificate policy expiry-date ignore'
   noOrDefaultSyntax = syntax
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'policy': 'Policy parameters',
            'expiry-date': 'Certificate expiry date',
            'ignore': 'Ignore a requirement',
          }
   handler = "SslHandler.IgnoreExpiryDateTrustCmd_handler"
   noOrDefaultHandler = "SslHandler.IgnoreExpiryDateTrustCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( IgnoreExpiryDateTrustCmd )

# --------------------------------------------------
# [no|default] trust certificate requirement hostname fqdn
# --------------------------------------------------
class VerifyTrustHostnameFqdnCmd( CliCommand.CliCommandClass ):
   syntax = 'trust certificate requirement hostname fqdn'
   noOrDefaultSyntax = syntax
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'requirement': requirementKwMatcher,
            'hostname': 'Verify hostname for the certificate',
            'fqdn': 'Hostname must be FQDN without wildcard',
          }
   handler = "SslHandler.VerifyTrustHostnameFqdnCmd_handler"
   noOrDefaultHandler = "SslHandler.VerifyTrustHostnameFqdnCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( VerifyTrustHostnameFqdnCmd )

#---------------------------------------------------------
# [no|default] crl policy expiry-date ignore
#---------------------------------------------------------
class IgnoreExpiryDateCrlCmd( CliCommand.CliCommandClass ):
   syntax = 'crl policy expiry-date ignore'
   noOrDefaultSyntax = syntax
   data = {
            'crl': 'Configure CRLs',
            'policy': 'Policy parameters',
            'expiry-date': 'Certificate expiry date',
            'ignore': 'Ignore a requirement',
          }
   hidden = True
   handler = "SslHandler.IgnoreExpiryDateCrlCmd_handler"
   noOrDefaultHandler = "SslHandler.IgnoreExpiryDateCrlCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( IgnoreExpiryDateCrlCmd )

#--------------------------------------------------
# [no|default] revocation crl policy expiry-date ignore
#--------------------------------------------------
class RevocationIgnoreExpiryDateCrlCmd( CliCommand.CliCommandClass ):
   syntax = 'revocation crl policy expiry-date ignore'
   noOrDefaultSyntax = syntax
   data = {
            'revocation': revocationKwMatcher,
            'crl': 'Configure CRLs',
            'policy': 'Policy parameters',
            'expiry-date': 'Certificate expiry date',
            'ignore': 'Ignore a requirement',
          }
   handler = "SslHandler.RevocationIgnoreExpiryDateCrlCmd_handler"
   noOrDefaultHandler = \
      "SslHandler.RevocationIgnoreExpiryDateCrlCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( RevocationIgnoreExpiryDateCrlCmd )

# -------------------------------------------------------------------------------
# Register convertSslProfileIgnExpDateCrlSyntax via "config convert new-syntax"
# -------------------------------------------------------------------------------
def convertSslProfileIgnExpDateCrlSyntax( mode ):
   config.ignExpDateCrlsNewSyntax = True

ConfigConvert.registerConfigConvertCallback( convertSslProfileIgnExpDateCrlSyntax )

#-----------------------------------------------------------------------------------
# [no|default] peer certificate requirement hostname match subject-alternative-name
#              [ common-name ]
#-----------------------------------------------------------------------------------
class VerifyPeerHostnameActionCmd( CliCommand.CliCommandClass ):
   syntax = 'peer certificate requirement hostname match subject-alternative-name ' \
            '[ common-name ]'
   noOrDefaultSyntax = 'peer certificate requirement hostname match ...'
   data = {
            'peer': peerKwMatcher,
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'hostname': 'Verify hostname for the peer certificate',
            'match': 'Hostname in certificate must match client reference '
                     'identifier',
            'subject-alternative-name': 'Expected hostname should be present in '
                                        'subject alternative name extension',
            'common-name': 'Expected hostname can be present in either subject '
                            'alternative name extension or in common name in subject'
          }
   handler = "SslHandler.VerifyPeerHostnameActionCmd_handler"
   noOrDefaultHandler = "SslHandler.VerifyPeerHostnameActionCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( VerifyPeerHostnameActionCmd )

# --------------------------------------------------
# [no|default] peer certificate requirement oid <oid> <value>
# --------------------------------------------------
class VerifyCertificateOidCmd( CliCommand.CliCommandClass ):
   syntax = ( 'peer certificate requirement oid { ( subject SUBJECT_VALUES ) | '
         '( issuer ISSUER_VALUES ) | ( OIDS_HIDDEN OID_VALUES_HIDDEN ) | ( san  '
         '( otherName OIDS OID_VALUES ) | ( dns DNS_VALUES ) | ( ip IP_VALUES ) | '
         '( uri URI_VALUES ) ) }' )
   noOrDefaultSyntax = 'peer certificate requirement oid ...'
   data = {
            'peer': peerKwMatcher,
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'otherName': "x509 certificate SAN otherName",
            'oid': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'oid',
               helpdesc='Verify x509 OID value matches a user-provided value' ) ),
            'OIDS_HIDDEN': CliCommand.Node( matcher=CliMatcher.PatternMatcher(
               pattern=r'[0-9.]+', helpdesc=( 'x509 certificate SAN otherName '
               'OID' ), helpname='OID' ), hidden=True ),
            'OID_VALUES_HIDDEN': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate value', helpname='WORD' ),
            'OIDS': CliMatcher.PatternMatcher( pattern=r'[0-9.]+',
               helpdesc='x509 certificate SAN otherName OID', helpname='OID' ),
            'OID_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate value', helpname='WORD' ),
            'issuer': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'issuer',
               helpdesc='x509 certificate issuer' ), maxMatches=1 ),
            'ISSUER_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate issuer', helpname='WORD' ),
            'subject': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'subject',
               helpdesc='x509 certificate subject' ), maxMatches=1 ),
            'SUBJECT_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate issuer', helpname='WORD' ),
            'san': 'x509 certificate subject alternative name',
            'dns': 'x509 certificate SAN DNS',
            'DNS_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate SAN DNS', helpname='WORD' ),
            'ip': 'x509 certificate SAN IPs',
            'IP_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate SAN IPs', helpname="WORD" ),
            'uri': 'x509 certificate SAN URIs',
            'URI_VALUES': CliMatcher.PatternMatcher( pattern=r'.+',
               helpdesc='x509 certificate SAN URIs', helpname='WORD' ),
          }
   handler = SslProfileConfigMode.addOIDChecks
   noOrDefaultHandler = SslProfileConfigMode.delOIDChecks

if MgmtSecurityToggleLib.toggleOCCertificateOIDValidationEnabled():
   SslProfileConfigMode.addCommandClass( VerifyCertificateOidCmd )


#---------------------------------------------------
# [no|default] tls versions
# tls versions [add | remove] { 1.0 | 1.1 | 1.2 | 1.3 }...
#---------------------------------------------------
class TlsVersions( CliCommand.CliCommandClass ):
   syntax = 'tls versions [ add | remove ] { 1.0 | 1.1 | 1.2 | 1.3 }'
   noOrDefaultSyntax = 'tls versions ...'
   data = {
            'tls': 'Configure TLS settings',
            'versions': 'Configure TLS versions',
            'add': 'Add versions to the current list',
            'remove': 'Remove versions from the current list',
            '1.0': CliCommand.singleKeyword( '1.0',
                                             helpdesc='TLS version 1.0' ),
            '1.1': CliCommand.singleKeyword( '1.1',
                                             helpdesc='TLS version 1.1' ),
            '1.2': CliCommand.singleKeyword( '1.2',
                                             helpdesc='TLS version 1.2' ),
            '1.3': CliCommand.singleKeyword( '1.3',
                                             helpdesc='TLS version 1.3' ),
          }
   handler = "SslHandler.TlsVersions_handler"
   noOrDefaultHandler = "SslHandler.TlsVersions_noOrDefaultHandler"
   
SslProfileConfigMode.addCommandClass( TlsVersions )

#--------------------------------------------------
# [no|default] fips restrictions
#--------------------------------------------------
class FipRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
            'fips': 'Configure FIPS settings',
            'restrictions': 'Configure FIPS restrictions',
          }
   handler = "SslHandler.FipRestrictionsCmd_handler"
   noOrDefaultHandler = "SslHandler.FipRestrictionsCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( FipRestrictionsCmd )

#--------------------------------------------------
# [no|default] diffie-hellman parameters DHPARAMS
# --------------------------------------------------
namedDhparams = {}
for namedDhparam in NamedDhparams.attributes:
   if namedDhparam == NamedDhparams.generated:
      namedDhparams[ namedDhparam ] = "2048-bit parameters generated by the switch"
   elif namedDhparam.startswith( "ffdhe" ):
      namedDhparams[ namedDhparam ] = f"{namedDhparam} from RFC7919"
namedDhparamMatcher = CliMatcher.EnumMatcher( namedDhparams )

class DiffieHellmanParamsCmd( CliCommand.CliCommandClass ):
   syntax = 'diffie-hellman parameters DHPARAMS'
   noOrDefaultSyntax = 'diffie-hellman parameters ...'
   data = {
            'diffie-hellman': 'Configure diffie-hellman parameters',
            'parameters': 'Configure diffie-hellman parameters',
            'DHPARAMS': namedDhparamMatcher,
          }
   handler = "SslHandler.DiffieHellmanParamsCmd_handler"
   noOrDefaultHandler = "SslHandler.DiffieHellmanParamsCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( DiffieHellmanParamsCmd )

# --------------------------------------------------
# [no|default] cipher-list
#--------------------------------------------------
class CipherListCmd( CliCommand.CliCommandClass ):
   syntax = 'cipher-list CIPHERS'
   noOrDefaultSyntax = 'cipher-list ...'
   data = {
            'cipher-list': 'Configure a cipher list',
            'CIPHERS': CliMatcher.PatternMatcher( r'[\w:@!+-]+',
                                        helpname='WORD',
                                        helpdesc='Cipher list' )
          }
   # The command will be hidden and transited to 'cipher v1.0 CIPHERS'
   # in el9, to be consistent with 'cipher v1.3 CIPHERS'
   hidden = True
   handler = "SslHandler.CipherListCmd_handler"
   noOrDefaultHandler = "SslHandler.CipherListCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( CipherListCmd )

# --------------------------------------------------
# [no|default] cipher VERSION CIPHERS
# --------------------------------------------------
class CipherCmd( CliCommand.CliCommandClass ):
   syntax = 'cipher VERSION CIPHERS'
   noOrDefaultSyntax = 'cipher VERSION ...'
   data = {
            'cipher': 'Configure TLS ciphers',
            'VERSION': CliMatcher.EnumMatcher(
               { 'v1.0': 'The cipher lists for TLS version 1.0, 1.1 and 1.2',
                 'v1.3': 'The cipher suites for TLS version 1.3' } ),
            'CIPHERS': CliMatcher.PatternMatcher( r'[\w:@!+-]+',
               helpname='WORD',
               helpdesc='Cipher list or suite names separated by colon ":"' ),
          }
   handler = "SslHandler.CipherCmd_handler"
   noOrDefaultHandler = "SslHandler.CipherCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( CipherCmd )

# -------------------------------------------------------------------------------
# Register convertLegacyConfigMulticast via "config convert new-syntax"
# -------------------------------------------------------------------------------
def convertSslProfileCiphersSyntax( mode ):
   config.ciphersNewSyntax = True

ConfigConvert.registerConfigConvertCallback( convertSslProfileCiphersSyntax )

# ----------------------------------------------------------------------------------
# [no|default] certificate ( auto AUTO_CERT_PROFILE ) or  ( CERT_NAME key KEY_NAME )
# ----------------------------------------------------------------------------------
class CertificateCmd( CliCommand.CliCommandClass ):
   if AutoCertMgmtLibToggleLib.toggleAutoCertMgmtEnabled():
      syntax = 'certificate ( auto AUTO_CERT_PROFILE ) | ( CERT_NAME key KEY_NAME )'
      data = {
               'certificate': certificateKwMatcher,
               'CERT_NAME': certificateNameMatcher,
               'key': 'Configure key matching the certificate',
               'KEY_NAME': keyNameMatcher,
               'auto': 'Automatically managed certificate profiles',
               'AUTO_CERT_PROFILE': autoCertProfileNameMatcher
            }
   else:
      syntax = 'certificate CERT_NAME key KEY_NAME '
      data = {
               'certificate': certificateKwMatcher,
               'CERT_NAME': certificateNameMatcher,
               'key': 'Configure key matching the certificate',
               'KEY_NAME': keyNameMatcher
            }
   noOrDefaultSyntax = 'certificate ...'
   handler = "SslHandler.CertificateCmd_handler"
   noOrDefaultHandler = "SslHandler.CertificateCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( CertificateCmd )

# ----------------------------------------------------
# [no|default] trust certificate (system | CERT_NAME )
# ----------------------------------------------------
class TrustCertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'trust certificate ( system | CERT_NAME )'
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'system': 'Use system-supplied trust certificates',
            'CERT_NAME': certificateNameMatcher
          }
   handler = "SslHandler.TrustCertificateCmd_handler"

class NoTrustCertificateCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'trust certificate ( system | CERT_NAME )'
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'system': 'Use system-supplied trust certificates',
            'CERT_NAME': CliMatcher.DynamicNameMatcher(
               lambda mode: config.profileConfig[ mode.profileName ].trustedCert,
               'Certificate name', 
               pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
          }
   noOrDefaultHandler = "SslHandler.NoTrustCertificateCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( TrustCertificateCmd )
SslProfileConfigMode.addCommandClass( NoTrustCertificateCmd )

#--------------------------------------------------
# [no|default] chain certificate CERT_NAME
#--------------------------------------------------
class ChainCertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'chain certificate CERT_NAME'
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': certificateNameMatcher
          }
   handler = "SslHandler.ChainCertificateCmd_handler"

class NoChainCertificateCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'chain certificate CERT_NAME'
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': CliMatcher.DynamicNameMatcher(
               lambda mode: config.profileConfig[ mode.profileName ].chainedCert,
               'Certificate name', 
               pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
          }
   noOrDefaultHandler = "SslHandler.NoChainCertificateCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( ChainCertificateCmd )
SslProfileConfigMode.addCommandClass( NoChainCertificateCmd )

#--------------------------------------------------
# [no|default] chain certificate requirement include root-ca
#--------------------------------------------------
class ChainIncludeRootCACmd( CliCommand.CliCommandClass ):
   syntax = 'chain certificate requirement include root-ca'
   noOrDefaultSyntax = syntax
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'requirement': requirementKwMatcher,
            'include': 'Add a requirement to be included when validating',
            'root-ca': 'Root Certificate Authority'
          }
   handler = "SslHandler.ChainIncludeRootCACmd_handler"
   noOrDefaultHandler = "SslHandler.ChainIncludeRootCACmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( ChainIncludeRootCACmd )

#--------------------------------------------------
# [no|default] crl CRL_NAME
#--------------------------------------------------
class CrlCmd( CliCommand.CliCommandClass ):
   syntax = 'crl CRL_NAME'
   data = {
            'crl': 'Configure CRLs',
            'CRL_NAME': crlFileNameMatcher,
          }
   hidden = True
   handler = "SslHandler.CrlCmd_handler"

class NoCrlCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'crl CRL_NAME'
   data = {
            'crl': 'Configure CRLs',
            'CRL_NAME': crlNameMatcher,
          }
   hidden = True
   noOrDefaultHandler = "SslHandler.NoCrlCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( CrlCmd )
SslProfileConfigMode.addCommandClass( NoCrlCmd )

#--------------------------------------------------
# [no|default] revocation crl name CRL_NAME
#--------------------------------------------------
class RevocationCrlCmd( CliCommand.CliCommandClass ):
   syntax = 'revocation crl name CRL_NAME'
   data = {
            'revocation': revocationKwMatcher,
            'crl': 'Configure CRLs',
            'name': 'CRL name',
            'CRL_NAME': crlFileNameMatcher,
          }
   handler = "SslHandler.RevocationCrlCmd_handler"

class RevocationNoCrlCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'revocation crl name CRL_NAME'
   data = {
            'revocation': revocationKwMatcher,
            'crl': 'Configure CRLs',
            'name': 'CRL name',
            'CRL_NAME': crlNameMatcher,
          }
   noOrDefaultHandler = "SslHandler.RevocationNoCrlCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( RevocationCrlCmd )
SslProfileConfigMode.addCommandClass( RevocationNoCrlCmd )

# -------------------------------------------------------------------------------
# Register convertSslProfileCrlSyntax via "config convert new-syntax"
# -------------------------------------------------------------------------------
def convertSslProfileCrlSyntax( mode ):
   config.crlsNewSyntax = True

ConfigConvert.registerConfigConvertCallback( convertSslProfileCrlSyntax )

#------------------------------------------------------------
# [no|default] revocation ocsp profile OCSP_PROFILE
#------------------------------------------------------------
class OcspProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'revocation ocsp profile OCSP_PROFILE'
   noOrDefaultSyntax = 'revocation ocsp profile ...'
   data = {
      'revocation': revocationKwMatcher,
      'ocsp': 'Configure OCSP',
      'profile': 'OCSP profile',
      'OCSP_PROFILE': ocspProfileNameMatcher
   }
   handler = "SslHandler.OcspProfileCmd_handler"
   noOrDefaultHandler = "SslHandler.OcspProfileCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( OcspProfileCmd )

#------------------------------------------------------------
# [no|default] certificate common-name username regexp REG_EX
#------------------------------------------------------------
class CommonNameRegexCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate common-name username regexp { <REGEX> }'
   noOrDefaultSyntax = 'certificate common-name username regexp ...'
   data = {
            'certificate': certificateKwMatcher,
            'common-name': 'Configure certificate Common Name settings',
            'username': 'Username settings',
            'regexp': 'Use regular expression to extract username',
            '<REGEX>': CliMatcher.StringMatcher( helpname='REGEX',
                                                    helpdesc='Regular expression' )
          }
   handler = "SslHandler.CommonNameRegexCmd_handler"
   noOrDefaultHandler = "SslHandler.CommonNameRegexCmd_noOrDefaultHandler"

SslProfileConfigMode.addCommandClass( CommonNameRegexCmd )

#-------------------------------------------------------------------------------
# The "show management security [ssl] certificate [system|<name>|auto<name>]" command
#-------------------------------------------------------------------------------
class ShowSecuritySslCert( ShowCommand.ShowCliCommandClass ):
   if AutoCertMgmtLibToggleLib.toggleAutoCertMgmtEnabled():
      syntax = ( 'show management security ssl certificate'
                 '[ ( system | CERT_NAME | ( auto [ AUTO_CERT_NAME ] ) ) ]' )
      data = {
               'management': ConfigMgmtMode.managementShowKwMatcher,
               'security': Security.securityShowMatcher,
               'ssl': sslShowMatcher,
               'certificate': 'Show certificate',
               'system': 'Show system-supplied trust certificates',
               'CERT_NAME': certificateNameMatcher,
               'auto': 'Show auto managed certificates',
               'AUTO_CERT_NAME': autoCertificateNameMatcher
            }
   else:
      syntax = 'show management security ssl certificate [ ( system | CERT_NAME ) ]'
      data = {
               'management': ConfigMgmtMode.managementShowKwMatcher,
               'security': Security.securityShowMatcher,
               'ssl': sslShowMatcher,
               'certificate': 'Show certificate',
               'system': 'Show system-supplied trust certificates',
               'CERT_NAME': certificateNameMatcher
            }

   privileged = True
   cliModel = SslModel.Certificates
   handler = "SslHandler.ShowSecuritySslCert_handler"

BasicCli.addShowCommandClass( ShowSecuritySslCert )

#-------------------------------------------------------------------------------
# The "show management security ssl key [<name>]" command
#-------------------------------------------------------------------------------
class ShowSecuritySslKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl key [ KEY_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'key': 'Show key',
            'KEY_NAME': keyNameMatcher

          }
   privileged = True
   cliModel = SslModel.PublicKeys
   handler = "SslHandler.ShowSecuritySslKey_handler"

BasicCli.addShowCommandClass( ShowSecuritySslKey )

#-------------------------------------------------------------------------------
# The "show management security ssl crl [<name>]" command
#-------------------------------------------------------------------------------
class ShowSecuritySslCrl( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl crl [ CRL_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'crl': 'Show crl',
            'CRL_NAME': crlFileNameMatcher

          }
   privileged = True
   cliModel = SslModel.Crls
   handler = "SslHandler.ShowSecuritySslCrl_handler"

BasicCli.addShowCommandClass( ShowSecuritySslCrl )

#-------------------------------------------------------------------------------
# The "show management security ssl diffie-hellman" command
#-------------------------------------------------------------------------------
class ShowSecuritySslDiffieHellman( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl diffie-hellman [ DHPARAMS ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'diffie-hellman': 'Show diffie-hellman parameters in use',
            'DHPARAMS': namedDhparamMatcher,
          }
   privileged = True
   cliModel = SslModel.DiffieHellman
   handler = "SslHandler.ShowSecuritySslDiffieHellman_handler"

BasicCli.addShowCommandClass( ShowSecuritySslDiffieHellman )

#-------------------------------------------------------------------------------
# The "show management security ssl profile [ PROFILE_NAME ] [ detail ]" command
#-------------------------------------------------------------------------------
class ShowSecuritySslProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl profile [ PROFILE_NAME ] [ detail ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'profile': 'Show SSL profile status',
            'PROFILE_NAME': statusProfileNameMatcher,
            'detail': 'Show detailed SSL status',
          }
   privileged = True
   cliModel = SslModel.SslStatus
   handler = "SslHandler.ShowSecuritySslProfile_handler"

BasicCli.addShowCommandClass( ShowSecuritySslProfile )

# -------------------------------------------------------------------------------
# The "show management security ssl profile [ PROFILE_NAME ] cipher" command
# ------------------------------------------------------------------------------
class ShowSecuritySslProfileCiphers( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl profile [ PROFILE_NAME ] cipher'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'profile': 'Show SSL profile status',
            'PROFILE_NAME': statusProfileNameMatcher,
            'cipher': 'Show SSL profile cipher suite names',
          }
   privileged = True
   cliModel = SslModel.ProfileCiphers
   handler = "SslHandler.ShowSecuritySslProfileCiphers_handler"

BasicCli.addShowCommandClass( ShowSecuritySslProfileCiphers )

#--------------------------------------------------------------------------------
# reset ssl diffie-hellman parameters
#--------------------------------------------------------------------------------
class ResetSslDiffieHellmanParametersCmd( CliCommand.CliCommandClass ):
   syntax = 'reset ssl diffie-hellman parameters'
   data = {
      'reset': CliToken.Reset.resetMatcher,
      'ssl': sslMatcher,
      'diffie-hellman': 'Diffie-hellman parameters',
      'parameters': 'Diffie-hellman parameters',
   }
   handler = "SslHandler.ResetSslDiffieHellmanParametersCmd_handler"

BasicCli.EnableMode.addCommandClass( ResetSslDiffieHellmanParametersCmd )

def Plugin( entityManager ):
   global config, status, allAutoCertProfileStatus
   config = ConfigMount.mount( entityManager, "mgmt/security/ssl/config",
                               "Mgmt::Security::Ssl::Config", "w" )
   status = LazyMount.mount( entityManager, "mgmt/security/ssl/status",
                             "Mgmt::Security::Ssl::Status", "r" )
   allAutoCertProfileStatus = LazyMount.mount( entityManager,
                                               "autocertmgmt/cert/status",
                                               "AutoCertMgmt::AllCertProfileStatus",
                                               "r" )
