# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AsuPStore
import AsuUtil
import Cell
import CliGlobal
import FileUrl
import Plugins
import SimpleConfigFile
import SsrPStore
import Swag
import Tac
from Toggles.SwagCoreToggleLib import toggleMultiSwagBringupEnabled
import Tracing
import Url
import datetime
import json
import os
import re
import subprocess
import sys
import traceback

from StageMgr import defaultStageInstance
from collections import defaultdict
from pathlib import Path

gv = CliGlobal.CliGlobal( dict(
   ssrConfig=None, ssrStatus=None, ssrHwStatus=None,
   shutdownStageEnabled=None, shutdownStageStatus=None,
   ssrDmaCancelConfig=None, ssrDmaCancelStatus=None,
   allIntfConfigDir=None, allIntfStatusDir=None,
   fatalError=None ) )

__defaultTraceHandle__ = Tracing.Handle( 'SsrShutdownSm' )
t0 = Tracing.trace0
t9 = Tracing.trace9

bootScriptTemplate = """
#!/bin/sh
parseconfig() {{
   return
}}

ifget() {{
   return
}}

writebootconfig() {{
   return
}}

export arch=i386
export swipath={0}

. /tmp/boot0
"""

# The set of pStore plugins to run for SSR shutdown.
ssrPStorePlugins = [
   'AclAgentAsuPStore',
   'AlePhyIntfPolicyManagerStatusAsuPStore',
   'EbraAsuPStore',
   'EthIntfAsuPStore',
   'IraAsuPStore',
   'LagAsuPStore',
   'StrataFastPktAsuPStore',
   'StrataLogicalPortAsuPStore',
   'SwagAsuPStore',
]

# Fatal error reboot reasons.
fatalErrorMsgDmaCancel = 'DMA cancellation failure'
fatalErrorMsgSsrShutdown = 'SSR shutdown failure'

def log( message ):
   msg = 'SsrShutdownSm: ' + message
   t0( msg )

   ssrEventLog = Tac.Type( 'Swag::Ssr::FS' ).getSsrEventLogFilePath()
   with open( ssrEventLog, 'a' ) as f:
      f.write( datetime.datetime.now().strftime( "%Y-%m-%d %H:%M:%S " ) +
               msg + '\n' )

class DmaCancelationSm( Tac.Notifiee ):
   """
   State machine to watch for Swag::Ssr::DmaCancelStatus.complete
   to be set by the PD agent that handles DMA cancellation.

   Once Swag::Ssr::DmaCancelStatus.complete is set this SM will call the passed
   in callback function ( SsrShutdownSm.handleDmaCancelationComplete ).
   """
   notifierTypeName = 'Swag::Ssr::DmaCancelStatus'

   def __init__( self, status, callbackFnc ):
      Tac.Notifiee.__init__( self, status )
      self.status = status
      self.callbackFnc = callbackFnc
      self.handleComplete()

   @Tac.handler( 'complete' )
   def handleComplete( self ):
      t0( "DmaCancelationSm :: handleComplete" )
      if not self.status.complete:
         return
      self.callbackFnc()

class ReadyPStorePluginSm( Tac.Notifiee ):
   """
   State machine to watch for all of the expected SSR pStore plugins to be loaded.

   Once all of the plugins are loaded this SM will call the passed in
   callback function ( SsrShutdownSm.handlePStorePluginsLoaded ).
   """
   notifierTypeName = 'Swag::Ssr::ReadyPlugins'

   def __init__( self, readyPlugins, expectedPlugins, callbackFnc ):
      Tac.Notifiee.__init__( self, readyPlugins )
      self.readyPlugins = readyPlugins
      self.expectedPlugins = expectedPlugins
      self.callbackFnc = callbackFnc

   @Tac.handler( 'plugin' )
   def handlePluginLoaded( self, name ):
      t0( f"ReadyPStorePluginSm :: handlePluginLoaded : {name}" )
      if len( self.readyPlugins.plugin ) != len( self.expectedPlugins ):
         t9( f"Expected plugins: {self.expectedPlugins}" )
         t9( f"Actual plugins: {self.readyPlugins.plugin.keys()}" )
         return
      self.callbackFnc()

class InstanceStatusSm( Tac.Notifiee ):
   """
   State machine to watch for Stage::InstanceStatus.complete to be set by StageMgr.

   Once the shutdown stages are complete this SM will call the passed in callback
   function ( SsrShutdownSm.handleShutdownStagesComplete ).
   """
   notifierTypeName = 'Stage::InstanceStatus'

   def __init__( self, status, callbackFnc ):
      Tac.Notifiee.__init__( self, status )
      self.status = status
      self.callbackFnc = callbackFnc
      self.handleComplete()

   @Tac.handler( 'complete' )
   def handleComplete( self ):
      t0( "InstanceStatusSm :: handleComplete" )
      if self.status.complete:
         t9( "InstanceStatusSm :: Shutdown stages complete" )
         self.callbackFnc()

class StageStatusSm( Tac.Notifiee ):
   """
   State machine to watch for the Stage::InstanceStatus for the shutdown stages to
   exist and then create an InstanceStatusSm for it.
   """
   notifierTypeName = 'Stage::Status'

   def __init__( self, status, callbackFnc ):
      Tac.Notifiee.__init__( self, status )
      self.status = status
      self.callbackFnc = callbackFnc
      self.instStatusSm = None

      for name in self.status.instStatus.keys():
         self.handleInstStatus( name )

   @Tac.handler( 'instStatus' )
   def handleInstStatus( self, name ):
      t0( f"StageStatusSm :: handleStatus : {name}" )
      if self.instStatusSm:
         return

      if name != defaultStageInstance:
         return

      if defaultStageInstance not in self.status.instStatus:
         return

      t9( "StageStatusSm :: Creating InstanceStatusSm" )
      self.instStatusSm = InstanceStatusSm(
            self.status.instStatus[ defaultStageInstance ],
            self.callbackFnc )

class SsrShutdownSm( Tac.Notifiee ):
   """
   This is the root state machine that drives the SSR shutdown sequence.

   SsrShutdownSm will react on Swag::Ssr::Config.initiated and then start
   the shutdown sequence. The shutdown sequence involves a mixture of
   synchronous and asynchronous tasks and as such is split into 4 sections;
   each section ending on kicking off an asynchronous tasks and then next
   section starting when the task is complete.

   1. handleConfig
    - Creating the boot stript.
    - Stopping ProcMgr.
    - Setting up the watchdog.
    - Starting the shutdown stages.

   2. handleShutdownStagesComplete
    - Starting the loading the pStore plugins.

   3. handlePStorePluginsLoaded
    - Creating the pStore JSON file.
    - Killing the forwarding agents.
    - Starting DMA cancellation.

   4. handleDmaCancelationComplete
    - Shutting down the management interfaces.
    - Triggering kexec.
   """
   notifierTypeName = 'Swag::Ssr::Config'

   def __init__( self, config, status, hwStatus, shutdownStageEnabled,
                 shutdownStageStatus, dmaCancelConfig, dmaCancelStatus,
                 allIntfConfigDir, allIntfStatusDir, fatalError, em ):
      Tac.Notifiee.__init__( self, config )

      self.config = config
      self.status = status
      self.hwStatus = hwStatus
      self.shutdownStageEnabled = shutdownStageEnabled
      self.shutdownStageStatus = shutdownStageStatus
      self.dmaCancelConfig = dmaCancelConfig
      self.dmaCancelStatus = dmaCancelStatus
      self.allIntfConfigDir = allIntfConfigDir
      self.allIntfStatusDir = allIntfStatusDir
      self.fatalError = fatalError

      self.em_ = em

      self.bootSwi_ = None

      self.stageStatusSm = None

      self.readyPStorePluginSm = None
      self.pStoreFeatureEventHandlerRegistry = {}
      self.readyPStorePlugins = Tac.newInstance( 'Swag::Ssr::ReadyPlugins' )

      self.dmaCancelationSm = None

      self.handleConfig()

   def requestFatalError( self, reason ):
      log( f"Requesting a fatal error due to: {reason}" )
      log( f"Fatal error backtrace:\n{traceback.format_exc()}" )
      # Mark the SSR status as unsuccessful.
      self.status.succeeded = False
      self.status.complete = True

      # Trigger a fatal error.
      fatalErrorResetMode = Tac.Type( 'Stage::FatalError::ResetMode' )
      self.fatalError.rebootReason = reason
      self.fatalError.requestReboot = fatalErrorResetMode.resetLocal

   @Tac.handler( 'initiated' )
   def handleConfig( self ):
      if not self.config.initiated:
         return

      # Cleanup any existing SSU/SSR related files prior to starting SSR.
      self.doPreSsrCleanup()

      log( "\n******************SSR Begins********************\n" )
      try:
         # Wait for agents to be warm.
         self.wfwAgents()

         # BUG967056: This can be removed once a supervisor ping check is added.
         if toggleMultiSwagBringupEnabled():
            Tac.run( [ 'sleep', '120' ], stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
                     asRoot=True )

         # Extract kernel.
         self.extractKernel()

         # Setting up the boot script.
         procOutput = self.generateProcOutput()
         self.setupBootScript( procOutput )

         # Stop ProcMgr
         self.stopProcMgr()

         # Start watchdog.
         self.killWatchdogAndArmScdWatchdog()

         # Set reload cause.
         self.setReloadCause()

         # Trigger shutdown stages.
         self.startShutdownStages()
      except Exception: # pylint: disable=broad-except
         self.requestFatalError( fatalErrorMsgSsrShutdown )

   @property
   def bootSwi( self ):
      if not self.bootSwi_:
         log( "Finding boot SWI" )

         url = FileUrl.localBootConfig( self.em_, True )
         bootConfig = SimpleConfigFile.SimpleConfigFileDict(
               url.localFilename(), createIfMissing=True, autoSync=True )
         assert bootConfig.get( 'SWI' )
         self.bootSwi_ = Url.parseUrl( bootConfig[ 'SWI' ],
                                 Url.Context( self.em_, True ),
                                 lambda fs: fs.fsType == 'flash' )
         assert self.bootSwi_.exists()
      return self.bootSwi_.localFilename()

   def doPreSsrCleanup( self ):
      ssrEventLog = Path( Tac.Type( 'Swag::Ssr::FS' ).getSsrEventLogFilePath() )
      ssrRestoreLog = Path( Tac.Type( 'Swag::Ssr::FS' ).getSsrRestoreLogFilePath() )

      files = [
         Path( Tac.Type( 'Asu::AsuFS' ).getAsuPStoreDefaultFilePath() ),
         ssrEventLog,
         ssrRestoreLog,
      ]

      # Check to see if there are SSR files from a previous SSR.
      if ssrEventLog.exists() or ssrRestoreLog.exists():
         # There was a previous SSR, persist the files before continuing.
         dstPath = Path( Tac.Type( 'Swag::Ssr::FS' ).getSsrPersistPath() )
         if not dstPath.exists():
            dstPath.mkdir()
            assert dstPath.exists()
         for file in files:
            dstFile = Path( f'{str( dstPath )}/{file.name}' )
            if file.exists():
               file.replace( dstFile )
      else:
         # There was no previous SSR, cleanup any lingering files.
         for file in files:
            if file.exists():
               file.unlink()

   def wfwAgents( self ):
      log( "Waiting for agents to be warm before doing SSR" )
      # BUG989540
      # Replace the hardcoded agents with dynamic state.
      # This will be required for supporting single chip systems.
      agents = [ 'Strata-Linecard0-1', 'Strata-Linecard0-0' ]
      cmd = f"wfw {' '.join( agents )} -t 600 -C"
      Tac.run( cmd.split( ' ' ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
               asRoot=True )

   def extractKernel( self ):
      log( "Extracting kernel from boot SWI" )
      cmd = os.environ.get( 'UNZIP_SQUASHFS_CMD',
         f"unzip -o {self.bootSwi} linux-i386 initrd-i386 boot0 -d /tmp" )
      Tac.run( cmd.split( ' ' ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
            asRoot=True )

   def generateProcOutput( self ):
      log( "Generating /proc/cmdline" )
      regexMatch = [ re.compile( r"SWI=[\S]* " ),
                     re.compile( "kexec_jump_back_entry=0x[0-9A-Fa-f]*" ),
                     re.compile( "arista.swagMode=1" ),
                     re.compile( r"NETIP=[\S]*" ),
                     re.compile( r"NETGW=[\S]*" ),
                     re.compile( r"dmamem=[\w-]+" ),
                     re.compile( r"SKIP_CMP=[\S]*" ),
                     re.compile( r"systemd.default_standard_output=tty" ), ]

      # pylint: disable-next=consider-using-with
      proc = subprocess.Popen( [ 'cat', '/proc/cmdline' ],
            stdout=subprocess.PIPE,
            universal_newlines=True )
      procOutput = proc.communicate()[ 0 ]
      procOutput = procOutput.strip( '\n' )
      for regex in regexMatch:
         procOutput = re.sub( regex, '', procOutput )

      # Remove duplicates.
      args = procOutput.split()
      procOutDict = {}
      procOutList = []
      for arg in args:
         n, v = arg, arg
         if '=' in arg:
            n = arg.split( '=' )[ 0 ]
         if n in procOutDict:
            # Use later arg if v is different, else skip adding to list.
            if v != procOutDict[ n ]:
               procOutList.remove( procOutDict[ n ] )
               procOutList.append( v )
         else:
            procOutList.append( v )
         procOutDict[ n ] = v

      procOutList.append( f"SWI={self.bootSwi}" )
      procOutList.append( 'arista.swagMode=1' )

      return procOutList

   def setupBootScript( self, procOutList ):
      log( "Setting up the boot script" )
      bootconfigFile = os.environ.get( 'ASUBOOTCONFIG_FILE', '/etc/boot-config' )
      if not os.path.exists( bootconfigFile ):
         bootConfigCmd = f"touch {bootconfigFile}"
         Tac.run( bootConfigCmd.split( ' ' ), asRoot=True )
         Tac.run( "touch /tmp/etc-bootconfig-created".split( ' ' ), asRoot=True )

      cmdlineFile = os.environ.get( 'ASUCMDLINE_FILE', '/etc/cmdline' )
      Tac.run( [ 'touch', cmdlineFile ], asRoot=True )
      Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
      procOutStr = '\n'.join( procOutList ) + '\n'
      with open( cmdlineFile, 'w' ) as f:
         f.write( procOutStr )
      Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
      Tac.run( "touch /tmp/etc-cmdline-created".split( ' ' ), asRoot=True )

      bootscriptFile = os.environ.get( 'BOOTSCRIPT_FILE', '/tmp/boot-script' )
      with open( bootscriptFile, 'w' ) as f:
         f.write( bootScriptTemplate.format( self.bootSwi ) )
      cmd = f"chmod 777 {bootscriptFile}"
      Tac.run( cmd.split( ' ' ), asRoot=True )

   def stopProcMgr( self ):
      log( "Stopping ProcMgr" )
      procMgrCmd = os.environ.get( 'PROCMGR_CMD', '/usr/bin/ProcMgr stoppm' )
      Tac.run( procMgrCmd.split( ' ' ), asRoot=True )

   def killWatchdogAndArmScdWatchdog( self ):
      """
      This function will attempt to kill the existing watchdog daemon on the system
      and then start a new watchdog with the timeout defined in Swag::Ssr::Config.

      In the event that this function fails to kill the existing watchdog or start
      the new watchdog we will:
       - Attempt to kill the watchdog daemon again. This is incase we might have
         somehow started the new watchdog even though the command returned an error.
       - Reboot the system. This will return the system into incarnation 2.
      """
      t0( "SsrShutdownSm :: killWatchdogAndArmScdWatchdog" )
      try:
         # Kill any existing watchdog daemon.
         log( "Killing watchdog daemon" )
         watchdogCmd = os.environ.get( 'WATCHDOGKILL_CMD', 'watchdog -k' )
         Tac.run( watchdogCmd.split( ' ' ), asRoot=True )

         # Start the SCD watchdog.
         log( "Starting SCD watchdog" )
         watchdogTimeoutCmd = os.environ.get( 'WATCHDOGTIMEOUT_CMD',
               f"watchdog -o {self.config.watchdogTimeout}" )
         Tac.run( watchdogTimeoutCmd.split( ' ' ), asRoot=True )
      except Tac.SystemCommandError:
         log( "Failed to kill watchdog daemon or start the SCD watchdog" )
         # Kill watchdog again in case.
         watchdogCmd = os.environ.get( 'WATCHDOGKILL_CMD', 'watchdog -k' )
         Tac.run( watchdogCmd.split( ' ' ), asRoot=True )

         # Proceed with normal reboot back into incarnation 2 since we failed
         # this step in the shutdown process.
         shutdownCmd = os.environ.get( 'SHUTDOWN_CMD', 'shutdown -r now' )
         Tac.run( shutdownCmd.split( ' ' ), asRoot=True )

   def setReloadCause( self ):
      log( "Setting the reload cause for SSR" )
      reloadCauseConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )
      reloadCauseConstants.writeLocalReloadCause(
            reloadCauseConstants.ssrReloadDesc, "", True )

   def startShutdownStages( self ):
      log( "Starting the shutdown stages" )
      # Trigger the shutdown stages.
      self.shutdownStageEnabled.enabled[ defaultStageInstance ] = True

      # Start the SM to watch for stages to be complete.
      # Once stages are complete, handleShutdownStagesComplete will be called
      # to continue the SSR shutdown steps.
      self.stageStatusSm = StageStatusSm(
            self.shutdownStageStatus,
            self.handleShutdownStagesComplete )

   def handleShutdownStagesComplete( self ):
      log( "Shutdown stages are complete" )
      try:
         # Load the pStore plugins.
         self.loadPStorePlugins()
      except Exception: # pylint: disable=broad-except
         self.requestFatalError( fatalErrorMsgSsrShutdown )

   def loadPStorePlugins( self ):
      log( "Loading pStore plugins" )

      # Handle btest plugins.
      plugins, pluginPath = AsuUtil.getEnvAsuPStorePlugins()
      if not plugins:
         assert not pluginPath
         plugins = ssrPStorePlugins

      self.readyPStorePluginSm = ReadyPStorePluginSm(
            self.readyPStorePlugins, plugins, self.handlePStorePluginsLoaded )

      ctx = SsrPStore.SsrPStoreContext( 'SsrLoad', self.em_,
            self.pStoreFeatureEventHandlerRegistry, self.readyPStorePlugins )
      Plugins.loadPlugins( 'AsuPStorePlugin', ctx,
            plugins=plugins, pluginPath=pluginPath )

   def handlePStorePluginsLoaded( self ):
      log( "Store plugins are loaded" )
      try:
         # Save pStore information.
         self.pStore()

         # Kill forwarding agents.
         self.killFowardingAgents()

         # Initiate DMA cancellation.
         self.startDmaCancelation()
      except Exception: # pylint: disable=broad-except
         self.requestFatalError( fatalErrorMsgSsrShutdown )

   def pStore( self ):
      log( "pStoring all required state" )
      pStoredata = defaultdict( dict )
      for fname, featureEventHandler in \
          self.pStoreFeatureEventHandlerRegistry.items():
         pStoredata[ featureEventHandler.getFileName() ][ fname ] = {}
         pStoreIO = AsuPStore.FeaturePStoreIO(
               pStoredata[ featureEventHandler.getFileName() ][ fname ] )
         featureEventHandler.save( pStoreIO )

      pStoreFilePath = os.environ.get(
            Tac.Type( 'Asu::AsuFS' ).asuPStoreDirEnv,
            Tac.Type( 'Asu::AsuFS' ).asuDataDirDefault )

      os.makedirs( pStoreFilePath, exist_ok=True )
      for fileName, data in pStoredata.items():
         with open( os.path.join( pStoreFilePath, fileName ), 'w' ) as outFile:
            t0( 'Writing json file' )
            json.dump( data, outFile )

   def killFowardingAgents( self ):
      log( "Killing forwarding agents" )
      agentsToKill = []
      for agent in self.hwStatus.forwardingAgent:
         try:
            if not os.environ.get( 'PLATFORM_CMD' ):
               # Use signal 0 to check whether the agent is running
               cmd = f"killall -q --signal 0 {agent}"
               Tac.run( cmd.split(), asRoot=True )
            agentsToKill.append( agent )
         except Tac.SystemCommandError:
            # Agent is not running, skipping.
            t9( f"Agent: {agent} is not running" )
            continue

      t9( f"Agents to kill: {agentsToKill}" )
      if agentsToKill:
         platformAgentCmd = os.environ.get(
               'PLATFORM_CMD', f'killall -w -q {" ".join( agentsToKill )}' )
         Tac.run( platformAgentCmd.split( ' ' ), asRoot=True )

   def startDmaCancelation( self ):
      log( "Starting DMA cancelation" )
      self.dmaCancelConfig.initiated = True
      self.dmaCancelationSm = DmaCancelationSm(
            self.dmaCancelStatus, self.handleDmaCancelationComplete )

   def handleDmaCancelationComplete( self ):
      log( "DMA cancelation is complete" )
      if not self.dmaCancelStatus.success:
         self.requestFatalError( fatalErrorMsgDmaCancel )
         return

      try:
         # Shutdown management interfaces.
         self.shutdownMgmtIntfs()

         # kexec
         self.kexec()
      except Exception: # pylint: disable=broad-except
         self.requestFatalError( fatalErrorMsgSsrShutdown )

   def shutdownMgmtIntfs( self ):
      log( "Shutting down management interfaces" )
      for intf in self.allIntfConfigDir.intfConfig:
         if intf not in self.allIntfStatusDir.intfStatus:
            continue
         if 'Management' in intf:
            mgmtIntfId = Tac.Value( 'Arnet::MgmtIntfId', intf )
            port = mgmtIntfId.port( intf )
            module = mgmtIntfId.module( intf )
            if module:
               kernelIntfName = f"ma{module}_{port}"
            else:
               kernelIntfName = f"ma{port}"

            cmd = os.environ.get(
                  'INTF_SHUT_CMD', f"ip link set {kernelIntfName} down" )
            Tac.run( cmd.split( ' ' ), asRoot=True,
                     stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   def kexec( self ):
      runBootScriptCmd = os.environ.get(
            'BOOTSCRIPT_CMD', 'bash -c /tmp/boot-script' )
      log( "Triggering kexec into incarnation 3 using "
           f"BOOTSCRIPT_CMD={runBootScriptCmd}" )

      # Trigger the kexec.
      self.status.succeeded = not bool(
            Tac.run( runBootScriptCmd.split( ' ' ), stdin=sys.stdin, asRoot=True ) )

      # Mark the SSR status as complete.
      self.status.complete = True

def Plugin( entityManager ):
   if Swag.role() != Tac.Type( 'Swag::Role' ).candidate:
      t0( "Switch is not a SWAG candidate. Not loading the SsrShutdownSm plugin" )
      t9( f"SWAG role: {Swag.role()}" )
      return
   t9( "Loading the SsrShutdownSm plugin" )

   mg = entityManager.mountGroup()
   gv.ssrConfig = mg.mount(
         'swag/ssr/config', 'Swag::Ssr::Config', 'r' )
   gv.ssrStatus = mg.mount(
         'swag/ssr/status', 'Swag::Ssr::Status', 'w' )
   gv.ssrHwStatus = mg.mount(
         'swag/ssr/hwStatus', 'Swag::Ssr::HwStatus', 'r' )
   gv.shutdownStageEnabled = mg.mount(
         Cell.path( 'stageEnable/shutdown' ), 'Stage::EnabledConfig', 'w' )
   gv.shutdownStageStatus = mg.mount(
         Cell.path( 'stage/shutdown/status' ), 'Stage::Status', 'r' )
   gv.ssrDmaCancelConfig = mg.mount(
         'swag/ssr/dmaCancelConfig', 'Swag::Ssr::DmaCancelConfig', 'w' )
   gv.ssrDmaCancelStatus = mg.mount(
         'swag/ssr/dmaCancelStatus', 'Swag::Ssr::DmaCancelStatus', 'r' )
   gv.allIntfConfigDir = mg.mount(
         'interface/config/all', 'Interface::AllIntfConfigDir', 'r' )
   gv.allIntfStatusDir = mg.mount(
         'interface/status/all', 'Interface::AllIntfStatusDir', 'r' )
   gv.fatalError = mg.mount(
         Cell.path( 'fatalError/config/request/Swag-Ssr' ),
         'Stage::FatalError', 'fcw' )

   def _mountsComplete():
      entityManager.ssrShutdownSm = SsrShutdownSm(
            gv.ssrConfig, gv.ssrStatus, gv.ssrHwStatus,
            gv.shutdownStageEnabled, gv.shutdownStageStatus,
            gv.ssrDmaCancelConfig, gv.ssrDmaCancelStatus,
            gv.allIntfConfigDir, gv.allIntfStatusDir,
            gv.fatalError, entityManager )

   mg.close( _mountsComplete )
