#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import ConfigMount
import BasicCliModes
import CliCommand
import CliMatcher

stageGlobalConfig = None

#------------------------------------------------------------------------------------
# The "[ no|default ] stage-config <stageClass> block <stage> command,
# in "config" mode.
#
# This command is used for ptesting stage progression deadlocks.
#------------------------------------------------------------------------------------
def getStageConfigParams( args ):
   stageClass = args.get( 'STAGECLASS' )
   stage = args.get( 'STAGE' )
   return ( stageClass, stage )

def doConfigBlockStage( mode, args ):
   stageClass, stage = getStageConfigParams( args )
   # Delay time of 0 or not providing one indicates that the stage is blocked
   # indefinitely.
   delayTime = args.get( 'STAGE_DELAY_TIME', 0 )
   stageInfo = Tac.Value( 'Stage::StageInfo', stageClass, stage )
   stageGlobalConfig.doNotStartStage[ stageInfo ] = delayTime

def doUnConfigBlockStage( mode, args ):
   stageClass, stage = getStageConfigParams( args )
   stageInfo = Tac.Value( 'Stage::StageInfo', stageClass, stage )
   del stageGlobalConfig.doNotStartStage[ stageInfo ]

#--------------------------------------------------------------------------------
# [ no | default ] stage-config STAGECLASS block STAGE [ STAGE_DELAY_TIME seconds ]
#--------------------------------------------------------------------------------
class StageConfigBlockCmd( CliCommand.CliCommandClass ):
   syntax = 'stage-config STAGECLASS block STAGE [ STAGE_DELAY_TIME seconds ]'
   noOrDefaultSyntax = syntax
   data = {
      'stage-config': 'Global stage configuration',
      'STAGECLASS': CliMatcher.PatternMatcher( pattern='.+',
                    helpdesc='Stage Class to configure', helpname='WORD' ),
      'block': 'Block a stage from staring',
      'STAGE': CliMatcher.PatternMatcher( pattern='.+',
               helpdesc='Stage to configure', helpname='WORD' ),
      'STAGE_DELAY_TIME': CliMatcher.IntegerMatcher( 1, 1000,
                                                     helpdesc='Block a stage from'
                                                     ' starting for STAGE_DELAY'
                                                     ' seconds' ),
      'seconds': 'STAGE_DELAY units'
   }
   hidden = True
   handler = doConfigBlockStage
   noOrDefaultHandler = doUnConfigBlockStage

BasicCliModes.GlobalConfigMode.addCommandClass( StageConfigBlockCmd )

def Plugin( em ):
   global stageGlobalConfig

   stageGlobalConfig = ConfigMount.mount( em, "stage/stageGlobalConfig",
                                          "Stage::StageGlobalConfig", "w" )
