#!/usr/bin/env python3
# Copyright (c) 2010-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable-msg=R0201

#-------------------------------------------------------------------------------
# This module implements bridging configuration.  In particular, it provides:
# -  the "[no] storm-control broadcast level <percent>" command
# -  the "[no] storm-control multicast level <percent>" command
#
# This module does not contain VLAN-specific commands.
#------------------------------------------------------------------------------

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import LagIntfCli
from CliPlugin import SwitchIntfCli
import ConfigMount
import LazyMount
import ShowCommand
import Tac
from CliToken.Hardware import hardwareMatcherForConfig


# Module globals set up by the Plugin
config = None
statusDir = None
lagStatus = None
errdisableStatus = None
fcFeatureConfigDir = None
bridgingHwCapabilities = None
stormControlCounterTable = None
stormControlSnapshotTable = None

FapId = Tac.Type( 'FlexCounters::FapId' )
FeatureId = Tac.Type( 'FlexCounters::FeatureId' )
defaultThreshold = Tac.Value( "Bridging::StormControl::Threshold" )

ethOrLagIntfTypes = (
   EthIntfCli.EthPhyAutoIntfType,
   SwitchIntfCli.SwitchAutoIntfType,
   LagIntfCli.LagAutoIntfType,
)
supportedIntfPrefixes = tuple( x.tagLong for x in ethOrLagIntfTypes )

def sliceStatusDir():
   assert statusDir
   return statusDir[ 'slice' ]

def status():
   assert statusDir
   return statusDir[ 'all' ]

def stormControlSupported():
   if status().stormControlSupported:
      return True
   for sliceStatus in sliceStatusDir().values():
      if sliceStatus.stormControlSupported:
         return True
   return False

def errdisableSupported():
   if status().errdisableSupported:
      return True
   return any( sliceStatus.errdisableSupported for sliceStatus in
               sliceStatusDir().values() )

def stormControlSubIntfSupported():
   if status().stormControlSubIntfSupported:
      return True
   # If J2 and SubIntfToggle is enabled, return true
   for sliceStatus in sliceStatusDir().values():
      if sliceStatus.stormControlSubIntfSupported:
         return True
   return False

class StormControlModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( supportedIntfPrefixes )

IntfCli.IntfConfigMode.addModelet( StormControlModelet )

def stormControlGuard( mode, token ):
   if status().stormControlSupported:
      return None
   else:
      if stormControlSupported():
         return None
      else:
         return CliParser.guardNotThisPlatform

def policeKnownMcastKnobGuard( mode, token ):
   parentGuard = stormControlGuard( mode, token )
   if parentGuard is not None:
      return parentGuard
   
   for sliceStatus in sliceStatusDir().values():
      if not sliceStatus.policeKnownMcastKnobSupported:
         return CliParser.guardNotThisPlatform
   
   return None

def stormControlDropCountGuard( mode, token ):
   if status().dropCountSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlDefaultConfig( name ):
   intfConfig = config.intfConfig[ name ]
   return ( intfConfig.uucastLevel == defaultThreshold and
            intfConfig.uucastCpuLevel == defaultThreshold and
            intfConfig.broadcastLevel == defaultThreshold and
            intfConfig.broadcastCpuLevel == defaultThreshold and
            intfConfig.multicastLevel == defaultThreshold and
            intfConfig.multicastCpuLevel == defaultThreshold and
            intfConfig.allLevel == defaultThreshold and
            intfConfig.allCpuLevel == defaultThreshold and
            intfConfig.stormControlDropLoggingMode == 'useGlobal' )

def stormControlSubIntfLevelGuard( mode, token ):
   if mode.intf.isSubIntf():
      if not stormControlSubIntfSupported():
         return CliParser.guardNotThisPlatform
   return None

def stormControlAllLevelGuard( mode, token ):
   if status().stormControlAllLevelSupported:
      return stormControlSubIntfLevelGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def stormControlUUcastLevelGuard( mode, token ):
   if status().dropCountSupported:
      if status().stormControlUUcastLevelSupported:
         return stormControlSubIntfLevelGuard( mode, token )
      else:
         return CliParser.guardNotThisPlatform
   elif stormControlSupported():
      for sliceStatus in sliceStatusDir().values():
         if sliceStatus.stormControlUUcastLevelSupported:
            return stormControlSubIntfLevelGuard( mode, token )
      return CliParser.guardNotThisPlatform
   else:
      return CliParser.guardNotThisPlatform

def aggregateStormControlGuard( mode, token ):
   if status().aggregateStormControlSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlPpsMeteringGuard( mode, token ):
   if status().ppsMeteringSupported:
      return None
   elif stormControlSupported():
      for sliceStatus in sliceStatusDir().values():
         if sliceStatus.ppsMeteringSupported:
            return None
   return CliParser.guardNotThisPlatform

def stormControlBpsMeteringGuard( mode, token ):
   if status().bpsMeteringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlCpuGuard( mode, token ):
   if status().cpuPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def stormLevelPercentRange( mode, ctx ):
   minlevel = 0.01
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, 99.99 )

def stormPpsPercentRange( mode, context ):
   minlevel = 1
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, defaultThreshold.maxLevelPps )

def stormBpsPercentRange( mode, context ):
   minlevel = 1
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, defaultThreshold.maxLevelBps )

def stormControlZeroRateSupported():
   agentRunning = False
   if status().zeroRateSupported:
      return True
   for sliceStatus in sliceStatusDir().values():
      agentRunning = True
      if sliceStatus.zeroRateSupported:
         return True
   if agentRunning:
      return False
   else:
      return True

def stormLevelPpsRangeGuard( mode, token ):
   if stormControlZeroRateSupported():
      return None
   try:
      if int( token ) < 1:
         return CliParser.guardNotThisPlatform
   except ValueError:
      pass
   return None

def stormLevelBpsRangeGuard( mode, token ):
   if stormControlZeroRateSupported():
      return None
   if int( token ) < 1:
      return CliParser.guardNotThisPlatform
   return None

#-------------------------------------------------------------------------------
# General tokens
#-------------------------------------------------------------------------------
stormControlKw = CliCommand.guardedKeyword( 'storm-control',
      helpdesc='Configure storm-control',
      guard=stormControlGuard )
levelKwMatcher = CliMatcher.KeywordMatcher( 'level',
      helpdesc='Configure the maximum storm control level' )
levelMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicFloatMatcher(
         rangeFn=stormLevelPercentRange,
         helpdesc='Maximum bandwidth percentage allowed by storm control',
         helpname='LEVEL',
         precisionString = '%.5g' ) )
nodePps = CliCommand.guardedKeyword( 'pps',
      helpdesc='Configure the maximum storm control level in packets per second',
      guard=stormControlPpsMeteringGuard )
ppsMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicIntegerMatcher( rangeFn=stormPpsPercentRange,
         helpdesc='Maximum packet rate allowed by storm control' ),
      guard=stormLevelPpsRangeGuard )

nodeBpsRate = CliCommand.guardedKeyword( "rate",
      helpdesc="Configure the maximum storm control level in (g|m|k)bits per second",
      guard=stormControlBpsMeteringGuard )
bpsMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicIntegerMatcher( rangeFn=stormBpsPercentRange,
         helpdesc="Maximum bandwidth rate allowed by storm control" ),
      guard=stormLevelBpsRangeGuard )
bpsRateUnitMatcher = CliMatcher.EnumMatcher( {
   "bps": "Rate in bits per second",
   "kbps": "Rate in kilobits per second",
   "mbps": "Rate in megabits per second",
   "gbps": "Rate in gigabits per second",
} )
stormControlCpuKw = CliCommand.guardedKeyword( 'cpu',
      helpdesc='Whether to police CPU-bound packets',
      guard=stormControlCpuGuard )
stormControlKnownMcastKw = CliCommand.guardedKeyword( 'known-multicast',
      helpdesc='Whether to police known multicast packets',
      guard=policeKnownMcastKnobGuard )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control bum aggregate
#                                      traffic-class TRAFFIC_CLASS level pps PPS
# command in config mode
#--------------------------------------------------------------------------------

class AggregateLevelPpsCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control bum aggregate traffic-class TRAFFIC_CLASS level pps PPS'
   noOrDefaultSyntax = 'storm-control bum aggregate traffic-class TRAFFIC_CLASS ...'
   data = {
      'storm-control': stormControlKw,
      'bum': CliCommand.guardedKeyword( 'bum',
         helpdesc='Broadcast, Unknown-unicast or Multicast traffic',
         guard=aggregateStormControlGuard ),
      'aggregate': 'Aggregate storm-control',
      'traffic-class': 'Select packet Traffic-class',
      'TRAFFIC_CLASS': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Traffic-class value' ),
      'level': levelKwMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
   }
   handler = 'StormControlHandler.setAggregateLevelPps'
   noOrDefaultHandler = 'StormControlHandler.noAggregateLevelPps'

BasicCli.GlobalConfigMode.addCommandClass( AggregateLevelPpsCmd )

#--------------------------------------------------------------------------------
# [ no ] hardware storm-control known-multicast
#--------------------------------------------------------------------------------
class HardwareStormControlKnownMulticastCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware storm-control known-multicast'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : hardwareMatcherForConfig,
      'storm-control' : 'Configure storm control at '
      'forwarding chip level granularity',
      'known-multicast' : stormControlKnownMcastKw
   }
   handler = 'StormControlHandler.setPoliceKnownMulticast'
   noHandler = 'StormControlHandler.unsetPoliceKnownMulticast'
   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( HardwareStormControlKnownMulticastCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control unknown-unicast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control unknown-unicast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------
class StormControlLevelExpr( CliCommand.CliExpression ):
   expression = 'level ( LEVEL | ( pps PPS ) | ( rate RATE RATE_UNIT ) )'
   data = {
      'level': levelKwMatcher,
      'LEVEL': levelMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
      'rate': nodeBpsRate,
      'RATE': bpsMatcher,
      'RATE_UNIT': bpsRateUnitMatcher,
   }



class StormControlUnknownUnicastLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control unknown-unicast ( ( cpu [ LEVEL_EXPR ] ) | LEVEL_EXPR )'
   noOrDefaultSyntax = 'storm-control unknown-unicast [ cpu ] ...'
   data = {
      'storm-control': stormControlKw,
      'unknown-unicast': CliCommand.guardedKeyword( 'unknown-unicast',
         helpdesc='Configure storm control for unknown unicast packets',
         guard=stormControlUUcastLevelGuard ),
      'cpu': stormControlCpuKw,
      'LEVEL_EXPR' : StormControlLevelExpr,
   }
   
   handler = 'StormControlHandler.unicastHandler'
   noOrDefaultHandler = 'StormControlHandler.noUUcastLevel'

StormControlModelet.addCommandClass( StormControlUnknownUnicastLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control broadcast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control broadcast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------


class StormControlBroadcastLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control broadcast ( ( cpu [ LEVEL_EXPR ] ) | LEVEL_EXPR )'
   noOrDefaultSyntax = 'storm-control broadcast [ cpu ] ...'
   data = {
      'storm-control': stormControlKw,
      'broadcast': CliCommand.guardedKeyword( 'broadcast',
         helpdesc='Configure storm control for broadcast packets',
         guard=stormControlSubIntfLevelGuard ),
      'cpu': stormControlCpuKw,
      'LEVEL_EXPR': StormControlLevelExpr,
   }

   handler = 'StormControlHandler.broadcastHandler'
   noOrDefaultHandler = 'StormControlHandler.noBroadcastLevel'

StormControlModelet.addCommandClass( StormControlBroadcastLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control multicast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control multicast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------


class StormControlMulticastLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control multicast ( ( cpu [ LEVEL_EXPR ] ) | LEVEL_EXPR )'
   noOrDefaultSyntax = 'storm-control multicast [ cpu ] ...'
   data = {
      'storm-control': stormControlKw,
      'multicast': CliCommand.guardedKeyword( 'multicast',
         helpdesc='Configure storm control for multicast packets',
         guard=stormControlSubIntfLevelGuard ),
      'cpu': stormControlCpuKw,
      'LEVEL_EXPR': StormControlLevelExpr,
   }
   
   handler = 'StormControlHandler.multicastHandler'
   noOrDefaultHandler = 'StormControlHandler.noMulticastLevel'

StormControlModelet.addCommandClass( StormControlMulticastLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control all level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control all level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------


class StormControlAllLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control all ( ( cpu [ LEVEL_EXPR ] ) | LEVEL_EXPR )'
   noOrDefaultSyntax = 'storm-control all [ cpu ] ...'
   data = {
      'storm-control': stormControlKw,
      'all': CliCommand.guardedKeyword( 'all',
         helpdesc='Configure storm control for all packets',
         guard=stormControlAllLevelGuard ),
      'cpu': stormControlCpuKw,
      'LEVEL_EXPR': StormControlLevelExpr,
   }
  
   handler = 'StormControlHandler.allLevelHandler'
   noOrDefaultHandler = 'StormControlHandler.noAllLevel'

StormControlModelet.addCommandClass( StormControlAllLevelCmd )

#-------------------------------------------------------------------------------
# The "[no|default] logging event storm-control discards interval <val>" command,
# in "config" mode.
# This command sets storm-control drop logging interval for all storm control
# enabled interfaces
#-------------------------------------------------------------------------------
stormControlDropKw = CliCommand.guardedKeyword(
   'discards',
   'Discards due to storm control',
   stormControlDropCountGuard )

class LoggingEventStormControlInterval( IntfCli.LoggingEventGlobalCmd ):
   syntax = "logging event storm-control discards interval INTERVAL"
   noOrDefaultSyntax = "logging event storm-control discards interval ..."
   data = {
      'storm-control' : stormControlKw,
      'discards' : stormControlDropKw,
      'interval' : 'Logging interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 10, 65535,
                                            helpdesc='Logging interval in seconds' )
      }

   handler = 'StormControlHandler.dropLoggingIntervalHandler'
   noOrDefaultHandler = 'StormControlHandler.dropLoggingIntervalHandler'

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventStormControlInterval )

#-------------------------------------------------------------------------------
# the global "[no|default] logging event storm-control discards global"
# command in config mode
# This command enables storm-control drop logging on all storm control enabled
# interfaces. This can be overriden by the interface specific command
#-------------------------------------------------------------------------------
class LoggingEventStormControlGlobal( IntfCli.LoggingEventGlobalCmd ):
   syntax = "logging event storm-control discards global"
   noOrDefaultSyntax = syntax
   data = {
      'storm-control' : stormControlKw,
      'discards' : stormControlDropKw,
      'global' : 'Configure global storm control discard logging'
      }
  
   
   handler = 'StormControlHandler.enableDropLogginghandler'
   noOrDefaultHandler = 'StormControlHandler.noDropLoggingHandler'

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventStormControlGlobal )

#-------------------------------------------------------------------------------
# [no|default] logging event storm-control discards [ use-global ]
# interface specific enable disable command.
#-------------------------------------------------------------------------------
class LoggingEventStormControlIntf( IntfCli.LoggingEventIntfCmd ):
   syntax = "logging event stormcontrol discards"
   data = {
      'stormcontrol' : stormControlKw,
      'discards' : stormControlDropKw
      }
   @classmethod
   def _enableHandler( cls, mode, args ):
      config.newIntfConfig( mode.intf.name ).stormControlDropLoggingMode = 'on'

   @classmethod
   def _disableHandler( cls, mode, args ):
      config.newIntfConfig( mode.intf.name ).stormControlDropLoggingMode = 'off'

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      name = mode.intf.name
      intfConfig = config.intfConfig.get( name )
      if intfConfig:
         intfConfig.stormControlDropLoggingMode = 'useGlobal'
         if stormControlDefaultConfig( name ):
            del config.intfConfig[ name ]


 

StormControlModelet.addCommandClass( LoggingEventStormControlIntf )

#-------------------------------------------------------------------------------
# the "show storm-control [ INTFS ]" command, in enable mode
#-------------------------------------------------------------------------------


class StormControlCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show storm-control [ INTFS ]'
   data = {
      'storm-control': CliCommand.guardedKeyword( 'storm-control',
         helpdesc='Configure storm-control', guard=stormControlGuard ),
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   cliModel = 'StormControlModel.StormControlStatus'
   handler = 'StormControlHandler.showStormControl'
   privileged = True

BasicCli.addShowCommandClass( StormControlCmd )

defaultBurst = Tac.Value( "Bridging::StormControl::BurstInfo" )

def stormControlBurstGuard( mode, token ):
   if status().stormControlBurstSupported:
      return None
   else:
      for sliceStatus in sliceStatusDir().values():
         if sliceStatus.stormControlBurstSupported:
            return None
      return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# General tokens
#-------------------------------------------------------------------------------
stormControlBurstKw = CliCommand.guardedKeyword( 'storm-control',
      helpdesc='Configure storm-control',
      guard=stormControlBurstGuard )
nodeBurst = CliMatcher.KeywordMatcher( 'burst',
      helpdesc='Configure the maximum storm control burst time' )
nodeTime = CliMatcher.KeywordMatcher( 'time',
      helpdesc='Configure the maximum storm control burst time' )
burstMatcher = CliMatcher.IntegerMatcher( 10, defaultBurst.maxBurstTimeUs,
      helpdesc='Maximum burst time in milliseconds '
               'or microseconds allowed by storm control' )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control burst time TIME (milliseconds | microseconds)
#--------------------------------------------------------------------------------

class StormControlBurstCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control burst time TIME UNITS' )
   noOrDefaultSyntax = 'storm-control burst ...'
   data = {
      'storm-control': stormControlBurstKw,
      'burst' : nodeBurst,
      'time' : nodeTime,
      'TIME' : burstMatcher,
      'UNITS' : CliMatcher.EnumMatcher( {
         'microseconds':
         'Configure the maximum storm control burst in microseconds',
         'milliseconds':
         'Configure the maximum storm control burst in milliseconds',
      } ),
   }
   handler = 'StormControlHandler.setBurst'
   noOrDefaultHandler = 'StormControlHandler.noBurst'

BasicCli.GlobalConfigMode.addCommandClass( StormControlBurstCmd )

#-------------------------------------------------------------------------------
# Destroy method of StormControlIntf class is called by Intf class when an interface
# object is deleted. Intf class will create a new instance of StormControlIntf
# and call destroy method on it.
# ------------------------------------------------------------------------------

def stormControlConfig():
   return config

class StormControlIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the StormControlIntf IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      del stormControlConfig().intfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, statusDir, lagStatus, errdisableStatus
   config = ConfigMount.mount( entityManager, "bridging/stormcontrol/config",
                               "Bridging::StormControl::Config", "w" )
   statusDir = LazyMount.mount( entityManager, "bridging/stormcontrol/status",
                                "Tac::Dir", "ri" )
   lagStatus = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                "Interface::EthLagIntfStatusDir", "r" )
   errdisableStatus = LazyMount.mount( entityManager,
                                       'interface/errdisable/status',
                                       'Errdisable::Status', 'ri' )

   IntfCli.Intf.registerDependentClass( StormControlIntf )
