# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=bad-string-format-type
# pylint: disable=consider-using-f-string

from Ark import timestampToStr
import Arnet
from ArnetModel import MacAddress
import ArPyUtils
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliModel import Enum
from CliModel import Bool
from IntfModels import Interface
from StpCliUtil import * # pylint: disable=wildcard-import
from CliPlugin import IntfCli
import Ethernet
import Vlan
from TableOutput import createTable, TableFormatter, Format


def printMappedVlansIntoTable( mappedVlans, fieldWidth ):
   firstLoop = True
   printed = False
   for vlanStr in Vlan.vlanSetToCanonicalStringGen( mappedVlans, strLimit=50 ):
      if not firstLoop:
         print( ' ' * ( 80 - fieldWidth ) )
         firstLoop = False
      print( vlanStr )
      printed = True
   if not printed:
      # If we don't do any iterations, print a newline to get us to the next
      # line of the table.
      print()

FOUR_LETTER_ROLES = {
'disabled': 'Dis',
'root': 'Root',
'alternate': 'Alt',
'designated': 'Desg',
'backup': 'Back',
'master': 'Mast'
}

THREE_LETTER_STATE = {
'discarding': 'DIS',
'learning': 'LRN',
'forwarding': 'FWD'
}

#-------------------------------------------------------------------------------
# Convert a TACC EthAddr to "canonical" string format, i.e., HHHH.HHHH.HHHH
#-------------------------------------------------------------------------------
def ethAddrToString( ethAddr ):
   return Ethernet.convertMacAddrCanonicalToDisplay( str( ethAddr ) )

class VlanList( Model ):
   vlans = List( valueType=int, help="A list of vlans" )
   
class SpanningTreePortId( Model ):
   priority = Int( help='Priority of port' )
   number = Int( help='Port number' )

class SpanningTreePathCost( Model ):
   # This is a base class
   pass

class SpanningTreePathCostDefault( SpanningTreePathCost ):
   cost = Int( help='Root Path Cost' )

class SpanningTreePathCostCist( SpanningTreePathCost ):
   internalCost = Int( help='Internal Root Path Cost' )
   externalCost = Int( help='External Root Path Cost' )

class SpanningTreeBridgeDetail( Model ):
   transmitHoldCount = Int( help='Transmit Hold Count. This is the burstiness '
                                 'for transmitted BPDUs. TransmitHoldCount number '
                                 'of BPDUs are allowed to be transmitted every '
                                 '(2000/(Hello Time * TransmitHoldCount)) seconds, '
                                 'where Hello Time is in milliseconds' )

class SpanningTreeBridgeBrief( Model ):
   priority = Int( help='Priority of the bridge (0 to 61440 in ' 
                        'increments of 4096)' )
   systemIdExtension = Int( help='System Id extension' )
   macAddress = MacAddress( help='MAC address of the bridge' )

class SpanningTreeBridge( SpanningTreeBridgeBrief ):
   helloTime = Float( help='Hello time for the bridge (seconds)' )
   maxAge = Int( help='Max age for the bridge (seconds)' )
   forwardDelay = Int( help='Forward delay for the bridge (seconds)' )
   detail = Submodel( valueType=SpanningTreeBridgeDetail,
                      help='Spanning tree bridge details',
                      optional=True )

   def renderSimple( self ):
      print( '  Bridge ID  Priority    %5d  (priority %d sys-id-ext %d)' % (
            self.priority + self.systemIdExtension , 
            self.priority,
            self.systemIdExtension ) )
      print( '             Address     %-14.14s' % (
            ethAddrToString( self.macAddress ) ) )
      # pylint: disable-next=redefined-builtin
      format = '             Hello Time  %2.3f sec  Max Age %2d sec  '
      format += 'Forward Delay %2d sec'
      print( format % ( self.helloTime, self.maxAge,
                       self.forwardDelay ) )

class SpanningTreeFeatureStatus( Model ):
   default = Bool( help='The feature setting is at its default value' )
   value = Str( help='The current feature setting' )

class SpanningTreeInterfaceCounters( Model ):
   bpduSent = Int( help='BPDUs sent' )
   bpduReceived = Int( help='BPDUs received' )
   bpduTaggedError = Int( help='BPDUs tagged error' )
   bpduOtherError = Int( help='BPDUs other error' )
   bpduRateLimitCount = Int( help='BPDU rate limiter count' )

class SpanningTreeInterfaceDetail( Model ):
   # Base class
   designatedRootPriority = Int( help='Priority of designated root bridge' )
   designatedRootAddress = MacAddress( help='Address of designated root bridge' )
   designatedBridgePriority = Int( help='Priority of designated bridge' )
   designatedBridgeAddress = MacAddress( help='Address of designated bridge' )
   designatedPortPriority = Int( help='Priority of designated port' )
   designatedPortNumber = Int( help='Port number of designated port' )
   designatedPathCost = Submodel( valueType=SpanningTreePathCost,
                                  help='Designated path cost' )

class SpanningTreeInterfaceMstDetail( SpanningTreeInterfaceDetail ):
   regionalRootPriority = Int( help='Priority of regional root bridge',
                               optional=True )
   regionalRootAddress = MacAddress( help='Address of regional root bridge',
                                     optional=True )

class SpanningTreeInterfaceAllDetail( SpanningTreeInterfaceMstDetail ):
   features = Dict( keyType=str, valueType=SpanningTreeFeatureStatus,
                    help='Spanning tree feature status' )
   messageAge = Int( help='Message age for port (seconds)' )
   forwardDelay = Int( help='Forwarding delay for port (seconds)' )
   maxAge = Int( help='Max Age for port (seconds)' )
   transitionsToForwarding = Int( help='Number of timer port transitioned to '
                                        'forwarding state' )
   counters = Submodel( valueType=SpanningTreeInterfaceCounters,
                        help='Spanning Tree Interface Counters' )
   rateLimitEnabled = Bool( help='BPDU rate limiter is enabled' )
   rateLimitInterval = Float( help='BPDU rate limiter window' )
   rateLimitMaxCount = Int( help='BPDU rate limiter maximum count' )


class SpanningTreeInconsistentFeatures( Model ):
   bridgeAssurance = Bool( help='Bridge assurance is inconsistent',
                           default=False )
   loopGuard = Bool( help='Loop Guard is inconsistent', default=False )
   rootGuard = Bool( help='Root Guard is inconsistent', default=False )
   mstPvstBorder = Bool( help='MST-PVST border is inconsistent',
                         optional=True, default=False )

class SpanningTreeInterface( Model ):
   role = Enum( values=['designated', 'root', 'alternate', 'backup', 'master',
                        'disabled' ], help='Spanning tree port role' )
   state = Enum( values=['forwarding', 'learning', 'discarding'],
                 help='Spanning tree port state' )
   cost = Int( help='Spanning tree path cost' )
   priority = Int( help='Spanning tree port priority' )
   portNumber = Int( help='Port number' )
   linkType = Enum( values=[ 'p2p', 'shared' ], help='Spanning tree link type' )
   boundaryType = Enum( values=[ 'stp', 'mstpRegion', 'mstpRegionPvst', 'none' ],
                        help='Spanning tree boundary type' )
   isEdgePort = Bool( help='Interface is an edge port', optional=True )
   edgePortType = Enum( values=[ 'portfast', 'auto' ], help='Edge port type' )
   inconsistentFeatures = Submodel( valueType=SpanningTreeInconsistentFeatures,
                                    help='Spanning tree inconsistent features' )
   # Used by Mst Interface
   mappedVlans = Submodel( valueType=VlanList,
                           help='Vlans mapped to the MST instance interface',
                           optional=True )
   detail = Submodel( valueType=SpanningTreeInterfaceDetail,
                      help='Spanning tree interface details',
                      optional=True )

   def renderSimple( self, interfaceName ):
      # pylint: disable-next=redefined-builtin
      format = '%-16.16s %-10.10s %-10.10s %-9d %3d.%-4d %-30.30s'

      portStatusTypeStr = ''
      if self.linkType == 'p2p':
         portStatusTypeStr += 'P2p'
      elif self.linkType == 'shared':
         portStatusTypeStr += 'Shared'

      if self.isEdgePort:
         portStatusTypeStr += ' Edge'
         if self.edgePortType == 'portfast':
            portStatusTypeStr += ' (portfast)'

      if self.inconsistentFeatures.bridgeAssurance:
         portStatusTypeStr += ' *BA_Inc'
      if self.inconsistentFeatures.loopGuard:
         portStatusTypeStr += ' *LOOP_Inc'
      if self.inconsistentFeatures.rootGuard:
         portStatusTypeStr += ' *ROOT_Inc'
      if self.inconsistentFeatures.mstPvstBorder:
         portStatusTypeStr += ' *PVST_Inc'

      if self.boundaryType == 'stp':
         portStatusTypeStr += ' Boundary(STP)'
      elif self.boundaryType == 'mstpRegion':
         portStatusTypeStr += ' Boundary'
      elif self.boundaryType == 'mstpRegionPvst':
         portStatusTypeStr += ' Boundary(PVST)'

      print( format % ( IntfCli.Intf.getShortname( interfaceName ),
                       self.role,
                       self.state,
                       self.cost,
                       self.priority,
                       self.portNumber,
                       portStatusTypeStr ) )
   
   def renderDetail( self, interfaceName, instanceName ):
      portStatusTypeStr = ''
      if self.linkType == 'p2p':
         portStatusTypeStr += 'point-to-point'
      elif self.linkType == 'shared':
         portStatusTypeStr += 'shared'
      if self.detail.features[ 'linkType' ].default:
         portStatusTypeStr += ' by default'

      if self.boundaryType == 'stp':
         portStatusTypeStr += ', Boundary STP'
      elif self.boundaryType == 'mstpRegion':
         portStatusTypeStr += ', Boundary'
      elif self.boundaryType == 'mstpRegionPvst':
         portStatusTypeStr += ', Boundary PVST'
      else:
         portStatusTypeStr += ', Internal'

      print( ' Port %d (%s) of %s is %s %s' % ( self.portNumber,
                                               interfaceName,
                                               instanceName,
                                               self.role,
                                               self.state ) )
      print( '   Port path cost %d, Port priority %d, Port Identifier %d.%d.' % (
         self.cost, self.priority, self.priority, self.portNumber ) )
      print( '   Designated root has priority %d, address %s' % (
         self.detail.designatedRootPriority, ethAddrToString(
                                             self.detail.designatedRootAddress ) ) )
      print( '   Designated bridge has priority %d, address %s' % (
         self.detail.designatedBridgePriority, ethAddrToString(
               self.detail.designatedBridgeAddress ) ) )
      print( '   Designated port id is %d.%d, designated path cost' % (
         self.detail.designatedPortPriority,
               self.detail.designatedPortNumber ), end=' ' )
      if isinstance( self.detail.designatedPathCost, SpanningTreePathCostCist ):
         print( "%d (Ext) %d (Int)" % ( self.detail.designatedPathCost.externalCost,
                                      self.detail.designatedPathCost.internalCost ) )
      else:
         print( "%d" % self.detail.designatedPathCost.cost )
         
      # TBD. I'm not sure about this. Our dsbu switches have these as all zeros,
      # like maybe they are countdown timers?
      print( '   Timers: message age %d, forward delay %d, hold %d' % (
         self.detail.messageAge,
         self.detail.forwardDelay,
         self.detail.maxAge ) )
      print( '   Number of transitions to forwarding state: %d' % (
         self.detail.transitionsToForwarding ) )
      print( '   Link type is %s' % portStatusTypeStr )

      if self.detail.features[ 'bpduGuard' ].value == 'enabled':
         defaultString = 'by default' if \
            self.detail.features[ 'bpduGuard' ].default else ''
         print( '   Bpdu guard is enabled %s' % defaultString )

      if 'interfaceGuard' in self.detail.features:
         defaultString = 'by default' if \
                           self.detail.features[ 'interfaceGuard' ].default else ''
         print( '   %s guard is enabled %s' % ( 
               self.detail.features[ 'interfaceGuard' ].value ,defaultString ) )
      if 'networkPortBridgeAssurance' in self.detail.features:
         print( '   Portfast network port with bridge assurance %s by default'
                % ( self.detail.features[ 'networkPortBridgeAssurance' ].value ) )
      if self.inconsistentFeatures.loopGuard:
         print( '   Port Inconsistent: Loop Guard detected STP Loop' )
      if self.inconsistentFeatures.rootGuard:
         print( '   Port Inconsistent: Root Guard detected superior BPDU' )
      if self.inconsistentFeatures.bridgeAssurance:
         print( '   Port Inconsistent: Bridge Assurance not receiving BPDUs' )
      if self.inconsistentFeatures.mstPvstBorder:
         print( '   Port Inconsistent: PVST border inconsistent' )
      if self.detail.features[ 'bpduFilter' ].value == 'enabled':
         defaultString = 'by default' if \
                              self.detail.features[ 'bpduFilter' ].default else ''
         print( '   Bpdu filter is enabled %s' % defaultString )
      print( '   BPDU: sent %d, received %d, taggedErr %d, otherErr %d,'
            ' rateLimiterCount %d' %
            ( self.detail.counters.bpduSent,
              self.detail.counters.bpduReceived,
              self.detail.counters.bpduTaggedError,
              self.detail.counters.bpduOtherError,
              self.detail.counters.bpduRateLimitCount ) )
      rateLimitStatus = 'enabled' if self.detail.rateLimitEnabled else 'disabled'
      print( '   Rate-Limiter: %s, Window: %d sec, Max-BPDU: %d'
            % ( rateLimitStatus, self.detail.rateLimitInterval,
                self.detail.rateLimitMaxCount ) )
      print()


   # pylint: disable-next=arguments-differ
   def render( self, interfaceName, instanceName ):
      if not self.detail:
         self.renderSimple( interfaceName )
      else:
         self.renderDetail( interfaceName, instanceName )

   def renderInterface( self, interfaceName, instanceName ):
      # Used by show spanning-tree interface <interface> [ detail ]
      if not self.detail:
         self.renderSimple( instanceName )
      else:
         self.renderDetail( interfaceName, instanceName )

class SpanningTreeRootPort( Model ):
   portNumber = Int( help='Port Number of Root Port' )
   interface = Interface( help='Root Port Interface Name' )
   cost = Submodel( valueType=SpanningTreePathCost, help='Root Path Cost' )

class SpanningTreeInstanceDetail( Model ):
   topologyChanges = Int( help='Number of topology changes occurred' )
   lastTopologyChangeTime = Int( help='Seconds since last topology change',
                                 optional=True )
   lastTopologyChangePort = Interface( help='Port on which last topology change'
                                      ' happened', optional=True )

class SpanningTreeInstance( Model ):
   protocol = Enum( values=[ 'mstp', 'rstp', 'rapidPvst' ],
                    help='Spanning tree protocol configured' )
   bridge = Submodel( valueType=SpanningTreeBridge,
                      help='Spanning tree bridge details' )
   rootPort = Submodel( valueType=SpanningTreeRootPort, help='Root Port',
                        optional=True )
   rootBridge = Submodel( valueType=SpanningTreeBridge,
                          help='Spanning tree root bridge details', optional=True)
   interfaces = Dict( keyType=Interface,
                      valueType=SpanningTreeInterface,
                      help='Map STP interface names to their details' )
   instanceName = Str( help='Name of spanning tree instance', optional=True )
   detail = Submodel( valueType=SpanningTreeInstanceDetail,
                      help='Spanning tree instance details',
                      optional=True )

   def renderSimple( self, instanceName ):
      print( instanceName )
      stpProtocol = self.protocol
      if stpProtocol == 'rapidPvst':
         stpProtocol = 'rapid-pvst'

      print( '  Spanning tree enabled protocol %s' % stpProtocol )

      rootBridge = self.bridge
      if self.rootPort:
         rootBridge = self.rootBridge
      
      rootPriority = rootBridge.priority + rootBridge.systemIdExtension

      print( '  Root ID    Priority    %d' % rootPriority )
      print( '             Address     %s' % (
            ethAddrToString( rootBridge.macAddress ) ) )
      if not self.rootPort:
         print( '             This bridge is the root' )
      else:
         if isinstance( self.rootPort.cost, SpanningTreePathCostCist ):
            print( '             Cost        %d (Ext) %d (Int)' %
                  ( self.rootPort.cost.externalCost,
                    self.rootPort.cost.internalCost ) ) 
         else:
            print( '             Cost        %d ' % self.rootPort.cost.cost )
         print( '             Port        %d (%s)' %
                ( self.rootPort.portNumber, self.rootPort.interface.stringValue ) )
         # pylint: disable-next=redefined-builtin
         format = '             Hello Time  %2.3f sec  Max Age %2d sec  '
         format += 'Forward Delay %2d sec'
         print( format % ( rootBridge.helloTime, rootBridge.maxAge,
                          rootBridge.forwardDelay ) )

      print()

      self.bridge.renderSimple()

      print()
      print( 'Interface        Role       State      Cost      Prio.Nbr Type' )
      print( '---------------- ---------- ---------- --------- -------- '
             '--------------------' )

      for portKey in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ portKey ].render( portKey, instanceName )

      print()

   def renderDetail( self, instanceName ):
      stpProtocol = self.protocol
      if stpProtocol == 'rapidPvst':
         stpProtocol = 'rapid-pvst'

      print( ' %s is executing the %s Spanning Tree protocol' % (
            instanceName, stpProtocol ) )

      # check that we are indeed dealing with the detail bridge here
      assert self.bridge.detail
      print( '  Bridge Identifier has priority %d, sysid %d, address %s' % (
            self.bridge.priority, self.bridge.systemIdExtension,
            ethAddrToString( self.bridge.macAddress ) ) )
      print( '  Configured hello time %2.3f, max age %d, forward delay %d,'
            ' transmit hold-count %d' % ( self.bridge.helloTime,
                                          self.bridge.maxAge,
                                          self.bridge.forwardDelay,
                                          self.bridge.detail.transmitHoldCount ) )

      if not self.rootPort:
         print( '  We are the root of the spanning tree' )
      else:
         rootPriority = self.rootBridge.priority + self.rootBridge.systemIdExtension

         print( '  Current root has priority %d, address %s' % (
            rootPriority, ethAddrToString( self.rootBridge.macAddress )
            ) )
         print( '  Root port is %d (%s), cost of root path is ' %
                ( self.rootPort.portNumber,
                      self.rootPort.interface.stringValue ), end=' ' )
         if isinstance( self.rootPort.cost, SpanningTreePathCostCist ):
            print( '%d (Ext) %d (Int)' % ( self.rootPort.cost.externalCost,
                                          self.rootPort.cost.internalCost ) )
         else:
            print( '%d' % self.rootPort.cost.cost )

      if self.detail.topologyChanges:
         print( '  Number of topology changes %d last change occurred %d '
               'seconds ago' % ( self.detail.topologyChanges,
                                 self.detail.lastTopologyChangeTime ) )
         print( '          from %s'
                % self.detail.lastTopologyChangePort.stringValue )

      print()

      for portKey in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ portKey ].render( portKey, instanceName )

   def render( self, instanceName ): # pylint: disable=arguments-differ
      if self.detail is None:
         self.renderSimple( instanceName )
      else:
         self.renderDetail( instanceName )

class SpanningTreeInstances( Model ):
   spanningTreeInstances = Dict( keyType=str,
                                 valueType=SpanningTreeInstance,
                                 help='Map STP instance names to their details' )
   _message = Str( help='Message to show to the user', optional=True )

   def render( self ):
      if self._message:
         print( self._message )
         return

      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeInstances ):
         self.spanningTreeInstances[ instanceName ].render( instanceName )
         if self.spanningTreeInstances[ instanceName ].detail:
            print()

class SpanningTreeBlockedPortsList( Model ):
   spanningTreeBlockedPorts = List( valueType=Interface,
                                    help='List of STP blocked ports' )

class SpanningTreeBlockedPorts( Model ):
   spanningTreeInstances = Dict( keyType=str,
                                 valueType=SpanningTreeBlockedPortsList,
                                 help='Map STP instance names to blocked ports' )
   
   def render( self ):
      print( 'Name       Blocked Interfaces List' )
      print( '---------- '
             '----------------------------------'
             '-----------------------------------' )

      blockedPortsTotal = 0
      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeInstances ):
         buf = '%-10.10s ' % instanceName
         blockedPortsMsti = 0

         intfs = self.spanningTreeInstances[ instanceName ].spanningTreeBlockedPorts
         for intfName in Arnet.sortIntf( intfs ):

            shortname = IntfCli.Intf.getShortname( intfName )
            if blockedPortsMsti == 0:
               buf += '%s' % shortname
            else:
               buf += ', %s' % shortname

            blockedPortsMsti += 1
            if len( buf ) > 80:
               # Line is getting too long. Print out, and indent into next line.
               print( buf )
               buf = '%-10.10s ' % ' '
               blockedPortsMsti = 0
               print( buf )

            blockedPortsTotal += 1
         print( buf )

      print()
      print( 'Number of blocked ports (segments) in the system : %d' % (
         blockedPortsTotal ) )

class PortChannelPortIdAlloc( Model ):
   reservedRangeMin = Int( help='Minimum value for reserved range' )
   reservedRangeMax = Int( help='Maximum value for reserved range' )

class SpanningTreeBridgesDetail( Model ):
   restartable = Bool( help='STP is restartable' )
   mstPvstBorderEffective = Bool( help='MST-PVST interoperation is active',
                                  optional=True )
   peerHeartbeatWhileEnabled = Bool( help='Peer heartbeat check is enabled',
                                     optional=True )
   overriddenBy = Str( help='By whom Stp is overridden', optional=True )
   heartbeatTimeout = Float( help='Time (in seconds) of no received STP '
         'heartbeat before a timeout is declared', optional=True )
   localHeartbeatTimeoutsSinceReboot = Int( help='Number of local STP heartbeat '
         'timeouts that occured since the last reload', optional=True )
   peerHeartbeatTimeoutsSinceReboot = Int( help='Number of peer STP heartbeat '
         'timeouts that occured since the last reload', optional=True )
   lastLocalHeartbeatTimeout = Float( help='Time of the last local STP heatbeat '
         'timeout', optional=True )
   lastPeerHeartbeatTimeout = Float( help='Time of the last peer STP heartbeat '
         'timeout', optional=True )
   superRoot = Bool( help='This switch is network super root', optional=True )
   portChannelPortIdAllocationRange = Submodel( valueType=PortChannelPortIdAlloc,
         help='Allocation range for port-channel port ID' )
   preserveOriginal2p5GAnd5GCost = Bool( help='Original 2.5G/5G cost (2000) is '
         'preserved', optional=True )
   mismatchedBpduMode = Enum( values=[ 'accept', 'drop' ],
                              help='VLAN-mismatched BPDU handling mode in PVST' )

class SpanningTreeBridges( Model ):
   spanningTreeBridges = Dict( keyType=str,
                               valueType=SpanningTreeBridge,
                               help='Map STP bridge names to their details' )
   detail = Submodel( valueType=SpanningTreeBridgesDetail,
                      help='Spanning tree bridge details',
                      optional=True )

   def renderSimple( self ):
      print( '                            Bridge ID                    ',
             'Hello  Max  Fwd' )
      print( 'Instance             Priority             MAC addr       ',
             'Time   Age  Dly' )
      print( '----------   ----------------------------------------    ',
             '-----  ---  ---' )
      # pylint: disable-next=redefined-builtin
      format = '%-10.10s   %5d(%5d, sys-id %-4d) %-14.14s      %2d    %2d   %2d'
      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeBridges ):
         bridge = self.spanningTreeBridges[ instanceName ]
         print( format % ( instanceName, bridge.priority + bridge.systemIdExtension,
                          bridge.priority, bridge.systemIdExtension,
                          ethAddrToString( bridge.macAddress ),
                          bridge.helloTime * 1000, bridge.maxAge,
                          bridge.forwardDelay ) )

   def renderDetail( self ):
      print( 'Stp Detailed Status:' )
      t = TableFormatter()
      d = self.detail
      t.newRow( 'Stp agent restartable', ': ',
               'true' if d.restartable else 'false' )
      t.newRow( '2.5G/5G port path cost', ': ',
            '2000/2000' if d.preserveOriginal2p5GAnd5GCost else '8000/4000' )
      defaultMismatchedBpduMode = \
         Tac.Type( 'Stp::ConfigBase' ).defaultMismatchedBpduMode
      if d.mismatchedBpduMode != defaultMismatchedBpduMode:
         t.newRow( 'VLAN-mismatched untagged PVST BPDU', ': ',
                     d.mismatchedBpduMode )
      if d.superRoot:
         t.newRow( 'Super root', ': ',
                  'true' if d.superRoot else 'false' )
      t.newRow( 'Port-channel port ID allocation range', ': ', 'unconfigured' \
            if not d.portChannelPortIdAllocationRange.reservedRangeMin and \
               not d.portChannelPortIdAllocationRange.reservedRangeMax else \
            '%d-%d' % ( d.portChannelPortIdAllocationRange.reservedRangeMin,
                        d.portChannelPortIdAllocationRange.reservedRangeMax ) )
      t.newRow( 'MST-PVST interoperation', ': ',
                'enabled' if d.mstPvstBorderEffective else 'disabled' )
      if not d.overriddenBy:
         t.newRow( 'Stp heartbeat timeout', ': ', d.heartbeatTimeout )
         t.newRow( 'Last local heartbeat timeout', ': ',
                    timestampToStr( d.lastLocalHeartbeatTimeout ) )
         t.newRow( 'Local heartbeat timeout since reboot', ': ',
                   d.localHeartbeatTimeoutsSinceReboot )
         if d.peerHeartbeatWhileEnabled:
            t.newRow( 'Last peer heartbeat timeout', ': ',
                      timestampToStr( d.lastPeerHeartbeatTimeout ) )
            t.newRow( 'Peer heartbeat timeout since reboot', ': ',
                      d.peerHeartbeatTimeoutsSinceReboot )
      t.formatColumns( Format( justify='left' ), Format(), Format() )
      print( t.output() )

      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeBridges ):
         print( instanceName )
         self.spanningTreeBridges[ instanceName ].renderSimple()

   def render( self ):
      if not self.spanningTreeBridges:
         return

      if self.detail:
         self.renderDetail()
      else:
         self.renderSimple()

class SpanningTreeBridgeAssuranceValue( Model ):
   bridgeAssuranceValue = Enum( values=[ 'consistent',
                                      'inconsistent', 'notNetworkPort' ],
                                      help='Spanning Tree Bridge Assurance values' )

class SpanningTreeBridgeAssuranceInterfaces( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=SpanningTreeBridgeAssuranceValue,
                      help='Map STP interfaces to their Bridge Assurance values' )

class SpanningTreeBridgeAssurance( Model ):
   spanningTreeInstances = Dict( keyType=str,
                                 valueType=SpanningTreeBridgeAssuranceInterfaces,
                                 help='Map STP instances to bridge assurance '
                                      'interfaces' )
   bridgeAssuranceEnabled = Bool( 
                  help='Bridge Assurance is enabled in the configuration' )

   def render( self ):
      if not self.bridgeAssuranceEnabled:
         print( 'Bridge assurance is globally disabled' )
         return

      title = 'Name                 Bridge Assurance Status'
      print( title )
      print( '-' * len( title ) )

      baInconsistentCount = 0

      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeInstances ):

         intfs = self.spanningTreeInstances[ instanceName ].interfaces
         for intfName in Arnet.sortIntf( intfs ):

            baValue = self.spanningTreeInstances[ instanceName ].interfaces[
                  intfName ].bridgeAssuranceValue
            shortname = IntfCli.Intf.getShortname( intfName )
            buf = '%-10.10s ' % instanceName
            buf += '%-10.10s       ' % shortname
            if baValue == 'notNetworkPort':
               buf += 'Not a network port'
            else:
               buf += baValue
            print( buf )
            if baValue == 'inconsistent':
               baInconsistentCount += 1

      print( '' )
      print( 'Number of bridge assurance inconsistent ports in the system : %d' % ( 
             baInconsistentCount ) )

class SpanningTreeCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=SpanningTreeInterfaceCounters,
                      help='Map interfaces to STP counters' )
   _message = Str( help='Message to show to the user', optional=True )

   def render( self ):
      if self._message:
         print( self._message )
         return

      t = createTable( ( "Port", "Sent", "Received",
          "Tagged Error", "Other Error", "RateLimiterCount" ) )
      for key in Arnet.sortIntf( self.interfaces ):
         t.newRow( key, self.interfaces[ key ].bpduSent, 
                        self.interfaces[ key ].bpduReceived,
                        self.interfaces[ key ].bpduTaggedError,
                        self.interfaces[ key ].bpduOtherError,
                        self.interfaces[ key ].bpduRateLimitCount )
      print( t.output() )

class SpanningTreeInterfaces( Model ):
   interface = Interface( help='Name of interface' )
   spanningTreeInstances = Dict( keyType=str,
                                 valueType=SpanningTreeInterface,
                                 help='Map STP instances to interfaces' )
   _interfaceExists = Bool( help='Interface exists in spanning tree',
                            default=True )
   _detail = Bool( help='Detail information' )

   def render( self ):
      if not self._interfaceExists:
         print( 'no spanning tree info available for', self.interface.stringValue )
         return
      
      instances = self.spanningTreeInstances
      if not self._detail:

         print( 'Instance         Role       State      Cost      Prio.Nbr Type' )
         print( '---------------- ---------- ---------- --------- -------- '
                '--------------------' )

      for instanceName in ArPyUtils.naturalsorted( instances ):
         instances[ instanceName ].renderInterface(
                                    self.interface.stringValue, instanceName )

class SpanningTreeMstConfiguration( Model ):
   mstRegionName = Str( help='MST region identifier' )
   mstConfigurationRevision = Int( help='MST configuration revision' )
   configuredMstInstances = Int( help='Number of MST instances configured' )
   mstInstances = Dict( keyType=int, valueType=VlanList,
                        help='Map MST instance numbers to corresponding Vlan list' )
   mstConfigurationDigest = Str( help='MST configuration digest', optional=True )
   _message = Str( help='Header for the output', optional=True )

   def render( self ):
      if self._message:
         print( self._message )
      print( "Name      [%s]" % self.mstRegionName )
      print( "Revision  %-4d Instances configured %d" %
             ( self.mstConfigurationRevision, self.configuredMstInstances ) )

      if self.mstConfigurationDigest:
         print( "Digest        ", self.mstConfigurationDigest )
      else:
         print( "\nInstance  Vlans mapped" )
         print( "-" * 8, "-" * 71 )
         for instId in sorted( self.mstInstances.keys() ):
            instIdPrinted = False
            for vlanSubStr in Vlan.vlanSetToCanonicalStringGen( 
                                             self.mstInstances[ instId ].vlans, 70 ):
               instStr = " " * 9 if instIdPrinted else "%-9d" % instId
               print( instStr, vlanSubStr )
               instIdPrinted = True
         print( "-" * 80 )


class SpanningTreeMstInstance( Model ):
   mappedVlans = Submodel( valueType=VlanList,
                           help='Vlans mapped to the MST instance' )
   bridge = Submodel( valueType=SpanningTreeBridgeBrief,
                      help='Spanning tree bridge details' )
   rootBridge = Submodel( valueType=SpanningTreeBridgeBrief,
                          help='Spanning tree root bridge details', optional=True)
   regionalRootBridge = Submodel( valueType=SpanningTreeBridgeBrief,
                          help='Spanning tree regional root bridge details', 
                          optional=True)
   interfaces = Dict( keyType=Interface,
                      valueType=SpanningTreeInterface,
                      help='Map interface names to details' )

   def render( self, instanceName, detail ): # pylint: disable=arguments-differ
      print( "##### %-7s" % instanceName, "vlans mapped:   ", end=' ' )

      printMappedVlansIntoTable( self.mappedVlans.vlans, 50 )

      print( 'Bridge        address', ethAddrToString( self.bridge.macAddress ),
            " priority      %d (%d sysid %d)" % ( self.bridge.priority +
                                                  self.bridge.systemIdExtension, 
                                                  self.bridge.priority,
                                                  self.bridge.systemIdExtension ) )

      print( "%-13s" % "Root", end=' ' )
      if self.rootBridge.macAddress == self.bridge.macAddress:
         print( 'this switch for', end=' ' )
         if instanceName == 'MST0':
            # For some reason, here the industry standard switches to CIST instead
            # of MST0.
            print( "the CIST" )
         else:
            print( instanceName )
      else:   
         print( 'address', ethAddrToString( self.rootBridge.macAddress ),
               " priority      %d (%d sysid %d)" % ( 
                                       self.rootBridge.priority +
                                       self.rootBridge.systemIdExtension,
                                       self.rootBridge.priority,
                                       self.rootBridge.systemIdExtension ) )

      if self.regionalRootBridge:
         print( "%-13s" % "Regional Root", end=' ' )
         if self.rootBridge.macAddress == self.bridge.macAddress:
            print( 'this switch for', end=' ' )
            if instanceName == 'MST0':
               # For some reason, here the industry standard switches to CIST instead
               # of MST0.
               print( "the CIST" )
            else:
               print( instanceName )
         else:
            print( 'address', ethAddrToString( self.regionalRootBridge.macAddress ),
                  " priority      %d (%d sysid %d)" % (
                                       self.regionalRootBridge.priority +
                                       self.regionalRootBridge.systemIdExtension,
                                       self.regionalRootBridge.priority,
                                       self.regionalRootBridge.systemIdExtension ) )
      print( '' )

      if not detail:

         print( 'Interface        Role       State      Cost      Prio.Nbr Type' )
         print( '---------------- ---------- ---------- --------- -------- '
                '--------------------' )

         for key in Arnet.sortIntf( self.interfaces ):
            self.interfaces[ key ].renderSimple( key )

      else:
         for key in Arnet.sortIntf( self.interfaces ):
            intf = self.interfaces[ key ]
            rootPathCost = intf.detail.designatedPathCost.externalCost if \
               self.regionalRootBridge else intf.detail.designatedPathCost.cost

            print( "%s of %s is %s %s" % ( key, instanceName,
                                         intf.role, intf.state ) )
            print( "Port info             port id %14s  priority %6d  cost %11d" %
                  ( "%d.%d" % ( intf.priority, intf.portNumber ),
                   intf.priority, intf.cost ) )
            print( "Designated root       address %s  priority %6d  cost %11d" %
                  ( ethAddrToString( intf.detail.designatedRootAddress ),
                        intf.detail.designatedRootPriority, rootPathCost ) )
            if self.regionalRootBridge:
               print( "Design. regional root address %s  priority %6d  cost %11d" %
                     ( ethAddrToString( intf.detail.regionalRootAddress ),
                      intf.detail.regionalRootPriority,
                      intf.detail.designatedPathCost.internalCost ) )
            print( "Designated bridge     address %s  priority %6d  port id %8s" %
                  ( ethAddrToString( intf.detail.designatedBridgeAddress ),
                        intf.detail.designatedBridgePriority,
                   "%d.%d" % ( intf.detail.designatedPortPriority,
                      intf.detail.designatedPortNumber ) ) )
            print( '' )
      print( '' )

class SpanningTreeMstInstances( Model ):
   spanningTreeMstInstances = Dict( keyType=str, valueType=SpanningTreeMstInstance,
                                    help='Map MST instance names to details' )
   _detail = Bool( help='Whether the render function uses detailed output',
                   optional=True )
   # Need this because of the difference between when MST is not running and
   # when there are no interfaces associated with a MST instance
   _message = Str( help='Message to show to the user', optional=True )

   def render( self ):
      if self._message:
         print( self._message )
         return

      for instanceName in ArPyUtils.naturalsorted( self.spanningTreeMstInstances ):
         self.spanningTreeMstInstances[ instanceName ].render( instanceName,
                                                               self._detail )

class SpanningTreeMstInterface( Model ):
   interfaceName = Interface( help='Interface Name' )
   isEdgePort = Bool( help='Interface is an edge port' )
   features = Dict( keyType=str, valueType=SpanningTreeFeatureStatus,
                    help='Spanning tree feature status' )
   counters = Submodel( valueType=SpanningTreeInterfaceCounters,
                        help='STP counters' )
   mstInstances = Dict( keyType=str, valueType=SpanningTreeInterface,
                         help='Map MST instance names to interface details' )

   def render( self ):
      # Make sure we have at least one key
      assert self.mstInstances
      mstInstanceName = ArPyUtils.naturalsorted( self.mstInstances )[ 0 ]
      intf = self.mstInstances[ mstInstanceName ]
      if intf.detail is None:
         print( "%s of %s is %s %s" % ( self.interfaceName.stringValue,
                                        mstInstanceName,
                                        intf.role,
                                        intf.state ) )

      edgePort = "yes" if self.isEdgePort else "no"
      bpduguard = self.features[ 'bpduGuard' ].value
      print( "Edge port: %-3s                             bpdu guard : %s" %
             ( edgePort, bpduguard ) )

      if self.features[ 'linkType' ].value == 'p2p':
         linkType = "point-to-point"
      else:
         linkType = "shared"
      bpduFilter = "yes" if self.features[ 'bpduFilter'].value == \
                                                         'enabled' else "no"
      print( "Link type: %14s                  bpdu filter: %s" %
             ( linkType, bpduFilter ) )

      boundary = None
      if intf.boundaryType == 'stp':
         boundary = 'Boundary STP'
      elif intf.boundaryType == 'mstpRegion':
         boundary = 'Boundary'
      elif intf.boundaryType == 'mstpRegionPvst':
         boundary = 'Boundary PVST'
      else:
         boundary = 'Internal'

      print( "Boundary : %s" % boundary )

      print( 'Bpdus sent %d, received %d, taggedErr %d, otherErr %d sinceTimer %d' %
             ( self.counters.bpduSent,
               self.counters.bpduReceived,
               self.counters.bpduTaggedError,
               self.counters.bpduOtherError,
               self.counters.bpduRateLimitCount ) )
      print( '' )

      if self.mstInstances[ next( iter( self.mstInstances ) ) ].detail is None:
         print( 'Instance Role Sts Cost      Prio.Nbr Vlans mapped' )
         print( '-------- ---- --- --------- '
                '-------- -------------------------------' )

         for key in ArPyUtils.naturalsorted( self.mstInstances ):
            print( "%-8s %4s %3s %-9d %-8s" %
                  ( key[ 3: ],
                    FOUR_LETTER_ROLES.get( self.mstInstances[ key ].role, 'Unkn' ),
                    THREE_LETTER_STATE.get( self.mstInstances[ key ].state, 'DIS' ),
                    self.mstInstances[ key ].cost,
                    "%d.%d" % ( self.mstInstances[ key ].priority,
                               self.mstInstances[ key ].portNumber ) ), end=' ' )
            printMappedVlansIntoTable( self.mstInstances[ key ].mappedVlans.vlans,
                                       43 )

      else:
         for key in ArPyUtils.naturalsorted( self.mstInstances ):
            intf = self.mstInstances[ key ]
            rootPathCost = intf.detail.designatedPathCost.externalCost if \
               key == 'MST0' else intf.detail.designatedPathCost.cost

            print( "%s of %s is %s %s" % ( self.interfaceName.stringValue, key,
                                         intf.role, intf.state ) )
            prefix = "Vlans mapped to %s" % key
            print( prefix, end=' ' )
            printMappedVlansIntoTable( intf.mappedVlans.vlans,
                                       80 - ( len(prefix) + 1 ) )
            print( "Port info             port id %14s  priority %6d  cost %11d" %
                  ( "%d.%d" % ( intf.priority, intf.portNumber ),
                   intf.priority, intf.cost ) )
            print( "Designated root       address %s  priority %6d  cost %11d" %
                  ( ethAddrToString( intf.detail.designatedRootAddress ),
                        intf.detail.designatedRootPriority, rootPathCost ) )
            if key == 'MST0':
               print( "Design. regional root address %s  priority %6d  cost %11d" %
                     ( ethAddrToString( intf.detail.regionalRootAddress ),
                      intf.detail.regionalRootPriority,
                      intf.detail.designatedPathCost.internalCost ) )
            print( "Designated bridge     address %s  priority %6d  port id %8s" %
                  ( ethAddrToString( intf.detail.designatedBridgeAddress ),
                        intf.detail.designatedBridgePriority,
                   "%d.%d" % ( intf.detail.designatedPortPriority,
                      intf.detail.designatedPortNumber ) ) )
            print()

class SpanningTreeMstInterfaces( Model ):
   spanningTreeMstInterface = Submodel( valueType=SpanningTreeMstInterface,
                                        help='Spanning Tree MST interface details',
                                        optional=True )
   _message = Str( help='Message to show to the user', optional=True )

   def render( self ):
      if self._message:
         print( self._message )
         return

      self.spanningTreeMstInterface.render()


class SpanningTreeRoot( Model ):
   rootPort = Submodel( valueType=SpanningTreeRootPort, help='Root Port',
                        optional=True )
   rootBridge = Submodel( valueType=SpanningTreeBridge,
                          help='Spanning tree root bridge details' )

class SpanningTreeRoots( Model ):
   instances = Dict( keyType=str, valueType=SpanningTreeRoot,
                     help='Map STP instance names to their root bridge' )
   _detail = Bool( help='Whether detailed information is to be rendered' )

   def render( self ):
      if not self._detail:
         print( '                   Root ID          Root    Hello  Max  Fwd' )
         print( 'Instance     Priority    MAC addr   Cost    Time   Age  Dly  '
                'Root Port' )
         # pylint: disable-next=redefined-builtin
         format = '----------   -------------------- --------- '
         format += '-----  ---  ---  ------------'
         print( format )
         format = '%-10.10s   %5d %-14.14s %9d   %2d    %2d  %2d   %-12.12s'

         for key in ArPyUtils.naturalsorted( self.instances ):
            rootPathCost = 0
            if self.instances[ key ].rootPort:
               if isinstance( self.instances[ key ].rootPort.cost,
                     SpanningTreePathCostCist ):
                  rootPathCost = self.instances[ key ].rootPort.cost.externalCost
               else:
                  rootPathCost = self.instances[ key ].rootPort.cost.cost

            rootPortName = self.instances[ key ].rootPort.interface.shortName if \
               self.instances[ key ].rootPort else 'None'
            rootPriority = self.instances[ key ].rootBridge.priority + \
                           self.instances[ key ].rootBridge.systemIdExtension
            print( format % ( key,
                             rootPriority,
                             ethAddrToString( 
                                self.instances[ key ].rootBridge.macAddress ),
                             rootPathCost,
                             self.instances[ key ].rootBridge.helloTime,
                             self.instances[ key ].rootBridge.maxAge,
                             self.instances[ key ].rootBridge.forwardDelay,
                             rootPortName ) )
      else:
         for key in ArPyUtils.naturalsorted( self.instances ):
            print( key )
            rootBridge = self.instances[ key ].rootBridge
            rootPort = self.instances[ key ].rootPort

            rootPriority = rootBridge.priority + rootBridge.systemIdExtension

            print( '  Root ID    Priority    %d' % rootPriority )
            print( '             Address     %s' % (
                  ethAddrToString( rootBridge.macAddress ) ) )
            if not self.instances[ key ].rootPort:
               print( '             This bridge is the root' )
            else:
               if isinstance( rootPort.cost, SpanningTreePathCostCist ):
                  print( '             Cost        %d (Ext) %d (Int)' %
                        ( rootPort.cost.externalCost,
                          rootPort.cost.internalCost ) ) 
               else:
                  print( '             Cost        %d ' % rootPort.cost.cost )
               print( '             Port        %d (%s)' %
                      ( rootPort.portNumber, rootPort.interface.stringValue ) )
               format = '             Hello Time  %2.3f sec  Max Age %2d sec  '
               format += 'Forward Delay %2d sec'
               print( format % ( rootBridge.helloTime, rootBridge.maxAge,
                                rootBridge.forwardDelay ) )

class SpanningTreeVlanInstance( Model ):
   spanningTreeVlanInstance = Submodel( valueType=SpanningTreeInstance,
                                        optional=True,
                                        help='Spanning tree vlan instance details. '
                                             'This field is None when a '
                                             'corresponding vlan exists, but '
                                             'has no spanning tree instance '
                                             'associated with it' )
   _message = Str( help='Message to show to the user', optional=True )

   def render( self, vlanId ): # pylint: disable=arguments-differ
      if self._message:
         print( self._message )
         return

      instance = self.spanningTreeVlanInstance
      assert instance # Should be present if _message is empty
      print( "Spanning tree instance for vlan %s" % vlanId )
      instance.render( instance.instanceName ) 
      if instance.detail:
         print()

class SpanningTreeVlanInstances( Model ):
   spanningTreeVlanInstances = Dict( keyType=int, valueType=SpanningTreeVlanInstance,
                                     help='Map vlans to their STP vlan instances. '
                                          'If a vlan exists, it will belong to this '
                                          'collection regardless of whether there '
                                          'is a spanning tree instance associated '
                                          'with it' )
   _message = Str( help='Message to show to the user', optional=True )

   def render( self ):
      # pylint: disable-msg=W0212
      if self._message:
         print( self._message )
         return

      for vlanId in sorted( self.spanningTreeVlanInstances.keys() ):
         instance = self.spanningTreeVlanInstances[ vlanId ]
         instance.render( vlanId )


class BridgeIntfInfo( Model ):
   externalPathCost = Int( help='Operational value to use for the external path cost'
                                ' for the port.  It is set based on the '
                                'extPathCost and linkPathCost values in the '
                                'PortConfig' )


class BridgeMstiIntfInfo( Model ):
   portId = Submodel( help='Port identifier for this MSTI. Each MSTI can have '
                           'a different port priority resulting in a different '
                           'port identifier.', valueType=SpanningTreePortId )
   role = Enum( help='Current port role for this instance and port', 
                values=( 'disabled', 'root', 'alternate', 'designated', 'backup', 
                         'master' ) )
   internalPathCost = Int( help='Path cost used on this port, either derived from '
                               'the link speed or specified via configuration' )
   fdbFlushes = Int( help='Incremented by one everytime fdb needs to be flushed' )


class BridgeMstiInfo( Model ):
   bridgeId = Submodel( help='Bridge identifier on this bridge', 
                        valueType=SpanningTreeBridgeBrief )
   designatedRoot = Submodel( help='Designated Root Bridge identifier', 
                              valueType=SpanningTreeBridgeBrief )
   interfaces = Dict( help='A mapping of interface name to its status info',
                        valueType=BridgeMstiIntfInfo, keyType=Interface )


class BridgeStpiInfo( Model ):
   cistRoot = Submodel( help='Common Internal Spanning Tree root identifier', 
                        valueType=SpanningTreeBridgeBrief  )
   cistPathCost = Int( help='Common Internal Spanning Tree path cost' )
   mstpInstances = Dict( help='A mapping of Multiple Spanning Tree Instance to its '
                              'status info', 
                    valueType=BridgeMstiInfo )
   interfaces = Dict( help='A mapping of interface name to its status info', 
                    valueType=BridgeIntfInfo, keyType=Interface )


class SpanningTreeBridgeMstTestInfoModel( Model ):
   name = Str( help= 'Name of bridge' )
   stpVersion = Enum( help='Version of the protocol being run', 
                      values=( 'none', 'stp', 'rstp', 'mstp', 'backup', 
                               'rapid-pvst', 'rapidPvstp') )
   mstpRegionId = Str( help='MSTP specific region Identifier' )
   bridgeAddr = MacAddress( help='MAC address of bridge' )
   stpInstances = Dict( help='A mapping of VLAN to spanning tree instances ('
                             'rapid-pvst) OR the entire group of MSTP '
                             'instances with key "Mst"', 
                    valueType=BridgeStpiInfo )

   def render( self ):
      pass
