#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import ConfigMount
import LazyMount
import HostnameCli

from Arnet import IntfId, IpGenAddr
from binascii import unhexlify
from CliMatcher import DynamicNameMatcher, IntegerMatcher, PatternMatcher
from CliMode.StunCliMode import ( StunBaseMode,
                                  StunServerBaseMode,
                                  StunClientBaseMode,
                                  StunServerProfileBaseMode )
from CliPlugin import ( EthIntfCli,
                        LoopbackIntfCli,
                        VlanIntfCli,
                        SubIntfCli,
                        LagIntfCli,
                        SharedSecretProfileCli,
                        IpAddrMatcher,
                        TechSupportCli )
from CliPlugin.Ssl import profileNameMatcher
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForConfig
from IpLibConsts import DEFAULT_VRF
from TypeFuture import TacLazyType
from IptablesHelper import RT
from Toggles.StunToggleLib import toggleStunEndpointDependentNatEnabled

StunTransportType = TacLazyType( "Stun::TransportType" )
StunTransactionId = TacLazyType( "Stun::TransactionId" )
ClearServerBindingsCommand = TacLazyType( "Stun::ClearServerBindingsCommand" )
SslConnectionLifetimeUnit = TacLazyType( "Stun::SslConnectionLifetimeUnit" )
SslConnectionLifetime = TacLazyType( "Stun::SslConnectionLifetime" )
TurnserverTracingLevel = TacLazyType( "Stun::TurnserverTracingLevel" )

stunServerConfig = None
stunServerStatus = None
stunClientConfig = None
stunServerProfile = None
stunClientStatus = None
ipStatus = None

serverProfileNameMatcher = DynamicNameMatcher(
      lambda mode: stunServerProfile.serverProfile,
      helpdesc='STUN server profile name',
      helpname='WORD' )

class StunMode( StunBaseMode, BasicCli.ConfigModeBase ):
   name = "STUN configuration"

   def __init__( self, parent, session ):
      StunBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class StunServerMode( StunServerBaseMode, BasicCli.ConfigModeBase ):
   name = "STUN server configuration"

   def __init__( self, parent, session ):
      StunServerBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class StunClientMode( StunClientBaseMode, BasicCli.ConfigModeBase ):
   name = "STUN client configuration"

   def __init__( self, parent, session ):
      StunClientBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class StunServerProfileMode( StunServerProfileBaseMode, BasicCli.ConfigModeBase ):
   name = "STUN server profile configuration"

   def __init__( self, parent, session, name ):
      self.profileName = name
      self.profile = self.getOrCreateServerProfileConfig()
      StunServerProfileBaseMode.__init__( self, self.profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateServerProfileConfig( self ):
      serverProfileConfig = stunServerProfile.serverProfile.get( self.profileName )
      if serverProfileConfig is None:
         serverProfileConfig = stunServerProfile.serverProfile.newMember(
                                                               self.profileName )
      return serverProfileConfig

def clearServerConfig():
   stunServerConfig.disabled = stunServerConfig.disabledDefault
   stunServerConfig.port = stunServerConfig.portDefault
   stunServerConfig.localIntfIds.clear()
   stunServerConfig.passwordProfile = stunServerConfig.passwordProfileDefault
   stunServerConfig.sslProfile = stunServerConfig.sslProfileDefault
   stunServerConfig.transportType = stunServerConfig.transportTypeDefault
   stunServerConfig.bindingResponseTimeout = \
      stunServerConfig.bindingResponseTimeoutDefault
   stunServerConfig.forceRestartOnConfigChange = False
   stunServerConfig.udpdtls = False
   stunServerConfig.turnTracing = TurnserverTracingLevel.errors
   stunServerConfig.sslConnectionLifetime = SslConnectionLifetime(
      stunServerConfig.sslConnectionLifetimeDefault,
      SslConnectionLifetimeUnit.minutes )

def clearClientConfig():
   clearServerProfileConfig()
   stunClientConfig.disabled = stunClientConfig.disabledDefault
   stunClientConfig.refreshInterval = stunClientConfig.refreshIntervalDefault

def clearServerProfileConfig():
   stunServerProfile.serverProfile.clear()

def clearConfig():
   clearServerConfig()
   clearClientConfig()
   clearServerProfileConfig()


class StunModeCmd( CliCommand.CliCommandClass ):
   syntax = '''stun'''
   noOrDefaultSyntax = syntax

   data = {
      'stun': 'STUN configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( StunMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearConfig()

BasicCli.GlobalConfigMode.addCommandClass( StunModeCmd )

class StunServerModeCmd( CliCommand.CliCommandClass ):
   syntax = '''server'''
   noOrDefaultSyntax = syntax

   data = {
      'server': 'Configure STUN server',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( StunServerMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearServerConfig()

class StunClientModeCmd( CliCommand.CliCommandClass ):
   syntax = '''client'''
   noOrDefaultSyntax = syntax

   data = {
      'client': 'Configure STUN client',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( StunClientMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearClientConfig()

class StunServerProfileModeCmd( CliCommand.CliCommandClass ):
   syntax = '''server-profile NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'server-profile': 'Create a STUN server profile',
      'NAME': serverProfileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      childMode = mode.childMode( StunServerProfileMode, name=name )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateServerProfileConfig()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'NAME' ]
      del stunServerProfile.serverProfile[ name ]

StunMode.addCommandClass( StunServerModeCmd )
StunMode.addCommandClass( StunClientModeCmd )
StunClientMode.addCommandClass( StunServerProfileModeCmd )

#----------------------------------------------------------------------------
# "[ no | default ] disabled" config command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
            'disabled': 'Disable STUN server',
          }

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.disabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.disabled = stunServerConfig.disabledDefault

intfMatcher = IntfMatcher()
intfMatcher |= VlanIntfCli.VlanIntf.matcher
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LoopbackIntfCli.LoopbackIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= LagIntfCli.subMatcher

#----------------------------------------------------------------------------
# "[ no | default ] local-interface INTF" config command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerLocalIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = syntax
   data = {
            'local-interface' : 'Configure STUN server to listen on an interface',
            'INTF' : intfMatcher
          }

   @staticmethod
   def handler( mode, args ):
      intfName = args.get( "INTF" )
      intfId = IntfId( intfName )
      intf = ipStatus.ipIntfStatus.get( intfId )

      if intf and intf.vrf != DEFAULT_VRF:
         mode.addErrorAndStop( "Interfaces in non-default VRFs are not supported" )

      stunServerConfig.localIntfIds.add( intfId )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = args.get( "INTF" )
      del stunServerConfig.localIntfIds[ IntfId( intfName ) ]

#----------------------------------------------------------------------------
# "[ no | default ] port PORT" config command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
            'port' : 'Configure listening port for STUN server (default: 3478)',
            'PORT' : IntegerMatcher( 1, 65535,
                                     helpdesc='STUN server port' ),
          }

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.port = args[ "PORT" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.port = stunServerConfig.portDefault

#---------------------------------------------------------------------------------
# "[ no | default ] password profile PASSWORD config command in "stun server" mode
#---------------------------------------------------------------------------------
class StunServerPasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'password profile PROFILE'
   noOrDefaultSyntax = 'password profile ...'
   data = {
      'password': 'Configure password related options for STUN server',
      'profile': 'Configure shared-secret profile',
      'PROFILE': DynamicNameMatcher(
         lambda mode: SharedSecretProfileCli.config.profile,
         'shared-secret profile name' )
   }

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.passwordProfile = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.passwordProfile = stunServerConfig.passwordProfileDefault

#------------------------------------------------------------------------------------
# [ no | default ] ssl profile PROFILE config command in "stun server" mode
#------------------------------------------------------------------------------------
class StunServerSslCmd( CliCommand.CliCommandClass ):
   syntax = "ssl profile PROFILE"
   noOrDefaultSyntax = "ssl profile ..."
   data = {
      "ssl": "Configure SSL related options for STUN server",
      "profile": "Configure SSL profile",
      "PROFILE": profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.sslProfile = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.sslProfile = stunServerConfig.sslProfileDefault

#------------------------------------------------------------------------------------
# [ no | default ] transport protocol (udp|tcp) config command in "stun server" mode
#------------------------------------------------------------------------------------
class StunServerTransportCmd( CliCommand.CliCommandClass ):
   syntax = "transport protocol ( udp | tcp )"
   noOrDefaultSyntax = "transport protocol ..."
   data = {
      "transport": "Transport related configuration for STUN server",
      "protocol": "Configure transport protocol for STUN server (default: UDP)",
      "udp": "Configure STUN server to listen for UDP requests (default)",
      "tcp": "Configure STUN server to listen for TCP requests",
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      transportType = StunTransportType.udp
      if "tcp" in args:
         transportType = StunTransportType.tcp
      stunServerConfig.transportType = transportType

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.transportType = stunServerConfig.transportTypeDefault

#----------------------------------------------------------------------------
# "[ no | default ] config change force-restart" command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerRestartEnabledCmd( CliCommand.CliCommandClass ):
   syntax = 'config change force-restart'
   noOrDefaultSyntax = syntax
   data = {
      'config': "STUN server configuration",
      'change': "Changes related to STUN server configuration",
      'force-restart': "Restart turnserver upon STUN server config change",
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.forceRestartOnConfigChange = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.forceRestartOnConfigChange = False

#----------------------------------------------------------------------------
# "[ no | default ] config change security migration" command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerMigrationModeCmd( CliCommand.CliCommandClass ):
   syntax = "config change security migration"
   noOrDefaultSyntax = syntax
   data = {
      'config': "STUN server configuration",
      'change': "Changes related to STUN server configuration",
      'security': "Security related config changes for STUN server",
      'migration': "Enable migration mode to support udp and dtls together",
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.udpdtls = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.udpdtls = False

#----------------------------------------------------------------------------
# "[ no | default ] ssl connection lifetime ( ( LIFETIME_MIN minutes ) |
#                                             ( LIFETIME_HRS hours ) )"
# command in "stun server" mode
#----------------------------------------------------------------------------
class StunServerSslConnectionLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = """
   ssl connection lifetime ( ( LIFETIME_MIN minutes ) | ( LIFETIME_HRS hours ) )
   """
   noOrDefaultSyntax = "ssl connection lifetime ..."
   data = {
      'ssl': "Configure SSL related options for STUN server",
      'connection': "Configure SSL connection related options for STUN server",
      'lifetime': "Configure lifetime for SSL connection",
      'LIFETIME_MIN': IntegerMatcher(
         1, 1440, helpdesc="SSL connection lifetime in minutes (default: 120)" ),
      'minutes': "Time unit for the SSL connection lifetime",
      'LIFETIME_HRS': IntegerMatcher(
         1, 24, helpdesc="SSL connection lifetime in hours (default: 2)" ),
      'hours': "Time unit for the SSL connection lifetime",
   }

   @staticmethod
   def handler( mode, args ):
      minutes = args.get( "LIFETIME_MIN" )
      hours = args.get( "LIFETIME_HRS" )
      if minutes:
         unit = SslConnectionLifetimeUnit.minutes
         seconds = minutes * 60
      else:                     # hours
         unit = SslConnectionLifetimeUnit.hours
         seconds = hours * 60 * 60

      lifetime = SslConnectionLifetime( seconds, unit )
      stunServerConfig.sslConnectionLifetime = lifetime

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.sslConnectionLifetime = \
         SslConnectionLifetime( stunServerConfig.sslConnectionLifetimeDefault,
                                SslConnectionLifetimeUnit.minutes )

#----------------------------------------------------------------------------
# "[ no | default ] trace level ( errors | info | verbose )" command in "stun server"
# mode
#----------------------------------------------------------------------------
class StunServerTurnserverTracingCmd( CliCommand.CliCommandClass ):
   syntax = "trace level ( errors | info | verbose )"
   noOrDefaultSyntax = syntax
   data = {
      'trace': "Configure tracing options for the turnserver",
      'level': "Configure tracing level for the turnserver",
      'errors': "Enable tracing for errors (default)",
      'info': "Enable informational traces",
      'verbose': "Enable all tracing levels",
   }

   @staticmethod
   def handler( mode, args ):
      if "errors" in args:
         stunServerConfig.turnTracing = TurnserverTracingLevel.errors
      elif "info" in args:
         stunServerConfig.turnTracing = TurnserverTracingLevel.info
      elif "verbose" in args:
         stunServerConfig.turnTracing = TurnserverTracingLevel.verbose
      else:
         raise ValueError( "Unsupported trace level configuration" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.turnTracing = TurnserverTracingLevel.errors

#----------------------------------------------------------------------------
# "[ no | default ] disabled" config command in "stun client" mode
#----------------------------------------------------------------------------
class StunClientDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
            'disabled': 'Disable STUN client',
          }

   @staticmethod
   def handler( mode, args ):
      stunClientConfig.disabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunClientConfig.disabled = stunClientConfig.disabledDefault

#----------------------------------------------------------------------------
# "[ no | default ] disabled" config command in "stun server-profile < name >" mode
#----------------------------------------------------------------------------
class StunServerProfileDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
            'disabled': 'Disable STUN server profile',
          }

   @staticmethod
   def handler( mode, args ):
      mode.profile.disabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.disabled = False

#----------------------------------------------------------------------------
# "[ no | default ] ip address IP_ADDR" config command in "stun server-profile" mode
#----------------------------------------------------------------------------
class StunServerProfileIpAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'ip address IP_ADDR'
   noOrDefaultSyntax = 'ip address ...'
   data = {
            'ip' : ipMatcherForConfig,
            'address': 'Configure STUN server address',
            'IP_ADDR': IpAddrMatcher.IpAddrMatcher(
                             helpdesc='Configure STUN server address' ),
          }

   @staticmethod
   def handler( mode, args ):
      mode.profile.serverAddress = IpGenAddr( args[ 'IP_ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.serverAddress = IpGenAddr()

#----------------------------------------------------------------------------
# "[ no | default ] port PORT" config command in "stun server-profile" mode
#----------------------------------------------------------------------------
class StunServerProfileServerPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
            'port' : 'Destination port for the request to STUN server '
                     '(default: 3478)',
            'PORT' : IntegerMatcher( 1, 65535,
                                     helpdesc='STUN server port' ),
          }

   @staticmethod
   def handler( mode, args ):
      mode.profile.serverPort = args[ "PORT" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.serverPort = mode.profile.serverPortDefault

#----------------------------------------------------------------------------
# "[ no | default ] serverHostName HOSTNAME" config command in
# "stun server-profile" mode
#----------------------------------------------------------------------------
class StunServerProfileHostNameCmd( CliCommand.CliCommandClass ):
   syntax = 'hostname HOSTNAME'
   noOrDefaultSyntax = 'hostname ...'
   data = {
            'hostname' : 'Configure STUN server hostname',
            'HOSTNAME': HostnameCli.HostnameMatcher( helpname='WORD',
                                                     helpdesc='Hostname' )
          }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.profile.serverHostName = args[ "HOSTNAME" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.serverHostName = mode.profile.serverHostNameDefault

#---------------------------------------------------------------------------------
# "[ no | default ] password profile PASSWORD config command in
# "stun server-profile" mode
#---------------------------------------------------------------------------------
class StunServerProfilePasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'password profile PROFILE'
   noOrDefaultSyntax = 'password profile ...'
   data = {
      'password': 'Configure password related options for STUN server',
      'profile': 'Configure shared-secret profile',
      'PROFILE': DynamicNameMatcher(
         lambda mode: SharedSecretProfileCli.config.profile,
         'shared-secret profile name' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.profile.passwordProfile = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.passwordProfile = mode.profile.passwordProfileDefault

#------------------------------------------------------------------------------------
# [ no | default ] ssl profile PROFILE config command in "stun server-profile" mode
#------------------------------------------------------------------------------------
class StunServerProfileSslCmd( CliCommand.CliCommandClass ):
   syntax = "ssl profile PROFILE"
   noOrDefaultSyntax = "ssl profile ..."
   data = {
      "ssl": "Configure SSL related options for STUN server",
      "profile": "Configure SSL profile",
      "PROFILE": profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.profile.sslProfile = args[ 'PROFILE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.sslProfile = mode.profile.sslProfileDefault

# -------------------------------------------------------------------------------
# [ no | default ] tunnel disabled config command in "stun server-profile" mode
# -------------------------------------------------------------------------------
class StunServerProfileTunnelModeCmd( CliCommand.CliCommandClass ):
   syntax = "tunnel disabled"
   noOrDefaultSyntax = syntax
   data = {
      "tunnel": "Send STUN packets through a tunnel",
      "disabled": "Disable tunneling of STUN packets",
   }

   @staticmethod
   def handler( mode, args ):
      mode.profile.tunnelMode = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.profile.tunnelMode = True


#------------------------------------------------------------------------------------
# clear stun server bindings ( all | ( transaction-id TID ) |
# ( ip address ADDRESS [ port PORT ] ) )
#------------------------------------------------------------------------------------
class ClearStunBindingsCmd( CliCommand.CliCommandClass ):
   syntax = """
   clear stun server bindings
   ( all
   | ( transaction-id TID )
   | ( ip address ADDRESS [ port PORT ] ) )
   """
   data = {
      "clear" : clearKwNode,
      "stun" : "Clear STUN information",
      "server" : "Clear STUN server related information",
      "bindings" : "Clear STUN bindings related information",
      "all" : "Clear all STUN bindings",
      "transaction-id" : "Clear STUN binding request for a transaction ID",
      "TID" : PatternMatcher( pattern=r'[a-fA-F0-9]+', helpdesc="Transaction ID",
                              helpname="HEX STRING" ),
      "ip" : "Clear STUN bindings matching a public IP address",
      "address" : "Clear STUN bindings matching a public IP address",
      "ADDRESS" : IpAddrMatcher.IpAddrMatcher( helpdesc='Public IP address' ),
      "port" : "Clear STUN bindings matching a public IP address and port",
      "PORT" : IntegerMatcher( 1, 65535, helpdesc="Public port" ),
   }

   @staticmethod
   def handler( mode, args ):
      tid = args.get( "TID" )
      publicIp = args.get( "ADDRESS" )
      publicPort = args.get( "PORT" )
      clearAll = "all" in args
      clearBr = ClearServerBindingsCommand( clearAll )
      if tid:
         clearBr.tid = StunTransactionId( unhexlify( tid ) )
      elif publicIp:
         clearBr.publicIp = IpGenAddr( publicIp )
         clearBr.publicPort = int( publicPort ) if publicPort else 0
      clearBr.version = stunServerConfig.clearBr.version + 1
      stunServerConfig.clearBr = clearBr

#------------------------------------------------------------------------------------
# [ no | default ] binding timeout TIMEOUT seconds
#------------------------------------------------------------------------------------
class StunServerBindingsTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = "binding timeout TIMEOUT seconds"
   noOrDefaultSyntax = "binding timeout ..."
   data = {
      "binding" : "Configuration for bindings stored on STUN server",
      "timeout" : "Configure timeout for bindings stored on STUN server",
      "TIMEOUT" : IntegerMatcher( 10, 7200, helpdesc="Timeout value in seconds" ),
      "seconds" : "Time unit for the binding timeout",
   }

   @staticmethod
   def handler( mode, args ):
      stunServerConfig.bindingResponseTimeout = args[ "TIMEOUT" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunServerConfig.bindingResponseTimeout = \
         stunServerConfig.bindingResponseTimeoutDefault

#------------------------------------------------------------------------------------
# [ no | default ] refresh interval <REFRESH_INTERVAL> seconds
#------------------------------------------------------------------------------------
class StunClientRefreshIntervalCmd( CliCommand.CliCommandClass ):
   syntax = "refresh interval REFRESH_INTERVAL seconds"
   noOrDefaultSyntax = "refresh interval ..."
   data = {
      "refresh" : "Bindings refresh configuration",
      "interval" : "Refresh interval for bindings",
      "REFRESH_INTERVAL" : IntegerMatcher( 10, 600,
                                           helpdesc="refresh interval in seconds" ),
      "seconds" : "Time unit for the refresh interval",
   }

   @staticmethod
   def handler( mode, args ):
      stunClientConfig.refreshInterval = args[ "REFRESH_INTERVAL" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stunClientConfig.refreshInterval = \
         stunClientConfig.refreshIntervalDefault

def stunClientRunning():
   return stunClientStatus.active

def stunServerRunning():
   return not stunServerStatus.disabled

def registerShowTechCommands():
   TechSupportCli.registerShowTechSupportCmd(
      '2022-06-28 09:00:00',
      cmds=[ "show stun client translations detail",
             RT( "bash sudo iptables -t filter -S {{inputchain}}" ), ],
      cmdsGuard=stunClientRunning )

   TechSupportCli.registerShowTechSupportCmd(
      '2022-06-28 09:00:00',
      cmds=[ "show stun server bindings detail",
             RT( "bash sudo iptables -t filter -S {{inputchain}}" ), ],
      cmdsGuard=stunServerRunning )

   TechSupportCli.registerShowTechSupportCmd(
      '2022-08-22 23:57:00',
      cmds=[ 'show stun server status' ],
      cmdsGuard=stunServerRunning )

   TechSupportCli.registerShowTechSupportCmd(
      '2022-11-29 01:43:15',
      cmds=[ 'show stun client counters' ],
      cmdsGuard=stunClientRunning )

StunServerMode.addCommandClass( StunServerDisabledCmd )
StunServerMode.addCommandClass( StunServerLocalIntfCmd )
StunServerMode.addCommandClass( StunServerPortCmd )
StunServerMode.addCommandClass( StunServerPasswordCmd )
StunServerMode.addCommandClass( StunServerSslCmd )
StunServerMode.addCommandClass( StunServerTransportCmd )
StunServerMode.addCommandClass( StunServerSslConnectionLifetimeCmd )
StunServerMode.addCommandClass( StunServerTurnserverTracingCmd )
StunServerMode.addCommandClass( StunServerRestartEnabledCmd )
StunServerMode.addCommandClass( StunServerMigrationModeCmd )
BasicCliModes.EnableMode.addCommandClass( ClearStunBindingsCmd )
StunServerMode.addCommandClass( StunServerBindingsTimeoutCmd )
StunClientMode.addCommandClass( StunClientDisabledCmd )
StunServerProfileMode.addCommandClass( StunServerProfileDisabledCmd )
StunServerProfileMode.addCommandClass( StunServerProfileIpAddressCmd )
StunServerProfileMode.addCommandClass( StunServerProfileServerPortCmd )
StunServerProfileMode.addCommandClass( StunServerProfileHostNameCmd )
StunServerProfileMode.addCommandClass( StunServerProfilePasswordCmd )
StunServerProfileMode.addCommandClass( StunServerProfileSslCmd )
if toggleStunEndpointDependentNatEnabled():
   StunServerProfileMode.addCommandClass( StunServerProfileTunnelModeCmd )
StunClientMode.addCommandClass( StunClientRefreshIntervalCmd )
registerShowTechCommands()

def Plugin( entityManager ):
   global stunServerConfig, stunServerStatus, stunClientConfig, stunServerProfile
   global stunClientStatus, ipStatus

   stunServerConfig = ConfigMount.mount( entityManager, 'stun/server/config',
                                         'Stun::ServerConfig', 'w' )
   stunServerStatus = LazyMount.mount( entityManager, 'stun/server/status',
                                       'Stun::ServerStatus', 'r' )
   stunClientConfig = ConfigMount.mount( entityManager, 'stun/client/cliConfig',
                                         'Stun::ClientCliConfig', 'w' )
   stunServerProfile = ConfigMount.mount( entityManager,
                                         'stun/server-profile',
                                         'Stun::ServerProfileConfig', 'w' )
   stunClientStatus = LazyMount.mount( entityManager,
                                         'stun/client/status',
                                         'Stun::StunClientStatus', 'r' )
   ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )

