# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import LazyMount
import ShowCommand
import Tac

from Arnet import IntfId
from ArnetModel import IpAddrAndPort
from binascii import unhexlify, Error
from CliMatcher import ( IntegerMatcher, KeywordMatcher, PatternMatcher,
                         DynamicNameMatcher )
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.StunModels import ( StunServerBindingModel,
                                   StunServerBindingsModel, StunTlvModel,
                                   StunClientBindingsModel,
                                   StunClientBindingModel,
                                   StunServerStatusModel,
                                   StunClientCountersModel,
                                   StunServerProfileCountersModel,
                                   StunTransportCountersModel )
from CliPlugin.StunTlvParser import tlvTypeToParser

stunServerConfig = None
stunServerStatus = None
stunClientAppStatusDir = None
stunClientAppConfigDir = None
stunCliConfig = None
stunClientStatus = None
stunServerProfile = None
stunKeywordMatcher = KeywordMatcher( "stun",
                                     helpdesc="Show STUN information" )

def calculateServerResponseTimeout( nextTimeoutAt ):
   timeRemaining = nextTimeoutAt - Tac.now()
   timeRemaining = int( timeRemaining ) if timeRemaining >= 0 else None
   return timeRemaining

def getServerTimeoutInterval():
   return int( stunServerConfig.bindingResponseTimeout )

def getClientRefreshInterval():
   return int( stunCliConfig.refreshInterval )

def populateStunTlvsModel( tid, tlvs ):
   tlvModelList = []
   for tlv in sorted( tlvs ):
      tlvParserClass = tlvTypeToParser.get( tlv.type )
      if not tlvParserClass:
         continue
      parsedTlv = tlvParserClass( tlv.value )
      tlvModel = StunTlvModel( tlvType=parsedTlv.typeStr(), tlvLen=tlv.length,
                               tlvValue=parsedTlv.valueStr() )
      tlvModelList.append( tlvModel )
   return tlvModelList

def populateServerBindingModel( tid, detail ):
   binding = stunServerStatus.bindingResponses.get( tid )
   if not binding:
      return None
   nextTimeoutAt = stunServerStatus.nextRespTimeout.get( tid )
   ipAddrAndPort = IpAddrAndPort( ip=binding.translatedAddress.v4Addr,
                                  port=binding.translatedPort )
   tlvsModel = []
   if detail and binding.tlv:
      tlvsModel = populateStunTlvsModel( tid, binding.tlv )

   timeoutInterval = getServerTimeoutInterval()
   timeRemaining = None
   if nextTimeoutAt:
      timeRemaining = calculateServerResponseTimeout( nextTimeoutAt )

   bindingModel = StunServerBindingModel( publicAddress=ipAddrAndPort,
                                          numTlvs=len( binding.tlv ),
                                          timeout=timeRemaining,
                                          timeoutInterval=timeoutInterval,
                                          tlvs=tlvsModel )
   return bindingModel

def populateServerBindingsModel( tid, detail ):
   tids = [ tid ] if tid else stunServerStatus.bindingResponses
   for _tid in tids:
      bindingModel = populateServerBindingModel( _tid, detail )
      if bindingModel is None:
         continue
      yield _tid.hexString(), bindingModel

def showStunServerBindingsHandler( mode, args ):
   detail = "detail" in args
   tid = args.get( 'TID' )
   if tid:
      # Convert hex string to bytes
      try:
         tidBytes = unhexlify( tid )
      except Error:
         return StunServerBindingsModel()
      tid = Tac.ValueConst( "Stun::TransactionId", tidBytes )
   bindings = populateServerBindingsModel( tid, detail )
   return StunServerBindingsModel( bindings=bindings,
                                   _detailedView=detail )

#----------------------------------------------------------------------
# show stun server bindings [ tid <TID> ] [ detail ]
#----------------------------------------------------------------------
class ShowStunServerBindings( ShowCommand.ShowCliCommandClass ):
   syntax = "show stun server bindings [ transaction-id TID ] [ detail ]"
   data = {
      'stun' : stunKeywordMatcher,
      'server': "Show STUN server information",
      'bindings': "Show STUN binding information",
      'transaction-id' : "Show bindings for a transaction ID",
      'TID' : PatternMatcher( pattern=r'[a-fA-F0-9]+', helpdesc="Transaction ID",
                              helpname="ABCdef1234" ),
      'detail': "Show STUN binding details",
   }

   handler = showStunServerBindingsHandler
   cliModel = StunServerBindingsModel

def populateClientBindingsModel( ipAddr, port, detailedView=False ):
   refreshInterval = getClientRefreshInterval()

   def populateEntry( agent, tid, srcIp, srcPort, publicIp=None, publicPort=None,
                      lastRefreshed=None ):
      srcAddr = None
      if srcIp and srcPort:
         srcAddr = IpAddrAndPort( ip=srcIp, port=srcPort )

      agentConfig = stunClientAppConfigDir.get( agent )
      tlvsModel = []

      req = agentConfig.stunRequest.get( tid ) if agentConfig else None
      tlvsModel = []
      if detailedView and req and req.tlv:
         tlvsModel = populateStunTlvsModel( tid, req.tlv )

      publicAddress = None
      if publicIp and publicPort:
         publicAddress = IpAddrAndPort( ip=publicIp, port=publicPort )

      return StunClientBindingModel(
         agentName=agent, sourceAddress=srcAddr, publicAddress=publicAddress,
         lastRefreshed=lastRefreshed, timeoutInterval=refreshInterval,
         tlvs=tlvsModel )


   for agent in stunClientAppConfigDir:
      for tid, req in stunClientAppConfigDir[ agent ].stunRequest.items():
         srcIp = req.sourceAddress.v4Addr
         srcPort = req.sourcePort

         if ipAddr and port and ( ipAddr != srcIp or port != srcPort ):
            # not the information user wants to see
            continue

         if agent in stunClientAppStatusDir:
            if tid in stunClientAppStatusDir[ agent ].stunResponse:
               # found an associated response - fill out public port/ip
               resp = stunClientAppStatusDir[ agent ].stunResponse[ tid ]
               lastRefreshed = int( Tac.utcNow() - resp.lastUpdateTime )
               yield tid.hexString(), populateEntry( agent, tid, srcIp, srcPort,
                                               resp.publicAddress.v4Addr,
                                               resp.publicPort, lastRefreshed )
               continue

         # no associated response found - no information about public ip/port
         yield tid.hexString(), populateEntry( agent, tid, srcIp, srcPort )

      # Look at responses that don't have associated requests...
      if ipAddr and port:
         # ... but user specified srcIp/port to filter output ...
         # ... so don't display entries that only contain information from response
         continue

      if agent not in stunClientAppStatusDir:
         continue

      for tid, resp in stunClientAppStatusDir[ agent ].stunResponse.items():
         # Skip if request exists
         if tid in stunClientAppConfigDir[ agent ].stunRequest:
            continue

         resp = stunClientAppStatusDir[ agent ].stunResponse[ tid ]
         lastRefreshed = int( Tac.utcNow() - resp.lastUpdateTime )

         yield tid.hexString(), populateEntry( agent, tid, None, None,
                                             resp.publicAddress.v4Addr,
                                             resp.publicPort, lastRefreshed )

def showClientBindingRequestsHandler( mode, args ):
   detail = "detail" in args
   ip = args.get( "IP" )
   port = args.get( "PORT" )
   bindings = populateClientBindingsModel( ip, port, detail )
   return StunClientBindingsModel( bindings=bindings, _detailedView=detail )

#----------------------------------------------------------------------
# show stun client translations [ IP PORT ] [ detail ]
#----------------------------------------------------------------------
class ShowClientTranslations( ShowCommand.ShowCliCommandClass ):
   syntax = "show stun client translations [ IP PORT ] [ detail ]"
   data = {
      'stun' : stunKeywordMatcher,
      'client': "Show STUN client information",
      'translations': "Show STUN client translations",
      'IP' : IpAddrMatcher( 'Local IP address' ),
      'PORT': IntegerMatcher( 1, 65535, helpdesc="Port" ),
      'detail': "Show client translation details",
   }

   handler = showClientBindingRequestsHandler
   cliModel = StunClientBindingsModel

def showStunServerStatusHandler( mode, args ):
   try:
      cmd = "pidof turnserver"
      out = Tac.run( cmd.split(), stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True )
      pid = int( out.split()[0] ) if out else 0
   except ValueError:
      pid = 0

   lastBindRcvd = 0
   if stunServerStatus.lastRcvdMsgTime:
      lastBindRcvd = int( Tac.now() - stunServerStatus.lastRcvdMsgTime )

   authMode = "none"
   if stunServerStatus.sslProfile:
      authMode = "ssl"
   elif stunServerStatus.passwordProfile:
      authMode = "stunLongTermCredentials"

   listeningIps = {}
   for addr, intf in stunServerStatus.ipAddressToIntf.items():
      listeningIps[ addr ] = IntfId( intf )
   listeningPort = stunServerStatus.port
   bindingTimeout = int( stunServerStatus.bindingResponseTimeout )
   sslConnectionLifetime = int( stunServerStatus.sslConnectionLifetime.lifetime )

   return StunServerStatusModel( enabled=not stunServerStatus.disabled,
                                 pid=pid, timeOfLastRcvdBindingReq=lastBindRcvd,
                                 authMode=authMode, listeningIps=listeningIps,
                                 listeningPort=listeningPort,
                                 bindingTimeout=bindingTimeout,
                                 sslConnectionLifetime=sslConnectionLifetime, )

#----------------------------------------------------------------------
# show stun server status
#----------------------------------------------------------------------
class ShowStunServerStatus( ShowCommand.ShowCliCommandClass ):
   syntax = "show stun server status"
   data = {
      "stun" : stunKeywordMatcher,
      "server" : "Show STUN server information",
      "status" : "Show status of STUN server",
   }

   handler = showStunServerStatusHandler
   cliModel = StunServerStatusModel

def getCounters( stunRequestCounters, tid, serverProfile ):
   spCntrsModel = StunServerProfileCountersModel()
   spCntrsModel.tid = tid.hexString()
   spCntrsModel.name = serverProfile
   spCntrs = stunRequestCounters[ tid ].stunServerProfileCounters[ serverProfile ]

   spCntrsModel.refreshRequests = spCntrs.requestRefreshes
   spCntrsModel.numRetries = spCntrs.numRetries
   spCntrsModel.retriesExhausted = spCntrs.retriesExhausted
   spCntrsModel.successfulResponses = spCntrs.successfulResponses
   spCntrsModel.invalidResponses = spCntrs.invalidResponses
   spCntrsModel.responseProcessingFailures = spCntrs.responseProcessFailures

   # Get SSL transport counters
   spCntrsModel.sslTransportCounters = StunTransportCountersModel( transport="SSL" )
   sslCntrsModel = spCntrsModel.sslTransportCounters
   sslCntrs = spCntrs.sslTransportSmCounters
   sslCntrsModel.successfulInitializations = sslCntrs.initSuccess
   sslCntrsModel.initializationFailures = sslCntrs.initFailures
   sslCntrsModel.readFailures = sslCntrs.readFailures
   sslCntrsModel.writeFailures = sslCntrs.writeFailures
   sslCntrsModel.connectionErrors = sslCntrs.connectionErrors

   # Get UDP transport counters
   spCntrsModel.udpTransportCounters = StunTransportCountersModel( transport="UDP" )
   udpCntrsModel = spCntrsModel.udpTransportCounters
   udpCntrs = spCntrs.udpTransportSmCounters
   udpCntrsModel.successfulInitializations = udpCntrs.initSuccess
   udpCntrsModel.initializationFailures = udpCntrs.initFailures
   udpCntrsModel.readFailures = udpCntrs.readFailures
   udpCntrsModel.writeFailures = udpCntrs.writeFailures
   udpCntrsModel.connectionErrors = udpCntrs.connectionErrors

   return spCntrsModel

def showStunClientCountersHandler( mode, args ):
   # Since we are using the iteration CLI operand "{}", the arguments passed will be
   # lists. Hence we need to access the individual elements.
   tid = None
   serverProfile = None
   if "TID" in args:
      tidBytes = unhexlify( args[ "TID" ][ 0 ] )
      tid = Tac.ValueConst( "Stun::TransactionId", tidBytes )
   if "SERVERPROFILE" in args:
      serverProfile = args[ "SERVERPROFILE" ][ 0 ]

   cntrsModel = StunClientCountersModel()

   stunCounters = stunClientStatus.stunCounters
   if not stunCounters:
      cntrsModel.cleanStarts = 0
      cntrsModel.nonCleanStarts = 0
      cntrsModel.stunServerProfileCounters = []
      return cntrsModel

   stunRequestCounters = stunCounters.stunRequestCounters

   spCntrs = []
   if tid and serverProfile:
      if tid in stunRequestCounters:
         if serverProfile in stunRequestCounters[ tid ].stunServerProfileCounters:
            spCntrs.append( getCounters( stunRequestCounters, tid, serverProfile ) )
   elif tid:
      if tid in stunRequestCounters:
         for sp in stunRequestCounters[ tid ].stunServerProfileCounters:
            spCntrs.append( getCounters( stunRequestCounters, tid, sp ) )
   elif serverProfile:
      for _tid in stunRequestCounters:
         if serverProfile in stunRequestCounters[ _tid ].stunServerProfileCounters:
            spCntrs.append( getCounters( stunRequestCounters, _tid, serverProfile ) )
   else:
      for _tid in stunRequestCounters:
         for sp in stunRequestCounters[ _tid ].stunServerProfileCounters:
            spCntrs.append( getCounters( stunRequestCounters, _tid, sp ) )

   if spCntrs:
      cntrsModel.cleanStarts = stunCounters.agentStarts
      cntrsModel.nonCleanStarts = stunCounters.agentRestarts
      cntrsModel.stunServerProfileCounters = spCntrs
   else:
      cntrsModel.cleanStarts = 0
      cntrsModel.nonCleanStarts = 0
      cntrsModel.stunServerProfileCounters = []

   return cntrsModel

# show stun client counters [ transaction-id <TID> ] [ server-profile <sp> ]
class ShowStunClientCounters( ShowCommand.ShowCliCommandClass ):
   # transaction-id and server-profile are optional arguments and we need the ability
   # to specify them in any order needed. This is achieved by using the iteration
   # operator {} which gives us the ability to specify the operand multiple times and
   # then restricting the operand themselves to a single instance by using the
   # CliCommand.singleKeyword object.
   syntax = "show stun client counters [ { ( transaction-id TID )" \
            " | ( server-profile SERVERPROFILE ) } ]"
   data = {
      "stun" : stunKeywordMatcher,
      "client" : "Show STUN client information",
      "counters" : "Display STUN client counters",
      "transaction-id" : CliCommand.singleKeyword( "transaction-id",
                                             helpdesc="Filter by transaction ID" ),
      "TID" : PatternMatcher( pattern=r'[a-fA-F0-9]+',
                              helpdesc="Transaction ID",
                              helpname="ABCDef1234" ),
      "server-profile" : CliCommand.singleKeyword( "server-profile",
                                             helpdesc="Filter by server profile" ),
      "SERVERPROFILE" : DynamicNameMatcher(
                                       lambda mode: stunServerProfile.serverProfile,
                                       helpdesc='Filter by server profile',
                                       helpname='WORD' )
   }

   handler = showStunClientCountersHandler
   cliModel = StunClientCountersModel

BasicCli.addShowCommandClass( ShowStunServerBindings )
BasicCli.addShowCommandClass( ShowClientTranslations )
BasicCli.addShowCommandClass( ShowStunServerStatus )
BasicCli.addShowCommandClass( ShowStunClientCounters )

def Plugin( entityManager ):
   global stunServerConfig, stunServerStatus, stunClientStatus
   global stunClientAppStatusDir, stunClientAppConfigDir, stunCliConfig
   global stunServerProfile

   stunServerConfig = LazyMount.mount( entityManager, 'stun/server/config',
                                       'Stun::ServerConfig', 'r' )
   stunServerStatus = LazyMount.mount( entityManager, 'stun/server/status',
                                       'Stun::ServerStatus', 'r' )
   stunClientAppStatusDir = LazyMount.mount( entityManager, 'stun/client/appStatus',
                                          'Tac::Dir', 'ri' )
   stunClientAppConfigDir = LazyMount.mount( entityManager, 'stun/client/appConfig',
                                          'Tac::Dir', 'ri' )
   stunCliConfig = LazyMount.mount( entityManager, 'stun/client/cliConfig',
                                    'Stun::ClientCliConfig', 'r' )
   stunClientStatus = LazyMount.mount( entityManager, 'stun/client/status',
                                    'Stun::StunClientStatus', 'r' )
   stunServerProfile = LazyMount.mount( entityManager, 'stun/server-profile',
                                        'Stun::ServerProfileConfig', 'r' )
