# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

import os
import sys

import AgentDirectory
import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IntfProfileCli as IntfProfileCli
import CliPlugin.PhysicalIntfRule
import CliPlugin.SwitchIntfCli as SwitchIntfCli
import CliPlugin.TechSupportCli as TechSupportCli
from CliPlugin.VirtualIntfRule import IntfMatcher
import CliPlugin.IntfStatusMatcher as IntfStatusMatcher
import CliPlugin.VrfCli as VrfCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.IntfCli import (
      intfMtuGuardHook,
      subintfMtuGuardHook,
)
from CliPlugin.IntfModel import (
      InterfaceCounters,
      InterfaceCountersRate,
)
from CliPlugin.VlanCli import (
      vlanIdMatcher,
      vlanNameMatcher,
      splitHorizonGroupMatcher,
      splitHorizonGroupSupportedGuard,
)
import CliToken.Mac
import ConfigMount
import EthIntfLib
import Ethernet
from Intf.IntfRange import subintfSupportedGuard
from IntfRangePlugin.SwitchIntf import SwitchAutoIntfType
import LazyMount
import SharedMem
import ShowCommand
import Smash
import Tac
import TacSigint
from Toggles.EbraToggleLib import (
      toggleSubIntfSpeedEnabled,
      toggleSplitHorizonGroupEnabled,
)
from TypeFuture import TacLazyType

gv = CliGlobal.CliGlobal( aleCliConfig=None,
                          allEthIntfConfigDir=None,
                          allIntfStatusDir=None,
                          bridgingHwCapabilities=None,
                          bridgingHwEnabled=None,
                          bridgingInputConfig=None,
                          intfCapability=None,
                          serviceIntfTypeWinner=None,
                          subIntfConfigDir=None,
                          subIntfCountersDir=None,
                          subIntfCountersCheckpointDir=None,
                          subIntfDpCountersDir=None,
                          subIntfDpCountersCheckpointDir=None,
                          subIntfStatusDir=None )

Dot1qEncapMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )
EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
EthSpeedApi = TacLazyType( "Interface::EthSpeedApi" )
FlexEncapConstants = TacLazyType( 'Ebra::FlexEncapConstants' )
FlexEncapRange = TacLazyType( 'Ebra::FlexEncapRange' )
FlexEncapField = TacLazyType( 'Ebra::FlexEncapField' )
FlexEncapSpec = TacLazyType( 'Ebra::FlexEncapSpec' )
FlexEncapEntry = TacLazyType( 'Ebra::FlexEncapEntry' )
IntfState = TacLazyType( 'Interface::IntfEnabledState' )
ServiceIntfType = TacLazyType( 'Ebra::ServiceIntfType' )
VlanEncapReason = TacLazyType( "Interface::SubIntfConfigDir::VlanEncapReason" )

def getPlatformName( sysname ):
   if AgentDirectory.agent( sysname, 'Sand' ) or \
      os.environ.get( 'SIMULATION_SAND' ):
      return 'Sand'
   if AgentDirectory.agent( sysname, 'StrataCentral' ) or \
      os.environ.get( 'SIMULATION_STRATA' ):
      return 'Strata'
   return None

def theSubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.l3SubintfSupported or
        gv.intfCapability.l2SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theQinQL3SubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.qinql3SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theL2SubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.l2SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theL2SubintfMacLearningDisableSupported( mode, token ):
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.l2SubintfMacLearningDisableSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def intfMtuGuard( mode, token ):
   if mode.intf.name.startswith( "Management" ):
      return None
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.perIntfL3MtuSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def subintfMtuGuard( mode, token ):
   if ( not mode.session_.isInteractive() or
        gv.intfCapability.subIntfMtuSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

subintfSupportedGuard.append( theSubintfSupportedGuard )
intfMtuGuardHook.append( intfMtuGuard )
subintfMtuGuardHook.append( subintfMtuGuard )

def ethernetSubintfParentCfg( parentName ):
   if not parentName.startswith( ( 'Et', 'Sw' ) ):
      return None
   sliceName = EthIntfLib.sliceName( parentName )
   # In case sliceName is empty, we just use allEthIntfConfigDir.
   if sliceName in gv.allEthIntfConfigDir:
      ethPhyDir = gv.allEthIntfConfigDir[ sliceName ]
      parentConf = ethPhyDir.intfConfig.get( parentName ) if ethPhyDir else None
   else:
      parentConf = gv.allEthIntfConfigDir.get( parentName )
   return parentConf

matcherLearning = CliMatcher.KeywordMatcher( 'learning',
      helpdesc='Learning keyword' )

# Currently because SubIntf is in Ebra package, we have to add this hook
# to get parentConfig for ethernet or port-channel sub interfaces. This
# is not a clean design, and needs to be revisited. Instead, the SubIntf
# should really be defined in Intf package. And in appropriate packages,
# we can createSubIntf corresponding sub classes of SubIntf class, to override
# the _parentConfig method, which will be a much cleaner design.
# BUG111733 created to track this.
subintfParentCfgHook = [ ethernetSubintfParentCfg ]

class SubIntf( IntfCli.VirtualIntf ):
   def __init__( self, name, mode ):
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.subIntfStatuses = gv.subIntfStatusDir.intfStatus
      self.intfStatus = None
      self.parentIntfStatus = None
      self.subIntfConfig = None

   #----------------------------------------------------------------------------
   # Returns the SubIntfConfig object for this interface.
   # SubIntfConfig is cached to prevent an update from the EntityLog thread
   # in the middle of processing a "show interfaces" command from causing an
   # inconsistent snapshot to be displayed.
   # The caching is done only if not in a config session, since the above
   # scenario cannot happen when in a config session.
   #----------------------------------------------------------------------------
   def config( self ):
      config = gv.subIntfConfigDir.intfConfig.get( self.name )
      if not self.mode_.session_.inConfigSession():
         self.subIntfConfig = config
      return config

   @Tac.memoize
   def status( self ):
      return self.subIntfStatuses.get( self.name )

   def slotId( self ):
      if not EthIntfId.isEthIntfId( self.name ):
         return None

      return ( EthIntfLib.sliceName( self.name )
               if Cell.cellType() == 'supervisor'
               else 'FixedSystem' )

   # Performance of this function is not optimal, because we need to
   # search for parent name in config collections every time.
   def _parentConfig( self ):
      parentName = self.name.split( '.' )[ 0 ]
      for hook in subintfParentCfgHook:
         parentConf = hook( parentName )
         if parentConf:
            return parentConf
      return None

   def parentStatus( self ):
      if not self.parentIntfStatus:
         parentName = self.name.split( '.' )[ 0 ]
         self.parentIntfStatus = gv.allIntfStatusDir.intfStatus.get( parentName )

      return self.parentIntfStatus

   def createPhysical( self, startupConfig=False ):
      gv.subIntfConfigDir.intfConfig.newMember( self.name )
      # The default adminEnabledState is unknownEnabledState. Override the state
      # to enabled (True) only if it's not already set
      if self.config().adminEnabledState() == IntfState.unknownEnabledState:
         self.config().adminEnabled = True
      if self.name not in gv.bridgingInputConfig.switchIntfConfig:
         sic = gv.bridgingInputConfig.newSwitchIntfConfig( self.name, 'dot1qTunnel' )
         # By default, no bridging on this interface. But if a vlan is configured on
         # this interface, then it makes this a l2 subintf, where only bridging is
         # allowed.
         sic.enabled = False

   #----------------------------------------------------------------------------
   # Determines whether the Interface::SubIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.subIntfStatuses.get( self.name, None )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      pass

   #----------------------------------------------------------------------------
   # Destroys the Interface::SubIntfConfig object for this interface if it already
   # exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del gv.subIntfConfigDir.intfConfig[ self.name ]
      del gv.bridgingInputConfig.switchIntfConfig[ self.name ]

   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      dot1qEncapModeNone = Dot1qEncapMode.subIntfDot1qEncapConfigNone
      reason = gv.subIntfConfigDir.deleteVlanEncap(
            self.name, FlexEncapConstants.defaultClientId, FlexEncapEntry(),
            dot1qEncapModeNone )
      if reason != VlanEncapReason.success:
         self.mode_.addError( f"Unexpected reason: {reason}" )
         return
      # At this point the bridging switchIntfConfig has been deleted by
      # SwitchIntfConfigCleaner, but let's instantiate a default entry
      # to be consistent with createPhysical().
      switchIntfConfig = gv.bridgingInputConfig.switchIntfConfig
      assert self.name not in switchIntfConfig
      sic = switchIntfConfig.newMember( self.name, 'dot1qTunnel' )
      # keep in sync with createPhysical()
      sic.enabled = False
      self.setMacAddr( None )

   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in gv.subIntfStatusDir.intfStatus:
         intf = SubIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   #----------------------------------------------------------------------------
   # Encap Vlan for a subinterface
   #----------------------------------------------------------------------------
   def setVlanEncap( self, mode, configMode, vlanId=None, innerVlanId=None,
                     clientId=FlexEncapConstants.defaultClientId ):
      ModeConflictStr = ( 'encapsulation dot1q vlan command cannot be used in '
                          'conjunction with encapsulation submode' )
      if not vlanId:
         reason = gv.subIntfConfigDir.deleteVlanEncap(
               self.name, clientId, FlexEncapEntry(), configMode )
         if reason == VlanEncapReason.modeConflict:
            mode.addWarning( ModeConflictStr )
         elif reason != VlanEncapReason.success:
            # sanity check
            mode.addError( f"Unexpected reason: {reason}" )
      else:
         # Create an equivalent FlexEncapEntry
         outer = Tac.nonConst( FlexEncapField.createDot1qTagged() )
         outerTag = int( vlanId.id )
         outer.insertTaggedRange( FlexEncapRange( outerTag, outerTag ) )
         innerTag = int( innerVlanId.id ) if innerVlanId else None
         if innerTag:
            inner = Tac.nonConst( FlexEncapField.createDot1qTagged() )
            inner.insertTaggedRange( FlexEncapRange( innerTag, innerTag ) )
         else:
            inner = FlexEncapField.createAnyOrNone()
         client = FlexEncapSpec.createTagged( outer, inner )
         network = FlexEncapSpec.createAnyOrNone()
         status = gv.subIntfConfigDir.setVlanEncap(
               self.name, clientId, FlexEncapEntry( client, network ), configMode,
               True )
         if status.reason == VlanEncapReason.modeConflict:
            mode.addError( ModeConflictStr )
         elif status.reason == VlanEncapReason.siblingConflict:
            errStr = 'VLAN %d' % outerTag
            if innerTag:
               errStr += ' inner VLAN %d' % innerTag
            errStr += ' already found in sibling %s' % status.siblingIntfId
            mode.addError( errStr )
         elif status.reason != VlanEncapReason.success:
            # sanity check
            mode.addError( f"Unexpected reason: {status.reason}" )

   def switchportEligible( self ):
      return False

   def isSubIntf( self ):
      return True

   def parentIntf( self ):
      parentIntf = self.name.split( '.' )[ 0 ]
      return parentIntf

   #----------------------------------------------------------------------------
   # Utility functions used by show functions.
   #----------------------------------------------------------------------------
   def addrStr( self ):
      return "address is %s" % self.addr()

   def addr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def hardware( self ):
      return "subinterface"

   def bandwidth( self ):
      if toggleSubIntfSpeedEnabled():
         return self.status().speed * 1000
      if self.parentStatus() is None:
         return 0
      parentSpeed = self.parentStatus().speed
      if isinstance( parentSpeed, str ): # Parent is Ethernet
         return EthSpeedApi.speedMbps( parentSpeed ) * 1000
      elif isinstance( parentSpeed, int ): # Parent is PortChannel
         return parentSpeed * 1000
      else:
         return 0

   def linkStatus( self ):
      ( _, ret ) = self.getStatus()
      return ret

   def autonegActive( self ):
      return False

   def getIntfDuplexStr( self ):
      if self.parentStatus() and hasattr( self.parentStatus(), 'duplex' ):
         return self.parentStatus().duplex
      else:
         return "duplexFull"

   def xcvrTypeStr( self ):
      '''
      With the addition of more complex encaps, we will simply use 'subinterface' for
      the encapsulationType string, while 'dot1q' will be retained for legacy encaps.

      groups.google.com/a/arista.com/g/cli-review/c/8xuzqHnSGlg/m/FLtsKZRCBgAJ
      '''
      configMode = self.config().dot1qEncapConfigMode
      if configMode == Dot1qEncapMode.subIntfDot1qEncapConfigFlexEncap:
         return 'subinterface'
      elif configMode == Dot1qEncapMode.subIntfDot1qEncapConfigLegacy:
         return 'dot1q-encapsulation'
      else:
         return 'Unknown'

   def isSubIntfBridgedOrPseudowire( self ):
      serviceType = \
               gv.serviceIntfTypeWinner.serviceIntfType.get( self.status().intfId )
      typeIsL2OrPw = ( serviceType in [ ServiceIntfType.l2, ServiceIntfType.pw ] )
      # TODO: Remove check for forwardingModel once BUG527309 is fixed.
      return ( typeIsL2OrPw or
               self.intfStatus.forwardingModel == 'intfForwardingModelBridged' )

   def ingressCountersSupported( self ):
      return ( ( gv.bridgingHwEnabled.subintfIngressCountersEnabled and
                 not self.isSubIntfBridgedOrPseudowire() ) or
               ( gv.bridgingHwEnabled.subintfL2IngressCountersEnabled and
                 self.isSubIntfBridgedOrPseudowire() ) )

   def ingressOctetCountersSupported( self ):
      return ( self.ingressCountersSupported() and
               gv.bridgingHwCapabilities.subintfInOctetCountersSupported )

   def egressCountersSupported( self ):
      return ( ( gv.bridgingHwEnabled.subintfEgressCountersEnabled and
                 not self.isSubIntfBridgedOrPseudowire() ) or
               ( gv.bridgingHwEnabled.subintfL2EgressCountersEnabled and
                 self.isSubIntfBridgedOrPseudowire() ) )

   def egressOctetCountersSupported( self ):
      return ( self.egressCountersSupported() and
               gv.bridgingHwCapabilities.subintfOutOctetCountersSupported )

   def bumCountersSupported( self ):
      return gv.bridgingHwEnabled.subintfBumCountersEnabled

   def umCountersSupported( self ):
      return gv.bridgingHwEnabled.subintfUMCountersEnabled

   def l3CountersSupported( self ):
      return ( gv.bridgingHwEnabled.subintfL3CountersEnabled and
               not self.isSubIntfBridgedOrPseudowire() )

   def l2CountersSupported( self ):
      return ( gv.bridgingHwEnabled.subintfL2CountersEnabled and
               self.isSubIntfBridgedOrPseudowire() )

   def l3CountersRateSupported( self ):
      return ( gv.bridgingHwEnabled.subintfL3CountersRateEnabled and
               not self.isSubIntfBridgedOrPseudowire() )

   def l2CountersRateSupported( self ):
      return ( gv.bridgingHwEnabled.subintfL2CountersRateEnabled and
               self.isSubIntfBridgedOrPseudowire() )

   def countersSupported( self ):
      if self.lookupPhysical():
         return self.l2CountersSupported() or self.l3CountersSupported()
      else:
         return False

   def dpCountersSupported( self ):
      return ( gv.bridgingHwEnabled.subintfEgressDpCountersEnabled or
               gv.bridgingHwEnabled.subintfIngressDpCountersEnabled )

   def countersRateSupported( self ):
      if not self.countersSupported():
         # Rate counters can't be supported without counters being supported at all.
         return False
      if not self.lookupPhysical():
         return False
      return self.l2CountersRateSupported() or self.l3CountersRateSupported()

   def countersDiscardSupported( self ):
      return False

   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return gv.subIntfCountersDir

   def dpCounters( self ):
      return (
         gv.subIntfDpCountersDir.intfDropPrecedenceCounter.get(
            self.status().intfId ) )

   def getLatestDpCounterCheckpoint( self ):
      return ( gv.subIntfDpCountersCheckpointDir.intfDropPrecedenceCounter[
               self.status().intfId ] )

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = InterfaceCounters(
            _name=self.status().intfId,
            _ingressCounters=self.ingressCountersSupported(),
            _egressCounters=self.egressCountersSupported(),
            _umCounters=self.umCountersSupported(),
            _bumCounters=self.bumCountersSupported(),
            _l3Counters=self.l3CountersSupported() )

      def stat( attr, supported=True ):
         if not supported:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )

         return currentValue - checkpointValue

      if not zeroOut:
         intfCountersModel.inOctets = stat( 'inOctets',
               self.ingressOctetCountersSupported() )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts',
               self.ingressCountersSupported() )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts',
               self.ingressCountersSupported() and
               ( self.bumCountersSupported() or self.umCountersSupported() ) )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts',
               self.ingressCountersSupported() and self.bumCountersSupported() )

         intfCountersModel.outOctets = stat( 'outOctets',
               self.egressOctetCountersSupported() )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts',
               self.egressCountersSupported() )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts',
               self.egressCountersSupported() and
               ( self.bumCountersSupported() or self.umCountersSupported() ) )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts',
               self.egressCountersSupported() and self.bumCountersSupported() )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0

      return intfCountersModel

   def ratePct( self, bps, pps ):
      """ Uses the link bandwidth to determinte the % of theoretical
      bitrate achieved including the 12 byte inter-frame gap and the
      8 byte preable.

      Returns None of bandwidth is not set"""

      maxBw = self.bandwidth() * 1000.0
      per = None
      if maxBw != 0:
         per = ( bps + 160 * pps ) * 100 / maxBw
      return per

   def rateValue( self, attr ):
      counter = self.counter()
      currentValue = getattr( counter, attr )
      return currentValue

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None:
         return None
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = self.description()
      intfCountersRateModel.interval = int( round(
            IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) ) )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.inPktsRate = \
                     self.ratePct( intfCountersRateModel.inBpsRate,
                     intfCountersRateModel.inPpsRate )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )
      intfCountersRateModel.outPktsRate = \
                     self.ratePct( intfCountersRateModel.outBpsRate,
                     intfCountersRateModel.outPpsRate )

      return intfCountersRateModel

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      if not sessionOnly:
         # Counter agent manages the checkpoint
         x = gv.subIntfCountersCheckpointDir
         if x is None:
            return None
         y = x.counterSnapshot.get( self.name )
         return y
      else:
         return IntfCli.Intf.getCounterCheckpoint( self,
               className='Interface::IntfCounter', sessionOnly=sessionOnly )

   def clearCounters( self, sessionOnly=False ):

      if not sessionOnly:
         # We have a hook to take care of global clear
         return
      else:
         IntfCli.Intf.clearCounters( self, sessionOnly=sessionOnly )

   # This method controls whether the "vlan ..." CLI commands are available in the
   # sub-interface if-mode.
   def vlanSubIntfSupported( self ):
      return True

def subIntfRuleValueFunc( mode, intf ):
   if theSubintfSupportedGuard( mode, intf ) is not None:
      raise CliParser.InvalidInputError()
   return SubIntf( intf, mode )

EthPhyAutoIntfType.cliSubIntfClazzIs( SubIntf )
SwitchAutoIntfType.cliSubIntfClazzIs( SubIntf )

subMatcher = IntfMatcher()
subMatcher |= CliPlugin.PhysicalIntfRule.PhysicalIntfMatcher( 'Ethernet',
                                                helpdesc="Ethernet Sub Interface",
                                                subIntf=True,
                                                subIntfGuard=subintfSupportedGuard,
                                                value=subIntfRuleValueFunc )
subMatcher |= IntfStatusMatcher.DefaultIntfMatcher(
   'Switch',
   SwitchIntfCli.defaultSubMatcher,
   helpdesc='Switch Sub Interface',
   subIntf=True,
   subIntfGuard=subintfSupportedGuard,
   guard=SwitchIntfCli.switchIntfGuard,
   value=subIntfRuleValueFunc )

IntfCli.Intf.addSubIntf( subMatcher, SubIntf, withIpSupport=True )

#-------------------------------------------------------------------------------
# Adds Subinterface-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class SubIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.isSubIntf()

SubIntfModelet.addCommandClass( IntfProfileCli.ApplyIntfProfile )
SubIntfModelet.addCommandClass( IntfProfileCli.OverrideIntfProfile )

#-------------------------------------------------------------------------------
# The "encapsulation dot1q vlan <vlan-id> [inner <vlan-id>]" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
class EncapsulationDot1QVlanCmd( CliCommand.CliCommandClass ):
   syntax = "encapsulation dot1q vlan VLAN [ inner INNER_VLAN ]"
   noOrDefaultSyntax = "encapsulation dot1q vlan ..."
   data = {
      'encapsulation': 'configure encapsulation',
      'dot1q': '802.1q encapsulation',
      'vlan': 'VLAN tag',
      'VLAN': vlanIdMatcher,
      'inner': CliCommand.guardedKeyword( 'inner', helpdesc="Inner VLAN tag",
                                          guard=theQinQL3SubintfSupportedGuard ),
      'INNER_VLAN': CliMatcher.IntegerMatcher( 1, 4094, helpdesc="Inner VLAN ID" )
   }

   handler = "SubIntfCliHandler.setVlanEncap"
   noOrDefaultHandler = "SubIntfCliHandler.noVlanEncap"

SubIntfModelet.addCommandClass( EncapsulationDot1QVlanCmd )
#-------------------------------------------------------------------------------
# Adds SubIntf VLAN commands to "config-if" mode.
#-------------------------------------------------------------------------------
class SubIntfVlanModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.isSubIntf() and \
             mode.intf.vlanSubIntfSupported()

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#--------------------------------------------------------------------------------
SubIntfVlanModelet.addCommandClass( IntfCli.MacAddressRouterAddrCmd )

#-------------------------------------------------------------------------------
# The "[ no | default ] mac address learning disabled" command,
# in "config-if" mode.
#
# The full syntax of this command is:
#
#   no mac address learning disabled
#   default mac address learning disabled
#   mac address learning disabled
#-------------------------------------------------------------------------------
matcherAddressKw = CliMatcher.KeywordMatcher(
   'address', helpdesc='Address keyword' )
matcherLearningKw = CliMatcher.KeywordMatcher(
   'learning', helpdesc='Learning keyword' )
matcherDisableKw = CliMatcher.KeywordMatcher(
   'disabled', helpdesc='Disable MAC address learning' )
nodeAddress = CliCommand.Node(
   matcher=matcherAddressKw, guard=theL2SubintfMacLearningDisableSupported )

class MacAddressLearningDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mac address learning disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'address' : nodeAddress,
      'learning': matcherLearningKw,
      'disabled': matcherDisableKw
   }

   handler = "SubIntfCliHandler.disableMacLearning"
   noOrDefaultHandler = "SubIntfCliHandler.defaultMacLearning"

SubIntfVlanModelet.addCommandClass( MacAddressLearningDisabledCmd )

#-------------------------------------------------------------------------------
# The "vlan id <vlan-id>" command, in "config-if" mode.
# The "vlan name <vlan-name>" command, in "config-if" mode.
# The "no vlan id <trailing-garbage>" command, in "config-if" mode.
#-------------------------------------------------------------------------------
class VlanIdOrNameCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan ( id VLAN_ID ) | ( name VLAN_NAME )'
   noOrDefaultSyntax = 'vlan ( id | name ) ...'
   data = {
      'vlan': CliCommand.guardedKeyword( 'vlan',
                                         helpdesc='Configure VLAN attributes',
                                         guard=theL2SubintfSupportedGuard ),
      'id': 'Identifier for a Virtual LAN',
      'VLAN_ID': vlanIdMatcher,
      'name': 'Name of the VLAN',
      'VLAN_NAME': vlanNameMatcher,
   }

   handler = "SubIntfCliHandler.setL2SubIntfVlan"
   noOrDefaultHandler = handler

SubIntfVlanModelet.addCommandClass( VlanIdOrNameCmd )

#-------------------------------------------------------------------------------
# Associate the SubIntfModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( SubIntfModelet )
IntfCli.IntfConfigMode.addModelet( SubIntfVlanModelet )

class SplitHorizonGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'split-horizon group GROUP'
   noOrDefaultSyntax = 'split-horizon group ...'
   data = {
      'split-horizon': CliCommand.guardedKeyword(
         'split-horizon',
         helpdesc='Configure the split horizon group the interface belongs to',
         guard=splitHorizonGroupSupportedGuard ),
      'group': 'group',
      'GROUP': splitHorizonGroupMatcher,
   }

   handler = "SubIntfCliHandler.setL2SubIntfSplitHorizonGroup"
   noOrDefaultHandler = handler

if toggleSplitHorizonGroupEnabled():
   SubIntfVlanModelet.addCommandClass( SplitHorizonGroupCmd )

#-------------------------------------------------------------------------------
# Clear counters hook to clear subIntf counters
#-------------------------------------------------------------------------------
def clearCountersHook( mode, intfs, sessionOnly, allIntfs ):
   request = gv.aleCliConfig.clearCountersRequest

   # We don't support per-session clear
   if sessionOnly:
      return

   if allIntfs:
      del request[ 'all' ]
      request[ 'all' ] = True
      return

   # Clear interface list
   for x in intfs:
      del request[ x.name ]
      request[ x.name ] = True

#-------------------------------------------------------------------------------
# A subinterface variant to the "show tech-support" cli
#-------------------------------------------------------------------------------
nodeSubIntf = CliCommand.guardedKeyword( 'subinterface',
      helpdesc='Show subinterface specific information',
      guard=theSubintfSupportedGuard )

def showTechSupportSubIntf( mode, args ):
   vrfs = list( VrfCli.getAllVrfNames() )
   arpCmds = [ 'show ip arp vrf ' + vrf for vrf in vrfs ]
   v6NeighborCmds = [ 'show ipv6 neighbors vrf ' + vrf for vrf in vrfs ]
   platform = getPlatformName( mode.sysname )
   platformCmds = []
   if platform == 'Sand':
      platformCmds = [
                       'show platform fap subinterface mapping',
                       'show platform fap subinterface vlan',
                     ]
   if platform == 'Strata':
      platformCmds = [
                       'show platform trident l2 shadow vlan-xlate',
                       'show platform trident l2 shadow egr-vlan-xlate',
                     ]
   cmds = [
            'show running-config sanitized',
            'show vrf',
            'show interfaces',
            'show ip interface brief',
            'show ipv6 interface brief',
          ] + arpCmds + v6NeighborCmds + platformCmds

   for cmd in cmds:
      print( "\n------------- %s -------------\n" % cmd )
      sys.stdout.flush()
      try:
         mode.session_.runCmd( cmd, aaa=False )
      except CliParser.GuardError as e:
         print( "(unavailable: %s)" % e.guardCode )
      except CliParser.AlreadyHandledError:
         pass
      except KeyboardInterrupt:
         raise
      except: # pylint: disable=bare-except
         print( "(unavailable)" )
      TacSigint.check()

class showTechSupportSubIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support subinterface'

   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'subinterface': nodeSubIntf,
   }

   handler = showTechSupportSubIntf
   privileged = True

BasicCli.addShowCommandClass( showTechSupportSubIntfCmd )

def mountCounters():
   gv.subIntfStatusDir.counterDir = gv.subIntfCountersDir

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   gv.allEthIntfConfigDir = ConfigMount.mount(
      entityManager, "interface/config/eth/phy/slice", "Tac::Dir", "w" )
   gv.allIntfStatusDir = LazyMount.mount(
      entityManager, "interface/status/all", "Interface::AllIntfStatusDir", "r" )
   gv.aleCliConfig = LazyMount.mount(
      entityManager, "hardware/ale/cliconfig", "Ale::HwCliConfig", "w" )
   gv.bridgingHwCapabilities = LazyMount.mount(
      entityManager, "bridging/hwcapabilities", "Bridging::HwCapabilities", "r" )
   gv.bridgingHwEnabled = LazyMount.mount(
      entityManager, "bridging/hwenabled", "Bridging::HwEnabled", "r" )
   gv.bridgingInputConfig = ConfigMount.mount(
      entityManager, "bridging/input/config/cli",
      "Bridging::Input::CliConfig", "w" )
   gv.intfCapability = LazyMount.mount(
      entityManager, "interface/hardware/capability",
      "Interface::Hardware::Capability", "r" )
   gv.serviceIntfTypeWinner = LazyMount.mount(
      entityManager, "interface/external/serviceType/winner",
      "EbraExt::ServiceIntfTypeWinner", "r" )
   gv.subIntfConfigDir = ConfigMount.mount(
      entityManager, "interface/config/subintf", "Interface::SubIntfConfigDir", "w" )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   gv.subIntfCountersDir = shmemEm.doMount(
      'interface/counter/subintf/current',
      'Smash::Interface::AllIntfCounterDir', smi )
   gv.subIntfCountersCheckpointDir = shmemEm.doMount(
      'interface/counter/subintf/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', smi )
   gv.subIntfDpCountersDir = shmemEm.doMount(
      'interface/dpCounters/subintf/current',
      'Interface::DropPrecedenceCounter::AllIntfDropPrecedenceCounters', smi )
   gv.subIntfDpCountersCheckpointDir = shmemEm.doMount(
      'interface/dpCounters/subintf/snapshot',
      'Interface::DropPrecedenceCounter::AllIntfDropPrecedenceCounters', smi )

   mg = entityManager.mountGroup()
   gv.subIntfStatusDir = mg.mount(
      "interface/status/subintf", "Interface::SubIntfStatusDir", "r" )
   mg.close( mountCounters )
