# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import (
   absolute_import,
   division,
   print_function,
)

from BasicCli import GlobalConfigMode
import BasicCliModes
from CliCommand import (
   CliCommandClass,
   CliExpression,
   Node,
   singleKeyword,
)
from CliGlobal import CliGlobal
from CliMatcher import (
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
)
import CliParser
from CliParser import guardNotThisPlatform
import CliToken.Clear
import ConfigMount
import LazyMount
from SynceTypesFuture import Synce
import six

import CliMode.Intf
from CliMode.Synce import (
   SynceConfigMode,
   SynceIntfConfigMode,
   synceModeStr,
)
from CliPlugin.EthIntfCli import EthIntfModelet
from CliPlugin.IntfCli import (
   Intf,
   IntfDependentBase,
)

gv = CliGlobal( dict( allCliClockConfig=None, cliConfig=None, hwCapability=None ) )

def getOrCreateIntfClockConfig( intfName ):
   """
   Get, or create, an intfClockConfig given an interface name.
   """
   # BUG637586: It is possible another session has removed the config submode; if
   # so, re-create the configuration.
   if intfName not in gv.allCliClockConfig.intfClockConfig:
      gv.allCliClockConfig.intfClockConfig.newMember( intfName )
   return gv.allCliClockConfig.intfClockConfig[ intfName ]

def synceSupported():
   return gv.hwCapability.synceSupported()

def synceSupportedGuard( mode, token ):
   return None if synceSupported() else guardNotThisPlatform

nodeSynce = Node( matcher=KeywordMatcher( synceModeStr, 'Synchronous Ethernet' ),
                  guard=synceSupportedGuard )
nodeNetwork = singleKeyword( 'network', 'Network type' )
nodeOption = singleKeyword( 'option', 'Network option' )

# [no] sync-e [ instance NAME ]
class SynceCmd( CliCommandClass ):
   syntax = 'SYNCE'
   noOrDefaultSyntax = syntax
   data = {
      'SYNCE': nodeSynce,
   }

   @staticmethod
   def handler( mode, args ):
      gv.cliConfig.enable = True
      childMode = mode.childMode( SynceConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.cliConfig.enable = False
      gv.cliConfig.networkOption = Synce.NetworkOption.invalid

# add sync-e to (config)
GlobalConfigMode.addCommandClass( SynceCmd )

# sync-e [ instance NAME ]
#    [no] network option ( 1 | 2 )
optionEnumToToken = dict(
   ( option, Synce.CliHelper.networkOptionCapiValue( option ) )
   for option in Synce.NetworkOption.attributes
   if option != Synce.NetworkOption.invalid )
optionTokenToEnum = dict(
   ( token, enum ) for enum, token in six.iteritems( optionEnumToToken ) )
option1Token = optionEnumToToken[ Synce.NetworkOption.option1 ]
option2Token = optionEnumToToken[ Synce.NetworkOption.option2 ]
optionDesc = {
   option1Token: 'Network option 1 (2048 kbps)',
   option2Token: 'Network option 2 (1544 kbps)',
}
nodeOption1 = singleKeyword( option1Token, optionDesc[ option1Token ] )
nodeOption2 = singleKeyword( option2Token, optionDesc[ option2Token ] )
optionMatcher = EnumMatcher( optionDesc )

class NetworkOptionCmd( CliCommandClass ):
   syntax = 'network option OPTION'
   noOrDefaultSyntax = 'network option ...'
   data = {
      'network': nodeNetwork,
      'option': nodeOption,
      'OPTION': optionMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      option = optionTokenToEnum.get( args.get( 'OPTION' ),
                                      Synce.NetworkOption.invalid )
      gv.cliConfig.networkOption = option

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.cliConfig.networkOption = Synce.NetworkOption.invalid

SynceConfigMode.addCommandClass( NetworkOptionCmd )

class WaitToRestoreCmd( CliCommandClass ):
   syntax = 'timer wait-to-restore WTR_TIME seconds'
   noOrDefaultSyntax = 'timer wait-to-restore ...'
   data = {
      'timer':
      'Timer duration',
      'wait-to-restore':
      'Time before a previously failed clock source becomes '
      'eligible for the selection process',
      'WTR_TIME':
      IntegerMatcher( Synce.WaitToRestore.min,
                      Synce.WaitToRestore.max,
                      helpdesc='Wait-to-restore timer duration' ),
      'seconds':
      'Wait-to-restore timer in seconds',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      gv.cliConfig.waitToRestore = Synce.WaitToRestore( args[ 'WTR_TIME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.cliConfig.waitToRestore = Synce.WaitToRestore()

SynceConfigMode.addCommandClass( WaitToRestoreCmd )

# interface EthernetXX
#    sync-e [ instance NAME ]
#       [no] priority PRIORITY
priorityMatcher = IntegerMatcher(
   Synce.Priority.highest,
   Synce.Priority.lowest,
   helpdesc='Priority ({} is the highest priority)'.format(
      Synce.Priority.highest ) )

class IntfPriorityCmd( CliCommandClass ):
   syntax = 'priority ( disabled | PRIORITY )'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': 'Clock source selection priority',
      'disabled': 'Clock source will not participate in selection process',
      'PRIORITY': priorityMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intfClockConfig = getOrCreateIntfClockConfig( mode.intf )
      intfClockConfig.priority = args.get( 'PRIORITY', Synce.Priority.disabled )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfClockConfig = gv.allCliClockConfig.intfClockConfig[ mode.intf ]
      intfClockConfig.priority = Synce.Priority.defaultValue

SynceIntfConfigMode.addCommandClass( IntfPriorityCmd )

# interface EthernetXX
#    sync-e [ instance NAME ]
#       [no] disabled
class IntfDisabledCmd( CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled': 'Disable synchronization functionality' }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      intfClockConfig = getOrCreateIntfClockConfig( mode.intf )
      intfClockConfig.synchronousMode = Synce.SynchronousMode.nonsynchronous

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfClockConfig = gv.allCliClockConfig.intfClockConfig[ mode.intf ]
      intfClockConfig.synchronousMode = Synce.SynchronousMode.synchronous

SynceIntfConfigMode.addCommandClass( IntfDisabledCmd )

def qlEnumMatcher( filterFunc ):
   '''
   Returns a filtered quality-level EnumMatcher instance for quality-levels which
   satisfy filterFunc (returns True)
   '''
   return EnumMatcher(
      dict( ( qlStr, qlStr ) for qlStr in [
         Synce.QualityLevelHelper.toStr( ql ) for ql in Synce.QualityLevel.attributes
         if filterFunc( ql )
      ] ) )

option1QlMatcher = qlEnumMatcher( Synce.QualityLevelHelper.isOption1 )
option2QlMatcher = qlEnumMatcher( Synce.QualityLevelHelper.isOption2 )

class QualityLevelExpression( CliExpression ):
   '''
   CliExpression for: 'network option ( 1 OPTION1QL ) | ( 2 OPTION2QL )'
   This can be used by both the rx and tx direction
      eg. quality-level rx network option 2 QL-PRS

   Quality-level written to args[ 'QL' ]
   '''
   expression = 'network option ( option1 OPTION1QL ) | ( option2 OPTION2QL )'
   data = {
      'network': nodeNetwork,
      'option': nodeOption,
      'option1': nodeOption1,
      'OPTION1QL': option1QlMatcher,
      'option2': nodeOption2,
      'OPTION2QL': option2QlMatcher,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      qlStr = args.pop( 'OPTION1QL', None )
      if qlStr is None:
         qlStr = args.pop( 'OPTION2QL', None )
      if qlStr is not None:
         args[ 'QL' ] = Synce.QualityLevelHelper.fromStr( qlStr )

# interface EthernetXX
#    sync-e [ instance NAME ]
#       [no] quality-level rx network option ( 1 OPTION1QL ) | ( 2 OPTION2QL )
class IntfRxQualityLevelCmd( CliCommandClass ):
   syntax = 'quality-level rx QLEXPR'
   noOrDefaultSyntax = 'quality-level rx ...'
   data = {
      'quality-level': 'Quality-level override',
      'rx': 'Received quality-level override',
      'QLEXPR': QualityLevelExpression,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      intfClockConfig = getOrCreateIntfClockConfig( mode.intf )
      intfClockConfig.rxQualityLevel = args[ 'QL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfClockConfig = gv.allCliClockConfig.intfClockConfig[ mode.intf ]
      intfClockConfig.rxQualityLevel = Synce.QualityLevelHelper.defaultValue

SynceIntfConfigMode.addCommandClass( IntfRxQualityLevelCmd )

# interface EthernetXX
#    [no] sync-e [ instance NAME ]
class SynceIntfCmd( CliCommandClass ):
   syntax = 'SYNCE'
   noOrDefaultSyntax = syntax
   data = {
      'SYNCE': nodeSynce,
   }

   @staticmethod
   def handler( mode, args ):
      if not isinstance( mode.session_.mode_, CliMode.Intf.IntfMode ):
         raise CliParser.InvalidInputError( ": sync-e does not support "
                                            "interface range mode" )
      gv.allCliClockConfig.intfClockConfig.newMember( mode.intf.name )
      childMode = mode.childMode( SynceIntfConfigMode, intf=mode.intf.name )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name in gv.allCliClockConfig.intfClockConfig:
         del gv.allCliClockConfig.intfClockConfig[ mode.intf.name ]

# add sync-e to (config-if)
EthIntfModelet.addCommandClass( SynceIntfCmd )

# clear sync-e esmc counters
class ClearSynceEsmcCountersCmd( CliCommandClass ):
   syntax = 'clear SYNCE esmc counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'SYNCE': nodeSynce,
      'esmc': 'ESMC message statistics',
      'counters': 'ESMC RX and TX counters',
   }

   @staticmethod
   def handler( mode, args ):
      gv.cliConfig.esmcStatClearRequest.trigger += 1

BasicCliModes.EnableMode.addCommandClass( ClearSynceEsmcCountersCmd )

class SynceIntfCleaner( IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in gv.allCliClockConfig.intfClockConfig:
         del gv.allCliClockConfig.intfClockConfig[ self.intf_.name ]

def Plugin( entityManager ):
   gv.allCliClockConfig = ConfigMount.mount( entityManager, 'synce/config/clocks',
                                             'Synce::AllCliClockConfig', 'w' )
   gv.cliConfig = ConfigMount.mount( entityManager, 'synce/config/instance',
                                     'Synce::CliConfig', 'w' )
   gv.hwCapability = LazyMount.mount( entityManager, 'synce/hw/capability',
                                      'Synce::Hardware::Capability', 'r' )
   Intf.registerDependentClass( SynceIntfCleaner )
