# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import (
   absolute_import,
   division,
   print_function,
)

import datetime
from functools import (
   cmp_to_key,
   partial,
)

import Ark
from Ark import (
   switchTimeToUtc,
   utcToSwitchTime,
)
from Arnet import sortIntf
from ArnetModel import MacAddress
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from IntfModels import Interface
from SynceTypesFuture import Synce
from TableOutput import (
   Format,
   createTable,
)
import Tac
import six
from six.moves import map

networkOptionValues = list(
   map( Synce.CliHelper.networkOptionCapiValue, [
      option_ for option_ in Synce.NetworkOption.attributes
      if option_ != Synce.NetworkOption.invalid
   ] ) )
qlValues = list( map( Synce.QualityLevelHelper.toStr,
                      Synce.QualityLevel.attributes ) )

class ClockSource( Model ):
   ethernet = Interface( help='Ethernet interface providing this clock',
                         optional=True )

   @staticmethod
   def factory_( clockSource ):
      model = ClockSource()
      if isinstance( clockSource, Synce.ClockSource ):
         model.ethernet = clockSource.intfId
      elif isinstance( clockSource, str ):
         model.ethernet = clockSource
      return model

   def str( self ):
      if self.ethernet is not None:
         # pylint: disable=no-member
         return self.ethernet.stringValue
      return ''

class Priority( Model ):
   disabled = Bool( help='Clock source will not participate in selection process',
                    optional=True )
   value = Int( help='Clock source selection priority', optional=True )

   @staticmethod
   def factory_( priority ):
      if isinstance( priority, int ):
         # Python may convert priority into builtin int
         priority = Synce.Priority( priority )
      if priority.isDisabled():
         return Priority( disabled=True )
      else:
         return Priority( value=priority.value )

   def str( self ):
      assert ( self.disabled is None ) != ( self.value is None )
      if self.disabled:
         return 'dis'
      else:
         return '{}'.format( self.value )

class SynceStatusInfo( Model ):
   networkOption = Enum( values=networkOptionValues,
                         help='Network option',
                         optional=True )
   # Quality level is optional because we do not advertise quality level if
   # qlEnabled == False
   qualityLevelEnabled = Bool( help='Quality level processing enabled' )
   holdoffDuration = Float( help='Duration of holdoff timer in seconds' )
   waitToRestoreDuration = Float(
      help='Duration of wait-to-restore timer in seconds' )
   selectedQualityLevel = Enum( values=qlValues,
                                help='Quality level of the selected clock source',
                                optional=True )
   selectedClock = Submodel( ClockSource,
                             help='Selected clock source to which the deviced '
                             'is synchronized',
                             optional=True )
   operState = Enum( values=Synce.HwState.attributes,
                     help='Operational state of the device' )
   freeRunningAndHoldoverQualityLevel = Enum(
      values=qlValues,
      help='Quality level of the device when '
      'operating in either free-running or holdover state',
      optional=True )
   clockIdentity = Str(
      help='Synce clockIdentity of the originator of the extended QlTlv',
      optional=True )
   cascadedEeecCount = Int( help='Number of cascaded eEECs', optional=True )
   cascadedEecCount = Int( help='Number of cascaded EECs', optional=True )
   etlvFlag = Int( help='EEC/eEEC flag', optional=True )

   @staticmethod
   def factory_( qlEnabled,
                 operState,
                 holdoffDuration,
                 waitToRestoreDuration,
                 networkOption=None,
                 clock=None,
                 ql=None,
                 localQl=None,
                 clockIdentity=None,
                 cascadedEeecCount=None,
                 cascadedEecCount=None,
                 etlvFlag=None ):
      model = SynceStatusInfo( qualityLevelEnabled=qlEnabled,
                               operState=operState,
                               holdoffDuration=holdoffDuration,
                               waitToRestoreDuration=waitToRestoreDuration )
      model.qualityLevelEnabled = qlEnabled
      model.operState = operState
      if clock is not None:
         model.selectedClock = ClockSource.factory_( clock )
      if networkOption is not None:
         model.networkOption = \
            Synce.CliHelper.networkOptionCapiValue( networkOption )
      if ql is not None:
         model.selectedQualityLevel = Synce.QualityLevelHelper.toStr( ql )
      if localQl is not None:
         model.freeRunningAndHoldoverQualityLevel = \
            Synce.QualityLevelHelper.toStr( localQl )
      if clockIdentity is not None:
         model.clockIdentity = clockIdentity.stringValue
      model.cascadedEeecCount = cascadedEeecCount
      model.cascadedEecCount = cascadedEecCount
      if etlvFlag is not None:
         model.etlvFlag = etlvFlag
      return model

   # pylint: disable=no-member
   def str( self ):
      networkOptionStr = self.networkOption or 'unconfigured'
      qlProcStr = 'enabled' if self.qualityLevelEnabled else 'disabled'
      selectedClockStr = 'none'
      if self.selectedClock:
         selectedClockStr = self.selectedClock.str()
      operStateStr = Synce.CliHelper.prettyOperState( self.operState )
      maybeQualityLevelStr = ''
      if self.qualityLevelEnabled and self.selectedQualityLevel:
         maybeQualityLevelStr = 'Quality level: {}\n'.format(
            self.selectedQualityLevel )
      maybeLocalQualityLevelStr = ''
      if self.qualityLevelEnabled and self.freeRunningAndHoldoverQualityLevel:
         maybeLocalQualityLevelStr = \
            '\nFree-running and holdover quality level: {}'.format(
               self.freeRunningAndHoldoverQualityLevel )
      extendedQualityLevelStr = 'Extended QL TLV received: false'
      if self.clockIdentity is not None:
         extendedQualityLevelStr = '''Clock identity: {clockIdentity}
Cascaded eEEC count: {cascadedEeecCount}
Cascaded EEC count: {cascadedEecCount}
EEC/eEEC flag: {flag}
Partial chain bit: {chainFlag}
Mixed EEC/eEEC bit: {mixedFlag}'''.format(
            clockIdentity=self.clockIdentity,
            cascadedEeecCount=self.cascadedEeecCount,
            cascadedEecCount=self.cascadedEecCount,
            flag=format( self.etlvFlag, '#010b' ),
            mixedFlag=str( bool( self.etlvFlag & 1 ) ).lower(),
            chainFlag=str( bool( self.etlvFlag & 2 ) ).lower() )
      formatStr = '''Operational state: {operState}
Clock source: {selectedClock}
{extendedQualityLevel}
Quality level processing: {qlProcessing}
{maybeQualityLevel}Network option: {networkOption}
Holdoff duration: {holdoffDuration} milliseconds
Wait-to-restore duration: {waitToRestoreDuration} seconds'''
      formatStr += '{maybeLocalQualityLevel}'
      return formatStr.format(
         networkOption=networkOptionStr,
         qlProcessing=qlProcStr,
         selectedClock=selectedClockStr,
         maybeQualityLevel=maybeQualityLevelStr,
         operState=operStateStr,
         maybeLocalQualityLevel=maybeLocalQualityLevelStr,
         # convert back to milliseconds
         holdoffDuration=int( self.holdoffDuration * 1000 ),
         waitToRestoreDuration=int( self.waitToRestoreDuration ),
         extendedQualityLevel=extendedQualityLevelStr )

# show sync-e
class SynceStatus( Model ):
   info = Submodel( SynceStatusInfo,
                    help='Synchronous Ethernet status',
                    optional=True )

   def render( self ):
      # We do not render state
      if self.info is not None:
         # pylint: disable=no-member
         print( self.info.str() )

class SynceSelectionStatus( Model ):
   clockSource = Submodel( ClockSource, help='Candidate clock source' )
   status = Str( help='Eligibility of clock source' )
   # Quality level is optional because we do not advertise quality level if
   # qlEnabled == False
   qualityLevel = Enum( values=qlValues, help='Quality level', optional=True )
   priority = Submodel( Priority, help='Selection priority' )
   holdoffTimeout = Float( help='UTC timestamp for the hold-off timer expiry',
                           optional=True )
   waitToRestoreTimeout = Float( help='UTC timestamp for the wait-to-restore '
                                 'timer expiry',
                                 optional=True )

   @staticmethod
   def factory_( clock,
                 status,
                 priority,
                 ql=None,
                 waitToRestore=None,
                 holdoff=None ):
      model = SynceSelectionStatus( clockSource=ClockSource.factory_( clock ),
                                    status=status,
                                    priority=Priority.factory_( priority ) )
      if ql is not None:
         model.qualityLevel = Synce.QualityLevelHelper.toStr( ql )
      if waitToRestore is not None:
         model.waitToRestoreTimeout = switchTimeToUtc( waitToRestore )
      if holdoff is not None:
         model.holdoffTimeout = switchTimeToUtc( holdoff )
      return model

def timestampToStr( timestamp, showMs=False ):
   """
   This helper routine functions similarly to Ark.timestampToStr,
   and in fact calls it directly when showMs == False.

   However, in some cases the milliseconds of the timestamp are
   required, and so this routine can be used with showMs == True
   to achieve this.
   """
   if not showMs or not timestamp:
      return Ark.timestampToStr( timestamp, relative=False, now=None )
   td = datetime.datetime.fromtimestamp( Tac.utcNow() - ( Tac.now() - timestamp ) )
   tdStr = td.strftime( '%Y-%m-%d %H:%M:%S.%f' )
   # Use [ :-3 ] to trim microseconds to 3 digits; strftime does not support
   # .%3f format specifier.
   #
   # https://stackoverflow.com/questions/11040177/datetime-round-trim-number-of-digits-in-microseconds
   return tdStr[ :-3 ]

# show sync-e selection
class SynceSelectionStatuses( Model ):
   clockSources = List( valueType=SynceSelectionStatus,
                        help='List of candidate clock sources participating '
                        'in the selection process' )

   def render( self ):
      if not self.clockSources:
         return
      headings = (
         'Input',
         'QL',
         'Priority',
         'Status',
      )
      fmtLeft = Format( justify='left', maxWidth=50, wrap=True )
      fmtLeft.noPadLeftIs( True )
      fmtLeft.padLimitIs( True )
      fmtRight = Format( justify='right' )
      fmtRight.padLimitIs( True )
      table = createTable( headings )
      table.formatColumns( fmtLeft, fmtLeft, fmtRight, fmtLeft )
      for status in self.clockSources:
         inputStr = status.clockSource.str()
         prioStr = status.priority.str()
         if status.holdoffTimeout:
            statusStr = Synce.CliHelper.prettySelectionStatusWithHoldoff(
               status.status,
               timestampToStr( utcToSwitchTime( status.holdoffTimeout ),
                               showMs=True ) )
         elif status.waitToRestoreTimeout:
            statusStr = Synce.CliHelper.prettySelectionStatusWithWaitToRestore(
               status.status,
               timestampToStr( utcToSwitchTime( status.waitToRestoreTimeout ) ) )
         else:
            statusStr = Synce.CliHelper.prettySelectionStatus( status.status )
         qlStr = status.qualityLevel or ''
         table.newRow( inputStr, qlStr, prioStr, statusStr )
      print( table.output() )

class SynceEsmcCountersRxInfo( Model ):
   validCount = Int( help='Number of valid ESMC messages received' )
   invalidCount = Int( help='Number of invalid ESMC messages received' )
   lastQualityLevel = Enum( values=qlValues,
                            help='Last received quality level',
                            optional=True )
   lastSsm = Int( help='Last received SSM code', optional=True )
   lastEssm = Int( help='Last received ESSM code', optional=True )
   lastCascadedEeecCount = Int( help='Last received number of cascaded eEECs',
                                optional=True )
   lastCascadedEecCount = Int( help='Last received number of cascaded EECs',
                               optional=True )

class SynceEsmcCountersTxInfo( Model ):
   sentCount = Int( help='Number of ESMC messages sent' )
   lastQualityLevel = Enum( values=qlValues,
                            help='Last sent quality level',
                            optional=True )
   lastSsm = Int( help='Last sent SSM code', optional=True )
   lastEssm = Int( help='Last sent ESSM code', optional=True )
   lastCascadedEeecCount = Int( help='Last sent number of cascaded eEECs',
                                optional=True )
   lastCascadedEecCount = Int( help='Last sent number of cascaded EECs',
                               optional=True )

class SynceEsmcCountersInfo( Model ):
   rxInfo = Submodel( SynceEsmcCountersRxInfo,
                      help='RX ESMC counters information',
                      optional=True )
   txInfo = Submodel( SynceEsmcCountersTxInfo,
                      help='TX ESMC counters information',
                      optional=True )

def maybeNaStr( strFunc, cond ):
   ''' Used for rendering optional fields in ESMC show commands '''
   return strFunc() if cond else 'n/a'

def directionStr( direction ):
   assert direction in { 'rx', 'tx' }
   return 'sent' if direction == 'tx' else 'received'

def esmcTypeCountStr( esmcTypeCount, direction ):
   ''' Used for rendering ESMC messages by type '''
   dirStr = directionStr( direction )
   countByEsmcType = '\n'.join( [
      '{}: {}'.format( esmcType, esmcTypeCount[ esmcType ] )
      for esmcType in sorted( esmcTypeCount )
   ] )
   return '\nESMC messages {} by type\n{}'.format(
      dirStr, countByEsmcType ) if countByEsmcType else ''

def qlCountStr( qlCount, direction ):
   ''' Used for rendering ESMC message's QL by count '''
   dirStr = directionStr( direction )
   countByQl = '\n'.join( [
      qlCount[ _ql ].str( _ql )
      for _ql in sorted( qlCount, key=cmp_to_key( qlSort ) )
   ] )
   return '\nESMC messages {} by QL (SSM code)\n{}'.format(
      dirStr, countByQl ) if countByQl else ''

class SynceEsmcCounters( Model ):
   clockSourceCounters = Dict( keyType=Interface,
                               valueType=SynceEsmcCountersInfo,
                               help='Clock source RX and TX ESMC counters' )

   def render( self ):
      fmtLeft = Format( justify='left' )
      fmtLeft.noPadLeftIs( True )
      fmtLeft.padLimitIs( True )
      fmtRight = Format( justify='right' )
      fmtRight.padLimitIs( True )
      rxHeadings = ( 'Interface', 'Valid Received', 'Invalid Received', 'QL', 'SSM',
                     'eSSM', 'eEECs', 'EECs' )
      rxTable = createTable( rxHeadings )
      rxTable.formatColumns( fmtLeft, fmtRight, fmtRight, fmtLeft, fmtRight,
                             fmtRight, fmtRight, fmtRight )
      txHeadings = ( 'Interface', 'Sent', 'QL', 'SSM', 'eSSM', 'eEECs', 'EECs' )
      txTable = createTable( txHeadings )
      txTable.formatColumns( fmtLeft, fmtRight, fmtLeft, fmtRight, fmtRight,
                             fmtRight, fmtRight )
      printRx = False
      printTx = False
      for intf in sortIntf( self.clockSourceCounters ):
         countersInfo = self.clockSourceCounters[ intf ]
         rxCounters = countersInfo.rxInfo
         txCounters = countersInfo.txInfo
         if rxCounters is not None:
            printRx = True
            # We cannot use lambda because of python's lambda scoping within
            # a for-loop. Use partial instead
            qlStr = maybeNaStr( partial( getattr, rxCounters, 'lastQualityLevel' ),
                                rxCounters.lastQualityLevel is not None )
            ssmStr = format( rxCounters.lastSsm,
                             '#006b' ) if rxCounters.lastSsm is not None else 'n/a'
            essmStr = format( rxCounters.lastEssm,
                              '#004x' ) if rxCounters.lastEssm is not None else 'n/a'
            cascadedEeecCountStr = maybeNaStr(
               partial( getattr, rxCounters, 'lastCascadedEeecCount' ),
               rxCounters.lastCascadedEeecCount is not None )
            cascadedEecCountStr = maybeNaStr(
               partial( getattr, rxCounters, 'lastCascadedEecCount' ),
               rxCounters.lastCascadedEecCount is not None )
            rxTable.newRow( intf, rxCounters.validCount, rxCounters.invalidCount,
                            qlStr, ssmStr, essmStr, cascadedEeecCountStr,
                            cascadedEecCountStr )
         if txCounters is not None:
            printTx = True
            # We cannot use lambda because of python's lambda scoping within
            # a for-loop. Use partial instead
            qlStr = maybeNaStr( partial( getattr, txCounters, 'lastQualityLevel' ),
                                txCounters.lastQualityLevel is not None )
            ssmStr = format( txCounters.lastSsm,
                             '#006b' ) if txCounters.lastSsm is not None else 'n/a'
            essmStr = format( txCounters.lastEssm,
                              '#004x' ) if txCounters.lastEssm is not None else 'n/a'
            cascadedEeecCountStr = maybeNaStr(
               partial( getattr, txCounters, 'lastCascadedEeecCount' ),
               txCounters.lastCascadedEeecCount is not None )
            cascadedEecCountStr = maybeNaStr(
               partial( getattr, txCounters, 'lastCascadedEecCount' ),
               txCounters.lastCascadedEecCount is not None )
            txTable.newRow( intf, txCounters.sentCount, qlStr, ssmStr, essmStr,
                            cascadedEeecCountStr, cascadedEecCountStr )
      if printRx:
         print( rxTable.output() )
      if printTx:
         print( txTable.output() )

def ssmSortCmp( lhs, rhs ):
   isEnhanced = lambda ssm: '/' in ssm
   if lhs == rhs:
      return 0
   elif isEnhanced( lhs ):
      return -1
   else:
      return 1

class QualityLevelCount( Model ):
   countBySsm = Dict( keyType=str, valueType=int, help='ESMC messages by SSM code' )

   def str( self, ql ):
      output = []
      for ssm in sorted( self.countBySsm, key=cmp_to_key( ssmSortCmp ) ):
         output.append( '{} ({}): {}'.format( ql, ssm, self.countBySsm[ ssm ] ) )
      return '\n'.join( output )

def qlSort( lhs, rhs ):
   lhs = Synce.QualityLevelHelper.fromStr( lhs )
   rhs = Synce.QualityLevelHelper.fromStr( rhs )
   if lhs == rhs:
      return 0
   elif Synce.QualityLevelHelper.preferred( lhs, rhs ):
      return -1
   else:
      return 1

class SynceEsmcDetailRxInfo( Model ):
   sourceMac = MacAddress( help='Source MAC address of ethernet clock source',
                           optional=True )
   validCount = Int( help='Number of valid ESMC messages received' )
   invalidCount = Int( help='Number of invalid ESMC messages received' )
   lastQualityLevel = Enum( values=qlValues,
                            help='Last received quality level',
                            optional=True )
   lastSsm = Int( help='Last received SSM code', optional=True )
   lastEssm = Int( help='Last received ESSM code', optional=True )
   lastClockIdentity = Str(
      help=
      'Last received Synce clockIdentity of the originator of the extended QlTlv',
      optional=True )
   lastCascadedEeecCount = Int( help='Last received number of cascaded eEECs',
                                optional=True )
   lastCascadedEecCount = Int( help='Last received number of cascaded EECs',
                               optional=True )
   lastEtlvFlag = Int( help='Last received EEC/eEEC flag', optional=True )
   lastValidTime = Float( help='UTC timestamp of the last received valid '
                          'ESMC message',
                          optional=True )
   lastInvalidTime = Float( help='UTC timestamp of the last received invalid '
                            'ESMC message',
                            optional=True )
   # BUG595834 - expirationTime
   lastEsmcType = Enum( values=Synce.EventFlag.attributes,
                        help='Last received ESMC message type',
                        optional=True )
   esmcTypeCount = Dict( keyType=str,
                         valueType=int,
                         help='ESMC messages received by type' )
   qualityLevelCount = Dict( keyType=str,
                             valueType=QualityLevelCount,
                             help='ESMC messages received by quality level' )

   @staticmethod
   def factory_( option,
                 validCount,
                 invalidCount,
                 sourceMac=None,
                 validRxTime=None,
                 invalidRxTime=None,
                 eventFlag=None,
                 ql=None,
                 ssmTuple=None,
                 eventFlagCountDict=None,
                 ssmTupleFromEsmcCountDict=None,
                 clockIdentity=None,
                 cascadedEeecCount=None,
                 cascadedEecCount=None,
                 etlvFlag=None ):
      model = SynceEsmcDetailRxInfo()
      eventFlagCountDict = eventFlagCountDict or {}
      ssmTupleFromEsmcCountDict = ssmTupleFromEsmcCountDict or {}
      model.sourceMac = sourceMac
      model.validCount = validCount
      model.invalidCount = invalidCount
      model.lastQualityLevel = Synce.QualityLevelHelper.toStr( ql ) if ql else None
      model.lastEsmcType = eventFlag
      model.lastSsm = ssmTuple.ssm if ssmTuple else None
      model.lastEssm = ssmTuple.essm if ssmTuple else None
      model.lastClockIdentity = clockIdentity.stringValue if clockIdentity else None
      model.lastCascadedEeecCount = cascadedEeecCount
      model.lastCascadedEecCount = cascadedEecCount
      model.lastEtlvFlag = etlvFlag
      model.lastValidTime = switchTimeToUtc( validRxTime ) if validRxTime else None
      model.lastInvalidTime = \
         switchTimeToUtc( invalidRxTime ) if invalidRxTime else None
      for flag, count in six.iteritems( eventFlagCountDict ):
         model.esmcTypeCount[ Synce.CliHelper.prettyEventFlag( flag ) ] = count
      qualityLevelCount = {}
      for ssmTupleFromEsmc, count in six.iteritems( ssmTupleFromEsmcCountDict ):
         ssmTuple = ssmTupleFromEsmc.ssmTuple()
         ql = Synce.QualityLevelHelper.fromSsm( ssmTuple, option )
         qlStr = Synce.QualityLevelHelper.toStr( ql )
         qualityLevelCount.setdefault( qlStr,
                                       {} )[ ssmTupleFromEsmc.stringValue() ] = count
      for qlStr, countBySsm in six.iteritems( qualityLevelCount ):
         model.qualityLevelCount[ qlStr ] = QualityLevelCount(
            countBySsm=countBySsm )
      return model

   def str( self ):
      # pylint: disable=no-member
      smacStr = maybeNaStr( lambda: self.sourceMac.displayString, self.sourceMac
                            is not None )

      # pylint: enable=no-member
      def maybeTimeStr( time ):
         return timestampToStr( utcToSwitchTime( time ) if time is not None else 0 )

      validTimeStr = maybeTimeStr( self.lastValidTime )
      maybeExtendedQualityLevelStr = 'Last received extended QL TLV: not received'
      if self.lastClockIdentity is not None:
         maybeExtendedQualityLevelStr = \
'''Last received clock identity: {lastClockIdentity}
Last received cascaded eEEC count: {lastCascadedEeecCount}
Last received cascaded EEC count: {lastCascadedEecCount}
Last received EEC/eEEC flag: {lastEtlvFlag}
Last received partial chain bit: {lastChainFlag}
Last received mixed EEC/eEEC bit: {lastMixedFlag}'''.format(
            lastClockIdentity=self.lastClockIdentity,
            lastCascadedEeecCount=self.lastCascadedEeecCount,
            lastCascadedEecCount=self.lastCascadedEecCount,
            lastEtlvFlag=format( self.lastEtlvFlag, '#010b' ),
            lastMixedFlag=str( bool( self.lastEtlvFlag & 1 ) ).lower(),
            lastChainFlag=str( bool( self.lastEtlvFlag & 2 ) ).lower() )
      invalidTimeStr = maybeTimeStr( self.lastInvalidTime )
      qlStr = maybeNaStr( lambda: self.lastQualityLevel, self.lastQualityLevel
                          is not None )
      ssmStr = 'n/a'
      if self.lastSsm is not None:
         lastEssm = self.lastEssm if self.lastEssm is not None else 0xff
         essmSet = True if self.lastEssm is not None else False
         ssmStr = Tac.Value( 'Synce::SsmTupleFromEsmc', self.lastSsm, lastEssm,
                             essmSet ).stringValue()
      esmcTypeStr = maybeNaStr(
         lambda: Synce.CliHelper.prettyEventFlag( self.lastEsmcType ),
         self.lastEsmcType is not None )
      esmcByTypeStr = esmcTypeCountStr( self.esmcTypeCount, 'rx' )
      qlByTypeStr = qlCountStr( self.qualityLevelCount, 'rx' )
      return '''Last received quality level: {ql}, SSM: {ssm}, Type: {esmcType}
Last received source MAC: {smac}
Last received valid message time: {lastValidRx}
{maybeExtendedQualityLevel}
Last received invalid message time: {lastInvalidRx}
Received valid message count: {validCount}
Received invalid message count: {invalidCount}{esmcByType}{qlByType}
'''.format( ql=qlStr,
            ssm=ssmStr,
            esmcType=esmcTypeStr,
            smac=smacStr,
            lastValidRx=validTimeStr,
            maybeExtendedQualityLevel=maybeExtendedQualityLevelStr,
            lastInvalidRx=invalidTimeStr,
            validCount=self.validCount,
            invalidCount=self.invalidCount,
            esmcByType=esmcByTypeStr,
            qlByType=qlByTypeStr )

class SynceEsmcDetailTxInfo( Model ):
   sourceMac = MacAddress( help='Source MAC address of ethernet clock source',
                           optional=True )
   sentCount = Int( help='Number of ESMC messages sent' )
   lastQualityLevel = Enum( values=qlValues,
                            help='Last sent quality level',
                            optional=True )
   lastSsm = Int( help='Last sent SSM code', optional=True )
   lastEssm = Int( help='Last sent ESSM code', optional=True )
   lastClockIdentity = Str(
      help='Last sent Synce clockIdentity of the originator of the extended QlTlv',
      optional=True )
   lastCascadedEeecCount = Int( help='Last sent number of cascaded eEECs',
                                optional=True )
   lastCascadedEecCount = Int( help='Last sent number of cascaded EECs',
                               optional=True )
   lastEtlvFlag = Int( help='Last sent EEC/eEEC flag', optional=True )
   lastSentTime = Float( help='UTC timestamp of the last sent ESMC message',
                         optional=True )
   lastEsmcType = Enum( values=Synce.EventFlag.attributes,
                        help='Last sent ESMC message type',
                        optional=True )
   esmcTypeCount = Dict( keyType=str,
                         valueType=int,
                         help='ESMC messages sent by type' )
   qualityLevelCount = Dict( keyType=str,
                             valueType=QualityLevelCount,
                             help='ESMC messages sent by quality level' )

   @staticmethod
   def factory_( option,
                 txCount,
                 sourceMac=None,
                 txTime=None,
                 eventFlag=None,
                 ql=None,
                 ssmTuple=None,
                 eventFlagCountDict=None,
                 ssmTupleFromEsmcCountDict=None,
                 clockIdentity=None,
                 cascadedEeecCount=None,
                 cascadedEecCount=None,
                 etlvFlag=None ):
      model = SynceEsmcDetailTxInfo()
      eventFlagCountDict = eventFlagCountDict or {}
      ssmTupleFromEsmcCountDict = ssmTupleFromEsmcCountDict or {}
      model.sourceMac = sourceMac
      model.sentCount = txCount
      model.lastQualityLevel = Synce.QualityLevelHelper.toStr( ql ) if ql else None
      model.lastEsmcType = eventFlag
      model.lastSsm = ssmTuple.ssm if ssmTuple else None
      model.lastEssm = ssmTuple.essm if ssmTuple else None
      model.lastClockIdentity = clockIdentity.stringValue if clockIdentity else None
      model.lastCascadedEeecCount = cascadedEeecCount
      model.lastCascadedEecCount = cascadedEecCount
      model.lastEtlvFlag = etlvFlag
      model.lastSentTime = switchTimeToUtc( txTime ) if txTime else None
      for flag, count in six.iteritems( eventFlagCountDict ):
         model.esmcTypeCount[ Synce.CliHelper.prettyEventFlag( flag ) ] = count
      qualityLevelCount = {}
      for ssmTupleFromEsmc, count in six.iteritems( ssmTupleFromEsmcCountDict ):
         ssmTuple = ssmTupleFromEsmc.ssmTuple()
         ql = Synce.QualityLevelHelper.fromSsm( ssmTuple, option )
         qlStr = Synce.QualityLevelHelper.toStr( ql )
         qualityLevelCount.setdefault( qlStr,
                                       {} )[ ssmTupleFromEsmc.stringValue() ] = count
      for qlStr, countBySsm in six.iteritems( qualityLevelCount ):
         model.qualityLevelCount[ qlStr ] = QualityLevelCount(
            countBySsm=countBySsm )
      return model

   def str( self ):
      # pylint: disable=no-member
      smacStr = maybeNaStr( lambda: self.sourceMac.displayString, self.sourceMac
                            is not None )
      # pylint: enable=no-member
      timeStr = timestampToStr(
         utcToSwitchTime( self.lastSentTime ) if self.lastSentTime is not None else 0
      )
      qlStr = maybeNaStr( lambda: self.lastQualityLevel, self.lastQualityLevel
                          is not None )
      ssmStr = 'n/a'
      if self.lastSsm is not None:
         lastEssm = self.lastEssm if self.lastEssm is not None else 0xff
         essmSet = True if self.lastEssm is not None else False
         ssmStr = Tac.Value( 'Synce::SsmTupleFromEsmc', self.lastSsm, lastEssm,
                             essmSet ).stringValue()
      maybeExtendedQualityLevelStr = ''
      if self.lastClockIdentity is not None:
         maybeExtendedQualityLevelStr = \
'''\nLast sent clock identity: {lastClockIdentity}
Last sent cascaded eEEC count: {lastCascadedEeecCount}
Last sent cascaded EEC count: {lastCascadedEecCount}
Last sent EEC/eEEC flag: {lastEtlvFlag}
Last sent partial chain bit: {lastChainFlag}
Last sent mixed EEC/eEEC bit: {lastMixedFlag}'''.format(
            lastClockIdentity=self.lastClockIdentity,
            lastCascadedEeecCount=self.lastCascadedEeecCount,
            lastCascadedEecCount=self.lastCascadedEecCount,
            lastEtlvFlag=format( self.lastEtlvFlag, '#010b' ),
            lastMixedFlag=str( bool( self.lastEtlvFlag & 1 ) ).lower(),
            lastChainFlag=str( bool( self.lastEtlvFlag & 2 ) ).lower() )
      esmcTypeStr = maybeNaStr(
         lambda: Synce.CliHelper.prettyEventFlag( self.lastEsmcType ),
         self.lastEsmcType is not None )
      esmcByTypeStr = esmcTypeCountStr( self.esmcTypeCount, 'tx' )
      qlByTypeStr = qlCountStr( self.qualityLevelCount, 'tx' )
      return '''Last sent quality level: {ql}, SSM: {ssm}, Type: {esmcType}
Last sent source MAC: {smac}{maybeExtendedQualityLevel}
Last sent message time: {lastTx}
Sent message count: {txCount}{esmcByType}{qlByType}
'''.format( ql=qlStr,
            ssm=ssmStr,
            esmcType=esmcTypeStr,
            smac=smacStr,
            maybeExtendedQualityLevel=maybeExtendedQualityLevelStr,
            lastTx=timeStr,
            txCount=self.sentCount,
            esmcByType=esmcByTypeStr,
            qlByType=qlByTypeStr )

class SynceEsmcDetailInfo( Model ):
   rxInfo = Submodel( SynceEsmcDetailRxInfo,
                      help='Detailed RX ESMC information',
                      optional=True )
   txInfo = Submodel( SynceEsmcDetailTxInfo,
                      help='Detailed TX ESMC information',
                      optional=True )

   # pylint: disable=no-member
   def str( self ):
      if self.rxInfo is not None and self.txInfo is not None:
         return '{}{}'.format( self.rxInfo.str(), self.txInfo.str() )
      elif self.rxInfo is not None:
         return self.rxInfo.str()
      elif self.txInfo is not None:
         return self.txInfo.str()
      return ''

class SynceEsmcDetail( Model ):
   clockSourceDetail = Dict( keyType=Interface,
                             valueType=SynceEsmcDetailInfo,
                             help='Detailed Clock source RX and TX ESMC '
                             'information' )

   def render( self ):
      for intf in sortIntf( self.clockSourceDetail or {} ):
         info = self.clockSourceDetail[ intf ]
         print( "Interface {}".format( intf ) )
         print( info.str() )
