# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the CLI config mode for specfying system wide L1 parameters.
'''

from ArPyUtils.Types import ArException
from CliMode.SystemL1 import SystemL1Mode
import BasicCli
import CliCommand
import CliToken.L1
import CliToken.System

cleanupHooks_ = {}

def registerCleanupHook( feature, hook ):
   '''Register a callback to cleanup state relating to commands under the system l1
   configuration mode.

   These hooks will be called after a "no|default system l1" command is issued.

   Args:
      feature ( str ): The feature name registering the cleanup hook.
      hook ( fn ): The cleanup callback to register. The function must be callable
                   without any arguments
   '''
   if feature in cleanupHooks_:
      raise ArException(
         f'Attempting to double register cleanup hook for: {feature}' )

   cleanupHooks_[ feature ] = hook

class SystemL1ConfigMode( SystemL1Mode, BasicCli.ConfigModeBase ):
   name = 'System-wide L1 parameter configuration'

   def __init__( self, parent, session ):
      SystemL1Mode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def abort( self ):
      self.session_.gotoParentMode()

class EnterSystemL1ConfigModeCommand( CliCommand.CliCommandClass ):
   syntax = '''system l1'''
   noOrDefaultSyntax = syntax
   data = { 'system': CliToken.System.systemMatcherForConfig,
            'l1': CliToken.L1.configKeywordL1 }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SystemL1ConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Execute all the L1 system-wide feature cleanup hooks.
      for cleanupHook in cleanupHooks_.values():
         cleanupHook()

BasicCli.GlobalConfigMode.addCommandClass( EnterSystemL1ConfigModeCommand )
