# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements TACACS+ configuration.
#
# In enable mode:
#
#     show tacacs
#     clear aaa counters tacacs
#
# In config mode:
#
#     [no] tacacs-server key [0] <key-text>
#     [no] tacacs-server timeout <1-1000>
#     [no] tacacs-server policy unknown-mandatory-attribute ignore
#     [no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#           [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]
#     [no] tacacs-server qos dscp <0-63>
#     [no] tacacs-server username max-length <1-255>
#
# Child mode of config mode:
#
#     aaa group server tacacs+ <server-group-name>
#        [no] server <ip-addr-or-hostname> [port <1-65535>]
#
#
# In global IP config mode:
#     [no] ip tacacs source-interace <interface-name>
#-------------------------------------------------------------------------------
import CliPlugin.TechSupportCli
from CliPlugin import AaaCli
from CliPlugin import IntfCli
from CliPlugin.VrfCli import VrfExprFactory
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliToken.Clear
import CliToken.Ip
import DscpCliLib
import HostnameCli
import ReversibleSecretCli
import ShowCommand
import Tac
import Tacacs
import TacacsGroup

# Need to define this here because I can't access these .tac definitions
# from Python due to lack of constAttr support
_timeoutMin = 1
_timeoutMax = 1000

tacacsConfig = None

class ShowTacacsCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show tacacs"
   data = { "tacacs" : 'TACACS+ server attributes' }
   cliModel = "TacacsModel.ShowTacacs"
   handler = "TacacsHandler.showTacacs"

BasicCli.addShowCommandClass( ShowTacacsCommand )

#-------------------------------------------------------------------------------
# "clear aaa counters tacacs" in enable mode
#-------------------------------------------------------------------------------
class ClearTacacsCounterCommand( CliCommand.CliCommandClass ):
   syntax = "clear aaa counters tacacs"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'aaa' : AaaCli.aaaAfterClearMatcher,
      'counters' : AaaCli.aaaCounterMatcher,
      'tacacs' : "Clear TACACS counters"
      }
   handler = "TacacsHandler.clearCounters"

BasicCli.EnableMode.addCommandClass( ClearTacacsCounterCommand )

#-------------------------------------------------------------------------------
# config mode commands
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

tacacsServerKwMatcher = CliMatcher.KeywordMatcher(
   'tacacs-server',
   helpdesc='Modify TACACS+ parameters' )

keyExpression = ReversibleSecretCli.defaultReversiblePwdCliExpr

#-------------------------------------------------------------------------------
# "[no] tacacs-server key <KEY>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerKeyCommand( CliCommand.CliCommandClass ):
   syntax = 'tacacs-server key <KEY>'
   noOrDefaultSyntax = 'tacacs-server key ...'
   data = { 'tacacs-server' : tacacsServerKwMatcher,
            'key' : 'Set TACACS+ secret key',
            '<KEY>' : keyExpression }

   @staticmethod
   def handler( mode, args ):
      tacacsConfig.key = args[ '<KEY>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tacacsConfig.key = ReversibleSecretCli.getDefaultSecret()

configMode.addCommandClass( TacacsServerKeyCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server timeout <1-1000>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server timeout <TIMEOUT>"
   noOrDefaultSyntax = "tacacs-server timeout ..."
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'timeout' : 'Time to wait for a TACACS+ server to respond',
      '<TIMEOUT>' : CliMatcher.IntegerMatcher(
         _timeoutMin, _timeoutMax,
         helpdesc='Number of seconds' )
   }
   @staticmethod
   def handler( mode, args ):
      tacacsConfig.timeout = args[ '<TIMEOUT>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tacacsConfig.timeout = tacacsConfig.defaultTimeout

configMode.addCommandClass( TacacsServerTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server policy unknown-mandatory-attribute ignore" in config mode
#-------------------------------------------------------------------------------
class TacacsServerUnknownAttrCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server policy unknown-mandatory-attribute ignore"
   noOrDefaultSyntax = syntax
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'policy' : 'Set TACACS+ policy',
      'unknown-mandatory-attribute' : 'Unknown mandatory attributes',
      'ignore' : 'Ignore unknown mandatory attributes'
   }
   @staticmethod
   def handler( mode, args ):
      tacacsConfig.ignoreUnknownMandatoryAttr = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tacacsConfig.ignoreUnknownMandatoryAttr = False

configMode.addCommandClass( TacacsServerUnknownAttrCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server qos dscp <0-63>" in config mode
#-------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( configMode,
                                   "TacacsHandler.setDscp",
                                   "TacacsHandler.noDscp",
                                   tokenProto=tacacsServerKwMatcher )

#-------------------------------------------------------------------------------
# "[no] tacacs-server username max-length <1-255>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerUsernameMaxLenCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server username max-length <LENGTH>"
   noOrDefaultSyntax = "tacacs-server username max-length ..."
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'username' : 'Configure username parameters',
      'max-length' : 'Maximum username length',
      '<LENGTH>' : CliMatcher.IntegerMatcher( 1, 255,
                                              helpdesc='Specify username length' )
   }
   @staticmethod
   def handler( mode, args ):
      tacacsConfig.maxUsernameLength = args[ '<LENGTH>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tacacsConfig.maxUsernameLength = tacacsConfig.maxUsernameLengthDefault

configMode.addCommandClass( TacacsServerUsernameMaxLenCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#     [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]"
# in config mode
#-------------------------------------------------------------------------------
def singularNode( keyword, helpdesc ):
   return CliCommand.Node( CliMatcher.KeywordMatcher( keyword,
                                                      helpdesc=helpdesc ),
                           maxMatches=1 )

class TacacsServerHostCommand( CliCommand.CliCommandClass ):
   syntax = 'tacacs-server host <HOSTNAME> ' \
            '[ { single-connection | ( VRF ) | ( port <PORT> ) | ' \
            '( timeout <TIMEOUT> ) } ] [ key <KEY> ]'
   noOrDefaultSyntax = 'tacacs-server host ' \
                       '[ <HOSTNAME> [ single-connection ] [ VRF ] ' \
                       '[ port <PORT> ] ] ...'
   data = { 'tacacs-server' : tacacsServerKwMatcher,
            'host' : 'TACACS+ server configuration',
            '<HOSTNAME>' : HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of TACACS+ server',
               ipv6=True ),
            'single-connection' :
            singularNode( 'single-connection',
                          'Use one TCP connection for multiple sessions' ),
            'VRF' : VrfExprFactory( helpdesc='VRF for this TACACS+ server',
                                    maxMatches=1 ),
            'port' :
            singularNode( 'port',
                          'TCP port of TACACS+ server ( default is % s )' % \
                          TacacsGroup.defaultPort ),
            '<PORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                             helpdesc="Number of the port to use" ),
            'timeout' :
            singularNode( 'timeout',
                          'Time to wait for this TACACS+ server to respond ' \
                          '( overrides default)' ),
            '<TIMEOUT>' : CliMatcher.IntegerMatcher(
               _timeoutMin, _timeoutMax,
               helpdesc='Timeout value in seconds to '
                        'wait for the server\'s response' ),
            'key' : 'Encryption key for this TACACS+ server (overrides default)',
            '<KEY>' : keyExpression
            }
   handler = "TacacsHandler.setTacacsServerHost"
   noOrDefaultHandler = "TacacsHandler.noTacacsServerHost"

configMode.addCommandClass( TacacsServerHostCommand )

#-------------------------------------------------------------------------------
# "[no] ip tacacs source-interface <interface-name>"
# global command
#-------------------------------------------------------------------------------
class TacacsSourceIntfCommand( CliCommand.CliCommandClass ):
   syntax = "ip tacacs [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip tacacs [ VRF ] source-interface ..."
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "tacacs" : "TACACS+ configuration",
      "VRF" : VrfExprFactory( helpdesc='Specify VRF' ),
      "source-interface" : 'Interface providing the IP source ' \
      'address of TACACS+ packets',
      "INTF" : IntfCli.Intf.matcherWithIpSupport
      }
   handler = "TacacsHandler.setTacacsSrcIntf"
   noOrDefaultHandler = "TacacsHandler.noTacacsSrvIntf"

configMode.addCommandClass( TacacsSourceIntfCommand )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
def showTacacsGuard():
   # skip if no tacacs-servers are configured
   return tacacsConfig.host

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-08-27 04:34:25',
   cmds=[ 'show tacacs' ],
   cmdsGuard=showTacacsGuard,
   summaryCmds=[ 'show tacacs' ],
   summaryCmdsGuard=showTacacsGuard )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global tacacsConfig
   tacacsConfig = ConfigMount.mount( entityManager, "security/aaa/tacacs/config",
                                     "Tacacs::Config", "w" )

