#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import ConfigMount
import CliParser
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from Tracing import Handle, t0

__defaultTraceHandle__ = Handle( 'TagConfigCliPlugin' )

tagConfig = None

class TagCmdsModelet( CliParser.Modelet ):
   pass

class TagCommand( CliCommand.CliCommandClass ):
   syntax = 'tag <TAGNAME> [ <TAGVALUE> ]'
   noOrDefaultSyntax = 'tag [ <TAGNAME> [ <TAGVALUE> ] ] ...'
   data = {
      'tag': CliMatcher.KeywordMatcher(
         'tag',
         helpdesc='Configure Tag for the interface' ),
      '<TAGNAME>': CliMatcher.PatternMatcher( pattern=r'[A-Za-z0-9_-]+',
                                              helpname="WORD",
                                              helpdesc="Tag name" ),
      '<TAGVALUE>': CliMatcher.PatternMatcher( pattern=r'[A-Za-z0-9_-]+',
                                              helpname="WORD",
                                              helpdesc="Tag value" ),
   }

   @staticmethod
   def handler( mode, args ):
      tagName = args[ '<TAGNAME>' ]
      tagValue = args.get( '<TAGVALUE>', '' )
      intfName = mode.intf.name
      if not tagName:
         t0( "Did not find tag name for", intfName )
         return
      else:
         t0( f"Configured {tagName} for {intfName}" )
         if intfName not in tagConfig.intfConfig:
            tagConfig.newIntfConfig( intfName )
         intfConfig = tagConfig.intfConfig[ intfName ]
         if len( intfConfig.tag ) == intfConfig.maxTags:
            if tagName not in intfConfig.tag:
               # pylint: disable-next=consider-using-f-string
               mode.addError( "Maximum of %d tags are configurable on an interface"
                                 % intfConfig.maxTags )
         intfConfig.tag[ tagName ] = tagValue

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tagName = args.get( '<TAGNAME>' )
      intfName = mode.intf.name
      if intfName in tagConfig.intfConfig:
         intfConfig = tagConfig.intfConfig[ intfName ]
         if tagName:
            if tagValue:= args.get( '<TAGVALUE>' ):
               if intfConfig.tag[ tagName ] == tagValue:
                  del intfConfig.tag[ tagName ]
            else:
               del intfConfig.tag[ tagName ]
         else:
            intfConfig.tag.clear()
         if not tagConfig.intfConfig[ intfName ].tag:
            del tagConfig.intfConfig[ intfName ]

TagCmdsModelet.addCommandClass( TagCommand )
IntfCli.IntfConfigMode.addModelet( TagCmdsModelet )

class TagIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del tagConfig.intfConfig[ self.intf_.name ]

#------------------------------------------------------
# Plugin method
#------------------------------------------------------

def Plugin( em ):
   global tagConfig
   tagConfig = ConfigMount.mount( em, "tag/config", "Tag::Config", "w" )
   IntfCli.Intf.registerDependentClass( TagIntfCleaner )
