#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in
# pylint: disable=consider-using-f-string

import os
import Tac, CliParser, BasicCli, LazyMount, ConfigMount
import CliPlugin.ConfigConvert
from CliPlugin import IntfCli
from CliPlugin import ( EthIntfCli, LagIntfCli, RecircIntfCli, SwitchIntfCli,
                        InternalRecircIntfCli, VlanIntfCli )
from CliPlugin import TapAggModels
from CliPlugin import VlanCli
from CliPlugin.IpGenAddrMatcher import isValidTunnelEndpointAddress
from CliPlugin.TapAggPmapCliLib import TapAggCliError
from CliPlugin.TapAggTcamLib import TapAggTcamUtil
import Cell
import MultiRangeRule
import Arnet
import Tracing
import CliExtensions
from CliMode.TapAgg import ( TapAggMode, IdentityMapMode )
from CliCommon import InvalidInputError, IncompleteCommandError
from FilteredDictView import FilteredDictView
import CliCommand
import CliMatcher
from TypeFuture import TacLazyType
import CliSession
import six

__defaultTraceHandle__ = Tracing.Handle( 'TapAggIntfCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1

aclIntfConfig = None
aegisPmapGroupConfig = None
bridgingConfig = None
bridgingSwitchIntfConfigDir = None
cliNexthopGroupConfig = None
ethIntfConfigDir = None
hwSlice = None
intfHwStatus = None
intfInputAegis = None
intfStatusAll = None
lagConfig = None
mcastNhgSelection = None
nhgStatus = None
tapAggCliConfig = None
tapAggHwConfig = None
tapAggHwStatus = None
tapAggIntfConfig = None
tapAggPmapConfig = None
tapAggPmapStatus = None
tapAggStatus = None

SwitchIntfConfigDefault = TacLazyType( "Bridging::Input::SwitchIntfConfigDefault" )
# BUG162413: upper and lower limit of TapAggGlobalTruncationSize is 169. If egress
# truncation isn't supported just use ingress sizes
def globalTruncationRange( mode, context ):
   if tapAggHwStatus.egressTruncationSupported:
      return ( 169, 169 )
   else:
      return ( 100, BrInSicType.tapTruncationSizeMax )

EthAddr = Tac.Type( 'Arnet::EthAddr' )
EthAddrTuple = Tac.Type( 'Arnet::EthAddrTuple' )
EthIntfId = Tac.Type( 'Arnet::EthIntfId' )
GreProtocol = Tac.Type( 'TapAgg::GreProtocol' )
FcsModeType = Tac.Type( 'TapAgg::TapAggFcsMode' )
RuntModeType = Tac.Type( 'TapAgg::TapAggRuntMode' )
InternalRecircIntfId = Tac.Type( 'Arnet::InternalRecircIntfId' )
sicType = Tac.Type( 'Bridging::Input::SwitchIntfConfig' )
SubIntfId = Tac.Type( 'Arnet::SubIntfId' )
TapAggModeType = Tac.Type( "TapAgg::TapAggMode" )
TimestampHeaderFormatType = Tac.Type( 'TapAgg::TapAggTimestampHeaderFormat' )
TimestampHeaderPlacementType = Tac.Type( 'TapAgg::TapAggTimestampHeaderPlacement' )
ToolIdentityMode = Tac.Type( 'Bridging::Input::ToolIdentityMode' )
IdentityTuple = Tac.Type( 'Bridging::IdentityTuple' )
TapHeaderRemove = Tac.Type( 'Bridging::Input::TapHeaderRemove' )
# pkgdeps: library TunnelIntfLib
TapDestination = Tac.Type( 'Interface::TapDestination' )
DzGreVlanTaggingMode = Tac.Type( 'Bridging::Input::DzGreVlanTaggingMode' )
SwitchIdPortId = Tac.Type( 'TapAgg::SwitchIdPortId' )
DzGreSwitchIdPortIdToVlan = Tac.Type( 'TapAgg::DzGreSwitchIdPortIdToVlan' )
DzGrePolicyIdToVlan = Tac.Type( 'TapAgg::DzGrePolicyIdToVlan' )

def tapaggGuard( mode, token ):
   if ( len( tapAggHwStatus.modeSupported ) == 0  or
        ( token == 'exclusive' and
          'tapAggModeExclusive' not in tapAggHwStatus.modeSupported ) or
        ( token == 'mixed' and
          'tapAggModeMixed' not in tapAggHwStatus.modeSupported ) ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def dot1qRemoveGuard( mode, token ):
   if not tapAggHwStatus.dot1qRemoveSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def ingressTruncationGuard( mode, token ):
   if not tapAggHwStatus.ingressTruncationSupported:
      return CliParser.guardNotThisPlatform
   elif hasattr( mode, 'intf' ) and mode.intf.name.startswith( "InternalRecirc" ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def egressTruncationGuard( mode, token ):
   if not tapAggHwStatus.egressTruncationSupported:
      return CliParser.guardNotThisPlatform
   elif hasattr( mode, 'intf' ) and mode.intf.name.startswith( "InternalRecirc" ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationSizePerIngressPortGuard( mode, token ):
   if not tapAggHwStatus.truncationSizePerIngressPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return ingressTruncationGuard( mode, token )

def truncationNoSizePerIngressPortGuard( mode, token ):
   if not tapAggHwStatus.globalIngressTruncationSizeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return ingressTruncationGuard( mode, token )

def truncationSizePerEgressPortGuard( mode, token ):
   if not tapAggHwStatus.truncationSizePerEgressPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def vxlanStripGuard( mode, token ):
   if not tapAggHwStatus.vxlanStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def greStripGuard( mode, token ):
   if not tapAggHwStatus.greStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None
   
def gtpStripGuard( mode, token ):
   if not tapAggHwStatus.gtpStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationNoSizePerEgressPortGuard( mode, token ):
   if not tapAggHwStatus.globalEgressTruncationSizeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationGlobalSizeGuard( mode, token ):
   if truncationNoSizePerIngressPortGuard( mode, token ) is None or \
      truncationNoSizePerEgressPortGuard( mode, token ) is None:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPopGuard( mode, token ):
   if not tapAggHwStatus.mplsPopSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def mplsPopToolGuard( mode, token ):
   if not tapAggHwStatus.mplsPopToolSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def vnBrTagStripToolGuard( mode, token ):
   if not tapAggHwStatus.brVnTagStripToolSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def guardMacAcls( mode, token ):
   if tapAggHwStatus.pmapMacSupported:
      return None
   return CliParser.guardNotThisPlatform

def macAclMatchIpConfigurableGuard( mode, token ):
   if tapAggHwStatus.macAclMatchIpConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def brVnTagStripGuard( mode, token ):
   if not tapAggHwStatus.brVnTagStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def lldpReceiveConfigurableGuard( mode, token ):
   if tapAggHwStatus.lldpReceiveConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def tcamProfileGuard( mode, token ):
   if tapAggHwStatus.tcamProfileSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampReplaceSmacGuard( mode, token ):
   if not tapAggHwStatus.timestampReplaceSmacSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def timestampCustomEthTypeGuard( mode, token ):
   if not tapAggHwStatus.timestampEthTypeCustomizable:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsErrorModeGuard( mode, token ):
   if not tapAggHwStatus.fcsErrorModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsErrorModePassthroughGuard( mode, token ):
   if not tapAggHwStatus.fcsErrorModePassthroughSupported \
         or not tapAggHwStatus.fcsErrorModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsModeGuard( mode, token ):
   if not tapAggHwStatus.fcsModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def runtModeGuard( mode, token ):
   if not tapAggHwStatus.runtModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def qinqIdentityTaggingGuard( mode, token ):
   if tapAggHwStatus.qinqIdentityTaggingSupported:
      return None
   return CliParser.guardNotThisPlatform

def macAddressGuard( mode, token ):
   if tapAggHwStatus.macReplaceSupported:
      return None
   return CliParser.guardNotThisPlatform

def tapMacAddressGuard( mode, token ):
   if tapAggHwStatus.tapMacReplaceSupported:
      return None
   return CliParser.guardNotThisPlatform

def sourceMacAddressGuard( mode, token ):
   if tapAggHwStatus.sourceMacReplaceSupported:
      return None
   return CliParser.guardNotThisPlatform

def tapToolPortGuard( mode, token ):
   if not tapAggHwStatus.tapToolPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return tapaggGuard( mode, token )

def counterAccessGuard( mode, token ):
   if ( tapPortCountersGetHook.extensions() and
        tapAggHwStatus.defaultForwardingCountersSupported and
        tapAggHwStatus.trafficSteeringCountersSupported ) or \
      ( tapTunnelCountersGetHook.extensions() and
        tapAggHwStatus.tapTunnelTerminationCountersSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def tapTunnelCounterAccessGuard( mode, token ):
   if tapTunnelCountersGetHook.extensions() and \
      tapAggHwStatus.tapTunnelTerminationCountersSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardNexthopGroupTapAgg( mode, token ):
   if tapAggHwStatus.nexthopGroupSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDefaultNexthopGroupTapAgg( mode, token ):
   if tapAggHwStatus.tapDefaultNexthopGroupSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampFooterSupportedGuard( mode, token ):
   if intfHwStatus.timestampFooterSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampHeaderSupportedGuard( mode, token ):
   if intfHwStatus.timestampHeaderSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampHeaderPlacementSupportedGuard( mode, token ):
   if intfHwStatus.timestampHeaderSupported and \
      tapAggHwStatus.timestampPlacementSupported:
      return None
   return CliParser.guardNotThisPlatform

def tapHeaderRemoveSupportedGuard( mode, token ):
   if tapAggHwStatus.tapHeaderRemoveSupported:
      return None
   return CliParser.guardNotThisPlatform

def dzGreEncodeSupportedGuard( mode, token ):
   if tapAggHwStatus.dzGreEncodeSupported:
      return None
   return CliParser.guardNotThisPlatform

def dzGreDecodeSupportedGuard( mode, token ):
   if tapAggHwStatus.dzGreDecodeSupported:
      return None
   return CliParser.guardNotThisPlatform

class TapAggModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      intfTypes = ( "Ethernet", "Port-Channel", "Recirc-Channel", "Switch",
                    "InternalRecirc" )
      return ( mode.intf.name.startswith( intfTypes ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( TapAggModelet )

class TimestampModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( not SubIntfId.isSubIntfId( mode.intf.name ) if
                   EthIntfId.isEthIntfId( mode.intf.name )
               else InternalRecircIntfId.isInternalRecircIntfId( mode.intf.name ) )

IntfCli.IntfConfigMode.addModelet( TimestampModelet )

#----------------------------------------------------------------------------
# Use TCAM profile utility class to retrieve profiles (from AleTcam, presumably)
#----------------------------------------------------------------------------

# Tcam profile utility class likely overridden by AleTcam
tcamUtilHook = CliExtensions.CliHook()

def getTapAggTcamUtil():
   if tcamUtilHook.extensions():
      return tcamUtilHook.extensions()[ 0 ]()
   return TapAggTcamUtil()

def allTcamProfiles( mode=None ):
   utils = getTapAggTcamUtil()
   allProfiles = utils.tcamProfileGetAll()
   # The hook is not available in breadth tests
   if not hasattr( utils, "tapAggUnsupportedSystemProfileHook" ):
      return { profile: ''
               for profile in allProfiles }
   hook = utils.tapAggUnsupportedSystemProfileHook()
   return { profile: ''
            for profile in allProfiles
            if not hook.all( mode, profile ) }

#----------------------------------------------------------------------------
# Helper functions
#----------------------------------------------------------------------------

def isValidSource( groupName, srcIntf ):
   # Source is invalid if it is present as a destination in any other
   # tap-agg group or if it is present as a destination in the current group
   tpc = tapAggCliConfig
   for name in tpc.group:
      if srcIntf in tpc.group[ name ].toolInterfaces:
         return( False, 'Duplicate' )
   #Source is invalid if its part of any Lag
   if srcIntf in lagConfig.phyIntf and \
          lagConfig.phyIntf[ srcIntf ].lag:
      return( False, 'Interface mode conflict, %s is part of lag %s' % \
                 ( srcIntf, lagConfig.phyIntf[ srcIntf ].lag.name ) )

   return( True, '' )

def intfUp( intf ):
   return intfStatusAll.intfStatus[ intf ].operStatus == 'intfOperUp'

# ------------------------------------------------------------------------
# '[no|default] switchport mode [tap|tool|tap-tool]'
# ------------------------------------------------------------------------
switchportModeCmdHook = CliExtensions.CliHook()
def setSwitchportTapToolMode( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   modeMapping = { 'tap': 'tap',
                   'tool': 'tool',
                   'tap-tool': 'tapTool' }
   chosenMode = None
   for availableMode in modeMapping:
      if availableMode in args:
         chosenMode = availableMode
         break
   assert chosenMode
   for hook in switchportModeCmdHook.extensions():
      if not hook( mode, mode.intf.name ):
         return
   sic.switchportMode = modeMapping[ chosenMode ]

# ------------------------------------------------------------------------
# ' tap aggregation'
# '[no|default] mode mixed|exclusive'
# ------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# tap-agg config mode. A new instance of this mode is created when the
# user enters "tap aggregation>" in config mode.
#-------------------------------------------------------------------------------
class TapAggConfigMode( TapAggMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Tap-agg group configuration'

   #----------------------------------------------------------------------------
   # Constructs a new TapAggMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      self.modeKey = 'tap-agg'
      assert isinstance( parent, BasicCli.GlobalConfigMode )
      TapAggMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def setTapAggNoErrdisableIntf( mode, args ):
   for intf in _normalizeIntfList( args[ 'INTFS' ] ):
      tapAggCliConfig.noerrdisableIntf[ intf ] = True

def noTapAggNoErrdisableIntf( mode, args ):
   for intf in _normalizeIntfList( args[ 'INTFS' ] ):
      del tapAggCliConfig.noerrdisableIntf[ intf ]

def setMode( mode, args ):
   tapAggModeMapping = {
      'exclusive' : TapAggModeType.tapAggModeExclusive,
      'mixed' : TapAggModeType.tapAggModeMixed,
   }
   tapAggMode = TapAggModeType.tapAggModeNone
   for opt in tapAggModeMapping: # pylint: disable=consider-using-dict-items
      if opt in args:
         tapAggMode = tapAggModeMapping[ opt ]
         break

   cardList = args.get( 'CARDNUMBERS' )
   isTapAggLc = args[ 'isTapAggLc' ] if 'isTapAggLc' in args else True

   tcamProfile = ''
   for opt in [ 'TAPAGGPROFILES', 'USERPROFILE' ]:
      if opt in args:
         tcamProfile = args[ opt ]
         break

   # modeChanged flag indicates whether to bump the mode version
   modeChanged = False

   # Save old mode
   oldMode = tapAggCliConfig.mode
   oldTcamProfile = tapAggCliConfig.tcamProfile
   # pylint: disable-next=unnecessary-comprehension
   oldSliceProfileConfig = { card: profile for card, profile in
                             six.iteritems( tapAggCliConfig.sliceProfileConfig ) }
   oldMirroringRunnability = tapAggCliConfig.mirroringRunnability

   if tapAggCliConfig.mode != tapAggMode:
      tapAggCliConfig.mode = tapAggMode
      modeChanged = True
      tapAggCliConfig.tcamProfile = ""
      tapAggCliConfig.sliceProfileConfig.clear()

   if not tcamProfile:
      tapAggEnabled = tapAggCliConfig.mode != TapAggModeType.tapAggModeNone
      tcamProfile = getTapAggTcamUtil().defaultTcamProfileToProgram(
            tapAggEnabled=tapAggEnabled )

   if tapAggMode == TapAggModeType.tapAggModeMixed:
      if cardList:
         for card in cardList:
            if isTapAggLc:
               if card not in tapAggCliConfig.sliceProfileConfig or \
                  tapAggCliConfig.sliceProfileConfig[ card ] != tcamProfile:
                  modeChanged = True
                  tapAggCliConfig.sliceProfileConfig[ card ] = tcamProfile
            elif card in tapAggCliConfig.sliceProfileConfig:
               modeChanged = True
               del tapAggCliConfig.sliceProfileConfig[ card ]
      # set mode to None if all linecard are removed out of TapAgg
      if len( tapAggCliConfig.sliceProfileConfig ) == 0:
         tapAggCliConfig.mode = TapAggModeType.tapAggModeNone
         tcamProfile = getTapAggTcamUtil().defaultTcamProfileToProgram( False )
   elif tapAggMode == TapAggModeType.tapAggModeExclusive:
      if tapAggCliConfig.tcamProfile != tcamProfile:
         modeChanged = True
         tapAggCliConfig.tcamProfile = tcamProfile

   # Set mirroringRunnability after mode is updated
   tapAggCliConfig.mirroringRunnability = \
      ( tapAggCliConfig.mode != TapAggModeType.tapAggModeNone )

   # Now bump the mode version
   if modeChanged:
      tapAggCliConfig.modeVersion = tapAggCliConfig.modeVersion + 1

   restartWarn = ''
   if tapAggHwStatus.modeChangeRestartsForwardingAgent:
      restartWarn = 'will restart the forwarding agent(s) and '

   # warn if the selected profile doesn't exist
   if tcamProfile not in allTcamProfiles():
      mode.addWarning( 'Configured TCAM profile ' + tcamProfile +
                       ' does not exist. ' \
                       'TAP aggregation will be enabled with the selected profile' \
                       ' once available.' )

   mode.addWarning( 'Changing modes ' + restartWarn +
         'may affect available functionality. '\
         'Unsupported configuration elements will be ignored.' )

   # Verify programming if we're using a non-system profile
   if tcamProfile not in getTapAggTcamUtil().tapAggSystemProfiles():
      if mode.session.inConfigSession():
         CliSession.registerSessionOnCommitHandler( mode.session.entityManager_,
               "tap-agg-tcam-profile-" + tcamProfile,
               lambda mode, onSessionCommit=False:
               not getTapAggTcamUtil().tcamProfileVerifyProgrammingComplete(
                  tcamProfile ) )
      elif not getTapAggTcamUtil().tcamProfileVerifyProgrammingComplete(
            tcamProfile ):
         mode.addError(
               'Failed to program {!r} TCAM profile. Reverting to previous '
               'configuration '.format( tcamProfile ) )
         tapAggCliConfig.mode = oldMode
         tapAggCliConfig.tcamProfile = oldTcamProfile
         tapAggCliConfig.sliceProfileConfig.clear()
         for card, profile in six.iteritems( oldSliceProfileConfig ):
            tapAggCliConfig.sliceProfileConfig[ card ] = profile
         tapAggCliConfig.mirroringRunnability = oldMirroringRunnability
         # Bump the version number
         tapAggCliConfig.modeVersion = tapAggCliConfig.modeVersion + 1

def noMode( mode, args ):
   tapAggModeMapping = {
      'exclusive' : TapAggModeType.tapAggModeExclusive,
      'mixed' : TapAggModeType.tapAggModeMixed,
   }
   tapAggMode = None
   tapAggModeName = ''
   for opt in tapAggModeMapping: # pylint: disable=consider-using-dict-items
      if opt in args:
         tapAggMode = tapAggModeMapping[ opt ]
         tapAggModeName = opt
         break

   cardList = args.get( 'CARDNUMBERS' )
   if tapAggMode and tapAggMode != tapAggCliConfig.mode:
      mode.addWarning( 'Mode ' + tapAggMode +' is not current active mode. '\
                          'Command is ignored.' )
      return
   if tapAggMode == TapAggModeType.tapAggModeMixed and cardList:
      args[ 'isTapAggLc' ] = False
   else:
      if tapAggMode:
         del args[ tapAggModeName ]
      if cardList:
         del args[ 'CARDNUMBERS' ]
   setMode( mode, args )

def gotoTapAggregationMode( mode, args ):
   childMode = mode.childMode( TapAggConfigMode )
   mode.session_.gotoChildMode( childMode )

def noTapAggregationConfig( mode, args ):
   noMode( mode, args )
   tapAggCliConfig.noerrdisableIntf.clear()
   tapAggCliConfig.trafficSteeringMacAclMatchIp = False
   tapAggCliConfig.globalTruncationSize = 0
   tapAggCliConfig.vnTagStrip = False
   tapAggCliConfig.brTagStrip = False
   tapAggCliConfig.timestampReplaceSmac = False
   tapAggCliConfig.timestampHeaderFormat = \
      TimestampHeaderFormatType.tapAggTimestampDefault
   tapAggCliConfig.timestampHeaderPlacement = \
      TimestampHeaderPlacementType.tapAggTimestampPlacementDefaultAfterSrcMac
   tapAggCliConfig.timestampHeaderEthType = Tac.enumValue( 'Arnet::EthType',
                                                           'ethTypeArista' )
   tapAggCliConfig.fcsMode = FcsModeType.tapAggFcsErrorDefault
   tapAggCliConfig.fcsErrorMode = FcsModeType.tapAggFcsErrorDefault
   tapAggCliConfig.runtMode = RuntModeType.tapAggRuntDrop
   tapAggCliConfig.lldpReceive = False
   setDefaultGreProtocol( mode, args )
   setDefaultDzGreSwitchId( mode, args )
   noIdentityMapDzGreConfig( mode, args )

def vnTagStripConfig( mode, args ):
   tapAggCliConfig.vnTagStrip = True

def noVnTagStripConfig( mode, args ):
   tapAggCliConfig.vnTagStrip = False

def brTagStripConfig( mode, args ):
   tapAggCliConfig.brTagStrip = True

def noBrTagStripConfig( mode, args ):
   tapAggCliConfig.brTagStrip = False

def lldpReceiveConfig( mode, no ):
   tapAggCliConfig.lldpReceive = not no

def setTrafficSteeringMacAclMatchIp( mode, args ):
   tapAggCliConfig.trafficSteeringMacAclMatchIp = True

def noTrafficSteeringMacAclMatchIp( mode, args ):
   tapAggCliConfig.trafficSteeringMacAclMatchIp = False

def setGlobalTruncationSize( mode, truncationSize ):
   tapAggCliConfig.globalTruncationSize = truncationSize

def setDefaultGlobalTruncationSize( mode, args ):
   setGlobalTruncationSize( mode, 0 ) # 0 means platform default

def getGlobalTruncationSize():
   configuredSize = tapAggCliConfig.globalTruncationSize
   if configuredSize != 0:
      return configuredSize
   else:
      return tapAggHwStatus.globalTruncationSizeDefault

def setGreProtocol( mode, args ):
   tapAggCliConfig.globalGreProtocol = args[ 'PROTOCOL' ]

def setDefaultGreProtocol( mode, args ):
   tapAggCliConfig.globalGreProtocol = GreProtocol.defaultProtocol

def timestampReplaceSmacConfig( mode, args ):
   tapAggCliConfig.timestampReplaceSmac = True
   if tapAggCliConfig.timestampHeaderFormat != \
         TimestampHeaderFormatType.tapAggTimestampDefault or \
      tapAggCliConfig.timestampHeaderPlacement != \
         TimestampHeaderPlacementType.tapAggTimestampPlacementDefaultAfterSrcMac:
      mode.addWarning(
         'More than one timestamp format configured. '
         'Replace source-mac takes precedence.' )

def setDefaultTimestampReplaceSmac( mode, args ):
   tapAggCliConfig.timestampReplaceSmac = False

def setTimestampHeaderPlacement( mode, args ):
   placementMapping = {
      'source-mac' : \
            TimestampHeaderPlacementType.tapAggTimestampPlacementDefaultAfterSrcMac,
      'l2' : TimestampHeaderPlacementType.tapAggTimestampPlacementAfterL2,
   }
   headerPlacement = None
   for placementOption in placementMapping:
      if placementOption in args:
         headerPlacement = args[ placementOption ]
         break
   assert headerPlacement in placementMapping
   tapAggCliConfig.timestampHeaderPlacement = placementMapping[ headerPlacement ]
   if tapAggCliConfig.timestampReplaceSmac: # replace-smac configured
      mode.addWarning(
         'More than one timestamp format configured. '
         'Replace source-mac takes precedence.' )

def setDefaultTimestampHeaderPlacement( mode, args ):
   tapAggCliConfig.timestampHeaderPlacement = \
      TimestampHeaderPlacementType.tapAggTimestampPlacementDefaultAfterSrcMac

def setTimestampHeaderFormat( mode, args ):
   formatMapping = {
      '48-bit' : TimestampHeaderFormatType.tapAggTimestamp48bit,
      '64-bit' : TimestampHeaderFormatType.tapAggTimestamp64bit,
   }
   headerFormat = None
   for formatOption in formatMapping:
      if formatOption in args:
         headerFormat = args[ formatOption ]
         break
   assert headerFormat in formatMapping
   tapAggCliConfig.timestampHeaderFormat = formatMapping[ headerFormat ]
   if tapAggCliConfig.timestampReplaceSmac: # replace-smac configured
      mode.addWarning(
         'More than one timestamp format configured. '
         'Replace source-mac takes precedence.' )

def setDefaultTimestampHeaderFormat( mode, args ):
   tapAggCliConfig.timestampHeaderFormat = \
      TimestampHeaderFormatType.tapAggTimestampDefault

def setTimestampHeaderEthType( mode, args ):
   tapAggCliConfig.timestampHeaderEthType = args[ 'ETHTYPE' ]

def setDefaultTimestampHeaderEthType( mode, args ):
   tapAggCliConfig.timestampHeaderEthType = Tac.enumValue( 'Arnet::EthType',
                                                           'ethTypeArista' )

def setFcsErrorMode( mode, args ):
   modeMapping = {
      'correct' : FcsModeType.tapAggFcsErrorCorrect,
      'discard' : FcsModeType.tapAggFcsErrorDiscard,
      'pass-through' : FcsModeType.tapAggFcsErrorPassThrough,
   }
   fcsErrorMode = None
   for modeOption in modeMapping:
      if modeOption in args:
         fcsErrorMode = args[ modeOption ]
         break
   assert fcsErrorMode in modeMapping
   tapAggCliConfig.fcsErrorMode = modeMapping[ fcsErrorMode ]
   if tapAggCliConfig.fcsMode != FcsModeType.tapAggFcsErrorDefault:
      mode.addWarning( "'mac fcs' has already been configured and takes "
                       "precedence over 'mac fcs-error'." )

def noFcsErrorMode( mode, args ):
   tapAggCliConfig.fcsErrorMode = FcsModeType.tapAggFcsErrorDefault

def setFcsMode( mode, args ):
   modeMapping = {
      'append' : FcsModeType.tapAggFcsAppend,
   }
   fcsMode = args[ 'ACTION' ]
   assert fcsMode in modeMapping
   tapAggCliConfig.fcsMode = modeMapping[ fcsMode ]

   if tapAggCliConfig.fcsErrorMode != FcsModeType.tapAggFcsErrorDefault:
      mode.addWarning( "'mac fcs-error' has already been configured. 'mac fcs' "
                       "takes precedence." )

def noFcsMode( mode, args ):
   tapAggCliConfig.fcsMode = FcsModeType.tapAggFcsErrorDefault

def setRuntMode( mode, args ):
   modeMapping = {
         'drop': RuntModeType.tapAggRuntDrop,
         'pad': RuntModeType.tapAggRuntPad,
         'pass-through': RuntModeType.tapAggRuntPassThrough,
   }
   runtMode = args[ 'MODE' ]
   assert runtMode in modeMapping
   tapAggCliConfig.runtMode = modeMapping[ runtMode ]

def noRuntMode( mode, args ):
   tapAggCliConfig.runtMode = RuntModeType.tapAggRuntDrop

def setDzGreSwitchId( mode, args ):
   tapAggCliConfig.dzGreSwitchId = args[ 'IDENTITY' ]

def setDefaultDzGreSwitchId( mode, args ):
   tapAggCliConfig.dzGreSwitchId = 0

def setDzGreSwitchPortToVlan( mode, args ):
   vlanId = args[ 'VLAN_ID' ].id
   if 'unmatched' in args:
      tapAggCliConfig.dzGreDefaultSwitchIdPortIdVlan = vlanId
      return
   switchId = args[ 'SWITCH_ID' ]
   portId = args[ 'PORT_ID' ]
   switchIdPortId = SwitchIdPortId( switchId, portId )
   entry = DzGreSwitchIdPortIdToVlan( switchIdPortId, vlanId )
   tapAggCliConfig.dzGreSwitchIdPortIdToVlanMap.addMember( entry )

def noDzGreSwitchPortToVlan( mode, args ):
   if 'unmatched' in args:
      tapAggCliConfig.dzGreDefaultSwitchIdPortIdVlan = 0
      return
   switchId = args[ 'SWITCH_ID' ]
   portId = args[ 'PORT_ID' ]
   switchIdPortId = SwitchIdPortId( switchId, portId )
   del tapAggCliConfig.dzGreSwitchIdPortIdToVlanMap[ switchIdPortId ]

def setDzGrePolicyToVlan( mode, args ):
   vlanId = args[ 'VLAN_ID' ].id
   if 'unmatched' in args:
      tapAggCliConfig.dzGreDefaultPolicyIdVlan = vlanId
      return
   policyId = args[ 'POLICY_ID' ]
   entry = DzGrePolicyIdToVlan( policyId, vlanId )
   tapAggCliConfig.dzGrePolicyIdToVlanMap.addMember( entry )

def noDzGrePolicyToVlan( mode, args ):
   if 'unmatched' in args:
      tapAggCliConfig.dzGreDefaultPolicyIdVlan = 0
      return
   policyId = args[ 'POLICY_ID' ]
   del tapAggCliConfig.dzGrePolicyIdToVlanMap[ policyId ]

# Note that tap-agg group names are silently truncated to 32 characters.
# This is consistent with VLAN names.

def getGroupList( mode ):
   result = set( tapAggStatus.group )
   for group in tapAggPmapConfig.group.values():
      result |= set( group.groupName )

   return result

matcherTapAggGroupName = CliMatcher.DynamicNameMatcher( getGroupList,
      'Group name', pattern=r'[A-Za-z0-9_:{}\[\]-]+' )

def getNexthopGroupList( mode ):
   return sorted( cliNexthopGroupConfig.nexthopGroup.members() )

notMac = "(?!mac$).*"
matcherNexthopGroupName = CliMatcher.PatternMatcher(
      notMac, helpname='WORD', helpdesc="Nexthop-group name", partialPattern=notMac )

matcherNexthopGroup = CliCommand.guardedKeyword(
      "nexthop-group",
      helpdesc="List of nexthop-groups to aggregate flow",
      guard=guardNexthopGroupTapAgg )

def filterInternal( token ):
   return token != sicType.tapGroupDefault

matcherTapAggGroupNameDynamic = CliMatcher.DynamicKeywordMatcher(
      lambda mode: FilteredDictView( tapAggStatus.group,
                                     filterInternal ),
      emptyTokenCompletion=[ CliParser.Completion( 'WORD',
                        'TapAgg group name', literal=False ) ] )

matcherMacAddress = CliCommand.guardedKeyword(
      'mac-address',
      helpdesc='MAC Addresses',
      guard=macAddressGuard )
matcherTapMacAddress = CliCommand.guardedKeyword(
      'mac-address',
      helpdesc='MAC Addresses',
      guard=tapMacAddressGuard )

matcherMacAddressDest = CliMatcher.KeywordMatcher(
      'dest',
      helpdesc='MAC address for the destination' )
matcherMacAddressSrc = CliCommand.guardedKeyword(
      'src',
      helpdesc='MAC address for the source',
      guard=sourceMacAddressGuard )

#-------------------------------------------------------------------------------
# identity-map config mode.
# A new instance of this mode is created when the user enters
# "identity map type dzgre" in tap-agg config mode.
#-------------------------------------------------------------------------------
class IdentityMapDzGreMode( IdentityMapMode, BasicCli.ConfigModeBase ):
   name = 'Identity map type dzgre configuration'

   def __init__( self, parent, session ):
      IdentityMapMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoIdentityMapDzGreMode( mode, args ):
   childMode = mode.childMode( IdentityMapDzGreMode )
   mode.session_.gotoChildMode( childMode )

def noIdentityMapDzGreConfig( mode, args ):
   tapAggCliConfig.dzGreSwitchIdPortIdToVlanMap.clear()
   tapAggCliConfig.dzGrePolicyIdToVlanMap.clear()
   tapAggCliConfig.dzGreDefaultSwitchIdPortIdVlan = 0
   tapAggCliConfig.dzGreDefaultPolicyIdVlan = 0

# ------------------------------------------------------------------------
# switchport tap default group <comma-saparated-group-list>
# [no|default] switchport tap default ( group |  ( group <name> group <name> ) )
# "no trailing garbage as it is a list but list is optional"
# ------------------------------------------------------------------------
def addTapGroups( mode, args, cfg ):
   groupList = args.get( 'GROUP_NAME', None )

   for group in groupList:
      cfg.tapGroup[ group ] = True

def removeTapGroups( mode, args, cfg ):
   groupList = args.get( 'GROUP_NAME', None )

   if groupList:
      for group in groupList:
         del cfg.tapGroup[ group ]
   else:
      for group in cfg.tapGroup:
         del cfg.tapGroup[ group ]

def handleTapGroups( mode, args ):
   t0( "handleTapGroups" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   addTapGroups( mode, args, sic )

def handleNoTapGroups( mode, args ):
   t0( "handleNoTapGroups" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   removeTapGroups( mode, args, sic )

def _normalizeIntfList( intfs ):
   if not intfs:
      return
   try:
      yield from intfs
   except TypeError:
      assert isinstance( intfs, ( EthIntfCli.EthPhyIntf,
                                  LagIntfCli.EthLagIntf,
                                  RecircIntfCli.EthRecircIntf,
                                  SwitchIntfCli.SwitchIntf,
                                  VlanIntfCli.VlanIntf,
                                  InternalRecircIntfCli.InternalRecircIntf, ) )
      # It's a singleton object, not a list of intfs
      yield from [ intfs.name ]

# ------------------------------------------------------------------------
# switchport tap default interface add|remove <interface-list>
# [no|default] switchport tap default interface <trailing-garbage>
# ------------------------------------------------------------------------
def addTapRawIntf( mode, args, cfg ):
   configuredIntfs = args.get( 'INTFS', None )
   for intf in _normalizeIntfList( configuredIntfs ):
      cfg.tapRawIntf[ intf ] = True

def removeTapRawIntf( mode, args, cfg ):
   configuredIntfs = args.get( 'INTFS', None )
   if configuredIntfs is None:
      for intf in cfg.tapRawIntf:
         del cfg.tapRawIntf[ intf ]
   else:
      for intf in _normalizeIntfList( configuredIntfs ):
         del cfg.tapRawIntf[ intf ]

def handleTapRawIntf( mode, args ):
   t0( "handleTapRawIntf" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   addTapRawIntf( mode, args, sic )

def handleNoTapRawIntf( mode, args ):
   t0( "handleNoTapRawIntf" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   removeTapRawIntf( mode, args, sic )

# ------------------------------------------------------------------------
# tap interface <interface-list> tool group <group>
# [no|default] tap interface [<interface-list> tool group [<group>]]
# tap interface <interface-list> tool interface <interface>
# [no|default] tap interface [<interface-list> tool interface [<interface>]]
# ------------------------------------------------------------------------
def addTunnelTapIntf( mode, args, cfg ):
   handler = addTapGroups if 'group' in args else addTapRawIntf
   tapIntfs = args.get( 'TAP_INTFS', None )

   intfList = _normalizeIntfList( tapIntfs )
   for intf in intfList:
      if intf not in cfg.tapMemberIntf:
         cfg.tapMemberIntf.newMember( intf )
      handler( mode, args, cfg.tapMemberIntf[ intf ] )

def removeTunnelTapIntf( mode, args, cfg ):
   handler = removeTapGroups if 'group' in args else removeTapRawIntf
   tapIntfs = args.get( 'TAP_INTFS', None )

   if not tapIntfs:
      cfg.tapMemberIntf.clear()
   else:
      intfList = _normalizeIntfList( tapIntfs )
      for intf in intfList:
         if intf in cfg.tapMemberIntf:
            handler( mode, args, cfg.tapMemberIntf[ intf ] )
            if ( not cfg.tapMemberIntf[ intf ].tapGroup and
                 not cfg.tapMemberIntf[ intf ].tapRawIntf ):
               del cfg.tapMemberIntf[ intf ]

# ------------------------------------------------------------------------
# switchport tap default nexthop-group add|remove <NHG-list>
# [no|default] switchport tap default nexthop-group <trailing-garbage>
# ------------------------------------------------------------------------
def handleTapNexthopGroups( mode, args ):
   t0( "handleTapNexthopGroups" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   addOp = not CliCommand.isNoOrDefaultCmd( args )
   groupList = args.get( 'NEXTHOP_GROUPS', [] )

   for group in groupList:
      if addOp:
         sic.tapNexthopGroup[ group ] = True
      else:
         del sic.tapNexthopGroup[ group ]
   if not groupList and not addOp:
      sic.tapNexthopGroup.clear()

# ------------------------------------------------------------------------
# switchport tap native vlan <vlan>
# [no|default] switchport tap native vlan <trailing-garbage>
# ------------------------------------------------------------------------

defaultTapNativeVlan = 1
defaultTapIdentity = 1

def handleTapNativeVlan( mode, args ):
   t0( "handleTapNativeVlan" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   if CliCommand.isNoOrDefaultCmd( args ):
      sic.tapNativeVlan = defaultTapNativeVlan
   else:
      sic.tapNativeVlan = args[ 'VLAN_ID' ].id

# ------------------------------------------------------------------------
# switchport tap identity <1-4094> [ inner <1-4094> ]
# [ no|default ] switchport tap identity <trailing garbage>
# ------------------------------------------------------------------------

def handleTapIdentity( mode, args ):
   t0( "handleTapIdentity" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   portId = args.get( 'PORT_ID', 0 )
   if not portId:
      portId = args.get( 'PORT_EXT_ID', 0 )
   innerPortId = args.get( 'INNER_PORT_ID', 0 )
   sic.tapIdentity = IdentityTuple( portId, innerPortId )

# ------------------------------------------------------------------------
# switchport tap mac address dest DMAC [ src SMAC ]
# [ no|default ] switchport tap mac-address <trailing garbage>
# ------------------------------------------------------------------------

def handleTapMacAddress( mode, args ):
   t0( "handleTapMacAddress" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   if 'DMAC' in args and Arnet.EthAddr( args[ 'DMAC' ] ) == EthAddr():
      mode.addError( TapAggCliError[ 'setMacAddress' ] )
      return
   elif 'SMAC' in args and Arnet.EthAddr( args[ 'SMAC' ] ) == EthAddr():
      mode.addError( TapAggCliError[ 'setMacAddress' ] )
      return
   destMac = args.get( 'DMAC' ) or EthAddr()
   srcMac = args.get( 'SMAC' ) or EthAddr()
   sic.tapMacAddress = EthAddrTuple( destMac, srcMac )

# ------------------------------------------------------------------------
# switchport tap encapsulation vxlan strip
# [ no|default ] switchport tap encapsulation vxlan strip
# ------------------------------------------------------------------------

def handleTapVxlanStrip( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapVxlanStrip = not CliCommand.isNoOrDefaultCmd( args )
   if tapVxlanStrip and sic.tapMplsPop:
      mode.addError( 'Error: Must disable TAP Port MPLS Pop' )
      return
   sic.tapVxlanStrip = tapVxlanStrip

# ------------------------------------------------------------------------
# switchport tap encapsulation gre strip
# [ no|default ] switchport tap encapsulation gre
#    [ destination dstIp [ source srcIp ] ]
#    ( strip | ( protocol PROTORANGE [ feature header length LENGTH ] strip
#    [ re-encasulation ethernet ] ) )
# ------------------------------------------------------------------------
TapTunnelInfo = Tac.Type( 'Bridging::Input::TapTunnelInfo' )

def tapTunnelIpStripAdapter( mode, args, argsList ):
   if args.get( 'DST_IP' ):
      errString = isValidTunnelEndpointAddress( Arnet.IpGenAddr( args[ 'DST_IP' ] ) )
      if errString:
         mode.addError( errString )
         return
   if args.get( 'SRC_IP' ):
      errString = isValidTunnelEndpointAddress( Arnet.IpGenAddr( args[ 'SRC_IP' ] ) )
      if errString:
         mode.addError( errString )
         return

def handleTapGreStrip( mode, args ):
   t0( "handleTapGreStrip" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapTunnelInfo = TapTunnelInfo(
         Arnet.IpGenAddr( args.get( 'DST_IP' ) or '0.0.0.0' ),
         Arnet.IpGenAddr( args.get( 'SRC_IP' ) or '0.0.0.0' ),
         args.get( 'PROTOCOL', TapTunnelInfo.protocolNull ) )
   if args.get( 'PROTOCOL' ):
      tapTunnelInfo.featureHeaderLength = args.get( 'LENGTH', 0 )
      tapTunnelInfo.ethEncapEnabled = 'ethernet' in args
   for key in sic.tapAllowedGreTunnel:
      # Exit if there is an existing entry that supercedes the new cmd
      if ( not key.dstIp.isAddrZero and key.srcIp.isAddrZero and
           key.dstIp.v4Addr == tapTunnelInfo.dstIp.v4Addr and
           key.protocol == tapTunnelInfo.protocol ):
         return
   for key in sic.tapAllowedGreTunnel:
      # Delete existing entries that are identical to the new cmd or superceded by
      # the new cmd
      if ( ( ( key.dstIp.isAddrZero and key.srcIp.isAddrZero ) or
             ( tapTunnelInfo.dstIp.isAddrZero or
               key.dstIp.v4Addr == tapTunnelInfo.dstIp.v4Addr ) and
             ( tapTunnelInfo.srcIp.isAddrZero or
               key.srcIp.v4Addr == tapTunnelInfo.srcIp.v4Addr ) ) and
             key.protocol == tapTunnelInfo.protocol ):
         del sic.tapAllowedGreTunnel[ key ]
   sic.tapAllowedGreTunnel[ tapTunnelInfo ] = True
   sic.tapGreStrip = True

def handleNoTapGreStrip( mode, args ):
   t0( "handleNoTapGreStrip" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   dstIp = Arnet.IpGenAddr( args.get( 'DST_IP' ) or '0.0.0.0' )
   srcIp = Arnet.IpGenAddr( args.get( 'SRC_IP' ) or '0.0.0.0' )
   for key in sic.tapAllowedGreTunnel:
      # Delete entries that matches the 'no' cmd
      if ( ( dstIp.isAddrZero or
             key.dstIp.v4Addr == dstIp.v4Addr ) and
           ( srcIp.isAddrZero or
             key.srcIp.v4Addr == srcIp.v4Addr ) and
           ( ( 'protocol' not in args ) or (
               ( 'PROTOCOL' not in args and
                 key.protocol != TapTunnelInfo.protocolNull ) or
               ( 'PROTOCOL' in args
                 and key.protocol == args[ 'PROTOCOL' ] ) and
                 ( 'LENGTH' not in args or
                   key.featureHeaderLength == args[ 'LENGTH' ] ) and
                 ( 'ethernet' not in args or key.ethEncapEnabled ) ) ) ):
         del sic.tapAllowedGreTunnel[ key ]
   sic.tapGreStrip = len( sic.tapAllowedGreTunnel ) > 0

# ------------------------------------------------------------------------
# switchport tap encapsulation gtp version VERSION strip
# [ no|default ] switchport tap encapsulation gtp version VERSION
#    [ destination dstIp [ source srcIp ] ] strip
# ------------------------------------------------------------------------

def handleTapGtpStrip( mode, args ):
   t0( "handleTapGtpStrip" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapTunnelInfo = TapTunnelInfo(
         Arnet.IpGenAddr( args.get( 'DST_IP', '0.0.0.0' ) ),
         Arnet.IpGenAddr( args.get( 'SRC_IP', '0.0.0.0' ) ),
         TapTunnelInfo.protocolNull )
   for key in sic.tapAllowedGtpTunnel:
      # Exit if there is an existing entry that supercedes the new cmd
      if ( not key.dstIp.isAddrZero and key.srcIp.isAddrZero and
           key.dstIp.v4Addr == tapTunnelInfo.dstIp.v4Addr and
           key.protocol == tapTunnelInfo.protocol ):
         mode.addErrorAndStop( "There is an existing entry "
                               "that supercedes the new cmd" )
   for key in sic.tapAllowedGtpTunnel:
      # Delete existing entries that are identical to the new cmd or superceded by
      # the new cmd
      if ( key.protocol == tapTunnelInfo.protocol and
            (
               ( key.dstIp.isAddrZero and key.srcIp.isAddrZero ) or
               (
                  ( tapTunnelInfo.dstIp.isAddrZero or
                  key.dstIp.v4Addr == tapTunnelInfo.dstIp.v4Addr ) and
                  ( tapTunnelInfo.srcIp.isAddrZero or
                  key.srcIp.v4Addr == tapTunnelInfo.srcIp.v4Addr )
               )
            )
              ):
         del sic.tapAllowedGtpTunnel[ key ]
   sic.tapAllowedGtpTunnel[ tapTunnelInfo ] = True
   sic.tapGtpStrip = True

def handleNoTapGtpStrip( mode, args ):
   t0( "handleNoTapGtpStrip" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   dstIp = Arnet.IpGenAddr( args.get( 'DST_IP', '0.0.0.0' ) )
   srcIp = Arnet.IpGenAddr( args.get( 'SRC_IP', '0.0.0.0' ) )
   for key in sic.tapAllowedGtpTunnel:
      # Delete entries that matches the 'no' cmd
      if ( ( dstIp.isAddrZero or key.dstIp.v4Addr == dstIp.v4Addr ) and
           ( srcIp.isAddrZero or key.srcIp.v4Addr == srcIp.v4Addr ) ):
         del sic.tapAllowedGtpTunnel[ key ]
   sic.tapGtpStrip = bool( sic.tapAllowedGtpTunnel )

# ------------------------------------------------------------------------
# switchport tap mpls pop all
# [ no|default ] switchport tap mpls pop [ all ]
# ------------------------------------------------------------------------

def handleTapMplsPop( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapMplsPop = not CliCommand.isNoOrDefaultCmd( args )
   if tapMplsPop and sic.tapVxlanStrip:
      mode.addError( 'Error: Must disable TAP Port VXLAN Strip' )
      return
   sic.tapMplsPop = tapMplsPop

# ------------------------------------------------------------------------
# switchport tool mpls pop all
# [ no|default ] switchport tool mpls pop [ all ]
# ------------------------------------------------------------------------

def handleToolMplsPop( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolMplsPop = not CliCommand.isNoOrDefaultCmd( args )
   sic.toolMplsPop = toolMplsPop
   if sic.toolVnStrip and toolMplsPop:
      mode.addWarning( "'vn-tag strip' has already been configured. 'mpls pop' "
                       "takes precedence." )
   if sic.toolBrStrip and toolMplsPop:
      mode.addWarning( "'dot1br strip' has already been configured. 'mpls pop' "
                       "takes precedence." )

# ------------------------------------------------------------------------
# switchport tool encapsulation vn-tag strip
# [ no|default ] switchport tool encapsulation vn-tag strip
# ------------------------------------------------------------------------

def handleToolVnStrip( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolVnStrip = not CliCommand.isNoOrDefaultCmd( args )
   sic.toolVnStrip = toolVnStrip
   if sic.toolMplsPop and toolVnStrip:
      mode.addWarning( "'mpls pop' has already been configured and takes "
                       "precedence over 'vn-tag strip'." )

# ------------------------------------------------------------------------
# switchport tool encapsulation dot1br strip
# [ no|default ] switchport tool encapsulation dot1br strip
# ------------------------------------------------------------------------

def handleToolBrStrip( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolBrStrip = not CliCommand.isNoOrDefaultCmd( args )
   sic.toolBrStrip = toolBrStrip
   if sic.toolMplsPop and toolBrStrip:
      mode.addWarning( "'mpls pop' has already been configured and takes "
                       "precedence over 'dot1br strip'." )

# ------------------------------------------------------------------------
# switchport tool group set <group-list> #old format
# switchport tool group <name> group <name> ...  #new format
# [no|default] switchport tool group <name> group <name>
# ------------------------------------------------------------------------

def handleInvalidGroupOpAndList( groupOp, groupList ):
   invalidGroupNames = [ 'set', 'group', 'add', 'remove' ]
   if groupList == None: # pylint: disable=singleton-comparison
      raise IncompleteCommandError()
   if groupOp == 'add' or groupOp == 'set':
      for groupName in invalidGroupNames:
         if groupName in groupList:
            errorMsg = 'Cannot name group using reserved keyword \'%s\'' \
                       % groupName
            raise InvalidInputError( ': ' + errorMsg )

def handleToolGroups( mode, args ):
   t0( "handleToolGroups" )
   groupList = []
   groupOp = 'set'
   if 'GROUP_NAME' in args:
      groupList = args[ 'GROUP_NAME' ]
      if CliCommand.isNoOrDefaultCmd( args ):
         groupOp = 'remove'
      elif 'set' not in args:
         groupOp = 'add'

   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   handleInvalidGroupOpAndList( groupOp, groupList )
   for group in groupList:
      if groupOp == 'add' or ( groupOp == 'set' and group ):
         sic.toolGroup[ group ] = True
      elif groupOp == 'remove':
         del sic.toolGroup[ group ]
   if groupOp == 'set':
      for group in sic.toolGroup:
         if group not in groupList:
            del sic.toolGroup[ group ]

# --------------------------------------------------------------------------
# switchport tool identity ( ( dot1q [ source dzgre ( policy | port ) ] ) |
#                            ( qinq [ source dzgre ( ( policy inner port ) |
#                            ( port inner policy ) ) ] ) )
# [no|default] switchport tool identity [ ( dot1q | qinq )
#                                       [ source dzgre ... ] ]
# --------------------------------------------------------------------------

def handleIdentityTagging( mode, args ):
   t0( "handleIdentityTagging" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   identityModeMapping = { 'dot1q' : ToolIdentityMode.dot1q,
                           'qinq'  : ToolIdentityMode.qinq }
   chosenIdentityMode = ToolIdentityMode.none
   # pylint: disable-next=consider-using-dict-items
   for identityModeOption in identityModeMapping:
      if identityModeOption in args:
         chosenIdentityMode = identityModeMapping[ identityModeOption ]
         break
   if 'source' not in args:
      sic.toolIdentityTagging = chosenIdentityMode
   else:
      if chosenIdentityMode == ToolIdentityMode.dot1q:
         policyTag = args.get( 'policy', None )
         portTag = args.get( 'port', None )
         if policyTag:
            sic.toolDzGreIdentityTagging = DzGreVlanTaggingMode.dzGreDot1qPolicy
         elif portTag:
            sic.toolDzGreIdentityTagging = DzGreVlanTaggingMode.dzGreDot1qPort
         else:
            sic.toolDzGreIdentityTagging = DzGreVlanTaggingMode.dzGreNone
      elif chosenIdentityMode == ToolIdentityMode.qinq:
         argsList = list( args )
         policyTokenPosition = argsList.index( 'policy' )
         portTokenPosition = argsList.index( 'port' )
         if policyTokenPosition < portTokenPosition:
            sic.toolDzGreIdentityTagging = \
                  DzGreVlanTaggingMode.dzGreQinqPolicyInnerPort
         elif policyTokenPosition > portTokenPosition:
            sic.toolDzGreIdentityTagging = \
                  DzGreVlanTaggingMode.dzGreQinqPortInnerPolicy
         else:
            sic.toolDzGreIdentityTagging = \
                  DzGreVlanTaggingMode.dzGreNone
      else:
         sic.toolDzGreIdentityTagging = DzGreVlanTaggingMode.dzGreNone

def handleNoIdentityTagging( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   if 'source' not in args:
      sic.toolIdentityTagging = ToolIdentityMode.none
   else:
      sic.toolDzGreIdentityTagging = DzGreVlanTaggingMode.dzGreNone

# ------------------------------------------------------------------------
# switchport tool dot1q remove outer <range>
# [no|default] switchport tool dot1q remove outer <trailing garbage>
# ------------------------------------------------------------------------

def handleDot1qRemoveVlan( mode, args ):
   tag = str( args.get( 'DOT_1Q_TAG', '' ) )
   if tag:
      # args[ 'DOT_1Q_TAG' ] only accepts the following input: '1' and '1-2'
      # with '1' indicates outer VLAN id, and '2' inner VLAN id.
      if ( mode.session.guardsEnabled() ) and \
         not tapAggHwStatus.dot1qRemoveInnerOnlySupported and '1' not in tag:
         mode.addError( "Removing only inner VLAN is not supported "
                        "on this hardware platform" )
         return
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   sic.toolDot1qRemoveVlans = str( tag )

matcherDot1QTag = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, tapAggHwStatus.dot1qRemoveMaxIndex ),
   False,
   helpdesc='Specify indices of vlan tags to be removed'
)

# ------------------------------------------------------------------------
# switchport tap header remove size <size> [preserve ethernet]
# [no|default] switchport tap header remove size <trailing garbage>
# ------------------------------------------------------------------------

def handleTapHeaderRemove( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   size = args.get( 'SIZE', 0 )
   # The size has to be an even number.
   if size % 2 == 1:
      mode.addError( "The size must be an even number." )
      return
   preserveEth = 'ethernet' in args
   sic.tapHeaderRemove = TapHeaderRemove( size, preserveEth )

# ------------------------------------------------------------------------
# switchport tap truncation <size>
# switchport tap truncation
# [no|default] switchport tap truncation <trailing garbage>
# ------------------------------------------------------------------------

def handleTapTruncation( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapTruncationSize = 0
   if not CliCommand.isNoOrDefaultCmd( args ):
      tapTruncationSize = args.get( 'SIZE', 1 )
   sic.tapTruncationSize = tapTruncationSize

BrInSicType = Tac.Type( "Bridging::Input::SwitchIntfConfig" )
matcherTapTruncationSize = CliMatcher.IntegerMatcher(
   100, BrInSicType.tapTruncationSizeMax,
   helpdesc='ingress packet truncation size in bytes' )

# ------------------------------------------------------------------------
# switchport tool truncation <size>
# switchport tool truncation
# [no|default] switchport tool truncation <trailing garbage>
# ------------------------------------------------------------------------

def handleToolTruncation( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolTruncationSize = 0
   if not CliCommand.isNoOrDefaultCmd( args ):
      toolTruncationSize = args.get( 'SIZE', 1 )
   sic.toolTruncationSize = toolTruncationSize

matcherToolTruncationSize = CliMatcher.IntegerMatcher(
   160, 160,
   helpdesc='egress packet truncation size in bytes' )

#------------------------------------------------------------------------
# 'show interface [ <intfName> ] tap [ detail ]'
# 'show interface [ <intfName> ] tool [ detail ]'
# 'show tap aggregation group [ <group-name> ... ] [ detail ]'
# 'show tap aggregation group detail [ <group-name> ]'
# 'show tap aggregation counters default-forwarding'
# ------------------------------------------------------------------------

def doShowTapAggMode( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args
   tunnel = 'tunnel' in args
   tapAggModeShow = args.get( 'tap' ) or args[ 'tool' ]

   def getAllowedVlanStr( allowedVlans ):
      if allowedVlans == SwitchIntfConfigDefault.defaultTapAllowedVlans:
         allowedVlans = 'All'
      elif allowedVlans == '':
         allowedVlans = 'None'
      return allowedVlans

   def getRemovedVlanStr( removedVlans ):
      return removedVlans if removedVlans else 'None'

   model = TapAggModels.TapAggProperties( _detail=detail,
      _tunnel=tunnel,
      _mplsPopToolSupported=tapAggHwStatus.mplsPopToolSupported,
      _brVnTagStripSupported=tapAggHwStatus.brVnTagStripSupported,
      _brVnTagStripToolSupported=tapAggHwStatus.brVnTagStripToolSupported,
      _vxlanStripSupported=tapAggHwStatus.vxlanStripSupported,
      _mplsPopSupported=tapAggHwStatus.mplsPopSupported )


   sicdir = bridgingSwitchIntfConfigDir
   ethIntfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   if ethIntfs is None:
      return None

   for intfName in ethIntfs:
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None:
         continue

      switchportMode = switchIntfConfig.switchportMode
      configuredMode = switchportMode

      if not switchIntfConfig.enabled:
         switchportMode = 'routed'
         configuredMode = 'routed'

      if intf is None and switchportMode != tapAggModeShow:
         if not( switchportMode == 'tapTool' and \
               ( tapAggModeShow == 'tap' or tapAggModeShow == 'tool' ) ):
            continue


      linkStatus = 'non-%s' % tapAggModeShow
      intfStatus = intfName.linkStatus()
      if intfStatus and intfStatus not in [ 'connected', 'up' ]:
         linkStatus = intfStatus
      elif configuredMode == tapAggModeShow or configuredMode == 'tapTool':
         linkStatus = tapAggModeShow

      if( configuredMode == 'tapTool' and \
            ( tapAggModeShow == 'tap' or tapAggModeShow == 'tool' ) ):
         configuredMode = 'tap-tool'

      if intfName.name in lagConfig.phyIntf and \
         lagConfig.phyIntf[ intfName.name ].lag.intfId:
         continue

      if tapAggModeShow == 'tap':
         nativeVlan = switchIntfConfig.tapNativeVlan
         idVlan = switchIntfConfig.tapIdentity

         truncationSize = switchIntfConfig.tapTruncationSize
         if truncationSize != 0:
            if not tapAggHwStatus.truncationSizePerIngressPortSupported:
               truncationSize = getGlobalTruncationSize()
            elif truncationSize == sicType.tapTruncationUseGlobalSize:
               # If per-port size is supported but configured to 1, this means to use
               # the global truncation size. We need this for backward compatibility
               # reasons.
               truncationSize = getGlobalTruncationSize()

         allowedVlans = getAllowedVlanStr( switchIntfConfig.tapAllowedVlans )
         tapProperty = TapAggModels.TapAggProperties.TapProperty(
            configuredMode=configuredMode,
            linkStatus=linkStatus,
            portIdentity=idVlan.outer,
            innerPortIdentity=idVlan.inner,
            allowedVlans=allowedVlans,
            nativeVlan=nativeVlan,
            truncationSize=truncationSize
            )
         if detail:
            tapGroup = []
            intfList = []
            defaultGroup = '---'
            if switchIntfConfig.tapGroup:
               tapGroup = list( switchIntfConfig.tapGroup )
            if switchIntfConfig.tapRawIntf:
               intfList = list( switchIntfConfig.tapRawIntf )
            if len( tapGroup ) != 0:
               defaultGroup = tapGroup[ 0 ]
            vxlanStrip = switchIntfConfig.tapVxlanStrip
            mplsPop = switchIntfConfig.tapMplsPop
            destMac = switchIntfConfig.tapMacAddress.destMac
            srcMac = switchIntfConfig.tapMacAddress.srcMac
            tapProperty.details = TapAggModels.TapAggProperties.TapProperty. \
                Details( defaultGroup=defaultGroup, groups=tapGroup,
                         toolPorts=intfList,
                         aclsAppliedPerType=getAppliedAcls( 'in', intfName.name ),
                         vxlanStrip=vxlanStrip,
                         mplsPop=mplsPop,
                         destMac=destMac,
                         srcMac=srcMac
                         )
         elif tunnel:
            tapProperty.tunnel = TapAggModels.TapAggProperties.TapProperty. \
                Tunnel( vnTagStrip=tapAggCliConfig.vnTagStrip,
                        brTagStrip=tapAggCliConfig.brTagStrip,
                        vxlanStrip=switchIntfConfig.tapVxlanStrip,
                        mplsPop=switchIntfConfig.tapMplsPop
                        )
         model.tapProperties[ intfName.name ] = tapProperty

      elif tapAggModeShow == 'tool':
         idTag = 'Off'
         if switchIntfConfig.toolIdentityTagging != ToolIdentityMode.none:
            idTag = switchIntfConfig.toolIdentityTagging
         allowedVlans = getAllowedVlanStr( switchIntfConfig.toolAllowedVlans )
         timestampMode = 'NA'
         if intfName.name.startswith( ( 'Ethernet', 'Switch', 'InternalRecirc' ) ):
            ethPhyIntfConfig = ethIntfConfigDir.intfConfig.get( intfName.name )
            timestamp = Tac.Type( 'Interface::EthTimestampMode' )
            timestampMode = 'None'
            if ethPhyIntfConfig.timestampMode == timestamp.timestampModeBeforeFcs:
               timestampMode = 'before-fcs'
            elif ethPhyIntfConfig.timestampMode == \
                   timestamp.timestampModeReplaceFcs:
               timestampMode = 'replace-fcs'
            elif ethPhyIntfConfig.timestampMode == timestamp.timestampModeHeader:
               timestampMode = 'header'

         toolProperty = TapAggModels.TapAggProperties.ToolProperty(
            configuredMode=configuredMode,
            linkStatus=linkStatus,
            identificationTag=idTag,
            allowedVlans=allowedVlans,
            timestampMode=timestampMode )
         if detail:
            removedVlans = getRemovedVlanStr( switchIntfConfig.toolDot1qRemoveVlans )

            truncationSize = switchIntfConfig.toolTruncationSize
            if truncationSize != 0 and \
               not tapAggHwStatus.truncationSizePerEgressPortSupported:
               truncationSize = getGlobalTruncationSize()
            toolGroup = []
            if list( switchIntfConfig.toolGroup ):
               toolGroup = list( switchIntfConfig.toolGroup )
            toolProperty.details = TapAggModels.TapAggProperties.ToolProperty. \
                Details( truncationSize=truncationSize,
                         aclsAppliedPerType=getAppliedAcls( 'out', intfName.name ),
                         groups=toolGroup,removedVlans=removedVlans )
         elif tunnel:
            mplsPop = switchIntfConfig.toolMplsPop
            vnTagStrip = switchIntfConfig.toolVnStrip
            brTagStrip = switchIntfConfig.toolBrStrip
            toolProperty.tunnel = TapAggModels.TapAggProperties.ToolProperty. \
                Tunnel( vnTagStrip=vnTagStrip,
                        brTagStrip=brTagStrip,
                        mplsPop=mplsPop )


         model.toolProperties[ intfName.name ] = toolProperty

   return model

def doShowTapAggMap( mode, args ):
   configured = 'configured' in args
   sicdir = bridgingSwitchIntfConfigDir
   ethIntfs = IntfCli.Intf.getAll( mode, None, intfType=EthIntfCli.EthIntf )

   if 'TAPPORT' in args:
      mapType = 'tap'
      model = TapAggModels.TapToToolMap()
   elif 'TOOLPORT' in args:
      mapType = 'tool'
      model = TapAggModels.ToolToTapMap()

   tapGroupsDict = { }
   toolGroupsDict = { }
   rawIntfDict = { }

   if ethIntfs is None:
      return None

   for intfName in ethIntfs:
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None:
         continue

      if configured:
         rawIntfDict[ intfName.name ] = list( switchIntfConfig.tapRawIntf )
         for tapGroup in switchIntfConfig.tapGroup:
            if tapGroup in tapGroupsDict:
               tapGroupsDict[ tapGroup ].append( intfName.name )
            else:
               tapGroupsDict[ tapGroup ] = [ intfName.name ]
         for toolGroup in switchIntfConfig.toolGroup:
            if toolGroup in toolGroupsDict:
               toolGroupsDict[ toolGroup ].append( intfName.name )
            else:
               toolGroupsDict[ toolGroup ] = [ intfName.name ]
      else:
         for intf in switchIntfConfig.tapRawIntf:
            if intf in tapAggHwConfig.toolPort and \
                   intfName.name in tapAggHwConfig.tapPort and \
                   intfUp( intf ) and intfUp( intfName.name ):
               if intfName.name in rawIntfDict:
                  rawIntfDict[ intfName.name ].append( intf )
               else:
                  rawIntfDict[ intfName.name ] = [ intf ]
         for tapaggGroup in tapAggStatus.group:
            if intfName.name in tapAggStatus.group[ tapaggGroup ].tapPort and \
                   intfName.name in tapAggHwConfig.tapPort and \
                   intfUp( intfName.name ):
               if tapaggGroup in tapGroupsDict:
                  tapGroupsDict[ tapaggGroup ].append( intfName.name )
               else:
                  tapGroupsDict[ tapaggGroup ] = [ intfName.name ]
            if intfName.name in tapAggStatus.group[ tapaggGroup ].toolPort and \
                   intfName.name in tapAggHwConfig.toolPort and \
                   intfUp( intfName.name ):
               if tapaggGroup in toolGroupsDict:
                  toolGroupsDict[ tapaggGroup ].append( intfName.name )
               else:
                  toolGroupsDict[ tapaggGroup ] = [ intfName.name ]

   for intfName in ethIntfs: # pylint: disable=too-many-nested-blocks
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None or \
             ( not configured and not intfUp( intfName.name ) ):
         continue
      tapProperty = []
      toolProperty = []

      # tap tool group config

      if mapType == 'tap':
         for tapGroup in switchIntfConfig.tapGroup:
            if tapGroup not in tapGroupsDict or tapGroup not in toolGroupsDict:
               continue
            for toolPort in toolGroupsDict[ tapGroup ]:
               tapIntfProperty = TapAggModels.MapProperty(
                  mappedIntfName=toolPort,
                  groupName=tapGroup,
                  policyName='' )
               tapProperty.append( tapIntfProperty )
      elif mapType == 'tool':
         for toolGroup in switchIntfConfig.toolGroup:
            if toolGroup not in tapGroupsDict or toolGroup not in toolGroupsDict:
               continue
            for tapPort in tapGroupsDict[ toolGroup ]:
               toolIntfProperty = TapAggModels.MapProperty(
                  mappedIntfName=tapPort,
                  groupName=toolGroup,
                  policyName='' )
               toolProperty.append( toolIntfProperty )

      # raw intf config
      if intfName.name in rawIntfDict:
         for intf in rawIntfDict[ intfName.name ]:
            rawIntfProperty = TapAggModels.MapProperty(
               mappedIntfName=intf,
               groupName='',
               policyName='' )
            tapProperty.append( rawIntfProperty )

      for key in rawIntfDict: # pylint: disable=consider-using-dict-items
         if intfName.name in rawIntfDict[ key ]:
            toolIntfProperty = TapAggModels.MapProperty(
               mappedIntfName=key,
               groupName="",
               policyName='' )
            toolProperty.append( toolIntfProperty )

      # policy config

      if intfName.name in tapAggIntfConfig.intf:
         pmap = tapAggIntfConfig.intf[ intfName.name ]
         if pmap in tapAggPmapConfig.pmapType.pmap:
            currCfg = tapAggPmapConfig.pmapType.pmap[ pmap ].currCfg
            for classMap in currCfg.classAction:
               if classMap not in currCfg.rawClassMap:
                  policyName = "%s/%s" % ( pmap, classMap )
               else:
                  policyName = "%s/*" % ( pmap )
               if 'setAggregationGroup' in \
                      currCfg.classAction[ classMap ].policyAction:
                  aggGroups = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggGroup
                  aggIntfs = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggIntf
                  for aggGroup in aggGroups:
                     if aggGroup not in toolGroupsDict:
                        continue
                     for toolPort in toolGroupsDict[ aggGroup ]:
                        tapIntfProperty = TapAggModels.MapProperty(
                           mappedIntfName=toolPort,
                           groupName=aggGroup,
                           policyName=policyName )
                        tapProperty.append( tapIntfProperty )
                  for aggIntf in aggIntfs:
                     if not configured and not intfUp( aggIntf ):
                        continue
                     tapIntfProperty = TapAggModels.MapProperty(
                        mappedIntfName=aggIntf,
                        groupName='',
                        policyName=policyName )
                     tapProperty.append( tapIntfProperty )

      for intf in tapAggIntfConfig.intf:
         pmap = tapAggIntfConfig.intf[ intf ]
         if pmap in tapAggPmapConfig.pmapType.pmap:
            currCfg = tapAggPmapConfig.pmapType.pmap[ pmap ].currCfg
            for classMap in currCfg.classAction:
               if classMap not in currCfg.rawClassMap:
                  policyName = "%s/%s" % ( pmap, classMap )
               else:
                  policyName = "%s/*" % ( pmap )
               if 'setAggregationGroup' in \
                      currCfg.classAction[ classMap ].policyAction:
                  aggGroups = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggGroup
                  aggIntfs = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggIntf
                  if intfName.name in aggIntfs:
                     toolIntfProperty = TapAggModels.MapProperty(
                        mappedIntfName=intf,
                        groupName='',
                        policyName=policyName )
                     toolProperty.append( toolIntfProperty )
                  for aggGroup in aggGroups:
                     if aggGroup not in toolGroupsDict:
                        continue
                     if intfName.name in toolGroupsDict[ aggGroup ]:
                        toolIntfProperty = TapAggModels.MapProperty(
                           mappedIntfName=intf,
                           groupName=aggGroup,
                           policyName=policyName )
                        toolProperty.append( toolIntfProperty )

      if mapType == 'tap':
         model.tapPortsMap[ intfName.name ] = \
             TapAggModels.MapPropertyList(
            mapPropertyList=tapProperty )
      elif mapType == 'tool':
         model.toolPortsMap[ intfName.name ] = \
             TapAggModels.MapPropertyList(
            mapPropertyList=toolProperty )

   return model

def getAppliedAcls( direction, interfaceName ):
   aclDic = {}
   for aclType in aclIntfConfig.config:
      aclIntfTypeConfig = aclIntfConfig.config[ aclType ]
      aclIntfs = aclIntfTypeConfig.intf[ direction ]
      for aclIntf in aclIntfs.intf:
         acl = aclIntfs.intf.get( aclIntf )
         if not acl:
            continue
         aclDic.setdefault( aclIntf, {} )
         aclDic[ aclIntf ].setdefault( aclType, [] ).append( acl )

   aclsAppliedPerType = {}
   if interfaceName in aclDic:
      for aclType in aclDic[ interfaceName ]:
         aclsApplied = aclDic[ interfaceName ][ aclType ]
         appliedAcls = TapAggModels.AclsAppliedPerType( aclsApplied=aclsApplied )
         aclsAppliedPerType[ aclType ] = appliedAcls
   return aclsAppliedPerType

def getPortStatuses( portInfo, policy=None ):
   model = TapAggModels.TapAggGroups.TapAggGroup.PortStatus()
   if not policy:
      model.policies = [ '*' ]
      if portInfo.active:
         model.statuses = [ 'Active' ]
      else:
         model.statuses = [ s for s in portInfo.reason.split( ',' ) if s ]
   else:
      model.policies = [ policy ]
      if not portInfo:
         model.statuses = [ 'Link unknown' ]
      elif portInfo == 'linkUp':
         model.statuses = [ 'Active' ]
      elif portInfo == 'linkDown':
         model.statuses = [ 'Link down' ]
   return model

def getPortStatusMap( port ):
   return { port: getPortStatuses( portInfo ) for ( port, portInfo ) in
            six.iteritems( port ) }

def appendOrCreatePortInTapPortMap( tapPortMap, port, policyClassTag ):
   if port in tapPortMap:
      tapPortMap[ port ].policies.append( policyClassTag )
   else:
      if port in intfStatusAll.intfStatus:
         portInfo = intfStatusAll.intfStatus[ port ].linkStatus
         tapPortMap[ port ] = getPortStatuses( portInfo,
                                               policyClassTag )
      else:
         t1( f'failed to fetch status for port {port}. Skipping' )


def updateTapPortByPolicyMap( tapPortMap, groupName ):
   for policyName, policyInfo in six.iteritems( tapAggPmapConfig.pmapType.pmap ):
      # Skip policies without a configured class action
      if not policyInfo.currCfg:
         continue
      for className, classInfo in six.iteritems( policyInfo.currCfg.classAction ):
         if 'setAggregationGroup' in classInfo.policyAction and \
            groupName in classInfo.policyAction[ 'setAggregationGroup' ].aggGroup:
            for port, portPolicy in six.iteritems( tapAggIntfConfig.intf ):
               if port in tapAggHwConfig.tapPort and portPolicy == policyName:
                  policyClassTag = policyName + '_' + className
                  appendOrCreatePortInTapPortMap( tapPortMap, port, policyClassTag )

def updateTapPortByTrafficPolicy( tapPortMap, groupName ):
   # we use policies' name as the key to bridge over group data and interface data
   policies = {}

   # filling the policies dictionary with relevant group data
   for idMapName, idMapInfo in six.iteritems( aegisPmapGroupConfig
                                              .tapAggInternalGroupToIdMap ):
      if groupName not in idMapInfo.aggGroup:
         continue

      tags = idMapName.split( '%' )
      policies.setdefault( tags[ 1 ], [] ).append( tags )

   for interface, policy in six.iteritems( intfInputAegis.intf ):
      if policy not in policies:
         continue
      for tags in policies[ policy ]:
         policyClassTag = tags[ 1 ] + "_" + tags[ 2 ]

         appendOrCreatePortInTapPortMap( tapPortMap, interface, policyClassTag )

def updateTapPortByPolicy( tapPortMap, groupName ):
   updateTapPortByPolicyMap( tapPortMap, groupName )
   updateTapPortByTrafficPolicy( tapPortMap, groupName )

def createTapAggGroupModel( group ):
   model = TapAggModels.TapAggGroups.TapAggGroup()
   model.tapPortStatuses = getPortStatusMap( group.tapPort )
   updateTapPortByPolicy( model.tapPortStatuses, group.name )
   model.toolPortStatuses = getPortStatusMap( group.toolPort )
   return model

def doShowTapAggGroups( mode, args ):
   groups = args.get( 'GROUPNAME' )
   detail = 'detail' in args
   model = TapAggModels.TapAggGroups( _detail=detail )

   if not tapAggCliConfig.enabled:
      mode.addError( "Tap-aggregation is not enabled" )
      return model
   if groups:
      for group in groups:
         if group not in tapAggStatus.group:
            mode.addError( 'Group %s not created' % group )
   elif not tapAggStatus.group:
      return model

   for groupName, group in six.iteritems( tapAggStatus.group ):
      if groups and groupName not in groups:
         continue
      if groupName == sicType.tapGroupDefault:
         continue
      model.groups[ groupName ] = createTapAggGroupModel( group )
   return model

def doShowTapAggNexthopGroups( mode, args ):
   nhgFilter = args.get( 'NHGNAME' )
   model = TapAggModels.TapAggNexthopGroups()

   if not tapAggCliConfig.enabled:
      mode.addError( "Tap-aggregation is not enabled" )
      return model
   if nhgFilter:
      for nhg in nhgFilter:
         if nhg not in nhgStatus.nexthopGroupHwAdj:
            mode.addError( 'Nexthop group %s not created' % nhg )

   connections = []
   nhgSet = set( nhgFilter ) if nhgFilter else None
   for intfId, tapConfig in tapAggHwConfig.tapPort.items():
      configuredNhgs = set( tapConfig.tapNexthopGroup )
      if nhgSet and not nhgSet & configuredNhgs:
         continue
      for intfs, nhgs in connections:
         if len( nhgs ) == len( tapConfig.tapNexthopGroup ) and \
            all( nhg in nhgs for nhg in tapConfig.tapNexthopGroup ):
            intfs.append( intfId )
            break
      else:
         nexthopGroups = list( tapConfig.tapNexthopGroup.keys() )
         connections.append( ( [ intfId ], nexthopGroups ) )

   model.mcgNexthopGroups = []
   model.fecNexthopGroups = {}
   for intfs, nhgs in connections:
      if len( nhgs ) == 1:
         for tap in intfs:
            model.addFecEntry( tap, '', nhgs[ 0 ], [ via.l2Intf for via in
               nhgStatus.nexthopGroupHwAdj[ nhgs[ 0 ] ].via.values() ] )
      elif len( nhgs ) > 1:
         for tap in intfs:
            model.addMcgEntry( tap, '', interfacesForNhgMulticast( tap, nhgs ) )

   # pylint: disable=too-many-nested-blocks
   for policyName, policyInfo in six.iteritems( tapAggPmapConfig.pmapType.pmap ):
      if not policyInfo.currCfg:
         continue
      for className, classInfo in six.iteritems( policyInfo.currCfg.classAction ):
         if not 'setNexthopGroup' in classInfo.policyAction:
            continue
         nhgs = classInfo.policyAction[ 'setNexthopGroup' ].nexthopGroup
         if nhgFilter and not any( nhg in nhgFilter for nhg in nhgs ):
            continue
         for port, portPolicy in six.iteritems( tapAggIntfConfig.intf ):
            if port not in tapAggHwConfig.tapPort or portPolicy != policyName:
               continue
            if len( nhgs ) == 1:
               nhg = nhgs.keys()[ 0 ]
               model.addFecEntry(
                  port, policyName + '_' + className, nhg, [ via.l2Intf for via in
                     nhgStatus.nexthopGroupHwAdj[ nhg ].via.values() ] )
            elif len( nhgs ) > 1:
               model.addMcgEntry( port, policyName + '_' + className,
                                  interfacesForNhgMulticast( port, nhgs ) )

   for pmap, config in aegisPmapGroupConfig.tapAggInternalNexthopGroupMap.items():
      if nhgFilter and not any( nhg in nhgFilter for nhg in config.nexthopGroup ):
         continue
      if len( config.nexthopGroup ) == 1:
         nhg = config.nexthopGroup.keys()[ 0 ]
         for tap, policy in six.iteritems( intfInputAegis.intf ):
            if policy != pmap.split( '%' )[ 1 ]:
               continue
            model.addFecEntry( tap, pmap.split( '%', 1 )[ 1 ].replace( '%', '_' ),
                               nhg, [ via.l2Intf for via in
                                  nhgStatus.nexthopGroupHwAdj[ nhg ].via.values() ] )
      elif len( config.nexthopGroup ) > 1:
         for tap, policy in six.iteritems( intfInputAegis.intf ):
            if policy != pmap.split( '%' )[ 1 ]:
               continue
            model.addMcgEntry( tap, pmap.split( '%', 1 )[ 1 ].replace( '%', '_' ),
               interfacesForNhgMulticast( tap, list( config.nexthopGroup.keys() ) ) )

   return model

def interfacesForNhgMulticast( source, nhgs ):
   mcgInterfaces = {}
   for nhg in nhgs:
      mcgInterfaces[ nhg ] = mcastNhgSelection.chosenIntf.get( nhg, '' )
   return mcgInterfaces

#----------------------------------------------------------------------------
# Use CLI hook to aggregate counter values of TAP ports (initially the hook's
# only extension is implemented by SandDanz)
#----------------------------------------------------------------------------
tapPortCountersGetHook = CliExtensions.CliHook()
tapTunnelCountersGetHook = CliExtensions.CliHook()

def doReadOrUpdateTapPortCounters( mode, args, counterReset ):
   aggregatedTapPortCounters = TapAggModels.IntfCounterModel()
   for hook in tapPortCountersGetHook.extensions():
      interfaces = hook( mode, counterReset=counterReset ).interfaces
      aggregatedTapPortCounters.interfaces.update( interfaces )
   return aggregatedTapPortCounters

def doReadOrUpdateTapTunnelCounters( mode, args, counterReset ):
   tapTunnelCounters = TapAggModels.TunnelCounterModel()
   tapAggCliConfig.tapTunnelCounterClear = counterReset
   tapAggCliConfig.tapTunnelCounterUpdateRequestTime = Tac.now()
   for hook in tapTunnelCountersGetHook.extensions():
      tunnels = hook( mode ).tunnels
      tapTunnelCounters.tunnels.update( tunnels )
   return tapTunnelCounters

def doShowAllTapAggCounters( mode, args ):
   """For every tap port, print the corresponding default-forwarded counter stats."""
   return doReadOrUpdateTapPortCounters( mode, args, False )

def doShowAllTapTunnelCounters( mode, args ):
   """For every tap tunnel interface, print the corresponding counter stats for the
      GRE packets terminated."""
   return doReadOrUpdateTapTunnelCounters( mode, args, False )

def doResetAllTapPortCounters( mode, args ):
   """For every tap port, reset the corresponding default-forwarded counter stats."""
   return doReadOrUpdateTapPortCounters( mode, args, True )

def doResetAllTapTunnelCounters( mode, args ):
   """For every tap tunnel interface, reset the corresponding counter stats for the
      GRE packets terminated."""
   return doReadOrUpdateTapTunnelCounters( mode, args, True )

tapShowKw = CliCommand.guardedKeyword( 'tap',
                                       'Show interface TAP information',
                                       tapaggGuard )
toolShowKw = CliCommand.guardedKeyword( 'tool',
                                        'Show interface TOOL information',
                                        tapaggGuard )
detailShowKw = CliMatcher.KeywordMatcher( 'detail',
                                          helpdesc='More comprehensive output' )
tunnelShowKw = CliMatcher.KeywordMatcher( 'tunnel',
                                          helpdesc='Encapsulation stripping output' )

class ShowIntfTapTool( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces tap | tool [ detail | tunnel ]'
   data = dict( tap=tapShowKw,
                tool=toolShowKw,
                detail=detailShowKw,
                tunnel=tunnelShowKw )
   handler = doShowTapAggMode
   moduleAtEnd = True
   cliModel = TapAggModels.TapAggProperties

BasicCli.addShowCommandClass( ShowIntfTapTool )

def convertConfigToolGroupsFormat( mode ):
   tapAggCliConfig.toolGroupNewFormat = True

# Register convertConfigToolGroups via "config convert new-syntax"
CliPlugin.ConfigConvert.registerConfigConvertCallback(
                           convertConfigToolGroupsFormat )

def Plugin( entityManager ):
   global tapAggCliConfig, lagConfig, bridgingConfig, bridgingSwitchIntfConfigDir
   global hwSlice, ethIntfConfigDir, aclIntfConfig, tapAggStatus, tapAggHwStatus
   global tapAggHwConfig, tapAggIntfConfig, tapAggPmapConfig, intfStatusAll
   global intfHwStatus, cliNexthopGroupConfig, aegisPmapGroupConfig, intfInputAegis
   global nhgStatus, mcastNhgSelection

   tapAggCliConfig = ConfigMount.mount( entityManager, 'tapagg/cliconfig',
                                      'TapAgg::CliConfig', 'w' )
   lagConfig = LazyMount.mount( entityManager, "lag/config",
                                "Lag::Config", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingSwitchIntfConfigDir = LazyMount.mount( entityManager,
                                               "bridging/switchIntfConfig",
                                               "Bridging::SwitchIntfConfigDir", "r" )
   hwSlice = LazyMount.mount( entityManager, 'hardware/slice', 'Tac::Dir', 'r' )
   ethIntfConfigDir = LazyMount.mount( entityManager,
                                       "interface/config/eth/intf",
                                       "Interface::EthIntfConfigDir", "r" )
   aclIntfConfig = LazyMount.mount( entityManager, "acl/intf/config/cli",
                                    "Acl::IntfConfig", "r" )
   tapAggStatus = LazyMount.mount( entityManager, 'tapagg/status',
                                   'TapAgg::Status', 'r' )
   tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                     'TapAgg::HwStatus', 'r' )
   tapAggHwConfig = LazyMount.mount( entityManager, 'tapagg/hwconfig',
                                     'TapAgg::HwConfig', 'r' )
   tapAggIntfConfig = LazyMount.mount( entityManager, 'tapagg/intfconfig',
                                       'PolicyMap::IntfConfig', 'r' )
   tapAggPmapConfig = LazyMount.mount( entityManager, 'tapagg/pmapconfig',
                                       'TapAgg::PmapConfig', 'r' )
   intfStatusAll = LazyMount.mount( entityManager,
                                    "interface/status/all",
                                    "Interface::AllIntfStatusDir", "r" )
   mcastNhgSelection = LazyMount.mount( entityManager,
      'hardware/sand/tapagg/mcastNhgSelection',
      'Hardware::Sand::Danz::McastNhgSelection', 'r' )
   nhgStatus = LazyMount.mount( entityManager,
                                'hardware/ale/nexthopGroup/status/tapagg',
                                'Ale::NexthopGroupStatus', 'r' )
   intfStatusAll = LazyMount.mount( entityManager,
                                    "interface/status/all",
                                    "Interface::AllIntfStatusDir", "r" )
   intfHwStatus = LazyMount.mount( entityManager, "interface/hwstatus",
                                   "Interface::HwStatus", "r" )
   cliNexthopGroupConfig = LazyMount.mount( entityManager,
                                            "routing/nexthopgroup/input/cli",
                                            "Routing::NexthopGroup::ConfigInput",
                                            "r" )
   aegisPmapGroupConfig = LazyMount.mount( entityManager,
                                           "tapagg/pmapgroupconfig/SandAegis",
                                           "TapAgg::PmapGroupConfig",
                                           "r" )
   intfInputAegis = LazyMount.mount( entityManager,
                                "trafficPolicies/intf/input/aegis/",
                                "PolicyMap::IntfConfig",
                                "r" )
   mg = entityManager.mountGroup()
   redStatus = entityManager.lookup( Cell.path( 'redundancy/status' ) )
   ethPhyIntfStatusDir = mg.mount( "interface/status/eth/phy/all",
                                   "Interface::AllEthPhyIntfStatusDir", "r" )
   internalRecircIntfConfigDir = mg.mount( "interface/config/recirc",
                                           "Interface::InternalRecircIntfConfigDir",
                                           "r" )
   tapAggConfig = mg.mount( "tapagg/cliconfig",
                            "TapAgg::CliConfig", "r" )
   timestampingStatus = mg.mount( "interface/status/timestamping",
                                  "Timestamping::Status", "w" )

   def _createTimestampingStatusSm():
      if 'NO_TIMESTAMPING_SM' not in os.environ:
         arxSmControl = Tac.newInstance( "Arx::SmControl" )
         entityManager.timestampingSm = Tac.newInstance(
            'Timestamping::StatusAggregateSm', redStatus, ethPhyIntfStatusDir,
            internalRecircIntfConfigDir,
            tapAggConfig,
            timestampingStatus, arxSmControl )
   mg.close( _createTimestampingStatusSm )
