#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from ArnetModel import MacAddress
from CliModel import Int, Str, Enum, Bool
from CliModel import List
from IntfModels import Interface
from CliPlugin.PolicyMapModel import ActionModel
from CliPlugin.TapAggPmapCliLib import tacStripHdrType
import Intf.IntfRange

class TapAggGroupModel( ActionModel ):
   aggGroup = List( valueType=str,
            help="List of TapAgg Groups to flow the traffic ", optional=True)
   aggIntf = List( valueType=Interface,
            help="List of Tool Interfaces to flow the traffic", optional=True)

   def renderText( self ):
      if self.aggIntf:
         intfTypeList = Intf.IntfRange.intfListToCanonical( self.aggIntf )
         printIntfs = ','.join( intfTypeList )

      if self.aggGroup and self.aggIntf:
         return 'set group %s interface %s' % ( ','.join( sorted( self.aggGroup ) ),
                           printIntfs )
      elif self.aggGroup:
         return 'set group %s' % ','.join( sorted( self.aggGroup ) )
      else:
         return 'set interface %s' % printIntfs 

class TapAggIdTagModel( ActionModel ):
   idTag = Int( help='Id-tag value' )
   innerIdTag = Int( help='Inner id-tag value' )

   def renderText( self ):
      actionStr = 'set id-tag %d' % self.idTag
      if self.innerIdTag != 0:
         actionStr += ' inner %d' % self.innerIdTag
      return actionStr

class TapAggStripHdrModel( ActionModel ):
   hdrType = Enum( values=list(tacStripHdrType.attributes), help='Header type' )
   dot1qRemoveVlans = Str( help='VLAN indices to be removed' )
   # Add any info for the new header-types here.

   def renderText( self ):
      actionStr = ''
      if self.hdrType != tacStripHdrType.noStrip:
         actionStr += 'remove %s' % self.hdrType
         if self.hdrType == tacStripHdrType.dot1q:
            actionStr += ' outer %s' % self.dot1qRemoveVlans
         # Add handling for new header types here.
      return actionStr

class TapAggNexthopGroupModel( ActionModel ):
   nexthopGroup = List( valueType=str,
         help="List of nexthop-groups to flow the traffic ", optional=True )

   def renderText( self ):
      return 'set nexthop-group %s' % ','.join( sorted( self.nexthopGroup ) )

class TapAggTimestampHeaderModel( ActionModel ):
   def renderText( self ):
      return 'set mac timestamp header'

class TapAggDropActionModel( ActionModel ):
   def renderText( self ):
      return 'drop'

class TapAggMacAddressActionModel( ActionModel ):
   destMac = MacAddress( help='Destination MAC address' )
   srcMac = MacAddress( help='Source MAC address' )

   def renderText( self ):
      if self.srcMac:
         actionStr = 'set mac-address dest %s src %s' % ( self.destMac, self.srcMac )
      else:
         actionStr = 'set mac-address dest %s' % self.destMac
      return actionStr

class TapAggHeaderRemoveActionModel( ActionModel ):
   size = Int( help='The number of bytes to remove after the Ethernet header, '
                    'including any VLAN tags' )
   preserveEth = Bool( help='Preserve the original Ethernet header, including VLAN '
                            ' tags if any' )

   def renderText( self ):
      actionStr = 'remove header size %d' % self.size
      if self.preserveEth:
         actionStr += ' preserve ethernet'
      return actionStr

class TapAggActionSetModel( ActionModel ):
   actionSet = List( valueType=str, help="List of action sets" )

   def renderText( self ):
      return 'action %s' % ( ','.join( sorted( self.actionSet ) ) )

