# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import json
import re
import sys
import traceback

import ArPyUtils
import BasicCli
import BasicCliModes
import BasicCliSession
import CliApi
import CliCommand
import CliCommon
from CliMode.TechSupport import TechSupportPolicyMode
import CliModel
import CliMatcher
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import ConfigMount
import ShowCommand
import Tac
import TacSigint

cliConfig = None

#------------------------------------------------------------------------------------
# The "show tech-support" command
#------------------------------------------------------------------------------------
showTechCmds = []
showTechSummaryCmds = []
showTechCountersCmds = []
cmdCallbacks = []
# show tech-support summary commands
summaryCmdCallbacks = []
# show tech-support counters commands
countersCmdCallbacks = []
# Extra "show tech <this and that keyword> [debugging ...]" commands can be regist-
# ered. Keep the <this and that> keywords in this list (for dynamic keyword matcher)
extendedOptions = [] # list(path); path=list(token)

# Dictionary of Callback lists for each of the options under 'extended'
extendedOptCallbacks = {}

class ShowTechModel( CliModel.DeferredModel ):
   __public__ = False
   showTech = CliModel.Str( help="Dummy show tech-support model" )

def _getCmdCallback( cmds, cmdsGuard ):
   def _listFunc():
      return cmds

   def _guardedListFunc():
      if cmdsGuard():
         return cmds
      return []

   # there are no commands, so don't return a function
   if not cmds:
      return None

   # if we have a cmds guard then return the function that invokes the guard.
   # if it is unconditional return function that always returns the cmds list
   retFunc = _listFunc
   if cmdsGuard is not None:
      assert callable( cmdsGuard )
      retFunc = _guardedListFunc

   # We assume that we are being called from CliPlugin -> registerShowTechSupportCmd,
   # so the limit = 3.  If we are called via some other path, we don't try to
   # annotate the retFunc.
   callers = traceback.extract_stack( limit=3 )
   if len( callers ) == 3:
      if callers[ 1 ][ 2 ] == 'registerShowTechSupportCmd':
         fileName = callers[ 0 ][ 0 ]
         m = re.search( 'CliPlugin/(.*).py', fileName )
         if m:
            module = "CliPlugin." + m.group( 1 )
         else:
            module = fileName
         retFunc.__module__ = module
         retFunc.__name__ = callers[ 0 ][ 2 ]

   return retFunc

def registerShowTechSupportCmd( timeStamp, cmds=None, cmdsGuard=None, extended=None,
      summaryTimeStamp=None, summaryCmds=None, summaryCmdsGuard=None,
      countersTimeStamp=None, countersCmds=None, countersCmdsGuard=None ):
   """
   API used by other CliPlugins to register commands to be called when
   'show tech-support' is called.

   'timesStamp' is a string in the format returned by time.strftime(
   '%Y-%m-%d %H:%M:%S ), for example '2010-06-11 23:46:19'.
   Commands are run in the ascending order of the time stamps so that
   we get the effect of adding commands at the end as we add them in
   new releases over time.  When you add a new command, please
   generate a new time stamp to use using the following python code:

   import time
   print time.strftime( '%Y-%m-%d %H:%M:%S' )

   It is important that we try to keep the order of the list consistent from
   release to release, so please make sure to generate a real time stamp based
   on the current time when you add a call to this function.  Using a time stamp
   with granularity down to seconds should remove the need for any tie breakers.

   The format chosen here explicitly avoids any locale specific formatting
   so that our code is not sensitive to the setting of the LANG environment
   variable.  We validate that the string matches the desired format with
   a regexp, then sort the strings as strings.

   'cmds' is a list of commands that should be added to 'show tech-support'

   'cmdsGuard' is a function that returns if the commands that there
   added should be run. This function is executed each time 'show tech-support'
   is called rather than when the commands are registered

   To register new commands with 'show tech-support', do this in your
   CliPlugin:

   import CliPlugin.TechSupportCli as TechSupportCli
   TechSupportCli.registerShowTechSupportCmd(
      '2010-06-11 23:46:19',
      # This would register these commands unconditionally
      cmds=[ 'show chickens'
             'show donkeys' ] )

   summaryTimeStamp and summaryCmds work similar to timeStamp and
   cmds, but for "show tech-support summary" command. If None is passed for
   summaryTimeStamp, then it assumes the same value as timeStamp
   """

   # track all show tech commands
   if cmds and not extended:
      showTechCmds.extend( [ cmd.lower() for cmd in cmds ] )
   if summaryCmds:
      showTechSummaryCmds.extend( summaryCmds )
   if countersCmds:
      showTechCountersCmds.extend( [ cmd.lower() for cmd in countersCmds ] )

   cmdCallback = _getCmdCallback( cmds, cmdsGuard )
   summaryCmdCallback = _getCmdCallback( summaryCmds, summaryCmdsGuard )
   countersCmdCallback = _getCmdCallback( countersCmds, countersCmdsGuard )
   _registerShowTechSupportCmdCallback(
      timeStamp,
      cmdCallback=cmdCallback,
      extended=extended,
      summaryTimeStamp=summaryTimeStamp,
      summaryCmdCallback=summaryCmdCallback,
      countersTimeStamp=countersTimeStamp,
      countersCmdCallback=countersCmdCallback )

def _registerShowTechSupportCmdCallback( timeStamp, cmdCallback=None, extended=None,
                                         summaryTimeStamp=None,
                                         summaryCmdCallback=None,
                                         countersTimeStamp=None,
                                         countersCmdCallback=None ):
   assert re.match( r'\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d', timeStamp )
   if extended is not None:
      assert cmdCallback
   assert cmdCallback or summaryCmdCallback or countersCmdCallback

   if extended is None:
      if cmdCallback:
         cmdCallbacks.append( ( timeStamp, cmdCallback ) )
      if countersCmdCallback:
         if countersTimeStamp is None:
            countersTimeStamp = timeStamp
         countersCmdCallbacks.append( ( countersTimeStamp, countersCmdCallback ) )
      if summaryCmdCallback:
         if summaryTimeStamp is None:
            summaryTimeStamp = timeStamp
         summaryCmdCallbacks.append( ( summaryTimeStamp, summaryCmdCallback ) )
   else:
      tokens = extended.split( " " )
      if tokens not in extendedOptions:
         extendedOptions.append( tokens )
      if extended not in extendedOptCallbacks:
         extendedOptCallbacks[ extended ] = []
      extendedOptCallbacks[ extended ].append( ( timeStamp, cmdCallback ) )

def showTechSupportExec( mode, inputCmd, inCmdCallbacks, benchmark=None,
                         outputFormat=None, showAll=True ):
   def sortKey( timestampAndCallback ):
      return timestampAndCallback[ 0 ]

   benchmarkData = {}
   cmds = []
   for ( _, cmd ) in sorted( inCmdCallbacks, key=sortKey ):
      try:
         cmds.extend( cmd() )
      except Exception as e:      # pylint: disable-msg=broad-except
         # The callback to tell us what commands to run raised an exception.
         callbackName = f'plugin for {cmd.__module__}.{cmd.__name__}'
         logFileNum = mode.session.writeInternalError( cmd=callbackName )
         cmds.append( f'! {callbackName} raised an exception: {e!r}' )
         if logFileNum is not None:
            cmds.append( 'show error %d' % logFileNum )

   # add user configurable commands
   # Don't do this for 'show tech-support summary'
   # and 'show tech-support counters'.
   # These are supposed to have their own
   # include/exclude commands
   if inputCmd not in ( 'show tech-support summary',
                        'show tech-support counters' ):
      cmds.extend( cliConfig.includedShowTechCmds )

   capiExecutor = None
   firstCmd = True
   jsonFmt = mode.session.outputFormat_ == 'json' or outputFormat == 'json'
   if jsonFmt:
      session = BasicCliSession.Session( BasicCliModes.EnableMode,
                                    mode.session.entityManager,
                                    privLevel=CliCommon.MAX_PRIV_LVL,
                                    disableAaa=True,
                                    disableAutoMore=True,
                                    isEapiClient=True,
                                    shouldPrint=False,
                                    disableGuards=not mode.session.guardsEnabled(),
                                    interactive=False,
                                    cli=mode.session.cli,
                                    aaaUser=mode.session.aaaUser() )
      capiExecutor = CliApi.CapiExecutor( mode.session.cli, session,
                                          stateless=False )
      print( '{\n"%s": {' % inputCmd )
      sys.stdout.flush()

   seenCmds = set()
   for cmd in cmds:
      # Excluded commands are stored in lowercase to eliminate misses from users
      # capitalizing commands.
      cmdLowercase = cmd.lower()

      # if a command has been registered more than once then ignore subsequent
      # registrations
      if cmdLowercase in seenCmds:
         continue
      seenCmds.add( cmdLowercase )

      if not showAll and cmdLowercase in cliConfig.excludedShowTechCmds:
         continue

      if jsonFmt:
         if cmd.startswith( 'bash' ):
            # BASH commands aren't don't work with JSON
            continue

         if cmd in CliCommon.skippedJsonCmds:
            # commands are buggy, skip them
            continue

         if not showAll and cmdLowercase in cliConfig.excludedShowTechJsonCmds:
            continue

         if not firstCmd:
            print( ',' )
         print( '"%s": ' % cmd )
         sys.stdout.flush()
      else:
         print( '\n------------- %s -------------\n' % cmd )
         sys.stdout.flush()
      startTime = Tac.now()
      try:
         if jsonFmt:
            print( '{ "json":' )
            sys.stdout.flush()
            result = capiExecutor.executeCommands( [ CliApi.CliCommand( cmd ) ],
                                                   textOutput=False,
                                                   autoComplete=True,
                                                   streamFd=sys.stdout.fileno() )
            sys.stdout.flush()
            if result[ 0 ].status == CliApi.CapiStatus.NOT_CAPI_READY:
               with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) \
                      as capturedStdout:
                  capiExecutor.executeCommands( [ CliApi.CliCommand( cmd ) ],
                                                textOutput=True,
                                                autoComplete=True,
                                                streamFd=sys.stdout.fileno() )
               output = capturedStdout.contents()
               if isinstance( output, bytes ):
                  output = output.decode( 'utf-8', 'replace' )
               print( ', "text": { "output": %s }' % json.dumps( output ) )
            print( '}' )
            sys.stdout.flush()
         else:
            mode.session_.runCmd( cmd, aaa=False )
         firstCmd = False
      except CliParser.GuardError as e:
         if not jsonFmt:
            print( "(unavailable: %s)" % e.guardCode )
      except CliParser.AlreadyHandledError as e:
         mode.session_.handleAlreadyHandledError( e )
      except KeyboardInterrupt:
         raise
      except: # pylint: disable=bare-except
         # Catch all errors, so that one command failure doesn't cause
         # output from others to be skipped.
         #
         # NOTE NOTE NOTE!!
         #
         # If you change what is printed when a command fails, PLEASE
         # ALSO UPDATE src/Eos/ptest/ShowTechCli.py, which looks for
         # this string in the output of "show tech-support" on our
         # various platforms.
         if not jsonFmt:
            print( "(unavailable)" )
      TacSigint.check()

      if benchmark is not None:
         duration = Tac.now() - startTime
         benchmarkData[ cmd ] = duration

   if benchmark is not None:
      if jsonFmt:
         # TODO: this needs to be converted to print json as well
         # TODO: add test for benchmark and JSON
         pass
      else:
         print()
         print( '-' * 40 )
         print( 'Benchmark for top %d commands:' % benchmark )
         print( '-' * 40 )
         for cmd in sorted( benchmarkData,
                            key=lambda c: -benchmarkData[ c ] )[ : benchmark ]:
            print( f'{cmd!r} took {benchmarkData[ cmd ]:.2f} seconds' )

   if jsonFmt:
      print( '}\n}\n' )
      sys.stdout.flush()
   else:
      # Clear any errors, they pertain to the last run command only anyway; we don't
      # want ugly syslogs about show tech failures when last cmd was 'Not supported'
      # and the message itself was already printed anyway (since !json mode).
      mode.session_.clearMessages()

   return CliModel.noValidationModel( ShowTechModel )

#-------------------------------------------------------
# tech-support config commands
#
# From config mode
#    [ no | default ] management tech-support
#       [ no | default ] policy show tech-support
#          [ no | default ] exclude command CMD
#-------------------------------------------------------
class TechSupportMgmtMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = 'Tech-Support configuration'

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session,
                                             'tech-support' )

class ManagementShowTechConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'management tech-support'
   noOrDefaultSyntax = syntax
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'tech-support': 'Configure tech-support policy',
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TechSupportMgmtMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # TechSupportMgmtMode only has one submode and no command of its own.
      # Hence, call the no command handler of a single child here.
      GotoTechPolicyCmd.noOrDefaultHandler( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( ManagementShowTechConfigCmd )

#-------------------------------------------------------
# policy show tech-support
#-------------------------------------------------------
class ShowTechPolicyMode( TechSupportPolicyMode, BasicCliModes.ConfigModeBase ):
   name = 'show tech-support configuration'

   def __init__( self, parent, session ):
      TechSupportPolicyMode.__init__( self, 'show tech-support' )
      BasicCliModes.ConfigModeBase.__init__( self, parent, session )

class GotoTechPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy show tech-support'
   noOrDefaultSyntax = syntax
   data = {
            'policy': 'Configure tech-support policy',
            'show': 'Configure a show command policy',
            'tech-support': 'Configure show tech-support policy'
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ShowTechPolicyMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.excludedShowTechCmds.clear()
      cliConfig.excludedShowTechJsonCmds.clear()
      cliConfig.includedShowTechCmds.clear()

TechSupportMgmtMode.addCommandClass( GotoTechPolicyCmd )

#-------------------------------------------------------
# [ no | default ] exclude command [ json ] CMD
# "no|default exclude command" will clear all currently excluded commands
#-------------------------------------------------------
class ExcludeCommand( CliCommand.CliCommandClass ):
   syntax = 'exclude command [ json ] CMD'
   noOrDefaultSyntax = 'exclude command [ json ] [ CMD ]'
   data = {
            'exclude': 'Exclude command from "show tech-support"',
            'command': 'Exclude command from "show tech-support"',
            'json': 'Exclude command from "show tech-support | json"',
            'CMD': CliMatcher.StringMatcher( helpname='CMD',
                                               helpdesc='Command to exclude' )
          }

   @staticmethod
   def handler( mode, args ):
      excludeJson = args.get( 'json', False )
      excludeCmd = args[ 'CMD' ].lower()

      # Add a warning if there is no matching command
      if excludeCmd not in showTechCmds:
         mode.addWarning( 'No matched command' )

      excludeConfig = ( cliConfig.excludedShowTechJsonCmds if excludeJson else
                        cliConfig.excludedShowTechCmds )
      excludeConfig[ excludeCmd ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      excludeJson = args.get( 'json' )
      excludeCmd = args.get( 'CMD' )
      excludeCmd = excludeCmd.lower() if excludeCmd else None
      excludeConfig = ( cliConfig.excludedShowTechJsonCmds if excludeJson else
                        cliConfig.excludedShowTechCmds )

      if excludeCmd and excludeCmd not in excludeConfig:
         mode.addWarning( 'No matched command' )
         return

      if excludeCmd:
         del excludeConfig[ excludeCmd ]
      else:
         excludeConfig.clear()

ShowTechPolicyMode.addCommandClass( ExcludeCommand )

#-------------------------------------------------------
# [ no | default ] include command CMD
# "no|default include command" will clear all currently included commands
#-------------------------------------------------------
class IncludeCommand( CliCommand.CliCommandClass ):
   syntax = 'include command CMD'
   noOrDefaultSyntax = 'include command [ CMD ]'
   data = {
            'include': 'Include command in "show tech-support"',
            'command': 'Include command in "show tech-support"',
            'CMD': CliMatcher.StringMatcher( helpname='CMD',
                                               helpdesc='Command to include' )
          }

   @staticmethod
   def handler( mode, args ):
      includedCmd = args[ 'CMD' ]
      cliConfig.includedShowTechCmds[ includedCmd ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'CMD' not in args:
         cliConfig.includedShowTechCmds.clear()
         return
      includedCmd = args[ 'CMD' ]
      del cliConfig.includedShowTechCmds[ includedCmd ]

ShowTechPolicyMode.addCommandClass( IncludeCommand )

#------------------------------------------------------------------------------------
# Show Commands
#------------------------------------------------------------------------------------
techSupportKwMatcher = CliMatcher.KeywordMatcher( 'tech-support',
                        helpdesc='Show aggregated status and configuration details' )
extendedKwMatcher = CliMatcher.KeywordMatcher( 'extended',
                                    helpdesc='Show tech-support extended command' )
summaryKwMatcher = CliMatcher.KeywordMatcher(
    'summary', helpdesc='Show summarized status and configuration details' )
countersKwMatcher = CliMatcher.KeywordMatcher(
    'counters', helpdesc='Show counter details' )
extendedValueMatcher = CliMatcher.KeywordListsMatcher( extendedOptions,
                                     helpdesc="Extended Show tech-support for %s" )
#-----------------------------------------------------------------------------------
# show tech-support [ all | ( extended EXTENDED ) ]
#                   [ benchmark BENCHMARK ] [ json | text ]
#-----------------------------------------------------------------------------------

class ShowTechSupport( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tech-support
               [ summary | all | counters | ( extended EXTENDED ) ]
               [ benchmark BENCHMARK ] [ json | text ]'''
   data = {
            'tech-support': techSupportKwMatcher,
            'summary': summaryKwMatcher,
            'counters': countersKwMatcher,
            'all': 'Show all aggregated status and configuration details',
            'extended': extendedKwMatcher,
            'EXTENDED': extendedValueMatcher,
            'benchmark': CliCommand.Node(
                                 matcher=CliMatcher.KeywordMatcher( 'benchmark',
                                             helpdesc='Show command benchmark' ),
                                 hidden=True ),
            'BENCHMARK': CliCommand.Node(
                                 matcher=CliMatcher.IntegerMatcher( 1, 9999,
                                            helpdesc='Show top N slowest commands' ),
                                 hidden=True ),
            'json': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'json',
                                                         helpdesc='output in json' ),
                        hidden=True,
                        alias='OUTPUT_FORMAT' ),
            'text': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'text',
                                                         helpdesc='output in text' ),
                        hidden=True,
                        alias='OUTPUT_FORMAT' ),
          }
   cliModel = ShowTechModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      inCmdCallbacks = cmdCallbacks
      showAll = True
      extended = args.get( 'EXTENDED' )
      if 'all' in args:
         inputCmd = 'show tech-support all'
      elif extended is not None:
         extended = ' '.join( extended )
         inputCmd = 'show tech-support extended %s' % extended
         inCmdCallbacks = extendedOptCallbacks[ extended ]
      elif 'summary' in args:
         inputCmd = 'show tech-support summary'
         inCmdCallbacks = summaryCmdCallbacks
      elif 'counters' in args:
         inputCmd = 'show tech-support counters'
         inCmdCallbacks = countersCmdCallbacks
      else:
         inputCmd = 'show tech-support'
         showAll = False

      return showTechSupportExec( mode, inputCmd=inputCmd,
                                  inCmdCallbacks=inCmdCallbacks,
                                  benchmark=args.get( 'BENCHMARK' ),
                                  outputFormat=args.get( 'OUTPUT_FORMAT' ),
                                  showAll=showAll )

BasicCli.addShowCommandClass( ShowTechSupport )

#-----------------------------------------------------------------------------------
# show tech-support [ extended EXTENDED ] debugging
# Debugging help for what callbacks exist and what they return
#-----------------------------------------------------------------------------------
class ShowTechSupportDebugging( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support [ summary | counters | ( extended EXTENDED ) ] \
                                 debugging'
   data = {
            'tech-support': techSupportKwMatcher,
            'summary': summaryKwMatcher,
            'counters': countersKwMatcher,
            'extended': extendedKwMatcher,
            'EXTENDED': extendedValueMatcher,
            'debugging': 'Show debugging for tech-support callbacks',
          }
   privileged = True
   hidden = True

   @staticmethod
   def handler( mode, args ):
      extended = args.get( 'EXTENDED' )
      if 'summary' in args:
         callbacks = summaryCmdCallbacks
      elif 'counters' in args:
         callbacks = countersCmdCallbacks
      elif extended is None:
         callbacks = cmdCallbacks
      else:
         extended = ' '.join( extended )
         callbacks = extendedOptCallbacks[ extended ]
      for ( ts, cmd ) in sorted( callbacks, key=lambda t: t[ 0 ] ):
         try:
            val = cmd()
         except Exception as e:   # pylint: disable-msg=broad-except
            val = e
         print( f'{ts} {cmd.__module__}.{cmd.__name__}: {val!r}' )

BasicCli.addShowCommandClass( ShowTechSupportDebugging )

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, 'cli/config',
                                  'Cli::Config', 'w' )
