#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
from CliCommand import CliCommandClass

import BasicCli
import BasicCliModes
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.EnvironmentCli as EnvironmentCli
import Tracing

t0 = Tracing.trace0

def fanSpeedRange( mode, context=None ):
   # Check the speed falls within the range defined in the product FDL.
   # For products with multiple cooling domains, the overall minimum is taken.
   
   coolingDomains = EnvironmentCli.coolingDomainDir.coolingDomain
   minSpeed = min( [ domain.minFanSpeed for domain in coolingDomains.values() ],
                     default=30 )

   return ( minSpeed, 100 )

matcherRecoveryValue = CliMatcher.KeywordMatcher( 'restricted',
      helpdesc='Run EOS in reduced capacity until overheat '
               'condition is considered over' )
matcherAmbientThreshold = CliMatcher.KeywordMatcher( 'ambient-threshold',
   helpdesc="Ambient temperature (in Celsius) above which EOS"
            " will run in reduced capacity" )
matcherCelsius = CliMatcher.KeywordMatcher( 'celsius',
   helpdesc='Ambient-threshold in degrees Celsius' )
matcherAction = CliMatcher.KeywordMatcher( 'action',
   helpdesc='Configure the action to take' )
matcherActionValue = CliMatcher.EnumMatcher( {
   'ignore': 'Ignore the warning',
   'shutdown': 'Shutdown the system',
} )
matcherActionIgnore = CliMatcher.KeywordMatcher( 'ignore',
   helpdesc='Ignore the warning' )
matcherAuto = CliMatcher.KeywordMatcher( 'auto',
   helpdesc='Automatic fan speed mode' ) 
matcherEnvironment = CliMatcher.KeywordMatcher( 'environment',
   helpdesc='Configure environment parameters' )
matcherFanSpeed = CliMatcher.KeywordMatcher( 'fan-speed',
   helpdesc='Configure the system fan speed' )
matcherFanSpeedValue = CliMatcher.DynamicIntegerMatcher( fanSpeedRange,
   helpdesc='% of max fan speed' )
matcherIgnore = CliMatcher.KeywordMatcher( 'ignore',
   helpdesc='Ignore invalid temperature readings in fan speed calculations' )
matcherInsufficientFans = CliMatcher.KeywordMatcher( 'insufficient-fans',
   helpdesc='Configure system behavior when insufficient fans are inserted' )
matcherInvalid = CliMatcher.KeywordMatcher( 'invalid',
   helpdesc='Configure the system behavior for invalid temperatures' )
matcherMinimum = CliMatcher.KeywordMatcher( 'minimum',
   helpdesc='Set a minimum value for fan speed algorithm' )
matcherOverheat = CliMatcher.KeywordMatcher( 'overheat',
   helpdesc='Configure system overheat behavior' )
matcherOverride = CliMatcher.KeywordMatcher( 'override',
   helpdesc='Override the automatic fan speed algorithm' )
matcherTemperature = CliMatcher.KeywordMatcher( 'temperature',
   helpdesc='Configure the system behavior for temperatures' )
matcherTransceiver = CliMatcher.KeywordMatcher( 'transceiver',
   helpdesc='Configure the system behavior for transceiver temperatures' )
matcherThirdParty = CliMatcher.KeywordMatcher( 'third-party',
   helpdesc='Configure the system behavior for third-party modules' )
matcherPollInterval = CliMatcher.KeywordMatcher(
   'poll-interval',
   helpdesc='Configure the polling interval' )
matcherInterval = CliMatcher.IntegerMatcher( 0, 10000,
                                             helpdesc='Polling interval' )
matcherSeconds = CliMatcher.KeywordMatcher( 'seconds', helpdesc='Units seconds' )
matcherMilliseconds = CliMatcher.KeywordMatcher( 'milliseconds',
                                                 helpdesc='Units milliseconds' )
matcherCooling = CliMatcher.KeywordMatcher( 'cooling',
   helpdesc='Configure switch cards cooling' )
coolingZonesNode = CliCommand.Node( matcher=matcherCooling,
      guard=EnvironmentCli.coolingZonesGuard )
matcherCoolingZones = CliMatcher.KeywordMatcher( 'zones',
   helpdesc='Configure number of switch cards cooling zones' )
matcherCoolingZonesNum = CliMatcher.EnumMatcher(
      EnvironmentCli.getSupportedCoolingZones )

#--------------------------------------------------------------------------------
# environment cooling zones NUM
#--------------------------------------------------------------------------------
class EnvCoolingZonesCmd( CliCommand.CliCommandClass ):
   syntax = 'environment cooling zones NUM'
   noOrDefaultSyntax = syntax.replace( 'NUM', '...' )
   data = {
      'environment' : matcherEnvironment,
      'cooling' : coolingZonesNode,
      'zones' : matcherCoolingZones,
      'NUM' : matcherCoolingZonesNum,
   }
   handler = EnvironmentCli.setNumCoolingZones
   noOrDefaultHandler = EnvironmentCli.setDefaultNumCoolingZones

BasicCliModes.GlobalConfigMode.addCommandClass( EnvCoolingZonesCmd )

#--------------------------------------------------------------------------------
# environment fan-speed auto
#--------------------------------------------------------------------------------
class EnvFanSpeedAutoCmd( CliCommand.CliCommandClass ):
   syntax = 'environment fan-speed auto'
   noOrDefaultSyntax = syntax.replace( 'auto', '...' )
   data = {
      'environment' : matcherEnvironment,
      'fan-speed' : matcherFanSpeed,
      'auto' : 'Automatic fan speed mode',
   }
   handler = EnvironmentCli.autoFanSpeed
   noOrDefaultHandler = EnvironmentCli.autoFanSpeed

BasicCliModes.GlobalConfigMode.addCommandClass( EnvFanSpeedAutoCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment fan-speed minimum FANSPEED
#--------------------------------------------------------------------------------
class EnvFanSpeedMinimumFanSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'environment fan-speed minimum FANSPEED'
   noOrDefaultSyntax = syntax.replace( 'FANSPEED', '...' )
   data = {
      'environment' : matcherEnvironment,
      'fan-speed' : matcherFanSpeed,
      'minimum' : matcherMinimum,
      'FANSPEED' : matcherFanSpeedValue, 
   }
   handler = EnvironmentCli.setMinFanSpeed
   noOrDefaultHandler = EnvironmentCli.noMinFanSpeed

BasicCliModes.GlobalConfigMode.addCommandClass( EnvFanSpeedMinimumFanSpeedCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment fan-speed override FANSPEED
#--------------------------------------------------------------------------------
class EnvFanSpeedOverrideFanSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'environment fan-speed override FANSPEED'
   noOrDefaultSyntax = syntax.replace( 'FANSPEED', '...' )
   data = {
      'environment' : matcherEnvironment,
      'fan-speed' : matcherFanSpeed,
      'override' : matcherOverride,
      'FANSPEED' : matcherFanSpeedValue,
   }
   handler = EnvironmentCli.overrideFanSpeed
   noOrDefaultHandler = EnvironmentCli.autoFanSpeed

BasicCliModes.GlobalConfigMode.addCommandClass( EnvFanSpeedOverrideFanSpeedCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment insufficient-fans action ( ignore | shutdown )
#--------------------------------------------------------------------------------
class EnvInsufficientFansActionCmd( CliCommand.CliCommandClass ):
   syntax = 'environment insufficient-fans action ACTION'
   noOrDefaultSyntax = syntax.replace( 'ACTION', '...' )
   data = {
      'environment' : matcherEnvironment,
      'insufficient-fans' : matcherInsufficientFans,
      'action' : matcherAction,
      'ACTION' : matcherActionValue,
   }
   handler = EnvironmentCli.setShutdownOnInsufficientFans
   noOrDefaultHandler = EnvironmentCli.setShutdownOnInsufficientFans

BasicCliModes.GlobalConfigMode.addCommandClass( EnvInsufficientFansActionCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment overheat action ( ignore | shutdown | power-cycle )
#--------------------------------------------------------------------------------
class EnvOverheatActionCmd( CliCommand.CliCommandClass ):
   syntax = 'environment overheat action ( ignore | shutdown | power-cycle )'
   noOrDefaultSyntax = 'environment overheat action ...'

   _powerCycleData = {
      'environment' : matcherEnvironment,
      'overheat' : matcherOverheat,
      'action' : matcherAction,
      'power-cycle' : CliCommand.guardedKeyword(
                         'power-cycle',
                         guard=EnvironmentCli.powercycleActionGuard,
                         helpdesc='Power-cycle the system' ),
   }

   data = _powerCycleData | {
      'ignore' : matcherActionIgnore,
      'shutdown' : CliCommand.guardedKeyword(
                      'shutdown',
                      guard=EnvironmentCli.shutdownActionGuard,
                      helpdesc='Shutdown the system' ),
   }
   handler = EnvironmentCli.setActionOnOverheat
   noOrDefaultHandler = EnvironmentCli.setNoOrDefaultActionOnOverheat

BasicCliModes.GlobalConfigMode.addCommandClass( EnvOverheatActionCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment overheat action power-cycle recovery restricted
#--------------------------------------------------------------------------------
class EnvOverheatRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'environment overheat action power-cycle recovery restricted'
   noOrDefaultSyntax = 'environment overheat action power-cycle recovery ...'
   # pylint: disable=protected-access
   data = EnvOverheatActionCmd._powerCycleData | {
      'recovery' : CliCommand.guardedKeyword(
                     'recovery',
                     guard=EnvironmentCli.overheatRecoveryGuard,
                     helpdesc="Specify recovery behavior after power-cycle" ),
      'restricted' : matcherRecoveryValue,
   }
   handler = EnvironmentCli.setRecoveryOnOverheat
   noOrDefaultHandler = EnvironmentCli.setNoOrDefaultRecoveryOnOverheat

BasicCliModes.GlobalConfigMode.addCommandClass( EnvOverheatRecoveryCmd )

#--------------------------------------------------------------------------------
# ( no | default ) environment overheat action power-cycle recovery restricted
# ambient-threshold <INT> celsius
#--------------------------------------------------------------------------------
class EnvOverheatRecoveryAmbientThresholdCmd( CliCommand.CliCommandClass ):
   syntax = ( 'environment overheat action power-cycle recovery restricted '
            'ambient-threshold THRESHOLD celsius' )
   noOrDefaultSyntax = ( 'environment overheat action power-cycle recovery '
                       'restricted ambient-threshold ...' )
   data = EnvOverheatRecoveryCmd.data | {
      'ambient-threshold' : matcherAmbientThreshold,
      'THRESHOLD': CliMatcher.IntegerMatcher( 0, 100,
                        helpdesc='Temperature value in Celsius' ),
      'celsius': matcherCelsius,
   }
   handler = EnvironmentCli.setRecoveryAmbientThresholdOnOverheat
   noOrDefaultHandler = \
         EnvironmentCli.setNoOrDefaultRecoveryAmbientThresholdOnOverheat

BasicCliModes.GlobalConfigMode.addCommandClass( 
      EnvOverheatRecoveryAmbientThresholdCmd )

#--------------------------------------------------------------------------------
# '[ no | default ] environment temperature invalid action ignore'
#--------------------------------------------------------------------------------
class TempActionCmd( CliCommandClass ):
   syntax = "environment temperature invalid action ignore"
   noOrDefaultSyntax = syntax.replace( 'ignore', '...' )
   data = {
      'environment' : matcherEnvironment,
      'temperature' : matcherTemperature,
      'invalid' : matcherInvalid,
      'action' : matcherAction,
      'ignore' : matcherIgnore,
   }
   hidden = True
   handler = EnvironmentCli.setIgnoreTempTrue
   noOrDefaultHandler = EnvironmentCli.setIgnoreTempFalse

BasicCli.GlobalConfigMode.addCommandClass( TempActionCmd )

#--------------------------------------------------------------------------------
# '[ no | default ] environment temperature transceiver third-party invalid
#  action ignore'
#--------------------------------------------------------------------------------
class ThirdPartyXcvrTempActionCmd( CliCommandClass ):
   syntax = "environment temperature transceiver third-party invalid action ignore"
   noOrDefaultSyntax = syntax.replace( 'ignore', '...' )
   data = {
      'environment' : matcherEnvironment,
      'temperature' : matcherTemperature,
      'transceiver' : matcherTransceiver,
      'third-party' : matcherThirdParty,
      'invalid' : matcherInvalid,
      'action' : matcherAction,
      'ignore' : matcherIgnore,
   }
   hidden = True

   handler = EnvironmentCli.setIgnoreThirdPartyXcvrTrue
   noOrDefaultHandler = EnvironmentCli.setIgnoreThirdPartyXcvrFalse

BasicCli.GlobalConfigMode.addCommandClass( ThirdPartyXcvrTempActionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] environment temperature poll-interval
#--------------------------------------------------------------------------------
class TemperaturePollIntervalCmd( CliCommandClass ):
   syntax = ( "environment temperature poll-interval "
              "INTERVAL ( seconds | milliseconds )" )
   noOrDefaultSyntax = "environment temperature poll-interval ..."
   data = {
      'environment' : matcherEnvironment,
      'temperature' : matcherTemperature,
      'poll-interval' : matcherPollInterval,
      'INTERVAL' : matcherInterval,
      'seconds' : matcherSeconds,
      'milliseconds' : matcherMilliseconds,
   }

   handler = EnvironmentCli.setTemperaturePollInterval
   noOrDefaultHandler = EnvironmentCli.setTemperaturePollInterval

BasicCli.GlobalConfigMode.addCommandClass( TemperaturePollIntervalCmd )
