# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Arnet
import Tac
import Vlan
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model

class TopoStatus( Model ):
   unmappedVlans = List( valueType=int, help="VLAN numbers that are not mapped"
                         " to any topology" )

   class Topology( Model ):
      vlans = List( valueType=int, help="VLANs mapped in this topology" )

      class PortStatus( Model ):
         state = Enum( values=( 'discarding', 'learning', 'forwarding',
                                'portStateUnknown' ),  help="State of the port" )
         numChanges = Int( help="Number of state changes for this port"
                           " since the switch booted", optional=True )
         lastChange = Float( help="UNIX timestamp with millisecond precision"
                             " of the last state change for this port",
                             optional=True )

      interfaces = Dict( valueType=PortStatus,
                         help="Maps an interface name to its status" )

   topologies = Dict( valueType=Topology, help="Maps a topology name to its"
                      " topology information" )

   def render( self ):
      if self.unmappedVlans:
         print( "The following vlans are not mapped to any topology" )
         print( ", ".join( str( v ) for v in sorted( self.unmappedVlans ) ) )
         print()
      timeDelta = Tac.utcNow() - Tac.now()  # Adjustment between monotonic and UTC
      for name, topo in sorted( self.topologies.items() ):
         print( "Topology:", name )
         vlanSet = { v: None for v in topo.vlans }
         print( "  Mapped Vlans:", Vlan.vlanSetToCanonicalString( vlanSet ) )
         for portName in Arnet.sortIntf( topo.interfaces ):
            portStatus = topo.interfaces[ portName ]
            if ( portStatus.numChanges is not None
                 and portStatus.lastChange is not None ):
               lastChange = Ark.timestampToStr( portStatus.lastChange - timeDelta )
               # pylint: disable-next=consider-using-f-string
               details = ( "(%s changes, last %s)"
                           % ( portStatus.numChanges, lastChange ) )
            else:
               details = ""
            print ( "  %-21s %-10s %s" # pylint: disable=consider-using-f-string
                    # pylint: disable-next=consider-using-f-string
                    % ( "%s:" % portName, portStatus.state, details ) )
         print()
      print()
