#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from AclCliLib import genericIpProtocols, genericIp6Protocols
import BasicCli
import Cell
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
import Logging
import SharkLazyMount
import Tac
import Tracing
import CliToken.Clear
import MultiRangeRule
from ClassificationLib import ( extraIpv4Protocols, extraIpv6Protocols,
                                getKeywordMap, )
import CliParser
import ShowCommand
from CliMode.Classification import ( FieldSetVlanConfigMode,
                                     FieldSetIntegerConfigMode )
from CliMode.TrafficPolicy import ( ActionsConfigMode,
                                    MacActionsConfigMode,
                                    MatchRuleIpv4ConfigMode,
                                    MatchRuleIpv6ConfigMode,
                                    MatchRuleMacConfigMode,
                                    MatchRuleDefaultConfigMode,
                                    TrafficPoliciesConfigMode,
                                    TrafficPolicyConfigMode, FEATURE, FEATURE_SHORT,
                                    FieldSetL4PortConfigMode,
                                    FieldSetIpPrefixConfigMode,
                                    FieldSetIpv6PrefixConfigMode,
                                    FieldSetMacAddrConfigMode,
                                    PacketTypeConfigMode,
                                    ReplicateMode,
                                    EncapInnerVxlanMatchIpv4ConfigMode,
                                    EncapDzGreConfigMode )
from CliPlugin import ( PolicyMapCliLib,
                        MirroringPolicyMapClassModeTapAgg,
                        TapAggIntfCli,
                        TunnelIntfCli )
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.PbrCli import ( NexthopExprFactory, guardPbrNexthopVrf )
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.TrafficPolicyCliLib import (
                                  neighborsConfigConflictMsg,
                                  matchL4ProtocolConflictMsg )
from CliPlugin.TrafficPolicyCliLib import ( MatchRuleContext,
                                  InnerMatchRuleContext,
                                  PacketTypeContext,
                                  ReplicateContext,
                                  TrafficPolicyMatchRuleAction,
                                  PolicyConfigCmdBase, ActionType,
                                  protectedTrafficPolicyNamesRegex,
                                  ProtocolTcpFlagsOnlyCmd,
                                  InnerMatchCmdBase,
                                  encapTypeMatcher,
                                  encapTypeVxlanMatcher,
                                  EncapInnerMatchIpv4ConfigMode,
                                  EncapInnerMatchIpv6ConfigMode,
                                  EncapInnerMatchMacConfigMode,
                                  rateUnitTokens,
                                  burstUnitTokens,
                                  factors,
                                  UniqueId,
                                  DropWithCliMsgType,
                                  DropWithMessageType4,
                                  DropWithMessageType6,
                                  icmpMsgTypeTokenToDropWithEnum,
                                  icmpMsgTypeTokens,
                                  icmp6MsgTypeTokens,
                                  )
from CliPlugin.TrafficPolicyCliContext import TrafficPolicyContext
from CliPlugin.ClassificationCliContextLib import (
   registerFieldSetConfigValidCallbacks )
from CliPlugin.ClassificationCliLib import ( generateMultiRangeMatcher,
                                   generateFieldSetExpression,
                                   PrefixFieldSetCmdMatcher,
                                   PrefixCmdBaseV2,
                                   PrefixIpv4Cmd,
                                   PrefixIpv4CmdV2,
                                   PrefixIpv6CmdV2,
                                   PrefixIpv6Cmd,
                                   PrefixLpmIpv4Cmd,
                                   PrefixLpmIpv6Cmd,
                                   ProtocolIcmpV4ConfigCmd,
                                   ProtocolIcmpV6ConfigCmd,
                                   ProtocolIcmpV4TypeCodeConfigCmd,
                                   ProtocolIcmpV6TypeCodeConfigCmd,
                                   IpLengthConfigCmd,
                                   MatchAllFragmentConfigCmd,
                                   FragmentOffsetConfigCmd,
                                   IpOptionsConfigCmd,
                                   NumericalRangeConfigCmdBase,
                                   CommitAbortModelet,
                                   PrefixFieldSetCmdBase,
                                   SourceMatcher,
                                   FieldSetLimitConfigCmd,
                                   FieldSetL4RangeLimitConfigCmd,
                                   FieldSetRangeLimitConfigCmd,
                                   FieldSetSourceConfigCmdBase,
                                   FieldSetIpPrefixConfigBase,
                                   FieldSetIpv6PrefixConfigCmds,
                                   FieldSetIpPrefixExceptConfigCmds,
                                   FieldSetIpv6PrefixExceptConfigCmds,
                                   FieldSetIpPrefixBaseConfigCmd,
                                   FieldSetL4PortBaseConfigCmd,
                                   FieldSetL4PortConfigBase,
                                   FieldSetL4PortExceptConfigBase,
                                   FieldSetVlanBaseConfigCmd,
                                   FieldSetVlanConfigCmds,
                                   FieldSetVlanExceptConfigCmds,
                                   FieldSetVlanRangeLimitConfigCmd,
                                   FieldSetRefreshAllCmdBase,
                                   FieldSetIpPrefixRefreshCmdBase,
                                   FieldSetL4PortRefreshCmdBase,
                                   FieldSetVlanRefreshCmdBase,
                                   ProtocolFieldSetOredBaseCmd,
                                   generateTcpFlagCombinationExpression,
                                   generateTcpFlagExpression,
                                   protectedFieldSetNamesRegex,
                                   ProtocolOredBase,
                                   nameAndNumberExpression,
                                   VlanTagConfigCmd,
                                   VlanTagCmdMatcher,
                                   VlanTagConfigCmdBase,
                                   VlanConfigCmd,
                                   EthTypeConfigCmd,
                                   TeidConfigCmd,
                                   NexthopGroupMatchBaseConfigCmd,
                                   LocationValueMatchBaseConfigCmd,
                                   locationKwMatcher,
                                   DestinationSelfConfigCmd,
                                   FieldSetIntegerBaseConfigCmd,
                                   FieldSetIntegerConfigCmds,
                                   FieldSetIntegerExceptConfigCmds,
                                   FieldSetIntegerRefreshCmdBase,
                                   FieldSetMacAddrBaseConfigCmd,
                                   FieldSetMacAddrConfigBase,
                                   FieldSetMacAddrRefreshCmdBase,
                                   MacCmdBase,
                                   SrcDstMacAddrCmdMatcher,
                                   SrcDstMacAddrFieldSetCmdMatcher,
                                   TeidCmdMatcher,
                                   TeidConfigCmdBase, )
from CliPlugin.AclCliRules import aclMirrorSessionNameFn
from CliPlugin.TapAggIdentityMapDzGreMode import ( matcherId )
from CliPlugin.MirroringTapAggConfigMode import ( nodeDzGre, matcherDzGreSwitch )
from Arnet import EthAddr
from TrafficPolicyLib import structuredFilterToCmds
from Toggles.TrafficPolicyToggleLib import (
      toggleTrafficPolicyTeidMatchEnabled,
      toggleTrafficPolicyMirrorActionEnabled,
      toggleTrafficPolicyDefaultMacMatchEnabled,
      toggleTrafficPolicyDestSelfMatchEnabled,
      toggleTrafficPolicyVlanMatchEnabled,
      toggleTrafficPolicyActionSetsEnabled,
      toggleTrafficPolicyEnforceGtsmEnabled,
      toggleTrafficPolicyDropWithActionEnabled,
      toggleTrafficPolicyVxlanPacketTypeMatchEnabled,
      toggleTrafficPolicyMulticastPacketTypeMatchEnabled,
      toggleVaclTrafficPolicyCountActionEnabled,
      toggleTrafficPolicyProportionalLagPolicerEnabled,
)
from Toggles.ClassificationToggleLib import (
      toggleUdfEncapGtpMatchEnabled,
      toggleUdfEncapVxlanMatchEnabled,
      toggleFieldSetRefreshAllEnabled,
      toggleFieldSetAndResultsEnabled,
      toggleFieldSetControllerEnabled,
)
from Toggles.PolicyMapToggleLib import togglePolicyBasedVpnEnabled
from TypeFuture import TacLazyType
import six
import math

t0 = Tracing.trace0

IntfId = TacLazyType( 'Arnet::IntfId' )

policiesCliConfig = None
policiesIntfParamConfig = None
policiesStatusRequestDir = None
policiesStatus = None
policyOpChkr = None
qosHwStatus = None
qosAclHwStatus = None
entityManager = None
cpuCounter = None
intfInputCounter = None
intfOutputCounter = None
fieldSetConfig = None
urlFieldSetConfig = None
interfaceTrafficPolicyHwStatus = None
tapAggStatus = None
tapAggHwStatus = None
counterStatusInput = None
counterStatusOutput = None
cliNexthopGroupConfig = None
locationAliasConfig = None
routingHwStatus = None
tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup
matchMacAccessGroup = tacMatchOption.matchMacAccessGroup
tacNeighborProtocol = Tac.Type( 'Classification::NeighborProtocol' )
protocolNone = tacNeighborProtocol.protocolNone
protocolBgp = tacNeighborProtocol.protocolBgp
protocolBgpTtlSec = tacNeighborProtocol.protocolBgpTtlSec
tacMatchL4Protocol = Tac.Type( 'Classification::MatchL4Protocol' )
matchL4None = tacMatchL4Protocol.matchL4None
matchL4Bgp = tacMatchL4Protocol.matchL4Bgp
tacFieldConflict = Tac.Type( 'Classification::FieldConflict' )
conflictNeighborProtocol = tacFieldConflict.conflictNeighborProtocol
conflictMatchL4Protocol = tacFieldConflict.conflictMatchL4Protocol
ReservedClassMapNames = Tac.Type( 'TrafficPolicy::ReservedClassMapNames' )
ActionType = Tac.Type( "PolicyMap::ActionType" )
IdentityTuple = Tac.Type( 'Bridging::IdentityTuple' )
TacEthAddr = Tac.Type( 'Arnet::EthAddr' )
TacEthAddrTuple = Tac.Type( 'Arnet::EthAddrTuple' )
TapHeaderRemove = Tac.Type( 'Bridging::Input::TapHeaderRemove' )
UdfAnchorConfig = Tac.Type( 'Classification::UdfAnchorConfig' )
UdfOffsetBase = Tac.Type( 'Classification::UdfOffsetBase' )
UdfLength = Tac.Type( 'Classification::UdfLength' )
NotifMessageType = Tac.Type( "PolicyMap::NotifMessageType" )

matchOptionToType = {
   matchIpAccessGroup : "ipv4",
   matchIpv6AccessGroup : "ipv6",
   matchMacAccessGroup : "mac"
}

TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED = Logging.LogHandle(
   'TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED',
   severity=Logging.logError,
   fmt="Failed to import %s field-set %s from source %s. "
       "Possible reason: %s. Please check the source's validity.",
   explanation="This field-set is set up with an import source URL to load "
               "its contents. The source is either a local file or a remote "
               "URL and must be available at the time of import. The entries "
               "in the file must conform to the field-set entry syntax.",
   recommendedAction="Check the URL and contents of the file." )

TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED = Logging.LogHandle(
   'TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED',
   severity=Logging.logInfo,
   fmt="Successfully imported %s field-set %s from source %s.",
   explanation="This field-set is set up with an import source URL to load "
               "its contents. The source is either a local file or a remote "
               "URL and must be available at the time of import. The entries "
               "in the file must conform to the field-set entry syntax.",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

def getMatchRules( mode ):
   currCfg = mode.trafficPolicy
   if currCfg:
      return list( currCfg.rawClassMap )
   return []

def getActionSets( mode ):
   asetNames = []
   currCfg = mode.trafficPolicy
   if currCfg and mode.ruleName in currCfg.classAction:
      policyActions = currCfg.classAction[ mode.ruleName ].policyAction
      if ActionType.actionSet in policyActions:
         for name in policyActions[ ActionType.actionSet ].namedActionSet:
            asetNames.append( name.rsplit( '%' )[ -1 ] )
   return asetNames

def getTrafficPolicyCommands( trafficPolicyContext, l4PortSaveAll,
                              matchRuleContext=None, innerMatchRuleContext=None ):
   """
   Takes in a trafficPolicyContext and optional matchRuleContext and returns
   a list of command dictionaries. Each command dictionary contains all commands used
   to create a match rule in the traffic-policy.
   If a matchRuleContext is given, we  use the matchRuleContext to generate
   the command dictionary rather then using the corresponding match rule in the
   trafficPolicyContext. We place the command dictionary in the list of commands
   according to its seqnum.
   If innerMatchRuleContext is given, it represent we're in the encapInnerMatchMode,
   we, then, use the innerMatchRuleContext to generate the command dictionary used in
   the current mode and outerMatchRuleContext is used for outer match rules.
   """
   assert trafficPolicyContext is not None
   reservedNames = trafficPolicyContext.reservedClassMapNames()
   filterCmds = []
   trafficPolicy = trafficPolicyContext.currentPolicy()
   if not trafficPolicy:
      return filterCmds

   # currentRuleName is used to distinguish the current matchRuleContext with others.
   currentRuleName = ""
   if matchRuleContext:
      currentRuleName = matchRuleContext.ruleName

   def matchRuleToCmds( prio, ruleName, sfilter, actions, matchType ):
      cmds = structuredFilterToCmds( sfilter, actions, matchType, l4PortSaveAll )
      innerCmds = None
      innerFilter = None
      if innerMatchRuleContext and currentRuleName == ruleName:
         innerFilter = innerMatchRuleContext.filter
         innerCmds = structuredFilterToCmds( innerFilter, None, innerFilter.af,
                                             l4PortSaveAll )
         if innerMatchRuleContext.encapType:
            innerCmds[ 'encap' ] = 'encapsulation %s %s' % (
                  innerMatchRuleContext.encapType,
                  'mac' if innerFilter.af == 'ipunknown' else innerFilter.af )
         cmds[ "innerMatch" ] = innerCmds
      if sfilter.encapField and sfilter.encapField.encapType == 'dzgre':
         cmds[ "dzgreEncap" ] = 'encapsulation dzgre'
      cmds[ "ruleHeader" ] = "match %s %s" % ( ruleName, matchType )
      cmds[ "ruleName" ] = ruleName
      cmds[ "prio" ] = prio
      return cmds

   def addMatchRule( ruleName ):
      matchRule = trafficPolicy.rawClassMap[ ruleName ]
      classAction = trafficPolicy.classAction.get( ruleName )
      match = list( matchRule.match.values() )[ 0 ]
      matchOption = list( matchRule.match )[ 0 ]
      matchType = matchOptionToType[ matchOption ]
      cmds = matchRuleToCmds( prio, ruleName, match.structuredFilter,
                              classAction.policyAction, matchType )
      filterCmds.append( cmds )

   def addMatchRuleContextRule( matchRuleContext ):
      # Add given rule to its appriopriate position in the command list
      # based on its seqnum, removing the rule if it is already present
      overrideCmd = matchRuleToCmds( matchRuleContext.seqnum,
                                     matchRuleContext.ruleName,
                                     matchRuleContext.filter,
                                     matchRuleContext.matchRuleAction.actions(),
                                     matchRuleContext.matchType )
      # Remove previous entry, if any (happens when already commited rule is edited)
      # And add current entry being edited
      for i, cmd in enumerate( filterCmds ):
         if cmd[ "ruleName" ] == matchRuleContext.ruleName:
            filterCmds.pop( i )
            break

      index = len( filterCmds )
      if matchRuleContext.seqnum:
         for i, cmd in enumerate( filterCmds ):
            if cmd[ "prio" ] > matchRuleContext.seqnum:
               index = i
               break
      filterCmds.insert( index, overrideCmd )

   for prio, ruleName in six.iteritems( trafficPolicy.classPrio ):
      if ruleName in reservedNames:
         # These rules should be displayed at the end
         continue
      addMatchRule( ruleName )

   if matchRuleContext and matchRuleContext.ruleName not in reservedNames:
      # do not add reserved rules here, add them once all unreserved rules have
      # been handled
      addMatchRuleContextRule( matchRuleContext )

   for prio, ruleName in six.iteritems( trafficPolicy.classPrio ):
      if ruleName not in reservedNames:
         continue
      elif matchRuleContext and matchRuleContext.ruleName == ruleName:
         addMatchRuleContextRule( matchRuleContext )
      else:
         addMatchRule( ruleName )

   return filterCmds

def printCmd( cmd, spacing=3 ):
   if cmd:
      print( "%s%s" % ( spacing * " ", cmd ) )

def printMatchRule( cmds ):
   printCmd( cmds.get( "ruleHeader" ) )
   ethTypeCmds = cmds.get( "ethType", [] )
   for cmd in ethTypeCmds:
      printCmd( cmd, spacing=6 )
   printCmd( cmds.get( "description" ), spacing=6 )
   printCmd( cmds.get( "srcMacAddr" ), spacing=6 )
   printCmd( cmds.get( "srcMacAddrSet" ), spacing=6 )
   printCmd( cmds.get( "dstMacAddr" ), spacing=6 )
   printCmd( cmds.get( "dstMacAddrSet" ), spacing=6 )
   vlanTagCmds = cmds.get( "vlanTag", [] )
   for cmd in vlanTagCmds:
      printCmd( cmd, spacing=6 )
   vlanCmds = cmds.get( "vlan", [] )
   for cmd in vlanCmds:
      printCmd( cmd, spacing=6 )
   printCmd( cmds.get( "source" ), spacing=6 )
   printCmd( cmds.get( "srcPrefixSet" ), spacing=6 )
   printCmd( cmds.get( "destination" ), spacing=6 )
   printCmd( cmds.get( "dstPrefixSet" ), spacing=6 )
   printCmd( cmds.get( "dstSelf" ), spacing=6 )
   printCmd( cmds.get( "protoNeighborsBgp" ), spacing=6 )
   if toggleTrafficPolicyEnforceGtsmEnabled():
      printCmd( cmds.get( "protoNeighborsBgpTtlSec" ), spacing=6 )
   printCmd( cmds.get( "protoBgp" ), spacing=6 )
   protoFields = cmds.get( "protocol", [] )
   for p in protoFields:
      printCmd( p, spacing=6 )
   printCmd( cmds.get( "ttl" ), spacing=6 )
   printCmd( cmds.get( "cos" ), spacing=6 )
   printCmd( cmds.get( "dscp" ), spacing=6 )
   printCmd( cmds.get( "sport" ), spacing=6 )
   printCmd( cmds.get( "dport" ), spacing=6 )
   printCmd( cmds.get( "length" ), spacing=6 )
   printCmd( cmds.get( "fragment" ), spacing=6 )
   printCmd( cmds.get( "matchIpOptions" ), spacing=6 )
   printCmd( cmds.get( "matchTracked" ), spacing=6 )
   printCmd( cmds.get( "nexthopGroup" ), spacing=6 )
   printCmd( cmds.get( "applicationProfile" ), spacing=6 )
   printCmd( cmds.get( "serviceSet" ), spacing=6 )
   locationMatches = cmds.get( "location", [] )
   for lm in locationMatches:
      printCmd( lm, spacing=6 )
   dzgreCmds = cmds.get( "dzgreEncap" )
   if dzgreCmds:
      printCmd( "!", spacing=6 )
      printCmd( dzgreCmds, spacing=6 )
      printCmd( cmds.get( "dzGreSwitchId" ), spacing=9 )
      printCmd( cmds.get( "dzGrePortId" ), spacing=9 )
      printCmd( cmds.get( "dzGrePolicyId" ), spacing=9 )
   packetTypeCmds = cmds.get( "packetTypes", {} )
   if packetTypeCmds:
      printCmd( "!", spacing=6 )
      printCmd( "packet type", spacing=6 )
      for pktType in packetTypeCmds:
         printCmd( packetTypeCmds[ pktType ], spacing=9 )
   innerCmds = cmds.get( "innerMatch" )
   if innerCmds:
      printCmd( "!", spacing=6 )
      printCmd( innerCmds.get( "encap" ), spacing=6 )
      teidCmds = innerCmds.get( "teid", [] )
      for cmd in teidCmds:
         printCmd( cmd, spacing=9 )
      printCmd( innerCmds.get( "source" ), spacing=9 )
      printCmd( innerCmds.get( "srcPrefixSet" ), spacing=9 )
      printCmd( innerCmds.get( "destination" ), spacing=9 )
      printCmd( innerCmds.get( "dstPrefixSet" ), spacing=9 )
      innerProtoFields = innerCmds.get( "protocol", [] )
      for p in innerProtoFields:
         printCmd( p, spacing=9 )
   actions = cmds.get( "actions" )
   if actions:
      printCmd( "!", spacing=6 )
      printCmd( "actions", spacing=6 )
      for action in actions:
         if isinstance( action, tuple ):
            printCmd( f'replicate { action[ 0 ] }', spacing=9 )
            for asetAction in action[ 1 ]:
               printCmd( asetAction, spacing=12 )
         else:
            printCmd( action, spacing=9 )

#
# Field-set change status callback that returns the status of traffic-policies
# as a result of a field-set change.
#
def isPolicyHwStatusValid( mode, fieldSetName, fieldSetType ):
   isValid = True
   issues = {}
   issues[ 'error' ] = ''

   if not mode.session_.startupConfig():
      if fieldSetType == 'l4-port':
         fieldSetType = 'fieldSetL4Port'
      elif fieldSetType == 'ipv4':
         fieldSetType = 'fieldSetIp'
      elif fieldSetType == 'ipv6':
         fieldSetType = 'fieldSetIpv6'
      elif fieldSetType == 'vlan':
         fieldSetType = 'fieldSetVlan'
      elif fieldSetType == 'integer':
         fieldSetType = 'fieldSetInteger'
      elif fieldSetType == 'mac':
         fieldSetType = 'fieldSetMacAddr'
      elif fieldSetType == 'service':
         fieldSetType = 'fieldSetService'
      else:
         assert False, 'FieldSetType %s not supported' % fieldSetType

      isValid, result = policyOpChkr.verify( fieldSetType, fieldSetName )
      if not isValid:
         issues[ 'error' ] = ( result.error if result and result.error else
                               'unknown traffic-policy status' )
      else:
         if result and result.warning:
            issues[ 'warning' ] = result.warning
   return isValid, issues

class ShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = { 'pending': 'show pending traffic-policy' }

   @staticmethod
   def handler( mode, args ):
      trafficPolicyContext = getattr( mode, "trafficPolicyContext", None )
      matchRuleContext = getattr( mode, "context", None )
      innerMatchRuleContext = None
      if hasattr( mode, "matchRuleContext" ):
         innerMatchRuleContext = matchRuleContext
         matchRuleContext = mode.matchRuleContext
      l4PortSaveAll = getattr( mode, "l4PortSaveAll", True )
      cmdOutput = getTrafficPolicyCommands( trafficPolicyContext, l4PortSaveAll,
                                            matchRuleContext, innerMatchRuleContext )

      # Print traffic-policy name along with all pending/committed match rules
      printCmd( "%s %s" % ( mode.feature,
                            trafficPolicyContext.pmapName_ ),
                spacing=0 )
      # Print policyDesc before rules
      if trafficPolicyContext.currentPolicy().policyDesc:
         cmd = f'description {trafficPolicyContext.currentPolicy().policyDesc}'
         printCmd( cmd, spacing=3 )
      # Print named counters before rules
      if trafficPolicyContext.currentPolicy().namedCounter:
         counterCmd = "counter %s" % (
            " ".join( sorted( trafficPolicyContext.currentPolicy().namedCounter ) ) )
         printCmd( counterCmd, spacing=3 )
      for matchRule in cmdOutput[ : -1 ]:
         printMatchRule( matchRule )
         printCmd( "!", spacing=3 )
      if cmdOutput:
         printMatchRule( cmdOutput[ -1 ] )

matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Clear traffic-policy counters in all sessions' )
matcherTrafficPolicy = CliMatcher.KeywordMatcher( 'traffic-policy',
      helpdesc='Clear traffic-policy' )
matcherTrafficPolicyRefresh = CliMatcher.KeywordMatcher( 'traffic-policy',
      helpdesc='Refresh traffic-policy information' )
matcherSet = CliMatcher.KeywordMatcher( 'set',
      helpdesc='Set packet header or property' )
matcherMac = CliMatcher.KeywordMatcher( 'mac',
      helpdesc='MAC config commands' )

def getTrafficPolicyNames():
   if not policiesCliConfig:
      return []
   return iter( policiesCliConfig.pmapType.pmap )

trafficPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: getTrafficPolicyNames(), "Traffic Policy Name",
      pattern=protectedTrafficPolicyNamesRegex() )

pMapConstants = Tac.Value( 'Classification::Constants' )
ttlRangeMatcher = generateMultiRangeMatcher( 'ttl', pMapConstants.maxTtl )
dscpRangeMatcher = generateMultiRangeMatcher( 'dscp', pMapConstants.maxDscp )
cosRangeMatcher = generateMultiRangeMatcher( 'cos', pMapConstants.maxCos,
                                             helpdesc='Class of Service '
                                             '(CoS) value' )

# protocol number/range
protoRangeMatcherV4 = generateMultiRangeMatcher( 'protocol ',
                                                 minVal=pMapConstants.minProtoV4,
                                                 maxVal=pMapConstants.maxProto )

protoRangeMatcherV6 = generateMultiRangeMatcher( 'protocol ',
                                                 minVal=pMapConstants.minProtoV6,
                                                 maxVal=pMapConstants.maxProto )

dzGrePolicyIdRangeMatcher = \
      generateMultiRangeMatcher( 'dzGrePolicyId',
                                 minVal=1,
                                 maxVal=pMapConstants.maxDzGrePolicyId )

dzGrePortIdRangeMatcher = \
      generateMultiRangeMatcher( 'dzGrePortId',
                                 minVal=1,
                                 maxVal=pMapConstants.maxDzGrePortId )

dzGreSwitchIdRangeMatcher = \
      generateMultiRangeMatcher( 'dzGreSwitchId',
                                 minVal=1,
                                 maxVal=pMapConstants.maxDzGreSwitchId )

def generateIpProtoMatcher( name, genericProtocols, extraProtocols,
                            protoRangeMatcher ):
   nameMap = getKeywordMap( genericProtocols, extraProtocols )
   protoNameMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                        nameMap )
   return nameAndNumberExpression( name, protoRangeMatcher, protoNameMatcher,
                                   ( genericProtocols, extraProtocols ) )

ipv4ProtoMatcher = generateIpProtoMatcher( 'PROTOCOL', genericIpProtocols,
                                           extraIpv4Protocols, protoRangeMatcherV4 )
ipv6ProtoMatcher = generateIpProtoMatcher( 'PROTOCOL', genericIp6Protocols,
                                           extraIpv6Protocols, protoRangeMatcherV6 )

notMacOrTtl = "(?!(mac|ttl)$).*"

def getNexthopGroupNames( mode ):
   return cliNexthopGroupConfig.nexthopGroup.members()

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher(
   getNexthopGroupNames, 'nexthop group Name',
   pattern=notMacOrTtl )

def getLocationAliasNames( mode ):
   return locationAliasConfig.udf.members()

locationAliasNameMatcher = CliMatcher.DynamicNameMatcher(
   getLocationAliasNames, 'Location alias name',
   pattern=notMacOrTtl )

def guardAegisTtlAction( mode, token ):
   if interfaceTrafficPolicyHwStatus.aegisTtlActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def getAegisTtlRange( mode, token ):
   return ( interfaceTrafficPolicyHwStatus.aegisTtlActionMinValue,
            interfaceTrafficPolicyHwStatus.aegisTtlActionMaxValue )

# CLI expression for TTL
class TtlExpression( CliCommand.CliExpression ):
   expression = 'ttl TTL_VALUE'
   data = {
            'ttl' : CliCommand.guardedKeyword( 'ttl', 'Set header TTL value',
                                               guard=guardAegisTtlAction ),
            'TTL_VALUE' : CliMatcher.DynamicIntegerMatcher(
                                                     getAegisTtlRange,
                                                     helpdesc='TTL value' ),
          }

#--------------------------------------
# The "traffic-policies" mode command
#--------------------------------------
class TrafficPoliciesConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-policies'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-policies' : 'Configure traffic policies'
   }

   _noHandlers = []

   @staticmethod
   def _registerNoHandler( handler ):
      TrafficPoliciesConfigCmd._noHandlers.append( handler )

   handler = "TrafficPolicyCliHandler.doTrafficPoliciesHandler"
   noOrDefaultHandler = "TrafficPolicyCliHandler.noTrafficPoliciesHandler"

#---------------------------------------------------------------
# The "[no|default] traffic-policy <traffic-policy-name>" mode command
#---------------------------------------------------------------
class TrafficPolicyConfigCmd( PolicyConfigCmdBase ):
   syntax = 'traffic-policy POLICY_NAME [ copy SRC_POLICY_NAME ]'
   noOrDefaultSyntax = 'traffic-policy POLICY_NAME'
   data = {
      'traffic-policy' : 'Configure traffic policy',
      'POLICY_NAME' : trafficPolicyNameMatcher,
      'copy' : 'Copy from traffic policy',
      'SRC_POLICY_NAME' : trafficPolicyNameMatcher,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name, mode ):
      inConfigSession = mode.session_.inConfigSession()
      return TrafficPolicyContext( policiesCliConfig, policiesStatusRequestDir,
                                   policiesStatus, name,
                                   rollbackEnabled=not inConfigSession )

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'POLICY_NAME' ]
      context = cls._context( name, mode )
      if 'copy' not in args:
         if context.hasPolicy( name ):
            context.copyEditPmap()
         else:
            context.newEditPmap()
      else:
         srcName = args[ 'SRC_POLICY_NAME' ]
         errorMsg = context.copyPolicyMapFromSource( srcName )
         if errorMsg:
            mode.addError( errorMsg )
            return
      childMode = mode.childMode( context.childMode(),
                              context=context,
                              feature=cls._feature() )
      mode.session_.gotoChildMode( childMode )

# -----------------------------------------------------------------------------------
# The "packet type" command under match criteria
# -----------------------------------------------------------------------------------
class PacketTypeConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'packet type'
   noOrDefaultSyntax = syntax
   data = {
      'packet' : 'Configure packet type match',
      'type' : 'Packet type'
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @classmethod
   def _context( cls, mode ):
      return PacketTypeContext( mode.trafficPolicyContext,
            mode.trafficPolicyContext.matchRuleContext )

   @classmethod
   def handler( cls, mode, args ):
      childMode = mode.childMode( PacketTypeConfigMode,
            trafficPolicyContext=mode.trafficPolicyContext,
            matchRuleContext=mode.trafficPolicyContext.matchRuleContext,
            packetTypeContext=cls._context( mode ),
            feature=cls._feature() )

      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = cls._context( mode )
      context.delAllPacketType()
      context.commit()

class VxlanDecapCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan decap [ exclude ]'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan' : 'Configure VXLAN decapsulation match',
      'decap' : 'VXLAN Decapsulated packet type match',
      'exclude' : 'Non-VXLAN Decapsulated packet type match'
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @classmethod
   def handler( cls, mode, args ):
      exclude = 'exclude' in args
      mode.context.updatePacketTypeSet( 'vxlanDecap', exclude=exclude,
                                        no=False )
      mode.context.commit()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      exclude = args.get( 'exclude', "" )
      mode.context.updatePacketTypeSet( 'vxlanDecap', exclude=exclude,
                                        no=True )
      mode.context.commit()

class MulticastCmd( CliCommand.CliCommandClass ):
   syntax = 'multicast'
   noOrDefaultSyntax = syntax
   data = {
      'multicast' : 'Configure multicast packet type match'
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @classmethod
   def handler( cls, mode, args ):
      mode.context.updatePacketTypeSet( 'multicast', no=False )
      mode.context.commit()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.updatePacketTypeSet( 'multicast', no=True )
      mode.context.commit()

#--------------------------------------------------------------------------------
# The "encapsulation ENCAP_TYPE ipv4 command
#--------------------------------------------------------------------------------
class InnerMatchCmd( InnerMatchCmdBase ):
   syntax = ( 'encapsulation ENCAP_TYPE AF' )
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation' : 'Configure encapsulation',
      'ENCAP_TYPE' : encapTypeMatcher,
      'AF' : CliMatcher.EnumMatcher( { 'ipv4' : 'IPv4 inner match rule',
                                       'ipv6' : 'IPv6 inner match rule' } )
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @staticmethod
   def _context( trafficPolicyContext, matchRuleContext, encapType, matchOption ):
      return InnerMatchRuleContext( trafficPolicyContext, matchRuleContext,
                                    encapType, matchOption )

# --------------------------------------------------------------------------------
# The "encapsulation vxlan( ipv4 |mac )" command
# --------------------------------------------------------------------------------
class InnerMatchVxlanCmd( InnerMatchCmdBase ):
   syntax = ( 'encapsulation ENCAP_TYPE AF' )
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation' : 'Configure encapsulation',
      'ENCAP_TYPE' : encapTypeVxlanMatcher,
      'AF' : CliMatcher.EnumMatcher( {
         'mac' : 'Non-IP inner match rule',
         'ipv4' : 'IPv4 inner match rule' } )
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @staticmethod
   def _context( trafficPolicyContext, matchRuleContext, encapType, matchOption ):
      return InnerMatchRuleContext( trafficPolicyContext, matchRuleContext,
                                    encapType, matchOption )

#--------------------------------------------------------------------------------
# The "match RULE_NAME <ipv4 | ipv6>" mode command
#--------------------------------------------------------------------------------
matchRuleName = CliMatcher.DynamicNameMatcher( getMatchRules, "Match Rule Name" )

class MatchRuleConfigCmdBase( CliCommand.CliCommandClass ):
   @staticmethod
   def _feature():
      raise NotImplementedError

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      raise NotImplementedError

   @staticmethod
   def _ruleNameToSeq( trafficPolicyContext, ruleName ):
      '''
      Returns the seqNo for 'ruleName'. Assumes 'ruleName' is present
      (checked by cls._matchRulePresent).
      '''
      return trafficPolicyContext.ruleToSeq[
         TrafficPolicyMatchRuleAction.keyTag() ][ ruleName ]

   @staticmethod
   def _matchRulePresent( currentPolicy, ruleName, matchOption=None ):
      """ Checks if match rule is present and the match type has not changed """
      if not ruleName:
         return False

      if matchOption:
         return ruleName in currentPolicy.classAction and \
                currentPolicy.rawClassMap[ ruleName ].match.get( matchOption )
      else:
         return ruleName in currentPolicy.rawClassMap

   @staticmethod
   def _matchRuleNameReserved( ruleName ):
      reservedRuleNames = [ ReservedClassMapNames.classV4Default,
                            ReservedClassMapNames.classV6Default ]
      if toggleTrafficPolicyDefaultMacMatchEnabled():
         reservedRuleNames.append( ReservedClassMapNames.classMacDefault )
      return ruleName in reservedRuleNames

   @classmethod
   def _createMatchRuleContext( cls, mode, ruleName, matchOption, seqno=None,
                                beforeRule=None, afterRule=None ):
      """
      This routine creates (or fetches) the appropriate match rule context
      (MatchRuleContext) for the given CLI command. If we are creating a new rule, a
      new match rule context is needed. In other cases, we are editing an existing
      rule, in which case we also create a new context but copy in the existing rule.
      In yet other cases we are _moving_ an existing rule to another sequence number.
      This is identical to editing an existing rule. Finally, we may be attempting to
      move a rule that doesn't exist for the provided match option (e.g. ipv6 when we
      meant ipv4). In this case, we return the previous match rule context, and issue
      a CLI warning.

      @return context - This routine returns a match rule context, or None if one
      could not be created.
      """
      currentPolicy = mode.trafficPolicyContext.currentPolicy()
      matchRulePresent = cls._matchRulePresent( currentPolicy, ruleName,
                                                  matchOption )
      otherMatchOptions = dict( matchOptionToType )
      del otherMatchOptions[ matchOption ]
      otherMatchRulePresent = False
      otherMatchType = None

      for option in otherMatchOptions:
         if cls._matchRulePresent( currentPolicy, ruleName, option ):
            otherMatchRulePresent = True
            otherMatchType = otherMatchOptions[ option ]
            break

      # Here we check if we are attempting to edit or move an existing rule but
      # with the opposite match option. In this case, we issue a warning and
      # return to the previous mode.
      if otherMatchRulePresent:
         # We can never get in a scenario where both rules are present, even
         # transiently
         assert not matchRulePresent
         if seqno and ( beforeRule or afterRule ):
            if beforeRule:
               cmd = 'match %s %s before %s' % ( ruleName, otherMatchType,
                                                 beforeRule )
            else:
               assert afterRule
               cmd = 'match %s %s after %s' % ( ruleName, otherMatchType,
                                                afterRule )
            warning = "Trying to move match rule with incorrect match type. " \
                      "Did you mean '%s'?" % ( cmd )
         else:
            cmd = 'match %s %s' % ( ruleName, otherMatchType )
            warning = "Trying to edit match rule with incorrect match type. " \
                      "Did you mean '%s'?" % ( cmd )

         mode.addWarning( warning )
         return None

      matchRuleContext = cls._context( mode.trafficPolicyContext, ruleName,
                                       matchOption )

      if matchRulePresent:
         matchRuleContext.copyEditMatchRule( ruleName, seqno )
      else:
         matchRuleContext.newEditMatchRule( ruleName, seqno )

      if seqno and ( beforeRule or afterRule ):
         # We are placing a rule via before / after
         mode.trafficPolicyContext.shouldResequence = True
         if matchRulePresent:
            mode.trafficPolicyContext.moving = True

      return matchRuleContext

   @staticmethod
   def _getMatchOption( args ):
      if 'ipv4' in args:
         matchOption = matchIpAccessGroup
      elif 'ipv6' in args:
         matchOption = matchIpv6AccessGroup
      elif 'mac' in args:
         matchOption = matchMacAccessGroup
      else:
         matchOption = None
      return matchOption

   @classmethod
   def handler( cls, mode, args ):
      ruleName = args[ 'RULE_NAME' ]
      beforeRule = afterRule = None
      if 'before' in args:
         beforeRule = args[ 'REFERENCE_RULE' ]
      elif 'after' in args:
         afterRule = args[ 'REFERENCE_RULE' ]

      matchOption = cls._getMatchOption( args )
      seqno = None
      locationRule = beforeRule or afterRule

      # The default rules are always the last two rules, with the IPV4 rule first,
      # followed by the IPV6 rule. Rules can be moved _before_ the IPV4 rule but not
      # before the IPV6 or after either rule.
      if cls._matchRuleNameReserved( locationRule ):
         errorMsg = 'User-defined rules cannot be moved such that they follow ' \
                    'a built-in rule.'
         if afterRule or beforeRule \
            and locationRule == ReservedClassMapNames.classV6Default:
            mode.addError( errorMsg )
            return

      if cls._matchRuleNameReserved( ruleName ) and locationRule:
         errorMsg = 'The built-in rules cannot be moved.'
         mode.addError( errorMsg )
         return

      # Verify match option is consistent with the reserved rule name
      matchOptionToReservedRule = {
            matchIpAccessGroup : ReservedClassMapNames.classV4Default,
            matchIpv6AccessGroup : ReservedClassMapNames.classV6Default,
            matchMacAccessGroup : ReservedClassMapNames.classMacDefault,
      }
      if ( cls._matchRuleNameReserved( ruleName ) and
           matchOptionToReservedRule.get( matchOption ) != ruleName ):
         expectedMatchOption = {
               v : k for k, v in matchOptionToReservedRule.items()
         }.get( ruleName )
         expectedMatchType = matchOptionToType[ expectedMatchOption ]
         cmd = f'match {ruleName} {expectedMatchType}'
         warning = "Trying to edit match rule with incorrect match type. " \
                   f"Did you mean '{cmd}'?"
         mode.addWarning( warning )
         return

      # Reserved rules have a fixed sequnce number
      if cls._matchRuleNameReserved( ruleName ):
         prioMap = mode.trafficPolicyContext.defaultClassMapToClassPriority()
         seqno = prioMap[ ruleName ]

      # Ignore 'before / after' if the rule is the same as the current rule
      if locationRule != ruleName:
         currentPolicy = mode.trafficPolicyContext.currentPolicy()
         if cls._matchRulePresent( currentPolicy, locationRule ):
            seqno = cls._ruleNameToSeq( mode.trafficPolicyContext, locationRule )
         elif cls._matchRulePresent( currentPolicy, ruleName ):
            seqno = cls._ruleNameToSeq( mode.trafficPolicyContext, ruleName )

      # If we have before / after, simply insert at the seqno of the before /
      # after rule - / + 1, then resequence. This is slow in general, but simple,
      # and could be optimized easily.
      if seqno:
         if beforeRule:
            seqno -= 1
         elif afterRule:
            seqno += 1

      context = cls._createMatchRuleContext( mode, ruleName, matchOption,
                                             seqno=seqno, beforeRule=beforeRule,
                                             afterRule=afterRule )

      if not context:
         # We are trying to edit a match rule with the wrong match option,
         # so simply return
         return
      mode.trafficPolicyContext.matchRuleContext = context

      childMode = mode.childMode( context.childMode( ruleName, matchOption ),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                                  matchRuleContext=context,
                                  feature=cls._feature() )

      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      ruleName = args[ 'RULE_NAME' ]

      if not mode.trafficPolicyContext.supportsDefaultRuleRemoval():
         # First, check if the ruleName is eligible
         if cls._matchRuleNameReserved( ruleName ):
            errorMsg = "Only the actions of the built-in rules can be changed. " \
                       "Rule %s cannot be removed." % ruleName
            mode.addError( errorMsg )
            return

      matchRule = mode.trafficPolicy.rawClassMap.get( ruleName )
      matchOption = cls._getMatchOption( args )

      # Only remove the rule if the match option matches the named rule
      if ( matchRule and
           ( matchOption is None or matchRule.match.get( matchOption, None ) ) ):
         matchRuleAction = TrafficPolicyMatchRuleAction( mode.trafficPolicyContext,
                                                         ruleName,
                                                         matchOption,
                                                         None )
         mode.trafficPolicyContext.delRuleCommon( 0, matchRuleAction )

def macRuleMatchingGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.trafficPolicyMacMatchSupported:
      return None
   return CliParser.guardNotThisPlatform

class MatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = ( 'match RULE_NAME ( ipv4 | ipv6 | mac ) '
              '[ before | after REFERENCE_RULE ]' )
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 | mac ] ... '
   data = {
      'match' : 'Configure match rule',
      'RULE_NAME' : matchRuleName,
      'ipv4' : 'IPv4 match rule',
      'ipv6' : 'IPv6 match rule',
      'mac' : CliCommand.guardedKeyword( 'mac',
         helpdesc='MAC address match rule',
         guard=macRuleMatchingGuard ),
      'before' : 'Add this rule immediately before another rule',
      'after' : 'Add this rule immediately after another rule',
      'REFERENCE_RULE' : matchRuleName,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return MatchRuleContext( trafficPolicyContext, ruleName, matchOption )

# -----------------------------------------------------------------------------------
# Base 'actions <matchOption> default' Command Class
# -----------------------------------------------------------------------------------
class DefaultActionConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'actions ( ipv4 | ipv6 ) default'
   noOrDefaultSyntax = syntax
   data = {
      'actions': 'Configure policy actions',
      'ipv4': 'IPv4 match rule(s) actions',
      'ipv6': 'IPv6 match rule(s) actions',
      'default': 'Default actions',
   }

   @staticmethod
   def _feature():
      raise NotImplementedError

   @staticmethod
   def _context( trafficPolicyContext, matchOption ):
      raise NotImplementedError

   @staticmethod
   def _getMatchOption( args ):
      matchOption = None

      if 'ipv4' in args:
         matchOption = matchIpAccessGroup
      elif 'ipv6' in args:
         matchOption = matchIpv6AccessGroup
      elif 'mac' in args:
         matchOption = matchMacAccessGroup
      else:
         assert False, 'Invalid match option'
      return matchOption

   @classmethod
   def handler( cls, mode, args ):
      defaultActionsContext = cls._context( mode.trafficPolicyContext,
                                            cls._getMatchOption( args ) )
      childMode = mode.childMode( defaultActionsContext.childMode(),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                                  defaultActionsContext=defaultActionsContext,
                                  feature=cls._feature() )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.trafficPolicyContext.delDefaultActions( cls._getMatchOption( args ) )

#------------------------------------------------------------------------------------
# Base Action Command Class
#------------------------------------------------------------------------------------
class ActionCmdBase( CliCommand.CliCommandClass ):
   _actionType = None

   @classmethod
   def checkErrors( cls, mode, args ):
      errorStr = mode.context.matchRuleAction.actionCombinationError()
      if errorStr:
         mode.addError( errorStr )
      return errorStr is not None

   @classmethod
   def handler( cls, mode, args ):
      mode.context.setAction( cls._actionType, no=False )
      hasError = cls.checkErrors( mode, args )
      if hasError:
         # revert change
         mode.context.setAction( cls._actionType, no=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.setAction( cls._actionType, no=True )
      hasError = cls.checkErrors( mode, args )
      if hasError:
         # revert change
         mode.context.setAction( cls._actionType, no=False )

#------------------------------------------------------------------------------------
# The "drop" action
#------------------------------------------------------------------------------------
# Dynamic matcher functions for 'drop notification...'
def getVersionKw( mode, context ):
   matchOption = mode.context.matchRuleAction.matchOption
   if matchOption == matchIpv6AccessGroup:
      return { 'icmpv6' : 'Drop and send ICMPv6 message' }
   else:
      return { 'icmp' : 'Drop and send ICMP message' }

def getCodeKw( mode, context ):
   version = context.sharedResult[ 'VERSION' ]
   if '6' in version:
      return { 'code' : 'ICMPv6 destination-unreachable (type 1) code' }
   else:
      return { 'code' : 'ICMP destination-unreachable (type 3) code' }

def getIcmpCode( mode, context ):
   version = context.sharedResult[ 'VERSION' ]
   if '6' in version:
      return icmp6MsgTypeTokens
   else:
      return icmpMsgTypeTokens

# This is used to generate either 'drop' or 'drop [ notification... ]' commands
class DropActionExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, notif ):
      super().__init__()
      self.notif = notif

   def generate( self, name ):
      class DropActionExpr( CliCommand.CliExpression ):
         _notify = ' [ notification ( tcp reset ) | ( VERSION [ CODE_KW CODE ] ) ]'
         expression = 'drop'
         data = {
            'drop' : 'Drop action',
         }
         if self.notif:
            expression += _notify
            data.update( {
               'notification' : 'Drop and send message',
               'tcp' : DropWithCliMsgType.tcpReset.helpMsg,
               'reset' : DropWithCliMsgType.tcpReset.helpMsg,
               'VERSION' : CliCommand.Node(
                  CliMatcher.DynamicKeywordMatcher( getVersionKw, passContext=True ),
                  storeSharedResult=True ),
               'CODE_KW' : CliCommand.Node(
                  CliMatcher.DynamicKeywordMatcher( getCodeKw, passContext=True ) ),
               'CODE' : CliCommand.Node(
                  CliMatcher.DynamicKeywordMatcher( getIcmpCode, passContext=True ) )
            } )

         # Populate args[ 'MSG_TYPE' ] with PolicyMap::NotifMessageType Union for the
         # handler to use.
         @staticmethod
         def adapter( mode, args, argsList ):
            msgType = None
            # Regular drop case
            if 'notification' not in args:
               msgType = NotifMessageType( noMsg=True )
            elif 'reset' in args:
               matchOption = mode.context.matchRuleAction.matchOption
               if matchOption == matchIpv6AccessGroup:
                  enumVal = DropWithMessageType6.tcp6Reset
                  msgType = NotifMessageType( dropWith6=enumVal )
               else:
                  enumVal = DropWithMessageType4.tcpReset
                  msgType = NotifMessageType( dropWith4=enumVal )
            # Drop with ICMP. If no code specified, pick a default code
            else:
               msgTypeToken = args.get( 'CODE' )
               version = args[ 'VERSION' ]
               enumVal = icmpMsgTypeTokenToDropWithEnum[ version ].get( msgTypeToken,
                     # Use default ICMP codes if CODE not explicitly specified
                     ( DropWithMessageType6.icmp6NoRoute if '6' in version else
                       DropWithMessageType4.icmpNetUnreachable ) )
               if '6' in version:
                  msgType = NotifMessageType( dropWith6=enumVal )
               else:
                  msgType = NotifMessageType( dropWith4=enumVal )
            args[ 'MSG_TYPE' ] = msgType

      return DropActionExpr

# Regular 'drop' and 'drop [ notification... ]' commands can share handlers. Their
# syntax and data differ and are generated using DropActionExprFactory
class DropActionCmdBase( ActionCmdBase ):
   _actionType = ActionType.deny

   syntax = 'DROP_EXPR'
   noOrDefaultSyntax = 'DROP_EXPR'

   handler = "TrafficPolicyCliHandler.dropActionCmdHandler"
   noOrDefaultHandler = "TrafficPolicyCliHandler.dropActionCmdNoOrDefaultHandler"

# The 'drop' action. Other features (e.g. FlowPinHole) may need to use this, as they
# don't support 'drop notification'
class DropActionWithoutNotifCmd( DropActionCmdBase ):
   data = {
      'DROP_EXPR' : DropActionExprFactory( notif=False )
   }

# 'drop [ notification ( tcp reset ) | ( ( icmp | icmp6 ) [ code CODE ] ) ]'
class DropActionWithNotifCmd( DropActionCmdBase ):
   data = {
      'DROP_EXPR' : DropActionExprFactory( notif=True )
   }

# --------------------------------------------------------------------------------
# set mac timestamp header
# --------------------------------------------------------------------------------
def ingressTimestampActionGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.ingressTimestampActionSupported:
      return None
   return CliParser.guardNotThisPlatform

class SetTimestampHeaderActionCmd( ActionCmdBase ):
   _actionType = ActionType.setTimestampHeader

   syntax = "set mac timestamp header"
   noOrDefaultSyntax = syntax
   data = {
      'set' : matcherSet,
      'mac' : matcherMac,
      'timestamp' : CliCommand.guardedKeyword(
         'timestamp', helpdesc='Configure timestamp properties',
         guard=ingressTimestampActionGuard ),
      'header' : 'Insert timestamp into the Ethernet header',
   }

#------------------------------------------------------------------------------------
# The "counter { NAMES }" command for policy counter definitions
#------------------------------------------------------------------------------------
def getNamedCounters( mode ):
   currCfg = mode.trafficPolicy
   if currCfg:
      return list( currCfg.namedCounter )
   return []

availNamedCounters = CliMatcher.DynamicNameMatcher( getNamedCounters,
                                                    "Counter name" )

class NamedCounterCmd( CliCommand.CliCommandClass ):
   syntax = '''counter { NAMES }'''
   noOrDefaultSyntax = '''counter [ { NAMES } ]'''
   data = {
      'counter' : 'counter',
      'NAMES' : availNamedCounters
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      counterNames = args[ 'NAMES' ]
      context.updateNamedCounters( counterNames, add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      counterNames = args.get( 'NAMES', [] )
      context.updateNamedCounters( counterNames, add=False )

#------------------------------------------------------------------------------------
# The "count [ NAME ]" action
#------------------------------------------------------------------------------------
class CountActionCmd( ActionCmdBase ):
   _actionType = ActionType.count

   syntax = '''count [ NAME ]'''
   noOrDefaultSyntax = syntax
   data = {
      'count' : 'Configure count action',
      'NAME' : availNamedCounters
   }

   @classmethod
   def handler( cls, mode, args ):
      # set the counter name if present
      namedCounter = args.get( "NAME", "" )
      mode.context.setAction( cls._actionType, namedCounter, no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      namedCounter = args.get( "NAME", "" )
      mode.context.setAction( cls._actionType, namedCounter, no=True )


#------------------------------------------------------------------------------------
# The "log" action
#------------------------------------------------------------------------------------
class LogActionCmd( ActionCmdBase ):
   _actionType = ActionType.log

   syntax = '''log'''
   noOrDefaultSyntax = '''log'''
   data = {
      'log' : 'Configure log action',
   }

#------------------------------------------------------------------------------------
# The "set dscp DSCP_VALUE" action
#------------------------------------------------------------------------------------
class SetDscpActionCmd( ActionCmdBase ):
   _actionType = ActionType.setDscp

   syntax = '''set dscp DSCP_VALUE'''
   noOrDefaultSyntax = '''set dscp ...'''
   data = {
      'set' : matcherSet,
      'dscp' : 'Set header DSCP',
      'DSCP_VALUE' : CliMatcher.IntegerMatcher( 0,
                                                63,
                                                helpdesc='specify dscp value' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      dscpValue = args.get( 'DSCP_VALUE' )
      mode.context.setAction( cls._actionType, dscpValue, no=False )

#------------------------------------------------------------------------------------
# The "mirror session SESSION_NAME" action
#------------------------------------------------------------------------------------
class MirrorActionCmd( ActionCmdBase ):
   _actionType = ActionType.mirror

   syntax = 'mirror session SESSION_NAME'
   noOrDefaultSyntax = 'mirror session ...'
   data = {
      'mirror' : 'Configure mirror action',
      'session' : 'Set mirror session',
      'SESSION_NAME' : CliMatcher.DynamicNameMatcher( aclMirrorSessionNameFn,
                                                      helpdesc='Session name' )
   }

   @classmethod
   def handler( cls, mode, args ):
      sessionName = args[ 'SESSION_NAME' ]
      mode.context.setAction( cls._actionType, sessionName, no=False )

# -----------------------------------------------------------------------------------
# The "sflow" action
# -----------------------------------------------------------------------------------
class SflowActionCmd( ActionCmdBase ):
   _actionType = ActionType.sflow

   syntax = 'sflow'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : 'Configure sflow action',
   }

#------------------------------------------------------------------------------------
# The "set traffic class TRAFFIC_CLASS" action
#------------------------------------------------------------------------------------
def tcRangeFunc( mode, context ):
   return ( 0, qosHwStatus.numTcSupported - 1 )

class SetTcActionCmd( ActionCmdBase ):
   _actionType = ActionType.setTc

   syntax = '''set traffic class TRAFFIC_CLASS'''
   noOrDefaultSyntax = '''set traffic class ...'''
   data = {
      'set' : matcherSet,
      'traffic' : 'Set traffic management property',
      'class' : 'Set forwarding traffic class',
      'TRAFFIC_CLASS' : CliMatcher.DynamicIntegerMatcher(
         tcRangeFunc, helpdesc='specify traffic class' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      tc = args.get( 'TRAFFIC_CLASS' )
      mode.context.setAction( cls._actionType, tc, no=False )

class ReplicateActionCmd( ActionCmdBase ):
   _actionType = ActionType.actionSet

   syntax = '''replicate ACTIONSET_NAME'''
   noOrDefaultSyntax = '''replicate ACTIONSET_NAME'''
   data = {
      'replicate' : 'Create an action set',
      'ACTIONSET_NAME' : CliMatcher.DynamicNameMatcher(
         getActionSets,
         helpdesc='Action set name'
      )
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @classmethod
   def handler( cls, mode, args ):
      actionSetName = args[ 'ACTIONSET_NAME' ]

      context = ReplicateContext( mode.context, mode, actionSetName )
      childMode = mode.childMode( ReplicateMode,
                                  context=context,
                                  feature=cls._feature()
      )
      mode.context.setAction( cls._actionType, actionSetName, no=False )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      actionSetName = args.get( 'ACTIONSET_NAME', '' )
      mode.context.setAction( cls._actionType, actionSetName, no=True )

#------------------------------------------------------------------------------------
# The "set idtag IDTAG_VALUE" action
#------------------------------------------------------------------------------------
class SetIdTagActionCmd( ActionCmdBase ):
   _actionType = ActionType.setIdentityTag

   syntax = '''set id-tag ( ( IDTAG_VALUE [ inner INNER_IDTAG_VALUE ] )
               | IDTAG_EXT_VALUE )'''
   noOrDefaultSyntax = '''set id-tag ...'''
   data = {
      'set' : matcherSet,
      'id-tag' : 'Identity tag',
      'IDTAG_VALUE' : CliMatcher.IntegerMatcher(
         1,
         4094,
         helpdesc='Dot1Q tag identifying the flow matching this policy' ),
      'IDTAG_EXT_VALUE' : CliCommand.Node( matcher=CliMatcher.IntegerMatcher(
         1,
         65535,
         helpdesc='Extended ID identifying the flow matching this policy',
         priority=CliParser.PRIO_LOW ),
         guard=TapAggIntfCli.dzGreEncodeSupportedGuard ),
      'inner' : CliCommand.guardedKeyword(
         'inner',
         helpdesc='Inner identity tag',
         guard=TapAggIntfCli.qinqIdentityTaggingGuard ),
      'INNER_IDTAG_VALUE' : CliMatcher.IntegerMatcher(
         1,
         4094,
         helpdesc='Inner Dot1Q tag for flow identity' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      outerId = args.get( 'IDTAG_VALUE', 0 )
      if not outerId:
         outerId = args.get( 'IDTAG_EXT_VALUE', 0 )
      idTagValue = IdentityTuple( outerId,
                                args.get( 'INNER_IDTAG_VALUE', 0 ) )
      mode.context.setAction( cls._actionType, idTagValue, no=False )

# -----------------------------------------------------------------------------------
# The "set mac address dest <DMAC> [ src <SMAC> ]" action
# -----------------------------------------------------------------------------------
class SetMacAddressActionCmd( ActionCmdBase ):
   _actionType = ActionType.setMacAddress

   syntax = '''set mac address dest DMAC [ src SMAC ]'''
   noOrDefaultSyntax = '''set mac address ...'''
   data = {
      'set' : matcherSet,
      'mac' : matcherMac,
      'address' : 'MAC addresses',
      'dest' : TapAggIntfCli.matcherMacAddressDest,
      'DMAC' : macAddrMatcher,
      'src' : TapAggIntfCli.matcherMacAddressSrc,
      'SMAC' : macAddrMatcher,
   }

   @classmethod
   def handler( cls, mode, args ):
      if EthAddr( args[ 'DMAC' ] ) == TacEthAddr():
         mode.addError( 'MAC address 0000.0000.0000 is not supported' )
         return
      elif 'SMAC' in args and EthAddr( args[ 'SMAC' ] ) == TacEthAddr():
         mode.addError( 'MAC address 0000.0000.0000 is not supported' )
         return
      macAddrValue = TacEthAddrTuple( args[ 'DMAC' ],
                                   args.get( 'SMAC', TacEthAddr() ) )
      mode.context.setAction( cls._actionType, macAddrValue, no=False )

# -----------------------------------------------------------------------------------
# The "set vrf secondary" action
# -----------------------------------------------------------------------------------
class SetVrfSecondaryActionCmd( ActionCmdBase ):
   _actionType = ActionType.useVrfSecondary

   syntax = 'set vrf secondary'
   noOrDefaultSyntax = syntax
   data = {
      'set' : matcherSet,
      'vrf' : 'Set destination VRF',
      'secondary' : 'Set destination based on secondary VRF lookup',
   }

def policerRateRangeFn( mode, context ):
   return ( 1, qosAclHwStatus.policeByteModeMaxRateKbps * 1000 )

def policerBurstSizeRangeFn( mode, context ):
   return ( 1, qosAclHwStatus.policeByteModeMaxBurstBytes )

#------------------------------------------------------------------------------------
# The "police rate RATE_VALUE [ RATE_UNIT ]
#     [ burst-size BURST [ BURST_UNIT ] ]" action
#------------------------------------------------------------------------------------
class PoliceActionCmd( ActionCmdBase ):
   _actionType = ActionType.police

   syntax = '''police rate RATE_VALUE [ RATE_UNIT ]
               [ burst-size BURST [ BURST_UNIT ] ]'''
   noOrDefaultSyntax = '''police ...'''
   data = {
      'police' : 'Configure policer parameters',
      'rate' : 'Set rate limit',
      'RATE_VALUE' : CliMatcher.DynamicIntegerMatcher(
                        rangeFn=policerRateRangeFn,
                        helpdesc='Specify rate limit value (Default Unit is Kbps)' ),
      'RATE_UNIT' : CliMatcher.EnumMatcher( rateUnitTokens ),
      'burst-size' : 'Set burst-size for rate limit',
      'BURST' : CliMatcher.DynamicIntegerMatcher(
                       rangeFn=policerBurstSizeRangeFn,
                       helpdesc='Specify burst size value (Default Unit is bytes)' ),
      'BURST_UNIT' : CliMatcher.EnumMatcher( burstUnitTokens ),
   }

   @staticmethod
   def _policerOutOfRangeCheck( mode, policeAction ):
      outOfRange = False
      if mode.session_.startupConfig():
         # Do not check this during startup
         return outOfRange
      rateValue = policeAction[ 0 ].rateValue
      rateUnit = policeAction[ 0 ].rateUnit
      if rateUnit == 'pps':
         rateLowerLimit = qosAclHwStatus.policePacketModeMinRatePps
         rateUpperLimit = qosAclHwStatus.policePacketModeMaxRatePps
      else:
         rateLowerLimit = qosAclHwStatus.policeByteModeMinRateKbps
         rateUpperLimit = qosAclHwStatus.policeByteModeMaxRateKbps
         # Representing rateLowerLimit and rateUpperLimit values in rateUnit
         rateLowerLimit = int( math.ceil( rateLowerLimit * factors[ rateUnit ] ) )
         rateUpperLimit = int( math.floor( rateUpperLimit * factors[ rateUnit ] ) )
      if not rateLowerLimit <= rateValue <= rateUpperLimit:
         mode.addError( 'Invalid rate limit value\n'
                        'Configurable rate value range in %s is <%d-%d>' % (
                        rateUnit, rateLowerLimit, rateUpperLimit ) )
         outOfRange = True
      burstValue = policeAction[ 1 ].burstSize
      if burstValue == 0:
         return outOfRange
      burstUnit = policeAction[ 1 ].burstUnit
      if burstUnit == 'packets':
         burstLowerLimit = max( 1, qosAclHwStatus.policePacketModeMinBurstPackets )
         burstUpperLimit = qosAclHwStatus.policePacketModeMaxBurstPackets
      else:
         burstLowerLimit = max( 1, qosAclHwStatus.policeByteModeMinBurstBytes )
         burstUpperLimit = qosAclHwStatus.policeByteModeMaxBurstBytes
         # Representing burstLowerLimit and burstUpperLimit values in burstUnit
         burstLowerLimit = int( math.ceil( burstLowerLimit * factors[ burstUnit ] ) )
         burstUpperLimit = int( math.floor(
            burstUpperLimit * factors[ burstUnit ] ) )
      if not burstLowerLimit <= burstValue <= burstUpperLimit:
         mode.addError( 'Invalid burst size value\n'
                        'Configurable burst size in %s is <%d-%d>' % (
                        burstUnit, burstLowerLimit, burstUpperLimit ) )
         outOfRange = True
      return outOfRange

   @classmethod
   def handler( cls, mode, args ):
      rateValue = args[ 'RATE_VALUE' ]
      rateUnit = args.get( 'RATE_UNIT', 'kbps' )
      burstSize = args.get( 'BURST', 0 )
      burstUnit = args.get( 'BURST_UNIT', 'bytes' )
      policeRate = Tac.newInstance( "PolicyMap::PoliceRate", rateValue, rateUnit )
      policeBurstSize = Tac.newInstance( "PolicyMap::PoliceBurstSize", burstSize,
                                         burstUnit )
      policeAction = [ policeRate, policeBurstSize ]
      if PoliceActionCmd._policerOutOfRangeCheck( mode, policeAction ):
         # Do nothing for out-of-range police rate configuration
         return
      mode.context.setAction( cls._actionType, policeAction, no=False )

class GotoActionCmd( ActionCmdBase ):
   _actionType = ActionType.actionGoto

   syntax = 'goto ( next | ( match REFERENCE_RULE ) )'
   noOrDefaultSyntax = 'goto ...'
   data = {
      'goto' : 'Configure goto rule to match next',
      'next' : 'Match on the next rule',
      'match' : 'Match on the rule to be specified next',
      'REFERENCE_RULE' : matchRuleName,
   }

   @classmethod
   def handler( cls, mode, args ):
      ruleNameToGoto = args.get( 'REFERENCE_RULE', '' )
      mode.context.setAction( cls._actionType,
                              actionValue=ruleNameToGoto,
                              no=False )

#--------------------------------------------------------------------------------
# remove dot1q outer VLAN_RANGE
# ( no | default ) remove dot1q outer ...
#--------------------------------------------------------------------------------
class RemoveDot1QOuterCmd( ActionCmdBase ):
   _actionType = ActionType.stripHeaderBytes

   syntax = '''remove dot1q outer VLAN_RANGE'''
   noOrDefaultSyntax = '''remove dot1q outer ...'''
   data = {
      'remove' : CliCommand.guardedKeyword(
         'remove',
         helpdesc='Remove a header',
         guard=MirroringPolicyMapClassModeTapAgg.guardPmapDot1qRemove ),
      'dot1q' : 'Remove dot1q tag',
      'outer' : 'Remove outer tag',
      'VLAN_RANGE' : CliCommand.Node(
         matcher=MultiRangeRule.MultiRangeMatcher(
            rangeFn=lambda : ( 1, tapAggHwStatus.dot1qRemoveMaxIndex ),
            helpdesc='Specify indices of VLAN tags to be removed',
            noSingletons=False ),
         guard=MirroringPolicyMapClassModeTapAgg.guardPmapRemoveDot1qRange ),
   }

   @classmethod
   def handler( cls, mode, args ):
      hdrInfo = str( args[ 'VLAN_RANGE' ] )
      mode.context.setAction( cls._actionType, hdrInfo, no=False )

# --------------------------------------------------------------------------------
# remove header size SIZE [ preserve ethernet ]
# ( no | default ) remove header size <trailing garbage>
# --------------------------------------------------------------------------------
def tapHeaderRemoveGuard( mode, token ):
   if tapAggHwStatus.modeSupported and tapAggHwStatus.tapHeaderRemoveSupported:
      return None
   return CliParser.guardNotThisPlatform

class RemoveHeaderCmd( ActionCmdBase ):
   _actionType = ActionType.setHeaderRemove

   syntax = 'remove header size SIZE [ preserve ethernet ]'
   noOrDefaultSyntax = 'remove header size ...'
   data = {
      'remove' : CliCommand.guardedKeyword(
         'remove',
         helpdesc='Remove a header',
         guard=tapHeaderRemoveGuard ),
      'header' : 'Remove a configurable number of bytes after the Ethernet header, '
                 'including any VLAN tags',
      'size' : 'Set the number of bytes to remove',
      'SIZE' : CliMatcher.IntegerMatcher(
         2, 62, helpdesc='The size in bytes (must be even)' ),
      'preserve' : 'Preserve the original header',
      'ethernet' : 'Preserve the original Ethernet header, including VLAN tags '
                   'if any',
   }

   @classmethod
   def handler( cls, mode, args ):
      size = args.get( 'SIZE', 0 )
      # The size has to be an even number.
      if size % 2 == 1:
         mode.addError( "The size must be an even number." )
         return
      preserveEth = 'ethernet' in args
      tapHeaderRemove = TapHeaderRemove( size, preserveEth )
      mode.context.setAction( cls._actionType, tapHeaderRemove, no=False )

class RedirectCmdBase( ActionCmdBase ):
   @classmethod
   def checkErrors( cls, mode, args ):
      actionTypeToCmdMap = {
         ActionType.setAggregationGroup :
         "redirect aggregation group | redirect interface",
         ActionType.setNexthopGroup : "redirect next-hop group",
         ActionType.setNexthop : "redirect next-hop",
         ActionType.redirectTunnel : "redirect next-hop interface" }

      actionCmd = actionTypeToCmdMap[ cls._actionType ]
      del actionTypeToCmdMap[ cls._actionType ]
      if cls._actionType == ActionType.setNexthop:
         del actionTypeToCmdMap[ ActionType.redirectTunnel ]

      for actionType in actionTypeToCmdMap:
         if actionType in mode.context.matchRuleAction.actions():
            errorStr = "'%s' action cannot be used together with '%s' actions." % (
               actionCmd, ' | '.join( actionTypeToCmdMap.values() ) )
            mode.addError( errorStr )
            return errorStr
      return None

def getAggGroupList( mode ):
   result = set( tapAggStatus.group )
   if policiesCliConfig and policiesCliConfig.actions:
      for action in policiesCliConfig.actions.setAggGroupAction.values():
         result |= set( action.aggGroup )
   return result

matcherAggGroupName = CliMatcher.DynamicNameMatcher( getAggGroupList,
   'Group name', pattern=r'[A-Za-z0-9_:{}\[\]-]+' )

#--------------------------------------------------------------------------------
# redirect aggregation group { GROUPS }
# ( no | default ) redirect aggregation group [ { GROUPS } ]
#--------------------------------------------------------------------------------
class RedirectAggregationGroupCmd( RedirectCmdBase ):
   _actionType = ActionType.setAggregationGroup

   syntax = "redirect aggregation group { GROUPS }"
   noOrDefaultSyntax = "redirect aggregation group [ { GROUPS } ]"
   data = {
      'redirect' : 'Redirect packet flow',
      'aggregation' : 'Redirect to aggregation groups in Tap Aggregation mode',
      'group' : 'Set names of aggregation groups',
      'GROUPS' : matcherAggGroupName,
   }

   @classmethod
   def handler( cls, mode, args ):
      if cls.checkErrors( mode, args ):
         return
      groups = args[ 'GROUPS' ]
      aggDict = {}
      aggDict[ 'groups' ] = groups
      mode.context.setAction( cls._actionType, aggDict, no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      groups = args.get( 'GROUPS', [] )
      aggDict = {}
      aggDict[ 'groups' ] = groups
      mode.context.setAction( cls._actionType, aggDict, no=True )

#--------------------------------------------------------------------------------
# redirect interface INTFS
# ( no | default ) redirect interface [ INTFS ]
#--------------------------------------------------------------------------------
class RedirectIntfCmd( RedirectCmdBase ):
   _actionType = ActionType.setAggregationGroup

   syntax = "redirect interface INTFS"
   noOrDefaultSyntax = "redirect interface [ INTFS ]"
   data = {
      'redirect' : 'Redirect packet flow',
      'interface' : 'Redirect to interface',
      'INTFS' : MirroringPolicyMapClassModeTapAgg.matcherIntfRanger,
   }

   @classmethod
   def handler( cls, mode, args ):
      if cls.checkErrors( mode, args ):
         return
      intfs = args[ 'INTFS' ]
      aggDict = {}
      aggDict[ 'intfs' ] = intfs
      mode.context.setAction( cls._actionType, aggDict, no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      intfs = args.get( 'INTFS', [] )
      aggDict = {}
      aggDict[ 'intfs' ] = intfs
      mode.context.setAction( cls._actionType, aggDict, no=True )

#--------------------------------------------------------------------------------
# redirect next-hop group { GROUPS } [ ttl TTL_VALUE ]
# ( no | default ) redirect next-hop group [ { GROUPS } ] [ ttl TTL_VALUE ]
#--------------------------------------------------------------------------------
class RedirectNexthopGroupCmd( RedirectCmdBase ):
   _actionType = ActionType.setNexthopGroup

   syntax = "redirect next-hop group { GROUPS } [ TTL_EXPR ]"
   noOrDefaultSyntax = "redirect next-hop group [ { GROUPS } ] [ TTL_EXPR ]"
   data = {
      'redirect' : 'Redirect packet flow',
      'next-hop' : 'Redirect to next-hop',
      'group' : 'Set groups to redirect flow',
      'GROUPS' : nexthopGroupNameMatcher,
      'TTL_EXPR' : TtlExpression,
   }

   @classmethod
   def handler( cls, mode, args ):
      if cls.checkErrors( mode, args ):
         return
      groups = args[ 'GROUPS' ]
      ttlValue = args.get( 'TTL_VALUE', 0 )
      mode.context.setAction( cls._actionType, ( groups, ttlValue ), no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      groups = args.get( 'GROUPS', [] )
      ttlValue = args.get( 'TTL_VALUE', 0 )
      mode.context.setAction( cls._actionType, ( groups, ttlValue ), no=True )

def guardPolicyBasedVpn( mode, token ):
   if routingHwStatus.policyBasedVpnSupported:
      return None
   return CliParser.guardNotThisPlatform

# ---------------------------------------------------------------------------
# redirect next-hop
# ( ( [ recursive ] NEXTHOP [ VRF ] [ ttl TTL_VALUE ] )
# | ( interface INTF ) )
# ( no | default ) redirect next-hop [ recursive ] ...
# ----------------------------------------------------------------------------
class RedirectNexthopCmd( RedirectCmdBase ):
   _actionType = ActionType.setNexthop

   syntax = '''redirect next-hop
               ( ( [ recursive ] NEXTHOPS [ VRF ] [ TTL_EXPR ] )
               | ( interface INTF ) )'''
   noOrDefaultSyntax = "redirect next-hop [ recursive ] ..."
   data = {
      'redirect' : 'Redirect packet flow',
      'next-hop' : 'Redirect to next-hop',
      'recursive' : 'Enable recursive next-hop resolution',
      'NEXTHOPS' : NexthopExprFactory(),
      'VRF' : VrfExprFactory(
         helpdesc='Resolve next-hop in a VRF',
         guard=guardPbrNexthopVrf ),
      'TTL_EXPR' : TtlExpression,
      'interface' : CliCommand.guardedKeyword(
         'interface', helpdesc='Redirect to a next-hop interface',
         guard=guardPolicyBasedVpn ),
      'INTF' : TunnelIntfCli.TunnelIntf.matcher,
   }

   @classmethod
   def handler( cls, mode, args ):
      if cls.checkErrors( mode, args ):
         return
      groups = args.get( 'NEXTHOPS', [] )
      recursive = 'recursive' in args
      vrfName = args.get( 'VRF', "" )
      ttlValue = args.get( 'TTL_VALUE', 0 )

      if togglePolicyBasedVpnEnabled():
         intf = args.get( 'INTF' )
         if intf:
            # delete setNexthop action and add redirectTunnel action
            mode.context.setAction( cls._actionType, no=True )
            mode.context.setAction( ActionType.redirectTunnel,
                                    actionValue=IntfId( intf.name ), no=False )
         else:
            # delete redirectTunnel action and add setNexthop action
            mode.context.setAction( ActionType.redirectTunnel, no=True )
            mode.context.setAction( cls._actionType,
                                    ( groups, recursive, vrfName, ttlValue ),
                                    no=False )
      else:
         mode.context.setAction( cls._actionType,
                                 ( groups, recursive, vrfName, ttlValue ),
                                 no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.setAction( cls._actionType, no=True )
      if togglePolicyBasedVpnEnabled():
         mode.context.setAction( ActionType.redirectTunnel, no=True )

#------------------------------------------------------------------------------------
# The "actions" sub-prompt mode command
#------------------------------------------------------------------------------------
class ActionsConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'actions'
   noOrDefaultSyntax = syntax
   data = {
      'actions' : 'Configure actions',
   }

   @staticmethod
   def _feature():
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      childMode = mode.childMode( mode.context.actionMode(),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                                  matchRuleContext=mode.context,
                                  feature=cls._feature() )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.delAllAction()

class ActionsConfigCmd( ActionsConfigCmdBase ):
   data = ActionsConfigCmdBase.data.copy()

   @staticmethod
   def _feature():
      return FEATURE

#--------------------------------------------------
# The "ttl <ttl-list>" command for traffic-policy
#--------------------------------------------------

class TtlConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'ttl'
   _rangeType = 'Classification::TtlRange'
   _argListName = 'TTL'

   syntax = 'ttl TTL'
   noOrDefaultSyntax = 'ttl [ TTL ]'
   data = {
      'ttl' : 'Configure ttl match criteria',
      'TTL' : ttlRangeMatcher
   }

#--------------------------------------------------
# The "dscp <dscp-list>" command for traffic-policy
#--------------------------------------------------

class DscpConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'dscp'
   _rangeType = 'Classification::DscpRange'
   _argListName = 'DSCP'

   syntax = 'dscp DSCP'
   noOrDefaultSyntax = 'dscp [ DSCP ]'
   data = {
      'dscp' : 'dscp',
      'DSCP' : dscpRangeMatcher
   }

# --------------------------------------------------
# The "cos <cos-list>" command for traffic-policy
# --------------------------------------------------
class CosConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'cos'
   _rangeType = 'Classification::CosRange'
   _argListName = 'COS'

   syntax = 'cos COS'
   noOrDefaultSyntax = 'cos [ COS ]'
   data = {
      'cos' : 'Class of Service (CoS) value',
      'COS' : cosRangeMatcher
   }

#------------------------------------------------------------------------------------
# The "protocol neighbors bgp" command
# for traffic-policy
#------------------------------------------------------------------------------------
class ProtoNeighborsConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol neighbors bgp'
   if toggleTrafficPolicyEnforceGtsmEnabled():
      syntax += '[ enforce ttl maximum-hops ]'
   noOrDefaultSyntax = syntax
   data = {
      'protocol' : 'Protocol',
      'neighbors' : 'Neighbors',
      'bgp' : 'BGP',
      'enforce' : 'Enforce the GTSM for BGP speakers',
      'ttl' : 'Time to Live',
      'maximum-hops' : 'Maximum IPv{4,6} hops'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolBgp )
      attrValue = protocolBgpTtlSec if 'enforce' in args else protocolBgp
      mode.context.updateFilterAttr( 'neighborProtocol', attrValue=attrValue )
      # If structuredFilter is no longer valid, revert the change
      if not mode.context.filter.isValidConfig( conflictNeighborProtocol ):
         mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolNone )
         mode.addError( neighborsConfigConflictMsg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolNone )

#------------------------------------------------------------------------------------
# The "protocol bgp" command for traffic-policy
#------------------------------------------------------------------------------------

class ProtoMatchConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol bgp'
   noOrDefaultSyntax = syntax
   data = {
      'protocol' : 'Protocol',
      'bgp' : 'BGP'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4Bgp )
      # If structuredFilter is no longer valid, revert the change
      if not mode.context.filter.isValidConfig( conflictMatchL4Protocol ):
         mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4None )
         mode.addError( matchL4ProtocolConflictMsg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4None )

# The protocol command for traffic-policy:
#  "protocol <proto-list> | <tcp-udp> [ (source|destination) port <port-list> ]"
#  where that list is similar to acl "permit":
#  ahp      Authentication Header Protocol
#  gre      Generic Routing Encapsulation
#  icmp     Internet Control Message Protocol
#  igmp     Internet Group Management Protocol (IGMP)
#  ospf     OSPF routing protocol
#  pim      Protocol Independent Multicast (PIM)
#  rsvp     Resource Reservation Protocol (RSVP)
#  tcp      Transmission Control Protocol
#  udp      User Datagram Protocol
#  vlan     Vlan
#  vrrp     Virtual Router Redundancy Protocol
#  <1-255>  IP protocol number for ipv4 ( 0 illegal for v4 )
#  <0-255>  IP protocol number for ipv6
#
#------------------------------------------------------------------------------------
def getL4PortFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetL4Port

l4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name",
   pattern=protectedFieldSetNamesRegex( 'port' ),
   priority=CliParser.PRIO_LOW )
#------------------------------------------------------------------------------------
# The "protocol (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port field-set
# FIELD_SET destination port field-set FIELD_SET" command for traffic-policy
#------------------------------------------------------------------------------------

l4PortSrcFieldSetExpr = generateFieldSetExpression( l4PortFieldSetNameMatcher,
                                                   'SRC_FIELD_SET_NAME',
                                                    allowMultiple=True )
l4PortDstFieldSetExpr = generateFieldSetExpression( l4PortFieldSetNameMatcher,
                                                   'DST_FIELD_SET_NAME',
                                                    allowMultiple=True )

class ProtocolFieldSetCmd( ProtocolFieldSetOredBaseCmd ):
   _sportFieldSetAttr = 'sportFieldSet'
   _dportFieldSetAttr = 'dportFieldSet'
   data = {
      'FLAGS_EXPR' : generateTcpFlagCombinationExpression(),
      'SRC_FIELD_SET_NAME' : l4PortSrcFieldSetExpr,
      'DST_FIELD_SET_NAME' : l4PortDstFieldSetExpr
   }
   data.update( ProtocolFieldSetOredBaseCmd._baseData )

class ProtocolFieldSetWithoutFlagsCmd( ProtocolFieldSetOredBaseCmd ):
   _sportFieldSetAttr = 'sportFieldSet'
   _dportFieldSetAttr = 'dportFieldSet'
   data = {
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=False ),
      'SRC_FIELD_SET_NAME' : l4PortSrcFieldSetExpr,
      'DST_FIELD_SET_NAME' : l4PortDstFieldSetExpr
   }
   data.update( ProtocolFieldSetOredBaseCmd._baseData )

class ProtocolIpv4ListConfigCmd( ProtocolOredBase ):
   data = {
      'PROTOCOL' : ipv4ProtoMatcher,
      'FLAGS_EXPR' : generateTcpFlagCombinationExpression(),
   }
   data.update( ProtocolOredBase._baseData )

class ProtocolIpv6ListConfigCmd( ProtocolOredBase ):
   data = {
      'PROTOCOL' : ipv6ProtoMatcher,
      'FLAGS_EXPR' : generateTcpFlagCombinationExpression(),
   }
   data.update( ProtocolOredBase._baseData )

class ProtocolListWithoutFlagsCmd( ProtocolOredBase ):
   """
   ProtocolListWithoutFlagsCmd is to implement the ORed L4 ports configuration in
   encapsulated mode ONLY for TCP and UDP protocols.
   """
   syntax = '''
protocol ( TCP_UDP | FLAGS_EXPR )
           [ ( source port SPORT [ destination port ( DPORT | all ) ] )
           | ( [ source port all ] destination port DPORT ) ]'''
   noOrDefaultSyntax = '''
protocol [ ( TCP_UDP | FLAGS_EXPR )
             [ ( source port [ SPORT [ destination port ( DPORT | all ) ] ] )
             | ( source port all destination port DPORT )
             | ( destination port [ DPORT ] ) ] ]'''
   data = {
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=False )
   }
   data.update( ProtocolOredBase._baseData )

#--------------------------------------------------------------------------------
# 'tracked' command under match rule config mode.
#--------------------------------------------------------------------------------
class MatchTrackedCmd( CliCommand.CliCommandClass ):
   syntax = 'tracked'
   noOrDefaultSyntax = syntax
   data = {
      'tracked' : 'Match packets in existing connections',
   }

   @staticmethod
   def handler( mode, args ):
      matchOnTracked = not CliCommand.isNoOrDefaultCmd( args )
      mode.context.updateFilterAttr( 'matchTracked', attrValue=matchOnTracked )

   noOrDefaultHandler = handler

def _clearCounterHelper( snapshotDir, pmapStatusDir, counterStatusDir,
                         sharkCountersSupported ):
   if sharkCountersSupported:
      counterStatusDirEntity = SharkLazyMount.force( counterStatusDir )
      return Tac.newInstance(
               "TrafficPolicyCli::TrafficPolicySharkCounterSnapshotHelper",
               counterStatusDirEntity, snapshotDir,
         pmapStatusDir.countersPerInterface )
   else:
      return Tac.newInstance(
                "TrafficPolicyCli::TrafficPolicySysdbCounterSnapshotHelper",
                pmapStatusDir, snapshotDir )

#--------------------------------------------------------------------------------
# clear traffic-policy counters
#--------------------------------------------------------------------------------
class ClearTrafficPolicyCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear traffic-policy counters'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'traffic-policy' : matcherTrafficPolicy,
      'counters' : matcherCounters,
   }

   @staticmethod
   def handler( mode, args ):
      for key, status in six.iteritems( policiesStatus ):
         counter = None
         if key == 'cpu-vrf-hardware':
            counter = cpuCounter
            counterStatus = None
            sharkCountersSupported = False
         elif key == 'interfaceInput':
            counter = intfInputCounter
            counterStatus = counterStatusInput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         elif key == 'interfaceOutput':
            counter = intfOutputCounter
            counterStatus = counterStatusOutput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         elif toggleVaclTrafficPolicyCountActionEnabled() and key == 'vlan':
            counter = intfInputCounter
            counterStatus = counterStatusInput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         else:
            # Counters not supported
            continue

         assert counter is not None
         assert status is not None
         LazyMount.force( counter )
         cliHelper = _clearCounterHelper( counter, status, counterStatus,
                                          sharkCountersSupported )
         cliHelper.clearCounter()

BasicCli.EnableMode.addCommandClass( ClearTrafficPolicyCountersCmd )

#--------------------------------------------------------------------------------
# clear traffic-policy counters session
#--------------------------------------------------------------------------------
class ClearTrafficPolicyCountersSessionCmd( CliCommand.CliCommandClass ):
   syntax = 'clear traffic-policy counters session'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'traffic-policy' : matcherTrafficPolicy,
      'counters' : matcherCounters,
      'session' : 'Clear traffic-policy counters in current session',
   }

   @staticmethod
   def handler( mode, args ):

      for key, status in six.iteritems( policiesStatus ):
         if key == 'cpu-vrf-hardware':
            sessionCounterDataName = 'cpuTrafficPolicySessionCounter'
            counterStatus = None
            sharkCountersSupported = False
         elif key == 'interfaceInput':
            sessionCounterDataName = 'intfInputTrafficPolicySessionCounter'
            counterStatus = counterStatusInput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         elif key == 'interfaceOutput':
            sessionCounterDataName = 'intfOutputTrafficPolicySessionCounter'
            counterStatus = counterStatusOutput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         elif toggleVaclTrafficPolicyCountActionEnabled() and key == 'vlan':
            sessionCounterDataName = 'vaclTrafficPolicySessionCounter'
            counterStatus = counterStatusInput
            sharkCountersSupported = (
                  interfaceTrafficPolicyHwStatus.sharkCountersSupported )
         else:
            # Counters not supported
            continue

         sessionCounter = mode.session.sessionData( sessionCounterDataName,
                                                    None )
         if sessionCounter is None:
            sessionCounter = Tac.newInstance( 'TrafficPolicy::Counter',
                                              sessionCounterDataName )
            mode.session.sessionDataIs( sessionCounterDataName,
                                        sessionCounter )

         assert status is not None
         assert sessionCounter is not None

         cliHelper = _clearCounterHelper( sessionCounter, status,
                                          counterStatus,
                                          sharkCountersSupported )
         cliHelper.clearCounter()

BasicCli.EnableMode.addCommandClass( ClearTrafficPolicyCountersSessionCmd )

# --------------------------------------------------------------------------
# The "field-set (ipv4 | ipv6) prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   return list( fieldSetConfig.fieldSetIpPrefix ) if fieldSetConfig else []

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'childMode' : FieldSetIpPrefixConfigMode,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

def getIpv6PrefixFieldSetNames( mode ):
   return list( fieldSetConfig.fieldSetIpv6Prefix ) if fieldSetConfig else []

ipv6PrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv6PrefixFieldSetNames,
   "IPv6 prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpv6PrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv6 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : ipv6PrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'childMode' : FieldSetIpv6PrefixConfigMode,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | bgp | URL | and-results }" command
# --------------------------------------------------------------------------
class FieldSetIpv4PrefixSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'prefixes'
   data = {
      'SOURCE_EXPR' : SourceMatcher(
                         supportsBgp=True,
                         supportsUrl=True,
                         supportsController=toggleFieldSetControllerEnabled(),
                         supportsIntersect=toggleFieldSetAndResultsEnabled(),
                         fieldSetMatcher=ipPrefixFieldSetNameMatcher ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher(
                              supportsBgp=False,
                              supportsUrl=False,
                              supportsIntersect=toggleFieldSetAndResultsEnabled(),
                              fieldSetMatcher=ipPrefixFieldSetNameMatcher,
                              isRemove=True )
   }

class FieldSetIpv6PrefixSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'prefixes'
   data = {
      'SOURCE_EXPR' : SourceMatcher(
                         supportsBgp=True,
                         supportsUrl=True,
                         supportsController=toggleFieldSetControllerEnabled(),
                         supportsIntersect=toggleFieldSetAndResultsEnabled(),
                         fieldSetMatcher=ipv6PrefixFieldSetNameMatcher ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher(
                              supportsBgp=False,
                              supportsUrl=False,
                              supportsIntersect=toggleFieldSetAndResultsEnabled(),
                              fieldSetMatcher=ipv6PrefixFieldSetNameMatcher,
                              isRemove=True )
   }

# XXX Temporary fix to address BUG578352. We need to come up with a long term fix.
# For these command classes, we need to declare its own 'data' class attribute
# becase the CLI infra may update the class's 'data' when expanding
# noOrDefaultSyntax. That ends up affecting the base class's attribute otherwise...

class FieldSetIntegerConfigNoAuthCmds( FieldSetIntegerConfigCmds ):
   data = FieldSetIntegerConfigCmds.data.copy()
   authz = False

class FieldSetIntegerExceptConfigNoAuthCmds( FieldSetIntegerExceptConfigCmds ):
   data = FieldSetIntegerExceptConfigCmds.data.copy()
   authz = False

class FieldSetMacAddrConfigNoAuthCmds( FieldSetMacAddrConfigBase ):
   data = FieldSetMacAddrConfigBase.data.copy()
   authz = False

class FieldSetVlanConfigNoAuthCmds( FieldSetVlanConfigCmds ):
   data = FieldSetVlanConfigCmds.data.copy()
   authz = False

class FieldSetVlanExceptConfigNoAuthCmds( FieldSetVlanExceptConfigCmds ):
   data = FieldSetVlanExceptConfigCmds.data.copy()
   authz = False

class FieldSetL4PortConfigNoAuthCmds( FieldSetL4PortConfigBase ):
   data = FieldSetL4PortConfigBase.data.copy()
   authz = False

class FieldSetL4PortExceptConfigNoAuthCmds( FieldSetL4PortExceptConfigBase ):
   data = FieldSetL4PortExceptConfigBase.data.copy()
   authz = False

class FieldSetIpPrefixConfigNoAuthCmds( FieldSetIpPrefixConfigBase ):
   data = FieldSetIpPrefixConfigBase.data.copy()
   authz = False

class FieldSetIpv6PrefixConfigNoAuthCmds( FieldSetIpv6PrefixConfigCmds ):
   data = FieldSetIpv6PrefixConfigCmds.data.copy()
   authz = False

class FieldSetIpPrefixExceptConfigNoAuthCmds( FieldSetIpPrefixExceptConfigCmds ):
   data = FieldSetIpPrefixExceptConfigCmds.data.copy()
   authz = False

class FieldSetIpv6PrefixExceptConfigNoAuthCmds( FieldSetIpv6PrefixExceptConfigCmds ):
   data = FieldSetIpv6PrefixExceptConfigCmds.data.copy()
   authz = False

# Allow multiple prefixes to be defined in prefix field-set
ipPrefixFieldSetExpr = generateFieldSetExpression( ipPrefixFieldSetNameMatcher,
                                                   'FIELD_SET',
                                                   allowMultiple=True )

# --------------------------------------------------------------------------
# The "( source|destination ) prefix field-set { FIELD_SET }" command
# --------------------------------------------------------------------------
class PrefixFieldSetIpv4BaseV2( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixFieldSetCmdMatcher( allowMultiple=True,
                                        fieldSetMatcher=ipPrefixFieldSetNameMatcher )
   }

class PrefixFieldSetIpv4CmdV2( PrefixFieldSetIpv4BaseV2 ):
   data = PrefixFieldSetIpv4BaseV2.data.copy()

class PrefixFieldSetIpv4Cmd( PrefixFieldSetCmdBase ):
   data = {
      "FIELD_SET" : ipPrefixFieldSetExpr
   }
   data.update( PrefixFieldSetCmdBase._baseData )

ipv6PrefixFieldSetExpr = generateFieldSetExpression( ipv6PrefixFieldSetNameMatcher,
                                                     'FIELD_SET',
                                                     allowMultiple=True )

class PrefixFieldSetIpv6CmdV2Base( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixFieldSetCmdMatcher( allowMultiple=True,
                                      fieldSetMatcher=ipv6PrefixFieldSetNameMatcher )
   }

class PrefixFieldSetIpv6CmdV2( PrefixFieldSetIpv6CmdV2Base ):
   data = PrefixFieldSetIpv6CmdV2Base.data.copy()

class PrefixFieldSetIpv6Cmd( PrefixFieldSetCmdBase ):
   data = {
      "FIELD_SET" : ipv6PrefixFieldSetExpr
   }
   data.update( PrefixFieldSetCmdBase._baseData )


# --------------------------------------------------------------------------
# The "( source|destination ) prefix longest-prefix field-set { FIELD_SET }"
# command
# --------------------------------------------------------------------------
class PrefixLpmFieldSetIpv4Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixFieldSetCmdMatcher(
         allowMultiple=True,
         fieldSetMatcher=ipPrefixFieldSetNameMatcher,
         longestPrefix=True )
   }

class PrefixLpmFieldSetIpv6Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixFieldSetCmdMatcher(
         allowMultiple=True,
         fieldSetMatcher=ipv6PrefixFieldSetNameMatcher,
         longestPrefix=True )
   }

# --------------------------------------------------------------------------
# The "next-hop group NEXTHOP_GROUP_NAME" command
# --------------------------------------------------------------------------
class NexthopGroupMatchConfigCmd( NexthopGroupMatchBaseConfigCmd ):
   syntax = 'next-hop group { NEXTHOP_GROUP_NAME }'
   noOrDefaultSyntax = 'next-hop group [ { NEXTHOP_GROUP_NAME } ]'
   _feature = "tp"
   data = {
      'NEXTHOP_GROUP_NAME' : nexthopGroupNameMatcher,
   }

   data.update( NexthopGroupMatchBaseConfigCmd._baseData )

def getSupportedBases( mode, encap=False ):
   supportedBase = {}
   udfOffsetBaseSupported = interfaceTrafficPolicyHwStatus.udfOffsetBaseSupported
   for base in udfOffsetBaseSupported:
      if encap:
         if base == UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderEnd:
            supportedBase[ 'ip-header-end' ] = 'Base is end of IP header'
         elif base == UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderStart:
            supportedBase[ 'ip-header-start' ] = 'Base is start of IP header'
         elif base == UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderEnd:
            supportedBase[ 'ip-header-end' ] = 'Base is end of IP header'
         elif base == UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderStart:
            supportedBase[ 'ip-header-start' ] = 'Base is start of IP header'
      else:
         if base == UdfOffsetBase.UdfOffsetBasePacketStart:
            supportedBase[ 'packet-start' ] = 'Base is start of the packet'
         elif base == UdfOffsetBase.UdfOffsetBaseMetadata:
            supportedBase[ 'metadata' ] = 'Base is metadata'
         elif base == UdfOffsetBase.UdfOffsetBaseL3HeaderEnd:
            supportedBase[ 'ip-header-end' ] = 'Base is end of IP header'
         elif base == UdfOffsetBase.UdfOffsetBaseL3HeaderStart:
            supportedBase[ 'ip-header-start' ] = 'Base is start of IP header'
   return supportedBase

encapMatcher = {}
if toggleUdfEncapGtpMatchEnabled():
   encapMatcher[ 'gtp' ] = 'Encapsulation GTP'
if toggleUdfEncapVxlanMatchEnabled():
   encapMatcher[ 'vxlan' ] = 'Encapsulation VXLAN'
locationEncapTypeMatcher = CliMatcher.EnumMatcher( encapMatcher )
locationCopyLengthMatcher = CliMatcher.EnumMatcher( {
      '16' : '16 bits',
      '32' : '32 bits',
      } )

def getSupportedBasesEncap( mode ):
   return getSupportedBases( mode, encap=True )

# ---------------------------------------------------------------------------
# The "location LOCATION_ALIAS_NAME encapsulation ENCAP_TYPE COPY_BASE offset
# COPY_OFFSET bytes length COPY_LENGTH bits" command
# ---------------------------------------------------------------------------
class LocationAliasConfigEncapCmd( CliCommand.CliCommandClass ):
   syntax = 'location LOCATION_ALIAS_NAME encapsulation ENCAP_TYPE COPY_BASE offset'\
            ' COPY_OFFSET bytes length COPY_LENGTH bits'
   data = {
      'location' : locationKwMatcher,
      'LOCATION_ALIAS_NAME' : CliMatcher.QuotedStringMatcher( helpname='WORD',
                                      helpdesc='Location alias name' ),
      'encapsulation' : 'Specify encapsulation',
      'ENCAP_TYPE' : locationEncapTypeMatcher,
      'COPY_BASE' : CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( getSupportedBasesEncap,
                                          alwaysMatchInStartupConfig=True ) ),
      'offset' : 'Location offset relative to base',
      'COPY_OFFSET' : CliMatcher.IntegerMatcher( 0, 128,
                                                helpdesc='Offset' ),
      'bytes' : 'Loction offset in bytes',
      'length' : 'Location length to copy from offset',
      'COPY_LENGTH' : locationCopyLengthMatcher,
      'bits' : 'Length in bits',
   }

   @staticmethod
   def handler( mode, args ):
      aliasName = args[ 'LOCATION_ALIAS_NAME' ]
      base = args[ 'COPY_BASE' ]
      offset = args[ 'COPY_OFFSET' ]
      length = args[ 'COPY_LENGTH' ]
      encapType = args[ 'ENCAP_TYPE' ]
      if base == 'ip-header-start':
         if encapType == 'gtp':
            udfBase = UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderStart
         elif encapType == 'vxlan':
            udfBase = UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderStart
      elif base == 'ip-header-end':
         if encapType == 'gtp':
            udfBase = UdfOffsetBase.UdfOffsetBaseEncapGtpIpHeaderEnd
         elif encapType == 'vxlan':
            udfBase = UdfOffsetBase.UdfOffsetBaseEncapVxlanIpHeaderEnd
      else:
         assert False, 'Invalid base %s' % base
      if length == '32':
         udfLength = UdfLength.udf32b
      elif length == '16':
         udfLength = UdfLength.udf16b
      else:
         assert False, 'Unsupported length %s' % length
      udfAnchorConfig = UdfAnchorConfig( aliasName, udfBase, offset, udfLength,
                                         UniqueId() )
      # if the member already exists it is overwritten
      locationAliasConfig.udf.addMember( udfAnchorConfig )

# --------------------------------------------------------------------------
# The "location LOCATION_ALIAS_NAME COPY_BASE offset COPY_OFFSET bytes
# length COPY_LENGTH bits" command
# --------------------------------------------------------------------------
class LocationAliasConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'location LOCATION_ALIAS_NAME COPY_BASE offset COPY_OFFSET bytes'\
            ' length COPY_LENGTH bits'
   noOrDefaultSyntax = 'location LOCATION_ALIAS_NAME'
   data = {
      'location' : locationKwMatcher,
      'LOCATION_ALIAS_NAME' : CliMatcher.QuotedStringMatcher( helpname='WORD',
                                      helpdesc='Location alias name' ),
      'COPY_BASE' : CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( getSupportedBases,
                                           alwaysMatchInStartupConfig=True ) ),
      'offset' : 'Location offset relative to base',
      'COPY_OFFSET' : CliMatcher.IntegerMatcher( 0, 128,
                                                helpdesc='Offset' ),
      'bytes' : 'Loction offset in bytes',
      'length' : 'Location length to copy from offset',
      'COPY_LENGTH' : locationCopyLengthMatcher,
      'bits' : 'Length in bits',
   }

   @staticmethod
   def handler( mode, args ):
      aliasName = args[ 'LOCATION_ALIAS_NAME' ]
      base = args[ 'COPY_BASE' ]
      offset = args[ 'COPY_OFFSET' ]
      length = args[ 'COPY_LENGTH' ]
      if base == 'packet-start':
         udfBase = UdfOffsetBase.UdfOffsetBasePacketStart
      elif base == 'ip-header-start':
         udfBase = UdfOffsetBase.UdfOffsetBaseL3HeaderStart
      elif base == 'ip-header-end':
         udfBase = UdfOffsetBase.UdfOffsetBaseL3HeaderEnd
      elif base == 'metadata':
         udfBase = UdfOffsetBase.UdfOffsetBaseMetadata
      else:
         assert False, 'Invalid base %s' % base

      if length == '32':
         udfLength = UdfLength.udf32b
      elif length == '16':
         udfLength = UdfLength.udf16b
      else:
         assert False, 'Unsupported length %s' % length
      udfAnchorConfig = UdfAnchorConfig( aliasName, udfBase, offset, udfLength,
                                         UniqueId() )
      # if the member already exists it is overwritten
      locationAliasConfig.udf.addMember( udfAnchorConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      locationAliasName = args[ 'LOCATION_ALIAS_NAME' ]
      # del is safe even when locationAliasName doesn't exist
      del locationAliasConfig.udf[ locationAliasName ]

# --------------------------------------------------------------------------
# The "location LOCATION_ALIAS_NAME value HEX_VALUE mask HEX_MASK" command
# --------------------------------------------------------------------------
class LocationValueMatchConfigCmd( LocationValueMatchBaseConfigCmd ):
   data = {
      'LOCATION_ALIAS_NAME' : locationAliasNameMatcher,
   }
   data.update( LocationValueMatchBaseConfigCmd._baseData )

# --------------------------------------------------------------------------
# The "field-set integer FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getIntegerFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetInteger

integerFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIntegerFieldSetNames,
   "Integer field-set name",
   pattern=protectedFieldSetNamesRegex( 'integer' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIntegerConfigCmd( FieldSetIntegerBaseConfigCmd ):
   syntax = 'field-set integer FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : integerFieldSetNameMatcher,
   }

   data.update( FieldSetIntegerBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIntegerName, mode=None ):
      return {
         'fieldSetIntegerName' : fieldSetIntegerName,
         'fieldSetConfig' : fieldSetConfig,
         'childMode' : FieldSetIntegerConfigMode,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

def getMacAddrFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetMacAddr

macAddrFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getMacAddrFieldSetNames,
   "MAC field-set name",
   pattern=protectedFieldSetNamesRegex( 'mac' ),
   priority=CliParser.PRIO_LOW )

nodeMacAddrFieldSet = CliCommand.Node( matcher=macAddrFieldSetNameMatcher,
   guard=macRuleMatchingGuard )

# --------------------------------------------------------------------------
# The "( source | destination ) mac { MAC_ADDR }" command
# --------------------------------------------------------------------------
class SrcDstMacAddrCmd( MacCmdBase ):
   data = {
      "MAC_EXPR" : SrcDstMacAddrCmdMatcher( allowMultiple=True )
   }

# --------------------------------------------------------------------------
# The "( source | destination ) mac field-set { FIELD_SET }" command
# --------------------------------------------------------------------------
class SrcDstMacAddrFieldSetCmd( MacCmdBase ):
   data = {
      "MAC_EXPR" : SrcDstMacAddrFieldSetCmdMatcher( allowMultiple=True,
                                        fieldSetMatcher=macAddrFieldSetNameMatcher )
   }

# --------------------------------------------------------------------------
# The "field-set mac FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetMacAddrConfigCmd( FieldSetMacAddrBaseConfigCmd ):
   syntax = 'field-set mac FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : nodeMacAddrFieldSet,
   }

   data.update( FieldSetMacAddrBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetMacAddrName, mode=None ):
      return {
         'fieldSetMacAddrName' : fieldSetMacAddrName,
         'fieldSetConfig' : fieldSetConfig,
         'childMode' : FieldSetMacAddrConfigMode,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | url }" command
# --------------------------------------------------------------------------
class FieldSetMacAddrSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'MACs'
   data = {
      'SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=True ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=False,
                                            isRemove=True )
   }

# ---------------------------------------------------------------------------
# The "teid field-set integer TEID_FIELD_SET" command
# ---------------------------------------------------------------------------
teidFieldSetExpr = generateFieldSetExpression( integerFieldSetNameMatcher,
                                               'TEID_FIELD_SET',
                                               allowMultiple=True )

class TeidFieldSetConfigCmd( TeidConfigCmdBase ):
   data = {
      'TEID_EXPR' : TeidCmdMatcher( isTeidFieldSet=True,
                                    fsExpr=teidFieldSetExpr ),
   }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | url }" command
# --------------------------------------------------------------------------
class FieldSetIntegerSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'INTEGERs'
   data = {
      'SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=True ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=False,
                                            isRemove=True )
   }

# --------------------------------------------------------------------------
# The "field-set vlan FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getVlanFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetVlan

vlanFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getVlanFieldSetNames,
   "VLAN field-set name",
   pattern=protectedFieldSetNamesRegex( 'vlan' ),
   priority=CliParser.PRIO_LOW )

vlanFieldSetExpr = generateFieldSetExpression( vlanFieldSetNameMatcher,
                                               'VLANTAG_FIELD_SET',
                                               allowMultiple=True )

class FieldSetVlanConfigCmd( FieldSetVlanBaseConfigCmd ):
   syntax = 'field-set vlan FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : vlanFieldSetNameMatcher,
   }
   data.update( FieldSetVlanBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetVlanName, mode=None ):
      return {
         'fieldSetVlanName' : fieldSetVlanName,
         'fieldSetConfig' : fieldSetConfig,
         'childMode' : FieldSetVlanConfigMode,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

#---------------------------------------------------------------------------
# The "dot1q vlan field-set VLANTAG_FIELD_SET" command
#---------------------------------------------------------------------------
class VlanTagFieldSetConfigCmd( VlanTagConfigCmdBase ):
   data = {
      'VLANTAG_EXPR' : VlanTagCmdMatcher( isVlanFieldSet=True,
                                          isInnerVlanFieldSet=False,
                                          fsExpr=vlanFieldSetExpr ),
   }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | url }" command
# --------------------------------------------------------------------------
class FieldSetVlanSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'VLANs'
   data = {
      'SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=True ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=False,
                                            isRemove=True )
   }

# --------------------------------------------------------------------------
# The "field-set l4-port PORT_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetL4PortConfigCmd( FieldSetL4PortBaseConfigCmd ):
   syntax = 'field-set l4-port FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : l4PortFieldSetNameMatcher,
   }
   data.update( FieldSetL4PortBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      return {
         'fieldSetL4PortName' : fieldSetL4PortName,
         'fieldSetConfig' : fieldSetConfig,
         'childMode' : FieldSetL4PortConfigMode,
         'featureName' : 'tp',
         # Enabled config rollback in the L4PortFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | url }" command
# --------------------------------------------------------------------------
class FieldSetL4PortSourceConfigCmd( FieldSetSourceConfigCmdBase ):
   _field = 'L4-ports'
   data = {
      'SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=True ),
      'REMOVE_SOURCE_EXPR' : SourceMatcher( supportsBgp=False, supportsUrl=False,
                                            isRemove=True )
   }

#---------------------------------------------------------------------------
# The "encapsulation dzgre" command
#---------------------------------------------------------------------------
class DzGreEncapMatchCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation dzgre'
   noOrDefaultSyntax = syntax
   data = {
      'encapsulation' : 'Configure encapsulation',
      'dzgre' : nodeDzGre,
   }

   @staticmethod
   def _feature():
      return FEATURE_SHORT

   @classmethod
   def handler( cls, mode, args ):
      encapType = 'dzgre'
      matchOption = matchIpAccessGroup
      context = mode.getContext()
      childMode = mode.childMode( EncapDzGreConfigMode,
                        trafficPolicyContext=mode.trafficPolicyContext,
                        matchRuleContext=mode.trafficPolicyContext.matchRuleContext,
                        feature=cls._feature() )

      sf = mode.trafficPolicyContext.matchRuleContext.filter
      sf.encapField = ( 'dzgre', )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      sf = mode.trafficPolicyContext.matchRuleContext.filter
      sf.encapField = None

# --------------------------------------------------------------------------
# The "policy id POLICYID" command
# --------------------------------------------------------------------------
class DzGreEncapSetPolicyIdCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'dzGrePolicyId'
   _rangeType = 'Classification::DzGrePolicyIdRange'
   _argListName = 'POLICYID'

   syntax = 'policy id POLICYID'
   noOrDefaultSyntax = 'policy id ...'
   data = {
      'policy' : 'policy identity',
      'id' : matcherId,
      'POLICYID' : dzGrePolicyIdRangeMatcher
   }

# --------------------------------------------------------------------------
# The "port id PORTID" command
# --------------------------------------------------------------------------
class DzGreEncapSetPortIdCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'dzGrePortId'
   _rangeType = 'Classification::DzGrePortIdRange'
   _argListName = 'PORTID'

   syntax = 'port id PORTID'
   noOrDefaultSyntax = 'port id ...'
   data = {
      'port' : 'port identity',
      'id' : matcherId,
      'PORTID' : dzGrePortIdRangeMatcher
   }

# --------------------------------------------------------------------------
# The "switch id SWITCHID" command
# --------------------------------------------------------------------------
class DzGreEncapSetSwitchIdCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'dzGreSwitchId'
   _rangeType = 'Classification::DzGreSwitchIdRange'
   _argListName = 'SWITCHID'

   syntax = 'switch id SWITCHID'
   noOrDefaultSyntax = 'switch id ...'
   data = {
      'switch' : matcherDzGreSwitch,
      'id' : matcherId,
      'SWITCHID' : dzGreSwitchIdRangeMatcher
   }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set all" command
# --------------------------------------------------------------------------
class FieldSetRefreshAllCmd( FieldSetRefreshAllCmdBase ):
   syntax = 'refresh traffic-policy field-set all'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetRefreshAllCmdBase._baseData )
   @classmethod
   def _getFieldSetConfig( cls ):
      return fieldSetConfig

   @classmethod
   def _freezeProcessing( cls, freeze ):
      fsConfig = cls._getFieldSetConfig()
      if not fsConfig:
         return
      if freeze:
         fsConfig.forcedPauseConfig.forcedPause += 1
         return
      fsConfig.forcedResumeConfig.forcedResume += 1

   @classmethod
   def _getIpPrefixContextKwargs( cls, fieldSetName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

   @classmethod
   def _getIpv6PrefixContextKwargs( cls, fieldSetName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

   @classmethod
   def _getL4PortContextKwargs( cls, fieldSetName, mode=None ):
      return {
         'fieldSetL4PortName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         # Enabled config rollback in the L4PortFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

   @classmethod
   def _getVlanContextKwargs( cls, fieldSetName, mode=None ):
      return {
         'fieldSetVlanName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

   @classmethod
   def _getIntegerContextKwargs( cls, fieldSetName, mode=None ):
      return {
         'fieldSetIntegerName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

   @classmethod
   def _getMacAddrContextKwargs( cls, fieldSetName, mode=None ):
      return {
         'fieldSetMacAddrName' : fieldSetName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set ipv4 prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIpv4PrefixRefreshCmd( FieldSetIpPrefixRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set ipv4 prefix FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set ipv6 prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIpv6PrefixRefreshCmd( FieldSetIpPrefixRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set ipv6 prefix FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : ipv6PrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'featureName' : 'tp',
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set l4-port FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetL4PortRefreshCmd( FieldSetL4PortRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set l4-port FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : l4PortFieldSetNameMatcher,
   }
   data.update( FieldSetL4PortRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      return {
         'fieldSetL4PortName' : fieldSetL4PortName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         # Enabled config rollback in the L4PortFieldSetContext.
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set vlan FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetVlanRefreshCmd( FieldSetVlanRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set vlan FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : vlanFieldSetNameMatcher,
   }
   data.update( FieldSetVlanRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetVlanName, mode=None ):
      return {
         'fieldSetVlanName' : fieldSetVlanName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set integer FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIntegerRefreshCmd( FieldSetIntegerRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set integer FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : integerFieldSetNameMatcher,
   }
   data.update( FieldSetIntegerRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIntegerName, mode=None ):
      return {
         'fieldSetIntegerName' : fieldSetIntegerName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

# --------------------------------------------------------------------------
# The "refresh traffic-policy field-set mac FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetMacAddrRefreshCmd( FieldSetMacAddrRefreshCmdBase ):
   syntax = 'refresh traffic-policy field-set mac FIELD_SET_NAME'
   data = {
      'traffic-policy' : matcherTrafficPolicyRefresh,
      'FIELD_SET_NAME' : macAddrFieldSetNameMatcher,
   }
   data.update( FieldSetMacAddrRefreshCmdBase._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetMacAddrName, mode=None ):
      return {
         'fieldSetMacAddrName' : fieldSetMacAddrName,
         'fieldSetConfig' : fieldSetConfig,
         'featureName' : 'tp',
         'rollbackSupported' : True,
         'urlFieldSetConfig' : urlFieldSetConfig,
         'urlRollbackSupported' : True,
         'urlLoadInitialMode' : TrafficPoliciesConfigMode,
         'urlImportFailSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_FAILED,
         'urlImportSuccessSyslogHandle' : TRAFFICPOLICY_FIELD_SET_IMPORT_SUCCEEDED,
      }

if toggleFieldSetRefreshAllEnabled():
   BasicCli.EnableMode.addCommandClass( FieldSetRefreshAllCmd )
BasicCli.EnableMode.addCommandClass( FieldSetIpv4PrefixRefreshCmd )
BasicCli.EnableMode.addCommandClass( FieldSetIpv6PrefixRefreshCmd )
BasicCli.EnableMode.addCommandClass( FieldSetL4PortRefreshCmd )
BasicCli.EnableMode.addCommandClass( FieldSetVlanRefreshCmd )
BasicCli.EnableMode.addCommandClass( FieldSetIntegerRefreshCmd )
BasicCli.EnableMode.addCommandClass( FieldSetMacAddrRefreshCmd )

BasicCli.GlobalConfigMode.addCommandClass( TrafficPoliciesConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( TrafficPolicyConfigCmd )
TrafficPolicyConfigMode.addCommandClass( MatchRuleConfigCmd )
TrafficPolicyConfigMode.addCommandClass( NamedCounterCmd )

# Most modes support 'commit', and 'abort'
for _mode in [ TrafficPolicyConfigMode,
               MatchRuleIpv4ConfigMode,
               MatchRuleIpv6ConfigMode,
               MatchRuleDefaultConfigMode,
               ActionsConfigMode,
               ReplicateMode,
               FieldSetL4PortConfigMode,
               FieldSetIpPrefixConfigMode,
               FieldSetIpv6PrefixConfigMode,
               FieldSetVlanConfigMode,
               FieldSetIntegerConfigMode, ]:
   _mode.addModelet( CommitAbortModelet )

MatchRuleMacConfigMode.addModelet( CommitAbortModelet )
MacActionsConfigMode.addModelet( CommitAbortModelet )
FieldSetMacAddrConfigMode.addModelet( CommitAbortModelet )
TrafficPoliciesConfigMode.addCommandClass( FieldSetMacAddrConfigCmd )
FieldSetMacAddrConfigMode.addCommandClass( FieldSetMacAddrConfigNoAuthCmds )
FieldSetMacAddrConfigMode.addCommandClass( FieldSetMacAddrSourceConfigCmd )
MatchRuleMacConfigMode.addCommandClass( CosConfigCmd )

for _cmd in [ ActionsConfigCmd,
              EthTypeConfigCmd,
              SrcDstMacAddrCmd,
              SrcDstMacAddrFieldSetCmd,
              MatchTrackedCmd,
              VlanTagConfigCmd,
              VlanTagFieldSetConfigCmd,
            ]:
   MatchRuleMacConfigMode.addCommandClass( _cmd )
MatchRuleMacConfigMode.addShowCommandClass( ShowPendingCmd )

# AF-agnostic rules
for _mode in [ MatchRuleIpv4ConfigMode,
               MatchRuleIpv6ConfigMode ]:
   for _cmd in [ ActionsConfigCmd,
                 ProtoMatchConfigCmd,
                 ProtoNeighborsConfigCmd,
                 DscpConfigCmd,
                 TtlConfigCmd,
                 IpLengthConfigCmd,
                 ProtocolFieldSetCmd,
                 ProtocolTcpFlagsOnlyCmd,
                 MatchTrackedCmd,
                 NexthopGroupMatchConfigCmd,
                 LocationValueMatchConfigCmd,
                 ]:
      _mode.addCommandClass( _cmd )
   if toggleTrafficPolicyDestSelfMatchEnabled():
      _mode.addCommandClass( DestinationSelfConfigCmd )
   if toggleTrafficPolicyVlanMatchEnabled():
      _mode.addCommandClass( VlanConfigCmd )

for _mode in [ MatchRuleIpv4ConfigMode,
               MatchRuleIpv6ConfigMode ]:
   _mode.addCommandClass( InnerMatchCmd )
   if ( toggleTrafficPolicyVxlanPacketTypeMatchEnabled() or
      toggleTrafficPolicyMulticastPacketTypeMatchEnabled() ):
      _mode.addCommandClass( PacketTypeConfigCmd )
   if toggleTrafficPolicyVxlanPacketTypeMatchEnabled():
      _mode.addCommandClass( InnerMatchVxlanCmd )
   _mode.addCommandClass( VlanTagConfigCmd )
   _mode.addCommandClass( EthTypeConfigCmd )
   _mode.addCommandClass( VlanTagFieldSetConfigCmd )
EncapInnerMatchIpv4ConfigMode.addCommandClass( PrefixIpv4CmdV2 )
EncapInnerMatchIpv4ConfigMode.addCommandClass( PrefixFieldSetIpv4CmdV2 )
EncapInnerMatchIpv6ConfigMode.addCommandClass( PrefixIpv6CmdV2 )
EncapInnerMatchIpv6ConfigMode.addCommandClass( PrefixFieldSetIpv6CmdV2 )
if toggleTrafficPolicyVxlanPacketTypeMatchEnabled():
   PacketTypeConfigMode.addCommandClass( VxlanDecapCmd )
if toggleTrafficPolicyMulticastPacketTypeMatchEnabled():
   PacketTypeConfigMode.addCommandClass( MulticastCmd )
for _encapMode in [ EncapInnerMatchIpv4ConfigMode,
                  EncapInnerMatchIpv6ConfigMode ]:
   _encapMode.addCommandClass( ProtocolFieldSetWithoutFlagsCmd )
   _encapMode.addCommandClass( ProtocolListWithoutFlagsCmd )
   if toggleTrafficPolicyTeidMatchEnabled():
      _encapMode.addCommandClass( TeidConfigCmd )
      _encapMode.addCommandClass( TeidFieldSetConfigCmd )
EncapInnerVxlanMatchIpv4ConfigMode.addCommandClass( PrefixIpv4CmdV2 )
EncapInnerVxlanMatchIpv4ConfigMode.addCommandClass( PrefixFieldSetIpv4CmdV2 )
EncapInnerVxlanMatchIpv4ConfigMode.addCommandClass(
                                                ProtocolIpv4ListConfigCmd )
EncapDzGreConfigMode.addCommandClass( DzGreEncapSetPolicyIdCmd )
EncapDzGreConfigMode.addCommandClass( DzGreEncapSetPortIdCmd )
EncapDzGreConfigMode.addCommandClass( DzGreEncapSetSwitchIdCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetVlanConfigCmd )
FieldSetVlanConfigMode.addCommandClass( FieldSetVlanConfigNoAuthCmds )
FieldSetVlanConfigMode.addCommandClass( FieldSetVlanExceptConfigNoAuthCmds )
FieldSetVlanConfigMode.addCommandClass( FieldSetVlanRangeLimitConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetIntegerConfigCmd )
FieldSetIntegerConfigMode.addCommandClass( FieldSetIntegerConfigNoAuthCmds )
FieldSetIntegerConfigMode.addCommandClass( FieldSetIntegerExceptConfigNoAuthCmds )
FieldSetIntegerConfigMode.addCommandClass( FieldSetRangeLimitConfigCmd )
FieldSetVlanConfigMode.addCommandClass( FieldSetVlanSourceConfigCmd )
FieldSetIntegerConfigMode.addCommandClass( FieldSetIntegerSourceConfigCmd )

# AF-specific rules
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIcmpV4ConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIcmpV6ConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIcmpV4TypeCodeConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIcmpV6TypeCodeConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( PrefixIpv4Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( PrefixFieldSetIpv4Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( IpOptionsConfigCmd ) # options are only v4
MatchRuleIpv4ConfigMode.addCommandClass( MatchAllFragmentConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( FragmentOffsetConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixIpv6Cmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixFieldSetIpv6Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIpv4ListConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIpv6ListConfigCmd )

MatchRuleIpv4ConfigMode.addCommandClass( PrefixLpmIpv4Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( PrefixLpmFieldSetIpv4Cmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixLpmIpv6Cmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixLpmFieldSetIpv6Cmd )

MatchRuleIpv4ConfigMode.addCommandClass( DzGreEncapMatchCmd )

MatchRuleDefaultConfigMode.addCommandClass( ActionsConfigCmd )

for _mode in [ ActionsConfigMode, MacActionsConfigMode ]:
   _mode.addCommandClass( PoliceActionCmd )
   _mode.addCommandClass( CountActionCmd )
   _mode.addCommandClass( RedirectAggregationGroupCmd )
   _mode.addCommandClass( RedirectIntfCmd )
   _mode.addCommandClass( RedirectNexthopCmd )
   _mode.addCommandClass( RedirectNexthopGroupCmd )
   _mode.addCommandClass( RemoveDot1QOuterCmd )
   _mode.addCommandClass( SetIdTagActionCmd )
   _mode.addCommandClass( SetMacAddressActionCmd )
   _mode.addCommandClass( RemoveHeaderCmd )
   _mode.addCommandClass( SetTimestampHeaderActionCmd )
   _mode.addCommandClass( SetTcActionCmd )
   if toggleTrafficPolicyActionSetsEnabled():
      _mode.addCommandClass( ReplicateActionCmd )

MacActionsConfigMode.addCommandClass( DropActionWithoutNotifCmd )
if toggleTrafficPolicyDropWithActionEnabled():
   ActionsConfigMode.addCommandClass( DropActionWithNotifCmd )
else:
   ActionsConfigMode.addCommandClass( DropActionWithoutNotifCmd )

ReplicateMode.addCommandClass( RedirectAggregationGroupCmd )
ReplicateMode.addCommandClass( RedirectIntfCmd )
ReplicateMode.addCommandClass( RemoveDot1QOuterCmd )
ReplicateMode.addCommandClass( SetIdTagActionCmd )
ReplicateMode.addCommandClass( SetMacAddressActionCmd )

ActionsConfigMode.addCommandClass( LogActionCmd )
ActionsConfigMode.addCommandClass( SetDscpActionCmd )
ActionsConfigMode.addCommandClass( SetVrfSecondaryActionCmd )
if toggleTrafficPolicyMirrorActionEnabled():
   ActionsConfigMode.addCommandClass( MirrorActionCmd )
ActionsConfigMode.addCommandClass( SflowActionCmd )
ActionsConfigMode.addCommandClass( GotoActionCmd )

TrafficPoliciesConfigMode.addCommandClass( FieldSetIpPrefixConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetIpv6PrefixConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetL4PortConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( LocationAliasConfigCmd )
if toggleUdfEncapGtpMatchEnabled() or toggleUdfEncapVxlanMatchEnabled():
   TrafficPoliciesConfigMode.addCommandClass( LocationAliasConfigEncapCmd )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortConfigNoAuthCmds )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortExceptConfigNoAuthCmds )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortSourceConfigCmd )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4RangeLimitConfigCmd )

# Add field-set commands to appropriate modes
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpv4PrefixSourceConfigCmd )
FieldSetIpv6PrefixConfigMode.addCommandClass( FieldSetIpv6PrefixSourceConfigCmd )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetLimitConfigCmd )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixConfigNoAuthCmds )
FieldSetIpv6PrefixConfigMode.addCommandClass( FieldSetLimitConfigCmd )
FieldSetIpv6PrefixConfigMode.addCommandClass( FieldSetIpv6PrefixConfigNoAuthCmds )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixExceptConfigNoAuthCmds )
FieldSetIpv6PrefixConfigMode.addCommandClass(
      FieldSetIpv6PrefixExceptConfigNoAuthCmds )

for m in [ TrafficPolicyConfigMode, MatchRuleIpv4ConfigMode,
           MatchRuleIpv6ConfigMode, MatchRuleDefaultConfigMode, ActionsConfigMode,
           EncapInnerMatchIpv4ConfigMode, EncapInnerMatchIpv6ConfigMode,
           EncapInnerVxlanMatchIpv4ConfigMode, EncapInnerMatchMacConfigMode,
           EncapDzGreConfigMode ]:
   m.addShowCommandClass( ShowPendingCmd )

def guardAegisInterface( mode, token ):
   # We support ingress traffic-policy on Strata through
   # DMF but not through the CLI. So add a guard here for
   # Strata
   if not interfaceTrafficPolicyHwStatus.ingressTrafficPolicyCliSupported:
      return CliParser.guardNotThisPlatform

   if ( interfaceTrafficPolicyHwStatus.ingressTrafficPolicySupported or
         interfaceTrafficPolicyHwStatus.egressTrafficPolicySupported ):
      return None
   return CliParser.guardNotThisPlatform

nodeCounterGlobalConfig = CliCommand.singleKeyword(
      'counter', 'Traffic policy counter parameters' )
nodeInterface = CliCommand.guardedKeyword(
      'interface', 'Set behavior for interface traffic policy', guardAegisInterface )

# --------------------------------------------------------------------------
# The "update interface default action drop" command
# --------------------------------------------------------------------------
class UpdateDefaultActionConfigCmd( CliCommand.CliCommandClass ):
   syntax = "update interface default action drop"
   noOrDefaultSyntax = "update interface default action ..."
   data = {
      'update' : 'Traffic policy update parameters',
      'interface' : nodeInterface,
      'default' : 'Set default behavior during update',
      'action' : 'Set default action during update',
      'drop' : 'Set default action during update to drop'
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.actionDuringUpdate = "drop"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.actionDuringUpdate = "permit"

TrafficPoliciesConfigMode.addCommandClass( UpdateDefaultActionConfigCmd )

# --------------------------------------------------------------------------
# The "update interface hitless strict" command
# --------------------------------------------------------------------------
def hitlessStrictGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.hitlessStrictSupported:
      return None
   return CliParser.guardNotThisPlatform

class HitfulUpdateSupportCmd( CliCommand.CliCommandClass ):
   syntax = "update interface hitless strict"
   noOrDefaultSyntax = syntax
   data = {
      'update' : 'Traffic policy update parameters',
      'interface' : nodeInterface,
      'hitless' : CliCommand.guardedKeyword(
         'hitless', helpdesc='Hitless policy updates',
         guard=hitlessStrictGuard ),
      'strict' : 'Only allow hitless policy updates',
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.allowHitfulUpdate = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.allowHitfulUpdate = True

TrafficPoliciesConfigMode.addCommandClass( HitfulUpdateSupportCmd )

# --------------------------------------------------------------------------
# The "counter interface per-interface ingress" command
# --------------------------------------------------------------------------
def ingressCounterPerInterfaceGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.ingressCountersPerInterfaceSupported:
      return None
   return CliParser.guardNotThisPlatform

class CounterIntfPerInterfaceIngressConfigCmd( CliCommand.CliCommandClass ):
   syntax = "counter interface per-interface ingress"
   noOrDefaultSyntax = syntax
   data = {
      'counter' : nodeCounterGlobalConfig,
      'interface' : nodeInterface,
      'per-interface' : 'Set count to be per-interface',
      'ingress' : CliCommand.guardedKeyword(
         'ingress', helpdesc='Set parameter for ingress traffic policies',
         guard=ingressCounterPerInterfaceGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.ingressCounterGranularity = "counterPerInterface"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.ingressCounterGranularity = "counterPerPolicy"

TrafficPoliciesConfigMode.addCommandClass( CounterIntfPerInterfaceIngressConfigCmd )

# --------------------------------------------------------------------------
# For the cli
# "transforms interface prefix common source-destination"
# --------------------------------------------------------------------------
def aegisXformGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.aegisXformSupported:
      return None
   return CliParser.guardNotThisPlatform

def aegisCommonTrieGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.aegisCommonTrieSupported:
      return None
   return CliParser.guardNotThisPlatform

class CommSrcDstTrieConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( "transforms interface prefix common source-destination" )
   noOrDefaultSyntax = syntax
   data = {
      'transforms' : CliCommand.guardedKeyword( 'transforms',
         helpdesc='Traffic policy transformed labels',
         guard=aegisCommonTrieGuard ),
      'interface' : nodeInterface,
      'prefix' : 'Prefixes of IP addresses',
      'common' : 'Common trie',
      'source-destination' : 'Common trie for source and destination IP',
   }

   @staticmethod
   def _setCommonTrieMode( mode, args, trieMode ):
      commonTrieInfo = Tac.Value( 'TrafficPolicy::CommonTrieInfo' )
      # set IPv4 for outer
      commonTrieInfo.commonOuterIpv4XformTrie = trieMode
      # set IPv6 for outer
      commonTrieInfo.commonOuterIpv6XformTrie = trieMode
      # set IPv4 for inner
      commonTrieInfo.commonInnerIpv4XformTrie = trieMode
      # set IPv6 for inner
      commonTrieInfo.commonInnerIpv6XformTrie = trieMode
      policiesIntfParamConfig.commonTrieConfig = commonTrieInfo
   @staticmethod
   def handler( mode, args ):
      CommSrcDstTrieConfigCmd._setCommonTrieMode( mode, args, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      CommSrcDstTrieConfigCmd._setCommonTrieMode( mode, args, False )

TrafficPoliciesConfigMode.addCommandClass( CommSrcDstTrieConfigCmd )

# --------------------------------------------------------------------------
# "transforms interface prefix scale outer"
# --------------------------------------------------------------------------
class PrefixScaleOuterConfigCmd( CliCommand.CliCommandClass ):
   syntax = "transforms interface prefix scale outer"
   noOrDefaultSyntax = syntax
   data = {
      'transforms' : CliCommand.guardedKeyword( 'transforms',
         helpdesc='Traffic policy transformed labels',
         guard=aegisXformGuard ),
      'interface' : 'Set behavior for interface traffic policy',
      'prefix' : 'Prefixes of IP addresses',
      'scale' : 'Prefix scale',
      'outer' : 'Prefer higher outer prefix scale',
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.prefixScaleOuter = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.prefixScaleOuter = False

TrafficPoliciesConfigMode.addCommandClass( PrefixScaleOuterConfigCmd )

def aegisProportionalLagPolicerGuard( mode, token ):
   if interfaceTrafficPolicyHwStatus.proportionalLagPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

# --------------------------------------------------------------------------
# "policer interface port-channel distribution proportional"
# --------------------------------------------------------------------------
class ProportionLagPolicerConfigCmd( CliCommand.CliCommandClass ):
   syntax = "policer interface port-channel distribution proportional"
   noOrDefaultSyntax = syntax
   data = {
      'policer' : CliCommand.guardedKeyword(
         'policer',
         helpdesc='Traffic policy policer parameters',
         guard=aegisProportionalLagPolicerGuard ),
      'interface' : 'Set behavior for interface traffic policy',
      'port-channel' : 'Set behavior for policer on LAG interfaces',
      'distribution' : 'Set distribution of policer on LAG interfaces',
      'proportional' :
         'Set distribution of policer on LAG interfaces to be proportional'
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.proportionalLagPolicerEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.proportionalLagPolicerEnabled = False

if toggleTrafficPolicyProportionalLagPolicerEnabled():
   TrafficPoliciesConfigMode.addCommandClass( ProportionLagPolicerConfigCmd )

def noTrafficPolicies( mode, args ):
   CommSrcDstTrieConfigCmd.noOrDefaultHandler( mode, args )
   CounterIntfPerInterfaceIngressConfigCmd.noOrDefaultHandler( mode, args )
   HitfulUpdateSupportCmd.noOrDefaultHandler( mode, args )
   PrefixScaleOuterConfigCmd.noOrDefaultHandler( mode, args )
   UpdateDefaultActionConfigCmd.noOrDefaultHandler( mode, args )
   ProportionLagPolicerConfigCmd.noOrDefaultHandler( mode, args )

# pylint: disable=protected-access
TrafficPoliciesConfigCmd._registerNoHandler( noTrafficPolicies )
# pylint: enable=protected-access

def Plugin( em ):
   global policiesCliConfig, policiesIntfParamConfig
   global policiesStatusRequestDir, policiesStatus
   global policyOpChkr
   global qosHwStatus
   global qosAclHwStatus
   global entityManager
   global cpuCounter
   global intfInputCounter
   global intfOutputCounter
   global fieldSetConfig
   global urlFieldSetConfig
   global interfaceTrafficPolicyHwStatus
   global tapAggStatus
   global tapAggHwStatus
   global counterStatusInput
   global counterStatusOutput
   global cliNexthopGroupConfig
   global locationAliasConfig
   global routingHwStatus

   policiesRootNode = 'trafficPolicies'
   policiesCellRootNode = 'cell/%d/trafficPolicies' % Cell.cellId()
   statusNode = 'status'
   policiesCliConfigNode = 'input/cli'
   policiesCliConfigPath = policiesRootNode + '/' + policiesCliConfigNode
   policiesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesIntfParamConfigPath = policiesRootNode + '/param/config/interface'
   policiesIntfParamConfigType = 'TrafficPolicy::TrafficPolicyIntfParamConfig'
   policiesStatusPath = policiesCellRootNode + '/' + statusNode
   policiesStatusType = 'Tac::Dir'
   statusRequestDirNode = 'statusRequest/cli'
   policiesStatusRequestDirPath = policiesRootNode + '/' + statusRequestDirNode
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'
   policiesCounterNode = 'counter'
   policiesCounterPath = policiesRootNode + '/' + policiesCounterNode
   policiesCounterType = 'TrafficPolicy::Counter'
   entityManager = em

   policiesStatus = LazyMount.mount( entityManager, policiesStatusPath,
                                     policiesStatusType, 'ri' )

   policiesCliConfig = ConfigMount.mount( entityManager, policiesCliConfigPath,
                                          policiesCliConfigType, 'wi' )
   policiesIntfParamConfig = ConfigMount.mount(
      entityManager, policiesIntfParamConfigPath, policiesIntfParamConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType,
                                               'wc' )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosAclHwStatus = LazyMount.mount( entityManager,
                                     "qos/hardware/acl/status/global",
                                     "Qos::AclHwStatus", "r" )
   cpuCounter = LazyMount.mount( entityManager, policiesCounterPath + '/cpu',
                                 policiesCounterType, "wr" )
   intfInputCounter = LazyMount.mount( entityManager,
                                       policiesCounterPath + '/interface/input',
                                       policiesCounterType, "wr" )
   intfOutputCounter = LazyMount.mount( entityManager,
                                        policiesCounterPath + '/interface/output',
                                        policiesCounterType, "wr" )
   cliNexthopGroupConfig = LazyMount.mount( entityManager,
                                            'routing/nexthopgroup/input/cli',
                                            'Routing::NexthopGroup::ConfigInput',
                                            'r' )
   fieldSetConfig = ConfigMount.mount( entityManager,
                                       'trafficPolicies/fieldset/input/cli',
                                       'Classification::FieldSetConfig', 'w' )
   urlFieldSetConfig = LazyMount.mount( entityManager,
                                        'trafficPolicies/fieldset/input/url',
                                        'Classification::FieldSetConfig', 'w' )
   locationAliasConfig = ConfigMount.mount( entityManager,
                                       'trafficPolicies/udfconfig/cli',
                                       'Classification::UdfConfig', 'w' )
   interfaceTrafficPolicyHwStatus = (
         LazyMount.mount( entityManager,
                          "trafficPolicies/hardware/status/interface",
                          "TrafficPolicy::HwStatus", "r" )
   )
   tapAggStatus = LazyMount.mount( entityManager, 'tapagg/status',
                                   'TapAgg::Status', 'r' )
   tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                     'TapAgg::HwStatus', 'r' )
   routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
                                      'Routing::Hardware::Status', 'r' )
   policyOpChkr = PolicyMapCliLib.PolicyOpChkr( policiesStatusRequestDir,
                                                policiesStatus )
   registerFieldSetConfigValidCallbacks( isPolicyHwStatusValid )

   sharkMountInfo = SharkLazyMount.mountInfo( 'shadow' )
   counterMountPath = "policyMap/counters/"
   autoUnmount = True
   counterStatusInput = SharkLazyMount.mount(
      entityManager,
      counterMountPath + "trafficPolicyInterface/input",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount )
   counterStatusOutput = SharkLazyMount.mount(
      entityManager,
      counterMountPath + "trafficPolicyInterface/output",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount )
