#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Definition of the CAPI model. All the rendering logic is deferred
and takes place in TrafficPolicyShowCliHelper(tac/tin)
"""

from __future__ import absolute_import, division, print_function
import TableOutput
from ArnetModel import ( IpGenericPrefix, MacAddress, IpGenericAddress )
from CliModel import ( Bool, Model, DeferredModel, Dict, Int, List, Str, Enum,
                       Submodel, Float )
from CliCommon import CliModelNotDegradable
from IntfModels import Interface
import Tac
from Toggles.TrafficPolicyToggleLib import (
      toggleTrafficPolicyEnforceGtsmEnabled,
      toggleTrafficPolicyVxlanPacketTypeMatchEnabled,
      toggleTrafficPolicyMulticastPacketTypeMatchEnabled,
)
class TcpFlag( DeferredModel ):
   name = Str( help="TCP flag name" )
   value = Bool( help="TCP flag bit value" )

class NumericalRange( DeferredModel ):
   low = Int( help="Minimum value" )
   high = Int( help="Maximum value" )

class HexNumericalRange( DeferredModel ):
   low = Str( help="Minimum value" )
   high = Str( help="Maximum value" )

class IcmpType( DeferredModel ):
   icmpTypes = Submodel( valueType=NumericalRange, help="ICMP type", optional=True )
   icmpCodes = List( valueType=NumericalRange, help="ICMP code", optional=True )

class L4PortFieldSet( DeferredModel ):
   name = Str( help="Field-set name", optional=True )
   numPorts = Int( help="Number of ports in field-set", optional=True )
   undefined = Bool( help="L4 port field-set has not been defined", optional=True )

class Port( DeferredModel ):
   srcPorts = List( valueType=NumericalRange, help="Source port", optional=True )
   destPorts = List( valueType=NumericalRange, help="Destination port",
                     optional=True )
   srcExtL4PortFieldSet = List( valueType=L4PortFieldSet,
                                help="Source l4-port field-set info",
                                optional=True )
   destExtL4PortFieldSet = List( valueType=L4PortFieldSet,
                                 help="Destination l4-port field-set info",
                                 optional=True )
   # Start of deprecated attributes. These will be retained in the model
   # until a new model revision is absolutely necessary.
   srcL4PortFieldSets = List( valueType=str,
                              help='(Deprecated - refer to "srcExtL4PortFieldSet") '
                                   'Source l4-port field-sets"',
                              optional=True )
   destL4PortFieldSets = List( valueType=str,
                               help='(Deprecated - refer to "destExtL4PortFieldSet"'
                                    ') Destination l4-port field-sets',
                               optional=True )

class Vni( DeferredModel ):
   vni = Int( help="vni", optional=True )
   vlan = Interface( help="dynamic vlan interface assigned for the vni",
                      optional=True )

class TcpFlags( DeferredModel ):
   tcpFlags = List( valueType=TcpFlag, help="List of ANDed TCP flags",
                    optional=True )

class ProtocolAndPort( DeferredModel ):
   protocolRange = Submodel( valueType=NumericalRange, help="Protocol",
                             optional=True )
   ports = List( valueType=Port, help="List of L4 ports that can be ORed together",
                 optional=True )

class Protocol( ProtocolAndPort ):
   __revision__ = 2
   icmps = List( valueType=IcmpType, help="ICMP info", optional=True )
   tcpFlagGroups = List( valueType=TcpFlags,
                         help="List of TCP flags that can be ORed together",
                         optional=True )

class Fragment( DeferredModel ):
   matchAllFragments = Bool( help="Match all fragments", optional=True )
   fragmentOffsets = List( valueType=NumericalRange, help="Fragment offset",
                           optional=True )

class MacAddressInfo( DeferredModel ):
   macAddr = MacAddress( help="MAC address" )
   # BUG706397: macAddrMask entry is populated when we support MAC address mask
   #macAddrMask = MacAddress( help='MAC address mask' )

class TeidFieldSet( DeferredModel ):
   name = Str( help="Name of TEID field-set", optional=True )
   numTeids = Int( help="Number of TEIDs in field-set", optional=True )
   undefined = Bool( help="TEID field-set has not been defined", optional=True )
   fieldSetType = Enum( values=( "integer", ), help="Type of field-set",
                        optional=True )

class PrefixFieldSet( DeferredModel ):
   name = Str( help="Name of prefix field-set" )
   numPrefixes = Int( help="Number of accept prefixes in field-set", optional=True )
   numExceptPrefixes = Int( help="Number of except prefixes in field-set",
                            optional=True )
   undefined = Bool( help="Prefix field-set has not been defined", optional=True )

class VlanFieldSet( DeferredModel ):
   name = Str( help="Name of VLAN field-set", optional=True )
   numVlans = Int( help="Number of VLANs in field-set", optional=True )
   undefined = Bool( help="VLAN field-set has not been defined", optional=True )

class ServiceFieldSet( DeferredModel ):
   name = Str( help="Name of service field-set", optional=True )
   numProtocols = Int( help="Number of protocols in field-set", optional=True )
   undefined = Bool( help="Service field-set has not been defined", optional=True )

class MacAddrFieldSet( DeferredModel ):
   name = Str( help="Name of MAC field-set" )
   numMacAddresses = Int( help="Number of MAC addresses in field-set",
                          optional=True )
   undefined = Bool( help="MAC field-set has not been defined", optional=True )

class CosMatch( DeferredModel ):
   coss = List( valueType=NumericalRange, help="Class of Service values",
                optional=True )

class Dot1QMatch( DeferredModel ):
   dot1qVlans = List( valueType=NumericalRange, help="Outer VLAN tags",
                      optional=True )
   dot1qVlanFieldSets = List( valueType=VlanFieldSet, help="Outer VLAN field sets",
                              optional=True )
   innerVlans = List( valueType=NumericalRange, help="Inner VLAN tags",
                      optional=True )

class Encapsulation( DeferredModel ):
   if toggleTrafficPolicyVxlanPacketTypeMatchEnabled():
      encapsulationType = Enum( values=( "gtpv1", "dzgre", "vxlan" ),
                                help="Encapsulation type",
                                optional=True )
   else:
      encapsulationType = Enum( values=( "gtpv1", "dzgre" ),
                                help="Encapsulation type",
                                optional=True )

class PacketTypes( DeferredModel ):
   if toggleTrafficPolicyVxlanPacketTypeMatchEnabled():
      vxlanDecap = Enum( values=( 'vxlanDecapOnly', 'vxlanDecapExclude' ),
                         help="Vxlan Decap packet type values", optional=True )

   if toggleTrafficPolicyMulticastPacketTypeMatchEnabled():
      multicast = Enum( values=( 'multicastOnly', ),
                        help="Mutlicast PacketType values", optional=True )

class ValueMask( DeferredModel ):
   value = Str( help="Value to match", optional=True )
   mask = Str( help="Mask for the match", optional=True )

class LocationMatch( DeferredModel ):
   alias = Str( help="Location alias name", optional=True )
   matches = List( valueType=ValueMask, help="Location alias matches",
                   optional=True )

class ApplicationProfile( DeferredModel ):
   name = Str( help="Name of application profile" )
   apps = Int( help="Number of apps in the profile", optional=True )
   undefined = Bool( help="Application profile has not been defined", optional=True )

class DestSelf( DeferredModel ):
   selfIpUnicast = Bool( help="Unicast packets to CPU for self IP", optional=True )

# See https://groups.google.com/a/arista.com/g/cli-dev/c/gRHwUMsbVxA/m/tpsA6EYpBAAJ
# In order to support innerMatch containing all the attributes of a Match, we need
# to define a base model that contains a fully defined Match. All new match
# attributes should be added to MatchesBase
class MatchesBase( DeferredModel ):
   # Only used by encap packets, outer packet has matchOption on outer model
   addressFamily = Enum( values=( "ipv4", "ipv6", "mac" ),
                         help="Address family of encapsulated packet",
                         optional=True )
   dot1qVlanTags = List( valueType=Dot1QMatch, help="802.1Q VLAN tags",
                         optional=True )
   vlans = List( valueType=NumericalRange, help="VLAN IDs", optional=True )
   cosData = List( valueType=CosMatch, help="COS match criteria",
                   optional=True )
   destPrefixes = List( valueType=IpGenericPrefix,
                        help="Destination prefix", optional=True )
   srcPrefixes = List( valueType=IpGenericPrefix,
                       help="Source prefix", optional=True )
   destLpmPrefixes = List( valueType=IpGenericPrefix,
                           help="Destination prefix (longest prefix match)",
                           optional=True )
   srcLpmPrefixes = List( valueType=IpGenericPrefix,
                          help="Source prefix (longest prefix match)",
                          optional=True )
   srcExtPrefixSet = List( valueType=PrefixFieldSet,
                           help="Source prefix field-sets", optional=True )
   destExtPrefixSet = List( valueType=PrefixFieldSet,
                            help="Destination prefix field-sets", optional=True )
   srcLpmPrefixSet = List( valueType=PrefixFieldSet,
                           help="Source prefix field-sets (longest prefix match)",
                           optional=True )
   destLpmPrefixSet = List( valueType=PrefixFieldSet,
                            help="Destination prefix field-sets "
                            "(longest prefix match)", optional=True )
   protocols = List( valueType=Protocol, help="Protocol",
                     optional=True )
   serviceSet = List( valueType=ServiceFieldSet,
                      help="Service field-sets",
                      optional=True )
   ipLengths = List( valueType=NumericalRange, help="IP length",
                     optional=True )
   fragment = Submodel( valueType=Fragment, help="Fragment", optional=True )
   dscps = List( valueType=NumericalRange, help="DSCP", optional=True )
   ecns = List( valueType=NumericalRange, help="ECN", optional=True )
   ttls = List( valueType=NumericalRange, help="TTL", optional=True )
   neighborProtocol = Str( help="Neighbor protocol", optional=True )
   matchL4Protocol = Str( help="Protocol BGP", optional=True )
   matchIpOptions = Bool( help="Match on any IP options", optional=True )
   matchTracked = Bool( help="Match on any tracked packet", optional=True )
   encapsulation = Submodel( valueType=Encapsulation,
                             help="Encapsulation match criteria", optional=True )
   packetTypes = Submodel( valueType=PacketTypes,
                           help="PacketType match criteria", optional=True )
   teids = List( valueType=NumericalRange, help="Tunnel endpoint identifier",
                 optional=True )
   teidFieldSets = List( valueType=TeidFieldSet, help="TEID field-sets",
                         optional=True )
   ethTypesExt = List( valueType=HexNumericalRange, help="Ethernet type",
                       optional=True )
   srcMacs = List( valueType=MacAddress, help="Source MAC addresses",
                   optional=True )
   dstMacs = List( valueType=MacAddress, help="Destination MAC addresses",
                   optional=True )
   srcMacAddrSet = List( valueType=MacAddrFieldSet,
                         help="Source MAC field-set", optional=True )
   dstMacAddrSet = List( valueType=MacAddrFieldSet,
                         help="Destination MAC field-set", optional=True )
   nexthopGroups = List( valueType=str,
                         help="Next-hop groups", optional=True )
   appProfiles = List( valueType=ApplicationProfile,
                        help="Application profile names", optional=True )
   locations = List( valueType=LocationMatch,
                        help="Location", optional=True )
   destSelf = Submodel( valueType=DestSelf,
                        help="Destination prefix self", optional=True )
   dzGreSwitchIds = List( valueType=NumericalRange,
                          help="DzGRE Switch Id", optional=True )
   dzGrePortIds = List( valueType=NumericalRange,
                          help="DzGRE Port Id", optional=True )
   dzGrePolicyIds = List( valueType=NumericalRange,
                          help="DzGRE Policy Id", optional=True )

   # Start of deprecated attributes. These will be retained in the model
   # until a new model revision is absolutely necessary.
   srcPrefixSets = List( valueType=str,
                         help='Deprecated - refer to "srcExtPrefixSet") '
                              'Source prefix field-sets', optional=True )
   destPrefixSets = List( valueType=str,
                          help='(Deprecated - refer to "destExtPrefixSet") '
                               'Destination prefix field-sets', optional=True )
   ethTypes = List( valueType=NumericalRange,
                    help='(Deprecated - refer to "ethTypesExt") Ethernet type',
                    optional=True )

# XXX This model cannot support inner match with inner match
class Matches( MatchesBase ):
   description = Str( "Match description", optional=True )
   innerMatch = Submodel( valueType=MatchesBase, help="Inner packet match criteria",
                          optional=True )
   dzGreMatch = Submodel( valueType=MatchesBase,
                          help="DzGRE packet match criteria",
                          optional=True )

class PoliceRate( DeferredModel ):
   rate = Int( help="Lower police rate" )
   unit = Str( help="Unit of police rate" )
   burstSize = Int( help="Lower burst size", optional=True )
   burstUnit = Str( help="Unit of police burst size", optional=True )

class CountData( DeferredModel ):
   packetCount = Int( help="Number of packets matched", optional=True )
   byteCount = Int( help="Number of bytes matched", optional=True )
   undefined = Bool( help="Named counter has not been defined", optional=True )

class TapAggRedirect( DeferredModel ):
   groups = List( valueType=str,
                    help="List of aggregation groups" )
   interfaces = List( valueType=str,
                   help="List of tool interfaces" )

class SetIdentityTagModel( DeferredModel ):
   idTag = Int( help='Id-tag value' )
   innerIdTag = Int( help='Inner id-tag value' )

class NexthopInfo( DeferredModel ):
   nexthops = List( valueType=IpGenericAddress, help="List of next hops" )

class RedirectNexthop( DeferredModel ):
   recursive = Bool( help="Enable recursive next hop resolution", optional=True )
   vrfName = Str( help="Resolve next hop in a VRF", optional=True )
   nexthops = List( valueType=IpGenericAddress,
                    help="List of configured next hops" )
   resolvedNexthops = Dict(
      keyType=str, valueType=NexthopInfo,
      help="A mapping of ingress VRF name to its active next hops",
      optional=True )

class GotoAction( DeferredModel ):
   nextMatch = Bool( help="Match on the next match rule", optional=True )
   matchName = Str( help="Specified match rule name to go to next", optional=True )

class SetMacAddressModel( DeferredModel ):
   destMacAddr = MacAddress( help='MAC address for the destination' )
   srcMacAddr = MacAddress( help='MAC address for the source' )

class SetHeaderRemoveModel( DeferredModel ):
   size = Int( help='The number of bytes (after the Ethernet header, including any '
                    'VLAN tags) to remove' )
   preserveEth = Bool( help='Preserve the original Ethernet header, including VLAN '
                            ' tags if any' )

class SetVrfSecondaryModel( DeferredModel ):
   useVrfSecondaryResult = Bool( help="Use secondary VRF route result",
                                 optional=True )
   vrfName = Str( help="Secondary forwarding VRF name", optional=True )

class SetDecapVrfModel( DeferredModel ):
   decapVrf = Str( help="Decapsulation VRF name", optional=False )
   fallbackVrf = Str( help="Fallback VRF name", optional=False )
   forwardVrf = Str( help="Forwarding VRF name", optional=False )

class TotalInfo( DeferredModel ):
   packetTotal = Int( help="Total packet count", optional=True )
   byteTotal = Int( help="Total byte count", optional=True )

class CounterRateAndUnit( DeferredModel ):
   rate = Float( help="Counter rate above/below the police rate limit",
                 optional=True )
   unit = Enum( values=( 'bps', 'kbps', 'mbps', 'gbps', 'tbps' ),
                help="Unit of counter rate above/below the police rate limit",
                optional=True )

class PoliceCount( DeferredModel ):
   below = Submodel( valueType=TotalInfo, help="Number of packets below police rate",
                     optional=True )
   above = Submodel( valueType=TotalInfo, help="Number of packets above police rate",
                     optional=True )
   belowRate = Submodel( valueType=CounterRateAndUnit,
                         help="Counter rate below the police rate limit",
                         optional=True )
   aboveRate = Submodel( valueType=CounterRateAndUnit,
                         help="Counter rate above the police rate limit",
                         optional=True )

class ReplicatePacketActions( DeferredModel ):
   stripHeaderBytes = Str( help='Remove 802.1Q outer tag', optional=True )
   setIdentityTag = Submodel( valueType=SetIdentityTagModel,
         help='Set Dot1Q identity tag', optional=True )
   tapAggRedirect = Submodel( valueType=TapAggRedirect,
         help='Redirect traffic action in Tap Aggregation mode', optional=True )
   setMacAddress = Submodel( valueType=SetMacAddressModel,
         help='Set MAC address', optional=True )

class ReplicatePacket( DeferredModel ):
   namedActionSets = Dict( keyType=str, valueType=ReplicatePacketActions,
         help="Action sets, keyed by action set name" )

DropWithMessageType4 = Tac.Type( 'PolicyMap::DropWithMessageType4' )
DropWithMessageType6 = Tac.Type( 'PolicyMap::DropWithMessageType6' )

class DropNotification( DeferredModel ):
   # For ICMP or ICMPv6 messages, includes the type and code. For tcp-reset,
   # messageType only
   messageType = Enum(
         values=( DropWithMessageType4.attributes +
                  DropWithMessageType6.attributes ),
         help="Message type" )
   icmpType = Int( help="ICMP type", optional=True )
   icmpCode = Int( help="ICMP code", optional=True )

class VerifyFlowReturnFailure( DeferredModel ):
   action = Enum( values=( 'drop', ),
                  help="Failure action", optional=True )

class Actions( DeferredModel ):
   drop = Bool( help="Drop action", optional=True )
   dropNotification = Submodel( valueType=DropNotification,
                                help="Drop notification action",
                                optional=True )
   police = Submodel( valueType=PoliceRate, help="Police action", optional=True )
   policeCount = Submodel( valueType=PoliceCount,
                           help="Descriptive count for police", optional=True )
   inputCounter = Int( help="Number of packets hitting the rule applied on input",
                       optional=True )
   inputNamedCounters = (
      Dict( keyType=str, valueType=CountData,
            help="A mapping of counter name to its input count data",
            optional=True )
   )
   outputCounter = Int( help="Number of packets hitting the rule applied on output",
                        optional=True )
   outputNamedCounters = (
      Dict( keyType=str, valueType=CountData,
            help="A mapping of counter name to its output count data",
            optional=True )
   )
   setDscp = Int( help="Set header DSCP value", optional=True )
   setTc = Int( help="Set forwarding traffic class", optional=True )
   setTtl = Int( help='Set header TTL value', optional=True )
   setVrf = Str( help="Set forwarding VRF", optional=True )
   setVrfSecondary = Submodel( valueType=SetVrfSecondaryModel,
                               help="Set secondary forwarding VRF",
                               optional=True )
   setDecapVrf = Submodel( valueType=SetDecapVrfModel,
                           help="Set Decap VRF",
                           optional=True )

   log = Bool( help="Log action", optional=True )
   sample = Bool( help="Sample action", optional=True )
   sampleAll = Bool( help="Sample all action", optional=True )
   goto = Submodel( valueType=GotoAction,
                    help="Match on the specified rule next",
                    optional=True )
   stripHeaderBytes = Str( help='Remove 802.1Q outer tag', optional=True )
   setIdentityTag = Submodel( valueType=SetIdentityTagModel,
         help='Set Dot1Q identity tag', optional=True )
   setTimestampHeader = Bool( help="Set MAC timestamp header", optional=True )
   replicatePacket = Submodel( valueType=ReplicatePacket,
         help='Replicate packet actions', optional=True )
   tapAggRedirect = Submodel( valueType=TapAggRedirect,
         help='Redirect traffic action in Tap Aggregation mode', optional=True )
   redirectInfo = Submodel( valueType=TapAggRedirect,
         help='Redirect traffic action', optional=True )
   redirectNexthopGroups = List( valueType=str,
         help="Action of redirect traffic to next-hop group(s)", optional=True )
   redirectNexthop = Submodel( valueType=RedirectNexthop,
                               help="Action of redirect traffic to next-hop(s)",
                               optional=True )
   redirectNexthopInterface = \
        Interface( help="Action of redirect traffic to a next-hop interface",
                   optional=True )
   mirrorSession = Str( help="Set mirror action", optional=True )
   sflow = Bool( help="Sflow action", optional=True )
   setMacAddress = Submodel( valueType=SetMacAddressModel,
         help='Set MAC address', optional=True )
   setHeaderRemove = Submodel( valueType=SetHeaderRemoveModel,
                               help='Set header remove', optional=True )
   verifyFlowReturnFailure = Submodel( valueType=VerifyFlowReturnFailure,
                               help='Verify flow return failure', optional=True )

   # Start of deprecated actions attributes. These will be retained in the model
   # until a new model revision is absolutely necessary.
   count = Int( help='(Deprecated - refer to the "inputCounter" attribute) '
                     'Number of packets hitting the rule applied on input',
                optional=True )
   countNamed = Dict( keyType=str, valueType=CountData,
                      help='(Deprecated - refer to the "inputNamedCounters" '
                           'attribute) A mapping of counter name to its '
                           'input count data',
                      optional=True )
   # End of deprecated actions attributes.

class RuleSummary( DeferredModel ):
   ruleString = Str( help="Match rule name" )
   matchOption = Str( help="Match on packet type" )

class Rule( RuleSummary ):
   matches = Submodel( valueType=Matches, help="Matches for the rule",
                       optional=True )
   actions = Submodel( valueType=Actions, help="Actions for the rule",
                       optional=True )

class TrafficPolicyIntfErrorInfo( DeferredModel ):
   ipv4FailedIntfs = List( valueType=Interface,
                           help="Failed interfaces for IPv4 traffic", optional=True )
   ipv6FailedIntfs = List( valueType=Interface,
                           help="Failed interfaces for IPv6 traffic", optional=True )
   macFailedIntfs = List( valueType=Interface,
                          help="Failed interfaces for MAC traffic", optional=True )
   ipv4FailedVni = List( valueType=Vni,
                         help="Failed VNIs for IPv4 traffic", optional=True )
   ipv6FailedVni = List( valueType=Vni,
                         help="Failed VNIs for IPv6 traffic", optional=True )
   macFailedVni = List( valueType=Vni,
                        help="Failed VNIs for MAC traffic", optional=True )
   ipv4FailedVrfs = List( valueType=str, help="Failed VRFs for IPv4 traffic",
                          optional=True )
   ipv6FailedVrfs = List( valueType=str, help="Failed VRFs for IPv6 traffic",
                          optional=True )
   macFailedVrfs = List( valueType=str,
                         help="Failed VRFs for MAC traffic", optional=True )
   ipv4InstallationErrors = List( valueType=str,
                                  help="List of IPv4 installation errors",
                                  optional=True )
   ipv6InstallationErrors = List( valueType=str,
                                  help="List of IPv6 installation errors",
                                  optional=True )
   macInstallationErrors = List( valueType=str,
                                 help="List of MAC installation errors",
                                 optional=True )
   ipv4VrfInstallationErrors = List( valueType=str,
                                  help="List of IPv4 VRF installation errors",
                                  optional=True )
   ipv6VrfInstallationErrors = List( valueType=str,
                                  help="List of IPv6 VRF installation errors",
                                  optional=True )
   macVrfInstallationErrors = List( valueType=str,
                                 help="List of MAC VRF installation errors",
                                 optional=True )

class TrafficPolicyVAclErrorInfo( DeferredModel ):
   ipv4FailedVlans = List( valueType=str,
                           help="Failed vlans for IPv4 traffic", optional=True )
   ipv6FailedVlans = List( valueType=str,
                           help="Failed vlans for IPv6 traffic", optional=True )
   ipv4InstallationErrors = List( valueType=str,
                                  help="List of IPv4 installation errors",
                                  optional=True )
   ipv6InstallationErrors = List( valueType=str,
                                  help="List of IPv6 installation errors",
                                  optional=True )

class TrafficPolicyErrors( DeferredModel ):
   __revision__ = 2
   # Start of deprecated attributes. These will be retained in the model
   # until a new model revision is absolutely necessary.
   ipv4FailedVrfs = List( valueType=str,
                          help="Deprecated - refer to cpu.ipv4FailedVrfs",
                          optional=True )
   ipv6FailedVrfs = List( valueType=str,
                          help="Deprecated - refer to cpu.ipv6FailedVrfs",
                          optional=True )
   ipv4FailedVrfsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv4FailedVrfs",
         optional=True )
   ipv6FailedVrfsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv6FailedVrfs",
         optional=True )
   ipv4InstallationErrors = List( valueType=str,
                                  help='List of IPv4 installation errors. '
                                       'For traffic-policies applied on interfaces, '
                                       'refer to the "ipv4InstallationErrors" in '
                                       'the input/output TrafficPolicyIntfErrorInfo '
                                       'sub-models.',
                                  optional=True )
   ipv6InstallationErrors = List( valueType=str,
                                  help='List of IPv6 installation errors. '
                                       'For traffic-policies applied on interfaces, '
                                       'refer to the "ipv6InstallationErrors" in '
                                       'the input/output TrafficPolicyIntfErrorInfo '
                                       'sub-models.',
                                  optional=True )
   macInstallationErrors = List( valueType=str,
                                 help='List of MAC installation errors. '
                                      'For traffic-policies applied on interfaces, '
                                      'refer to the "macInstallationErros" in '
                                      'the input/output TrafficPolicyIntfErrorInfo '
                                      'sub-models.',
                                 optional=True )
   ipv4InstallationErrorsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv4VrfInstallationErrors",
         optional=True )
   ipv6InstallationErrorsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv6VrfInstallationErrors",
         optional=True )
   ipv4FailedIntfs = List( valueType=Interface,
                           help='(Deprecated - refer to "ipv4FailedIntfs" in the '
                                '"input" TrafficPolicyIntfErrorInfo sub-model) '
                                'Failed interfaces for IPv4 traffic',
                           optional=True )
   ipv6FailedIntfs = List( valueType=Interface,
                           help='(Deprecated - refer to "ipv6FailedIntfs" in the '
                                '"input" TrafficPolicyIntfErrorInfo sub-model) '
                                'Failed interfaces for IPv6 traffic',
                           optional=True )
   # End of deprecated attributes

   input = Submodel( valueType=TrafficPolicyIntfErrorInfo,
                     help='Installation errors for policies applied on '
                          'input of interfaces',
                     optional=True )
   output = Submodel( valueType=TrafficPolicyIntfErrorInfo,
                      help='Installation errors for policies applied '
                           'on output of interfaces',
                      optional=True )
   vacl = Submodel( valueType=TrafficPolicyVAclErrorInfo,
                      help='Installation errors for policies applied '
                           'on vlan',
                      optional=True )

   cpu = Submodel( valueType=TrafficPolicyIntfErrorInfo,
                   help='Installation errors for policies applied to CPU '
                        '(enforced in hardware)',
                   optional=True )

   cpuSoftware = Submodel( valueType=TrafficPolicyIntfErrorInfo,
                           help='Installation errors for policies applied to CPU '
                                '(enforced in software)',
                           optional=True )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # The next revision of the model has no information for the rules as we
         # display errors encountered on the configured interface or vrf. So we can't
         # show any useful data in the dictRepr for rules.
         raise CliModelNotDegradable( "No resonable values for rules available" )

class InterfaceInfo( DeferredModel ):
   packetCount = Int( help="Number of packets matched", optional=True )
   byteCount = Int( help="Number of bytes matched", optional=True )

class CounterInfo( DeferredModel ):
   interfaces = Dict( valueType=InterfaceInfo, help="Count data per interface",
                      optional=True )
   totals = Submodel( valueType=TotalInfo, help="Total hit counts", optional=True )
   policeCount = Submodel( valueType=PoliceCount,
                           help="Descriptive count for police", optional=True )

class TrafficPolicyCounters( DeferredModel ):
   namedCounters = Dict( keyType=str, valueType=CounterInfo,
                         help="Mapping of named counter to count data",
                         optional=True )
   unnamedCounters = Dict( keyType=str, valueType=CounterInfo,
                           help="Mapping of rule name to count data", optional=True )

class TrafficPolicyCountersModel( DeferredModel ):
   ingressCounterGranularity = Enum(
      values=( "per-policy", "per-interface" ),
      help="Enabled counter granularity for traffic-policies applied on input of "
           "interfaces",
      optional=True )
   ingressCounterGranularityVlanIntf = Enum(
      values=( "per-policy", "per-interface" ),
      help="Enabled counter granularity for traffic-policies applied on input of "
           "VLAN interfaces",
      optional=True )
   trafficPolicies = Dict( valueType=TrafficPolicyCounters,
                           help="Traffic policy indexed by policy name. Contains "
                                "either policies applied on input of interfaces "
                                "or CPU policies",
                           optional=True )
   egressCounterGranularity = Enum(
      values=( "per-policy", "per-interface" ),
      help="Enabled counter granularity for traffic-policies applied on output of "
           "interfaces",
      optional=True )
   egressTrafficPolicies = Dict( valueType=TrafficPolicyCounters,
                                 help="Traffic policies applied on output of "
                                      "interfaces indexed by policy name.",
                                 optional=True )

class InterfaceCommon( DeferredModel ):
   configuredIntfs = List( valueType=Interface,
                           help="List of configured interfaces",
                           optional=True )
   ipv4AppliedIntfs = List( valueType=Interface,
                            help="Applied interfaces for IPv4 traffic",
                            optional=True )
   ipv6AppliedIntfs = List( valueType=Interface,
                            help="Applied interfaces for IPv6 traffic",
                            optional=True )
   macAppliedIntfs = List( valueType=Interface,
                           help="Applied interfaces for MAC traffic",
                           optional=True )

   configuredVni = List( valueType=Vni,
                         help="List of configured VNIs",
                         optional=True )
   ipv4AppliedVni = List( valueType=Vni,
                            help="Applied VNIs for IPv4 traffic",
                            optional=True )
   ipv6AppliedVni = List( valueType=Vni,
                            help="Applied VNIs for IPv6 traffic",
                            optional=True )
   macAppliedVni = List( valueType=Vni,
                           help="Applied VNIs for MAC traffic",
                           optional=True )

   configuredVrfs = List( valueType=str, help="List of configured VRFs",
                          optional=True )
   ipv4AppliedVrfs = List( valueType=str, help="Applied VRFs for IPv4 traffic",
                           optional=True )
   ipv6AppliedVrfs = List( valueType=str, help="Applied VRFs for IPv6 traffic",
                           optional=True )
   macAppliedVrfs = List( valueType=str, help="Applied VRFs for MAC traffic",
                          optional=True )

class VAclCommon( DeferredModel ):
   configuredVlans = List( valueType=str,
                           help="List of configured vlans",
                           optional=True )
   ipv4AppliedVlans = List( valueType=str,
                            help="Applied vlans for IPv4 traffic",
                            optional=True )
   ipv6AppliedVlans = List( valueType=str,
                            help="Applied vlans for IPv6 traffic",
                            optional=True )

# Common state shared between summary and detailed view of policy.
class TrafficPolicyCommon( DeferredModel ):
   description = Str( "Policy description", optional=True )
   namedCounters = List( valueType=str,
                         help="Defined named counters for a traffic-policy",
                         optional=True )
   input = Submodel( valueType=InterfaceCommon,
                     help="Interface information for input policy",
                     optional=True )
   output = Submodel( valueType=InterfaceCommon,
                      help="Interface information for output policy",
                      optional=True )
   vacl = Submodel( valueType=VAclCommon,
                    help="Vlan information for VAcl policy",
                    optional=True )
   cpu = Submodel( valueType=InterfaceCommon,
                   help="Information for CPU policy (enforced in hardware)",
                   optional=True )
   cpuSoftware = Submodel( valueType=InterfaceCommon,
                           help="Information for CPU policy (enforced in software)",
                           optional=True )

   # Start of deprecated TrafficPolicyCommon attributes. These have been moved
   # to the 'input' InterfaceCommon sub-model, but will be retained in the model
   # until a new model revision is absolutely necessary.
   configuredIntfs = List( valueType=Interface,
                           help='(Deprecated - refer to the "configuredIntfs" '
                                'attribute in the "input" InterfaceCommon '
                                'sub-model) List of interfaces the policy is '
                                'configured on input',
                           optional=True )
   ipv4AppliedIntfs = List( valueType=Interface,
                            help='(Deprecated - refer to the "ipv4AppliedIntfs" '
                                 'attribute in the "input" InterfaceCommon '
                                 'sub-model) Applied interfaces for IPv4 '
                                 'input traffic',
                            optional=True )
   ipv6AppliedIntfs = List( valueType=Interface,
                            help='(Deprecated - refer to the "ipv6AppliedIntfs" '
                                 'attribute in the "input" InterfaceCommon '
                                 'sub-model) Applied interfaces for IPv6 '
                                 'input traffic',
                            optional=True )
   configuredVrfs = List(
         valueType=str,
         help="Deprecated - refer to cpu.configuredVrfs attribute",
         optional=True )
   ipv4AppliedVrfs = List(
         valueType=str,
         help="Deprecated - refer to cpu.ipv4AppliedVrfs attribute",
         optional=True )
   ipv6AppliedVrfs = List(
         valueType=str,
         help="Deprecated - refer to cpu.ipv6AppliedVrfs attribute",
         optional=True )
   ipv4AppliedVrfsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv4AppliedVrfs",
         optional=True )
   ipv6AppliedVrfsSoftware = List(
         valueType=str,
         help="Deprecated - refer to cpuSoftware.ipv6AppliedVrfs",
         optional=True )
   # End of deprecated TrafficPolicyCommon attributes.

# Rules are ordered based on priority enforced in CLI configuration
class TrafficPolicySummary( TrafficPolicyCommon ):
   rules = List( valueType=RuleSummary, help="Summary of match rules configured" )

class TrafficPolicy( TrafficPolicyCommon ):
   rules = List( valueType=Rule, help="Detailed information of match rules" )

class TrafficPolicySummaryModel( DeferredModel ):
   trafficPolicies = Dict( valueType=TrafficPolicySummary,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class TrafficPolicyModel( DeferredModel ):
   __revision__ = 2
   proportionalLagPolicerEnabled = Bool(
      help="Proportional lag policer mode is enabled", optional=True )
   trafficPolicies = Dict( valueType=TrafficPolicy,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class TrafficPolicyErrorsModel( DeferredModel ):
   __revision__ = 2
   trafficPolicies = Dict( valueType=TrafficPolicyErrors,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class ExcludedVrfIntfInfo( DeferredModel ):
   vrfName = Str( "VRF Name", optional=True )
   excludedIntfs = List( valueType=Interface,
                         help="List of excluded VRF interfaces",
                         optional=True )

class OverriddenVrfInfo( DeferredModel ):
   vrfs = List( valueType=ExcludedVrfIntfInfo,
                help="List of overridden VRFs",
                optional=True )

class VrfIntfOverrideInput( DeferredModel ):
   input = Submodel( valueType=OverriddenVrfInfo,
                     help='Interface information for input policy',
                     optional=True )

class TrafficPolicyVrfIntfOverrideModel( DeferredModel ):
   trafficPolicies = Dict( valueType=VrfIntfOverrideInput,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class IpPrefixWithExceptions( DeferredModel ):
   prefix = IpGenericPrefix( help='Allowed prefix', optional=True )
   exceptPrefixes = List( valueType=IpGenericPrefix,
                              help="Exception prefixes", optional=True )

class UrlInfo( DeferredModel ):
   urlPath = Str( "Path to URL file", optional=True )
   vrf = Str( "Name of VRF for URL fetch", optional=True )

class PolicyUsage( DeferredModel ):
   policyNames = List( valueType=str, help="policies applied with the field-set",
                       optional=True )
   # We used a more detatiled model in order to support classnames in the output at
   # a later time
   # classNames = List( valueType=str, help="match rules applied with the field-set",
   #                    optional=True )

class _FieldSet( DeferredModel ):
   source = Enum( values=( "static", "bgp", "url", "controller", "andResults",
                           "other" ),
                           help="Contents source type" )
   urlInfo = Submodel( valueType=UrlInfo, help="URL information", optional=True )
   policies = Submodel( valueType=PolicyUsage,
                        help="Policies applied with the field-set", optional=True )

class IpPrefixFieldSet( _FieldSet ):
   prefixes = List( valueType=IpPrefixWithExceptions,
                    help="Allowed prefixes with associated exception prefixes",
                    optional=True )
   andResultsFieldSetNames = List( valueType=str,
                    help="Field sets for which and-results operation is done",
                    optional=True )

class TrafficPolicyPrefixFieldSetModel( DeferredModel ):
   prefixFieldSets = Dict( keyType=str, valueType=IpPrefixFieldSet,
                           help="Prefix field sets by name",
                           optional=True )

class TrafficPolicyL4PortFieldSet( _FieldSet ):
   ports = List( valueType=NumericalRange, help="Port ranges", optional=True )

class TrafficPolicyL4PortFieldSetModel( DeferredModel ):
   l4PortFieldSets = Dict( keyType=str, valueType=TrafficPolicyL4PortFieldSet,
                           help="Layer 4 port field sets by name",
                           optional=True )

class TrafficPolicyVlanFieldSet( _FieldSet ):
   vlans = List( valueType=NumericalRange, help="VLAN ranges", optional=True )

class TrafficPolicyVlanFieldSetModel( DeferredModel ):
   vlanFieldSets = Dict( keyType=str, valueType=TrafficPolicyVlanFieldSet,
                         help="VLAN field sets by name",
                         optional=True )

class TrafficPolicyServiceFieldSet( _FieldSet ):
   protocols = List( valueType=ProtocolAndPort,
                     help="List of protocols",
                     optional=True )

class TrafficPolicyServiceFieldSetModel( DeferredModel ):
   serviceFieldSets = Dict( keyType=str, valueType=TrafficPolicyServiceFieldSet,
                         help="service field sets by name",
                         optional=True )

class TrafficPolicyIntegerFieldSet( _FieldSet ):
   integers = List( valueType=NumericalRange, help="Integer ranges", optional=True )

class TrafficPolicyIntegerFieldSetModel( DeferredModel ):
   integerFieldSets = Dict( keyType=str, valueType=TrafficPolicyIntegerFieldSet,
                            help="Integer field sets by name",
                            optional=True )

class TrafficPolicyMacFieldSet( _FieldSet ):
   macAddrs = List( valueType=MacAddressInfo,
                    help="MAC addresses with optional mask", optional=True )

class TrafficPolicyMacFieldSetModel( DeferredModel ):
   macFieldSets = Dict( keyType=str, valueType=TrafficPolicyMacFieldSet,
                        help="MAC address field sets by name",
                        optional=True )

class MacTrafficPolicyMacFieldSet( DeferredModel ):
   macAddrs = List( valueType=MacAddress,
                help="List of all MAC addresses" )

class MacTrafficPolicyMacFieldSetSource( DeferredModel ):
   macFieldSets = Dict( keyType=str, valueType=MacTrafficPolicyMacFieldSet,
               help="Mapping of field set name to MAC address information" )

class MacTrafficPolicyMacFieldSetModel( DeferredModel ):
   sources = Dict( keyType=str, valueType=MacTrafficPolicyMacFieldSetSource,
                   help="Mapping of source name to MAC field set configuration" )

# Models for 'show traffic-policy protocol neighbors bgp' commands
# NOTE: Model is used here (not DeferredModel) as this command is implemented as a
# standard Python CAPI command (not CliPrint). This was a deliberate choice to take
# advantage of TableOutput to implement dynamic column widths. CliPrint is best for
# high scale commands. As the output from this command is O( number of BGP peers ),
# it should be limited to the low tens of thousands at the most. Therefore, a Python
# implementation should be fast enough.
class NeighborRule( Model ):
   sourcePrefix = IpGenericPrefix( help="Source prefix" )
   sourcePort = Int( help="Source port", optional=True )
   destinationPort = Int( help="Destination port", optional=True )
   maximumHops = Int( help="Maximum hop(s) allowed per GTSM for this speaker",
                      optional=True )

class VrfRules( Model ):
   rules = List( valueType=NeighborRule, help="Neighbor rules for IPV4/6" )

class ProtocolNeighbor( Model ):
   vrfs = Dict( valueType=VrfRules, help="Neighbors per VRF" )

class ProtocolNeighborsModel( Model ):
   # Keyed on policy name
   policies = Dict( valueType=ProtocolNeighbor, help="Neighbors per policy" )

   def createTable( self ):
      headingEntries = [ "VRF", "Source IP", "Source Port", "Destination Port" ]
      if toggleTrafficPolicyEnforceGtsmEnabled():
         headingEntries.append( "Maximum Hops" )
      # NOTE: 81 is the maximum length of a row in the table assuming the longest
      # possible IPV6 address.
      table = TableOutput.createTable( headingEntries, indent=0 )
      # Center justify and wrap all columns
      textColFormat = TableOutput.Format( justify='left', wrap=True )
      textColFormat.noPadLeftIs( True )
      textColFormat.padLimitIs( True )
      numberColFormat = TableOutput.Format( justify='right', wrap=True )
      numberColFormat.noPadLeftIs( True )
      numberColFormat.padLimitIs( True )
      if toggleTrafficPolicyEnforceGtsmEnabled():
         table.formatColumns( textColFormat, textColFormat, numberColFormat,
                              numberColFormat, numberColFormat )
      else:
         table.formatColumns( textColFormat, textColFormat, numberColFormat,
                              numberColFormat )
      return table

   def render( self ):
      for policyName in sorted( self.policies.keys() ):
         print( 'Traffic Policy %s' % policyName )

         table = self.createTable()

         protocolNeighbor = self.policies.get( policyName )
         for vrfName in sorted( protocolNeighbor.vrfs ):
            vrfRules = protocolNeighbor.vrfs[ vrfName ]
            for rule in sorted( vrfRules.rules, key=lambda key: key.sourcePrefix ):
               sourcePort = 'any'
               if rule.sourcePort:
                  sourcePort = rule.sourcePort
               destinationPort = 'any'
               if rule.destinationPort:
                  destinationPort = rule.destinationPort
               if toggleTrafficPolicyEnforceGtsmEnabled():
                  maximumHops = 'any'
                  if rule.maximumHops:
                     maximumHops = rule.maximumHops
                  table.newRow( vrfName, rule.sourcePrefix, sourcePort,
                              destinationPort, maximumHops )
               else:
                  table.newRow( vrfName, rule.sourcePrefix, sourcePort,
                              destinationPort )
         print( table.output() )
