#Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function

import struct

import AgentDirectory
import Arnet
import BasicCli
import CliCommon
import CliGlobal
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import Toggles.IpLibToggleLib

from CliPlugin import (
   TechSupportCli,
   TunnelModels,
)
from CliPlugin.TunnelRibCli import (
   systemTunnelRibName,
   systemTunnelRibId,
   systemColoredTunnelRibName,
   systemColoredTunnelRibId,
   systemTunnelingLdpTunnelRibName,
   systemIgpShortcutTunnelRibName,
   TunnelRibShowNameMatcher,
   systemColoredTunnelingLdpTunnelRibName,
   systemColoredIgpShortcutTunnelRibName
)
from CliPlugin import TunnelRibModel
from TunnelLib import ( # pylint: disable-msg=unused-import, W0611
   getDyTunTidFromIntfId,
   getNhAndIntfStrs,
   getTunnelIdFromIndex,
   getTunnelViaStatusFromTunnelId,
   isDyTunIntfId,
   readMountTunnelTable,
   srTunnelTypeStrs,
   staticTunnelTypeStrs,
   tunnelViaStatusTacToCapi,
   writeMountTunnelTable,
)
from collections import defaultdict
import ShowCommand
import SharedMem
import SmashLazyMount
import SharkLazyMount
import Tac
import Tracing
import LazyMount
import TunnelAgent
import AgentCommandRequest
from Toggles.TunnelToggleLib import toggleMvpnRsvpP2mpEnabled

from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'TunnelCli' )
t0 = Tracing.trace0

AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
ColoredTunnelEndpoint = TacLazyType( "Tunnel::TunnelTable::ColoredTunnelEndpoint" )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
LdpP2mpFecElement = TacLazyType( "Ldp::LdpP2mpFecElement" )
MetricTypeString = TacLazyType( "Arnet::MetricTypeString" )
MldpOpaqueValTlvParser = TacLazyType( 'Mpls::MldpOpaqueValTlvParser' )
MldpP2mpTunnelRib = TacLazyType( "Tunnel::TunnelTable::MldpP2mpTunnelRib" )
RsvpP2mpTunnelRib = TacLazyType( "Tunnel::TunnelTable::RsvpP2mpTunnelRib" )
StaticTunnelMrib = TacLazyType( "Tunnel::TunnelTable::StaticTunnelMrib" )
TunnelIdConstants = TacLazyType( "Tunnel::TunnelTable::TunnelIdConstants" )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
TunnelViaStatus = TacLazyType( 'Tunnel::Hardware::TunnelViaStatus' )

keyshadowInfo = SmashLazyMount.mountInfo( 'keyshadow' )
readerInfo = SmashLazyMount.mountInfo( 'reader' )

em = None
sysname = None
ldpTunnelTable = None
staticTunnelTable = None
srTunnelTable = None
ospfSrTunnelTable = None
tilfaTunnelTable = None
nexthopGroupTunnelTable = None
rsvpFrrTunnelTable = None
rsvpLerSubTunnelTable = None
rsvpLerTunnelTable = None
bgpLuTunnelTable = None
ipInIpTunnelTable = None
gueTunnelFib = None
srTePolicyTunnelTable = None
isisFlexAlgoTunnelTable = None
systemSharkStatus = None
systemTunnelRib = None
systemColoredTunnelRib = None
tunnelRibNameIdMap = None
forwardingStatus = None
forwarding6Status = None
forwardingGenStatus = None
programmingStatus = None
tunnelSmashEm = None
tunnelRibWatcher = None
tunnelMldpMrib = None
tunnelRsvpMrib = None
tunnelStaticMrib = None
vrfIdMapStatus = None

gv = CliGlobal.CliGlobal( dict( srv6TransportTunnelTable=None ) )

# The 44th bit is used to differentiate between tunnel endpoints of the same
# type but different address families.
tunnelIdAfBitMask = TunnelIdConstants.tunnelAfMask
tunnelIndexMask = TunnelIdConstants.tunnelIndexMask

#  must remain consistent (or have values added) to remain revision compatible
tunnelViaStatusCapiEnumVals = list( tunnelViaStatusTacToCapi.values() ) + [
   'notProgrammed' ]

#-------------------------------------------------------------------------
# Common CLI tokens for Tunnel package
#-------------------------------------------------------------------------
tokenColored = CliMatcher.KeywordMatcher(
   "colored", helpdesc="Colored tunnel RIBs" )
tokenTunnelMatcher = CliMatcher.KeywordMatcher(
   'tunnel', helpdesc="Show tunnel information" )
tokenTunnelFibMatcher = CliMatcher.KeywordMatcher(
   'fib', helpdesc="Forwarding Information Base" )
tokenMulticastMatcher = CliMatcher.KeywordMatcher(
   'multicast', helpdesc="Show multicast information" )
tokenMulticastRibMatcher = CliMatcher.KeywordMatcher(
   'multicast', helpdesc="Multicast tunnel RIBs" )
tokenMldpRibMatcher = CliMatcher.KeywordMatcher(
   'mldp', helpdesc="MLDP tunnel RIB" )
tokenRsvpRibMatcher = CliMatcher.KeywordMatcher(
   'rsvp', helpdesc="RSVP tunnel RIB" )
tokenStaticRibMatcher = CliMatcher.KeywordMatcher(
   'static', helpdesc="Static tunnel RIB" )
tokenTunnelRibShowNameMatcher = TunnelRibShowNameMatcher()
tokenCountersMatcher = CliMatcher.KeywordMatcher(
   "counters", helpdesc="Tunnel egress hardware counters" )

tunnelIndexMin = 0
tunnelIndexMax = 0x00000FFFFFFFFFFF
tunnelIndexMatcher = CliMatcher.IntegerMatcher( tunnelIndexMin, tunnelIndexMax,
                                                helpdesc="Tunnel Index" )

tunnelIgpPreferenceMatcher = CliMatcher.KeywordMatcher(
   'preference', helpdesc="IGP preference for tunnel" )
tunnelIgpPreferenceRangeMatcher = CliMatcher.IntegerMatcher(
   0, 255, helpdesc="IGP preference for tunnel" )

tunnelIgpMetricMatcher = CliMatcher.KeywordMatcher(
   'metric', helpdesc="IGP metric for tunnel" )
tunnelIgpMetricRangeMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
                                                         helpdesc='IGP metric' )

#-------------------------------------------------------------------------
# Common helper methods for Tunnel package CLI
#-------------------------------------------------------------------------
def isAfBitSetInTunnelId( tunnelId ):
   # Mask out the tunnel type and tunnel index bit
   if ( tunnelId & tunnelIdAfBitMask ) >> 43:
      return True
   return False

TunnelViaStatus = Tac.Type( 'Tunnel::Hardware::TunnelViaStatus' )

def getTunnelEntry( tunnelId ):
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   tunnelTable = None
   if tunnelType == 'bgpLuTunnel':
      tunnelTable = bgpLuTunnelTable
   elif tunnelType == 'ldpTunnel':
      tunnelTable = ldpTunnelTable
   elif tunnelType in staticTunnelTypeStrs:
      tunnelTable = staticTunnelTable
   elif tunnelType in srTunnelTypeStrs:
      tunnelTable = srTunnelTable
   elif tunnelType == 'ospfSrTunnel':
      tunnelTable = ospfSrTunnelTable
   elif tunnelType == 'nexthopGroupTunnel':
      tunnelTable = nexthopGroupTunnelTable
   elif tunnelType == 'rsvpFrrTunnel':
      tunnelTable = rsvpFrrTunnelTable
   elif tunnelType == 'rsvpLerSubTunnel':
      tunnelTable = rsvpLerSubTunnelTable
   elif tunnelType == 'rsvpLerTunnel':
      tunnelTable = rsvpLerTunnelTable
   elif tunnelType == 'tiLfaTunnel':
      tunnelTable = tilfaTunnelTable
   elif tunnelType == 'isisFlexAlgoTunnel':
      tunnelTable = isisFlexAlgoTunnelTable
   elif tunnelType == 'srv6TransportTunnel':
      tunnelTable = gv.srv6TransportTunnelTable
   elif tunnelType == 'ipInIpTunnel':
      tunnelTable = ipInIpTunnelTable
   elif tunnelType == 'gueTunnel':
      tunnelTable = gueTunnelFib

   if tunnelTable and tunnelTable.entry:
      tunnelEntry = tunnelTable.entry.get( tunnelId )
      return tunnelEntry

   return None

def getEndpointFromTunnelId( tunnelId ):
   tunnelEntry = getTunnelEntry( tunnelId )
   tid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId )
   tunnelType = tid.tunnelType()
   if tunnelEntry:
      if tunnelType == 'ipInIpTunnel':
         return Tac.Value( 'Arnet::IpGenPrefix', str( tunnelEntry.tunnelInfo.dst ) )
      elif tunnelType == 'gueTunnel':
         # Need to retrieve GueTunnelInfo from gueTunnelEncap collection in TunnelFib
         if len( tunnelEntry.tunnelVia ) > 0:
            encapId = tunnelEntry.tunnelVia[ 0 ].encapId
            gueTunnelEncap = gueTunnelFib.gueTunnelEncap.get( encapId )
            if gueTunnelEncap:
               gueTunnelInfo = gueTunnelEncap.encapInfo
               prefix = str( gueTunnelInfo.dst ) + '/32'
               return Tac.Value( 'Arnet::IpGenPrefix', prefix )
         return None
      return tunnelEntry.tep
   return None

def getColoredEndpointFromTunnelId( tunnelId ):
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   tunnelTable = None
   if tunnelType == 'srTePolicyTunnel':
      tunnelTable = srTePolicyTunnelTable

   if tunnelTable and tunnelTable.entry:
      tunnelEntry = tunnelTable.entry.get( tunnelId )
      if tunnelEntry:
         return tunnelEntry.tep, tunnelEntry.color
   return None, None

def getTunnelIndexFromId( tunnelId ):
   return int( Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId ).tunnelIndex() 
                & tunnelIndexMask )

def getViaModel( nexthop, intfId, labels ):
   if isDyTunIntfId( intfId ):
      tid = getDyTunTidFromIntfId( intfId )
      return TunnelModels.MplsTunnelVia( tunnelId=getTunnelIdModel( tid ),
                                         type='tunnel', labels=labels )
   else:
      return TunnelModels.MplsVia( nexthop=nexthop, interface=intfId,
                                   type='ip', labels=labels )

def getTunnelTypeFromTunnelId( tunnelId ):
   # The 44th bit is used to differentiate between tunnel endpoints of the same
   # type but different address families. Currently only SR
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   # If this is a SR or static tunnel, return tunnel type corresponding to AF
   if tunnelType == 'srTunnel':
      if isAfBitSetInTunnelId( tunnelId ):
         return 'srV6Tunnel'
      else:
         return 'srV4Tunnel'
   elif tunnelType == 'staticTunnel':
      if isAfBitSetInTunnelId( tunnelId ):
         return 'staticV6Tunnel'
      else:
         return 'staticV4Tunnel'
   else:
      return tunnelType

# create a TunnelId Model object for a tunnelId tacc object.
def getTunnelIdModel( tunnelId ):
   tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId )
   # If tunnelId type is 'staticTunnel' or 'srTunnel', we need to make a
   # distinction between IPv4 and IPv6 tunnels. We do this by looking at the
   # 44th bit of the tunnel ID.
   TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
   tunnelType = tacTid.tunnelType()
   if tunnelType == TunnelType.staticTunnel:
      if not isAfBitSetInTunnelId( tunnelId ): # IPv4
         tunnelType = 'staticV4Tunnel'
      else: # IPv6
         tunnelType = 'staticV6Tunnel'
   elif tunnelType == TunnelType.srTunnel:
      if not isAfBitSetInTunnelId( tunnelId ): # IPv4
         tunnelType = 'srV4Tunnel'
      else: # IPv6
         tunnelType = 'srV6Tunnel'
   return TunnelModels.TunnelId(
         index=tacTid.tunnelIndex(),
         type=TunnelModels.tunnelTypeStrDict[ tunnelType ] )

# create a TunnelVia object from via dictionary.
def getTunnelViaModel( via ):
   if 'tunnelId' not in via:
      # this can happen with EosImage/test/CliModelRevisionTest.py 
      # The test choose all possible values of type ['ip', 'tunnel'] 
      # to generate a dictionary of via for MplsVia with type as 'tunnel'.
      # MplsVia does not have tunnelId argument, so to avoid keyError
      # we raise CliModelNotDegradable to make CliModelRevisionTest.py
      # happy.
      raise CliCommon.CliModelNotDegradable( "Invalid via dictionary" )
   tunnelId = TunnelModels.TunnelId(
         index=via[ 'tunnelId' ][ 'index' ],
         type=via[ 'tunnelId' ][ 'type' ] )
   return TunnelModels.TunnelVia( tunnelId=tunnelId, type='tunnel' )

# create a IpVia object from via dictionary.
def getIpViaModel( via ):
   return TunnelModels.IpVia( nexthop=via[ 'nexthop' ], 
                              interface=via[ 'interface' ],
                              type='ip' )

# create a TunnelVia or IpVia Model object from via dict.
def getViaModelFromViaDict( via ):
   if via[ 'type' ] == 'tunnel':
      return getTunnelViaModel( via )
   else:
      return getIpViaModel( via )

# create a list of TunnelVias and/or IpVias from generic Via Model object.
def getViaModels( via, bgpLuForwardingTunnelTable=None, flattenLuForwarding=True,
                  flatteningCall=False, useMplsVias=False ):
   viaModels = []
   Model = TunnelModels.MplsTunnelVia if useMplsVias else TunnelModels.TunnelVia
   if FecIdIntfId.isSrTePolicyIntfId( via.intfId ):
      tid = FecIdIntfId.tunnelId( via.intfId )
      model = Model( tunnelId=getTunnelIdModel( tid ), type='tunnel' )
      viaModels.append( model )
   elif not isDyTunIntfId( via.intfId ):
      if useMplsVias:
         labels = []
         for mplsStackIndex in range( via.labels.stackSize - 1, -1, -1 ):
            labels.append( str( via.labels.labelStack( mplsStackIndex ) ) )
         viaModels.append( TunnelModels.MplsVia( nexthop=via.nexthop,
                                                 interface=via.intfId,
                                                 labels=labels,
                                                 type='ip' ) )
      else:
         viaModels.append( TunnelModels.IpVia( nexthop=via.nexthop,
                                               interface=via.intfId, type='ip' ) )
   else:
      tid = getDyTunTidFromIntfId( via.intfId )
      tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
      TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
      if ( tacTid.tunnelType() != TunnelType.bgpLuForwardingTunnel or
           not flattenLuForwarding ):
         model = Model( tunnelId=getTunnelIdModel( tid ), type='tunnel' )
         viaModels.append( model )
      else:
         flattenedVias = getFlattenedViaModels( via, bgpLuForwardingTunnelTable,
                                                tacTid, TunnelType )
         viaModels.extend( flattenedVias )
   return viaModels

# recursively flatten out any BGP LU Push Via Models
def getFlattenedViaModels( via, bgpLuForwardingTunnelTable, tacTid, TunnelType ):
   viaModels = []
   labels = []
   entry = bgpLuForwardingTunnelTable.entry.get( tacTid )
   if entry:
      for i in range( entry.labels.stackSize - 1, -1, -1 ):
         labels.append( str( entry.labels.labelStack( i ) ) )
      for nextVia in entry.via.values():
         if FecIdIntfId.isSrTePolicyIntfId( nextVia.intfId ):
            tid = FecIdIntfId.tunnelId( nextVia.intfId )
            viaModels.append( TunnelModels.MplsTunnelVia( labels=labels,
               tunnelId=getTunnelIdModel( tid ), type='tunnel' ) )
         elif not isDyTunIntfId( nextVia.intfId ):
            viaModels.append( TunnelModels.MplsVia(
                                 nexthop=nextVia.nexthop, labels=labels,
                                 interface=nextVia.intfId, type='ip' ) )
         else:
            tid = getDyTunTidFromIntfId( nextVia.intfId )
            tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
            if tacTid.tunnelType() != TunnelType.bgpLuForwardingTunnel:
               viaModels.append( TunnelModels.MplsTunnelVia( labels=labels,
                                    tunnelId=getTunnelIdModel( tid ),
                                    type='tunnel' ) )
            else:
               flattenedVias = getFlattenedViaModels( nextVia, 
                                                      bgpLuForwardingTunnelTable,
                                                      tacTid, TunnelType )
               fViasWithLabels = appendEntryLabelsToViaModels( flattenedVias,
                                                               labels )
               viaModels.extend( fViasWithLabels )
   return viaModels

def appendEntryLabelsToViaModels( vias, labels ):
   viaModels = []
   for via in vias:
      if hasattr( via, 'labels' ):
         baseLabels = []
         for l in via.labels:
            baseLabels.append( l )
         via.labels = baseLabels + labels
      viaModels.append( via )
   return viaModels

def getNhAndIntfAndLabelStrs( via ):
   nhStr, intfStr = getNhAndIntfStrs( via )
   if hasattr( via, 'labels' ) and via.labels:
      labelsStr = '[ ' + ' '.join( via.labels ) + '  ]'
   else:
      labelsStr = '-'
   return nhStr, intfStr, labelsStr

#-------------------------------------------------------------------------
# The "show tunnel rib [<prefix>] [brief]" command.
#-------------------------------------------------------------------------
def getTunnelRibEntry( tunnelRib, key ):
   if tunnelRib and key:
      tunnelRibEntry = tunnelRib.entry.get( key )
      if not tunnelRibEntry:
         return None
      if not tunnelRibEntry.tunnelId:
         # Somehow we have ended up in a transient state where there exists a
         # Tunnel RIB entry with no Tunnel IDs. We should just ignore this
         # Tunnel RIB entry and not add it to the CLI/CAPI output.
         return None
      return tunnelRibEntry
   return None

def getTunnelType( tunnelRib, key ):
   tunnelRibEntry = getTunnelRibEntry( tunnelRib, key )
   if not tunnelRibEntry:
      return None

   # only care about the tunnelType for first tunnelId
   tunnelIds = tunnelRibEntry.tunnelId
   tunnelType = getTunnelTypeFromTunnelId( tunnelIds[ 0 ] )
   return TunnelModels.tunnelTypeStrDict[ tunnelType ]

def getTunnelRibEntryHelper( tunnelRibEntry, tunnelType, tunnelIds ):
   tunnelIndexes = \
      [ getTunnelIndexFromId( tunnelId ) for tunnelId in tunnelIds ]
   tunnelIndexes.sort()

   metricType = MetricTypeString().metricTypeCode( tunnelRibEntry.metricType )
   return TunnelRibModel.TunnelRibEntry(
      tunnelType=TunnelModels.tunnelTypeStrDict[ tunnelType ],
      tunnelIndexes=tunnelIndexes,
      metric=tunnelRibEntry.metric, metric2=tunnelRibEntry.metric2,
      pref=tunnelRibEntry.pref, pref2=tunnelRibEntry.pref2,
      tunnelPreference=tunnelRibEntry.entryPref,
      metricType=metricType )

def getTunnelRibEntryModel( tunnelRib, key ):
   tunnelRibEntry = getTunnelRibEntry( tunnelRib, key )
   if not tunnelRibEntry:
      return None

   # get tunnel indexes
   tunnelIds = list( tunnelRibEntry.tunnelId.values() )
   # only care about the tunnelType for first tunnelId
   tunnelType = getTunnelTypeFromTunnelId( tunnelIds[ 0 ] )
   return getTunnelRibEntryHelper( tunnelRibEntry, tunnelType, tunnelIds )

def getTunnelRibEntriesByTunnelTypeModel( tunnelRib, key ):
   tunnelRibEntry = getTunnelRibEntry( tunnelRib, key )
   if not tunnelRibEntry:
      return None

   tunnelTypeToIds = defaultdict( list )
   for tid in tunnelRibEntry.tunnelId.values():
      tunnelType = getTunnelTypeFromTunnelId( tid )
      tunnelTypeToIds[ tunnelType ].append( tid )
   entriesModel = TunnelRibModel.TunnelRibEntriesByTunnelType()

   for tunnelType, tids in tunnelTypeToIds.items():
      entriesModel.entries.append( getTunnelRibEntryHelper( tunnelRibEntry,
                                                            tunnelType, tids ) )

   return entriesModel

def getColoredTunnelRibModel( trName, tunnelRib, prefix, color ):
   coloredRibModel = TunnelRibModel.ColoredTunnelRib( name=trName )
   tep = Arnet.IpGenPrefix( str( prefix ) ) if prefix is not None else None
   if tep and color is not None:
      key = Tac.Value( "Tunnel::TunnelTable::ColoredTunnelEndpoint",
                       tep, color, True )
      tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, key )
      if tunnelRibEntryModel:
         coloredRibModel.addColoredRibEntry( key.tep, key.color,
                                             tunnelRibEntryModel )
      return coloredRibModel
   if tunnelRib:
      for coloredTep in tunnelRib.entry:
         tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, coloredTep )
         if not tunnelRibEntryModel:
            continue
         if tep and tep != coloredTep.tep:
            continue
         if color is not None and color != coloredTep.color:
            continue
         coloredRibModel.addColoredRibEntry( coloredTep.tep, coloredTep.color,
                                             tunnelRibEntryModel )
   return coloredRibModel

def getTunnelRibAllSummaryModel( tunnelRibs ):
   tepCountByTunnelRibName = {}
   for tunnelName, tunnelRib in tunnelRibs.items():
      if tunnelRib:
         tepCountByTunnelRibName[ tunnelName ] = len( tunnelRib.entry )
   return TunnelRibModel.TunnelRibAllSummary( tunnels=tepCountByTunnelRibName )

def getTunnelRibSummaryModel( trName, tunnelRib ):
   tepCountByTunnelType = {}
   if tunnelRib:
      for tep in tunnelRib.entry:
         tunnelType = getTunnelType( tunnelRib, tep )
         tepCountByTunnelType[ tunnelType ] = \
               tepCountByTunnelType.get( tunnelType, 0 ) + 1
   return TunnelRibModel.TunnelRibSummary( name=trName,
                                           tunnels=tepCountByTunnelType )

def getTunnelRibModel( trName, tunnelRib, prefix=None, color=None ):
   entries = {}
   if tunnelRib:
      if tunnelRib.isColored():
         return getColoredTunnelRibModel( trName, tunnelRib, prefix, color )
      if prefix is None:
         for tep in tunnelRib.entry:
            tunnelRibEntryModel = getTunnelRibEntriesByTunnelTypeModel( tunnelRib,
                                                                        tep )
            if tunnelRibEntryModel:
               entries[ tep ] = tunnelRibEntryModel
      else:
         if not isinstance( prefix, str ):
            prefix = str( prefix )
         prefix = Arnet.IpGenPrefix( prefix )
         tunnelRibEntryModel = getTunnelRibEntriesByTunnelTypeModel( tunnelRib,
                                                                     prefix )
         if tunnelRibEntryModel:
            entries[ prefix ] = tunnelRibEntryModel
   return TunnelRibModel.TunnelRib( name=trName, entries=entries )

class TunnelRibWatcher( Tac.Notifiee ) :
   '''
   TunnelRibWatcher reacts to the nameToId attribute of TunnelRibNameIdMap,
   mounts, and caches all tunnel ribs including the system tunnel rib. This
   allows for more efficient mounting( and unmounting ) of tunnel rib
   entities required for show commands.

   Reactor:
   TunnelRibNameIdMap:: | mountedTunnelRibs | action
      nameToId          |                   |
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
          no            |        no         |  nothing to do
   --------------------------------------------------------------------------------
          no            |        yes        |  tunnel has been
                        |                   |  deleted, remove from
                        |                   |  watcher
   --------------------------------------------------------------------------------
         yes            |       yes         | already mounted so
                        |                   | nothing to do
   --------------------------------------------------------------------------------
         yes            |       no          | tunnel not mounted so
                        |                   | mount it

   show command:
   TunnelRibNameIdMap:: | mountedTunnelRibs | action
      nameToId          |                   |
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
          no            |        no         |  nothing to show
   --------------------------------------------------------------------------------
          no            |       yes         |  Do nothing as watcher will be called
                        |                   |  to delete the entry
   --------------------------------------------------------------------------------
         yes            |       yes         | tunnel in Watcher
                        |                   | return the model
   --------------------------------------------------------------------------------
         yes            |       no          | tunnel not mounted so
                        |                   | mount and return model


   '''
   notifierTypeName = "Tunnel::TunnelTable::TunnelRibNameIdMap"
   def __init__( self ):
      Tac.Notifiee.__init__( self, tunnelRibNameIdMap )
      # mountedTunnelRibs is dictionary of mounted TunnelRibs indexed by id 
      self.mountedTunnelRibs = {}
      self.tunnelRibBaseSmashPath = 'tunnel/tunnelRibs/status/'
      for tribName, tribId in tunnelRibNameIdMap.nameToId.items():
         tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str( tribId )
         if tribName in ( systemColoredTunnelRibName,
                          systemColoredTunnelingLdpTunnelRibName,
                          systemColoredIgpShortcutTunnelRibName ):
            tunnelRibType = 'Tunnel::TunnelTable::ColoredTunnelRib'
         else:
            tunnelRibType = 'Tunnel::TunnelTable::TunnelRib'
         tunnelRib = tunnelSmashEm.doMount( tunnelRibSmashPath,
                                            tunnelRibType,
                                            readerInfo )
         tunnelRib.id = tribId
         self.mountedTunnelRibs[ tribName ] = tunnelRib

   # Get Id of from TunnelRibNameIdMap using name as the key
   def getTunnelId( self, name ):
      return tunnelRibNameIdMap.nameToId.get( name )

   def getAllTunnelRibs( self ):
      ''' cache and return all tunnel ribs '''
      for tunnelName, tunnelId in tunnelRibNameIdMap.nameToId.items():
         if tunnelName not in tunnelRibWatcher.mountedTunnelRibs:
            tunnelRibWatcher.cacheTunnelRib( tunnelName, tunnelId )
      return self.mountedTunnelRibs

   def getMountedTunnelRib( self, name ):
      tunnelId = getTunnelId( name )
      if tunnelId is None:
         return None
      if name not in tunnelRibWatcher.mountedTunnelRibs:
         tunnelRibWatcher.cacheTunnelRib( name, tunnelId )
      return self.mountedTunnelRibs[ name ]

   def cacheTunnelRib( self, name, tunnelId ):
      '''
         For a new key in tunnelRibNameIdMap, mount and add the respective tunnel
         rib to the mountedTunnelRibs class variable
      '''
      t0( "Creating/mounting tunnel rib" )
      if tunnelId is None:
         t0( "Tried to add Tunnel %s that is not in the TunnelRibNameIdMap" % name )
         return
      if name in self.mountedTunnelRibs:
         t0( "TunnelRib %s with Id %s is already mounted" % (name, tunnelId ) )
         return
      tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str( tunnelId )
      if name in ( systemColoredTunnelRibName,
                   systemColoredTunnelingLdpTunnelRibName,
                   systemColoredIgpShortcutTunnelRibName ):
         tunnelRibType = 'Tunnel::TunnelTable::ColoredTunnelRib'
      else:
         tunnelRibType = 'Tunnel::TunnelTable::TunnelRib'
      tunnelRib = tunnelSmashEm.doMount( tunnelRibSmashPath,
                                         tunnelRibType,
                                         readerInfo )
      if not tunnelRib:
         t0( "Mounting %s unsuccessful" % name )
         return
      tunnelRib.id = tunnelId
      t0( "Successfully mounted ", name, " from ", tunnelRibSmashPath )
      self.mountedTunnelRibs[ name ] = tunnelRib

   def removeTunnelRibFromCache( self, name ):
      '''
        Deletes cached Tunnel Rib when we detect that an entry has been deleted
        from tunnelRibNameIdMap. We delete when we detect that a key is in
        our dictionary of cached Tunnel Ribs but not in the tunnelRibNameIdMap.
      '''
      # At this point we know tunnelId is None
      if not name in self.mountedTunnelRibs:
         t0( "Tunnel %s is not mounted" % name )
         return
      if name is tunnelRibNameIdMap.systemTunnelRibName:
         t0( "Can't delete system tunnel rib" )
         return
      if name is tunnelRibNameIdMap.systemColoredTunnelRibName:
         t0( "Can't delete system colored tunnel rib" )
         return
      if name is tunnelRibNameIdMap.systemColoredTunnelingLdpTunnelRibName:
         t0( "Can't delete system colored tunneling ldp tunnel rib" )
         return
      if name is tunnelRibNameIdMap.systemColoredIgpShortcutTunnelRibName:
         t0( "Can't delete system colored igp shortcut tunnel rib" )
         return
      tunnelId = self.mountedTunnelRibs[ name ].name
      del self.mountedTunnelRibs[ name ]
      tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str ( tunnelId )
      t0( "Deleting %s from %s" % ( name, tunnelRibSmashPath ) )
      tunnelSmashEm.doUnmount( tunnelRibSmashPath )
      return

   @Tac.handler( 'nameToId' )
   def handletunRibNametunnelIdMap( self, trName ):
      t0( "Updating tunnel config for tunnel rib %s" % trName )
      tunnelId = self.getTunnelId( trName )
      if tunnelId is None and trName in self.mountedTunnelRibs:
         # Delete cached tunnelRibs for absent tunnel rib config
         self.removeTunnelRibFromCache( trName )
         return
      # Tunnel is being added to TunnelRibNameIdMap
      if trName in self.mountedTunnelRibs:
         t0( "Tunnel %s already mounted" % trName )
         return
      # trName is not in Mounted TunnelRibs so we add 
      self.cacheTunnelRib( trName, tunnelId )

def getTunnelRibWatcher():
   global tunnelRibWatcher
   if not tunnelRibWatcher:
      t0( "Instantiating TunnelRibWatcher..." )
      tunnelRibWatcher = TunnelRibWatcher()
   return tunnelRibWatcher

def showTunnelRib( mode, args ):
   tunnelRibs = [ 'TUNNEL_RIB_NAME', systemColoredTunnelRibName,
                  systemTunnelRibName, systemTunnelingLdpTunnelRibName,
                  systemIgpShortcutTunnelRibName ]
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      tunnelRibs.append( systemColoredTunnelingLdpTunnelRibName )
   if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
      tunnelRibs.append( systemColoredIgpShortcutTunnelRibName )
   for name in tunnelRibs:
      trName = args.get( name )
      if trName:
         break

   prefix = args.get( 'PREFIX' )
   color = args.get( 'COLOR' )
   if trName is None:
      if 'colored' in args:
         trName = systemColoredTunnelRibName
      else:
         trName = systemTunnelRibName
   _tunnelRibWatcher = getTunnelRibWatcher()
   mountedTunnelRib = _tunnelRibWatcher.getMountedTunnelRib( trName )
   return getTunnelRibModel( trName, mountedTunnelRib, prefix, color )

def getTunnelId( trName ):
   return tunnelRibNameIdMap.nameToId.get( trName )

def showTunnelRibAllSummary( mode, args ):
   _tunnelRibWatcher = getTunnelRibWatcher()
   tunnelRibs = _tunnelRibWatcher.getAllTunnelRibs()
   return getTunnelRibAllSummaryModel( tunnelRibs )

def showTunnelRibSummary( mode, args ):
   tunnelRibs = [ 'TUNNEL_RIB_NAME', systemTunnelRibName,
                  systemTunnelingLdpTunnelRibName, 
                  systemColoredTunnelRibName,
                  systemIgpShortcutTunnelRibName ]
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      tunnelRibs.append( systemColoredTunnelingLdpTunnelRibName )
   if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
      tunnelRibs.append( systemColoredIgpShortcutTunnelRibName )
   for name in tunnelRibs:
      trName = args.get( name )
      if trName:
         break

   if trName is None:
      trName = systemTunnelRibName

   _tunnelRibWatcher = getTunnelRibWatcher()
   mountedTunnelRib = _tunnelRibWatcher.getMountedTunnelRib( trName )
   return getTunnelRibSummaryModel( trName, mountedTunnelRib )

#-------------------------------------------------------------------------
# The "show tunnel rib [<prefix>] [candidates]" command.
#-------------------------------------------------------------------------
def getInternalTunnelRibCandidateEntry( cliRibEntry, candId ):
   if cliRibEntry and candId:
      cliRibCandidate = cliRibEntry.candidates.get( candId, None )
      if cliRibCandidate and cliRibCandidate.tunnelId:
         candTunnelIds = list( cliRibCandidate.tunnelId )
         if candTunnelIds: # BUG604426: transient state in stress test
            return cliRibCandidate, candTunnelIds 
      else:
         # Somehow we have ended up in a transient state where there exists a
         # CliRibEntry with no Tunnel IDs. We should just ignore this
         # CliRibEntry and not add it to the CLI/CAPI output.
         return None
   return None

def getInternalTunnelRibCandidateEntryModel( cliRibCandidate, cachedTunnelIds ):
   # get tunnel indexes
   tunnelIndexes = \
      [ getTunnelIndexFromId( tunnelId ) for tunnelId in cachedTunnelIds ]
   # only care about the tunnelType for first tunnelId
   tunnelType = getTunnelTypeFromTunnelId( cachedTunnelIds[ 0 ] )
   tunnelIndexes.sort()

   return TunnelRibModel.InternalTunnelRibCandidateEntry(
      tunnelType=TunnelModels.tunnelTypeStrDict[ tunnelType ],
      tunnelIndexes=tunnelIndexes,
      tunnelPreference=cliRibCandidate.id.entryPref,
      tunnelMetric=cliRibCandidate.id.entryMetric )

def getInternalTunnelRibCandidatesModel( cliTunnelRib, coloredTepKey ):
   """ Take TEP's unsorted RibCandidate entries, and sort into CAPI
   Model list.  To avoid a stale entry list, we must guarantee that every
   CliRib show command triggers a new on-the-fly mount, populate and sort.
   Do not reuse a cached list.
   """
   candidates = []
   cliRibEntry = cliTunnelRib.entry.get( coloredTepKey )
   if cliRibEntry:
      # Invariant: must trigger a new sort on entries for every 'mount/show' call
      for candId in sorted( cliRibEntry.candidates ):
         cliRibCandPair = getInternalTunnelRibCandidateEntry( cliRibEntry, candId )
         if not cliRibCandPair:
            # candidate in transient state or doesn't exist.
            continue
         cliRibCandidateEntryModel = \
               getInternalTunnelRibCandidateEntryModel( cliRibCandPair[ 0 ],
                                                        cliRibCandPair[ 1 ] )
         candidates.append( cliRibCandidateEntryModel )
      return TunnelRibModel.InternalTunnelRibCandidates( candidates=candidates )
   return None

def createColoredTep( tep_, color=None ):
   # Make a ColoredTunnelEndpoint object out of an IpGenPrefix or string.
   if color is None:
      isColored = False
      color = 0
   else:
      isColored = True
   return ColoredTunnelEndpoint( Arnet.IpGenPrefix( str( tep_ ) ), color, isColored )

def getInternalTunnelRibModel( trName, cliTunnelRib, prefix=None, color=None ):
   endpoints = {}

   if cliTunnelRib:
      if color is None: #TODO (BUG575074) ColoredTunnelRib implementation
         if prefix is None:
            for coloredTep in cliTunnelRib.entry:
               if coloredTep.isColored:
                  continue
               internalTunnelRibEntryModel = \
                  getInternalTunnelRibCandidatesModel( cliTunnelRib, coloredTep )
               if internalTunnelRibEntryModel:
                  endpoints[ coloredTep.tep ] = internalTunnelRibEntryModel
         else:
            prefix = Arnet.IpGenPrefix( str( prefix ) )
            coloredTep = createColoredTep( prefix, color )
            internalTunnelRibEntryModel = \
               getInternalTunnelRibCandidatesModel( cliTunnelRib, coloredTep )
            if internalTunnelRibEntryModel:
               endpoints[ prefix ] = internalTunnelRibEntryModel
   return TunnelRibModel.InternalTunnelRib( name=trName, endpoints=endpoints )

def getMountedCliTunnelRib( trName ):
   tribId = getTunnelId( trName )
   cliTunnelRibCollection = systemSharkStatus.cliTunnelRibColl
   if tribId is None:
      cliTunnelRib = None
   else:
      cliTunnelRib = cliTunnelRibCollection.get( tribId )
   return cliTunnelRib

def showInternalTunnelRib( mode, args ):
   tunnelRibs = [ 'TUNNEL_RIB_NAME', systemTunnelRibName,
                  systemTunnelingLdpTunnelRibName, 
                  systemIgpShortcutTunnelRibName ]
   
   for name in tunnelRibs:
      trName = args.get( name )
      if trName:
         break

   if trName is None:
      trName = systemTunnelRibName

   prefix = args.get( 'PREFIX' )
   cliTunnelRib = getMountedCliTunnelRib( trName )
   return getInternalTunnelRibModel( trName, cliTunnelRib, prefix )

matcherSystemTunnelRib = CliMatcher.KeywordMatcher(
   'system-tunnel-rib', helpdesc='The system tunnel RIB' )

matcherSystemTunnelingLdpTunnelRib = CliMatcher.KeywordMatcher(
   'system-tunneling-ldp-tunnel-rib',
   helpdesc='The system tunneling LDP tunnel RIB' )

matcherColoredSystemTunnelingLdpTunnelRib = CliMatcher.KeywordMatcher(
   'system-colored-tunneling-ldp-tunnel-rib',
   helpdesc='The system colored tunneling LDP tunnel RIB' )

matcherColoredSystemIgpShortcutTunnelRib = CliMatcher.KeywordMatcher(
   'system-colored-igp-shortcut-tunnel-rib',
   helpdesc='The system colored IGP shortcut tunnel RIB' )

matcherSystemIgpShortcutTunnelRib = CliMatcher.KeywordMatcher(
   'system-igp-shortcut-tunnel-rib', helpdesc='The system IGP shortcut tunnel RIB' )

matcherSystemColoredTunnelRib = CliMatcher.KeywordMatcher(
   'system-colored-tunnel-rib', helpdesc='The system colored tunnel RIB' )
matcherColor = CliMatcher.KeywordMatcher( 'color', helpdesc='Match with color' )
matcherColorValue = CliMatcher.IntegerMatcher( 0, 2 ** 32 - 1,
                                               helpdesc='Match this color value' )
matcherRib = CliMatcher.KeywordMatcher(
      'rib', helpdesc='Tunnel routing information' )

class ShowColoredTunnelRibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tunnel rib colored [ system-colored-tunnel-rib ]'
              '[ ( PREFIX [ color COLOR ] ) | ( color COLOR ) ] brief' )
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
         syntax = '''show tunnel rib colored 
                     [ system-colored-tunnel-rib 
                     | system-colored-tunneling-ldp-tunnel-rib 
                     | system-colored-igp-shortcut-tunnel-rib ]
                     [ ( PREFIX [ color COLOR ] ) | ( color COLOR ) ] brief'''
      else:
         syntax = '''show tunnel rib colored 
                     [ system-colored-tunnel-rib 
                     | system-colored-tunneling-ldp-tunnel-rib ]
                     [ ( PREFIX [ color COLOR ] ) | ( color COLOR ) ] brief'''

   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'colored': tokenColored,
      'system-colored-tunnel-rib': matcherSystemColoredTunnelRib,
      'system-colored-tunneling-ldp-tunnel-rib': \
            matcherColoredSystemTunnelingLdpTunnelRib,
      'system-colored-igp-shortcut-tunnel-rib': \
            matcherColoredSystemIgpShortcutTunnelRib,
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
      'brief': 'Only show Tunnel IDs',
   }

   cliModel = TunnelRibModel.ColoredTunnelRib
   handler = showTunnelRib

class ShowTunnelRibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tunnel rib
               [ system-tunneling-ldp-tunnel-rib
               | system-igp-shortcut-tunnel-rib
               | system-tunnel-rib
               | TUNNEL_RIB_NAME ]
               [ PREFIX ] brief'''

   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'system-tunneling-ldp-tunnel-rib': matcherSystemTunnelingLdpTunnelRib,
      'system-igp-shortcut-tunnel-rib': matcherSystemIgpShortcutTunnelRib,
      'system-tunnel-rib': matcherSystemTunnelRib,
      'TUNNEL_RIB_NAME':  tokenTunnelRibShowNameMatcher,
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'brief': 'Only show Tunnel IDs',
   }
   cliModel = TunnelRibModel.TunnelRib
   handler = showTunnelRib

class ShowNamedTunnelRibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tunnel rib
               ( system-tunneling-ldp-tunnel-rib
               | system-igp-shortcut-tunnel-rib
               | system-tunnel-rib
               | ( colored system-colored-tunnel-rib )
               | TUNNEL_RIB_NAME ) summary'''
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
         syntax = '''show tunnel rib
                     ( system-tunneling-ldp-tunnel-rib
                     | system-igp-shortcut-tunnel-rib
                     | system-tunnel-rib
                     | ( colored ( system-colored-tunnel-rib 
                                 | system-colored-tunneling-ldp-tunnel-rib 
                                 | system-colored-igp-shortcut-tunnel-rib ) )
                     | TUNNEL_RIB_NAME ) summary'''
      else:
         syntax = '''show tunnel rib
                     ( system-tunneling-ldp-tunnel-rib
                     | system-igp-shortcut-tunnel-rib
                     | system-tunnel-rib
                     | ( colored ( system-colored-tunnel-rib 
                                 | system-colored-tunneling-ldp-tunnel-rib ) )
                     | TUNNEL_RIB_NAME ) summary'''
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'colored': tokenColored,
      'system-tunneling-ldp-tunnel-rib': matcherSystemTunnelingLdpTunnelRib,
      'system-igp-shortcut-tunnel-rib': matcherSystemIgpShortcutTunnelRib,
      'system-tunnel-rib': matcherSystemTunnelRib,
      'system-colored-tunnel-rib': matcherSystemColoredTunnelRib,
      'system-colored-tunneling-ldp-tunnel-rib': \
            matcherColoredSystemTunnelingLdpTunnelRib,
      'system-colored-igp-shortcut-tunnel-rib': \
            matcherColoredSystemIgpShortcutTunnelRib,
      'TUNNEL_RIB_NAME': tokenTunnelRibShowNameMatcher,
      'summary': 'Tunnel RIB summary'
   }
   cliModel = TunnelRibModel.TunnelRibSummary
   handler = showTunnelRibSummary 

class ShowTunnelRibsSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tunnel rib summary'
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'summary': 'All tunnel RIBs summary'
   }
   cliModel = TunnelRibModel.TunnelRibAllSummary
   handler = showTunnelRibAllSummary 

class ShowTunnelRibCandidatesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tunnel rib
               [ system-tunneling-ldp-tunnel-rib
               | system-igp-shortcut-tunnel-rib
               | system-tunnel-rib
               | TUNNEL_RIB_NAME ]
               [ PREFIX ] candidates'''
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'system-tunneling-ldp-tunnel-rib': matcherSystemTunnelingLdpTunnelRib,
      'system-igp-shortcut-tunnel-rib': matcherSystemIgpShortcutTunnelRib,
      'system-tunnel-rib': matcherSystemTunnelRib,
      'TUNNEL_RIB_NAME': tokenTunnelRibShowNameMatcher,
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'candidates': 'Show all Tunnel IDs, primary AND backups',
   }
   cliModel = TunnelRibModel.InternalTunnelRib
   handler = showInternalTunnelRib

def getMldpPrettyPrintOpaqueValue( mcastTreeId ):
   p2mpFecElement = LdpP2mpFecElement.fromProtoData( mcastTreeId.protoData )
   rootIp = p2mpFecElement.rootIp
   opaqueVal = p2mpFecElement.opaqueVal
   tlvParser = MldpOpaqueValTlvParser( opaqueVal, True )
   if not p2mpFecElement or not tlvParser.valid:
      return 'Invalid: {}'.format( mcastTreeId )
   opaqueValPrettyString = tlvParser.getOpaqueString( False, False )
   return '{}, {}'.format( rootIp, opaqueValPrettyString )

def getTunnelMulticastMldpRibModel( mrib ):
   entries = {}
   for mcastTreeId, mribEntry in mrib.entry.items():
      tunnelId = mribEntry.tunnelId
      tunnelType = getTunnelTypeFromTunnelId( tunnelId )
      entry = TunnelRibModel.MulticastTunnelRibEntry()
      entry.tunnelType = TunnelModels.tunnelTypeStrDict[ tunnelType ]
      entry.tunnelIndexes = [ getTunnelIndexFromId( tunnelId ) ]
      entries[ getMldpPrettyPrintOpaqueValue( mcastTreeId ) ] = entry

   return TunnelRibModel.MulticastTunnelRib( name='mldp', tunnels=entries )

def getRsvpPrettyPrintSessionId( mcastTreeId ):
   sessionId = Tac.Type( 'Routing::Multicast::MvpnRsvpSessionId' ).fromProtoData(
                                                            mcastTreeId.protoData )
   return sessionId.toStrepForUI()

def getTunnelMulticastRsvpRibModel( mrib ):
   entries = {}
   for mcastTreeId, mribEntry in mrib.entry.items():
      tunnelId = mribEntry.tunnelId
      tunnelType = getTunnelTypeFromTunnelId( tunnelId )
      entry = TunnelRibModel.MulticastTunnelRibEntry()
      entry.tunnelType = TunnelModels.tunnelTypeStrDict[ tunnelType ]
      entry.tunnelIndexes = [ getTunnelIndexFromId( tunnelId ) ]
      entries[ getRsvpPrettyPrintSessionId( mcastTreeId ) ] = entry

   return TunnelRibModel.MulticastTunnelRib( name='rsvp', tunnels=entries )

def getStaticMcastPrettyPrintOpaqueValue( mcastTreeId ):
   afByte = mcastTreeId.protoData[ 0 ]
   afStr = 'Invalid AF'
   if afByte == Tac.enumValue( 'Arnet::AddressFamily', AddressFamily.ipv4 ):
      afStr = 'IPv4'
   elif afByte == Tac.enumValue( 'Arnet::AddressFamily', AddressFamily.ipv6 ):
      afStr = 'IPv6'
   vrfId = struct.unpack( "!I", mcastTreeId.protoData[ 1 : 5 ] )[ 0 ]
   vrfEntry = vrfIdMapStatus.vrfIdToName.get( vrfId )
   if vrfEntry is not None:
      return '{}, VRF {}'.format( afStr, vrfEntry.vrfName )
   return '{}, VRF ID {}'.format( afStr, vrfId )

def getTunnelMulticastStaticRibModel( mrib ):
   entries = {}
   for mcastTreeId, mribEntry in mrib.entry.items():
      tunnelId = mribEntry.tunnelId
      tunnelType = getTunnelTypeFromTunnelId( tunnelId )

      entry = TunnelRibModel.MulticastTunnelRibEntry()
      entry.tunnelType = TunnelModels.tunnelTypeStrDict[ tunnelType ]
      entry.tunnelIndexes = [ getTunnelIndexFromId( tunnelId ) ]
      entries[ getStaticMcastPrettyPrintOpaqueValue( mcastTreeId ) ] = entry

   return TunnelRibModel.MulticastTunnelRib( name='staticMulticast',
                                             tunnels=entries )

def showMulticastTunnelRib( mode, args ):
   if args.get( 'mldp' ):
      return getTunnelMulticastMldpRibModel( tunnelMldpMrib )
   elif args.get( 'rsvp' ):
      return getTunnelMulticastRsvpRibModel( tunnelRsvpMrib )
   elif args.get( 'static' ):
      return getTunnelMulticastStaticRibModel( tunnelStaticMrib )
   else:
      return None

showTunnelRibSyntax = 'show tunnel rib multicast ( mldp | static )'
showTunnelRibData = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'multicast': tokenMulticastRibMatcher,
      'mldp': tokenMldpRibMatcher,
      'static': tokenStaticRibMatcher,
   }
if toggleMvpnRsvpP2mpEnabled():
   showTunnelRibData[ 'rsvp' ] = tokenRsvpRibMatcher
   showTunnelRibSyntax = 'show tunnel rib multicast ( mldp | rsvp | static )'

class ShowTunnelMulticastRibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = showTunnelRibSyntax
   data = showTunnelRibData
   cliModel = TunnelRibModel.MulticastTunnelRib
   handler = showMulticastTunnelRib

BasicCli.addShowCommandClass( ShowColoredTunnelRibCmd )
BasicCli.addShowCommandClass( ShowTunnelRibCmd )
BasicCli.addShowCommandClass( ShowTunnelMulticastRibCmd )
BasicCli.addShowCommandClass( ShowNamedTunnelRibSummaryCmd )
BasicCli.addShowCommandClass( ShowTunnelRibsSummaryCmd )
BasicCli.addShowCommandClass( ShowTunnelRibCandidatesCmd )

#------------------------------------------------------------------------------------
# show tech-support tunnel graceful-restart
#------------------------------------------------------------------------------------
class ShowTechTunnelGr( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tech-support tunnel graceful-restart' )
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'tunnel' : "Show aggregated status details for Tunnel",
      'graceful-restart' : "Graceful restart information for Tunnel agent",
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      AgentCommandRequest.runSocketCommand( em, TunnelAgent.name,
                                            'debug', 'DUMP_GR_STATE' )

BasicCli.addShowCommandClass( ShowTechTunnelGr )


#------------------------------------------------------------------------------------
# show <ldp | segment-routing> tunnel [ <tunnel-index> | <tep> ]
#------------------------------------------------------------------------------------
def getTunnelEntryAndIdFromIndexAndAf( tunnelIndex=None, tunnelTable=None,
                                       tunnelModel=None ):
   tunnelTableModel = tunnelModel( entries={} )
   # the tunnel with corresponding tunnelIndex can belong to
   # v4 tunnel or v6 tunnel but not both
   tunnelIdV4 = tunnelTableModel.getTunnelIdFromIndex( index=tunnelIndex,
                                                       addrFamily='ipv4' )
   tunnelV4 = tunnelTable.entry.get( tunnelIdV4 )
   tunnelIdV6 = tunnelTableModel.getTunnelIdFromIndex( index=tunnelIndex,
                                                       addrFamily='ipv6' )
   tunnelV6 = tunnelTable.entry.get( tunnelIdV6 )
   tunnel = tunnelV4 if tunnelV4 else tunnelV6
   tunnelId = tunnelIdV4 if tunnelV4 else tunnelIdV6

   return ( tunnel, tunnelId )

#--------------------------------------------------------------------------
# register show tech-support for tunnel infrastructure
#--------------------------------------------------------------------------
def _showTechGuard():
   return AgentDirectory.agentIsRunning( sysname, TunnelAgent.name )

cmdsList = [ 'show tunnel rib brief',
             'show tunnel rib candidates',
             'show tunnel rib colored system-colored-tunnel-rib brief',
             'show tunnel rib system-tunneling-ldp-tunnel-rib brief',
             'show tunnel rib system-igp-shortcut-tunnel-rib brief',
             'show tunnel fib debug' ]
if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled() :
   cmdsList.append( 'show tunnel rib colored '
                    'system-colored-tunneling-ldp-tunnel-rib brief' )
if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled() :
   cmdsList.append( 'show tunnel rib colored '
                     'system-colored-igp-shortcut-tunnel-rib brief' )


TechSupportCli.registerShowTechSupportCmd(
   '2020-07-28 14:04:13',
   cmds=cmdsList,
   cmdsGuard=_showTechGuard )

TechSupportCli.registerShowTechSupportCmd(
   '2020-06-12 11:31:01',
   summaryCmds=[ 'show tunnel rib summary' ],
   summaryCmdsGuard=_showTechGuard )

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2017-11-03 12:06:10',
   cmds=[ 'show tunnel rib brief',
          'show tunnel rib colored system-colored-tunnel-rib brief',
          'show tunnel rib system-tunneling-ldp-tunnel-rib brief',
          'show tunnel rib system-igp-shortcut-tunnel-rib brief',
          'show tunnel fib debug', ],
   cmdsGuard=_showTechGuard,
   extended='evpn' )

#------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------
def Plugin( entityManager ):
   global sysname
   global nexthopGroupTunnelTable
   global systemTunnelRib
   global systemColoredTunnelRib
   global systemSharkStatus
   global tunnelRibNameIdMap
   global ldpTunnelTable
   global staticTunnelTable
   global srTunnelTable
   global ospfSrTunnelTable
   global rsvpFrrTunnelTable
   global rsvpLerSubTunnelTable
   global rsvpLerTunnelTable
   global bgpLuTunnelTable
   global srTePolicyTunnelTable
   global isisFlexAlgoTunnelTable
   global programmingStatus
   global tunnelSmashEm
   global tilfaTunnelTable
   global ipInIpTunnelTable
   global gueTunnelFib
   global tunnelRibWatcher
   global tunnelMldpMrib
   global tunnelRsvpMrib
   global tunnelStaticMrib
   global vrfIdMapStatus
   global em

   em = entityManager
   sysname = entityManager.sysname()
   systemTunnelRib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
      readerInfo )
   systemTunnelRib.id = systemTunnelRibId
   tunnelRibNameIdMap = LazyMount.mount( entityManager,
                                         'tunnel/tunnelRibs/tunnelRibNameIdMap',
                                         'Tunnel::TunnelTable::TunnelRibNameIdMap',
                                         'r' )
   systemColoredTunnelRib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelRibs/status/1',
      'Tunnel::TunnelTable::ColoredTunnelRib',
      readerInfo )
   systemColoredTunnelRib.id = systemColoredTunnelRibId
   tunnelRibWatcher = None
   tunnelMldpMrib = LazyMount.mount( entityManager,
                                     MldpP2mpTunnelRib.protocolRibMountPath(),
                                     "Tunnel::TunnelTable::MldpP2mpTunnelRib",
                                     "r" )
   tunnelRsvpMrib = LazyMount.mount( entityManager,
                                     RsvpP2mpTunnelRib.protocolRibMountPath(),
                                     "Tunnel::TunnelTable::RsvpP2mpTunnelRib",
                                     "r" )
   tunnelStaticMrib = LazyMount.mount( entityManager,
                                       StaticTunnelMrib.protocolRibMountPath(),
                                       "Tunnel::TunnelTable::StaticTunnelMrib",
                                       "r" )
   vrfIdMapStatus = SmashLazyMount.mount( entityManager,
                                          "vrf/vrfIdMapStatus",
                                          "Vrf::VrfIdMap::Status",
                                          readerInfo,
                                          autoUnmount=True )

   # mount top-level SharkStatus root; internalCliRib dirs are read on-demand
   systemSharkStatus = SharkLazyMount.mount(
      entityManager,
      'tunnel/sharkStatus',
      'Tunnel::TunnelTable::TunnelSharkStatus',
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount=True )

   ldpTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.ldpTunnelTable, entityManager )
   staticTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.staticTunnelTable, entityManager )
   srTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTunnelTable, entityManager )
   ospfSrTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.ospfSrTunnelTable, entityManager )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
   nexthopGroupTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.nexthopGroupTunnelTable, entityManager )
   bgpLuTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.bgpLuTunnelTable, entityManager )
   srTePolicyTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTePolicyTunnelTable, entityManager )
   rsvpFrrTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.rsvpFrrTunnelTable, entityManager )
   rsvpLerSubTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.rsvpLerSubTunnelTable, entityManager )
   rsvpLerTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )
   isisFlexAlgoTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.isisFlexAlgoTunnelTable, entityManager )
   gv.srv6TransportTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srv6TransportTunnelTable, entityManager )
   ipInIpTunnelTable = readMountTunnelTable(
       TunnelTableIdentifier.ipInIpTunnelTable, entityManager )

   gueTunnelFib = SmashLazyMount.mount( entityManager,
                                        "tunnel/gueTunnelFib",
                                        "Tunnel::TunnelFib::TunnelFib",
                                        readerInfo )

   programmingStatus = SmashLazyMount.mount( entityManager,
                                       "routing/hardware/tunnelProgramming/status",
                                       "Tunnel::Hardware::TunnelProgrammingStatus",
                                             readerInfo )
   tunnelSmashEm = SharedMem.entityManager( sysdbEm=entityManager )
