# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from operator import attrgetter
from Arnet import IpGenPrefix
from ArnetModel import (
   IpGenericAddress,
   IpGenericPrefix
)
from CliModel import (
   Str,
   Dict,
   Enum,
   List,
   Int,
   Bool,
   Model,
   Submodel
)
from CliPlugin.TunnelCli import (
   tunnelViaStatusTacToCapi,
)
from CliPlugin.TunnelModels import (
   getTunnelViaModelFromTunnelIntf,
   IpVia,
   TunnelId,
   tunnelTypeEnumValues,
   tunnelTypesReverseStrDict,
   SrTePolicyId
)
import SmashLazyMount
import LazyMount
import copy
from TunnelLib import (
   getDyTunTidFromIntfId,
   getTunnelIdFromIndex,
   isDyTunIntfId,
)
import TypeFuture
from Toggles import TunnelToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'TunnelFibModel' )
t0 = Tracing.trace0

# pylint: disable-msg=too-many-nested-blocks

TacTunnelId = TypeFuture.TacLazyType( "Tunnel::TunnelTable::TunnelId" )
readerInfo = SmashLazyMount.mountInfo( 'reader' )
TacMplsStackIndex = TypeFuture.TacLazyType( "Arnet::MplsStackEntryIndex" )
TacNexthopGroupIntfId = TypeFuture.TacLazyType( 'Arnet::NexthopGroupIntfId' )
FecId = TypeFuture.TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TypeFuture.TacLazyType( 'Arnet::FecIdIntfId' )
TunnelTableIdentifier = TypeFuture.TacLazyType(
   "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TypeFuture.TacLazyType(
   "Tunnel::TunnelTable::TunnelTableMounter" )
DsfConstants = TypeFuture.TacLazyType( "Dsf::DsfConstants" )
DsfIntfNameHelper = TypeFuture.TacLazyType( "Dsf::DsfIntfNameHelper" )
SystemPortId = TypeFuture.TacLazyType( "Dsf::SystemPortId" )

#  must remain consistent (or have values added) to remain revision compatible
tunnelViaStatusCapiEnumVals = list( tunnelViaStatusTacToCapi.values() ) +\
      [ 'notProgrammed' ]

fecModeStatus = None
fecModeSm = None
fwdStatus = None
fwd6Status = None
fwdGenStatus = None
unifiedFwdStatus = None
unifiedFwd6Status = None
unifiedFwdGenStatus = None

unifiedTunnelFibViewHelper = None
unifiedTunnelFib = None
tunnelMulticastFib = None
nhgStatus = None
policyForwardingStatus = None
srTePolicyTunnelTable = None
dsfConfig = None
virtualOutputIntfIdTable = None

def getTunnelViaStatusStr( tunnelViaStatus ):
   if tunnelViaStatus == 'usingBackupVias':
      return "Backup"
   if tunnelViaStatus == 'usingPrimaryVias':
      return "Primary"
   return "None"

class L3TunnelInfo ( Model ):
   srcAddr = IpGenericAddress(
         help="Location in the IP network where the tunnel starts" )
   dstAddr = IpGenericAddress(
         help="Location in the IP network where the tunnel terminates" )
   dscp = Int( optional=True, help="Differentiated services code point" )
   hoplimit = Int( optional=True, help="Hop limit" )
   tos = Int( optional=True, help="Type of service" )
   ttl = Int( optional=True, help="Time to live" )

   def updateOptionalAttributes(
         self, af, dscp=None, hoplimit=None, tos=None, ttl=None ):
      if af == "ipv4":
         if tos:
            self.tos = tos
         if ttl:
            self.ttl = ttl
      else:
         if dscp:
            self.dscp = dscp
         if hoplimit:
            self.hoplimit = hoplimit

   def getList( self ):
      attributes = list()
      attributes.append( f"destination {self.dstAddr}" )
      attributes.append( f"source {self.srcAddr}" )
      if self.hoplimit:
         attributes.append( f"hoplimit {self.hoplimit}" )
      if self.dscp:
         attributes.append( f"dscp 0x{self.dscp:x}" )
      if self.ttl:
         attributes.append( f"ttl {self.ttl}" )
      if self.tos:
         attributes.append( f"tos {self.tos}" )
      return attributes

   def getStr( self ):
      return ", ".join( self.getList() )

class GreTunnelInfo ( L3TunnelInfo ):
   sequence = Bool( help="Sequence numbers in use", default=False )
   checksum = Bool( help="Checksum in use", default=False )
   key = Int( optional=True, help="GRE key" )

   def getList( self ):
      attributes = super().getList()
      if self.key:
         attributes.append( f"key {self.key}" )
      if self.sequence:
         attributes.append( "sequence" )
      if self.checksum:
         attributes.append( "checksum" )
      return attributes

   def getStr( self ):
      return ", ".join( [ "GRE" ] + self.getList() )

class IpsecTunnelInfo ( L3TunnelInfo ):
   def getStr( self ):
      return ", ".join( [ "IPsec" ] + self.getList() )

class IpsecGreTunnelInfo ( GreTunnelInfo ):
   def getStr( self ):
      return ", ".join( [ "GRE over IPsec" ] + self.getList() )

class IpInIpTunnelInfo( L3TunnelInfo ):
   def getStr( self ):
      return ", ".join( [ "IP-in-IP" ] + self.getList() )

class MplsTunnelInfo ( Model ):
   labelStack = List( valueType=str, help="MPLS label stack" )

   def getList( self ):
      attributes = list()
      if self.labelStack:
         attributes.append( "label " + " ".join( self.labelStack ) )
      return attributes

   def getStr( self ):
      return ", ".join( self.getList() )

class Srv6TransportTunnelInfo( Model ):
   srcAddr = IpGenericAddress(
      help="Location in the IP network where the tunnel starts" )

   def getList( self ):
      attributes = list()
      attributes.append( f"source {self.srcAddr}" )
      return attributes

   def getStr( self ):
      return ", ".join( [ "SRv6" ] + self.getList() )

class VoqTunnelInfo( Model ):
   systemPortId = Int( help="VOQ tunnel system port ID" )
   rewriteIndex = Int( help="VOQ tunnel rewrite index" )

   def getList( self ):
      attributes = list()
      attributes.append( f"system port ID {self.systemPortId}" )
      attributes.append( f"rewrite index 0x{self.rewriteIndex:x}" )
      return attributes

   def getStr( self ):
      return ", ".join( self.getList() )

class TunnelFibVia( IpVia ):
   __revision__ = 2
   encapId = Int( optional=True, help="Encapsulation identifier" )
   mplsEncap = Submodel(
         optional=True, valueType=MplsTunnelInfo,
         help="Encapsulation parameters for MPLS tunnel" )
   ipsecEncap = Submodel(
         optional=True, valueType=IpsecTunnelInfo,
         help="Encapsulation parameters for IPsec tunnel" )
   greEncap = Submodel(
         optional=True, valueType=GreTunnelInfo,
         help="Encapsulation parameters for GRE tunnel" )
   ipsecGreEncap = Submodel(
         optional=True, valueType=IpsecGreTunnelInfo,
         help="Encapsulation parameters for GRE over IPsec tunnel" )
   ipInIpEncap = Submodel(
         optional=True, valueType=IpInIpTunnelInfo,
         help="Encapsulation parameters for IP-in-IP tunnel" )
   srv6Encap = Submodel(
         optional=True, valueType=Srv6TransportTunnelInfo,
         help="Encapsulation parameters for SRv6 transport tunnel" )
   if TunnelToggleLib.toggleDsfPhase1Enabled():
      voqEncap = Submodel(
            optional=True, valueType=VoqTunnelInfo,
            help="Encapsulation parameters for VOQ tunnel" )
   nhgName = Str( optional=True, help="Nexthop Group name for NHG tunnel" )
   nhgId = Int( optional=True, help="Nexthop Group Id for NHG tunnel" )
   _isBackupVia = Bool( help="Backup Via", default=False )
   ucmpGenerationMode = Enum( values=( 'linkBandwidth', ),
                              help="UCMP mode", optional=True )

   resolvingTunnel = Submodel(
         optional=True, valueType=TunnelId,
         help="Resolving tunnel Information" )
   srTePolicyId = Submodel(
         optional=True, valueType=SrTePolicyId,
         help="SR-TE policy information" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if not isDyTunIntfId( dictRepr.get( 'interface' ) ) and \
            dictRepr.get( 'resolvingTunnel' ):
            dictRepr.pop( 'resolvingTunnel' )
      return dictRepr

   def getNhgEntry( self ):
      cachedNHG = {}
      for key, entry in nhgStatus.nexthopGroupEntry.items():
         cachedNHG[ entry.nhgId ] = key.nhgName()
      return cachedNHG

   def getSrTePolicyVias( self, tunnelId ):
      vias = []
      if tunnelId.type == 'SR-TE Policy':
         tacTid = TacTunnelId( tunnelId.toRawValue() )
         intfId = tacTid.intfId()
         fecId = FecIdIntfId.intfIdToFecId( intfId )
         policyFec = policyForwardingStatus.fec.get( fecId )
         if policyFec:
            for via in policyFec.via.values():
               resolvingTunnel = getTunnelViaModelFromTunnelIntf( via.intfId )
               vias.append( resolvingTunnel )
      return vias

   def getEncap( self ):
      # Modify for each Encap type
      return None

   def getEncapStr( self ):
      # pylint: disable=assignment-from-none
      encap = self.getEncap()
      if encap:
         return f" {encap.getStr()}"
      return ""

   def getViaOrBackUpViaStr( self ):
      if self._isBackupVia:
         return 'backup via '
      return 'via '

   def getUcmpEligibleViaStr( self ):
      if self.ucmpGenerationMode == 'linkBandwidth':
         return ', weight link bandwidth'
      return ''

   def appendDebugAttribute( self, attributes ):
      # Debug command option is used
      # encapId is only set with debug option
      if self.getEncap() and self.encapId is not None:
         attributes.append( f"encapId {self.encapId}" )

   def render( self, indent=3 ):
      """
      Renders each via attribute
      Example Output for Vias:
         via 6.1.1.2, 'Test6' label 50 40 30
      Example Output for NHG Vias:
         via NexthopGroupUnknown, NHG ID 1
      """
      attributes = list()
      viaStr = self.getViaOrBackUpViaStr()
      if self.nhgName:
         attributes.append( viaStr + self.nhgName )
         attributes.append( f"NHG ID {self.nhgId}" )
      else:
         attributes.append( viaStr + str( self.nexthop ) )
         attributes.append( str( self.interface ) )

      self.appendDebugAttribute( attributes )

      print( " " * indent + ", ".join( attributes ) + self.getEncapStr()
             + self.getUcmpEligibleViaStr() )

   def renderSrTePolicyTunnelVia( self, cachedNHG, visitedTunnelIds, indent ):
      '''
      This function prints the details required for a tunnelVia with
      an SR-TE Policy resolving tunnel. It then recurses over the SR-TE policy
      vias (which will be SR-TE segment list tunnels).

      Example text output:
      Type 'BGP LU', index 1, endpoint 1.0.5.1/32, forwarding Primary
         via SR-TE Policy index 0, endpoint 1.0.5.1/32, color 200 label 110 120
            via SR-TE tunnel index 1
               via 1.0.4.2, 'Ethernet4' label 3000

      Example JSON output:
      {
          "categories": {
               "BGP LU": {
                   "entries": {
                        "1": {
                            "tunnelIndex": 1,
                            "endpoint": "1.0.5.1/32",
                            "tunnelType": "BGP LU",
                            "vias": [
                                {
                                    "mplsEncap": {
                                        "labelStack": [
                                            "110",
                                            "120"
                                        ]
                                    },
                                    "resolvingTunnel": {
                                        "type": "SR-TE Policy",
                                        "index": 0
                                    },
                                    "interface": "",
                                    "srTePolicyId": {
                                        "color": 200,
                                        "endpoint": "1.0.5.1/32",
                                        ]
                                    },
                                    "nexthop": "5.1.1.2",
                                    "type": "ip"
                                }
                            ],
                            "tunnelViaStatus": "usingPrimaryVias",
                            "backupVias": []
                       }
                   }
               }
           }
      }
      '''
      tunnelIdStr = self.resolvingTunnel.renderTunnelIdStr( tunStr="index" )
      if self.srTePolicyId:
         tunnelIdStr += f", {self.srTePolicyId.renderStr()}"
      print( f"{' ' * indent}via {tunnelIdStr} {self.getEncapStr()}" )
      encapId = Tac.Value( "Tunnel::TunnelTable::EncapId" )
      resolvingTunnels = self.getSrTePolicyVias( self.resolvingTunnel )
      for resolvingTunnel in resolvingTunnels:
         tacTid = TacTunnelId( resolvingTunnel.toRawValue() )
         intfId = tacTid.intfId()
         tunVia = Tac.newInstance( 'Tunnel::TunnelTable::TunnelVia',
                                   self.nexthop, intfId, encapId )
         viaModel = getTunnelFibViaModel( tunVia, cachedNHG,
                                          resolvingTunnel=resolvingTunnel )
         viaModel.renderTunnelVia( cachedNHG, visitedTunnelIds, indent + 3 )

   def renderResolvingTunnelVia( self, cachedNHG, visitedTunnelIds, indent=3 ):
      '''
      This function will recurse over resolving tunnels to print the details
      of the further tunnels.
      It will fetch the tunnel Id from the models interface and fetch the tunnel
      entry from the tunnel fib using this tunnelId. It will print the current tunnel
      information and it will recurse over this tunnel fib entry and repeat
      this process until it gets a nexthop and an interface without any
      further tunnels. It then prints the final nexthop and interface and returns.
      e.g.:
      Type 'IS-IS SR', index 4, endpoint 26::1:1:0/112, forwarding None
         via TI-LFA tunnel index 4 label 350 340 330
            via 47::2:3:9, 'Test47' label 330 320 310
            backup via 48::2:3:9, 'Test48' label 340 330 320
      '''
      if self.resolvingTunnel.type == 'SR-TE Policy':
         self.renderSrTePolicyTunnelVia( cachedNHG, visitedTunnelIds, indent )
         return
      nextTunnelId = getDyTunTidFromIntfId( self.interface )
      viaStr = self.getViaOrBackUpViaStr()
      tunnelIdStr = self.resolvingTunnel.renderTunnelIdStr()
      print( f"{' ' * indent}{viaStr} {tunnelIdStr} {self.getEncapStr()}" )
      # To detect cycles in the tunnel, this map has been added.
      if nextTunnelId in visitedTunnelIds:
         print( f"{' ' * indent}cycle detected at {tunnelIdStr}" )
         return
      visitedTunnelIds[ nextTunnelId ] = True
      tunnelFibEntry = unifiedTunnelFib.entry.get( nextTunnelId )
      if not tunnelFibEntry:
         print( f"{' ' * indent}{tunnelIdStr} not present in Tunnel FIB" )
         return
      for via in sorted( tunnelFibEntry.tunnelVia.values() ):
         # If the tunnel via contains an optimized FEC then render FEC's vias.
         # Otherwise, recursively render the tunnel vias.
         if FecIdIntfId.isFecIdIntfId( via.intfId ):
            self.renderTunnelFibViaFromFec( via, cachedNHG, visitedTunnelIds,
                                             indent + 3 )
         else:
            resolvingTunnel = getTunnelViaModelFromDynTunIntf( via.intfId )
            viaModel = getTunnelFibViaModel( via, cachedNHG,
                                             resolvingTunnel=resolvingTunnel )
            viaModel.renderTunnelVia( cachedNHG, visitedTunnelIds, indent + 3 )
      for via in sorted( tunnelFibEntry.backupTunnelVia.values() ):
         assert not FecIdIntfId.isFecIdIntfId( via.intfId ), \
                "Unexpected FEC ID interface ID for backup via"
         resolvingTunnel = getTunnelViaModelFromDynTunIntf( via.intfId )
         viaModel = getTunnelFibViaModel( via, cachedNHG, backup=True,
                                          resolvingTunnel=resolvingTunnel )
         viaModel.renderTunnelVia( cachedNHG, visitedTunnelIds, indent + 3 )

   def renderTunnelVia( self, cachedNHG, visitedTunnelIds, indent=3 ):
      if self.resolvingTunnel:
         self.renderResolvingTunnelVia( cachedNHG, visitedTunnelIds, indent )
      else:
         self.render( indent )

   def renderTunnelFibVia( self, selfTunnelId, indent=3 ):
      cachedNHG = self.getNhgEntry()
      # Using this dictionary to break the loop for a cycle
      visitedTunnelIds = {}
      visitedTunnelIds[ selfTunnelId ] = True
      self.renderTunnelVia( cachedNHG, visitedTunnelIds, indent )

   def renderTunnelFibViaFromFec( self, tunVia, cachedNHG, visitedTunnelIds,
                                  indent ):
      # Renders a TunnelVia which contains an optimized FEC.

      # Get the vias for the optimized FEC from the forwardingStatus
      fec = getFecFromIntfId( tunVia.intfId )
      if fec is None:
         return
      # Render each of the FEC vias.
      for fecVia in sorted( fec.via.values() ):
         resolvingTunnel = getTunnelViaModelFromDynTunIntf( fecVia.intfId )
         # Convert the FEC via into a TunnelVia for rendering.
         addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
         interface = fecVia.intfId
         createdTunVia = Tac.newInstance( 'Tunnel::TunnelTable::TunnelVia',
                                          addr, interface, tunVia.encapId )
         viaModel = getTunnelFibViaModel( createdTunVia, cachedNHG,
                                          resolvingTunnel=resolvingTunnel )
         viaModel.renderTunnelVia( cachedNHG, visitedTunnelIds, indent )

class MplsFibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.mplsEncap

class IpsecFibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.ipsecEncap

class GreFibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.greEncap

class IpsecGreFibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.ipsecGreEncap

class IpInIpFibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.ipInIpEncap

class Srv6FibVia( TunnelFibVia ):
   def getEncap( self ):
      return self.srv6Encap

class GueTunnelEntryFields( Model ):
   source = IpGenericAddress( help="Source address of the tunnel" )
   payloadType = Str( help="Payload type" )
   tos = Int( optional=True, help="Type of service" )
   ttl = Int( optional=True, help="Time to live" )

   def payloadTypeToStr( self ):
      if self.payloadType == 'ipvx':
         return 'IP'
      elif self.payloadType == 'ipv4':
         return 'IPv4'
      elif self.payloadType == 'ipv6':
         return 'IPv6'
      elif self.payloadType == 'mpls':
         return 'MPLS'
      return ''

   def attributes( self ):
      attributes = list()
      attributes.append( "source " + str( self.source ) )
      attributes.append( "payload type " + self.payloadTypeToStr() )
      attributes.append( "tos " + str( self.tos ) )
      attributes.append( "ttl " + str( self.ttl ) )
      return attributes

class VoqFibVia( TunnelFibVia ):
   interfaceDescription = Str( help="Interface description" )

   def getEncap( self ):
      return self.voqEncap

   def render( self, indent=3 ):
      """
      Renders each via attribute
      Example Output for VOQ Vias:
         via Leaf1:Cpu1, system port ID 201, rewrite index 2097151
      """
      attributes = list()
      viaStr = self.getViaOrBackUpViaStr()

      attributes.append( viaStr + str( self.interfaceDescription ) )
      self.appendDebugAttribute( attributes )

      print( " " * indent + ", ".join( attributes ) + "," + self.getEncapStr() )

class TunnelFibEntry( Model ):
   __revision__ = 2
   tunnelType = Enum( values=tunnelTypeEnumValues, help="Tunnel type" )
   tunnelIndex = Int( help="Tunnel index within tunnel type" )
   vias = List( valueType=TunnelFibVia, help="List of tunnel vias" )
   backupVias = List( valueType=TunnelFibVia,
                      help="List of tunnel backup vias" )
   endpoint = IpGenericPrefix( optional=True,
         help="Route prefix used for underlay nexthop resolution" )
   aliasEndpoints = List( valueType=IpGenericPrefix,
                     help="Additional prefixes used for underlay nexthop resolution",
                     optional=True )
   algorithm = Str( optional=True, help="Algorithm associated with the endpoint" )
   tunnelViaStatus = Enum( values=tunnelViaStatusCapiEnumVals,
                     help="Tunnel programming status", optional=True )
   # Attributes included with debug command option
   interface = Str( optional=True, help="Interface name for tunnel" )
   tunnelId = Int( optional=True, help="Tunnel identifier" )
   tunnelFecId = Int( optional=True, help="FEC identifier" )
   seqNo = Int( optional=True, help="Sequence number of tunnel entry" )

   gueTunnelEntryFields = Submodel( valueType=GueTunnelEntryFields,
                                    help="GUE common fields", optional=True )

   def degradeToV1( self, dictRepr, revision ):
      tr = Tac.newInstance( "Tunnel::TunnelFib::TunnelResolver",
                            unifiedTunnelFib.tunnelFib )

      if revision == 1:
         viaListToRemove = []
         ansList = []
         for i, tunnelFibVia in enumerate( dictRepr[ 'vias' ] ):
            tr.renderedTunnelId.clear()
            tr.renderedTunnelId[ self.tunnelId ] = True
            if tunnelFibVia.get( 'resolvingTunnel' ):
               nextTunnelId = getDyTunTidFromIntfId( tunnelFibVia.get(
                  'interface' ) )
               nextTunnelTableId = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                                              nextTunnelId )
               if not tunnelFibVia.get( 'mplsEncap' ):
                  continue
               if not tr.consolidatedEntry.get( nextTunnelId ):
                  tr.updateFlattenedInfo( nextTunnelTableId )
               processList = [ consEntry.viaInfo for consEntry in
                               tr.consolidatedEntry.values()
                               if consEntry.tunnelId == nextTunnelId ]
               for setVia in processList:
                  for mplsVia in setVia:
                     fibViaCopy = copy.deepcopy( tunnelFibVia )
                     fibViaCopy[ 'interface' ] = mplsVia.intfId
                     fibViaCopy[ 'nexthop' ] = mplsVia.nexthop.stringValue
                     labelOp = mplsVia.labels
                     labels = []
                     fibLabelStack = fibViaCopy[ 'mplsEncap' ][ 'labelStack' ]
                     maxStackSize = TacMplsStackIndex.max
                     stackSize = labelOp.stackSize
                     if stackSize + len( fibLabelStack ) <= maxStackSize:
                        for index in range( stackSize ):
                           labels.append( str( labelOp.labelStack(
                              stackSize - index - 1 ) ) )
                        labels.extend( fibLabelStack )
                     else:
                        labels = fibLabelStack
                     fibViaCopy[ 'mplsEncap' ] = MplsTunnelInfo(
                                                labelStack=labels ).toDict()
                     fibViaCopy.pop( 'resolvingTunnel' )
                     ansList.append( fibViaCopy )
               # Remove all the vias having further tunnel information
               viaListToRemove.append( i )
         # Remove all the vias with further tunnels in them
         for i in sorted( viaListToRemove, reverse=True ):
            dictRepr[ 'vias' ].pop( i )
         # Add the resolved vias back for further tunnels.
         dictRepr[ 'vias' ].extend( ansList )
      return dictRepr

   def getCommonTunnelAttributes( self ):
      """
      Returns a list of the default attributes that are common between tunnel types
      """
      attributes = list()
      attributes.append( f"Type '{self.tunnelType}'" )
      attributes.append( f"index {self.tunnelIndex}" )
      if self.endpoint:
         attributes.append( f"endpoint {self.endpoint}" )
      if self.algorithm:
         attributes.append( f"algorithm {self.algorithm}" )
      if self.tunnelViaStatus:
         attributes.append(
            f"forwarding {getTunnelViaStatusStr( self.tunnelViaStatus )}" )

      return attributes

   def renderTunnelAttributes( self ):
      """
      Renders each tunnel attributes
      Example Output:

      Type 'BGP LU', index 1, endpoint 11.1.1.0/24, forwarding None
      """
      attributes = self.getCommonTunnelAttributes()
      print( ", ".join( attributes ) )

   def renderGueTunnel( self ):
      """
      Renders tunnel attributes of a GUE tunnel
      Example Output:

      Type 'UDP', index 1, endpoint 1.0.2.5/32, forwarding Primary, source 1.0.0.1,
      payload type MPLS, ttl 62, tos 108
      """
      attributes = self.getCommonTunnelAttributes()
      if self.gueTunnelEntryFields:
         attributes += self.gueTunnelEntryFields.attributes()
      print( ", ".join( attributes ) )

   def renderAliasEndpoints( self, indent=3 ):
      """
      Renders each alias endpoint
      Example Output:
         additional endpoints
            28.1.1.0/24
            30.1.1.0/24
      """
      if self.aliasEndpoints:
         print( f"{' ' * indent}additional endpoints" )
         for ep in self.aliasEndpoints:
            print( f"{' ' * ( indent + 3 )}{ep}" )

   def renderDebugAttributes( self, indent=3 ):
      """
      Renders debug attributes when debug command option is included
      Example Output:
         DynamicTunnel2097152.1, id 35184372088833, sequence num 0
         FEC id 432380748599656449
      """
      debugAttributes = list()
      if self.interface:
         debugAttributes.append( self.interface )
      if self.tunnelId is not None:
         debugAttributes.append( f"id {self.tunnelId}" )
      if self.seqNo is not None:
         debugAttributes.append( f"sequence num {self.seqNo}" )

      if debugAttributes:
         print( " " * indent + ", ".join( debugAttributes ) )

      # FEC id goes on seperate line, since it is too long to fit all on same line
      if self.tunnelFecId is not None:
         print( f"{' ' * indent}FEC id {self.tunnelFecId}" )

   def renderSortedVias( self, indent=3 ):
      """
      Renders each via, then backup via in sorted order
      Example Output:
         via 6.1.1.2, 'Test6' label 50 40 30
         via 7.1.1.2, 'Test7' label 50 40 30
      """
      af = self.endpoint.af if self.endpoint else None
      selfTunnelId = self.tunnelId or getTunnelIdFromIndex(
         tunnelTypesReverseStrDict[ self.tunnelType ], self.tunnelIndex, af )
      for via in sorted( self.vias, key=attrgetter( 'nexthop', 'interface' ) ):
         via.renderTunnelFibVia( selfTunnelId, indent )
      for via in sorted( self.backupVias, key=attrgetter( 'nexthop', 'interface' ) ):
         via.renderTunnelFibVia( selfTunnelId, indent )

   def renderVoqTunnel( self ):
      """
      Renders VOQ tunnel
      Example Output:
      Type 'VOQ Fabric', index 4294967296
      """
      attributes = list()
      attributes.append( f"Type '{self.tunnelType}'" )
      attributes.append( f"index {self.tunnelIndex}" )
      print( ", ".join( attributes ) )

   def render( self, indent=3 ):
      """
      Renders each tunnel fib with vias
      If applicable, also renders alias endpoints and debug attributes
      Example Output:

      Type 'BGP LU', index 1, endpoint 11.1.1.0/24, forwarding None
         DynamicTunnel2097152.1, id 35184372088833, sequence num 0
         FEC id 432380748599656449
         via 6.1.1.2, 'Test6' label 50 40 30
         via 7.1.1.2, 'Test7' label 50 40 30
      """
      # Extra empty line before each tunnel
      print()

      # Dictionary used for custom outputs for the tunnel statement
      customTunnelTypeRenderDict = {
         "UDP": self.renderGueTunnel,
         "VOQ Fabric": self.renderVoqTunnel,
      }

      customTunnelTypeRender = customTunnelTypeRenderDict.get( self.tunnelType )

      if customTunnelTypeRender:
         customTunnelTypeRender()
      else:
         self.renderTunnelAttributes()
      self.renderAliasEndpoints( indent )

      self.renderDebugAttributes( indent )
      self.renderSortedVias( indent )

class TunnelFibCategory( Model ):
   __revision__ = 2
   entries = Dict( keyType=int, valueType=TunnelFibEntry,
                   help="Tunnel FIB entries for a tunnel type" )

   def degradeToV1( self, dictRepr, revision ):
      if revision == 1:
         idsToRemove = []
         entries = dictRepr[ 'entries' ]
         for key in sorted( entries ):
            entry = entries[ key ]
            endpoint = entry.get( 'endpoint' )
            af = IpGenPrefix( endpoint ).af if endpoint else None
            selfTunnelId = entry.get( 'tunnelId' ) or \
                           getTunnelIdFromIndex(
                           tunnelTypesReverseStrDict[
                           entry[ 'tunnelType' ] ],
                           entry[ 'tunnelIndex' ], af )

            tunnelFibEntryModel = TunnelFibEntry( tunnelId=selfTunnelId )
            newDict = tunnelFibEntryModel.degradeToV1( entry, revision )
            if not newDict[ 'vias' ]:
               idsToRemove.append( key )
            else:
               entries[ key ] = newDict
         for key in idsToRemove:
            entries.pop( key )
      return dictRepr

   def render( self ):
      for key in sorted( self.entries ):
         self.entries[ key ].render()

class TunnelFib( Model ):
   __revision__ = 2
   categories = Dict( keyType=str, valueType=TunnelFibCategory,
         help="Collections of tunnels in the FIB, grouped by tunnel service type" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         keysToRemove = []
         categories = dictRepr[ 'categories' ]
         for key in sorted( categories ):
            newDict = TunnelFibCategory().degradeToV1(
               categories[ key ], revision )
            if not newDict[ 'entries' ]:
               keysToRemove.append( key )
            else:
               categories[ key ] = newDict
         for key in keysToRemove:
            categories.pop( key )
      return dictRepr

   def render( self ):
      for key in sorted( self.categories ):
         self.categories[ key ].render()

def getTunnelFibToUse( multicast=False, gueTunnel=False, dsfTunnel=False ):
   if multicast:
      return tunnelMulticastFib
   if ( TunnelToggleLib.toggleDynamicGueTunnelsEnabled() and gueTunnel ):
      return unifiedTunnelFib.gueTunnelFib
   if ( TunnelToggleLib.toggleDsfPhase1Enabled() and dsfTunnel ):
      return unifiedTunnelFib.dsfTunnelFib
   else:
      return unifiedTunnelFib.tunnelFib

def getLabelsFromLabelStack( tunnelFibToUse, encapId ):
   labelStackEncap = tunnelFibToUse.labelStackEncap.get(
      encapId, Tac.Value( "Tunnel::TunnelFib::LabelStackEncap" ) )
   labelOp = labelStackEncap.labelStack

   labels = []
   for mplsStackIndex in range( labelOp.stackSize - 1, -1, -1 ):
      labels.append( str( labelOp.labelStack( mplsStackIndex ) ) )
   return labels

def getMplsViaModelFromFecVia( tunnelFibToUse, encapId, fecVia ):
   labels = getLabelsFromLabelStack( tunnelFibToUse, encapId )

   if labels:
      encap = MplsTunnelInfo( labelStack=labels )
      return MplsFibVia( mplsEncap=encap, nexthop=fecVia.hop,
                         interface=fecVia.intfId, type="ip" )
   else:
      return MplsFibVia( nexthop=fecVia.hop, interface=fecVia.intfId, type="ip" )

def getMplsViaModel( tunnelFibToUse, tunnelVia ):
   labels = getLabelsFromLabelStack( tunnelFibToUse, tunnelVia.encapId )

   if labels:
      encap = MplsTunnelInfo( labelStack=labels )
      return MplsFibVia( mplsEncap=encap, nexthop=tunnelVia.nexthop,
                         interface=tunnelVia.intfId, type="ip" )
   else:
      return MplsFibVia( nexthop=tunnelVia.nexthop, interface=tunnelVia.intfId,
                         type="ip" )

def getGreViaModel( tunnelFibToUse, tunnelVia ):
   greEncap = tunnelFibToUse.greEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::TunnelFib::GreEncap' ) )
   greInfo = greEncap.tunnelInfo

   encap = GreTunnelInfo( srcAddr=greInfo.src, dstAddr=greInfo.dst )

   super( GreTunnelInfo, encap ).updateOptionalAttributes(
         af=greInfo.src.af, dscp=greInfo.dscp, hoplimit=greInfo.hoplimit,
         tos=greInfo.tos, ttl=greInfo.ttl )

   if greInfo.oKey or greInfo.iKey:
      encap.key = greInfo.oKey or greInfo.iKey

   return GreFibVia( greEncap=encap, nexthop=tunnelVia.nexthop,
                     interface=tunnelVia.intfId, type="ip" )

def getIpsecViaModel( tunnelFibToUse, tunnelVia ):
   ipsecEncap = tunnelFibToUse.ipsecVtiEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::IpsecVtiTunnelInfo' ) )
   ipsecInfo = ipsecEncap.tunnelInfo

   encap = IpsecTunnelInfo( srcAddr=ipsecInfo.src, dstAddr=ipsecInfo.dst )

   super( IpsecTunnelInfo, encap ).updateOptionalAttributes(
         af=ipsecInfo.src.af, dscp=ipsecInfo.dscp, hoplimit=ipsecInfo.hoplimit,
         tos=ipsecInfo.tos, ttl=ipsecInfo.ttl )

   return IpsecFibVia( ipsecEncap=encap, nexthop=tunnelVia.nexthop,
                       interface=tunnelVia.intfId, type="ip" )

def getIpsecGreViaModel( tunnelFibToUse, tunnelVia ):
   ipsecGreEncap = tunnelFibToUse.ipsecGreEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::IpsecGreTunnelInfo' ) )
   ipsecGreInfo = ipsecGreEncap.tunnelInfo

   encap = IpsecGreTunnelInfo(
         srcAddr=ipsecGreInfo.src, dstAddr=ipsecGreInfo.dst )

   super( IpsecGreTunnelInfo, encap ).updateOptionalAttributes(
         af=ipsecGreInfo.src.af, dscp=ipsecGreInfo.dscp,
         hoplimit=ipsecGreInfo.hoplimit, tos=ipsecGreInfo.tos,
         ttl=ipsecGreInfo.ttl )

   if ipsecGreInfo.oKey or ipsecGreInfo.iKey:
      encap.key = ipsecGreInfo.oKey or ipsecGreInfo.iKey

   return IpsecGreFibVia( ipsecGreEncap=encap, nexthop=tunnelVia.nexthop,
                          interface=tunnelVia.intfId, type="ip" )

def getIpInIpViaModel( tunnelFibToUse, tunnelVia ):
   ipInIpEncap = tunnelFibToUse.ipInIpEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::IpInIpTunnelInfo' ) )
   ipInIpInfo = ipInIpEncap.tunnelInfo

   encap = IpInIpTunnelInfo(
         srcAddr=ipInIpInfo.src, dstAddr=ipInIpInfo.dst )

   super( IpInIpTunnelInfo, encap ).updateOptionalAttributes(
         af=ipInIpInfo.src.af, ttl=ipInIpInfo.ttl )

   return IpInIpFibVia( ipInIpEncap=encap, nexthop=tunnelVia.nexthop,
                        interface=tunnelVia.intfId, type="ip" )

def getSrv6TunnelViaModel( tunnelFibToUse, tunnelVia ):
   srv6Encap = tunnelFibToUse.srv6TunnelEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::Srv6TunnelInfo' ) )
   srv6Info = srv6Encap.encapInfo.tunnelInfo

   encap = Srv6TransportTunnelInfo( srcAddr=srv6Info.src )
   return Srv6FibVia( srv6Encap=encap, nexthop=tunnelVia.nexthop,
                      interface=tunnelVia.intfId, type="ip" )

def getVoqTunnelViaModel( tunnelFibToUse, tunnelVia ):
   voqEncap = tunnelFibToUse.dsfTunnelEncap.get(
      tunnelVia.encapId, Tac.Value( 'Tunnel::DsfTunnelInfo' ) )
   voqInfo = voqEncap.encapInfo
   encap = VoqTunnelInfo( systemPortId=voqInfo.sppId,
                           rewriteIndex=voqInfo.encapIdx )

   spId = SystemPortId( voqInfo.sppId )
   interfaceDescription = DsfIntfNameHelper.longLongNameFromSpId(
      virtualOutputIntfIdTable, dsfConfig, spId )
   spIdIntfId = virtualOutputIntfIdTable.virtualOutputIntfId.get( spId )
   if spIdIntfId:
      interface = spIdIntfId.intfId
   else:
      interface = None

   return VoqFibVia( voqEncap=encap, interfaceDescription=interfaceDescription,
                     interface=interface, nexthop="", type="tunnel" )

def getTunnelFibViaModel( tunnelVia, cachedNHG, debug=False, backup=False,
                          resolvingTunnel=None, multicast=False, gueTunnel=False,
                          dsfTunnel=False, ucmpEligible=False ):
   tunnelFibToUse = getTunnelFibToUse( multicast=multicast, gueTunnel=gueTunnel,
                                       dsfTunnel=dsfTunnel )

   encapTypeViaModelGetDict = {
      "mplsEncap": getMplsViaModel,
      "greEncap": getGreViaModel,
      "ipSecEncap": getIpsecViaModel,
      "ipSecGreEncap": getIpsecGreViaModel,
      "ipInIpEncap": getIpInIpViaModel,
      "srv6TunnelEncap": getSrv6TunnelViaModel,
      "dsfEncap": getVoqTunnelViaModel
   }

   encapTypeViaModelGet = encapTypeViaModelGetDict.get( tunnelVia.encapId.encapType )

   if encapTypeViaModelGet:
      viaModel = encapTypeViaModelGet( tunnelFibToUse, tunnelVia )
   else:
      viaModel = TunnelFibVia( nexthop=tunnelVia.nexthop,
                               interface=tunnelVia.intfId, type="ip" )

   viaModel.resolvingTunnel = resolvingTunnel

   if ucmpEligible:
      viaModel.ucmpGenerationMode = 'linkBandwidth'

   if backup:
      # pylint: disable-msg=protected-access
      viaModel._isBackupVia = True
   if debug:
      viaModel.encapId = tunnelVia.encapId.encapIdValue

   isNhgTunnel = TacNexthopGroupIntfId.isNexthopGroupIntfId( tunnelVia.intfId )
   if isNhgTunnel:
      nhgId = TacNexthopGroupIntfId.nexthopGroupId( tunnelVia.intfId )
      nhgName = cachedNHG.get( nhgId, "NexthopGroupUnknown" )
      viaModel.nhgName = nhgName
      viaModel.nhgId = nhgId
      viaModel.interface = ""

   isSrTePolicyTunnel = resolvingTunnel and resolvingTunnel.type == 'SR-TE Policy'
   if isSrTePolicyTunnel:
      # For SR-TE Policy tunnels, the policy information (endpoint, color)
      # is obtained from it's tunnel table.
      viaModel.interface = ""
      if srTePolicyTunnelTable and srTePolicyTunnelTable.entry:
         tidValue = resolvingTunnel.toRawValue()
         tunnelEntry = srTePolicyTunnelTable.entry.get( tidValue )
         if tunnelEntry:
            viaModel.srTePolicyId = SrTePolicyId(
               endpoint=tunnelEntry.tep.ipGenAddr,
               color=tunnelEntry.color )
   return viaModel

def getTunnelViaModelFromDynTunIntf( intfId ):
   '''Get the resolving tunnel of a dynamic tunnel interface ID, or return None if
   the interface is not a dynamic tunnel interface'''
   if isDyTunIntfId( intfId ):
      return getTunnelViaModelFromTunnelIntf( intfId )
   return None

def getFecFromIntfId( intfId ):
   fecId = FecId( FecIdIntfId.intfIdToFecId( intfId ) )
   fecId = FecId( FecId.convertUsedByTunnelToOriginal( fecId ) )
   unifiedMode = Tac.Type( 'Smash::Fib::FecMode' ).fecModeUnified

   fec = None
   if fecModeStatus.fecMode == unifiedMode:
      if fecId.adjType() == 'fibV4Adj':
         fec = unifiedFwdStatus.fec.get( fecId )
      elif fecId.adjType() == 'fibV6Adj':
         fec = unifiedFwd6Status.fec.get( fecId )
      elif fecId.adjType() == 'fibGenAdj':
         fec = unifiedFwdGenStatus.fec.get( fecId )
   else:
      if fecId.adjType() == 'fibV4Adj':
         fec = fwdStatus.fec.get( fecId )
      elif fecId.adjType() == 'fibV6Adj':
         fec = fwd6Status.fec.get( fecId )
      elif fecId.adjType() == 'fibGenAdj':
         fec = fwdGenStatus.fec.get( fecId )

   return fec

def Plugin( entityManager ):
   global unifiedTunnelFibViewHelper
   global unifiedTunnelFib
   global tunnelMulticastFib
   global nhgStatus
   global policyForwardingStatus
   global srTePolicyTunnelTable
   global dsfConfig
   global virtualOutputIntfIdTable

   mg = entityManager.mountGroup()
   l3Config = mg.mount( "l3/config", "L3::Config", "ri" )

   def _onMountsComplete():
      global fecModeSm, fecModeStatus
      global fwdStatus, fwd6Status, fwdGenStatus
      global unifiedFwdStatus, unifiedFwd6Status, unifiedFwdGenStatus

      # pkgdeps: library RoutingLib
      fecModeStatus = Tac.newInstance( "Smash::Fib::FecModeStatus", "fecModeStatus" )
      fecModeSm = Tac.newInstance( "Ira::FecModeSm", l3Config, fecModeStatus )

      # BUG527092: In future when all the agents in the system use unifiedFecStatus
      # we can clean up the else block.
      unifiedFwdStatus = SmashLazyMount.mount( entityManager,
                                               "forwarding/unifiedStatus",
                                               "Smash::Fib::ForwardingStatus",
                                               readerInfo )
      unifiedFwd6Status = SmashLazyMount.mount( entityManager,
                                                "forwarding6/unifiedStatus",
                                                "Smash::Fib6::ForwardingStatus",
                                                readerInfo )
      unifiedFwdGenStatus = SmashLazyMount.mount( entityManager,
                                                  "forwardingGen/unifiedStatus",
                                                  "Smash::FibGen::ForwardingStatus",
                                                  readerInfo )
      fwdStatus = SmashLazyMount.mount( entityManager,
                                        "forwarding/status",
                                        "Smash::Fib::ForwardingStatus",
                                        readerInfo )
      fwd6Status = SmashLazyMount.mount( entityManager,
                                         "forwarding6/status",
                                         "Smash::Fib6::ForwardingStatus",
                                         readerInfo )
      fwdGenStatus = SmashLazyMount.mount( entityManager,
                                           "forwardingGen/status",
                                           "Smash::FibGen::ForwardingStatus",
                                           readerInfo )

   mg.close( _onMountsComplete )

   unifiedTunnelFibViewHelper = Tac.newInstance(
      "Tunnel::TunnelFib::UnifiedTunnelFibViewHelper",
      entityManager.cEntityManager(), "reader" )
   unifiedTunnelFib = unifiedTunnelFibViewHelper.unifiedView
   tunnelMulticastFib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelMfib', 'Tunnel::TunnelFib::TunnelFib',
      readerInfo )
   nhgStatus = SmashLazyMount.mount( entityManager,
                                     "routing/nexthopgroup/entrystatus",
                                     "NexthopGroup::EntryStatus",
                                     readerInfo )
   policyForwardingStatus = SmashLazyMount.mount( entityManager,
                                                  'forwarding/srte/status',
                                                  'Smash::Fib::ForwardingStatus',
                                                  readerInfo )
   tableInfo = TunnelTableMounter.getMountInfo(
      TunnelTableIdentifier.srTePolicyTunnelTable ).tableInfo
   srTePolicyTunnelTable = SmashLazyMount.mount(
      entityManager, tableInfo.mountPath, tableInfo.tableType, readerInfo )
   dsfConfig = LazyMount.mount(
      entityManager, "dsf/config", "Dsf::DsfConfig", "r" )
   virtualOutputIntfIdTable = SmashLazyMount.mount(
      entityManager, DsfConstants.virtualOutputIntfIdMountPath,
      "Dsf::VirtualOutputIntfIdTable", readerInfo, autoUnmount=True )
