#!/usr/bin/env python3
# Copyright (c) 2009-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Tunnel interface type.  In particular, it provides
# the TunnelIntf class.
#-------------------------------------------------------------------------------
import math
import re

import AgentDirectory
import Ark
import BasicCli
import Cell
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import ConfigMount
import Ethernet
from CliDynamicSymbol import LazyCallback
from CliPlugin import IntfCli
from CliPlugin import TechSupportCli
from CliPlugin import VirtualIntfRule
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.IntfModel import ErrorCounters, InterfaceCounters, \
      InterfaceCountersRate
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.IraIpIntfCli import canSetVrfHook
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
from CliPlugin.TunnelIntfModel import GreIntfTunnelTable
from CliPlugin.TunnelIntfModel import TunnelIntfStatus, TunnelIntfConfigSanity
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.MirroringModelets import (
      matcherDefault,
      matcherGroupName,
      nodeTap,
)
from CliPlugin.TunnelCli import (
      TunnelTableIdentifier,
      readMountTunnelTable,
)
import LazyMount
import SharedMem
import ShowCommand
import Smash
import Tac
import Tracing
import TunnelIntfUtil
from Arnet.NsLib import DEFAULT_NS
from IntfRangePlugin.TunnelIntfRange import TunnelAutoIntfType
from Toggles.TunnelIntfToggleLib import (
      toggleTapAggGreTerminationRoutedPortEnabled,
)
import HostnameCli
from CliParserCommon import MatchResult

em = None

tunModeEnum = Tac.Type( 'Arnet::TunnelIntfMode' )
afEnum = Tac.Type( 'Arnet::AddressFamily' )

# Maps Eos tunnel (mode, address-family) to Linux tunnel (mode, device)
kernelTunnelAttributesMap = {
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv4 )  : ( 'gre', 'gre0' ),
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv6 )  : ( 'ip6gre', 'ip6gre0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv4 ) : ( 'ipip', 'tunl0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv6 ) : ( 'ipip6', 'ip6tnl0' ),
   ( tunModeEnum.tunnelIntfModeIpsec, afEnum.ipv4 ) : ( 'vti', 'ip_vti0' ) }

ip4ConfigDir = None
ip6ConfigDir = None
tunIntfConfigDir = None
tunIntfStatusDir = None
allIntfNsConfigDir = None
hwCapabilitiesCommon = None
hwCapabilities = None
greIntfTunnelTable = None
hw6Capabilities = None
tunnelCountersDir = None
tunnelCountersCheckpointDir = None
aleCliConfig = None
tunnelIntfCliConfig = None
bridgingHwCapabilities = None
internalVlanStatus = None
duplicateConfigDir = None

U32_MAX_VALUE =  0xFFFFFFFF

# Add trace filter class to the Cli agent for tunnel interface commands
__defaultTraceHandle__ = Tracing.Handle( 'TunnelIntfCli' )
t0 = Tracing.trace0

# Returns the Linux device name for an Eos interface name
def intfNameToDeviceName( intfName ):
   return Tac.Value('Arnet::IntfId', intfName ).shortName

# Returns the interface namespace name or 'None' if default namespace.
def getIntfNetnsName( ifName ):
   nscfg = allIntfNsConfigDir.intfNamespaceConfig.get( ifName )
   ns = nscfg.netNsName if nscfg else DEFAULT_NS
   t0( ifName, 'namespace is', ns )
   return ns

# Returns tunnel source address. In case the source is an interface, returns
# IPV4 or IPv6 address (matching the address family of the destination IP)
# of the interface
def findTunnelSourceAddr( srcAddr, srcIntf, dstAddr ):
   if srcAddr and not srcAddr.isUnspecified:
      return srcAddr
   elif srcIntf:
      if dstAddr.af == afEnum.ipv4:
         ip4Config = ip4ConfigDir.ipIntfConfig.get( srcIntf )
         if not ip4Config or not ip4Config.addrWithMask:
            return None
         srcAddr = Tac.Value( 'Arnet::IpGenAddr', ip4Config.addrWithMask.address )
         return None if srcAddr.isUnspecified else srcAddr
      elif dstAddr.af == afEnum.ipv6:
         ip6Config = ip6ConfigDir.intf.get( srcIntf )
         if not ip6Config or not ip6Config.addr :
            return None
         # take the first configured Ipv6 address
         addr = next( iter( ip6Config.addr ) )
         srcAddr = addr.address.stringValue
         srcAddr = Tac.Value( 'Arnet::IpGenAddr', srcAddr )
         return srcAddr
   return None

#-------------------------------------------------------------------------------
# Clear counters hook to clear tunnel counters
#-------------------------------------------------------------------------------
def clearTunnelIntfCountersHook( mode, intfs, sessionOnly, allIntfs ):
   if hwCapabilities.staticTunIntfPlatformCounterSupported:
      # Platform handles counters. TunnelIntf agent doesn't do anything.
      return
   request = tunnelIntfCliConfig.clearCountersRequest

   # "clear counters session" shouldn't trigger copy from current to snapshot table.
   if sessionOnly:
      return

   if allIntfs:
      del request[ 'all' ]
      request[ 'all' ] = True
      return

   # Clear interface list
   for x in intfs:
      del request[ x.name ]
      request[ x.name ] = True

IntfCli.registerClearCountersHook( clearTunnelIntfCountersHook )

#-------------------------------------------------------------------------------
# A subclass for tunnel interfaces
#-------------------------------------------------------------------------------
class TunnelIntf( IntfCli.VirtualIntf ):

   #----------------------------------------------------------------------------
   # Creates a new TunnelIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, cliMode ):
      m = re.match( r'Tunnel(\d+)$', name )
      self.tunnelId = int( m.group( 1 ) )
      IntfCli.VirtualIntf.__init__( self, name, cliMode )
      self.intfConfigDir = tunIntfConfigDir
      self.intfStatuses = tunIntfStatusDir.intfStatus
      self.intfStatus = None

   #----------------------------------------------------------------------------
   # The rule for matching Tunnel interface names. When this pattern matches,
   # it returns an instance of the TunnelIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Tunnel',
      None, None,
      value=lambda cliMode, intf: TunnelIntf( intf, cliMode ),
      helpdesc='Tunnel interface',
      rangeFunc=lambda mode, ctx: ( 0, hwCapabilities.maxTunnelIntfNum - 1
                                    if hwCapabilities
                                    else TunnelIntfUtil.maxTunnelIntfNum ) )

   #----------------------------------------------------------------------------
   # Creates the Interface::TunnelIntfConfig object for this interface if it
   # does not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      self.intfConfigDir.intfConfig.newMember( self.name )

   #----------------------------------------------------------------------------
   # Determines if the Interface::TunnelIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.intfStatuses.get( self.name, None )
      return ( self.intfStatus is not None ) # pylint: disable=superfluous-parens

   #----------------------------------------------------------------------------
   # Destroys the Interface::TunnelIntfConfig object for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del self.intfConfigDir.intfConfig[ self.name ]

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return self.intfConfigDir.intfConfig.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      if not self.intfStatus and not self.lookupPhysical():
         return None
      return self.intfStatus

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      s = self.status()
      assert s
      sParent = s.parent
      if not sParent:
         return None
      # Read s.counter to trigger the externally implemented accessor which may
      # populate s.parent.counterDir as a side effect.
      _ignore = s.counter
      return tunnelCountersDir

   def ingressCountersSupported( self ):
      return self.status().ingressCounterSupported

   def egressCountersSupported( self ):
      return self.status().egressCounterSupported

   def countersSupported( self ):
      return self.ingressCountersSupported() or self.egressCountersSupported()

   def bumCountersSupported( self ):
      return bridgingHwCapabilities.tunnelIntfBumCountersSupported

   def umCountersSupported( self ):
      return bridgingHwCapabilities.tunnelIntfUmCountersSupported

   def countersRateSupported( self ):
      return self.countersSupported()

   def getUpdateTime( self, checkpoint ):
      currentTime = self.counter().statistics.lastUpdate

      # when counters have been cleared, return zero until they have been updated
      if checkpoint and checkpoint.statistics.lastUpdate == currentTime:
         return 0.0

      # convert 'uptime' to UTC
      return Ark.switchTimeToUtc( currentTime )

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = InterfaceCounters( _name=self.status().intfId )
      # pylint: disable-next=protected-access
      intfCountersModel._ingressCounters = self.ingressCountersSupported()
      # pylint: disable-next=protected-access
      intfCountersModel._egressCounters = self.egressCountersSupported()
      # pylint: disable-next=protected-access
      intfCountersModel._umCounters = self.umCountersSupported()
      # _bumCounters will be True only on software forwarded platforms.
      # pylint: disable-next=protected-access
      intfCountersModel._bumCounters = self.bumCountersSupported()
      intfCountersModel._tunnelIntfCounters = True # pylint: disable=protected-access

      def stat( attr, supported=True ):
         if not supported or self.counter() is None:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )
         return currentValue - checkpointValue

      if not zeroOut:
         intfCountersModel.inOctets = stat( 'inOctets' )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts' )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts',
                                                   ( self.bumCountersSupported() or
                                                     self.umCountersSupported()) )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts',
                                                   self.bumCountersSupported() )
         intfCountersModel.inDiscards = stat( 'inDiscards' )
         intfCountersModel.outDiscards = stat( 'outDiscards' )
         intfCountersModel.outOctets = stat( 'outOctets' )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts' )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts',
                                                    ( self.bumCountersSupported() or
                                                      self.umCountersSupported()) )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts',
                                                    self.bumCountersSupported() )
         # NOTE: do NOT use 'stat()' since last update time should never be a delta
         intfCountersModel.lastUpdateTimestamp = self.getUpdateTime( checkpoint )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.inBroadcastPkts = 0
         intfCountersModel.outBroadcastPkts = 0
         intfCountersModel.inDiscards = 0
         intfCountersModel.outDiscards = 0
         intfCountersModel.lastUpdateTimestamp = 0.0

      return intfCountersModel

   def rateValue( self, attr ):
      # Calculate the rate counter value by exponentially decay the
      # previously saved value and subtract it from the current rate
      # value.
      if self.counter() is None or not self.countersRateSupported():
         return 0.0
      counter = self.counter()
      ckpt = self.getLatestCounterCheckpoint()
      loadInterval = IntfCli.getActualLoadIntervalValue(
         self.config().loadInterval )

      currentValue = getattr( counter.rates, attr )
      if ckpt:
         ckptValue = getattr( ckpt.rates, attr, 0 )
         ckptStatsUpdateTime = getattr( ckpt.rates, "statsUpdateTime", 0 )
      else:
         ckptValue = 0
         ckptStatsUpdateTime = 0

      # If loadInterval = 0, return rate is the rate at the most recent counter
      # without being decayed.
      if loadInterval == 0:
         return currentValue

      # If an interface is error disabled, its counters would never be updated. For
      # an error disabled interface, If clear command is issued, the last time when
      # counters are updated will be less than the last time counter are cleared.
      # So Addin a guard for this case!
      if counter.rates.statsUpdateTime < ckptStatsUpdateTime:
         return 0.0

      expFactor = - ( ( counter.rates.statsUpdateTime - ckptStatsUpdateTime ) /
                      float( loadInterval ) )
      decayedCkptValue = ckptValue * math.exp( expFactor )
      return max( 0.0, currentValue - decayedCkptValue )

   def updateInterfaceCountersRateModel( self ):
      cfg = self.config()
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = cfg.description
      interval = IntfCli.getActualLoadIntervalValue( cfg.loadInterval )
      intfCountersRateModel.interval = int( round( interval ) )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )

      return intfCountersRateModel

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      if not sessionOnly:
         # Platform agent manages the checkpoint for hardware forwarded platforms.
         # TunnelIntf agent manages the checkpoint for software forwarded platforms.
         x = tunnelCountersCheckpointDir
         if x is None:
            return None
         y = x.counterSnapshot.get( self.name )
         return y
      else:
         return IntfCli.Intf.getCounterCheckpoint( self,
                                          className='Interface::IntfCounter',
                                          sessionOnly=sessionOnly )

   def clearCounters( self, sessionOnly=False ):
      if not sessionOnly:
         # We have a hook to take care of global clear
         return
      else:
         IntfCli.Intf.clearCounters( self, sessionOnly=sessionOnly )

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   # pylint: disable-next=arguments-renamed
   def showPhysical( self, cliMode, intfStatusModel ):
      pass

   def bandwidth( self ):
      return 0

   def hardware( self ):
      return 'tunnel'

   #----------------------------------------------------------------------------
   # Returns the MAC address of the Tunnel interface. Overrides the addr()
   # function of the base class.
   #----------------------------------------------------------------------------
   def addr( self ):
      tunMacAddr = self.status().macAddr
      if tunMacAddr is None:
         t0( self.name, 'has no MAC address' )
         return None
      result = Ethernet.convertMacAddrCanonicalToDisplay( tunMacAddr )
      t0( self.name, 'canonical MAC address is', result )
      return result

   #----------------------------------------------------------------------------
   # Returns a address string to be used in the "show interface" output
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if not self.addr():
         return None
      # Tunnels do not have burned-in addresses (bia), so leave bia out
      return 'address is ' + self.addr()

   #----------------------------------------------------------------------------
   # Returns an unsorted list of TunnelIntf objects representing all the
   # existing Interface::TunnelIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( cliMode ):
      intfs = []
      iDir = tunIntfStatusDir.intfStatus
      for name in iDir:
         intf = TunnelIntf( name, cliMode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   # Everytime you add a new tunnel config value
   # please please update this function. Else it is a
   # _LOT_ of debugging.
   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      cfg = self.config()
      cfg.tos = TunnelIntfUtil.tunnelDefaultTos
      cfg.ttl = TunnelIntfUtil.tunnelDefaultTtl
      cfg.mode = tunModeEnum.tunnelIntfModeUnspecified
      cfg.srcIntf = ''
      cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
      cfg.dstAddr = cfg.srcAddr
      cfg.dstHost = ''
      cfg.pathMtuDiscovery = False
      cfg.mtu = TunnelIntfUtil.tunnelDefaultMtu
      cfg.key = TunnelIntfUtil.tunnelDefaultKey
      cfg.maxMss = TunnelIntfUtil.tunnelDefaultMaxMss
      cfg.ipsec = False
      cfg.ipsecProfile = ""
      cfg.tapGroup.clear()
      cfg.tapRawIntf.clear()
      cfg.tapEncapGrePreserve = False
      cfg.tapGreProtocolTunnel.clear()
      cfg.tapMemberIntf.clear()
      cfg.decapDipOnlyMatch = False

   def getIntfStatusModel( self ):
      tunIntfStatus = TunnelIntfStatus( name=self.status().intfId )
      tunIntfStatus.tunSrcAddr = self.status().srcAddr
      tunIntfStatus.tunDstAddr = self.status().dstAddr
      if self.config().dstHost:
         tunIntfStatus.tunDstHost = self.config().dstHost
      tunIntfStatus.tunMode = self.status().mode
      tunIntfStatus.tunTtl = self.status().ttl
      tunIntfStatus.tunTos = self.status().tos
      tunIntfStatus.tunMtu = self.status().mtu
      tunIntfStatus.tunPathMtuDiscovery = self.status().pathMtuDiscovery
      tunIntfStatus.tunOKey = self.status().okey
      tunIntfStatus.tunIKey = self.status().ikey
      tunIntfStatus.tunHwFwd = self.status().hardwareForwarded
      tunIntfStatus.tunUnderlayVrf = self.status().underlayVrfName
      tunIntfStatus.tunDiagnosticsInfo = list( self.status().diagnosticsInfo )
      tunIntfStatus.tunDecapDipOnlyMatch = self.status().decapDipOnlyMatch
      return tunIntfStatus

   def getIntfConfigSanityModel( self ):
      tunIntfConfigSanity = TunnelIntfConfigSanity( name=self.status().intfId )
      tunIntfConfigSanity.tunSrcAddr = self.status().srcAddr
      tunIntfConfigSanity.tunDstAddr = self.status().dstAddr
      if self.config().dstHost:
         tunIntfConfigSanity.tunDstHost = self.config().dstHost
      tunIntfConfigSanity.tunMode = self.status().mode
      tunIntfConfigSanity.tunTtl = self.status().ttl
      tunIntfConfigSanity.tunTos = self.status().tos
      tunIntfConfigSanity.tunMtu = self.status().mtu
      tunIntfConfigSanity.tunPathMtuDiscovery = self.status().pathMtuDiscovery
      tunIntfConfigSanity.tunOKey = self.status().okey
      tunIntfConfigSanity.tunIKey = self.status().ikey
      tunIntfConfigSanity.tunHwFwd = self.status().hardwareForwarded
      tunIntfConfigSanity.tunUnderlayVrf = self.status().underlayVrfName
      tunIntfConfigSanity.tunDecapDipOnlyMatch = self.status().decapDipOnlyMatch
      tunIntfConfigSanity.tunDiagnosticsInfo = (
            list( self.status().diagnosticsInfo ) )
      dupConfig = duplicateConfigDir.intfConfig.get( self.status().intfId )
      tunIntfConfigSanity.tunDuplicateTunnels = \
         list( dupConfig.duplicatesIntfId ) if dupConfig else []
      return tunIntfConfigSanity

   def getCountersErrorsModel( self ):
      if not self.countersErrorsSupported():
         return None
      tunCE = ErrorCounters()
      counter = self.counter()
      ckptInErrors = 0
      ckptOutErrors = 0
      ckpt = self.getLatestCounterCheckpoint()
      if ckpt:
         ckptInErrors = getattr( ckpt.statistics, 'inErrors', 0 )
         ckptOutErrors = getattr( ckpt.statistics, 'outErrors', 0 )
      tunCE.fcsErrors = 0
      tunCE.inErrors = counter.statistics.inErrors - ckptInErrors
      tunCE.outErrors = counter.statistics.outErrors - ckptOutErrors
      # fields that are not related to tunnel intfs.
      tunCE.alignmentErrors = 0
      tunCE.frameTooShorts = 0
      tunCE.frameTooLongs = 0
      tunCE.symbolErrors = 0
      return tunCE

   def countersErrorsSupported( self ):
      return bridgingHwCapabilities.tunnelIntfErrorCountersSupported

#-------------------------------------------------------------------------------
# Add a group of tunnel-specific CLI commands to the "config-if" mode
#-------------------------------------------------------------------------------
class TunnelIntfConfigModelet ( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, TunnelIntf )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   em = entityManager

   global tunIntfConfigDir
   tunIntfConfigDir = ConfigMount.mount(
         entityManager, 'interface/config/tunnel/intf',
         'Interface::TunnelIntfConfigDir', 'w' )

   global ip4ConfigDir
   ip4ConfigDir = LazyMount.mount(
         entityManager, "ip/config", "Ip::Config", "r" )

   global ip6ConfigDir
   ip6ConfigDir = LazyMount.mount(
         entityManager, "ip6/config", "Ip6::Config", "r" )

   global tunIntfStatusDir
   tunIntfStatusDir = LazyMount.mount(
         entityManager, 'interface/status/tunnel/intf',
         'Interface::TunnelIntfStatusDir', 'r' )

   global allIntfNsConfigDir
   allIntfNsConfigDir = LazyMount.mount(
         entityManager, Cell.path( 'interface/nsconfig' ),
         'Interface::AllIntfNamespaceConfigDir', 'r' )

   global hwCapabilities
   hwCapabilities = LazyMount.mount(
         entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

   global hwCapabilitiesCommon
   hwCapabilitiesCommon = LazyMount.mount(
         entityManager, 'routing/hardware/statuscommon',
         'Routing::Hardware::StatusCommon', 'r' )

   global internalVlanStatus
   internalVlanStatus = LazyMount.mount(
         entityManager, 'bridging/internalvlan/status',
         'Bridging::InternalVlanStatusDir', 'r' )

   global greIntfTunnelTable
   greIntfTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.greTunnelInterfaceTunnelTable, entityManager )

   global hw6Capabilities
   hw6Capabilities = LazyMount.mount(
         entityManager, 'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )

   sMountGroup = SharedMem.entityManager( sysdbEm=em )
   sMountInfo = Smash.mountInfo( 'reader' )
   global tunnelCountersDir
   tunnelCountersDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/current',
      'Smash::Interface::AllIntfCounterDir', sMountInfo )

   global tunnelCountersCheckpointDir
   tunnelCountersCheckpointDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', sMountInfo )

   global aleCliConfig
   aleCliConfig = LazyMount.mount( entityManager,
         'hardware/ale/cliconfig',
         'Ale::HwCliConfig', 'r' )

   global tunnelIntfCliConfig
   tunnelIntfCliConfig = LazyMount.mount( entityManager,
         'tunnelintf/cliconfig',
         'TunnelIntf::TunnelIntfCliConfig', 'w' )

   global bridgingHwCapabilities
   bridgingHwCapabilities = LazyMount.mount(
         entityManager, 'bridging/hwcapabilities',
         'Bridging::HwCapabilities', 'r' )

   global duplicateConfigDir
   duplicateConfigDir = LazyMount.mount(
         entityManager, 'tunnelintf/duplicates',
         'TunnelIntf::TunnelIntfDuplicateConfigDir', 'r' )

   canSetVrfHook.addExtension( LazyCallback( 
      'TunnelIntfCli.intfCanSetVrfHookForTunnel' ) )

#-------------------------------------------------------------------------------
# Register the TunnelIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( TunnelIntf, TunnelAutoIntfType,
                                  withIpSupport=True )

#-------------------------------------------------------------------------------
# Associate the TunnelIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( TunnelIntfConfigModelet )

#-----------------------------------------------------------------
#ipsecHook that allows the TunnelIntf agent to check if a profile is
#present in ipsec/ike/config.
#-----------------------------------------------------------------
ipsecHook = CliExtensions.CliHook()

modelet = TunnelIntfConfigModelet

class ShowGreTunnelStaticCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show gre tunnel static [ TUNNEL-INDEX ]'
   data = {
      'gre': 'Show GRE information',
      'tunnel': 'GRE tunnel information',
      'static': 'Static GRE tunnel information',
      'TUNNEL-INDEX': CliMatcher.IntegerMatcher( TunnelIntfUtil.minTunnelIntfNum,
                                                 TunnelIntfUtil.maxTunnelIntfNum,
                                                 helpdesc='Tunnel Index' )
   }
   cliModel = GreIntfTunnelTable

   handler = 'TunnelIntfCli.ShowGreTunnelStaticCmdHandler'

BasicCli.addShowCommandClass( ShowGreTunnelStaticCmd )

#-------------------------------------------------------------------------
# register show tech-support extended gre
#-------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2018-07-06 06:42:29',
   cmds=[ 'bash sudo iptables --list-rules -t mangle',
          'bash sudo iptables --list-rules -t raw',
          'show interface Tunnel0-$ config-sanity' ],
   extended='gre' )

TechSupportCli.registerShowTechSupportCmd(
   '2018-07-06 06:42:29',
   cmds=[ 'show ip hardware fib routes' ],
   # Check if we aren't vEOS(Sfa)
   cmdsGuard=lambda: not AgentDirectory.agent( em.sysname(), 'Sfa' ),
   extended='gre' )

TechSupportCli.registerShowTechSupportCmd(
   '2018-07-06 06:42:29',
   cmds=[ 'show platform fap fec all',
          'show platform fap eedb ip-tunnel gre interface',
          'platform fap diag getreg EPNI_IPV4_TOS',
          'platform fap diag getreg EPNI_IPV4_TTL',
          'platform fap diag getreg EPNI_IPV4_SIP' ],
   # Check if we are Sand platform
   cmdsGuard=lambda: AgentDirectory.agent( em.sysname(), 'Sand' ),
   extended='gre' )

matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Source of tunnel packets' )
matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Protocol-over-protocol tunneling' )
supportedEncapModes = { 'gre': 'Generic route encapsulation protocol',
                        'ipsec': 'IPsec-over-IP encapsulation' }
hiddenEncapModes = { 'ipip': 'IP-over-IP encapsulation' }
matcherEncapMode = CliMatcher.EnumMatcher( supportedEncapModes, hiddenEncapModes )

destIntfRangeConfigMatcher = IntfCli.intfRangeWithSingleExpression(
   'INTFS', explicitIntfTypes=( EthPhyAutoIntfType,
                                SwitchAutoIntfType,
                                LagAutoIntfType ) )
tapIntfRangeConfigMatcher = IntfCli.intfRangeWithSingleExpression(
   'TAP_INTFS', explicitIntfTypes=( EthPhyAutoIntfType,
                                    SwitchAutoIntfType ) )

#--------------------------------------------------------------------------------
# tunnel destination HOST_OR_ADDR
#--------------------------------------------------------------------------------
class IpAddrOrHostnameMatcher( IpGenAddrMatcher ):
   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )

   def match( self, mode, context, token ):
      addrRe = self.ipGenAddrRe6_ if ':' in token else self.ipGenAddrRe4_
      m = addrRe.match( token )
      if m is None or m.group( 0 ) != token:
         if not HostnameCli.validateHostname( token ):
            raise CliParser.InvalidInputError()
         return MatchResult( token, token )
      return super().match( mode, context, token )

   def __str__( self ):
      return '<IP (v4 or v6) address> or hostname'

class TunnelDestinationCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel destination HOST_OR_ADDR'
   noOrDefaultSyntax = 'tunnel destination ...'
   data = {
      'tunnel': matcherTunnel,
      'destination': CliMatcher.KeywordMatcher( 'destination',
         helpdesc='Destination of tunnel' ),
   }
   data[ 'HOST_OR_ADDR' ] = IpAddrOrHostnameMatcher(
      helpdesc='IP address or hostname of tunnel interface destination' )
   handler = 'TunnelIntfCli.setDst'
   noOrDefaultHandler = 'TunnelIntfCli.unsetDst'

modelet.addCommandClass( TunnelDestinationCmd )

#--------------------------------------------------------------------------------
# tunnel ipsec profile PROFILE
#--------------------------------------------------------------------------------
class TunnelIpsecProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel ipsec profile PROFILE'
   noOrDefaultSyntax = 'tunnel ipsec profile ...'
   data = {
      'tunnel': matcherTunnel,
      'ipsec': CliMatcher.KeywordMatcher( 'ipsec',
         helpdesc='Secure tunnel with IPsec' ),
      'profile': CliMatcher.KeywordMatcher( 'profile',
         helpdesc='IPsec profile Name' ),
      'PROFILE': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
         helpdesc='Name of the profile', helpname='WORD' )
   }
   handler = 'TunnelIntfCli.setIpsec'
   noOrDefaultHandler = 'TunnelIntfCli.noSetIpsec'

modelet.addCommandClass( TunnelIpsecProfileCmd )

#--------------------------------------------------------------------------------
# tunnel key TUNNELKEY
#--------------------------------------------------------------------------------
class TunnelKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel key TUNNEL_KEY'
   noOrDefaultSyntax = 'tunnel key ...'
   data = {
      'tunnel': matcherTunnel,
      'key': CliMatcher.KeywordMatcher( 'key', helpdesc='Set tunnel key' ),
      'TUNNEL_KEY': CliMatcher.IntegerMatcher( 1, 2**32-1, helpdesc='Key value' )
   }
   handler = 'TunnelIntfCli.setKey'
   noOrDefaultHandler = 'TunnelIntfCli.unsetKey'

modelet.addCommandClass( TunnelKeyCmd )

#--------------------------------------------------------------------------------
# tunnel mode ( gre | ipip | ipsec )
#--------------------------------------------------------------------------------

class TunnelModeCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel mode TUNNEL_MODE'
   noOrDefaultSyntax = 'tunnel mode ...'
   data = {
      'tunnel': matcherTunnel,
      'mode': 'Tunnel encapsulation method',
      'TUNNEL_MODE': matcherEncapMode
   }
   handler = 'TunnelIntfCli.setMode'
   noOrDefaultHandler = 'TunnelIntfCli.unsetMode'

modelet.addCommandClass( TunnelModeCmd )

#--------------------------------------------------------------------------------
# tunnel mss ceiling TUNNELMSS
#--------------------------------------------------------------------------------
class TunnelMssCeilingCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel mss ceiling TUNNEL_MSS'
   noOrDefaultSyntax = 'tunnel mss ceiling ...'
   # hidden as generalized "tcp mss ceiling" is preferred
   hidden = True
   data = {
      'tunnel': matcherTunnel,
      'mss': CliMatcher.KeywordMatcher( 'mss',
         helpdesc='Maximum segment in TCP SYN' ),
      'ceiling': CliMatcher.KeywordMatcher( 'ceiling',
         helpdesc='Set maximum limit' ),
      'TUNNEL_MSS': CliMatcher.IntegerMatcher( 64, 16384, helpdesc='MSS' )
   }
   handler = 'TunnelIntfCli.setMss'
   noOrDefaultHandler = 'TunnelIntfCli.unsetMss'

modelet.addCommandClass( TunnelMssCeilingCmd )

#--------------------------------------------------------------------------------
# tunnel path-mtu-discovery
#--------------------------------------------------------------------------------
class TunnelPathMtuDiscoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel path-mtu-discovery'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnel,
      'path-mtu-discovery': 'Enable Path MTU discovery on tunnel'
   }
   handler = 'TunnelIntfCli.setPathMtuDiscovery'
   noOrDefaultHandler = 'TunnelIntfCli.unsetPathMtuDiscovery'

modelet.addCommandClass( TunnelPathMtuDiscoveryCmd )

#--------------------------------------------------------------------------------
# tunnel source IPGENADDR
#--------------------------------------------------------------------------------
class TunnelSourceIpGenAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel source IPGENADDR'
   noOrDefaultSyntax = 'tunnel source ...'
   data = {
      'tunnel': 'Protocol-over-protocol tunneling',
      'source': matcherSource,
      'IPGENADDR': IpGenAddrMatcher( helpdesc='IP address' )
   }
   handler = 'TunnelIntfCli.setSrcAddr'
   noOrDefaultHandler = 'TunnelIntfCli.unsetSrc'

modelet.addCommandClass( TunnelSourceIpGenAddrCmd )

#--------------------------------------------------------------------------------
# tunnel source interface INTF
#--------------------------------------------------------------------------------
class TunnelSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel source interface INTF'
   data = {
      'tunnel': matcherTunnel,
      'source': matcherSource,
      'interface': 'Source interface of tunnel packets',
      'INTF': IntfCli.Intf.matcherWithIpSupport
   }
   handler = 'TunnelIntfCli.setSrcIntf'

modelet.addCommandClass( TunnelSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# tunnel tos TUNNEL_TOS
#--------------------------------------------------------------------------------
def tunnelIntfTosSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.staticTunIntfTosSupported:
      return CliParser.guardNotThisPlatform
   return None

tosGuardedKw = CliCommand.guardedKeyword( 'tos',
      helpdesc='Set IP type of service',
      guard=tunnelIntfTosSupportedGuard )

class TunnelTosCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel tos TUNNEL_TOS'
   noOrDefaultSyntax = 'tunnel tos ...'
   data = {
      'tunnel': matcherTunnel,
      'tos': tosGuardedKw,
      'TUNNEL_TOS': CliMatcher.IntegerMatcher( 0, 255, helpdesc='ToS' )
   }
   handler = 'TunnelIntfCli.setTos'
   noOrDefaultHandler = 'TunnelIntfCli.unsetTos'

modelet.addCommandClass( TunnelTosCmd )

#--------------------------------------------------------------------------------
# tunnel ttl TUNNEL_TTL
#--------------------------------------------------------------------------------
def tunnelIntfTtlSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.staticTunIntfTtlSupported:
      return CliParser.guardNotThisPlatform
   return None

ttlGuardedKw = CliCommand.guardedKeyword( 'ttl',
      helpdesc='Set time to live',
      guard=tunnelIntfTtlSupportedGuard )

class TunnelTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel ttl TUNNEL_TTL'
   noOrDefaultSyntax = 'tunnel ttl ...'
   data = {
      'tunnel': matcherTunnel,
      'ttl': ttlGuardedKw,
      'TUNNEL_TTL': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL' )
   }
   handler = 'TunnelIntfCli.setTtl'
   noOrDefaultHandler = 'TunnelIntfCli.unsetTtl'

modelet.addCommandClass( TunnelTtlCmd )

#--------------------------------------------------------------------------------
# tunnel underlay vrf VRF
#--------------------------------------------------------------------------------
class TunnelUnderlayVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel underlay VRF'
   noOrDefaultSyntax = 'tunnel underlay ...'
   data = {
      'tunnel': matcherTunnel,
      'underlay': 'Tunnel underlay',
      'VRF': VrfExprFactory( helpdesc='Set tunnel underlay VRF' ),
   }
   handler = 'TunnelIntfCli.setUnderlayVrf'
   noOrDefaultHandler = handler

modelet.addCommandClass( TunnelUnderlayVrfCmd )

#--------------------------------------------------------------------------------
# tunnel decap source any
#--------------------------------------------------------------------------------
def tunnelDecapDipOnlyMatchSupportedGuard( mode, token ):
   if hwCapabilities and not hwCapabilities.tunnelIntfDecapDipOnlyMatch:
      return CliParser.guardNotThisPlatform
   return None

decapGuardedKw = CliCommand.guardedKeyword( 'decap',
      helpdesc='Configure decapsulation parameter',
      guard=tunnelDecapDipOnlyMatchSupportedGuard )

class TunnelDecapDipOnlyMatchCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap source any'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnel,
      'decap': decapGuardedKw,
      'source': matcherSource,
      'any': 'Match on any source IP',
   }
   handler = 'TunnelIntfCli.setTunnelDecapDipOnlyMatch'
   noOrDefaultHandler = 'TunnelIntfCli.unsetTunnelDecapDipOnlyMatch'

#--------------------------------------------------------------------------------
# [ no | default ] tap default interface INTFS
#--------------------------------------------------------------------------------
class TunnelTapDefaultInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tap default interface INTFS' )
   noOrDefaultSyntax = ( 'tap default interface [ INTFS ]' )
   data = {
      'tap' : nodeTap,
      'default' : matcherDefault,
      'interface' : 'Set interfaces for the tap port',
      'INTFS' : destIntfRangeConfigMatcher,
   }

   handler = 'TunnelIntfCli.handleTunnelTapRawIntf'
   noOrDefaultHandler = 'TunnelIntfCli.handleNoTunnelTapRawIntf'

#--------------------------------------------------------------------------------
# [ no | default ] tap default { group GROUP }
#--------------------------------------------------------------------------------
class TunnelTapDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'tap default { group GROUP_NAME }'
   noOrDefaultSyntax = ( 'tap default '
                         ' ( group | { group GROUP_NAME } )' )
   data = {
      'tap' : nodeTap,
      'default' : matcherDefault,
      'group': 'Set tap group for the interface',
      'GROUP_NAME' : matcherGroupName,
   }

   handler = 'TunnelIntfCli.handleTunnelTapGroups'
   noOrDefaultHandler = 'TunnelIntfCli.handleNoTunnelTapGroups'

#--------------------------------------------------------------------------------
# [ no | default ] tap encapsulation gre preserve
#--------------------------------------------------------------------------------
class TunnelTapEncapGrePreserveCmd( CliCommand.CliCommandClass ):
   syntax = 'tap encapsulation gre preserve'
   noOrDefaultSyntax = syntax
   data = {
      'tap' : nodeTap,
      'encapsulation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='Configure encapsulation parameters' ) ),
      'gre' : 'Configure GRE parameters',
      'preserve' : 'Preserve encapsulation header',
   }

   handler = 'TunnelIntfCli.handleTapEncapGrePreserve'
   noOrDefaultHandler = handler

#--------------------------------------------------------------------------------
# [ no | default ] tap interface TAP_INTFS
#                  [ tool group GROUP_NAME | interface INTFS ]
#--------------------------------------------------------------------------------
class TunnelTapIntfCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tap interface TAP_INTFS '
              '[ tool ( { group GROUP_NAME } ) | '
              '       ( toolIntf INTFS ) ] ' )
   noOrDefaultSyntax = (
      'tap interface [ TAP_INTFS '
      '[ tool ( group | { group GROUP_NAME } ) | '
      '       ( toolIntf INTFS ) ] ] ' )

   data = {
      'tap' : nodeTap,
      'interface' : 'Set interfaces for incoming traffic',
      'TAP_INTFS' : tapIntfRangeConfigMatcher,
      'tool' : 'Set tool characteristics of the interface',
      'group': 'Set tap group for the interface',
      'GROUP_NAME' : matcherGroupName,
      'toolIntf' : CliCommand.singleKeyword( 'interface',
         'Set destination interfaces' ),
      'INTFS' : destIntfRangeConfigMatcher,
   }

   handler = 'TunnelIntfCli.handleTunnelTapIntf'
   noOrDefaultHandler = 'TunnelIntfCli.handleNoTunnelTapIntf'

class TunnelTapEncapGreProtocolCmd( CliCommand.CliCommandClass ):
   syntax = '''tap encapsulation gre protocol PROTOCOL
               [ feature header length LENGTH ]
               [ re-encapsulation ethernet ]'''
   noOrDefaultSyntax = 'tap encapsulation gre protocol [ PROTOCOL ] ...'
   data = {
      'tap' : nodeTap,
      'encapsulation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='Configure encapsulation parameters' ) ),
      'gre' : 'Configure GRE parameters',
      'protocol' : 'Protocol type in GRE header',
      'PROTOCOL': CliMatcher.IntegerMatcher( 0, 0xFFFF,
         helpdesc='Protocol range', helpname='0x0000-0xFFFF' ),
      'feature' : 'Feature header options',
      'header' : 'Feature header options',
      'length' : 'Feature header length in bytes',
      'LENGTH' : CliMatcher.IntegerMatcher( 1, 16,
         helpdesc='Header length in bytes' ),
      're-encapsulation' : 'Extra header to prepend to the terminated packet',
      'ethernet' : 'Ethernet header'
   }

   handler = 'TunnelIntfCli.handleTapEncapGreProtocol'
   noOrDefaultHandler = 'TunnelIntfCli.handleNoTapEncapGreProtocol'

modelet.addCommandClass( TunnelTapDefaultInterfaceCmd )
modelet.addCommandClass( TunnelTapDefaultCmd )
modelet.addCommandClass( TunnelTapEncapGrePreserveCmd )
modelet.addCommandClass( TunnelTapEncapGreProtocolCmd )
if toggleTapAggGreTerminationRoutedPortEnabled():
   modelet.addCommandClass( TunnelTapIntfCmd )
modelet.addCommandClass( TunnelDecapDipOnlyMatchCmd )

