# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This file contains tunnel CAPI model definitions that are shared between
# multiple packages. Tunnel models specific to certain packages should be
# defined elsewhere.

from __future__ import absolute_import, division, print_function
from ArnetModel import IpGenericAddress, IpGenericPrefix
from CliModel import Bool, Enum, List, Int, Str
from CliModel import Model, Submodel
from IntfModels import Interface
import Tac
from Toggles import TunnelToggleLib
from TunnelLib import (
   getDyTunTidFromIntfId,
)
from TunnelTypeLib import (
encapTypeEnumValues,
   tunnelTypeEnumValues,
   tunnelTypesReverseStrDict,
   tunnelTypeStrDict,
)
from TypeFuture import TacLazyType

tableTypeAttrs = list( tunnelTypeStrDict.values() )

TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
TunnelIdConstants = TacLazyType( "Tunnel::TunnelTable::TunnelIdConstants" )

TunnelType = Enum( values=tableTypeAttrs, help="Tunnel type" )

class IpTunnelInfo( Model ):
   tunnelEncap = Enum( values=encapTypeEnumValues, help="Tunnel encapsulation type" )
   tunnelSource = IpGenericAddress(
      help="Location in the IP network where the IP tunnel starts" )
   tunnelDestination = IpGenericAddress(
      help="Location in the IP network where the IP tunnel terminates" )
   tunnelKey = Int( help="GRE IP tunnel key", optional=True )
   tunnelDscp = Int( help="Differentiated services code point", optional=True )
   tunnelHoplimit = Int( help="Hop limit", optional=True )
   tunnelTos = Int( help="Type of service", optional=True )
   tunnelTtl = Int( help="Time to live", optional=True )

class TunnelId( Model ):
   __revision__ = 2
   index = Int( help="Tunnel index per tunnel type" )
   type = TunnelType
   _af = Str( help='Address family of tunnel' )

   def renderTunnelIdStr( self, tunStr="tunnel index" ):
      tunnelViaStr = ( self.type + " " + tunStr + " " +
                       str( self.index ) )
      return tunnelViaStr

   def renderStr( self ):
      # pylint: disable-next=consider-using-f-string
      return '%s (%d)' % ( self.type, self.index )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if dictRepr[ 'type' ] == 'Nexthop Group':
            dictRepr[ 'type' ] = 'Nexthop-Group'

      return dictRepr

   def toRawValue( self ):
      tunType = tunnelTypesReverseStrDict[ self.type ]
      index = self.index
      if self._af == 'ipv6':
         index = self.index | TunnelIdConstants.tunnelAfMask
      return TunnelIdType.convertToTunnelValue( tunType, index )

   def fromRawValue( self, value ):
      tun = TunnelIdType( value )
      self.index = tun.tunnelIndex()
      self.type = tun.typeCliStr()
      self._af = tun.tunnelAf()

def getTunnelViaModelFromTunnelIntf( tunnelIntf ):
   tunnelId = getDyTunTidFromIntfId( tunnelIntf )
   recurTunnelId = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                               tunnelId )
   tunnel = TunnelId(
      type=recurTunnelId.typeCliStr(),
      index=recurTunnelId.tunnelIndex() )
   return tunnel

class SrTePolicyId( Model ):
   endpoint = IpGenericAddress(
      help="IP address of the SR-TE policy tunnel endpoint" )
   color = Int( help="Color of SR-TE policy tunnel" )

   def renderStr( self ):
      return 'endpoint ' + str( self.endpoint ) + ', color ' + str( self.color )

class TunnelViaInfo( Model ):
   nexthopAddr = IpGenericAddress( help="Next hop IP address", optional=True )
   interface = Interface( help="Egress L3 interface of next hop",
                                    optional=True )
   nhgName = Str( help="Next-Hop-Group name", optional=True )
   nextHop = Str( help="Next-Hop name", optional=True )
   interfaceDescription = Str( help="Interface description", optional=True )
   labelStack = List( valueType=int, optional=True,
                      help="MPLS label stack (top-of-stack label first)" )
   resolvingTunnel = Submodel(
      valueType=TunnelId, help="Resolving tunnel information", optional=True )
   # SR-TE policy, present when resolving tunnel is SR-TE policy tunnel
   srTePolicyId = Submodel( valueType=SrTePolicyId,
                            help="SR-TE policy id", optional=True )
   if TunnelToggleLib.toggleDsfPhase1Enabled():
      rewriteIndex = Int( help="VOQ tunnel rewrite index", optional=True )
      systemPortId = Int( help="VOQ tunnel system port ID", optional=True )

class TunnelInfo( Model ):
   tunnelType = Enum( values=tunnelTypeEnumValues, help="Tunnel type" )
   tunnelIndex = Int( help="Tunnel table index" )
   tunnelName = Str( help="Tunnel name", optional=True )
   tunnelEndPoint = IpGenericPrefix(
      help="Route prefix for MPLS underlay nexthop resolution",
      optional=True )
   tunnelAlgorithm = Str( help="Flexible algorithm of the tunnel",
                          optional=True )
   tunnelAddressFamily = Enum( values=( 'IPv4', 'IPv6' ),
                               help="Address family for MPLS tunnels",
                               optional=True )
   staticInterfaceTunnelInfo = Submodel(
      valueType=IpTunnelInfo,
      help="Tunnel encap information for a static interface tunnel",
      optional=True )
   tunnelVias = List( valueType=TunnelViaInfo, help="List of tunnel vias",
                      optional=True )
   tunnelColor = Int( help="Tunnel color", optional=True )

class Via( Model ):
   type = Enum( values=[ 'ip', 'tunnel' ],
                 help="Type of via, indicating the expected attributes" )

class IpVia( Via ):
   nexthop = IpGenericAddress( help="Nexthop IP address" )
   interface = Interface( help="Egress L3 interface of the next-hop" )

   def degradeToV1( self, via ):
      """Degrade the Ip via dict to revision 1"""
      del via[ 'type' ]

   def renderStr( self ):
      # pylint: disable-next=consider-using-f-string
      return "via %s %s" % ( self.nexthop, self.interface )

   def render( self ):
      print( self.renderStr() )

class TunnelVia( Via ):
   tunnelId = Submodel( valueType=TunnelId, help="Tunnel Identifier" )

   def degradeToV1( self, via ):
      """Degrade the Tunnel via dict to revision 1"""
      via[ 'nexthop' ] = '0.0.0.0'
      tidType = tunnelTypesReverseStrDict[ via[ 'tunnelId' ][ 'type' ] ]

      afBitMask = 0x0
      if tidType in [ 'staticV6Tunnel', 'srV6Tunnel' ]:
         # FIXME BUG200023 Can remove this afBitMask after convertToTunnelValue
         # is refactored.
         # if ID is a V6 channel, need to flip the address family bit to 1
         afBitMask = TunnelIdConstants.tunnelAfMask
      if tidType in [ 'staticV4Tunnel', 'staticV6Tunnel' ]:
         tidType = 'staticTunnel'
      elif tidType in [ 'srV4Tunnel', 'srV6Tunnel' ]:
         tidType = 'srTunnel'
      tid = Tac.Type( 'Tunnel::TunnelTable::TunnelId' ).convertToTunnelValue(
            tidType, via[ 'tunnelId' ][ 'index' ] ) | afBitMask
      intfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' ).tunnelIdToIntfId( tid )
      via[ 'interface' ] = intfId
      del via[ 'type' ]
      del via[ 'tunnelId' ]

class MplsVia( IpVia ):
   labels = List( valueType=str, help="Label stack" )
   _isBackupVia = Bool( help="Backup Via", default=False )
   # This attribute will be populated if a tunnel is being protected
   # by another tunnel. For e.g SR tunnel being protected by TI-LFA tunnels
   backupTunnelInfo = Submodel( valueType=TunnelInfo,
                                help="Backup Tunnel Information",
                                optional=True )

   def render( self ):
      viaStr = "backup via" if self._isBackupVia else "via"
      labelsStr = " ".join( self.labels )
      # pylint: disable-next=consider-using-f-string
      print( " " * 3 + "%s %s, %s" % ( viaStr, self.nexthop, self.interface ) )
      # pylint: disable-next=consider-using-f-string
      print( " " * 6 + "label stack %s" % labelsStr )

def sortKey( self ):
   if self.type == 'ip':
      return ( self.type, self.nexthop, self.interface, self.labels )

   if self.type == 'tunnel':
      return ( self.type, self.tunnelId.type, self.tunnelId.index, self.labels )

   assert False, "handle new via type"
   return ( self.type, )

class MplsTunnelVia( TunnelVia ):
   labels = List( valueType=str, help="Label stack" )

class TunnelTableEntry( Model ):
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel" )

class MulticastTunnelTableEntry( Model ):
   mcastTreeId = Str( help="Multicast tree identifier" )
