# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import re

import Arnet
import BasicCli
import CliMatcher
import CliMode.TunnelRib
import CliParser
import CliParserCommon
import ConfigMount
import Tac

from TypeFuture import TacLazyType
import Toggles.TunnelToggleLib
import Toggles.IpLibToggleLib

cliConfig = None
systemTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                              ).systemTunnelRibName
systemTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                            ).systemTunnelRibId
systemColoredTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemColoredTunnelRibName
systemColoredTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                   ).systemColoredTunnelRibId
systemTunnelingLdpTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemTunnelingLdpTunnelRibName
systemTunnelingLdpTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                     ).systemTunnelingLdpTunnelRibId
systemColoredTunnelingLdpTunnelRibName = Tac.Type( "Tunnel::"\
      "TunnelTable::TunnelRibNameIdMap"
       ).systemColoredTunnelingLdpTunnelRibName
systemColoredTunnelingLdpTunnelRibId = Tac.Type( "Tunnel::"\
      "TunnelTable::TunnelRibId"
      ).systemColoredTunnelingLdpTunnelRibId
systemIgpShortcutTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemIgpShortcutTunnelRibName
systemIgpShortcutTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                     ).systemIgpShortcutTunnelRibId
systemColoredIgpShortcutTunnelRibName = Tac.Type( "Tunnel::"\
      "TunnelTable::TunnelRibNameIdMap").systemColoredIgpShortcutTunnelRibName
systemColoredIgpShortcutTunnelRibId = Tac.Type( "Tunnel::"\
      "TunnelTable::TunnelRibId").systemColoredIgpShortcutTunnelRibId

ProtoPrefAction = TacLazyType( "Tunnel::TunnelTable::ProtoPrefAction" )
ProtoPrefMapping = TacLazyType( "Tunnel::TunnelTable::ProtoPrefMapping" )

# -----------------------------------------------------------------------------------
#      modes
# -----------------------------------------------------------------------------------

class TunnelRibsMode( CliMode.TunnelRib.TunnelRibsBaseMode,
                      BasicCli.ConfigModeBase ):

   def __init__( self, parent, session ):
      CliMode.TunnelRib.TunnelRibsBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TunnelRibMode( CliMode.TunnelRib.TunnelRibBaseMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, tunnelRibName ):
      CliMode.TunnelRib.TunnelRibBaseMode.__init__( self, tunnelRibName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config = cliConfig.config.get( self.tunnelRibName )
      if not self.config:
         self.config = cliConfig.config.newMember( self.tunnelRibName )

   def getTunnelTableId( self, args ):
      tokenList = [ 'ldp', 'static', 'labeled-unicast', 
                    'segment-routing', 'nexthop-group', 'rsvp-ler', 'flex-algo' ]
      sourceProtocolLastToken = ''
      for token in tokenList:
         if token in args:
            sourceProtocolLastToken = token
            break
      if sourceProtocolLastToken == 'segment-routing':
         if 'isis' in args:
            sourceProtocolLastToken = 'isis-segment-routing'
         elif 'policy' in args:
            sourceProtocolLastToken = 'segment-routing-policy'
         elif Toggles.TunnelToggleLib.toggleOspfSegmentRoutingTunnelEnabled():
            sourceProtocolLastToken = 'ospf-segment-routing'
      tunnelTableId = CliMode.TunnelRib.tunnelTableIdFromCliToken( 
            sourceProtocolLastToken )
      return tunnelTableId

   def handleEntry( self, args ):
      tunnelTableId = self.getTunnelTableId( args )
      # Today,TunRibSubProtoEntry.subProtoName is used for flex-algo algoName
      # In future, if we implement similar granular preference for other tables,
      # we should handle subProtoEntry configure accordingly here.
      algoName = args.get("ALGO-NAME")
      entry = self.config.entry.get( tunnelTableId )
      preference = args.get( "PREFERENCE" )

      prefAction = ( ProtoPrefAction.prefDefault
                     if preference is None else ProtoPrefAction.prefStatic )
      prefMapping = ProtoPrefMapping( prefAction, preference or 0 )

      cost = args.get( "COST" )
      igpCostAction = ( ProtoPrefAction.prefDefault
                        if cost is None else ProtoPrefAction.prefStatic )
      igpCostPrefMapping = ProtoPrefMapping( igpCostAction, cost or 0 )

      # if entry is present and config is for proto 
      #  -> update entry.prefMapping
      # if entry is present and config is for subProto (e.g. flexAlgo algorithm )
      #  -> update subProtoEntry.prefMapping
      # if entry is not present and config is for proto 
      #  -> create entry with given prefMapping
      # if entry is not present and config is for subProto
      #  -> create entry with default prefMapping and create subProtoEntry with
      #     given prefMapping
      if not entry:
         if algoName:
            defPrefMapping = ProtoPrefMapping( ProtoPrefAction.prefDefault, 0 )
            entry = self.config.entry.newMember( tunnelTableId, defPrefMapping,
                                                 igpCostPrefMapping )
         else:
            entry = self.config.entry.newMember( tunnelTableId, prefMapping,
                                                 igpCostPrefMapping )
      if algoName:
         if entry.tunRibSubProtoEntries.get( algoName ):
            entry = entry.tunRibSubProtoEntries.get( algoName ) 
         else:
            entry = entry.tunRibSubProtoEntries.newMember( algoName, prefMapping,
                                                           igpCostPrefMapping )

      # this maybe a redundant update if we have just created but should be ok
      # as they are no-op
      entry.prefMapping = prefMapping
      entry.igpCostPrefMapping = igpCostPrefMapping

   def defaultEntryHandler( self, args ):
      entry = self.config.entry.get( self.getTunnelTableId( args ) )
      algoName = args.get("ALGO-NAME")
      if not entry: # no table pref configured.
         return
      elif algoName: # delete subProtoPref
         del entry.tunRibSubProtoEntries[ algoName ]
         return

      entry.prefMapping = ProtoPrefMapping( ProtoPrefAction.prefDefault, 0 )
      entry.igpCostPrefMapping = ProtoPrefMapping( ProtoPrefAction.prefDefault, 0 )

class CustomTunnelRibMode( TunnelRibMode, BasicCli.ConfigModeBase ):
   def deleteEntry( self, args ):
      tunnelTableId = self.getTunnelTableId( args )
      del self.config.entry[ tunnelTableId ]

class SystemTunnelRibMode( TunnelRibMode, BasicCli.ConfigModeBase ):
   pass

class SystemColoredTunnelRibMode( TunnelRibMode, BasicCli.ConfigModeBase ):
   pass

# -----------------------------------------------------------------------------------
#      config commands
# -----------------------------------------------------------------------------------
def getCustomTunnelRibNames( mode ):
   ctrNames = list( cliConfig.config )
   ctrNames.remove( systemTunnelRibName )
   ctrNames.remove( systemColoredTunnelRibName )
   ctrNames.remove( systemTunnelingLdpTunnelRibName )
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      ctrNames.remove( systemColoredTunnelingLdpTunnelRibName )
   if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
      ctrNames.remove( systemColoredIgpShortcutTunnelRibName )
   ctrNames.remove( systemIgpShortcutTunnelRibName )
   return ctrNames

class TunnelRibNameMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches custom tunnel rib names that are not ip address or
   ipv6 address or brief keywords. This is to avoid confusion between
   'show tunnel-rib [tep] brief', 'show tunnel-rib [trib-name] brief' and
   'show tunnel-rib brief'. With this restriction custom tunnel ribs cannot be named
   as brief or any other case combination of "brief" say "BRIEF", "bRIEF" etc or a
   valid ip or ipv6 address.
   '''
   def __init__( self, helpdesc='Name of the tunnel RIB',
                 excludeSystemSpecialTunnelRibs=True,
                 **kargs ):
      # pylint: disable-next=super-with-arguments
      super( TunnelRibNameMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.expectedPatternRe_ = re.compile( r"(?:^([a-zA-Z0-9:\._-]{1,100})$)" )
      # Excluded keywords
      self.briefRe_ = re.compile( r"(?:^(brief)$)", re.IGNORECASE )
      self.coloredRe_ = re.compile( r"(?:^(colored)$)", re.IGNORECASE )
      self.summaryRe_ = re.compile( r"(?:^(summary)$)", re.IGNORECASE )
      self.excludedRegs = [ self.briefRe_, self.coloredRe_, self.summaryRe_ ]
      self.excludeSystemSpecialTunnelRibs = excludeSystemSpecialTunnelRibs

   def match( self, mode, context, token ):
      # First check that the token contains valid characters and expected length
      if not self.expectedPatternRe_.match( token ):
         return CliParserCommon.noMatch

      # see if the token matches to brief keyword
      for excludedRe in self.excludedRegs:
         if excludedRe.match( token ):
            return CliParserCommon.noMatch

      excludeTunnelRibs = [ systemColoredTunnelRibName,
                            systemTunnelingLdpTunnelRibName,
                            systemIgpShortcutTunnelRibName ]
      if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
         excludeTunnelRibs.append( systemColoredTunnelingLdpTunnelRibName )

      if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
         excludeTunnelRibs.append( systemColoredIgpShortcutTunnelRibName )

      if self.excludeSystemSpecialTunnelRibs and token in excludeTunnelRibs:
         return CliParserCommon.noMatch

      # see if the token matches to ipv4 address or ipv6 address regex
      ipv4Match = Arnet.IpAddrCompiledRe.match( token )
      ipv6Match = Arnet.Ip6AddrCompiledRe.match( token )
      if ( ipv4Match is None or ipv4Match.group( 0 ) != token ) and \
         ( ipv6Match is None or ipv6Match.group( 0 ) != token ):
         # It doesn't match with either ipv4 address or ipv6 address regex. So it's a
         # valid tunnel rib name token
         return CliParserCommon.MatchResult( token, token )

      # token matches ipv4 or ipv6 address regex but IpAddrCompiledRe,
      # Ip6AddrCompiledRe are bit lenient in matching the token. So, try to get a
      # IpGenAddr from the token
      try:
         _ = Arnet.IpGenAddr( token )
      except ( IndexError, ValueError ):
         return CliParserCommon.MatchResult( token, token )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      if token == '':
         return [ CliParser.Completion( "TUNNEL_NAME", self.helpdesc_, False ) ]
      # completions are case-sensitive
      return [ CliParser.Completion( k, k ) for k in getCustomTunnelRibNames( mode )
               if k.startswith( token ) ]

class TunnelRibShowNameMatcher( TunnelRibNameMatcher ):
   '''
   Same purpose as TunnelRibNameMatcher, but to avoid upgrade issues after we added
   new exceptions (after the initial release, like the 'multicast' keyword), we
   need a matcher to that does not block possibly already existing ribs with those
   new exceptions as names
   '''
   def __init__( self, helpdesc='Name of the tunnel RIB',
                 excludeSystemSpecialTunnelRibs=True,
                 **kargs ):
      # pylint: disable-next=super-with-arguments
      super( TunnelRibShowNameMatcher, self ).__init__( helpdesc=helpdesc,
                                                            **kargs )
      self.multicastRe_ = re.compile( r"(?:^(multicast)$)", re.IGNORECASE )
      self.excludedRegs.append( self.multicastRe_ )
#----------------------------------------------
# [no] tunnel-ribs
#----------------------------------------------
def gotoTunnelRibsMode( mode, args ):
   childMode = mode.childMode( TunnelRibsMode )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelRibs( mode, args ):
   systemTunnelRibs = [ systemTunnelRibName, systemColoredTunnelRibName,
                        systemTunnelingLdpTunnelRibName,
                        systemIgpShortcutTunnelRibName ]
   if Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      systemTunnelRibs.append( systemColoredTunnelingLdpTunnelRibName )
   if Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
      systemTunnelRibs.append( systemColoredIgpShortcutTunnelRibName )
   for tunnelRibName, config in cliConfig.config.items():
      # system tunnel RIBs shouldn't be removed but entries should've default values.
      if tunnelRibName in systemTunnelRibs:
         for entry in config.entry.values():
            defaultPrefMapping = ProtoPrefMapping( ProtoPrefAction.prefDefault, 0 )
            entry.prefMapping = defaultPrefMapping
            entry.igpCostPrefMapping = defaultPrefMapping
            for subProtoEntry in entry.tunRibSubProtoEntries:
               del entry.tunRibSubProtoEntries[ subProtoEntry ]
      else:
         del cliConfig.config[ tunnelRibName ]

#----------------------------------------------
# [no] tunnel-rib <name>
#----------------------------------------------
def gotoTunnelRibMode( mode, args ):
   tunnelRibName = args.get( "TUNNEL_NAME" )
   if systemTunnelRibName in args:
      childMode = mode.childMode( SystemTunnelRibMode, 
                                  tunnelRibName=systemTunnelRibName )
      mode.session_.gotoChildMode( childMode )
   elif systemColoredTunnelRibName in args:
      childMode = mode.childMode( SystemColoredTunnelRibMode, 
                                  tunnelRibName=systemColoredTunnelRibName )
      mode.session_.gotoChildMode( childMode )
   elif systemTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                     systemTunnelingLdpTunnelRibName )
   elif Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled() and \
        systemColoredTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemColoredTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                        systemColoredTunnelingLdpTunnelRibName )
   elif Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled() and \
         systemColoredIgpShortcutTunnelRibName in args or \
         tunnelRibName == systemColoredIgpShortcutTunnelRibName:
      mode.addError( "%s is system-defined, can not be modified" %
                        systemColoredIgpShortcutTunnelRibName )
   elif systemIgpShortcutTunnelRibName in args or \
        tunnelRibName == systemIgpShortcutTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                     systemIgpShortcutTunnelRibName )
   else:
      childMode = mode.childMode( CustomTunnelRibMode, tunnelRibName=tunnelRibName )
      mode.session_.gotoChildMode( childMode )

def deleteTunnelRib( mode, args ):
   tunnelRibName = args.get( "TUNNEL_NAME" )
   if systemTunnelRibName in args:
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemTunnelRibName )
   elif systemColoredTunnelRibName in args:
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemColoredTunnelRibName )
   elif systemTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemTunnelingLdpTunnelRibName )
   elif Toggles.IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled() and \
        systemColoredTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemColoredTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                        systemColoredTunnelingLdpTunnelRibName )
   elif Toggles.IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled() and \
         systemColoredIgpShortcutTunnelRibName in args or \
         tunnelRibName == systemColoredIgpShortcutTunnelRibName:
      mode.added( "%s is system-defined, can not be deleted" %
                        systemColoredIgpShortcutTunnelRibName )
   elif systemIgpShortcutTunnelRibName in args or \
        tunnelRibName == systemIgpShortcutTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemIgpShortcutTunnelRibName )
   elif tunnelRibName in cliConfig.config:
      del cliConfig.config[ tunnelRibName ]

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, "tunnel/tunnelRibs/config",
                                  "Tunnel::TunnelTable::TunnelRibConfigDir", "wi" )
