#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.TunnelRibCli as TunnelRibCli
import BasicCliModes
import Toggles.TunnelToggleLib

#--------------------------------------------------------------------------------
# [ no | default ] tunnel-ribs
#--------------------------------------------------------------------------------
class TunnelRibsCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-ribs'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-ribs': 'Tunnel RIBs configuration',
   }

   handler = TunnelRibCli.gotoTunnelRibsMode
   noOrDefaultHandler = TunnelRibCli.deleteTunnelRibs

BasicCliModes.GlobalConfigMode.addCommandClass( TunnelRibsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel-rib 
#     ( system-tunnel-rib
#     | ( colored system-colored-tunnel-rib )
#     | TUNNEL_NAME )
#--------------------------------------------------------------------------------
class TunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tunnel-rib '
              '( system-tunnel-rib '
              '| ( colored system-colored-tunnel-rib )'
              '| TUNNEL_NAME )' )
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-rib': 'Configure a tunnel RIB',
      'system-tunnel-rib': 'The system tunnel RIB',
      'colored': 'Colored tunnel RIB',
      'system-colored-tunnel-rib': 'The system colored tunnel RIB',
      'TUNNEL_NAME': TunnelRibCli.TunnelRibNameMatcher(
         helpdesc='Name of the tunnel RIB' ),
   }
   
   handler = TunnelRibCli.gotoTunnelRibMode
   noOrDefaultHandler = TunnelRibCli.deleteTunnelRib

TunnelRibCli.TunnelRibsMode.addCommandClass( TunnelRibCmd )

matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='BGP tunnel' )
matcherCostValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The IGP cost value' )
flexAlgoKw = 'flex-algo'
matcherFlexAlgo = CliMatcher.KeywordMatcher( flexAlgoKw,
      helpdesc='Flex algo tunnel' )
matcherIgpCost = CliMatcher.KeywordMatcher( 'igp-cost',
      helpdesc='Configure the preference to influence IGP cost '
               'of next-hops resolving over tunnels' )
matcherIgpPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='Configure the preference attribute of igp cost' )
matcherIsis = CliMatcher.KeywordMatcher( 'isis',
      helpdesc='IS-IS tunnel' )
matcherLabeledUnicast = CliMatcher.KeywordMatcher( 'labeled-unicast',
      helpdesc='BGP labeled unicast tunnel' )
matcherLdp = CliMatcher.KeywordMatcher( 'ldp',
      helpdesc='LDP tunnel' )
matcherNexthopGroup = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='Nexthop group tunnel' )
matcherPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='Configure the preference to influence tunnel RIB election' )
matcherPreferenceValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The preference value' )
matcherSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='IS-IS segment routing tunnel' )
matcherOspfSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='Ospf segment routing tunnel' )
matcherTeSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='Segment routing tunnel' )
matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Segment routing policy tunnel' )
matcherSourceProtocol = CliMatcher.KeywordMatcher( 'source-protocol',
      helpdesc='Configure the tunnel source' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Static tunnel' )
matcherRsvpLer = CliMatcher.KeywordMatcher( 'rsvp-ler',
      helpdesc='RSVP LER tunnel' )
matcherOspf = CliMatcher.KeywordMatcher( 'ospf',
      helpdesc='OSPF tunnel' )

namePattern = r'^[^\d\s|>&;`$()\\][^\s|>&;`$()\\]+'
matcherAlgoName = CliCommand.Node( matcher=CliMatcher.PatternMatcher(
   namePattern, helpname='ALGO-NAME',
   helpdesc='Algorithm name not starting with a number' ) )

class SourceProtocolCmd( CliCommand.CliCommandClass ):
   _prefArgs = ( '[ ( preference PREFERENCE [ igp-cost igp-preference COST ] ) '
                 '| ( igp-cost igp-preference COST [ preference PREFERENCE ] ) ]' )

   syntax = ( 'source-protocol '
              '( ldp '
              '| rsvp-ler '
              '| static '
              '| ( bgp labeled-unicast ) '
              '| ( isis segment-routing ) '
              '| nexthop-group ' )

   data = {
         'source-protocol': matcherSourceProtocol,
         'ldp': matcherLdp,
         'rsvp-ler': matcherRsvpLer,
         'static': matcherStatic,
         'bgp': matcherBgp,
         'labeled-unicast': matcherLabeledUnicast,
         'isis': matcherIsis,
         'segment-routing': matcherSegmentRouting,
         flexAlgoKw : matcherFlexAlgo,
         'nexthop-group': matcherNexthopGroup,
         'preference': matcherPreference,
         'PREFERENCE': matcherPreferenceValue,
         'igp-cost': matcherIgpCost,
         'igp-preference': matcherIgpPreference,
         'COST': matcherCostValue,
         'ALGO-NAME' : matcherAlgoName,
      }

   if Toggles.TunnelToggleLib.toggleIsisFlexAlgoTunPrefEnabled():
      syntax += '| ( isis flex-algo [ ALGO-NAME ] )'
   else:
      syntax += '| ( isis flex-algo )'

   syntax += ' ) %s' % _prefArgs # pylint: disable=consider-using-f-string
   defaultSyntax = syntax + ' ...'


#--------------------------------------------------------------------------------
# [ no | default ] source-protocol
#                             ( ldp
#                             | rsvp-ler
#                             | static
#                             | ( bgp labeled-unicast )
#                             | ( isis segment-routing )
#                             | ( isis flex-algo [ ALGO-NAME ])
#                             | nexthop-group )
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] )
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolCmdForCustomTunnelRibMode( SourceProtocolCmd ):
   noSyntax = SourceProtocolCmd.defaultSyntax

   # This inherited class needs data dict to be separately defined because of 
   # BUG380538. The fix for BUG380538 should move this dict to the parent class
   handler = TunnelRibCli.CustomTunnelRibMode.handleEntry
   noOrDefaultHandler = TunnelRibCli.CustomTunnelRibMode.deleteEntry

#--------------------------------------------------------------------------------
# [ default ] source-protocol
#                             ( ldp
#                             | rsvp-ler
#                             | static
#                             | ( bgp labeled-unicast )
#                             | ( isis segment-routing )
#                             | ( isis flex-algo [ALGO-NAME] )
#                             | nexthop-group )
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] )
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolCmdForSystemTunnelRibMode( SourceProtocolCmd ):
   # noSyntax is not present for this command as we cannot remove
   # source-protocols from contributing to the system-tunnel-rib

   # This inherited class needs data dict to be separately defined because of
   # BUG380538. The fix for BUG380538 should move this dict to the parent class

   data = {
         'source-protocol': matcherSourceProtocol,
         'ldp': matcherLdp,
         'rsvp-ler': matcherRsvpLer,
         'static': matcherStatic,
         'bgp': matcherBgp,
         'labeled-unicast': matcherLabeledUnicast,
         'isis': matcherIsis,
         'segment-routing': matcherSegmentRouting,
         flexAlgoKw : matcherFlexAlgo,
         'nexthop-group': matcherNexthopGroup,
         'preference': matcherPreference,
         'PREFERENCE': matcherPreferenceValue,
         'igp-cost': matcherIgpCost,
         'igp-preference': matcherIgpPreference,
         'COST': matcherCostValue,
         'ALGO-NAME' : matcherAlgoName,
      }

   handler = TunnelRibCli.SystemTunnelRibMode.handleEntry
   defaultHandler = TunnelRibCli.SystemTunnelRibMode.defaultEntryHandler

#--------------------------------------------------------------------------------
# [ default ] source-protocol 
#                             ( rsvp-ler 
#                             | ( segment-routing policy )
#                             | ( isis flex-algo ) )
#                                [ ( preference PREFERENCE ) ]
#--------------------------------------------------------------------------------
class SourceProtocolCmdForSystemColoredTunnelRibMode( CliCommand.CliCommandClass ):
   # noSyntax is not present for this command as we cannot remove
   # source-protocols from contributing to the system-tunnel-rib

   syntax = ( 'source-protocol '
              '( rsvp-ler '
              '| ( segment-routing policy ) ' )

   if Toggles.TunnelToggleLib.toggleIsisFlexAlgoTunPrefEnabled():
      syntax += '| ( isis flex-algo [ ALGO-NAME ] )'
   else:
      syntax += '| ( isis flex-algo )'

   data = {
      'source-protocol': matcherSourceProtocol,
      'rsvp-ler': matcherRsvpLer,
      'segment-routing': matcherTeSegmentRouting,
      'policy': matcherPolicy,
      'isis': matcherIsis,
      flexAlgoKw: matcherFlexAlgo,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceValue,
      'ALGO-NAME' : matcherAlgoName,
   }

   syntax += ') [ preference PREFERENCE ]'
   defaultSyntax = syntax
   handler = TunnelRibCli.SystemColoredTunnelRibMode.handleEntry
   defaultHandler = TunnelRibCli.SystemTunnelRibMode.defaultEntryHandler

TunnelRibCli.CustomTunnelRibMode.addCommandClass( 
           SourceProtocolCmdForCustomTunnelRibMode )
TunnelRibCli.SystemTunnelRibMode.addCommandClass( 
           SourceProtocolCmdForSystemTunnelRibMode )
TunnelRibCli.SystemColoredTunnelRibMode.addCommandClass( 
   SourceProtocolCmdForSystemColoredTunnelRibMode )

class SourceProtocolOspfCmd( CliCommand.CliCommandClass ):
   _prefArgs = ( '[ ( preference PREFERENCE [ igp-cost igp-preference COST ] ) '
                 '| ( igp-cost igp-preference COST [ preference PREFERENCE ] ) ]' )
   syntax = ( 'source-protocol ' 
              ' ( ( ospf segment-routing ) ' )
   syntax += ' ) %s' % _prefArgs # pylint: disable=consider-using-f-string
   defaultSyntax = syntax + ' ...'
   
   data = {
      'source-protocol': matcherSourceProtocol,
      'ospf': matcherOspf,
      'segment-routing': matcherOspfSegmentRouting,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceValue,
      'igp-cost': matcherIgpCost,
      'igp-preference': matcherIgpPreference,
      'COST': matcherCostValue,
   }

#--------------------------------------------------------------------------------
# [ no | default ] source-protocol 
#                             | ( ospf segment-routing )
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] ) 
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolOspfCmdForCustomTunnelRibMode( SourceProtocolOspfCmd ):
   noSyntax = SourceProtocolOspfCmd.defaultSyntax
   # This inherited class needs data dict to be separately defined because of 
   # BUG380538. The fix for BUG380538 should move this dict to the parent class
   handler = TunnelRibCli.CustomTunnelRibMode.handleEntry
   noOrDefaultHandler = TunnelRibCli.CustomTunnelRibMode.deleteEntry

#--------------------------------------------------------------------------------
# [ default ] source-protocol 
#                             ( ( Ospf segment-routing ) )
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] ) 
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolOspfCmdForSystemTunnelRibMode( SourceProtocolOspfCmd ):
   # noSyntax is not present for this command as we cannot remove
   # source-protocols from contributing to the system-tunnel-rib
   # This inherited class needs data dict to be separately defined because of 
   # BUG380538. The fix for BUG380538 should move this dict to the parent class
   data = {
      'source-protocol': matcherSourceProtocol,
      'ospf': matcherOspf,
      'segment-routing': matcherOspfSegmentRouting,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceValue,
      'igp-cost': matcherIgpCost,
      'igp-preference': matcherIgpPreference,
      'COST': matcherCostValue,
   }
   handler = TunnelRibCli.SystemTunnelRibMode.handleEntry
   defaultHandler = TunnelRibCli.SystemTunnelRibMode.defaultEntryHandler

if Toggles.TunnelToggleLib.toggleOspfSegmentRoutingTunnelEnabled():
   TunnelRibCli.CustomTunnelRibMode.addCommandClass( 
              SourceProtocolOspfCmdForCustomTunnelRibMode )
   TunnelRibCli.SystemTunnelRibMode.addCommandClass( 
              SourceProtocolOspfCmdForSystemTunnelRibMode )
