# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.TwampLight import (
    MonitorTwampMode,
    TwampLightMode,
    TwampLightSenderProfileMode,
    TwampLightSenderDefaultsMode,
    TwampLightReflectorDefaultsMode
)
import ConfigMount
import CliParser
import CliToken.Monitor
import LazyMount

twampLightConfig = None
twampLightHwStatus = None
TWAMP_LIGHT_PROFILE_REFLECTOR = 0
TWAMP_LIGHT_PROFILE_SENDER = 1

def twampLightSupportedGuard( mode, token ):
   if twampLightHwStatus.twampLightSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# (config)# monitor twamp
# -----------------------------------------------------------------------------------
class MonitorTwampModeCmd( CliCommand.CliCommandClass ):
   syntax = "monitor twamp"
   noOrDefaultSyntax = "monitor twamp"
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'twamp': CliCommand.guardedKeyword( 'twamp',
                                          helpdesc='Configure TWAMP',
                                          guard=twampLightSupportedGuard )
   }
   handler = "TwampLightCliHandler.handlerMonitorTwampModeCmd"
   noOrDefaultHandler = "TwampLightCliHandler.noOrDefaultHandlerMonitorTwampModeCmd"

BasicCli.GlobalConfigMode.addCommandClass( MonitorTwampModeCmd )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] twamp-light
# -----------------------------------------------------------------------------------
class TwampLightModeCmd( CliCommand.CliCommandClass ):
   syntax = "twamp-light"
   noOrDefaultSyntax = "twamp-light"
   data = {
      'twamp-light': 'Configure TWAMP light'
   }
   handler = "TwampLightCliHandler.handlerTwampLightModeCmd"
   noOrDefaultHandler = "TwampLightCliHandler.noOrDefaultHandlerTwampLightModeCmd"

MonitorTwampMode.addCommandClass( TwampLightModeCmd )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] reflector defaults
# -----------------------------------------------------------------------------------
class TwampLightReflectorDefaultsModeCmd( CliCommand.CliCommandClass ):
   syntax = "reflector defaults"
   noOrDefaultSyntax = "reflector defaults"
   data = {
      'reflector': 'Configure TWAMP light reflector',
      'defaults': 'Configure TWAMP light reflector defaults',
      }
   handler = "TwampLightCliHandler.handlerTwampLightReflectorDefaultsModeCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerTwampLightReflectorDefaultsModeCmd"

TwampLightMode.addCommandClass( TwampLightReflectorDefaultsModeCmd )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] sender defaults
# -----------------------------------------------------------------------------------
class TwampLightSenderDefaultsModeCmd( CliCommand.CliCommandClass ):
   syntax = "sender defaults"
   noOrDefaultSyntax = "sender defaults"
   data = {
      'sender': 'Configure TWAMP light sender',
      'defaults': 'Configure TWAMP light sender defaults',
      }
   handler = "TwampLightCliHandler.handlerTwampLightSenderDefaultsModeCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerTwampLightSenderDefaultsModeCmd"

TwampLightMode.addCommandClass( TwampLightSenderDefaultsModeCmd )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] sender profile NAME
# -----------------------------------------------------------------------------------
senderNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: twampLightConfig.senderProfile,
   'Profile name' )

class TwampLightSenderProfileModeCmd( CliCommand.CliCommandClass ):
   syntax = "sender profile NAME"
   noOrDefaultSyntax = 'sender profile NAME ...'
   data = {
      'sender': 'Configure TWAMP light sender',
      'profile': 'Configure TWAMP light sender profile',
      'NAME': senderNameMatcher,
      }
   handler = "TwampLightCliHandler.handlerTwampLightSenderProfileModeCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerTwampLightSenderProfileModeCmd"

TwampLightMode.addCommandClass( TwampLightSenderProfileModeCmd )

# -----------------------------------------------------------------------------------
# (twamp-light-reflector-defaults)# listen port PORT
# -----------------------------------------------------------------------------------
class ReflectorListenPortCmd( CliCommand.CliCommandClass ):
   syntax = "listen port PORT"
   noOrDefaultSyntax = "listen port ..."
   data = {
      'listen': 'Reflector listen configuration',
      'port': 'Port on which TWAMP light is listening for incoming packets',
      'PORT': CliMatcher.IntegerMatcher(
          1, 65535, helpdesc="Listen port number" ),
   }
   handler = "TwampLightCliHandler.handlerReflectorListenPortCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerReflectorListenPortCmd"

TwampLightReflectorDefaultsMode.addCommandClass( ReflectorListenPortCmd )

# -----------------------------------------------------------------------------------
# (twamp-light-sender-defaults)# source port PORT
# -----------------------------------------------------------------------------------
class SenderSourcePortCmd( CliCommand.CliCommandClass ):
   syntax = "source port PORT"
   noOrDefaultSyntax = "source port ..."
   data = {
      'source': 'Source configuration',
      'port': 'Source port configuration',
      'PORT': CliMatcher.IntegerMatcher(
          1, 65535, helpdesc='Source port number' ),
   }
   handler = "TwampLightCliHandler.handlerSenderSourcePortCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerSenderSourcePortCmd"

TwampLightSenderDefaultsMode.addCommandClass( SenderSourcePortCmd )

# -----------------------------------------------------------------------------------
# (twamp-light-sender-defaults)# destination port PORT
# -----------------------------------------------------------------------------------
class SenderDestinationPortCmd( CliCommand.CliCommandClass ):
   syntax = "destination port PORT"
   noOrDefaultSyntax = "destination port ..."
   data = {
      'destination': 'Destination configuration',
      'port': 'Destination port configuration',
      'PORT': CliMatcher.IntegerMatcher(
         1, 65535, helpdesc='Destination port number' ),
   }
   handler = "TwampLightCliHandler.handlerSenderDestinationPortCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerSenderDestinationPortCmd"

TwampLightSenderDefaultsMode.addCommandClass( SenderDestinationPortCmd )

# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# measurement interval INTERVAL
# -----------------------------------------------------------------------------------
class SenderProfileIntervalPortCmd( CliCommand.CliCommandClass ):
   syntax = "measurement interval INTERVAL seconds"
   noOrDefaultSyntax = "measurement interval ..."
   data = {
      'measurement': 'TWAMP light measurement configuration',
      'interval': 'TWAMP light measurement interval',
      'INTERVAL': CliMatcher.IntegerMatcher(
         1, 255, helpdesc='Measurement interval duration in seconds' ),
      'seconds': 'Seconds',
   }
   handler = "TwampLightCliHandler.handlerSenderProfileIntervalCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerSenderProfileIntervalCmd"

TwampLightSenderProfileMode.addCommandClass( SenderProfileIntervalPortCmd )

# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# measurement samples NUMBER
# -----------------------------------------------------------------------------------
class SenderProfileSamplesPortCmd( CliCommand.CliCommandClass ):
   syntax = "measurement samples NUMBER"
   noOrDefaultSyntax = "measurement samples ..."
   data = {
      'measurement': 'TWAMP light measurement configuration',
      'samples': 'Samples used for TWAMP light measurement',
      'NUMBER': CliMatcher.IntegerMatcher(
         1, 65535,
         helpdesc='Number of samples used to calculate TWAMP light metrics' ),
   }
   handler = "TwampLightCliHandler.handlerSenderProfileSamplesCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerSenderProfileSamplesCmd"

TwampLightSenderProfileMode.addCommandClass(
      SenderProfileSamplesPortCmd )

microsecondsMatcher = CliMatcher.KeywordMatcher( 'microseconds',
                  helpdesc='Microseconds' )
# -----------------------------------------------------------------------------------
# (config-twamp-light-sender-profile-NAME)# significance SIGNIFICANCE microseconds
#                                           offset OFFSET microseconds
# -----------------------------------------------------------------------------------
class SenderProfileSignificanceOffsetCmd( CliCommand.CliCommandClass ):
   syntax = "significance SIGNIFICANCE SIG_UNIT offset OFFSET OFF_UNIT"
   noOrDefaultSyntax = "significance ..."
   data = {
      'significance': 'Significance used to round up calculated TWAMP light'
                      ' delay statistics',
      'SIGNIFICANCE': CliMatcher.IntegerMatcher(
         1, 1000000, helpdesc='Significance value in microseconds' ),
      'SIG_UNIT': microsecondsMatcher,
      'offset': 'Offset used to round up calculated TWAMP light delay statistics',
      'OFFSET': CliMatcher.IntegerMatcher(
         1, 1000000, helpdesc='Offset value' ),
      'OFF_UNIT': microsecondsMatcher
   }
   handler = "TwampLightCliHandler.handlerSenderProfileSignificanceOffsetCmd"
   noOrDefaultHandler =\
      "TwampLightCliHandler.noOrDefaultHandlerSenderProfileSignificanceOffsetCmd"

TwampLightSenderProfileMode.addCommandClass(
      SenderProfileSignificanceOffsetCmd )

def Plugin( entityManager ):
   global twampLightConfig
   global twampLightHwStatus

   twampLightConfig = ConfigMount.mount(
      entityManager, 'monitor/twamp/twampLight/config',
      'Twamp::Light::Config', 'w' )
   twampLightHwStatus = LazyMount.mount(
      entityManager, 'monitor/twamp/twampLight/hwstatus',
      'Twamp::Light::TwampLightHwStatus', 'r' )
