# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import Tac

from CliMode.Upnp import UpnpMode
from CliPlugin import IntfCli
from CliPlugin.UpnpCliLib import getUpnpConfig, upnpSupportedGuard

nodeUpnp = CliCommand.guardedKeyword( 'upnp',
      helpdesc='Configure UPNP service',
      guard=upnpSupportedGuard )

# When upnp config is removed, we need to clear config from the submodes
# as well.  The other submodes can implement clearConfig method
# and register themselves via UpnpCli.registerDependentClass
class UpnpDependentBase:
   def clearConfig( self ):
      pass

UpnpCli = IntfCli.DependentClassRegistry()

def hexAtoI( hexStr ):
   return int( hexStr, 16 )

class UpnpConfigMode( UpnpMode, BasicCli.ConfigModeBase ):
   name = "UPNP configuration"

   def __init__( self, parent, session ):
      UpnpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# [ no ] upnp
class UpnpCmd( CliCommand.CliCommandClass ):
   syntax = '''upnp'''
   noOrDefaultSyntax = syntax
   data = {
      'upnp': nodeUpnp,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( UpnpConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      UpnpCmd.clearConfig_()

   @staticmethod
   def clearConfig_():
      # Clear global UPNP config
      upnpConfig = getUpnpConfig()
      upnpConfig.uuid = Tac.Value( "Arnet::Uuid" )

      # Clear config from dependent modes
      for cls in UpnpCli:
         cls().clearConfig()

BasicCli.GlobalConfigMode.addCommandClass( UpnpCmd )

# [no] uuid <UUID>
class UuidCmd( CliCommand.CliCommandClass ):
   syntax = '''uuid VALUE'''
   noOrDefaultSyntax = '''uuid ...'''
   data = {
      'uuid': 'Universally Unique IDentifier',
      'VALUE': CliMatcher.PatternMatcher(
         '^[0-9a-fA-F]{1,8}-[0-9a-fA-F]{1,4}-[0-9a-fA-F]'
         '{1,4}-[0-9a-fA-F]{1,4}-[0-9a-fA-F]{1,12}$',
         helpname='A-B-C-D-E', helpdesc='UUID for this device' )
   }

   @staticmethod
   def handler( mode, args ):
      upnpConfig = getUpnpConfig()
      uuid = args.get( 'VALUE', "" )
      parts = uuid.split( '-' )
      w0 = hexAtoI( parts[ 0 ] )
      w1 = ( hexAtoI( parts[ 1 ] ) << 16 ) | hexAtoI( parts[ 2 ] )
      w2 = ( hexAtoI( parts[ 3 ] ) << 16 ) | ( hexAtoI( parts[ 4 ] ) >> 32 )
      w3 = hexAtoI( parts[ 4 ] ) & 0xFFFFFFFF
      upnpConfig.uuid = Tac.Value( "Arnet::Uuid", w0, w1, w2, w3 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      upnpConfig = getUpnpConfig()
      upnpConfig.uuid = Tac.Value( "Arnet::Uuid" )

UpnpConfigMode.addCommandClass( UuidCmd )
