# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import time

import BasicCli
import CliMatcher
import EosVersion
import ShowCommand
import Tac
import TacSigint
import Tracing
import LazyMount
import OnieVersionLib

from .VersionModel import ( # pylint: disable=relative-beyond-top-level
   Version,
   Component,
   Details,
   Firmware,
   Package,
   Licenses
)
from PlatformArchLogger import getRequiredArchs, checkArchitectureCompatibility

__defaultTraceHandle__ = Tracing.Handle( 'VersionCli' )
t0 = Tracing.trace0

hwEpochStatus_ = None
onieStatus_ = None
hwEntMib_ = None
sysMacAddrConfig_ = None

versionKwMatcher = CliMatcher.KeywordMatcher(
   'version',
   helpdesc='Software and hardware versions' )

#-----------------------------------------------------------------------------------
# show version [ detail ]
#-----------------------------------------------------------------------------------
# We do not count meminfo[ 'Cached' ] as free memory because this number
# includes unreclaimable space from tmpfs/ramfs filesystems
def _getMemUsage():
   # pylint: disable-next=consider-using-with
   meminfo = open( '/proc/meminfo' ).read().strip().split( '\n' )
   meminfo = ( m.split() for m in meminfo )
   meminfo = { m[ 0 ][ : -1 ]: int( m[ 1 ] ) for m in meminfo }
   if "MemAvailable" not in meminfo:
      return ( meminfo[ "MemTotal" ], meminfo[ "MemFree" ] )
   else:
      return ( meminfo[ "MemTotal" ], meminfo[ "MemAvailable" ] )

def _getPackages():
   resultPackage = {}
   packageInfo = Tac.run( [ 'rpm', '-qa', '--qf', '%-20{N} %-15{V} %{R}\n' ],
                          stdout=Tac.CAPTURE )
   packages = packageInfo.strip().split( '\n' )
   packages.sort()
   packages = ( package.split() for package in packages )
   for package in packages:
      if not package:
         continue
      p, v, r = package
      resultPackage[ p ] = Package( version=v, release=r )

   return resultPackage

def _getNewComponent( name, version ):
   return Component( name=name, version=version )

def _getChassisComponents( entityMibRoot ):
   components = []
   cardMibs = [ x.card for x in entityMibRoot.cardSlot.values( ) if x.card ]
   for cm in sorted( cardMibs, key=lambda x: ( x.tag, int( x.label ) ) ):
      prefix = f"{cm.tag}{cm.label}-"
      if cm.firmwareRev:
         name = prefix + "Aboot"
         components.append( _getNewComponent( name, cm.firmwareRev ) )
      for chip in sorted( cm.chip.values(), key=lambda x: x.tag ):
         if chip.firmwareRev:
            name = prefix + chip.tag
            components.append( _getNewComponent( name, chip.firmwareRev ) )

   return components

def _getFixedSystemComponents( entityMibRoot ):
   components = []
   components.append( _getNewComponent( "Aboot", entityMibRoot.firmwareRev ) )
   for chip in sorted( entityMibRoot.chip.values(), key=lambda x: x.tag ):
      if chip.firmwareRev:
         components.append( _getNewComponent( chip.tag, chip.firmwareRev ) )

   return components

def _getComponents( mode ):
   entityMibRoot = mode.entityManager.lookup( 'hardware/entmib' ).root
   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      # System is not yet initialized
      return ( [], "unknown" )

   if entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      return ( _getChassisComponents( entityMibRoot ), "chassis" )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      return ( _getFixedSystemComponents( entityMibRoot ), "fixedSystem" )

   return ( [], "unknown" )

def _getSystemEpoch():
   if hwEpochStatus_ and hwEpochStatus_.systemHwEpoch:
      return hwEpochStatus_.systemHwEpoch
   return "unknown"

def _getOnieVersion():
   if onieStatus_ and onieStatus_.version:
      return onieStatus_.version
   else:
      return "unknown"

def _getDetails( mode, versionInfo ):
   details = Details()
   details.packages = _getPackages()
   ( details.components, details.switchType ) = _getComponents( mode )
   deviations = versionInfo.deviations()
   if deviations:
      details.deviations = deviations
   details.systemEpoch = _getSystemEpoch()
   if OnieVersionLib.getOniePlatform():
      details.onieVersion = _getOnieVersion()

   return details

def platformSupportsArch( modelName, arch ):
   if not modelName or not arch:
      return None

   reqArchs = getRequiredArchs( modelName )
   return checkArchitectureCompatibility( arch, reqArchs, modelName,
                                          log=False )

def opt( var, default ):
   return var if var is not None else default

def showVersion( mode, args ):
   detail = 'detail' in args
   result = Version()

   # Some of these values are optional because they might not be initialized.
   # Usually this happens only on standby supervisors.
   vi = EosVersion.VersionInfo( mode.sysdbRoot )

   modelName = vi.modelName()
   arch = vi.architecture()
   archSupported = platformSupportsArch( modelName, arch )
   if archSupported is not None and not archSupported:
      arch += " (UNSUPPORTED)"

   result.mfgName = opt( vi.mfgName(), "Arista" ).replace( "Arista Networks",
                                                           "Arista" )
   result.modelName = opt( vi.modelNameExtended(), "Switch (model not available)" )
   result.hardwareRevision = opt( vi.hardwareRev(), "Not available" )
   result.serialNumber = opt( vi.serialNum(), "Not available" )
   result.version = opt( vi.version(), "Not available" )
   result.architecture = opt( arch, "(unknown)" )
   result.internalVersion = opt( vi.internalVersion(), "(unknown)" )
   result.internalBuildId = opt( vi.internalBuildId(), "(unknown)" )
   result.imageFormatVersion = opt( vi.imageFormatVersion(), "(unknown)" )
   result.imageOptimization = opt( vi.swiOptimization(), "(unknown)" )
   result.systemMacAddress = hwEntMib_.systemMacAddr 
   result.hwMacAddress = hwEntMib_.hwMacAddr 
   result.configMacAddress = sysMacAddrConfig_.macAddr
   curTime = time.time()
   if vi.isCEos():
      boot = os.stat( '/proc/1/cmdline' )
      result.bootupTimestamp = float( boot.st_ctime )
      result.uptime = curTime - result.bootupTimestamp
      result.kernelVersion = opt( os.uname()[ 2 ], '(unknown)' )
   else:
      uptime = open( "/proc/uptime" ).read() # pylint: disable=consider-using-with
      result.uptime = float( uptime.split()[ 0 ] )
      result.bootupTimestamp = curTime - result.uptime
   result.isIntlVersion = vi.isIntlVersion()
   ( result.memTotal, result.memFree ) = _getMemUsage()

   if detail:
      result.details = _getDetails( mode, vi )

   return result

class ShowVersionDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show version [ detail ]'
   data = {
            'version': versionKwMatcher,
            'detail': 'Show additional version information',
          }
   cliModel = Version
   handler = showVersion

BasicCli.addShowCommandClass( ShowVersionDetailCmd )

#-----------------------------------------------------------------------------------
# show version license
#-----------------------------------------------------------------------------------
licenseDir = '/usr/share/licenses'
docDir = '/usr/share/doc'
def showLicense( mode, args ):
   licenses = {}
   def _walk( dirname, fnames ):
      t0( "Processing", dirname, "with contents", fnames )
      for fn in fnames:
         TacSigint.check()
         # All files in licenseDir, no matter what, or
         # certain files from docDir.
         if dirname.startswith( licenseDir ) or \
               fn.startswith( "COPY" ) or fn.startswith( "LICEN" ) or \
               fn.startswith( "NOTICE" ):
            fp = dirname + "/" + fn
            if os.path.isfile( fp ):
               try:
                  with open( fp, errors="backslashreplace" ) as f:
                     licenses[ fp ] = f.read()
               except OSError:
                  # pylint: disable-next=consider-using-f-string
                  mode.addError( "Failed to display file %s" % fp )

   for d in ( licenseDir, docDir ):
      for dirpath, _, filenames in os.walk( d ):
         _walk( dirpath, filenames )

   if len( licenses ) == 0:
      mode.addError( "No software licenses found." )

   return Licenses( licenses=licenses )

class ShowVersionLicenseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show version license'
   data = {
            'version': versionKwMatcher,
            'license': 'Show software license information',
          }
   cliModel = Licenses
   handler = showLicense

BasicCli.addShowCommandClass( ShowVersionLicenseCmd )

#-----------------------------------------------------------------------------------
# show version firmware
#-----------------------------------------------------------------------------------
def showFirmware( mode, args ):
   result = Firmware()

   ( result.components, result.switchType ) = _getComponents( mode )

   return result

class ShowVersionFirmwareCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show version firmware'
   data = {
            'version': versionKwMatcher,
            'firmware': 'Show device firmware versions'
          }
   cliModel = Firmware
   handler = showFirmware

BasicCli.addShowCommandClass( ShowVersionFirmwareCmd )

def Plugin( entityManager ):
   global hwEpochStatus_
   global onieStatus_
   global hwEntMib_
   global sysMacAddrConfig_
   hwEpochStatus_ = LazyMount.mount( entityManager,
                                     'hwEpoch/status', 'HwEpoch::Status', 'r' )
   onieStatus_ = LazyMount.mount( entityManager, 'onie/status', 'Onie::Status', 'r' )
   hwEntMib_ = LazyMount.mount( entityManager,  'hardware/entmib', 
                                          "EntityMib::Status", 'r' )
   sysMacAddrConfig_ = LazyMount.mount( entityManager, 'sys/macAddress/config',
                                        "Fru::MacAddressConfig", 'r' )
