# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.VirtualCableMode import VirtualCableMode, VirtualCablePortMode
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import MultiRangeRule
import ConfigMount
from EthIntfLib import lookupEthPhyIntfConfigDir
import Intf.IntfRange
import LazyMount
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'VirtualCable' )

#-------------------------------------------------------------------------------
# globals
#-------------------------------------------------------------------------------
configDir = None
statusDir = None
ethPhyIntfConfigSliceDir = None

#-------------------------------------------------------------------------------
# config-virtual-cable mode
#-------------------------------------------------------------------------------
class VirtualCableConfigMode( BasicCli.ConfigModeBase, VirtualCableMode ):
   name = 'Virtual-Cable Configuration'

   def __init__( self, parent, session ):
      self.config = configDir
      self.defaultTpid = 0x88a8
      VirtualCableMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setDefaultRunningTpid( self ):
      if self.config.tpid == 0:
         self.config.tpid = self.defaultTpid
      
   def setTpid( self, tpid ):
      self.config.tpid = tpid

   def clearTpid( self ):
      self.config.tpid = self.defaultTpid

   def createPort( self, intf ):
      if intf not in self.config.groups:
         self.config.groups.newMember( intf )
         #self.config.groups[ intf.name ].muxIntf = intf.name

   def deletePort( self, intf ):
      self.deleteVcableGroup( intf )
      
   def deleteVcableGroup( self, grp ):
      if grp not in self.config.groups:
         return
      del self.config.groups[ grp ]

   def deletePortAll( self ):
      for key in self.config.groups:
         self.deleteVcableGroup( key )
   
#-------------------------------------------------------------------------------
# config-virtual-cable-mux-port mode
#-------------------------------------------------------------------------------
class VirtualCablePortConfigMode( BasicCli.ConfigModeBase, VirtualCablePortMode ):
   name = 'Virtual-Cable Mux Port Configuration'

   def __init__( self, parent, session, interface ):
      self.loopback = False
      self.configDir = configDir
      self.groupDir = configDir.groups.get( interface.name )
      VirtualCablePortMode.__init__( self, interface )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setLoopbackPort( self, intfs, MUX_IDS, no=None ):
      interfaces = IntfCli.Intf.getAll( self, intfs, config=True )
      if not interfaces:
         return
      
      if no is None and len( interfaces ) != len( MUX_IDS ):
         # pylint: disable-next=consider-using-f-string
         print( 'Number of interfaces: %d, Not equal to number of mux-ids: %d.'
                % ( len( interfaces ), len( MUX_IDS ) ) )
         return
      
      if no is None:
         # Add new loopback ports
         for i in interfaces:
            vcableIntf = Tac.newInstance( 'VirtualCable::VcableIntf', i.name )
            vcableIntf.muxId = MUX_IDS.pop( 0 )
            self.groupDir.vcableIntf.addMember( vcableIntf )
      else:
         # Delete new loopback ports
         for i in interfaces:
            del self.groupDir.vcableIntf[ i.name ]
      
#--------------------------------------------------------------------------------
# [ no | default ] virtual-cable
#--------------------------------------------------------------------------------
class VirtualCableCmd( CliCommand.CliCommandClass ):
   syntax = 'virtual-cable'
   noOrDefaultSyntax = syntax
   data = {
      'virtual-cable' : 'Configure virtual cable parameters',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( VirtualCableConfigMode )
      mode.session_.gotoChildMode( childMode )
      childMode.setDefaultRunningTpid( )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( VirtualCableConfigMode )
      childMode.setTpid( 0 )
      childMode.deletePortAll()

BasicCli.GlobalConfigMode.addCommandClass( VirtualCableCmd )

#--------------------------------------------------------------------------------
# show virtual-cable
#--------------------------------------------------------------------------------
class ShowVirtualCableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show virtual-cable'
   data = {
      'virtual-cable' : 'Show virtual cable status and configuration',
   }

   @staticmethod
   def handler( mode, args ):
      Tracing.trace0( 'running show virtual-cable command' )
      mount = statusDir
      
      print( '' )
      print( 'Virtual-Cable Status ' )
      print( '----------------------------------------------------'
             '-----------------' )
      # pylint: disable-next=consider-using-f-string
      print( '%-30s : %#06x' % ( 'TPID', mount.tpid ) )
      # pylint: disable-next=consider-using-f-string
      print( '%-30s : %u' % ( 'Total Mux Groups', len( mount.groups ) ) )
      print( '' )
      print( '----------------------------------------------------'
             '-----------------' )
      if not mount.groups:
         return
      print( '| Mux-Group      | Loopback-Port  | MuxId          |'
             ' LoopbackMode   |' )
      print( '+----------------+----------------+----------------+'
             '----------------+' )
      
      groups = list( mount.groups )
      groups.sort()
      for group in groups:
         groupConfig = mount.groups[ group ]
         
         groupStr = group
         lbIntfs = list( groupConfig.vcableIntf )
         lbIntfs.sort()
         if lbIntfs == []: # pylint: disable=use-implicit-booleaness-not-comparison
            # pylint: disable-next=consider-using-f-string
            print( '| %-15s| %-15s| %-15s| %-15s|'
                   % ( groupStr, '-', '-', '-' ) )
         else:
            for lbIntf in lbIntfs:
               lbConfig = groupConfig.vcableIntf[ lbIntf ]
               ethPhyIntfConfigDir = lookupEthPhyIntfConfigDir( 
                  lbIntf, ethPhyIntfConfigSliceDir )
               ethConfig = ethPhyIntfConfigDir.intfConfig.get( lbIntf )
               lbMode = ethConfig.loopbackMode if ethConfig else '-'
               # pylint: disable-next=consider-using-f-string
               print( '| %-15s| %-15s| %6u         | %-15s|'
                      % ( groupStr, lbIntf, lbConfig.muxId, lbMode ) )
      print( '----------------------------------------------------------------'
             '-----' )

BasicCli.addShowCommandClass( ShowVirtualCableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tpid TPID
#--------------------------------------------------------------------------------
class TpidCmd( CliCommand.CliCommandClass ):
   syntax = 'tpid TPID'
   noOrDefaultSyntax = 'tpid ...'
   data = {
      'tpid' : 'Configure tag protocol identifier (TPID) value',
      'TPID' : CliMatcher.IntegerMatcher( 1, 65534,
         helpdesc='Value for the tag protocol identifier (TPID)' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setTpid( args[ 'TPID' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.clearTpid()

VirtualCableConfigMode.addCommandClass( TpidCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mux-port INTF
#--------------------------------------------------------------------------------
class MuxPortIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'mux-port INTF'
   noOrDefaultSyntax = syntax
   data = {
      'mux-port' : 'Configure mux port characteristics',
      'INTF' : EthIntfCli.EthPhyIntf.ethMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      intfId = Tac.Value( 'Arnet::IntfId', intf.name )
      mode.createPort( intfId )
      childMode = mode.childMode( VirtualCablePortConfigMode, interface=intf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      intfId = Tac.Value( 'Arnet::IntfId', intf.name )
      mode.deletePort( intfId )

VirtualCableConfigMode.addCommandClass( MuxPortIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] loopback-port INTF MUX_IDS
#--------------------------------------------------------------------------------
def portListFunc( mode, grList ):
   return list( grList.values() )

class LoopbackPortCmd( CliCommand.CliCommandClass ):
   syntax = 'loopback-port INTF MUX_IDS'
   noOrDefaultSyntax = 'loopback-port INTF ...'
   data = {
      'loopback-port' : 'Configure loopback port',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'MUX_IDS' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 4096 ),
         noSingletons=False,
         helpdesc='Set mux-id(s) for given interface(s)',
         tagLong='mux-id' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setLoopbackPort( args[ 'INTF' ],
                            list( args[ 'MUX_IDS' ].values() ), None )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setLoopbackPort( args[ 'INTF' ], None, True )

VirtualCablePortConfigMode.addCommandClass( LoopbackPortCmd )

#------------------------------------------------------------------------------
# plugin definition
#------------------------------------------------------------------------------

def Plugin( entityManager ):
   global configDir, statusDir, ethPhyIntfConfigSliceDir

   configDir = ConfigMount.mount( entityManager, 'virtualcable/config',
                                  'VirtualCable::Config', 'w' )
   statusDir = LazyMount.mount( entityManager, 'virtualcable/status',
                                'VirtualCable::Status', 'r' )
   ethPhyIntfConfigSliceDir = ConfigMount.mount( entityManager, 
                                                 'interface/config/eth/phy/slice',
                                                 'Tac::Dir', 'wi' )

