# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=not-callable
# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in
# pylint: disable=protected-access

#-------------------------------------------------------------------------------
# This module implements VLAN configuration.  In particular, it provides:
# -  the Vlan class
# -  the VlanSet class
# -  the "config-vlan" mode
# -  the "[no] vlan <vlan_set>" command
# -  the "[no] mac address learning" command
# -  the "[no] name <vlan_name>" command
# -  the "[no] shutdown" command
# -  the "[no] state {active|suspend}" command
# -  the "[no] shutdown vlan <vlan_id>" command
# -  the "[no] trunk group <group_name>" command
# -  the "[no] vlan dynamic range <vlan_set>
# -  the "[no] vlan internal order {ascending|descending}" command
# -  the "[no] e-tree role leaf" command
# -  the "[no] remote leaf host drop" command
# -  the "show vlan" command
# -  the "show vlan mtu" command
# -  the "show interfaces [<intf>] trunk" command
# -  the "show vlan summary" command
# -  the "show vlan [[id] <vlan_set> | name <vlan_name>] trunk group" command
# -  the "show vlan internal allocation policy" command
# -  the "show vlan internal usage" command
# -  the "show vlan interface dynamic" command
# -  the "show vlan counters" command
# -  the "show split-horizon group [vlan id] [shgName] command"
# -  the "clear vlan counters" command
# -  the "[no] switchport" comand
# -  the "[no] switchport mode {access|trunk|dot1q-tunnel}" command
# -  the "[no] switchport mode trunk phone" command
# -  the "[no] switchport access vlan" command
# -  the "[no] switchport trunk native vlan" command
# -  the "[no] switchport trunk allowed vlan" command
# -  the "[no] switchport trunk group <group_name>" command
# -  the "[no] switchport vlan translation <vlan-id> <new-vlan-id>" command
# -  the "[no] switchport vlan translation in <vlan-id> <new-vlan-id>" command
# -  the "[no] switchport vlan translation out <vlan-id> <new-vlan-id>" command
# -  the "[no] encapsulation dot1q vlan <vlan-id>" command
# -  the "[no] switchport phone vlan <vlan-id>" command
# -  the "[no] switchport phone trunk tagged [ phone ] | untagged [ phone ]" command
# -  the "[no] switchport default phone vlan <vlan-id>" command
# -  the "[no] switchport default phone cos <0-7>" command
# -  the "[no] switchport default phone trunk tagged [ phone ] | \
#       untagged [ phone ] " command
# -  the "[no] switchport default phone access-list bypass" command
# -  the "[ no | default ] switchport vlan tag validation" command
# -  the "show interfaces [<name>] vlan mapping" command
# -  the "show interfaces [<name>] encapsulation dot1q vlan" command
# -  the "show interfaces [<name>] switchport configuration source" command
# -  the "vlan database" command.
# -  the "[no] mac address forwarding {unicast | multicast} miss \
#       action {drop | flood | log}" command
# -  the "[no] floodset expanded vlan <vlan-id> command
# -  the "show vlan [[id] <vlan_set> | name <vlan_name>] mac \
#       address forwarding" command
# -  the "show e-tree vlan" command
# -  the "[no] vlan tpid" command
#-------------------------------------------------------------------------------
import collections
import Tac
import CliParser
import BasicCli
import TacSigint
from Vlan import computeVlanRangeSet, vlanSetToCanonicalStringGen
from Vlan import  vlanSetToCanonicalString, VlanIdRangeList
import LazyMount, Intf.Log
import BasicCliModes
import ConfigMount
import CliCommand
import CliMatcher
import ShowCommand
import CliExtensions
import re
import MultiRangeRule
from CliMode.Vlan import VlanMode
from CliMode.VlanEtree import VlanEtreeMode
import CliRangeExpansion
import CliToken.Clear
import CliToken.Mac
import SmashLazyMount
import Toggles.EbraToggleLib
import Url
import FileUrl
from CliPlugin import IntfCli, EthIntfCli, EbraCliLib
from CliPlugin.VlanModel import (
   EncapsulationDot1QVlans,
   EncapsulationVlans,
   EtreeRoleModel,
   FlexEncapTaggedSpec,
   FlexEncapEntry,
   FlexEncapSpec,
   FlexEncapTaggedValue,
   FlexEncapValue,
   FlexibleEncapsulations,
   FloodsetExpandedVlanModel,
   FloodsetExpandedVlanTableModel,
   InnerTagMappings,
   InterfaceTrunk,
   InterfaceVlanMappings,
   MappedVlan,
   TrunkInterfaces,
   TrunkVlans,
   Vlans,
   VlanCounters,
   VlanDynamic,
   VlanEtreeRolesModel,
   VlanInterfaceDynamic,
   VlanIds,
   VlanIntfs,
   VlanInternalAllocationPolicy,
   VlanInternalUsage,
   VlanInternalUsageReserved,
   VlanMappings,
   VlanSummary,
   VlanTrunkGroups,
)
from CliPlugin.SplitHorizonGroupModel import (
   SplitHorizonGroups,
   maxGroupNameSize as maxSplitHorizonGroupNameSize
)
from EbraLib import defaultTpid, egressVlanXlateKeyIfBoth
from CliSavePlugin.EbraCliSave import addSwitchIntfConfigCmds
from CliDynamicSymbol import CliDynamicPlugin
import CliSaveBlock
from TypeFuture import TacLazyType
from CliModel import (
   Dict,
   Int,
   List,
   Model,
   Str
)
import Arnet

BridgingCliModel = CliDynamicPlugin( "BridgingCliModel" )

EncapConfigMode = TacLazyType( 'Interface::SubIntfDot1qEncapConfigMode' )
EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
EtreeRole = TacLazyType( 'Bridging::EtreeRole' )
EtreeRemoteLeafInstallMode = TacLazyType( 'Bridging::EtreeRemoteLeafInstallMode' )
FlexEncapConstants = TacLazyType( 'Ebra::FlexEncapConstants' )
FlexEncapEntryType = TacLazyType( 'Ebra::FlexEncapEntry' )
FlexEncapFieldType = TacLazyType( 'Ebra::FlexEncapField::FieldType' )
FlexEncapSpecType = TacLazyType( 'Ebra::FlexEncapSpec::SpecType' )
IntfOperStatus = TacLazyType( 'Interface::IntfOperStatus' )
L2MissPolicyConfig = TacLazyType( "Bridging::L2MissPolicyConfig" )
PeerEthIntfId = TacLazyType( 'Arnet::PeerEthIntfId' )
PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )
ServiceIntfType = TacLazyType( 'Ebra::ServiceIntfType' )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )
TrustMode = TacLazyType( 'Qos::TrustMode' )
VlanConfig = TacLazyType( 'Bridging::Input::VlanConfig' )
VlanTpid = TacLazyType( 'Bridging::VlanTpid' )
VlanFloodsetExpandedReason = TacLazyType( 'Bridging::VlanFloodsetExpandedReason' )
VlanMappingDirection = TacLazyType( 'Bridging::VlanMappingDirection' )
VlanTagFormat = TacLazyType( 'Bridging::VlanTagFormat' )
VlanTypeAndPrimaryVlan = TacLazyType( "Bridging::VlanTypeAndPrimaryVlan" )

# Module globals set up by the Plugin function below
cliConfig = None
bridgingConfig = None
vlanXlateStatusDir = None
bridgingHwCapabilities = None
bridgingHwEnabled = None
bridgingSwitchIntfConfig = None
vlanFloodsetExpandedStatusDir = None
topologyConfig = None
ethIntfStatusDir = None
dynAllowedVlanDir = None
dynBlockedVlanDir = None
dynVlanDir = None
dynVlanIntfDir = None
peerVlanSet = None
vlanIntfConfigDir = None
responseReservedVlanId = None
requestPseudowireReservedVlanId = None
requestIpsecReservedVlanId = None
requestGreTunnelReservedVlanId = None
requestSandGreTunnelReservedVlanId = None
bridgingInputConfigDir = None
vlanMappingRequiredStatusDir = None
vlanCountersCliReq = None
vlanCounterDir = None
vlanCounterSnapshotDir = None
conflictStatusDir = None
conflictStatusClient = None
conflictStatusNetwork = None
serviceIntfTypeWinner = None
subIntfConfigDir = None
subIntfStatusDir = None
intfCapability = None

canRunVlanRangeCmdHook = CliExtensions.CliHook()

vlanMatcher = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Details on VLAN operation' )
switchportMatcher = CliMatcher.KeywordMatcher( 'switchport',
      helpdesc='Set switching mode characteristics' )

def passTheBuck( mode, **args ):
   return args

def vlanInUseErrStr( vlanId ):
   return "VLAN %d is in use as an internal VLAN.  Cannot create it." % ( vlanId, )

def bridgingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.bridgingSupported:
      return None
   return CliParser.guardNotThisPlatform

def phoneTrustModeGuard( mode, token ):
   if bridgingHwCapabilities.supportsPhoneTrustMode:
      return None
   return CliParser.guardNotThisPlatform

def _sessionCounterDir( mode ):
   sessionCounterDir = mode.session.sessionData( 'vlanSessionCounterDir', None )
   if sessionCounterDir is None:
      sessionCounterDir = {}
      mode.session.sessionDataIs( 'vlanSessionCounterDir', sessionCounterDir )

   return sessionCounterDir

#------------------------------------------------------------------------------
# Base class for all dependent classes of Vlan.
#------------------------------------------------------------------------------
class VlanDependentBase:
   """ This class is responsible for cleaning up vlan configurations when
   the vlan is deleted or defaulted
   """
   def __init__( self, vlan ):
      self.vlan_ = vlan

   def destroy( self ):
      pass

#-------------------------------------------------------------------------------
# Class to represent a VLAN within the CLI.
#
# Note that existence of an instance of this class is independent of existence
# of the corresponding VLAN and C++ objects.  Whether or not the VLAN
# exists can be tested by calling the lookup() method.  The VLAN can be
# created and destroyed by calling the create() and destroy() methods.
#-------------------------------------------------------------------------------
class Vlan:

   #----------------------------------------------------------------------------
   # Create a new Vlan instance.
   #----------------------------------------------------------------------------
   def __init__( self, vid ):
      self.id_ = vid
      # We create this variable here to maintain consistency with the
      # VlanSet class. Please look at VlanSet class for more details.
      self.individualVlans_ = ( self, )

   id = property( lambda self: self.id_ )

   def __str__( self ):
      return str( self.id_ )

   #---------------------------------------------------------------------------
   # Registers a dependent class to perform customized actions when a vlan is
   # deleted.
   #---------------------------------------------------------------------------
   depClsRegistry_ = IntfCli.DependentClassRegistry()
   registerDependentClass = depClsRegistry_.registerDependentClass

   #----------------------------------------------------------------------------
   # Creates the VLAN corresponding to this object.  This function is called
   # whenever "config-vlan" mode is entered (i.e. by the "vlan" command).
   #----------------------------------------------------------------------------
   def create( self, mode ):
      if self.id not in cliConfig.vlanConfig:
         cliConfig.doCreateVlan( self.id )

   #----------------------------------------------------------------------------
   # Destroys the VLAN corresponding to this object.  This function is called by
   # the "no vlan" command.
   #----------------------------------------------------------------------------
   def destroy( self, mode ):
      del cliConfig.vlanConfig[ self.id ]
      self._removeCommentAndCallDestroyCallbacks( mode )

   #----------------------------------------------------------------------------
   # Resets the VLAN corresponding to this object.  This function is called by
   # the "default vlan" command. NB: VLAN 1 is on by default.
   #----------------------------------------------------------------------------
   def setDefault( self, mode ):
      if self.id == 1:
         self._removeCommentAndCallDestroyCallbacks( mode )
         config = cliConfig.vlanConfig[ self.id ]
         config.adminState = config.defaultAdminState
         config.agingTime = config.defaultAgingTime
         config.configuredName = config.defaultName
         config.etreeRole = config.defaultEtreeRole
         config.etreeRemoteLeafInstallMode = config.defaultEtreeRemoteLeafInstallMode
         config.floodsetExpandedVlanId = config.defaultFloodsetExpandedVlanId
         config.l2MissPolicyConfig = L2MissPolicyConfig()
         config.macLearning = config.defaultMacLearning
         config.maxLearnedDynamicMac = config.defaultMaxLearnedDynamicMac
         config.trunkGroup.clear()
         config.vlanTypeAndPrimaryVlan = VlanTypeAndPrimaryVlan()
      else:
         self.destroy( mode )

   #----------------------------------------------------------------------------
   # Call any registered destruction callbacks.
   #----------------------------------------------------------------------------
   def _removeCommentAndCallDestroyCallbacks( self, mode ):
      BasicCliModes.removeCommentWithKey( f"vlan-{self.id}" )
      for cls in self.depClsRegistry_:
         cls( self ).destroy()

   #----------------------------------------------------------------------------
   # Searches for the VLAN corresponding to this object, returning True if
   # the VLAN exists.
   #----------------------------------------------------------------------------
   def _lookup( self, config ):
      # FIXME: `Vlan` should subclass `int` and this check should be during init.
      return 0 < self.id < 4095 and self.id in config.vlanConfig

   def lookup( self, mode ):
      """Return True if the vlan is operationally present (in
      bridging/config )"""
      return self._lookup( bridgingConfig )

   def lookupConfig( self, mode ):
      """Return True if the vlan is configured in bridging/input/config/cli
      tree"""
      return self._lookup( cliConfig )

   #----------------------------------------------------------------------------
   # Returns an unordered list of Vlan objects for all existing VLANs.
   # This shows *operational* vlans, not local CLI configuration.
   # If internal is True, returns an unordered list of Vlan objects for all
   # internal VLANS.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAll( mode, internal=False, secondaryVlans=True ):
      vlans = []
      for vlanCfg in bridgingConfig.vlanConfig.values():
         vlanId = vlanCfg.vlanId
         if not secondaryVlans and bridgingConfig.vidToFidMap.get( vlanId ):
            continue
         if vlanCfg.internal == internal:
            vlans.append( Vlan( vlanId ) )
      return vlans

   #----------------------------------------------------------------------------
   # Returns a merged set of dynamic vlans from all the agent.
   #----------------------------------------------------------------------------
   @staticmethod
   def getDynVlanSet( mode ):
      vlanSet = set()
      for dynVlanSet in dynVlanDir.values():
         if dynVlanSet.vlans:
            vlanSet = vlanSet.union( computeVlanRangeSet( dynVlanSet.vlans ) )
      if peerVlanSet.vlans:
         vlanSet = vlanSet.union( computeVlanRangeSet( peerVlanSet.vlans ) )
      return vlanSet

   #----------------------------------------------------------------------------
   # Returns a list of all the responseReservedVlanId items that have allocated
   # vlanIds.
   #----------------------------------------------------------------------------
   @staticmethod
   def getReservedMapping( mode ):
      mapping = {}
      # Iterate Through all pseudowireIntfs
      if requestPseudowireReservedVlanId:
         PwIntfId = Tac.Type( 'Arnet::PseudowireIntfId' )
         for intfId, requested in \
         requestPseudowireReservedVlanId.request.items():
            if ( requested ): # pylint: disable=superfluous-parens
               vlanId = responseReservedVlanId.response[ intfId ]
               if vlanId > 0:
                  intfName = str( PwIntfId( intfId ) )
                  mapping[ vlanId ] = intfName

      # Iterate Through all Sand Gre Tunnel Intfs
      if requestSandGreTunnelReservedVlanId:
         for intfId, requested in \
         requestSandGreTunnelReservedVlanId.request.items():
            if ( requested ): # pylint: disable=superfluous-parens
               vlanId = responseReservedVlanId.response[ intfId ]
               if vlanId > 0:
                  intfName = str( intfId ) + " (GRE)"
                  mapping[ vlanId ] = intfName

      # Iterate Through all Gre Tunnel Intfs
      if requestGreTunnelReservedVlanId:
         for intfId, requested in \
         requestGreTunnelReservedVlanId.request.items():
            if ( requested ): # pylint: disable=superfluous-parens
               vlanId = responseReservedVlanId.response[ intfId ]
               if vlanId > 0:
                  intfName = str( intfId ) + " (GRE)"
                  mapping[ vlanId ] = intfName

      # Iterate Through all Ipsec Tunnel Intfs
      if requestIpsecReservedVlanId:
         for intfId, requested in \
         requestIpsecReservedVlanId.request.items():
            if ( requested ): # pylint: disable=superfluous-parens
               vlanId = responseReservedVlanId.response[ intfId ]
               if vlanId > 0:
                  intfName = str( intfId ) + " (IPsec)"
                  mapping[ vlanId ] = intfName

      return mapping

   #----------------------------------------------------------------------------
   # Returns the name of this VLAN, from the operational config
   #----------------------------------------------------------------------------
   def name( self, mode ):
      vc = bridgingConfig.vlanConfig.get( self.id )
      return getattr( vc, 'configuredName', '' )

   #----------------------------------------------------------------------------
   # Sets the configured name of this VLAN.
   #----------------------------------------------------------------------------
   def setName( self, mode, name ):
      vc = cliConfig.vlanConfig[ self.id ]
      vc.configuredName = name or vc.defaultName

   #----------------------------------------------------------------------------
   # Sets the configured name of this VLAN to its default.
   #----------------------------------------------------------------------------
   def noName( self, mode ):
      self.setName( mode, '' )

   #----------------------------------------------------------------------------
   # Returns the mac learning status of this VLAN, from the operational config
   #----------------------------------------------------------------------------
   def macLearning( self, mode ):
      vc = bridgingConfig.vlanConfig.get( self.id )
      return getattr( vc, 'macLearning', VlanConfig.defaultMacLearning )

   #----------------------------------------------------------------------------
   # Sets the mac learning status of this VLAN.
   #----------------------------------------------------------------------------
   def setMacLearning( self, mode, enable ):
      cliConfig.vlanConfig[ self.id ].macLearning = enable

   #----------------------------------------------------------------------------
   # Disable mac learning of this VLAN.
   #----------------------------------------------------------------------------
   def noMacLearning( self, mode ):
      self.setMacLearning( mode, False )

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy value for VLAN.
   #----------------------------------------------------------------------------
   def setMissPolicy( self, mode, policyType, policyAction ):
      policy = L2MissPolicyConfig()
      curPolicy = cliConfig.vlanConfig[ self.id ].l2MissPolicyConfig
      policy.l2UcMissPolicy = curPolicy.l2UcMissPolicy
      policy.l2McMissPolicy = curPolicy.l2McMissPolicy
      if policyType == 'unicast':
         policy.l2UcMissPolicy = policyAction
      else:
         policy.l2McMissPolicy = policyAction
      cliConfig.vlanConfig[ self.id ].l2MissPolicyConfig = policy

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy value for VLAN to its default (flood)
   #----------------------------------------------------------------------------
   def noMissPolicy( self, mode, policyType ):
      policy = L2MissPolicyConfig()
      self.setMissPolicy( mode, policyType, policy.l2UcMissPolicy )

   #----------------------------------------------------------------------------
   # Returns the administrative state of this VLAN.
   #----------------------------------------------------------------------------
   def adminState( self, mode ):
      """Return the operational administrative state of the Vlan."""
      vc = bridgingConfig.vlanConfig.get( self.id )
      return getattr( vc, 'adminState', 'inactive' )

   def adminStateConfig( self, mode ):
      """Return the configured (via bridging/input/config/cli) administrative
      state of the Vlan."""
      return cliConfig.vlanConfig[ self.id ].adminState

   #----------------------------------------------------------------------------
   # Sets the administrative state of this VLAN.
   #----------------------------------------------------------------------------
   def setAdminState( self, state ):
      cliConfig.vlanConfig[ self.id ].adminState = state

   def trunkGroups( self, mode ):
      """Return the configured trunk groups of the Vlan."""
      vlanConfig = cliConfig.vlanConfig.get( self.id, None )
      if vlanConfig:
         return list( vlanConfig.trunkGroup )
      else:
         # Must be a dynamic vlan
         vlanConfig = bridgingConfig.vlanConfig.get( self.id )
         return list( vlanConfig.trunkGroup )

   #----------------------------------------------------------------------------
   # Add or remove trunk groups to/from this VLAN.
   #----------------------------------------------------------------------------
   def setTrunkGroup( self, mode, group ):
      """Add this vlan to a trunk group."""
      v = cliConfig.vlanConfig[ self.id ]
      v.trunkGroup[ group ] = True

   def noTrunkGroup( self, mode, group ):
      """Remove this vlan from a trunk group. If no group is specified, remove
      from all trunk groups."""
      v = cliConfig.vlanConfig[ self.id ]
      if group:
         del v.trunkGroup[ group ]
      else:
         v.trunkGroup.clear()

   #----------------------------------------------------------------------------
   # Associate or unassociate MAC-VRF with this VLAN.
   # ----------------------------------------------------------------------------
   def setMacVrf( self, mode, name ):
      """Associate / Unassociate this vlan with a MAC-VRF."""
      v = cliConfig.vlanConfig[ self.id ]
      v.macVrf = name

   # ----------------------------------------------------------------------------
   # Sets the vlan type and primary vlan for this VLAN.
   #----------------------------------------------------------------------------
   def setPrivateVlan( self, mode, vlanType, primaryVlan ):
      if primaryVlan.id == self.id:
         mode.addError(
            'Unable to set VLAN ID %d as its own primary VLAN' % self.id )
         return
      vlanTypeAndPrimaryVlan = VlanTypeAndPrimaryVlan()
      vlanTypeAndPrimaryVlan.vlanType = vlanType
      vlanTypeAndPrimaryVlan.primaryVlan = primaryVlan.id
      cliConfig.vlanConfig[ self.id ].vlanTypeAndPrimaryVlan = vlanTypeAndPrimaryVlan

   #----------------------------------------------------------------------------
   # Sets the vlan type and primary vlan for this VLAN to its default.
   #----------------------------------------------------------------------------
   def noPrivateVlan( self, mode ):
      vlanTypeAndPrimaryVlan = VlanTypeAndPrimaryVlan()
      cliConfig.vlanConfig[ self.id ].vlanTypeAndPrimaryVlan = vlanTypeAndPrimaryVlan

   #----------------------------------------------------------------------------
   # Sets the e-tree role for this VLAN to the provided role
   #----------------------------------------------------------------------------
   def setEtreeRole( self, mode, etreeRole ):
      cliConfig.vlanConfig[ self.id ].etreeRole = etreeRole

   # ----------------------------------------------------------------------------
   # Sets the e-tree remote leaf install mode for this VLAN to the provided mode
   # ----------------------------------------------------------------------------
   def setEtreeRemoteLeafInstallMode( self, mode, remoteLeafInstallMode ):
      cliConfig.vlanConfig[ self.id ].etreeRemoteLeafInstallMode = \
            remoteLeafInstallMode

   #----------------------------------------------------------------------------
   # Returns VLAN ID given VLAN name.
   #----------------------------------------------------------------------------
   @staticmethod
   def getVlanIdFromName( mode, vlanName ):
      vlans = Vlan.getAll( mode )
      return next( ( v.id for v in vlans if v.name( mode ) == vlanName ), 0 )

   #----------------------------------------------------------------------------
   # Returns a list of EthIntf objects corresponding to all of the access and
   # trunk ports configured in this VLAN.  For trunk ports, this means that the
   # VLAN is allowed on the trunk. dotq1-tunnel ports are treated as access.
   #----------------------------------------------------------------------------
   def configuredPorts( self, mode ):
      dynVlanSet = Vlan.getDynVlanSet( mode )
      dynamic = self.id in dynVlanSet and not self.lookupConfig( mode )
      mvrpDynVlans = dynVlanDir[ 'mvrp' ].vlans if 'mvrp' in dynVlanDir else []
      isMvrpDynVlan = dynamic and self.id in computeVlanRangeSet( mvrpDynVlans )

      config = bridgingSwitchIntfConfig
      ports = []
      # pylint: disable-next=too-many-nested-blocks
      for ( sw, switchIntfConfig ) in config.switchIntfConfig.items():
         intfId = switchIntfConfig.intfId
         if switchIntfConfig.enabled == False: # pylint: disable=singleton-comparison
            continue
         if switchIntfConfig.l2SubIntfVlan.nameOrId != 'none':
            # This is a L2 sub-interface. Get the VLAN ID from config.
            assert SubIntfId.isSubIntfId( intfId )
            if switchIntfConfig.l2SubIntfVlan.nameOrId == 'id':
               vlanId = int( switchIntfConfig.l2SubIntfVlan.vlanId )
            else:
               assert switchIntfConfig.l2SubIntfVlan.nameOrId == 'name'
               vlanName = switchIntfConfig.l2SubIntfVlan.vlanName
               vlanId = Vlan.getVlanIdFromName( mode, vlanName )
            if vlanId == self.id:
               ports.append( EthIntfCli.EthIntf( sw, mode ) )
         elif ( switchIntfConfig.switchportMode in ( 'access', 'dot1qTunnel' ) and
                switchIntfConfig.accessVlan == self.id ):
            if not isMvrpDynVlan:
               ports.append( EthIntfCli.EthIntf( sw, mode ) )
         elif switchIntfConfig.switchportMode == 'trunk':
            intfPresent = False
            vlanSet = computeVlanRangeSet( switchIntfConfig.trunkAllowedVlans )
            if self.id in vlanSet and not isMvrpDynVlan:
               # We check only the dynamic allowed vlan list for MVRP dynamic
               # VLANs.
               intfPresent = True
            else:
               # Check if the vlan is present in any dynamic allowed vlan list
               # for this interface.
               noDynamicInfo = (
                     SubIntfId.isSubIntfId( intfId ) or
                     EthIntfId.isUnconnected( intfId ) or
                     PortChannelIntfId.isRecirc( intfId )
                     )

               if not noDynamicInfo:
                  for dynAgent in dynAllowedVlanDir.entityPtr:
                     entity = dynAllowedVlanDir.entityPtr.get( dynAgent, None )
                     if entity:
                        vlanSet = computeVlanRangeSet(
                           entity.vlans.get( sw, "" ) )
                        if self.id in vlanSet:
                           intfPresent = True
                           break
            if intfPresent:
               ports.append( EthIntfCli.EthIntf( sw, mode ) )
      return ports

   #----------------------------------------------------------------------------
   # Returns a list of EthIntf objects corresponding to all of the ports
   # configured to be part of this VLAN that are active.  We assume this is
   # only called for Vlan's whose adminState is not "inactive"
   #----------------------------------------------------------------------------
   def activePorts( self, mode, promoted=True ):
      vlanConfig = bridgingConfig.vlanConfig
      ports = []
      tc = topologyConfig.vidToTopoMap.get( self.id )
      vc = vlanConfig.get( self.id )
      # Is it possible that someone in another Cli has destroyed the
      # vlanConfig, so that adminState is now 'inactive'?  And, if so,
      # do we need to check here, before going ahead?
      if tc and vc:
         for portname in tc.topoPortConfig:
            tpcExists = portname in tc.topoPortConfig
            vpcExists = portname in vc.intf
            if tpcExists and vpcExists:
               if not promoted and vc.intf[ portname ].promotedPort:
                  continue
               ports.append( EthIntfCli.EthIntf( portname, mode ) )
      return ports

   #----------------------------------------------------------------------------
   # Returns a set of EthIntf objects corresponding to all blocked ports
   # for this VLAN.
   #----------------------------------------------------------------------------
   def blockedPorts( self, mode ):
      blocked = set()
      for dynAgent in dynBlockedVlanDir.entityPtr:
         entity = dynBlockedVlanDir.entityPtr.get( dynAgent )
         if entity:
            for intf, vlans in entity.vlans.items():
               vlanSet = computeVlanRangeSet( vlans )
               if self.id in vlanSet:
                  blocked.add( intf )
      return blocked

   def __hash__( self ):
      return hash( self.id_ )

   def __eq__( self, other ):
      return self.id_ == other.id_

   def __ne__( self, other ):
      return self.id_ != other.id_

   def __lt__( self, other ):
      return self.id_ < other.id_

   def __le__( self, other ):
      return self.id_ <= other.id_

   def __gt__( self, other ):
      return self.id_ > other.id_

   def __ge__( self, other ):
      return self.id_ >= other.id_

   #----------------------------------------------------------------------------
   # Support for VLAN counters.
   #----------------------------------------------------------------------------
   def counter( self ):
      # Returns the current VlanCounter object for this vlan.
      if not self.countersSupported():
         return None
      return vlanCounterDir.vlanCounter.get( self.id )

   def getCounterSnapshot( self, mode, sessionOnly=False ):
      # Returns the global or session counter snapshot depending on sessionOnly.
      if sessionOnly:
         vlanCounter = _sessionCounterDir( mode ).get( self.id )
         if not vlanCounter:
            vlanCounter = Tac.Value( "Smash::Bridging::VlanCounter", self.id )
      else:
         vlanCounter = vlanCounterSnapshotDir.vlanCounterSnapshot.get( self.id )
      return vlanCounter

   def getLatestCounterSnapshot( self, mode ):
      # Get the latest counter snapshot whether global or for this session only.

      snapshotGlobal = self.getCounterSnapshot( mode, sessionOnly=False )
      snapshotSession = self.getCounterSnapshot( mode, sessionOnly=True )
      if snapshotGlobal and snapshotSession:
         # Compare timestamps if both are there
         if snapshotGlobal.statsUpdateTime > snapshotSession.statsUpdateTime:
            return snapshotGlobal
         else:
            return snapshotSession
      else:
         # Return one if at least one not there
         if snapshotGlobal:
            return snapshotGlobal
         else:
            return snapshotSession

   def clearCounters( self, mode ):
      if self.countersSupported():
         _sessionCounterDir( mode )[ self.id ] = self.getCounterSnapshot( mode,
                                                               sessionOnly=True )
         if self.counter() is None:
            return
         _sessionCounterDir( mode )[ self.id ] = self.counter()

   def countersSupported( self ):
      return ( bridgingHwEnabled.vlanIngressCountersEnabled or
               bridgingHwEnabled.vlanEgressCountersEnabled or
               bridgingHwEnabled.vlanBumCountersEnabled )

   def updateVlanCountersModel( self, snapshot=None, notFoundString=None ):
      counter = self.counter()
      def stat( attr ):
         if counter is None:
            return 0

         currentValue = getattr( counter.vlanStats, attr, None )

         if currentValue is None:
            return notFoundString

         snapshotValue = 0
         if snapshot:
            snapshotValue = getattr( snapshot.vlanStats, attr, None )
         return currentValue - snapshotValue

      return VlanCounters.VlanStats(
            inOctets=stat( 'inOctets' ),
            inUcastPkts=stat( 'inUcastPkts' ),
            inMcastPkts=stat( 'inMcastPkts' ),
            inBcastPkts=stat( 'inBcastPkts' ),
            outOctets=stat( 'outOctets' ),
            outUcastPkts=stat( 'outUcastPkts' ),
            outMcastPkts=stat( 'outMcastPkts' ),
            outBcastPkts=stat( 'outBcastPkts' ) )

#------------------------------------------------------------------------------------
# Represents a set of VLANs within the CLI.
#------------------------------------------------------------------------------------
class VlanSet( set ):
   def __init__( self, mode, vlanSetString, vlanIds=None ):
      if vlanIds is None:
         try:
            vlanIds = MultiRangeRule.setFromCanonicalString( vlanSetString )
         except MultiRangeRule.MalformedCanonicalStringError as ex:
            mode.addErrorAndStop( str( ex ) )

      super().__init__( vlanIds )
      self.string_ = vlanSetString

      # We create individualVlans_ list consisting of individual
      # Vlan objects from the VlanSet so that if a cli command needs to
      # be executed on individual Vlans, it can be done easily since we
      # have each of the Vlan objects in this list.
      self.individualVlans_ = tuple( Vlan( vid ) for vid in self )

   def copy( self ):
      return VlanSet( None, self.string_, self )

   def __str__( self ):
      return self.string_

   #----------------------------------------------------------------------------
   # Creates the VLANs corresponding to this object.  This function is called
   # whenever "config-vlan" mode is entered (i.e. by the "vlan" command).
   #----------------------------------------------------------------------------
   def create( self, mode ):
      for vid in self:
         Vlan( vid ).create( mode )

   #----------------------------------------------------------------------------
   # Destroys the VLANs corresponding to this object.  This function is called by
   # the "no vlan" command.
   #----------------------------------------------------------------------------
   def destroy( self, mode ):
      for vid in self:
         Vlan( vid ).destroy( mode )

   #----------------------------------------------------------------------------
   # Resets the VLANs corresponding to this object.  This function is called by
   # the "default vlan" command.
   #----------------------------------------------------------------------------
   def setDefault( self, mode ):
      for vid in self:
         Vlan( vid ).setDefault( mode ) # TODO: use `individualVlans_` for these.

   #----------------------------------------------------------------------------
   # Sets the configured name of each VLAN in this set.
   #----------------------------------------------------------------------------
   def setName( self, mode, name ):
      for vid in self:
         Vlan( vid ).setName( mode, name )

   #----------------------------------------------------------------------------
   # Sets the configured name of each VLAN in this set to its default.
   #----------------------------------------------------------------------------
   def noName( self, mode ):
      for vid in self:
         Vlan( vid ).noName( mode )

   #----------------------------------------------------------------------------
   # Sets the mac learning status of each VLAN in this set.
   #----------------------------------------------------------------------------
   def setMacLearning( self, mode, enable ):
      for vid in self:
         Vlan( vid ).setMacLearning( mode, enable )

   #----------------------------------------------------------------------------
   # Disable mac learning of each VLAN in this set.
   #----------------------------------------------------------------------------
   def noMacLearning( self, mode ):
      for vid in self:
         Vlan( vid ).noMacLearning( mode )

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy of each VLAN in this set.
   #----------------------------------------------------------------------------
   def setMissPolicy( self, mode, policyType, policyAction ):
      for vid in self:
         Vlan( vid ).setMissPolicy( mode, policyType, policyAction )

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy value for VLAN to its default (flood)
   #----------------------------------------------------------------------------
   def noUcMissPolicy( self, mode, policyType ):
      for vid in self:
         Vlan( vid ).noMissPolicy( mode, policyType )

   #----------------------------------------------------------------------------
   # Sets the administrative state of each VLAN in this set.
   #----------------------------------------------------------------------------
   def setAdminState( self, state ):
      for vid in self:
         Vlan( vid ).setAdminState( state )

   #----------------------------------------------------------------------------
   # Add or remove allowed trunk group(s) from this Vlan
   #----------------------------------------------------------------------------
   def setTrunkGroup( self, mode, group ):
      for vid in self:
         Vlan( vid ).setTrunkGroup( mode, group )

   def noTrunkGroup( self, mode, group ):
      for vid in self:
         Vlan( vid ).noTrunkGroup( mode, group )

   #----------------------------------------------------------------------------
   # Sets the vlan type and primary vlan for each VLAN in this set.
   #----------------------------------------------------------------------------
   def setPrivateVlan( self, mode, vlanType, primaryVlan ):
      for vid in self:
         Vlan( vid ).setPrivateVlan( mode, vlanType, primaryVlan )

   #----------------------------------------------------------------------------
   # Sets the vlan type and primary vlan of each VLAN in this set to its default.
   #----------------------------------------------------------------------------
   def noPrivateVlan( self, mode ):
      for vid in self:
         Vlan( vid ).noPrivateVlan( mode )

   #----------------------------------------------------------------------------
   # Sets the e-tree role for each VLAN in this set
   #----------------------------------------------------------------------------
   def setEtreeRole( self, mode, etreeRole ):
      for vid in self:
         Vlan( vid ).setEtreeRole( mode, etreeRole )

   # ----------------------------------------------------------------------------
   # Sets the e-tree remote leaf install mode for each VLAN in this set
   # ----------------------------------------------------------------------------
   def setEtreeRemoteLeafInstallMode( self, mode, remoteLeafInstallMode ):
      for vid in self:
         Vlan( vid ).setEtreeRemoteLeafInstallMode( mode, remoteLeafInstallMode )

#-------------------------------------------------------------------------------
# Rule to match the token class <vlan_id>, returning a corresponding Vlan
# object.
#-------------------------------------------------------------------------------
vlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
                                           helpdesc='Identifier for a Virtual LAN',
                                           value=lambda mode, match: Vlan( match ) )

#-------------------------------------------------------------------------------
# Rule to match the token class <vlan_set>, returning a corresponding VlanSet
# object.
#-------------------------------------------------------------------------------
def vlanIdListFunc( mode, grList ):
   return VlanSet( mode, vlanIds=list( grList.values() ),
                   vlanSetString=str( grList ) )

vlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'VLAN ID or range(s) of VLAN IDs',
   value=vlanIdListFunc )

#-------------------------------------------------------------------------------
# The "config-vlan" mode.
#-------------------------------------------------------------------------------
class VlanConfigMode( VlanMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'VLAN configuration'

   #----------------------------------------------------------------------------
   # Constructs a new VlanConfigMode instance for a particular Vlan or VlanSet
   # object.  This is called every time the user enters "config-vlan" mode.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, vlan ):
      self.vlan = vlan
      VlanMode.__init__( self, ( None, str( self.vlan ), True ) )
      self.multiInstance = False
      if len( self.vlan.individualVlans_ ) > 1:
         self.multiInstance = True
         self.constructIndividualVlanConfigModes( parent, session )
      BasicCli.ConfigModeBase.__init__( self, parent, session,
                                        self.multiInstance,
                                        self.vlan.individualVlans_ )

   #----------------------------------------------------------------------------
   # Returns vlanConfig objects for vlan
   # ----------------------------------------------------------------------------
   def vlanConfigs( self ):
      # FIXME: both should have individualVlans_. Looks tricky, so next MUT.
      if self.multiInstance:
         for m in self.vlan.individualVlans_:
            yield cliConfig.vlanConfig[ m.vlan.id_ ]
      else:
         yield cliConfig.vlanConfig[ self.vlan.id ]

   #----------------------------------------------------------------------------
   # Since the vlan is a VlanSet type, we construct individual
   # VlanConfigMode objects for each of the vlans in the set and
   # stores them back into the individualVlans_ list. This is done so
   # that (if needed) cli commands can be executed on indiviual vlans
   # instead of the entire vlan set.
   # ----------------------------------------------------------------------------
   def constructIndividualVlanConfigModes( self, parent, session ):
      # FIXME: It's very sus that we put VLAN config modes under individual VLANs.
      self.vlan.individualVlans_ = tuple( VlanConfigMode( parent, session, vlan )
                                          for vlan in self.vlan.individualVlans_ )

   # pylint: disable-next=inconsistent-return-statements
   def parse( self, tokens, autoComplete=True, authz=True, acct=True ):
      # This is very exciting.  Not many modes override mode.parse().
      # What I do is ask all of my babies to parse for me.  I never
      # use my modeRule at all.  This way, the configuration applies
      # to all vlans.
      for hook in canRunVlanRangeCmdHook.extensions():
         if not hook( self.vlan.individualVlans_,
                      tokens, autoComplete=autoComplete ):
            return
      # Since there is only one class VlanConfigMode for Vlans as well
      # as Vlan ranges, we have to seperate out functionality of the
      # parser.  It is needed for processing of commands like show
      # running-config. -risha
      if self.multiInstance:
         # Try parsing a range expansion.
         if CliRangeExpansion.tryRangeParse( self, tokens, autoComplete=autoComplete,
                                             authz=authz, acct=acct ):
            return

         # Try the parse myself.  This handles things like
         # "exit" without doing something crazy (like invoking "exit"
         # once for each vlan in the set.)
         try:
            return BasicCli.ConfigModeBase.parse( self, tokens,
                                                  autoComplete=autoComplete,
                                                  authz=authz, acct=acct )
         except CliParser.ParseError:
            pass

         results = {}
         for m in self.vlan.individualVlans_:
            results[ m ] = m.parse( tokens, autoComplete=autoComplete,
                                    authz=authz, acct=acct )

         CmdHandler = collections.namedtuple( 'CmdHandler', [ 'handler', 'dummy' ] )
         cmdHandler = CmdHandler( self._invokeValueFunc, True )
         return { "cmdHandler": cmdHandler,
                  "allowCache": False,
                  "kargs": { "results": results, "authz": authz, "acct": acct },
                  "aaa": None,
                  "cmdDeprecatedBy": None }
      else:
         return BasicCli.ConfigModeBase.parse( self, tokens, autoComplete )

   def _invokeValueFunc( self, mode, results, authz, acct ):
      for m in self.vlan.individualVlans_:
         # pylint: disable-msg=W0212
         m.session._invokeValueFunc( m, results[ m ], authz, acct )

         # We do authorization for each child mode (since RBAC might reject it),
         # but not accounting
         acct = False

         # Allow keyboard interrupt
         TacSigint.check()

   def printShowActiveForVlanRange( self, url, configmode='', prefix='' ):
      # This is a set of all the vlan ranges in this group
      vlanIdsToEvaluate = []
      matched = False
      inConfigMode = False
      # FIXME: This is hacky.
      # `VlanSet.individualVlans` is an iterable of `Vlan` objects, but
      # `VlanConfigMode.individualVlans` is an iterable of `VlanConfigMode` objects.
      for m in self.vlan.individualVlans_:
         vlanIdsToEvaluate.append( getattr( m, 'vlan', m ).id_ )
      try: # pylint: disable=too-many-nested-blocks
         f = url.open()
         try:
            for line in f.readlines():
               if matched:
                  if line.startswith( '   ' + prefix ):
                     print( line, end=( '' if line.endswith( '\n' ) else ' ' ) )
                     continue
               matched = False

               # Enter the configmode
               if configmode and not inConfigMode:
                  if re.match( r'^' + configmode, line ):
                     inConfigMode = True
                     print( configmode )
                  continue
               # Exit the configmode, stop parsing
               if inConfigMode and not re.match( r'^' + prefix, line ):
                  break

               if not re.match( r'^' + prefix + r'vlan \d+', line ):
                  continue
               # range to list
               ranges = re.findall( r'[0-9\-,]+', line )
               vlans = set()
               for rng in ranges: # Somewhat pointless since `len( ranges ) == 1`.
                  vlans |= MultiRangeRule.setFromCanonicalString( rng )
               vlansToShow = set( vlanIdsToEvaluate ) & vlans
               if vlansToShow:
                  matched = True
                  # list to range
                  start = end = 0
                  vlansString = []
                  for vlan in sorted( list( vlansToShow ) ) + [ 0 ]:
                     if vlan == end + 1:
                        end = vlan
                        if configmode and vlan == 1:
                           start = 1
                     else:
                        if start:
                           if start == end:
                              vlansString += [ str( start ) ]
                           else:
                              vlansString += [ '%d-%d' % ( start, end ) ]
                        start = end = vlan
                  print( prefix + 'vlan ' + ','.join( vlansString ) )
         finally:
            f.close()
      except OSError:
         pass

   def showActive( self ):
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ) )
      self.printShowActiveForVlanRange( runningConfigUrl )

   def showActiveAll( self, showDetail ):
      runningConfigAllUrl = FileUrl.localRunningConfigAll(
         *Url.urlArgsFromMode( self ), showDetail=showDetail )
      self.printShowActiveForVlanRange( runningConfigAllUrl )

   def bridgingInputConfigs( self ):
      return self.vlanConfigs()

   #----------------------------------------------------------------------------
   # Sets the mac learning status of this VLAN.
   #----------------------------------------------------------------------------
   def setMacLearning( self ):
      for vlanConfig in self.vlanConfigs():
         vlanConfig.macLearning = True

   #----------------------------------------------------------------------------
   # Sets the maximum limit for locally learned dynamic macs of this VLAN.
   #----------------------------------------------------------------------------
   def setMaxLearnedDynamicMac( self, limit ):
      for vlanConfig in self.vlanConfigs():
         vlanConfig.maxLearnedDynamicMac = limit

   #----------------------------------------------------------------------------
   # Disable mac learning of this VLAN.
   #----------------------------------------------------------------------------
   def noMacLearning( self, args ):
      for vlanConfig in self.vlanConfigs():
         vlanConfig.macLearning = False
      self.setMaxLearnedDynamicMac( VlanConfig.defaultMaxLearnedDynamicMac )

   #----------------------------------------------------------------------------
   # Resets the mac limit to 0 and enabled learning
   #----------------------------------------------------------------------------
   def defaultMacLearning( self, args ):
      self.setMaxLearnedDynamicMac( VlanConfig.defaultMaxLearnedDynamicMac )
      self.setMacLearning()

   #----------------------------------------------------------------------------
   # Sets the maximum limit for locally learned dynamic macs and the
   # mac learning status of this VLAN.
   #----------------------------------------------------------------------------
   def setMacLearningConfig( self, args ):
      limit = args.get( 'MAXIMUM', VlanConfig.defaultMaxLearnedDynamicMac )
      self.setMaxLearnedDynamicMac( limit )
      self.setMacLearning()

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy value for VLAN.
   #----------------------------------------------------------------------------
   def setMissPolicy( self, args ):
      policyType = args[ 'ADDR_TYPE' ]
      policyAction = args[ 'ACTION_TYPE' ]
      policy = L2MissPolicyConfig()
      for vlanConfig in self.vlanConfigs():
         curPolicy = vlanConfig.l2MissPolicyConfig
      policy.l2UcMissPolicy = curPolicy.l2UcMissPolicy
      policy.l2McMissPolicy = curPolicy.l2McMissPolicy
      if policyType == 'unicast':
         policy.l2UcMissPolicy = policyAction
      else:
         policy.l2McMissPolicy = policyAction
      for vlanConfig in self.vlanConfigs():
         vlanConfig.l2MissPolicyConfig = policy

   #----------------------------------------------------------------------------
   # Sets unicast/multicast miss policy value for VLAN to its default (flood)
   #----------------------------------------------------------------------------
   def noMissPolicy( self, args ):
      policyType = args[ 'ADDR_TYPE' ] # pylint: disable=unused-variable
      policy = L2MissPolicyConfig()
      args[ 'ACTION_TYPE' ] = policy.l2UcMissPolicy
      self.setMissPolicy( args )

#-------------------------------------------------------------------------------
# The "[no] vlan <vlan_set>" command, in "config" mode.
#
# The full syntax of this command is:
#
#   [no|default] vlan <vlan_set>
#-------------------------------------------------------------------------------

# extraGotoVlanModeHook extensions accept two arguments: the mode and
# the vlanSet and returns nothing
extraGotoVlanModeHook = CliExtensions.CliHook()

# canDeleteVlanHook extensions accept two arguments: the mode and the
# vlanSet and returns True if the set of vlans can be deleted, and
# False otherwise
canDeleteVlanHook = CliExtensions.CliHook()

def gotoVlanMode( mode, args ):
   vlanSet = args[ 'VLAN_SET' ]
   for vid in vlanSet:
      vc = bridgingConfig.vlanConfig.get( vid )
      # If one tries to create a vlan which is internal vlan while in interactive
      # mode, do not allow that. Allow vlan creation in non interactive modes.
      if vc and vc.internal and mode.session_.interactive_:
         mode.addError( vlanInUseErrStr( vid ) )
         return
   if len( vlanSet ) == 1:
      vlan = Vlan( list( vlanSet )[ 0 ] )
   else:
      vlan = vlanSet
   # Note that we create the VLAN(s) before entering the child mode, so that if for
   # some reason the VLAN(s) can't be created, we remain in the parent mode.
   childMode = mode.childMode( VlanConfigMode, vlan=vlan )
   childMode.vlan.create( mode )
   mode.session_.gotoChildMode( childMode )
   for hook in extraGotoVlanModeHook.extensions():
      hook( mode, vlanSet )

def noVlanHandler( mode, args ):
   vlans = args[ 'VLAN_SET' ]
   if canDeleteVlanHook.all( mode, vlans ):
      vlans.destroy( mode )

def defaultVlanHandler( mode, args ):
   # Default means 'no' for all VLANs except 1, so we need to check the hook.
   vlans = args[ 'VLAN_SET' ]
   if canDeleteVlanHook.all( mode, vlans - { 1 } ):
      vlans.setDefault( mode )

class VlanModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_SET'
   noOrDefaultSyntax = syntax
   data = {
      'vlan' : CliCommand.guardedKeyword( 'vlan',
                  helpdesc='VLAN commands',
                  guard=bridgingSupportedGuard ),
      'VLAN_SET' : vlanSetMatcher,
      }

   allowCache = False
   handler = gotoVlanMode
   noHandler = noVlanHandler
   defaultHandler = defaultVlanHandler

BasicCli.GlobalConfigMode.addCommandClass( VlanModeCmd )

# -------------------------------------------------------------------------------
# The "config-vlan-etree" mode.
# -------------------------------------------------------------------------------
class VlanEtreeConfigMode( VlanEtreeMode, BasicCli.ConfigModeBase ):
   # ----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   # ----------------------------------------------------------------------------
   name = 'VLAN E-Tree configuration'

   # ----------------------------------------------------------------------------
   # Constructs a new VlanConfigMode instance for a particular Vlan or VlanSet
   # object.  This is called every time the user enters "config-vlan" mode.
   # ----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      VlanEtreeMode.__init__( self, str( parent.vlan ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -------------------------------------------------------------------------------
# The "[no|default] e-tree role leaf" command, in "config-vlan" mode
#
# The full syntax of this command is:
#
#     e-tree role leaf
#     no e-tree role leaf
#     default e-tree role leaf
# -------------------------------------------------------------------------------
def gotoVlanEtreeMode( mode, args ):
   mode.vlan.setEtreeRole( mode, EtreeRole.etreeRoleLeaf )
   childMode = mode.childMode( VlanEtreeConfigMode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultVlanEtreeHandler( mode, args ):
   mode.vlan.setEtreeRole( mode, EtreeRole.etreeRoleRoot )
   mode.vlan.setEtreeRemoteLeafInstallMode( mode,
         EtreeRemoteLeafInstallMode.dontInstall )

class VlanEtreeModeCmd( CliCommand.CliCommandClass ):
   syntax = 'e-tree role leaf'
   noOrDefaultSyntax = 'e-tree role ...'
   data = {
      'e-tree': 'E-Tree configuration',
      'role': 'E-Tree role configuration',
      'leaf': 'Leaf VLAN',
   }

   handler = gotoVlanEtreeMode
   noOrDefaultHandler = noOrDefaultVlanEtreeHandler

VlanConfigMode.addCommandClass( VlanEtreeModeCmd )

# -------------------------------------------------------------------------------
# The "[no|default] remote leaf host drop" command, in "config-vlan-etree" mode
#
# The full syntax of this command is:
#
#     remote leaf host drop
#     no remote leaf host drop
#     default remote leaf host drop
# -------------------------------------------------------------------------------
class EtreeRemoteLeafHostInstallModeCmd( CliCommand.CliCommandClass ):
   syntax = 'remote leaf host drop'
   noOrDefaultSyntax = syntax
   data = {
      'remote': 'Remote entities',
      'leaf': 'Leaf entities',
      'host': 'Host entries',
      'drop': 'Install explicit drop nexthops for remote leaf hosts',
   }

   @staticmethod
   def handler( mode, args ):
      mode.parent_.vlan.setEtreeRemoteLeafInstallMode( mode,
            EtreeRemoteLeafInstallMode.installDrop )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.parent_.vlan.setEtreeRemoteLeafInstallMode( mode,
            EtreeRemoteLeafInstallMode.dontInstall )

if Toggles.EbraToggleLib.toggleEtreeRemoteLeafInstallModeEnabled():
   VlanEtreeConfigMode.addCommandClass( EtreeRemoteLeafHostInstallModeCmd )

#-------------------------------------------------------------------------------
# The "[no] name <vlan_name>" command, in "config-vlan" mode.
#
# The full syntax of this command is:
#
#   name <vlan_name>
#   [no|default] name [<vlan_name>]
#-------------------------------------------------------------------------------

# Note that VLAN names are silently truncated to 32 characters.  This is consistent
# with the industry-standard.
vlanNameMatcher = CliMatcher.PatternMatcher( pattern='.+',
                     helpname='WORD',
                     helpdesc='The ASCII name for the VLAN',
                     value=lambda mode, match: match[ :32 ] )

def setName( mode, args ):
   name = args.get( 'VLAN_NAME' )
   mode.vlan.setName( mode, name )

def noName( mode, args ):
   mode.vlan.noName( mode )

class VlanNameCmd( CliCommand.CliCommandClass ):
   syntax = 'name VLAN_NAME'
   noOrDefaultSyntax = 'name ...'
   data = {
      'name' : 'ASCII name of the VLAN',
      'VLAN_NAME' : vlanNameMatcher,
      }

   handler = setName
   noOrDefaultHandler = noName

VlanConfigMode.addCommandClass( VlanNameCmd )

#-------------------------------------------------------------------------------
# The "mac address learning" command, in "config-vlan" mode.
#
# The full syntax of this command is:
#
#   [no|default] mac address learning [local limit <maximum> hosts]
#-------------------------------------------------------------------------------
def vlanMacAddressCliGuard( mode, token ):
   if( bridgingHwCapabilities.vlanMacLearningSupported or \
         bridgingHwCapabilities.vlanL2MissPolicySupported ):
      return None
   return CliParser.guardNotThisPlatform

def vlanMacLearningSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanMacLearningSupported:
      return None
   return CliParser.guardNotThisPlatform

def macLimitPerVlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.macLimitPerVlanSupported:
      return None
   return CliParser.guardNotThisPlatform

nodeMac = CliCommand.guardedKeyword( 'mac',
             helpdesc='Configure mac property of the VLAN',
             guard=vlanMacAddressCliGuard )

matcherAddress = CliMatcher.KeywordMatcher( 'address',
                    helpdesc='Configure mac address property of the VLAN' )

nodeLearning = CliCommand.guardedKeyword( 'learning',
                  helpdesc='Configure mac address learning property of the VLAN',
                  guard=vlanMacLearningSupportedGuard )

nodeLocalHelp = 'Configure local mac address learning property of the VLAN'
nodeLocal = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'local', helpdesc=nodeLocalHelp ),
      guard=macLimitPerVlanSupportedGuard )

matcherMaximum = CliMatcher.IntegerMatcher( 1, 10000,
                    helpdesc='Maximum number of locally learned dynamic hosts' )

class MacAddressLearningCmd( CliCommand.CliCommandClass ):

   syntax = 'mac address learning [ local limit MAXIMUM hosts ]'
   noOrDefaultSyntax = syntax
   data = {
         'mac': nodeMac,
         'address': matcherAddress,
         'learning': nodeLearning,
         'local': nodeLocal,
         'limit': ( 'Configure limit on locally learned dynamic MAC addresses on '
                    'the VLAN' ),
         'MAXIMUM': matcherMaximum,
         'hosts': 'Locally learned dynamic MACs',
   }
   handler = VlanConfigMode.setMacLearningConfig
   noHandler = VlanConfigMode.noMacLearning
   defaultHandler = VlanConfigMode.defaultMacLearning

VlanConfigMode.addCommandClass( MacAddressLearningCmd )

#-------------------------------------------------------------------------------
# The "shutdown" command, in "config-vlan" mode.
#
# The full syntax of this command is:
#
#   [no|default] shutdown
#-------------------------------------------------------------------------------
def setShutdown( mode, args ):
   mode.addError( "The 'shutdown' command is not supported.  "
                  "Please use 'state suspend' instead." )

def noShutdown( mode, args ):
   mode.addError( "The 'no shutdown' command is not supported.  "
                  "Please use 'state active' instead." )

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown VLAN switching',
   }
   handler = setShutdown
   noOrDefaultHandler = noShutdown
   hidden = True

VlanConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# The "shutdown vlan VLAN_ID" command, in "config" mode.
#
# The full syntax of this command is:
#
#   [no|default] shutdown vlan VLAN_ID
#-------------------------------------------------------------------------------

def setShutdownFromConfigMode( mode, args ):
   mode.addError( "The 'shutdown' command is not supported.  "
                  "Please use 'state suspend' in config-vlan mode instead." )

def noShutdownFromConfigMode( mode, args ):
   mode.addError( "The 'no shutdown' command is not supported.  "
                  "Please use 'state active' in config-vlan mode instead." )

class ShutdownVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown vlan VLAN_ID'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown system elements',
      'vlan': 'Shutdown VLAN switching',
      'VLAN_ID': vlanIdMatcher,
   }
   handler = setShutdownFromConfigMode
   noOrDefaultHandler = noShutdownFromConfigMode
   hidden = True

BasicCliModes.GlobalConfigMode.addCommandClass( ShutdownVlanVlanidCmd )

#-------------------------------------------------------------------------------
# The "state" command, in "config-vlan" mode.
#
# The full syntax of this command is:
#
# [ no | default ] state ( active | suspend )
#
#-------------------------------------------------------------------------------

class StateCmd( CliCommand.CliCommandClass ):
   syntax = 'state ( active | suspend ) '
   noOrDefaultSyntax = 'state ...'
   data = {
      'state': 'Operational state of the VLAN',
      'active': 'VLAN Active State',
      # added alternate suspended token to address Bug438127
      'suspend': CliMatcher.KeywordMatcher( 'suspend',
                    helpdesc='VLAN Suspended State',
                    value=lambda mode, match: 'suspended',
                    alternates=[ 'suspended' ] )
   }

   @staticmethod
   def handler( mode, args ):
      # 'active' by default.
      mode.vlan.setAdminState( args.get( 'suspend', 'active' ) )

   noOrDefaultHandler = handler

VlanConfigMode.addCommandClass( StateCmd )

#-------------------------------------------------------------------------------
# The "[no] trunk group <group>", in "config-vlan" mode
#
# The full syntax of this command is:
#
#     trunk group <group>
#     no trunk group [<group>]
#-------------------------------------------------------------------------------
trunkGroupMatcher = CliMatcher.PatternMatcher( pattern='.+',
                       helpname='WORD',
                       helpdesc='trunk group name',
                       value=lambda mode, match: match[ :32 ] )

def doTrunkGroup( mode, args ):
   group = args.get( 'GROUP' )
   mode.vlan.setTrunkGroup( mode, group )

def doNoTrunkGroup( mode, args ):
   group = args.get( 'GROUP' )
   mode.vlan.noTrunkGroup( mode, group )

class TrunkGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'trunk group GROUP'
   noOrDefaultSyntax = 'trunk group [ GROUP ]'
   data = {
      'trunk' : 'Configure trunking characteristics of the VLAN',
      'group' : 'Configure a trunk group',
      'GROUP' :  trunkGroupMatcher,
      }

   handler = doTrunkGroup
   noOrDefaultHandler = doNoTrunkGroup

VlanConfigMode.addCommandClass( TrunkGroupCmd )

# -------------------------------------------------------------------------------
# The "[no] mac-vrf <mac-vrf-name>", in "config-vlan" mode
#
# The full syntax of this command is:
#
#     mac-vrf <NAME>
#     no mac-vrf
# -------------------------------------------------------------------------------
macVrfNameMatcher = CliMatcher.PatternMatcher( r'[A-Za-z0-9_.:{}\[\]-]+',
                                               helpdesc='MAC-VRF name',
                                               helpname='WORD' )

class MacVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-vrf NAME'
   noOrDefaultSyntax = 'mac-vrf ...'
   data = {
      'mac-vrf': 'The MAC-VRF configuration',
      'NAME': macVrfNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args.get( 'NAME', '' )
      mode.vlan.setMacVrf( mode, name )

   noOrDefaultHandler = handler

if Toggles.EbraToggleLib.toggleEvpnVxlanOCSupportEnabled():
   VlanConfigMode.addCommandClass( MacVrfCmd )

#-------------------------------------------------------------------------------
# new syntax:
# The "[no] vlan internal order" command, in "config" mode.
#
# legacy:
# The "[no] vlan internal allocation policy" command, in "config" mode.
#
# The full syntax of this command is:
#
#   [no|default] vlan internal order {ascending | descending} [range
#                       VLAN_BEGIN VLAN_END ]
#-------------------------------------------------------------------------------

def setAllocationPolicy( mode, args ):
   vlanBeginId = args.get( 'VLAN_BEGIN', 1006 )
   vlanEndId = args.get( 'VLAN_END', 4094 )
   if vlanBeginId > vlanEndId:
      mode.addError( "Beginning VLAN ID must be less than ending VLAN ID" )
      return
   direction = args.get( 'DIRECTION', 'ascending' )
   cliConfig.internalVlanAllocationPolicy = direction
   cliConfig.internalVlanRange = Tac.Value(
                        'Bridging::InternalVlanRange', vlanBeginId, vlanEndId )

nodeVlan = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'vlan',
              helpdesc='VLAN commands' ),
              guard=bridgingSupportedGuard )

matcherAllocation = CliMatcher.KeywordMatcher( 'allocation',
                       helpdesc='Internal VLAN allocation' )

internalVlanIdBeginMatcher = CliMatcher.IntegerMatcher( 2, 4094,
      helpdesc='Beginning VLAN ID of the range', priority=CliParser.PRIO_HIGH )
internalVlanIdEndMatcher = CliMatcher.IntegerMatcher( 2, 4094,
      helpdesc='Ending VLAN ID of the range', priority=CliParser.PRIO_HIGH )

class VlanInternalOrderCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan internal order DIRECTION [ range VLAN_BEGIN VLAN_END ] '
   noOrDefaultSyntax = 'vlan internal order ...'
   data = {
      'vlan': nodeVlan,
      'internal': 'Internal VLAN',
      'order': 'Internal VLAN order',
      'DIRECTION':CliMatcher.EnumMatcher( {
         'ascending': 'Allocate internal VLANs from low to high',
         'descending': 'Allocate internal VLANs from high to low',
       } ),
      'range': 'Specify range of VLAN IDs',
      'VLAN_BEGIN': internalVlanIdBeginMatcher,
      'VLAN_END': internalVlanIdEndMatcher,
   }

   handler = setAllocationPolicy
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( VlanInternalOrderCmd )

class VlanInternalAllocationCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vlan internal allocation policy DIRECTION '
            '[ range VLAN_BEGIN VLAN_END ]' )
   noOrDefaultSyntax = 'vlan internal allocation policy ...'
   data = {
      'vlan': nodeVlan,
      'internal': 'Internal VLAN',
      'allocation': CliCommand.Node( matcher=matcherAllocation,
         deprecatedByCmd='vlan internal order' ),
      'policy': 'Internal VLAN allocation policy',
      'DIRECTION': CliMatcher.EnumMatcher( {
         'ascending': 'Allocate internal VLANs from low to high',
         'descending': 'Allocate internal VLANs from high to low',
       } ),
      'range': 'Specify range of VLAN IDs',
      'VLAN_BEGIN': internalVlanIdBeginMatcher,
      'VLAN_END': internalVlanIdEndMatcher,
   }

   handler = noOrDefaultHandler = setAllocationPolicy

BasicCliModes.GlobalConfigMode.addCommandClass( VlanInternalAllocationCmd )

#-------------------------------------------------------------------------------
# The "show vlan" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan [id <vlan_set>|name <vlan_name>|brief] [configure]
#
# where configure = [ active-configuration | configured-ports ]
#
#-------------------------------------------------------------------------------

configPortsMatcher = CliMatcher.KeywordMatcher( 'configured-ports',
                        helpdesc='Show all configured ports' )

activeConfigMatcher = CliMatcher.KeywordMatcher( 'active-configuration',
                         helpdesc='Show only active configuration' )

def emitUnknownVlanError( mode, vlanSpec ):
   """ Command attempted to use a non-existent VLAN ID, name, or
   set. vlanSpec is either a VLAN ID, a VLAN name, or a VLAN set. """
   mode.addError( 'VLAN %s not found in current VLAN database' % vlanSpec )

def doShowVlan( mode, args ):
   vlanSet = args.get( 'VLAN_SET' )
   name = args.get( 'NAME' )
   configured = 'configured-ports' in args
   vlansModel = Vlans()
   legends = set() # pylint: disable=unused-variable

   vlansModel.sourceDetail = bridgingConfig.vlanConfigSourceDetail

   def createIntfModel( intf, vlanId, isBlocked ):
      eis = ethIntfStatusDir.intfStatus.get( intf.name )
      intfModel = Vlans.Status.Interface()
      if ( configured and eis ):
         if eis.forwardingModel == 'intfForwardingModelLinkQualification':
            ( _reason, _, legendPhrase, annot ) = \
               intf.linkQualificationExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
         elif eis.forwardingModel in [ 'intfForwardingModelLayer1Patch',
               'intfForwardingModelLayer1PatchAndLldp' ]:
            ( _reason, _, legendPhrase, annot ) = \
               intf.layer1PatchExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
         elif eis.forwardingModel == 'intfForwardingModelRecirculation':
            ( _reason, _, legendPhrase, annot ) = \
               intf.recirculationExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
         elif eis.forwardingModel == 'intfForwardingModelDataLink':
            ( _reason, _, legendPhrase, annot ) = \
               intf.dataLinkExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
         elif eis.forwardingModel == 'intfForwardingModelQuietDataLink':
            ( _reason, _, legendPhrase, annot ) = \
               intf.quietDataLinkExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
         elif eis.forwardingModel == 'intfForwardingModelUnauthorized':
            ( _reason, _, legendPhrase, annot ) = \
               intf.unauthorizedExplanation( mode, intf.status() )
            if legendPhrase:
               intfModel.annotation = legendPhrase
               intfModel._annotationSymbol = annot
      vc = bridgingConfig.vlanConfig.get( vlanId )
      if vc:
         vpc = vc.intf.get( intf.name )
         if vpc and vpc.promotedPort:
            intfModel.privatePromoted = True

      if isBlocked:
         intfModel.blocked = True

      intfIdDisplayContext = IntfCli.intfIdDisplayContextHelper.entry.get(
         intf.name )
      if intfIdDisplayContext:
         intfModel.function = intfIdDisplayContext.shortContext
         intfModel.functionDetail = intfIdDisplayContext.fullContext

      return intfModel

   # getAll() returns an unordered list. Sorting it here allows the
   # generator to return the entries indexed by vlan id.
   vlans = sorted( Vlan.getAll( mode ) )

   if vlanSet is not None:
      vlans = [ vlan for vlan in vlans if vlan.id in vlanSet ]
      if not vlans:
         emitUnknownVlanError( mode, vlanSet )
         return vlansModel

   if name is not None:
      vlans = [ vlan for vlan in vlans if vlan.name( mode ) == name ]
      if not vlans:
         emitUnknownVlanError( mode, name )
         return vlansModel

   dynVlanSet = Vlan.getDynVlanSet( mode )

   #----------------------------------------------------------------------------
   # Generator function which returns a vlan-id and the list of ports which
   # have that vlan. Since the list of "vlans" is already sorted, this
   # generator will return the vlans in order.
   #----------------------------------------------------------------------------
   def vlanToPortMapGenerator():
      for v in vlans:
         dynamic = v.id in dynVlanSet and not v.lookupConfig( mode )
         blocked = v.blockedPorts( mode )

         vlan = Vlans.Status( name=v.name( mode ), dynamic=dynamic,
                              status=v.adminState( mode ) )

         ports = v.configuredPorts( mode ) if configured else v.activePorts( mode )

         ports.sort()
         for intf in ports:
            vlan.interfaces[ intf.name ] = createIntfModel(
               intf, v.id, intf.name in blocked )

         yield v.id, vlan

         TacSigint.check()

   vlansModel.vlans = vlanToPortMapGenerator()

   return vlansModel

# BUG191 "show vlan brief" is hidden, since it is currently identical to "show vlan".

class VlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vlan '
              '[ ( name NAME ) | ( [ id ] VLAN_SET ) ] '
              '[ configured-ports | active-configuration ]' )
   data = {
      'vlan': vlanMatcher,
      'name': 'Show the status of a specific named VLAN',
      'NAME': vlanNameMatcher,
      'id': 'Show the status of a specific VLAN ID',
      'VLAN_SET': vlanSetMatcher,
      'configured-ports': configPortsMatcher,
      'active-configuration': activeConfigMatcher,
   }

   handler = doShowVlan
   cliModel = Vlans

BasicCli.addShowCommandClass( VlanCmd )

class VlanBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vlan brief [ configured-ports | active-configuration ]' )
   data = {
      'vlan': vlanMatcher,
      'brief': 'VLAN summary information',
      'configured-ports': configPortsMatcher,
      'active-configuration': activeConfigMatcher,
   }

   hidden = True
   handler = doShowVlan
   cliModel = Vlans

BasicCli.addShowCommandClass( VlanBriefCmd )

#-------------------------------------------------------------------------------
# The "show vlan trunk group" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan [[id] <vlan_set> | name <vlan_name>] trunk group
#-------------------------------------------------------------------------------
def doShowVlanTrunkGroup( mode, args ):
   vlanSet = args.get( 'VLAN_SET' )
   name = args.get( 'VLAN_NAME' )
   result = VlanTrunkGroups()
   if bridgingConfig.vlanConfigSourceDetail:
      result.sourceDetail = bridgingConfig.vlanConfigSourceDetail
   vlans = Vlan.getAll( mode )

   if vlanSet is not None:
      vlans = [ v for v in vlans if v.id in vlanSet ]
      if not vlans:
         emitUnknownVlanError( mode, vlanSet )
         return result

   if name is not None:
      vlans = [ v for v in vlans if v.name( mode ) == name ]
      if not vlans:
         emitUnknownVlanError( mode, name )
         return result

   for v in vlans:
      trunkGroups = v.trunkGroups( mode )
      trunkGroups.sort()
      result.trunkGroups[ v.id ] = VlanTrunkGroups.GroupNames( names=trunkGroups )
      TacSigint.check()

   return result

class ShowVlanTrunkGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan [ ( [ id ] VLAN_SET ) | ( name VLAN_NAME ) ] trunk group'
   data = {
      'vlan': vlanMatcher,
      'id': 'Show the status of a specific VLAN ID',
      'VLAN_SET': vlanSetMatcher,
      'name': 'ASCII name of the VLAN',
      'VLAN_NAME': vlanNameMatcher,
      'trunk': 'VLAN trunk information',
      'group': 'Trunk group information',
      }

   handler = doShowVlanTrunkGroup
   cliModel = VlanTrunkGroups

BasicCli.addShowCommandClass( ShowVlanTrunkGroupCmd )

#-------------------------------------------------------------------------------
# [no] mac address forwarding {unicast | multicast} miss action {drop | flood | log}
# in "config-vlan" mode
#
# The full syntax of this command is:
#
#  mac address forwarding {unicast | multicast} miss action {drop | flood | log}
#  no mac address forwarding {unicast | multicast} miss action {drop | flood | log}
#-------------------------------------------------------------------------------
def vlanMacPolicySupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanL2MissPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def vlanMacLogPolicySupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanL2MissLogPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

matcherPolicyType = CliMatcher.EnumMatcher( {
                       'unicast': 'Configure unicast miss action for VLAN',
                       'multicast': 'Configure multicast miss action for VLAN',
          } )
matcherPolicyAction = CliMatcher.EnumMatcher( {
                         'drop': 'Drop L2 miss packets',
                         'flood': 'Flood L2 miss packets',
          } )
nodeForwarding = CliCommand.guardedKeyword( 'forwarding',
                    helpdesc='Configure mac address fowarding property of the VLAN',
                    guard=vlanMacPolicySupportedGuard )
nodePolicyLog = CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'log', helpdesc='Log L2 miss packets' ),
         guard=vlanMacLogPolicySupportedGuard,
         alias='ACTION_TYPE' )

class MacAddressFrwdUniOrMulticastCmd( CliCommand.CliCommandClass ):
   syntax = 'mac address forwarding ADDR_TYPE miss action ( ACTION_TYPE | log )'
   noOrDefaultSyntax = syntax
   data = {
         'mac': nodeMac,
         'address': matcherAddress,
         'forwarding': nodeForwarding,
         'ADDR_TYPE': matcherPolicyType,
         'miss': 'Configure unicast miss action for VLAN',
         'action': 'Configure unicast miss action for VLAN',
         'ACTION_TYPE': matcherPolicyAction,
         'log': nodePolicyLog,
   }

   handler = VlanConfigMode.setMissPolicy
   noOrDefaultHandler = VlanConfigMode.noMissPolicy

VlanConfigMode.addCommandClass( MacAddressFrwdUniOrMulticastCmd )

#-------------------------------------------------------------------------------
# [no] floodset expanded vlan <v>
# in "config-vlan" mode
#
# Expand vlan floodset to include ports of a given vlan
# The full syntax of this command is:
#
# floodset expanded vlan <v>
# no floodset expanded vlan
#-------------------------------------------------------------------------------
class floodsetExpandedVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'floodset expanded vlan VLAN'
   noOrDefaultSyntax = 'floodset expanded vlan ...'
   data = {
      'floodset': 'floodset configuration',
      'expanded' : 'floodset expanded configuration ',
      'vlan' : 'floodset expanded to include ports of another vlan ',
      'VLAN' : vlanIdMatcher
   }

   @staticmethod
   def handler( mode, args ):
      default = VlanConfig.defaultFloodsetExpandedVlanId
      vlanId = getattr( args.get( 'VLAN' ), 'id', default )
      for config in mode.vlanConfigs():
         config.floodsetExpandedVlanId = vlanId

   noOrDefaultHandler = handler

VlanConfigMode.addCommandClass( floodsetExpandedVlanCmd )

class showFloodsetExpandedVlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vlan [ ( [ id ] VLAN_SET ) | ( name VLAN_NAME ) ] '
              'floodset expanded' )
   data = {
      'vlan': 'Details on VLAN operation',
      'id': 'Show the status of a specific VLAN ID',
      'VLAN_SET': vlanSetMatcher,
      'name': 'ASCII name of the VLAN',
      'VLAN_NAME': vlanNameMatcher,
      'floodset': 'Show floodset configuration of the VLAN',
      'expanded': 'Show floodset expanded configuration of the VLAN'
   }

   cliModel = FloodsetExpandedVlanTableModel

   @staticmethod
   def handler( mode, args ): # pylint: disable=inconsistent-return-statements
      vlanFloodsetExpandedStatus = (
            vlanFloodsetExpandedStatusDir.vlanFloodsetExpandedStatus )

      vlanSet = args.get( 'VLAN_SET' )
      name = args.get( 'VLAN_NAME' )

      vlans = Vlan.getAll( mode )
      vlans.sort()
      if vlanSet:
         vlans = [ v for v in vlans if v.id in vlanSet ]
         if not vlans:
            emitUnknownVlanError( mode, vlanSet )
            return
      elif name:
         vlans = [ v for v in vlans if v.name( mode ) == name ]
         if not vlans:
            emitUnknownVlanError( mode, name )
            return

      reasonToModel = {
         VlanFloodsetExpandedReason.floodsetExpandedVlanValid: 'valid',
         VlanFloodsetExpandedReason.floodsetExpandedVlanNonexistent: 'nonexistent',
         VlanFloodsetExpandedReason.floodsetExpandedVlanConfigConflict: 'conflict',
      }
      model = FloodsetExpandedVlanTableModel()
      for vlan in vlans:
         vlanConfig = cliConfig.vlanConfig.get( vlan.id )
         if not vlanConfig or not vlanConfig.floodsetExpandedVlanId:
            continue
         status = vlanFloodsetExpandedStatus.get( vlan.id )
         if not status:
            continue
         vlanModel = FloodsetExpandedVlanModel()
         vlanModel.floodsetExpandedVlanId = vlanConfig.floodsetExpandedVlanId
         vlanModel.reason = reasonToModel[ status.reason ]
         model.vlans[ vlan.id ] = vlanModel

      return model

BasicCli.addShowCommandClass( showFloodsetExpandedVlanCmd )

#-------------------------------------------------------------------------------
# The "show vlan mac address forwarding" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan [[id] <vlan_set> | name <vlan_name>] mac address forwarding
#-------------------------------------------------------------------------------

nodeMacPolShow = CliCommand.guardedKeyword( 'mac',
                    helpdesc='Show MAC property of the VLAN',
                    guard=vlanMacPolicySupportedGuard )

def doShowVlanPolicy( mode, args ):
   vlanSet = args.get( 'VLAN_SET' )
   name = args.get( 'VLAN_NAME' )
   if bridgingConfig.vlanConfigSourceDetail:
      print( bridgingConfig.vlanConfigSourceDetail )
   vlans = Vlan.getAll( mode )
   vlans.sort()

   if vlanSet:
      vlans = [ v for v in vlans if v.id in vlanSet ]
      if not vlans:
         emitUnknownVlanError( mode, vlanSet )
         return

   if name :
      vlans = [ v for v in vlans if v.name( mode ) == name ]
      if not vlans:
         emitUnknownVlanError( mode, name )
         return

   fmt = '%4d  %-12s  %-12s'

   print()
   print( 'VLAN  UcMissAction  McMissAction' )
   print( '----  ------------  ------------' )

   for v in vlans:
      vlanCfg = bridgingConfig.vlanConfig.get( v.id )
      policy = vlanCfg.l2MissPolicyConfig
      print( fmt % ( v.id, policy.l2UcMissPolicy, policy.l2McMissPolicy ) )

class VlanMacAddressForwardingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show vlan [ ( [ id ] VLAN_SET ) | ( name VLAN_NAME ) ] '
              'mac address forwarding' )
   data = {
      'vlan': 'Details on VLAN operation',
      'id': 'Show the status of a specific VLAN ID',
      'VLAN_SET': vlanSetMatcher,
      'name': 'ASCII name of the VLAN',
      'VLAN_NAME': vlanNameMatcher,
      'mac': nodeMacPolShow,
      'address': 'Show MAC address property of the VLAN',
      'forwarding': 'Show MAC address fowarding property of the VLAN',
   }

   handler = doShowVlanPolicy

BasicCli.addShowCommandClass( VlanMacAddressForwardingCmd )

#-------------------------------------------------------------------------------
# The "show interfaces <intf list> vlans" command
#-------------------------------------------------------------------------------
class InterfaceVlansModel( Model ):
   _shortname = Str( help='Shortname of interface for CLI output' )
   untaggedVlan = Int( help='Untagged VLAN for an interface', optional=True )
   taggedVlans = List( valueType=int, help='Tagged VLANs for an interface',
                       optional=True )

   def renderInterfaceVlans( self ):
      if not self.untaggedVlan:
         if not self.taggedVlans:
            untagged = '-'
         else:
            untagged = "None"
      else:
         untagged = self.untaggedVlan

      padding = "%-10s %-8s" % ( self._shortname, untagged )
      if self.taggedVlans:
         entries = vlanSetToCanonicalStringGen( self.taggedVlans, 50 )
      else:
         entries = ( '-', )
      for s in entries:
         print( "%-19s %-s" % ( padding, s ) )
         padding = ""

class ShowInterfacesVlanModel( Model ):
   interfaces = Dict( keyType=str, valueType=InterfaceVlansModel,
                 help="Map interfaces to VLANs", optional=True )
   sourceDetail = Str( help=
                       'Description of the source of the VLAN configs',
                       optional=True )

   def render( self ):

      if self.sourceDetail:
         print( self.sourceDetail )

      if not self.interfaces:
         print( "Specified interface(s) carry no VLANs" )
         return

      print( "%-10s %8s %-s" % ( "Port", "Untagged", "Tagged" ) )
      for i in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ i ].renderInterfaceVlans( )

def doShowInterfacesVlan( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfType = ( EthIntfCli.EthIntf, IntfCli.VxlanVirtualIntf )
   intfs = IntfCli.Intf.getAll( mode, intf, mod, intfType=intfType )
   if mode.session.hasError():
      return None
   if not intfs:
      return ShowInterfacesVlanModel( interfaces=None, sourceDetail=None )

   if bridgingConfig.vlanConfigSourceDetail:
      configSource = bridgingConfig.vlanConfigSourceDetail
   else:
      configSource = None

   # Invert the topoPortConfig.
   intfVlanSet = {}
   for ( vid, tc ) in topologyConfig.vidToTopoMap.items():
      vc = bridgingConfig.vlanConfig.get( vid )
      if not vc:
         continue
      intfSet = set( tc.topoPortConfig ).intersection( set( vc.intf ) )
      for intf in intfSet:
         intfVlanSet.setdefault( intf, set() ).add( vid )

   interfaces = {}
   native = None

   for i in intfs: # pylint: disable=too-many-nested-blocks
      if i.name not in intfVlanSet:
         # Skip interfaces that are not members of any VLANs.  This
         # includes down interfaces, routed interfaces, mirror
         # destinations, port-channel members, etc.
         continue
      switchIntfConfig = bridgingSwitchIntfConfig.switchIntfConfig.get( i.name )
      tagged = set()
      if switchIntfConfig and switchIntfConfig.enabled:
         if switchIntfConfig.switchportMode == 'trunk':
            native = None if switchIntfConfig.trunkNativeVlan == 0 else \
                                      switchIntfConfig.trunkNativeVlan
            if i.name in intfVlanSet:
               tagged = intfVlanSet[ i.name ]
               if native:
                  if switchIntfConfig.trunkNativeVlan in tagged:
                     tagged.remove( switchIntfConfig.trunkNativeVlan )
                  else:
                     native = None
         elif switchIntfConfig.switchportMode in ( 'access', 'dot1qTunnel' ):
            native = switchIntfConfig.accessVlan
         elif switchIntfConfig.switchportMode in ( 'tap', 'tapTool' ):
            native = switchIntfConfig.tapNativeVlan
         elif switchIntfConfig.switchportMode == 'tool':
            native = None
            tagged.clear()
      else:
         # Skip routed interfaces altogether.
         continue
      vlans = InterfaceVlansModel( _shortname=i.shortname,
                                   untaggedVlan=int( native ) if native else None,
                                   taggedVlans=list( tagged ) if tagged else None )
      interfaces[ i.name ] = vlans
   return ShowInterfacesVlanModel( interfaces=interfaces,
                                   sourceDetail=configSource )


class ShowIntfVlans( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces vlans'
   data = {
      'vlans': 'Details on VLAN interfaces',
   }
   cliModel = ShowInterfacesVlanModel
   handler = doShowInterfacesVlan

BasicCli.addShowCommandClass( ShowIntfVlans )

#------------------------------------------------------------------------------
# show interfaces [<name>] trunk
#
# show interfaces module <modnum> trunk
# show interfaces trunk module <modnum>
# ----------------------------------------------------------------------------

def doShowTrunk( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   showPhoneTrunk = "phone" in args
   trunkInterfaces = TrunkInterfaces()
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   if not intfs:
      trunkInterfaces._noIntfs = True
      return trunkInterfaces

   # Note that if no interface name is specified, we only show the interfaces that
   # are configured as trunks and are active.  However, if an interface name is
   # specified, we show that interface regardless of whether it is configured
   # as a trunk.
   if not intf and not mod :
      def intfIsTrunkport( i ):
         switchIntfConfig = switchIntfConfigIfEnabled( i, checkTrunk=True )
         return switchIntfConfig and switchIntfConfig.phoneTrunk == showPhoneTrunk

      intfs = [ x for x in intfs if intfIsTrunkport( x ) ]
      if not intfs:
         return trunkInterfaces

   # Invert the topoPortConfig and the mst information.
   intfVlanSet = {}
   mstVlanSet = {}
   for ( vid, tc ) in topologyConfig.vidToTopoMap.items():
      vc = bridgingConfig.vlanConfig.get( vid )
      if not vc:
         continue
      tpc = tc.topoPortConfig
      intfSet = set( tpc ).intersection( set( vc.intf ) )
      for intf in intfSet:
         intfVlanSet.setdefault( intf, set() ).add( vid )
         # Since we do not snapshot tpc, it is possible that it has changed
         # while we are performing iteration. To account for this, we first
         # check if intf is present in tpc.
         tpcIntf = tpc.get( intf )
         if not tpcIntf:
            continue
         if tpcIntf.state == 'forwarding':
            mstVlanSet.setdefault( intf, set() ).add( vid )

   def _getTrunkAndVlans( intf, vlans ):
      interfaceTrunk = trunkInterfaces.trunks.get( intf.name )
      if not interfaceTrunk:
         interfaceTrunk = InterfaceTrunk()
         trunkInterfaces.trunks[ intf.name ] = interfaceTrunk
      vlans = TrunkVlans() if vlans else None
      return interfaceTrunk, vlans

   for i in intfs:
      switchIntfConfig = switchIntfConfigIfEnabled( i )
      if switchIntfConfig is None:
         continue
      if switchIntfConfig.switchportMode == 'trunk':
         native = switchIntfConfig.trunkNativeVlan
      elif switchIntfConfig.switchportMode in ( 'access', 'dot1qTunnel' ):
         native = switchIntfConfig.accessVlan
      elif switchIntfConfig.switchportMode == 'tap' or \
            switchIntfConfig.switchportMode == 'tapTool':
         native = switchIntfConfig.tapNativeVlan
      elif switchIntfConfig.switchportMode == 'tool':
         native = 0

      linkStatus = 'nonTrunking'
      if i.linkStatus() == 'disabled':
         linkStatus = 'down'
      elif switchIntfConfig.switchportMode == 'trunk':
         linkStatus = 'trunking'

      interfaceTrunk, _ = _getTrunkAndVlans( i, False )
      interfaceTrunk.portMode = switchIntfConfig.switchportMode
      interfaceTrunk.phoneTrunk = switchIntfConfig.phoneTrunk
      interfaceTrunk.nativeVlan = native
      interfaceTrunk.linkStatus = linkStatus

   # Allowed ports information
   for i in intfs:
      switchIntfConfig = switchIntfConfigIfEnabled( i )
      if switchIntfConfig is None:
         continue
      interfaceTrunk, vlans = _getTrunkAndVlans( i, True )
      if switchIntfConfig.switchportMode in [ 'trunk', 'tap', 'tool' ]:
         if switchIntfConfig.switchportMode == 'trunk':
            allowedVlans = switchIntfConfig.trunkAllowedVlans
         elif switchIntfConfig.switchportMode == 'tap':
            allowedVlans = switchIntfConfig.tapAllowedVlans
         else:
            allowedVlans = switchIntfConfig.toolAllowedVlans
         if not allowedVlans:
            vlans.vlanIds = [ 0 ]
         elif allowedVlans == '1-4094':
            vlans.allVlans = True
         else:
            vSet = computeVlanRangeSet( allowedVlans )
            vlans.vlanIds = list( vSet )
         interfaceTrunk.allowedVlans = vlans
      elif switchIntfConfig.switchportMode in ( 'access', 'dot1qTunnel' ):
         vlans.vlanIds = [ switchIntfConfig.accessVlan ]
         interfaceTrunk.allowedVlans = vlans

   # Active ports information
   for i in intfs:
      if i.linkStatus() == 'disabled' or i.name not in intfVlanSet:
         continue
      interfaceTrunk, vlans = _getTrunkAndVlans( i, True )
      tagged = intfVlanSet[ i.name ]
      if len( tagged ) == 4094:
         vlans.allVlans = True
      else:
         vlans.vlanIds = list( tagged )
      interfaceTrunk.activeVlans = vlans

   # Spanning tree information
   for i in intfs:
      if i.linkStatus() == 'disabled' or i.name not in mstVlanSet:
         continue
      interfaceTrunk, vlans = _getTrunkAndVlans( i, True )
      tagged = mstVlanSet[ i.name ]
      if len( tagged ) == 4094:
         vlans.allVlans = True
      else:
         vlans.vlanIds = list( tagged )
      interfaceTrunk.forwardingVlans = vlans
   return trunkInterfaces


# If the SwitchInterfaceConfig for the given interface is present and not
# disabled, return it.  Otherwise return None.
def switchIntfConfigIfEnabled( intf, checkTrunk=False ):
   switchIntfConfig = bridgingSwitchIntfConfig.switchIntfConfig.get( intf.name )
   enabled = switchIntfConfig and switchIntfConfig.enabled
   if checkTrunk:
      enabled = ( enabled and switchIntfConfig.switchportMode == 'trunk' and
                  intf.linkStatus() != 'disabled' )
   if enabled:
      return switchIntfConfig
   else:
      return None

class ShowIntfTrunk( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces trunk [ phone ] '
   data = {
      'trunk': 'Show interface trunk information',
      'phone': 'Show interface trunk phone information',
   }
   handler = doShowTrunk
   moduleAtEnd = True
   cliModel = TrunkInterfaces

BasicCli.addShowCommandClass( ShowIntfTrunk )

#-------------------------------------------------------------------------------
# The "show vlan mtu" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan mtu
#-------------------------------------------------------------------------------
def doShowVlanMtu( mode, args ):
   if bridgingConfig.vlanConfigSourceDetail:
      print( bridgingConfig.vlanConfigSourceDetail )
   vlans = Vlan.getAll( mode )
   vlans.sort()

   fmt = '%-4d %-13d %-14d %-14d %s'

   print()
   print( 'VLAN    SVI_MTU    MinMTU(port)   MaxMTU(port)  MTU_Mismatch' )
   print( '---- ------------- -------------  ------------  ------------' )

   for v in vlans:
      # BUG191 These values shouldn't all be hard-coded.
      print( fmt % ( v.id, 1500, 1500, 1500, 'No' ) )

class VlanMtuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan mtu'
   data = {
         'vlan': vlanMatcher,
         'mtu': 'VLAN MTU information',
   }
   handler = doShowVlanMtu

#BasicCli.addShowCommandClass( VlanMtuCmd )
# BUG191 This command is removed until it displays something sensible.

#-------------------------------------------------------------------------------
# The "show vlan brief count" command, in "enable" mode.
#
# The full syntax of this command is:
#   show vlan brief count
#
# legacy:
#   show vlan summary
#-------------------------------------------------------------------------------
matcherSummaryLegacy = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                          'summary', helpdesc='VLAN summary information' ),
                          deprecatedByCmd='show vlan brief count' )

def doShowVlanSummary( mode, args ):
   return VlanSummary( sourceDetail=bridgingConfig.vlanConfigSourceDetail,
                       totalVlans=len( Vlan.getAll( mode ) ) )

class ShowVlanSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan brief count'
   data = {
      'vlan' : vlanMatcher,
      'brief' : 'VLAN summary information',
      'count' : 'VLAN summary count information',
      }

   handler = doShowVlanSummary
   cliModel = VlanSummary

BasicCli.addShowCommandClass( ShowVlanSummaryCmd )

#-----------------------------------------------------------
#Show vlan summary
#-----------------------------------------------------------

class VlanSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan summary'
   data = {
      'vlan' : vlanMatcher,
      'summary' : matcherSummaryLegacy,
      }

   handler = doShowVlanSummary
   cliModel = VlanSummary

BasicCli.addShowCommandClass( VlanSummaryCmd )

def switchportEligible( mode ):
   """Return True if the interface being configured is allowed to be a switchport.
      If the interface has a switchportEligible method, use it; otherwise assume
      'no'."""
   return getattr( mode.intf, 'switchportEligible', bool )() # bool() => False.

#-------------------------------------------------------------------------------
# The "show vlan dynamic" command
#
# The full syntax of this command is:
#
#   show vlan dynamic
#-------------------------------------------------------------------------------
def doShowDynVlans( mode, args ):
   ret = VlanDynamic()
   def addDynVlan( agent, dynVlanSet ):
      vlanIds = VlanIds()
      if dynVlanSet.vlans:
         vlanIds.vlanIds = list( computeVlanRangeSet( dynVlanSet.vlans ) )
      ret.dynamicVlans[ agent ] = vlanIds

   for agent, dynVlanSet in dynVlanDir.items():
      addDynVlan( agent, dynVlanSet )
   addDynVlan( "mlag", peerVlanSet )
   return ret

class ShowDynVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan dynamic'
   data = {
      'vlan' : vlanMatcher,
      'dynamic' : 'Show dynamic VLANs',
      }

   handler = doShowDynVlans
   cliModel = VlanDynamic

BasicCli.addShowCommandClass( ShowDynVlansCmd )

#-------------------------------------------------------------------------------
# The "show vlan interface dynamic" command
#
# The full syntax of this command is:
#
#   show vlan interface dynamic
#-------------------------------------------------------------------------------
def doShowDynIntfVlans( mode, args ):
   ret = VlanInterfaceDynamic()
   def addDynVlans( agent, dynVlanIntfSet ):
      intfs = VlanIntfs()
      for intfName, intfEntity in dynVlanIntfSet.intfConfig.items():
         if intfEntity.dynamic == True: # pylint: disable=singleton-comparison
            intfs.interfaces.append( intfName )
      ret.agents[ agent ] = intfs

   for agent, dynVlanIntfSet in dynVlanIntfDir.items():
      addDynVlans( agent, dynVlanIntfSet )
   return ret

class ShowDynIntfVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan interface dynamic'
   data = {
      'vlan' : vlanMatcher,
      'interface' : 'Show VLANs interface status',
      'dynamic' : 'Show dynamic VLANs ',
      }

   handler = doShowDynIntfVlans
   cliModel = VlanInterfaceDynamic

BasicCli.addShowCommandClass( ShowDynIntfVlansCmd )

#-------------------------------------------------------------------------------
# The "show vlan internal allocation policy" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan internal allocation policy
#-------------------------------------------------------------------------------

def doShowInternalVlanAllocationPolicy( mode, args ):
   ret = VlanInternalAllocationPolicy()
   ret.policy = cliConfig.internalVlanAllocationPolicy
   ret.startVlanId = cliConfig.internalVlanRange.vlanBegin
   ret.endVlanId = cliConfig.internalVlanRange.vlanEnd
   return ret

class InternalVlanAllocationPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan internal allocation policy'
   data = {
         'vlan' : vlanMatcher,
         'internal' : 'Internal VLAN',
         'allocation' : 'Internal VLAN allocation',
         'policy' :  'Internal VLAN allocation policy',
         }
   handler = doShowInternalVlanAllocationPolicy
   cliModel = VlanInternalAllocationPolicy

BasicCli.addShowCommandClass( InternalVlanAllocationPolicyCmd )

#-------------------------------------------------------------------------------
# The "show vlan internal usage" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan internal usage
#-------------------------------------------------------------------------------

def doShowInternalVlanUsage( mode, args ):
   ret = VlanInternalUsage()
   for v in Vlan.getAll( mode, internal=True ):
      ret.internalVlans[ v.id ] = v.name( mode )
   return ret

class InternalVlanUsageCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan internal usage'
   data = {
         'vlan' : vlanMatcher,
         'internal' : 'Internal VLAN',
         'usage' : 'VLAN internal usage',
         }
   handler = doShowInternalVlanUsage
   cliModel = VlanInternalUsage

BasicCli.addShowCommandClass( InternalVlanUsageCmd )

#-------------------------------------------------------------------------------
# The "show vlan internal usage reserved" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan internal usage reserved
#-------------------------------------------------------------------------------

def doShowInternalVlanUsageReserved( mode, args ):
   ret = VlanInternalUsageReserved()
   for vlanId, intfName in Vlan.getReservedMapping( mode ).items():
      ret.internalVlans[ vlanId ] = intfName
   return ret

class InternalVlanUsageReserved( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan internal usage reserved'
   data = {
         'vlan' : vlanMatcher,
         'internal' : 'Internal VLAN',
         'usage' : 'VLAN internal usage',
         'reserved' : 'VLAN internal usage reserved',
         }
   handler = doShowInternalVlanUsageReserved
   cliModel = VlanInternalUsageReserved

BasicCli.addShowCommandClass( InternalVlanUsageReserved )

#-------------------------------------------------------------------------------
# The "show vlan [vlanId | vlanRange] counters" command
# The "clear vlan [vlanId | vlanRange] counters" command
#-------------------------------------------------------------------------------

def countersGuard( mode, token ):
   if bridgingHwCapabilities.vlanCountersSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

matcherCounters = CliCommand.guardedKeyword( 'counters',
      helpdesc='VLAN counters information', guard=countersGuard )

matcherId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Show the status of a specific VLAN' )

def getVlans( vlanSet=None ):
   inactiveVlans = []
   vlanIds = []
   if vlanSet is not None:
      for vlanId in vlanSet:
         if vlanCounterDir.vlanCounter.get( vlanId ):
            vlanIds.append( vlanId )
         else:
            inactiveVlans.append( vlanId )
   else:
      vlanIds = list( vlanCounterDir.vlanCounter )
   return ( inactiveVlans, vlanIds )

def doShowVlanCounters( mode, args ):
   vlanSet = args.get( 'VLANS' )
   inactiveVlans, vlanIds = getVlans( vlanSet )
   vlanCountersInfo = {}
   for vlanId in vlanIds:
      snapshot = Vlan( vlanId ).getLatestCounterSnapshot( mode )
      vlanCountersInfo[ vlanId ] = Vlan( vlanId ).updateVlanCountersModel(
            snapshot=snapshot )
   displayIngressCounters = bridgingHwEnabled.vlanIngressCountersEnabled
   displayEgressCounters = bridgingHwEnabled.vlanEgressCountersEnabled
   displayBumCounters = bridgingHwEnabled.vlanBumCountersEnabled
   return VlanCounters( _displayIngressCounters=displayIngressCounters,
                        _displayEgressCounters=displayEgressCounters,
                        _displayBumCounters=displayBumCounters,
                        _inactiveVlans=inactiveVlans,
                        vlanCountersInfo=vlanCountersInfo )

def clearCounters( mode, args ):
   vlanSet = args.get( 'VLANS' )
   _, vlanIds = getVlans( vlanSet )
   request = vlanCountersCliReq.clearCountersRequest

   if vlanSet is None:
      Intf.Log.logClearCounters( "", "all VLANs" )
   elif vlanIds:
      if len( vlanIds ) > 1:
         vlanIds.sort()
         Intf.Log.logClearCounters( "", "VLANs %d to %d" % ( vlanIds[ 0 ],
                                                             vlanIds[ -1 ] ) )
      else:
         Intf.Log.logClearCounters( "", "VLAN%d" % vlanIds[ 0 ] )

   # Clear vlan list
   for vlanId in vlanIds:
      del request[ vlanId ]
      request.add( vlanId )

def clearCountersSession( mode, args ):
   vlanSet = args.get( 'VLANS' )
   _, vlanIds = getVlans( vlanSet )

   for vlanId in vlanIds:
      Vlan( vlanId ).clearCounters( mode )

class VlanVlansetCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan [ [ id ] VLANS ] counters'
   data = {
      'vlan': vlanMatcher,
      'id': matcherId,
      'VLANS': vlanSetMatcher,
      'counters': matcherCounters,
   }
   handler = doShowVlanCounters
   cliModel = VlanCounters

BasicCli.addShowCommandClass( VlanVlansetCountersCmd )

class ClearVlanVlansetCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear vlan [ [ id ] VLANS ] counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'vlan': vlanMatcher,
      'id': matcherId,
      'VLANS': vlanSetMatcher,
      'counters': matcherCounters,
   }
   handler = clearCounters

BasicCliModes.EnableMode.addCommandClass( ClearVlanVlansetCountersCmd )

class ClearVlanVlansetCountersSessionCmd( CliCommand.CliCommandClass ):
   syntax = 'clear vlan [ [ id ] VLANS ] counters session'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'vlan': vlanMatcher,
      'id': 'matcherId',
      'VLANS': vlanSetMatcher,
      'counters': matcherCounters,
      'session': IntfCli.counterSessionKw,
   }
   handler = clearCountersSession

BasicCliModes.EnableMode.addCommandClass( ClearVlanVlansetCountersSessionCmd )

#-------------------------------------------------------------------------------
# Adds switchport CLI commands to the "config-if" mode.
#---------------------------------------------------------------------------
class SwitchportModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthIntfCli.EthIntf ) and
               switchportEligible( mode ) )

#-------------------------------------------------------------------------------
# The "[no] switchport" command, in "config-if" mode.
#-------------------------------------------------------------------------------
# switchportCmdHook extensions accept 3 arguments: the CLI mode, the
# interface name and the no state, and returns True if the switch port
# configuration is allowed on the interface and False otherwise.
switchportCmdHook = CliExtensions.CliHook()

def setSwitchport( mode, args ):
   for hook in switchportCmdHook.extensions():
      if not hook( mode, mode.intf.name, no=False ):
         return
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   sic.enabled = True

def noSwitchport( mode, args ):
   for hook in switchportCmdHook.extensions():
      if not hook( mode, mode.intf.name, no=True ):
         return

   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   sic.enabled = False

class SwitchportCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': switchportMatcher,
      }

   handler = defaultHandler = setSwitchport
   noHandler = noSwitchport

SwitchportModelet.addCommandClass( SwitchportCmd )

#-------------------------------------------------------------------------------
# The "[no|default] switchport mode {access|trunk|dot1q-tunnel}"
# command, in "config-if" mode.
#-------------------------------------------------------------------------------
def dot1qTunnelSupportedGuard( mode, token ):
   if bridgingHwCapabilities.dot1qTunnelSupported:
      return None
   return CliParser.guardNotThisPlatform

class SwitchIntfConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing entries from the cliConfig's
   switchIntfConfig collection when the interfaces associated with those
   switchIntfConfigs are removed."""

   def setDefault( self ):
      self.removeSwitchIntfConfigs()

   def removeSwitchIntfConfigs( self ):
      # this delete is safe even if self.intfName_ is not present in
      # the collection because it is a tacc collection, i.e. no KeyError will
      # be generated
      del cliConfig.switchIntfConfig[ self.intf_.name ]

def getSwitchIntfConfigEvenIfDisabled( mode ):
   return getSwitchInputConfig( mode.intf.name )

# switchportModeCmdHook determines validity of requested config
# switchportModeCmdHook extensions currently accept 3 arguments
# arguments in order: CLI mode, interface name, switchportMode(to be set)
# extensions return True if configuration is allowed, and False if not
switchportModeCmdHook = CliExtensions.CliHook()

modeNode = CliCommand.guardedKeyword( 'mode',
      helpdesc='Set trunking mode of the interface',
      guard=bridgingSupportedGuard )

def setSwitchportMode( mode, args ):
   switchportMode = 'access'
   if 'trunk' in args:
      switchportMode = 'trunk'
   elif 'dot1q-tunnel' in args:
      switchportMode = 'dot1qTunnel'
   for hook in switchportModeCmdHook.extensions():
      if not hook( mode, mode.intf.name, switchportMode ):
         return
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.phoneTrunk = False
   sic.switchportMode = switchportMode

class SwitchportModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport mode ( access | trunk | dot1q-tunnel )'
   noOrDefaultSyntax = 'switchport mode ...'
   data = {
      'switchport': switchportMatcher,
      'mode':  modeNode,
      'access': 'Set trunking mode to ACCESS unconditionally',
      'trunk': 'Set trunking mode to TRUNK unconditionally',
      'dot1q-tunnel': CliCommand.guardedKeyword( 'dot1q-tunnel',
                          helpdesc='Set trunking mode to DOT1Q-TUNNEL '
                          'unconditionally',
                          guard=dot1qTunnelSupportedGuard ),
      }

   handler = setSwitchportMode
   noOrDefaultHandler = handler

SwitchportModelet.addCommandClass( SwitchportModeCmd )

#-------------------------------------------------------------------------------
# The "switchport mode trunk phone" command, in "config-if" mode.
#
# switchport mode trunk phone
#-------------------------------------------------------------------------------

def macBasedVlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.macBasedVlanSupported:
      return None
   return CliParser.guardNotThisPlatform

def mbvaScopeGuard( mode, token ):
   if bridgingHwCapabilities.macBasedVlanAssignmentScopeSupported:
      return None
   return CliParser.guardNotThisPlatform

phoneNode = CliCommand.guardedKeyword( 'phone',
   helpdesc='Set trunking mode to phone trunk unconditionally',
   guard=macBasedVlanSupportedGuard )

def setSwitchportModePhoneTrunk( mode, args ):
   switchportMode = 'trunk'
   for hook in switchportModeCmdHook.extensions():
      if not hook( mode, mode.intf.name, switchportMode ):
         return
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.phoneTrunk = True
   sic.switchportMode = switchportMode

class SwitchportModeTrunkPhoneCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport mode trunk phone'
   data = {
      'switchport': switchportMatcher,
      'mode': CliCommand.guardedKeyword( 'mode',
         helpdesc='Set trunking mode of the interface',
         guard=bridgingSupportedGuard ),
      'trunk': 'Set trunking mode to TRUNK unconditionally',
      'phone': phoneNode,
   }
   handler = setSwitchportModePhoneTrunk

SwitchportModelet.addCommandClass( SwitchportModeTrunkPhoneCmd )

MacBasedVlanAssignmentScope = Tac.Type( 'Bridging::MacBasedVlanAssignmentScope' )

def setMacBasedVlanAssignmentScope( mode, args ):
   if args.get( 'SCOPE' ) == 'interface':
      value = MacBasedVlanAssignmentScope.mbvaScopeInterface
   else: # Default.
      value = MacBasedVlanAssignmentScope.mbvaScopeGlobal
   cliConfig.macBasedVlanAssignmentScope = value

mbvaScopeMatcher = CliCommand.guardedKeyword( 'based',
      helpdesc='MAC based configuration commands',
      guard=mbvaScopeGuard )

class MacBasedVlanAssignmentScopeCmd( CliCommand.CliCommandClass ):
   syntax = 'mac based vlan assignment scope SCOPE'
   noOrDefaultSyntax = 'mac based vlan assignment scope ...'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'based': mbvaScopeMatcher,
      'vlan': 'MAC based VLAN configuration commands',
      'assignment': 'MAC based VLAN assignment configuration commands',
      'scope': 'MAC based VLAN assignment scope',
      'SCOPE': CliMatcher.EnumMatcher( {
         'interface': 'Assign VLAN to MAC on a per-interface basis',
         'global': 'Assign VLAN to MAC globally',
      } ),
   }
   handler = setMacBasedVlanAssignmentScope
   noOrDefaultHandler = handler

if Toggles.EbraToggleLib.togglePerInterfaceMbvaCliEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass(
         MacBasedVlanAssignmentScopeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] switchport dot1q ethertype" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport dot1q ethertype <ethertype>
#   {no|default} switchport dot1q ethertype [<ethertype>]
#-------------------------------------------------------------------------------
def switchportDot1QGuard( mode, token ):
   if not ( bridgingHwCapabilities.vlanTpidSupported or
            bridgingHwCapabilities.vlanTagRequiredSupported or
            bridgingHwCapabilities.vlanTagDisallowedSupported ):
      return CliParser.guardNotThisPlatform
   return None

def vlanTpidSupportedGuard( mode, token ):
   if not bridgingHwCapabilities.vlanTpidSupported:
      return CliParser.guardNotThisPlatform
   return None

dot1qMatcher = CliMatcher.KeywordMatcher( 'dot1q', helpdesc='802.1q parameters' )
dot1qNode = CliCommand.Node( matcher=dot1qMatcher, guard=switchportDot1QGuard )
ethertypeMatcher = CliMatcher.KeywordMatcher( 'ethertype',
      helpdesc='Ethertype/TPID (Tag Protocol IDentifier) for VLAN tagged frames' )
ethertypeNode = CliCommand.Node( matcher=ethertypeMatcher,
      guard=vlanTpidSupportedGuard )
vlanTpidMatcher = CliMatcher.IntegerMatcher( VlanTpid.min, VlanTpid.max,
      helpdesc='Ethertype/TPID value' )

def setVlanTpid( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if sic:
      sic.vlanTpid = args.get( 'VLANTPID', defaultTpid )

class SwitchportVlanTpidCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport dot1q ethertype VLANTPID'
   noOrDefaultSyntax = 'switchport dot1q ethertype ...'
   data = {
      'switchport': switchportMatcher,
      'dot1q': dot1qNode,
      'ethertype': ethertypeNode,
      'VLANTPID': vlanTpidMatcher
   }

   handler = setVlanTpid
   noOrDefaultHandler = handler


SwitchportModelet.addCommandClass( SwitchportVlanTpidCmd )

#-------------------------------------------------------------------------------
# The "[no|default] switchport dot1q vlan tag required" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport dot1q vlan tag required
#   switchport dot1q vlan tag disallowed
#   {no|default} switchport dot1q vlan tag
#-------------------------------------------------------------------------------


def dot1qFrameDropSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanTagRequiredSupported or \
      bridgingHwCapabilities.vlanTagDisallowedSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def vlanTagRequiredSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanTagRequiredSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def vlanTagDisallowedSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanTagDisallowedSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

vlanNode = CliCommand.guardedKeyword( 'vlan',
      helpdesc='VLAN Tagged Frames', guard=dot1qFrameDropSupportedGuard )

tagMatcher = CliMatcher.KeywordMatcher( 'tag', helpdesc='Tagged frames' )

def setVlanTagRequired( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   requiredTagFormats = [ VlanTagFormat.untagged, VlanTagFormat.priority ]
   for vlanTagFormat in sic.vlanTagFormatDrop:
      if vlanTagFormat not in requiredTagFormats:
         del sic.vlanTagFormatDrop [ vlanTagFormat ]
   sic.vlanTagFormatDrop[ VlanTagFormat.untagged ] = True
   sic.vlanTagFormatDrop[ VlanTagFormat.priority ] = True

def setVlanTagDisallowed( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   disallowedTagFormats = [ VlanTagFormat.tagged ]
   for vlanTagFormat in sic.vlanTagFormatDrop:
      if vlanTagFormat not in disallowedTagFormats:
         del sic.vlanTagFormatDrop [ vlanTagFormat ]
   sic.vlanTagFormatDrop[ VlanTagFormat.tagged ] = True

def noDot1qVlanTag( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   sic.vlanTagFormatDrop.clear()

class vlanTagRequiredCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport dot1q vlan tag required'
   data = {
      'switchport': switchportMatcher,
      'dot1q': dot1qNode,
      'vlan': vlanNode,
      'tag': tagMatcher,
      'required': CliCommand.guardedKeyword( 'required',
                       helpdesc='Necessary frame type',
                       guard=vlanTagRequiredSupportedGuard ),
      }

   handler = setVlanTagRequired

SwitchportModelet.addCommandClass( vlanTagRequiredCmd )

class VlanTagCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport dot1q vlan tag disallowed'
   noOrDefaultSyntax = 'switchport dot1q vlan tag ...'
   data = {
      'switchport': switchportMatcher,
      'dot1q': dot1qNode,
      'vlan': vlanNode,
      'tag': tagMatcher,
      'disallowed': CliCommand.guardedKeyword( 'disallowed',
                        helpdesc='Configure the port to '
                        'discard 802.1q encapsulated frames',
                        guard=vlanTagDisallowedSupportedGuard ),
      }

   handler = setVlanTagDisallowed
   noOrDefaultHandler = noDot1qVlanTag

SwitchportModelet.addCommandClass( VlanTagCmd )

#-------------------------------------------------------------------------------
# The "[no] switchport access vlan" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport access vlan <vlan_id>
#   {no|default} switchport access vlan [<vlan_id>]
#-------------------------------------------------------------------------------
def setAccessVlan( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      vlanId = Vlan( defaultAccessVlan )
   else:
      vlanId = args.get( 'VLANID' )
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   vc = bridgingConfig.vlanConfig.get( vlanId.id )
   # If one tries to create a vlan which is internal vlan while in interactive mode,
   # do not allow that. Allow vlan creation in non interactive modes.
   if vc and vc.internal and mode.session_.interactive_:
      mode.addError( vlanInUseErrStr( vlanId.id ) )
   else:
      if not vlanId.lookupConfig( mode ) and mode.session_.interactive_:
         mode.addWarning(
            'Access VLAN does not exist. Creating vlan %d' % vlanId.id )
         vlanId.create( mode )
      # It is important to create the vlan first and then assign it to
      # switchIntfConfig to avoid needless setting/resetting of nativeVlan.
      # In otherwords, if we don't follow this order, the code goes through
      # needless churn of first setting the nativeVlan to '0' and then to the
      # 'vlanId'
      sic.accessVlan = vlanId.id

defaultAccessVlan = 1

accessKwMatcher = CliMatcher.KeywordMatcher( 'access',
      helpdesc='Set access mode characteristics of the interface' )
vlanKwMatcher = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Set VLAN when interface is in access mode' )

class AccessVlan( CliCommand.CliCommandClass ):
   syntax = 'switchport access vlan VLANID'
   noOrDefaultSyntax = 'switchport access vlan ...'
   data = {
      'switchport': switchportMatcher,
      'access': accessKwMatcher,
      'vlan': vlanKwMatcher,
      'VLANID': vlanIdMatcher,
      }

   handler = setAccessVlan
   noOrDefaultHandler = handler

SwitchportModelet.addCommandClass( AccessVlan )

#-------------------------------------------------------------------------------
# The "[no] switchport trunk native vlan" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport trunk native vlan <vlan_id>
#   {no|default} switchport trunk native vlan [<vlan_id>]
#-------------------------------------------------------------------------------
trunkKwMatcher = CliMatcher.KeywordMatcher( 'trunk',
      helpdesc='Set trunking characteristics of the interface' )
trunkNode = CliCommand.Node(
      matcher=trunkKwMatcher,
      guard=bridgingSupportedGuard )
nativeMatcher = CliMatcher.KeywordMatcher( 'native',
      helpdesc='Set trunking native characteristics when '
      'interface is in trunking mode' )
nativeVlanMatcher = CliMatcher.KeywordMatcher( 'vlan',
       helpdesc='Set native VLAN when interface is in trunking mode' )

def setTrunkNativeVlan( mode, args ):
   if 'tag' in args:
      vlanId = Vlan( 0 )
   elif CliCommand.isNoOrDefaultCmd( args ):
      vlanId = Vlan( defaultTrunkNativeVlan )
   else:
      vlanId = args.get( 'VLANID' )
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.trunkNativeVlan = vlanId.id
   if vlanId.id != 0:
      if not vlanId.lookupConfig( mode ):
         mode.addWarning( 'VLAN id %d not found in current VLAN configuration' %
                          vlanId.id )
defaultTrunkNativeVlan = 1

class TrunkNativeVlan( CliCommand.CliCommandClass ):
   syntax = 'switchport trunk native vlan VLANID'
   noOrDefaultSyntax = 'switchport trunk native vlan ...'
   data = {
      'switchport': switchportMatcher,
      'trunk': trunkNode,
      'native': nativeMatcher,
      'vlan': nativeVlanMatcher,
      'VLANID': vlanIdMatcher,
      }

   handler = setTrunkNativeVlan
   noOrDefaultHandler = handler

SwitchportModelet.addCommandClass( TrunkNativeVlan )

#-------------------------------------------------------------------------------
# The "[no] switchport trunk native vlan tag" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport trunk native vlan tag
#   {no|default} switchport trunk native vlan tag
#-------------------------------------------------------------------------------
def taggedNativeVlanSupportedGuard( mode, token ):
   if bridgingHwCapabilities.taggedNativeVlanSupported:
      return None
   return CliParser.guardNotThisPlatform

tagKwMatcher = CliMatcher.KeywordMatcher( 'tag',
      helpdesc='Set native VLAN to be tagged when '
                'interface is in trunking mode' )
tagNode = CliCommand.Node(
      matcher=tagKwMatcher,
      guard=taggedNativeVlanSupportedGuard )

class TrunkNativeVlanTag( CliCommand.CliCommandClass ):
   syntax = 'switchport trunk native vlan tag'
   data = {
      'switchport': switchportMatcher,
      'trunk': trunkNode,
      'native': nativeMatcher,
      'vlan': nativeVlanMatcher,
      'tag': tagNode,
      }

   handler = setTrunkNativeVlan

SwitchportModelet.addCommandClass( TrunkNativeVlanTag )

#-------------------------------------------------------------------------------
# The "[no] switchport trunk allowed vlan" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport trunk allowed vlan all
#   switchport trunk allowed vlan none
#   switchport trunk allowed vlan VLAN_SET
#   switchport trunk allowed vlan except VLAN_SET
#   switchport trunk allowed vlan add VLAN_SET
#   switchport trunk allowed vlan remove VLAN_SET
#   {no|default} switchport trunk allowed vlan [ ... ]
#-------------------------------------------------------------------------------
vlanIdPattern = \
'(0*([1-9]|[1-9][0-9]|[1-9][0-9][0-9]|[1-3][0-9][0-9][0-9]|40[0-8][0-9]|409[0-4]))'
vlanRangePattern = '(%s(-%s)?)' % ( vlanIdPattern, vlanIdPattern )
vlanSetPattern = '%s(,%s)*' % ( vlanRangePattern, vlanRangePattern )

def vlanIdRangeListFunc( mode, grList ):
   return VlanIdRangeList( initialList=grList.ranges() )

vlanIdRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4094 ),
   noSingletons=False,
   helpdesc='VLAN ID or range(s) of VLAN IDs',
   value=vlanIdRangeListFunc )

tapToolAllowedVlanIdRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4095 ),
   noSingletons=False,
   helpdesc='VLAN ID or range(s) of VLAN IDs',
   value=vlanIdRangeListFunc )

def allVlanIds( args ):
   if any( switchmode in ( 'tap', 'tool' ) for switchmode in args ):
      return VlanIdRangeList( initialList=[ ( 1, 4095 ) ] )
   return VlanIdRangeList( initialList=[ ( 1, 4094 ) ] )

def noVlanIds():
   return VlanIdRangeList( initialList=[] )

allowedVlanOpMatcher = CliMatcher.EnumMatcher( {
   'add' : 'Add VLANS to the current list',
   'except' : 'All VLANS except the following',
   'remove' : 'Remove VLANS from the current list',
} )


# Do the actual work of changing the allowed VLANs string.
# This function is called from outside of this file as well.
# It exists to avoid duplicating the logic here. The
# oldStringHelpFn and its arguments allow the callers to
# retrieve the old value only when needed, which saves a
# GenericIf call when not needed. The resulting structure
# is a compromise between avoiding duplicating the
# code and preserving the performance of the duplicated code.
def vlanRangeOpResultString( args, argsVlansKey, opKey, noIsAll,
                             oldStringHelpFn, arg1, arg2, arg3 ):
   op = args.get( opKey )
   if 'none' in args:
      allowedVlanRanges = noVlanIds()
   elif 'all' in args or ( noIsAll and CliCommand.isNoOrDefaultCmd( args ) ) :
      allowedVlanRanges = allVlanIds( args )
   elif op == 'add':
      oldAllowedStr = oldStringHelpFn( arg1, arg2, arg3 )
      allowedVlanRanges = VlanIdRangeList( initialString=oldAllowedStr )
      allowedVlanRanges.add( args[ argsVlansKey ].ranges )
   elif op == 'remove':
      oldAllowedStr = oldStringHelpFn( arg1, arg2, arg3 )
      allowedVlanRanges = VlanIdRangeList( initialString=oldAllowedStr )
      allowedVlanRanges.remove( args[ argsVlansKey ].ranges )
   elif op == 'except':
      allowedVlanRanges = allVlanIds( args )
      allowedVlanRanges.remove( args[ argsVlansKey ].ranges )
   else:
      allowedVlanRanges = args[ argsVlansKey ]

   s = str( allowedVlanRanges )
   return s

def handleSwitchportModeAllowedVlan( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   sicAttrMap = {
      'trunk' : 'trunkAllowedVlans',
      'tap' : 'tapAllowedVlans',
      'tool' : 'toolAllowedVlans',
   }

   chosenMode = None
   for switchportMode in sicAttrMap:
      if switchportMode in args:
         chosenMode = switchportMode
         break
   assert chosenMode

   def switchportModeAllowedVlanHelper( sic, sicAttrMap, sicMode ):
      return getattr( sic, sicAttrMap[ sicMode ] )

   # The actual string manipulation is delegated to another function so that
   # the code can be shared with other CliPlugin functions doing the same
   # manipulations of allowed VLAN range strings. How to access the current
   # string is handled through a callback function so that we can defer the
   # GenericIf call to get the string and only do it in the cases that require
   # it. GenericIf accesses are expensive.
   s = vlanRangeOpResultString( args, 'VLANS', 'OP', True,
                                switchportModeAllowedVlanHelper,
                                sic, sicAttrMap, chosenMode )
   setattr( sic, sicAttrMap[ chosenMode ], s )

class SwitchportTrunkAllowedVlanCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport trunk allowed vlan ( all | none | ( [ OP ] VLANS ) )' )
   noOrDefaultSyntax = 'switchport trunk allowed vlan ...'
   data = {
      'switchport': switchportMatcher,
      'trunk': trunkNode,
      'allowed': 'Set allowed VLAN characteristics when interface is in '
                 'trunking mode',
      'vlan': 'Set allowed VLANs when interface is in trunking mode',
      'all': 'All VLANs',
      'none': 'No VLANs',
      'OP': allowedVlanOpMatcher,
      'VLANS': vlanIdRangeMatcher,
   }

   handler = handleSwitchportModeAllowedVlan
   noOrDefaultHandler = handler

SwitchportModelet.addCommandClass( SwitchportTrunkAllowedVlanCmd )


def setPortTrunkGroup( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.trunkGroup[ args[ 'GROUP' ] ] = True

canDeletePortTrunkGroupHook = CliExtensions.CliHook()

def noPortTrunkGroup( mode, args ):
   group = args.get( 'GROUP' )
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   for hook in canDeletePortTrunkGroupHook.extensions():
      if not hook( mode, group ):
         return
   if group is None:
      sic.trunkGroup.clear()
   else:
      del sic.trunkGroup[ group ]

class SwitchportTrunkGroupGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport trunk group GROUP'
   noOrDefaultSyntax = 'switchport trunk group [ GROUP ]'
   data = {
      'switchport': switchportMatcher,
      'trunk': trunkNode,
      'group': 'Trunk group information',
      'GROUP': trunkGroupMatcher,
   }

   handler = setPortTrunkGroup
   noOrDefaultHandler = noPortTrunkGroup

SwitchportModelet.addCommandClass( SwitchportTrunkGroupGroupCmd )

#-------------------------------------------------------------------------------
# Adds vlan mapping CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
def getSwitchInputConfig( intf ):
   intfs = cliConfig.switchIntfConfig
   intfConfig = intfs.get( intf )
   if intfConfig:
      return intfConfig
   switchportMode = 'dot1qTunnel' if SubIntfId.isSubIntfId( intf ) else 'access'
   return intfs.newMember( intf, switchportMode )

VlanXlateKey = Tac.Type( "Bridging::VlanXlateKey" )
VlanXlateMapping = Tac.Type( "Bridging::VlanXlateMapping" )

def ingressVlanXlateKeyIfBoth( ingressVlanXlate,
                               egressVlanXlateKey, egressVlanXlateMapping ):
   if egressVlanXlateMapping is None:
      return None

   ingressVlanXlateKey = Tac.Value( "Bridging::VlanXlateKey",
         egressVlanXlateMapping.vlanId, egressVlanXlateMapping.innerVid )
   ingressVlanXlateMapping = ingressVlanXlate.get( ingressVlanXlateKey )

   if ingressVlanXlateMapping and \
         ingressVlanXlateMapping.vlanId == egressVlanXlateKey.outerVid and \
         ingressVlanXlateMapping.deQinQXlate \
            == egressVlanXlateMapping.deQinQXlate and \
         ingressVlanXlateMapping.tunnel == egressVlanXlateMapping.tunnel:
      return ingressVlanXlateKey

   return None

def addVlanMapping( sic, wireVid, actualVid, wireInnerVid=0,
                    direction="both", dot1qTunnel=False, allInnerVlanId=False,
                    deQinQXlate=False ):
   # Update the cli switch input config to register the desired mapping
   if direction == "both":
      ingressKey = VlanXlateKey( wireVid, wireInnerVid )
      ingressMapping = sic.ingressVlanXlate.get( ingressKey )
      oldEgressKey = egressVlanXlateKeyIfBoth( sic.egressVlanXlate,
                                               ingressKey, ingressMapping )
      egressKey = VlanXlateKey( actualVid, wireVid if dot1qTunnel else 0 )
      egressMapping = sic.egressVlanXlate.get( egressKey )
      oldIngressKey = ingressVlanXlateKeyIfBoth( sic.ingressVlanXlate,
                                                 egressKey, egressMapping )
      if oldIngressKey is not None and ingressKey != oldIngressKey:
         del sic.ingressVlanXlate[ oldIngressKey ]
      if oldEgressKey is not None and egressKey != oldEgressKey:
         del sic.egressVlanXlate[ oldEgressKey ]
      for d in [ "in", "out" ]:
         addVlanMapping( sic, wireVid, actualVid, wireInnerVid=wireInnerVid,
                         direction=d, dot1qTunnel=dot1qTunnel,
                         deQinQXlate=deQinQXlate )
   elif direction == "in":
      ingressKey = VlanXlateKey( wireVid, wireInnerVid )
      mapping = VlanXlateMapping( actualVid, 0, dot1qTunnel )
      mapping.deQinQXlate = deQinQXlate
      sic.ingressVlanXlate[ ingressKey ] = mapping

   elif direction == "out":
      innerVlanId = 0
      if dot1qTunnel and not allInnerVlanId:
         innerVlanId = wireVid
      egressKey = VlanXlateKey( actualVid, innerVlanId )

      # If there is a deQinQXlate mapping, delete both mappings. This is because
      # deQinQXlate is always "both", never "in" or "out".
      oldEgressMapping = sic.egressVlanXlate.get( egressKey )
      oldIngressKey = ingressVlanXlateKeyIfBoth( sic.ingressVlanXlate,
                                                 egressKey, oldEgressMapping )
      if oldIngressKey is not None:
         ingressMapping = sic.ingressVlanXlate.get( oldIngressKey )
         if ingressMapping.deQinQXlate:
            del sic.ingressVlanXlate[ oldIngressKey ]

      # Add the new mapping
      egressMapping = VlanXlateMapping( wireVid, wireInnerVid, dot1qTunnel )
      egressMapping.allInnerVid = allInnerVlanId
      egressMapping.deQinQXlate = deQinQXlate
      sic.egressVlanXlate[ egressKey ] = egressMapping

   else:
      assert False, "Unexpected direction"

def delVlanMapping( sic, wireVid, actualVid, wireInnerVid=0,
                    direction="both", dot1qTunnel=False, allInnerVlanId=False,
                    deQinQXlate=False ):
   # Update the cli switch input config to delete the given mapping
   if direction == "both":
      ingressKey = VlanXlateKey( wireVid, wireInnerVid )
      ingressMapping = sic.ingressVlanXlate.get( ingressKey )
      egressKey = egressVlanXlateKeyIfBoth( sic.egressVlanXlate,
                                            ingressKey, ingressMapping )
      if egressKey is not None:
         del sic.ingressVlanXlate[ ingressKey ]
         del sic.egressVlanXlate[ egressKey ]
   elif direction == "in":
      ingressKey = VlanXlateKey( wireVid, wireInnerVid )
      del sic.ingressVlanXlate[ ingressKey ]
   elif direction == "out":
      innerVlanId = 0
      if dot1qTunnel and not allInnerVlanId:
         innerVlanId = wireVid
      egressKey = VlanXlateKey( actualVid, innerVlanId )
      del sic.egressVlanXlate[ egressKey ]
   else:
      assert False, "Unexpected direction"

def isNativeVlan( sic, vid ):
   nativeVlan = 0
   if sic and sic.enabled:
      if sic.switchportMode == 'trunk':
         nativeVlan = sic.trunkNativeVlan
      elif sic.switchportMode == 'access' or sic.switchportMode == 'dot1qTunnel':
         nativeVlan = sic.accessVlan
   return nativeVlan == vid

def isInternalVlan( vid ):
   vcfg = bridgingConfig.vlanConfig.get( vid )
   return vcfg and vcfg.internal

def nativeVlanWarningStr( vid ):
   return ( "VLAN %d is the native VLAN. Please note that VLAN mapping will "
            "only take effect for tagged packets." % vid )

def invalidConfigErrStr( intf ):
   return "Vlan mapping is temporarily unavailable for %s." % intf

def validateTranslationVlans( mode, sic, wireVidSet, actualVidSet,
                              wireInnerVid=0, dot1qTunnel=False ):
   if not ( wireInnerVid or dot1qTunnel ):
      if wireVidSet == actualVidSet:
         mode.addError( "VLANs cannot be the same" )
         return False
      elif wireVidSet & actualVidSet:
         if len( wireVidSet ) > 1:
            wireVidSet -= actualVidSet
            mode.addWarning( "No mapping created for VLAN %d" %
                             list( actualVidSet )[ 0 ] )
         elif len( actualVidSet ) > 1:
            actualVidSet -= wireVidSet
            mode.addWarning( "No mapping created for VLAN %d" %
                             list( wireVidSet )[ 0 ] )

   missingVlan = []
   internalVlan = []
   for actualVid in actualVidSet:
      if isInternalVlan( actualVid ):
         internalVlan.append( actualVid )
      elif not Vlan( actualVid ).lookupConfig( mode ):
         missingVlan.append( actualVid )

   if internalVlan:
      mode.addWarning( 'VLAN(s) %s in use as internal VLAN' %
                        vlanSetToCanonicalString( internalVlan ) )
   if missingVlan:
      mode.addWarning( 'VLAN(s) %s not found in current VLAN configuration' %
                        vlanSetToCanonicalString( missingVlan ) )

   for wireVid in wireVidSet:
      if isNativeVlan( sic, wireVid ):
         mode.addWarning( "VLAN %d is the native VLAN. Please note that VLAN "
                          "mapping will only take effect for tagged packets."
                          % wireVid )

   return True

def validateVlanMapping( mode, sic, wireVidSet, actualVidSet,
                         wireInnerVid=0, dot1qTunnel=False ):
   if not validateTranslationVlans( mode, sic, wireVidSet, actualVidSet,
                                    wireInnerVid=wireInnerVid,
                                    dot1qTunnel=dot1qTunnel ):
      return False
   if not sic.enabled or sic.switchportMode != 'trunk':
      mode.addWarning( 'Not a trunk port. '
                       'VLAN mapping will take effect only for trunk ports' )

   return True

def setVlanMapping( mode, origVlanSet, newVlanSet,
                    origInnerVlan=None, newInnerVlan=None,
                    direction="both", dot1qTunnel=False, delete=False,
                    allInnerVlanId=False, deQinQXlate=False ):
   assert len( origVlanSet ) == 1 or len( newVlanSet ) <= 1
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      mode.addError( invalidConfigErrStr( mode.intf.name ) )
      return

   if direction == "out":
      wireVidSet = newVlanSet
      actualVidSet = origVlanSet
      assert origInnerVlan is None
      wireInnerVid = newInnerVlan if newInnerVlan is not None else 0
   else:
      wireVidSet = origVlanSet
      actualVidSet = newVlanSet
      assert newInnerVlan is None
      wireInnerVid = origInnerVlan if origInnerVlan is not None else 0

   if delete:
      func = delVlanMapping
   else:
      func = addVlanMapping
      if not validateVlanMapping( mode, sic, wireVidSet, actualVidSet,
                                  wireInnerVid=wireInnerVid,
                                  dot1qTunnel=dot1qTunnel ):
         return

   if direction == "out":
      if dot1qTunnel:
         actualVid, = actualVidSet # A VLAN ID from set; `ValueError` if len > 1.
         for wireVid in wireVidSet:
            func( sic, wireVid=wireVid, actualVid=actualVid,
                  wireInnerVid=wireInnerVid,
                  direction=direction, dot1qTunnel=dot1qTunnel,
                  allInnerVlanId=allInnerVlanId, deQinQXlate=deQinQXlate )
      else:
         if delete:
            assert not wireVidSet
            wireVid = None
         else:
            wireVid, = wireVidSet # A VLAN ID from set; `ValueError` if len > 1.
         for actualVid in actualVidSet:
            func( sic, wireVid=wireVid, actualVid=actualVid,
                  wireInnerVid=wireInnerVid,
                  direction=direction, dot1qTunnel=dot1qTunnel,
                  deQinQXlate=deQinQXlate )
   else:
      if delete:
         assert not actualVidSet
         actualVid = None
      else:
         actualVid, = actualVidSet # A VLAN ID from set; `ValueError` if len > 1.
      for wireVid in wireVidSet:
         func( sic, wireVid=wireVid, actualVid=actualVid,
               wireInnerVid=wireInnerVid,
               direction=direction, dot1qTunnel=dot1qTunnel,
               deQinQXlate=deQinQXlate )

def noVlanMapping( mode, origVlanSet, newVlanSet=VlanSet( None, None, vlanIds=[] ),
                   origInnerVlan=None, newInnerVlan=None,
                   direction="both", dot1qTunnel=False, allInnerVlanId=False,
                   deQinQXlate=False ):
   setVlanMapping( mode, origVlanSet, newVlanSet,
                   origInnerVlan=origInnerVlan, newInnerVlan=newInnerVlan,
                   direction=direction, dot1qTunnel=dot1qTunnel,
                   allInnerVlanId=allInnerVlanId, deQinQXlate=deQinQXlate,
                   delete=True )

def addNativeVlanMapping( switchIntfConfig, wireVid ):
   # Update the cli switch input config to register the desired native mapping
   switchIntfConfig.ingressNativeVlanXlate = wireVid
   switchIntfConfig.egressNativeVlanXlate = wireVid

def delNativeVlanMapping( switchIntfConfig ):
   # Update the cli switch input config to delete the given native mapping
   switchIntfConfig.ingressNativeVlanXlate = 0
   switchIntfConfig.egressNativeVlanXlate = 0

def vlanMappingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanXlateSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def encapDot1qSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanXlateSupported and \
         bridgingHwCapabilities.encapDot1qSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ingressVlanMappingRequiredSupportedGuard( mode, token ):
   if bridgingHwCapabilities.ingressVlanMappingRequiredSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def egressVlanMappingRequiredSupportedGuard( mode, token ):
   if bridgingHwCapabilities.egressVlanMappingRequiredSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def doubleVlanMappingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.doubleVlanXlateSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def vlanMappingToDot1qTunelSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanXlateToDot1qTunnelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def egressVlanXlateToDot1qAllSupported( mode, token ):
   if bridgingHwCapabilities.egressVlanXlateToDot1qAllSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def deQinQXlateSupportedGuard( mode, token ):
   if bridgingHwCapabilities.deQinQXlateSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def innerVlanMappingSupportedGuard( mode, token ):
   if ( bridgingHwCapabilities.doubleVlanXlateSupported or
        bridgingHwCapabilities.deQinQXlateSupported ):
      return None
   else:
      return CliParser.guardNotThisPlatform

vlanForMapping = CliCommand.guardedKeyword( 'vlan',
      helpdesc='VLAN mapping commands', guard=vlanMappingSupportedGuard )

translationMatcher = CliMatcher.KeywordMatcher( 'translation',
      helpdesc='Map packets from one VLAN to another' )

mappingDeprecatedMatcher = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
      'mapping', helpdesc='Map packets from one VLAN to another' ),
      deprecatedByCmd='switchport vlan translation' )

origVlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='VLAN ID to map from', priority=CliParser.PRIO_HIGH )

newVlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='VLAN ID to map to' )

origInnerNode = CliCommand.guardedKeyword( 'inner',
      helpdesc='Inner VLAN ID to map from', guard=innerVlanMappingSupportedGuard )

origInnerVlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Inner VLAN ID to map from' )

origVlanSetIdsMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ), False,
      'VLAN ID or range(s) of VLAN IDs to map from', value=vlanIdListFunc )

mapToDot1qTunnelMatcher = CliCommand.guardedKeyword( 'dot1q-tunnel',
      helpdesc="Dot1q tunnel by retaining the tag",
      guard=vlanMappingToDot1qTunelSupportedGuard )

newInnerVlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Inner VLAN ID to map to' )

newInnerNode = CliCommand.guardedKeyword( 'inner',
      helpdesc='Inner VLAN ID to map to', guard=doubleVlanMappingSupportedGuard )

deQinQXlateNetworkNode = CliCommand.guardedKeyword( 'network',
      helpdesc='Network-side VLAN ID to map to',
      guard=deQinQXlateSupportedGuard )

newVlanSetIdsMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, 4094 ), False,
      'VLAN ID or range(s) of VLAN IDs to map to', value=vlanIdListFunc )

#-------------------------------------------------------------------------------
# switchport vlan forwarding accept all
#-------------------------------------------------------------------------------
def vlanForwardingModeSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vlanForwardingModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def setVlanForwardingModeAllConfiguredVlans( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      mode.addError( invalidConfigErrStr( mode.intf.name ) )
      return
   sic.vlanForwardingMode = "vlanForwardingModeAllConfiguredVlans"

def setVlanForwardingModeAllowedVlansOnPort( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      mode.addError( invalidConfigErrStr( mode.intf.name ) )
      return
   sic.vlanForwardingMode = "vlanForwardingModeAllowedVlansOnPort"

class SwitchportVlanForwarding( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan forwarding accept all'
   noOrDefaultSyntax = 'switchport vlan forwarding ...'
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'forwarding': CliCommand.guardedKeyword( 'forwarding',
                       helpdesc='Forward packets belonging to VLAN',
                       guard=vlanForwardingModeSupportedGuard ),
      'accept': 'Accept packets for VLAN',
      'all': 'Accept all VLANs',
      }

   handler = setVlanForwardingModeAllConfiguredVlans
   noOrDefaultHandler = setVlanForwardingModeAllowedVlansOnPort

SwitchportModelet.addCommandClass( SwitchportVlanForwarding )

#-------------------------------------------------------------------------------
# new syntax:
#    "switchport vlan translation <vid> [inner <inner-vid>] <new-vid>" command,
# legacy:
#    "switchport vlan mapping <vid> [inner <inner-vid>] <new-vid>" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanMappingBoth( mode, args ):
   origVlan = args[ 'VID' ]
   newVlan = args[ 'NEW_ID' ]
   origInnerVlan = args.get( 'INNER_ID' )
   setVlanMapping( mode, VlanSet( mode, None, vlanIds=[ origVlan ] ),
                   VlanSet( mode, None, vlanIds=[ newVlan ] ),
                   origInnerVlan=origInnerVlan )

def noVlanMappingBoth( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   origInnerVlan = args.get( 'INNER_ID' )
   noVlanMapping( mode, origVlanSet, origInnerVlan=origInnerVlan )

class VlanTranslationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan ( translation | mapping ) VID [ inner INNER_ID ] NEW_ID'
   noOrDefaultSyntax = ( 'switchport vlan ( translation | mapping ) VLANS '
                         '[ inner INNER_ID ] ...' )
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'VID': origVlanIdMatcher,
      'inner': origInnerNode,
      'VLANS': origVlanSetIdsMatcher,
      'INNER_ID': origInnerVlanIdMatcher,
      'NEW_ID': newVlanIdMatcher,
      }

   handler = setVlanMappingBoth
   noOrDefaultHandler = noVlanMappingBoth

SwitchportModelet.addCommandClass( VlanTranslationCmd )

def setDeQinQXlateMapping( mode, args ):
   ivid = args[ 'IVID' ]
   newVid = args[ 'NEW_VID' ]
   vid = args[ 'VID' ]

   setVlanMapping( mode,
                   VlanSet( mode, None, vlanIds=[ vid ] ),
                   VlanSet( mode, None, vlanIds=[ newVid ] ),
                   ivid,
                   deQinQXlate=True )

def noDeQinQXlateMapping( mode, args ):
   ivid = args[ 'IVID' ]
   vid = args[ 'VID' ]

   noVlanMapping( mode,
                  VlanSet( mode, None, vlanIds=[ vid ] ),
                  origInnerVlan=ivid )

class DeQinQXlateCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan translation VID inner IVID network NEW_VID'
   noOrDefaultSyntax = 'switchport vlan translation VID inner IVID ...'
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'IVID': origInnerVlanIdMatcher,
      'inner': origInnerNode,
      'network': deQinQXlateNetworkNode,
      'NEW_VID': newVlanIdMatcher,
      'VID': origVlanIdMatcher,
      }

   handler = setDeQinQXlateMapping
   noOrDefaultHandler = noDeQinQXlateMapping

SwitchportModelet.addCommandClass( DeQinQXlateCmd )

#-------------------------------------------------------------------------------
# new syntax:
#    "switchport vlan translation <vid> dot1q-tunnel <new-vid>" command,
#
# legacy:
#    "switchport vlan mapping <vid> dot1q-tunnel <new-vid>" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanMappingDot1qTunnel( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   newVlan = args[ 'NEW_ID' ]
   setVlanMapping( mode, origVlanSet, VlanSet( mode, None, vlanIds=[ newVlan ] ),
                   direction="both", dot1qTunnel=True )

class VlanMappingDot1qTunnelCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan ( translation | mapping ) VLANS dot1q-tunnel NEW_ID'
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'VLANS': origVlanSetIdsMatcher,
      'dot1q-tunnel': mapToDot1qTunnelMatcher,
      'NEW_ID': newVlanIdMatcher,
      }

   handler = setVlanMappingDot1qTunnel

SwitchportModelet.addCommandClass( VlanMappingDot1qTunnelCmd )

#-------------------------------------------------------------------------------
# new syntax:
# switchport vlan translation in <vid> [inner <inner-vid>] <new-vid>" command,
# in "config-if" mode.
#
# legacy:
# "switchport vlan mapping in <vid> [inner <inner-vid>] <new-vid>" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
mapInMatcher = CliMatcher.KeywordMatcher( 'in', helpdesc="Map ingress packets only" )

def setVlanMappingIngress( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   newVlan = args[ 'NEW_ID' ]
   origInnerVlan = args.get( 'INNER_ID' )
   setVlanMapping( mode, origVlanSet, VlanSet( mode, None, vlanIds=[ newVlan ] ),
                   origInnerVlan=origInnerVlan, direction="in" )

def noVlanMappingIngress( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   origInnerVlan = args.get( 'INNER_ID' )
   noVlanMapping( mode, origVlanSet, origInnerVlan=origInnerVlan, direction="in" )

class VlanMappingIngressCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport vlan ( translation | mapping ) in VLANS '
              '[ inner INNER_ID ] NEW_ID' )
   noOrDefaultSyntax = ( 'switchport vlan ( translation | mapping ) in VLANS '
                         '[ inner INNER_ID ] ...' )
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'in': mapInMatcher,
      'inner': origInnerNode,
      'VLANS': origVlanSetIdsMatcher,
      'INNER_ID': origInnerVlanIdMatcher,
      'NEW_ID': newVlanIdMatcher,
      }

   handler = setVlanMappingIngress
   noOrDefaultHandler = noVlanMappingIngress

SwitchportModelet.addCommandClass( VlanMappingIngressCmd )

#-------------------------------------------------------------------------------
# new syntax :
#   "switchport vlan translation in required" command, in "config-if" mode.
# legacy:
#   "switchport vlan mapping in required" command, in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanMappingRequired( mode, direction ):
   sic = getSwitchInputConfig( mode.intf.name )
   sic.vlanMappingRequired[ direction ] = True

def noVlanMappingRequired( mode, direction ):
   sic = getSwitchInputConfig( mode.intf.name )
   del sic.vlanMappingRequired[ direction ]

def setVlanMappingRequiredIngress( mode, args ):
   setVlanMappingRequired( mode, VlanMappingDirection.vlanMappingIngress )

def noVlanMappingRequiredIngress( mode, args ):
   noVlanMappingRequired( mode, VlanMappingDirection.vlanMappingIngress )

class VlanMappingRequiredIngressCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan ( translation | mapping ) in required'
   noOrDefaultSyntax = syntax
   data = {
         'switchport': switchportMatcher,
         'vlan': vlanForMapping,
         'translation': translationMatcher,
         'mapping': mappingDeprecatedMatcher,
         'in': mapInMatcher,
         'required': CliCommand.guardedKeyword( 'required',
                        helpdesc='Drop the packets that do '
                              'not match any VLAN mapping',
                        guard=ingressVlanMappingRequiredSupportedGuard ),
         }

   handler = setVlanMappingRequiredIngress
   noOrDefaultHandler = noVlanMappingRequiredIngress

SwitchportModelet.addCommandClass( VlanMappingRequiredIngressCmd )

#-------------------------------------------------------------------------------
# new syntax
# "switchport vlan translation in <vid> dot1q-tunnel <new-vid>" command,
# in "config-if" mode.
#
# legacy:
# "switchport vlan mapping in <vid> dot1q-tunnel <new-vid>" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanMappingDot1qTunnelIngress( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   newVlan = args[ 'NEW_ID' ]
   setVlanMapping( mode, origVlanSet, VlanSet( mode, None, vlanIds=[ newVlan ] ),
                   direction="in", dot1qTunnel=True )

class VlanMappingDot1qTunnelIngressCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan ( translation | mapping ) in VLANS dot1q-tunnel NEW_ID'
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'in': mapInMatcher,
      'VLANS': origVlanSetIdsMatcher,
      'dot1q-tunnel': mapToDot1qTunnelMatcher,
      'NEW_ID': newVlanIdMatcher,
      }

   handler = setVlanMappingDot1qTunnelIngress

SwitchportModelet.addCommandClass( VlanMappingDot1qTunnelIngressCmd )

#-------------------------------------------------------------------------------
# new syntax:
# "switchport vlan translation out <vid> <new-vid> [inner <new-inner-vid>]" command,
# in "config-if" mode.
#
# legacy
#  "switchport vlan mapping out <vid> <new-vid> [inner <new-inner-vid>]" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
mapOutMatcher = CliMatcher.KeywordMatcher( 'out',
      helpdesc="Map egress packets only" )

def setVlanMappingEgress( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   newVlan = args[ 'NEW_ID' ]
   newInnerVlan = args.get( 'NEW-INNER_ID' )
   setVlanMapping( mode, origVlanSet, VlanSet( mode, None, vlanIds=[ newVlan ] ),
                   newInnerVlan=newInnerVlan, direction="out" )

def noVlanMappingEgress( mode, args ):
   origVlanSet = args[ 'VLANS' ]
   noVlanMapping( mode, origVlanSet, direction="out" )

class VlanMappingEgressCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switchport vlan ( translation | mapping ) out VLANS '
              'NEW_ID [ inner NEW-INNER_ID ]' )
   noOrDefaultSyntax = 'switchport vlan ( translation | mapping ) out VLANS ...'
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'out': mapOutMatcher,
      'VLANS': origVlanSetIdsMatcher,
      'NEW_ID': newVlanIdMatcher,
      'inner': newInnerNode,
      'NEW-INNER_ID': newInnerVlanIdMatcher,
      }

   handler = setVlanMappingEgress
   noOrDefaultHandler = noVlanMappingEgress

SwitchportModelet.addCommandClass( VlanMappingEgressCmd )

# -------------------------------------------------------------------------------
#  "switchport vlan translation out required" command, in "config-if" mode.
# -------------------------------------------------------------------------------
def setVlanMappingRequiredEgress( mode, args ):
   setVlanMappingRequired( mode, VlanMappingDirection.vlanMappingEgress )

def noVlanMappingRequiredEgress( mode, args ):
   noVlanMappingRequired( mode, VlanMappingDirection.vlanMappingEgress )

class VlanMappingRequiredEgressCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan translation out required'
   noOrDefaultSyntax = syntax
   data = {
         'switchport': switchportMatcher,
         'vlan': vlanForMapping,
         'translation': translationMatcher,
         'out': mapOutMatcher,
         'required': CliCommand.guardedKeyword( 'required',
                        helpdesc='Drop the packets that do '
                              'not match any VLAN mapping',
                        guard=egressVlanMappingRequiredSupportedGuard ),
         }

   handler = setVlanMappingRequiredEgress
   noOrDefaultHandler = noVlanMappingRequiredEgress

SwitchportModelet.addCommandClass( VlanMappingRequiredEgressCmd )

#-------------------------------------------------------------------------------
# New Syntax:
# "switchport vlan translation out <vid> dot1q-tunnel <new-vid>" command,
# in "config-if" mode.
#
# Legacy:
# "switchport vlan mapping out <vid> dot1q-tunnel <new-vid>" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanMappingDot1qTunnelEgress( mode, args ):
   origVlan = args[ 'VID' ]
   newVlanSet = args[ 'NEW_IDS' ]
   setVlanMapping( mode, VlanSet( mode, None, vlanIds=[ origVlan ] ),
                   newVlanSet, direction="out", dot1qTunnel=True )

def noVlanMappingDot1qTunnelEgress( mode, args ):
   origVlan = args[ 'VID' ]
   newVlanSet = args[ 'NEW_IDS' ]
   noVlanMapping( mode, VlanSet( mode, None, vlanIds=[ origVlan ] ),
                  newVlanSet, direction="out", dot1qTunnel=True )

class VlanMappingDot1qTunnelEgressCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan ( translation | mapping ) out VID dot1q-tunnel NEW_IDS'
   noOrDefaultSyntax = ( 'switchport vlan ( translation | mapping ) out VID '
                         'dot1q-tunnel NEW_IDS ...' )
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'mapping': mappingDeprecatedMatcher,
      'out': mapOutMatcher,
      'VID': origVlanIdMatcher,
      'dot1q-tunnel': mapToDot1qTunnelMatcher,
      'NEW_IDS': newVlanSetIdsMatcher,
      }

   handler = setVlanMappingDot1qTunnelEgress
   noOrDefaultHandler = noVlanMappingDot1qTunnelEgress

SwitchportModelet.addCommandClass( VlanMappingDot1qTunnelEgressCmd )

# -------------------------------------------------------------------------------
# Syntax
# "switchport vlan translation out <vid> dot1q-tunnel all" command,
# in "config-if" mode.
# -------------------------------------------------------------------------------
def setVlanMappingDot1qTunnelEgressAll( mode, args ):
   origVlan = args[ 'VID' ]
   setVlanMapping( mode, VlanSet( mode, None, vlanIds=[ origVlan ] ),
                   VlanSet( mode, None, vlanIds=[ origVlan ] ),
                   direction="out", dot1qTunnel=True, allInnerVlanId=True )

def noVlanMappingDot1qTunnelEgressAll( mode, args ):
   origVlan = args[ 'VID' ]
   noVlanMapping( mode, VlanSet( mode, None, vlanIds=[ origVlan ] ),
                  VlanSet( mode, None, vlanIds=[ origVlan ] ),
                  direction="out", dot1qTunnel=True, allInnerVlanId=True )

class VlanMappingDot1qTunnelEgressAllCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan translation out VID dot1q-tunnel all'
   noOrDefaultSyntax = ( 'switchport vlan translation out VID '
                         'dot1q-tunnel all ...' )
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanForMapping,
      'translation': translationMatcher,
      'out': mapOutMatcher,
      'VID': origVlanIdMatcher,
      'dot1q-tunnel': mapToDot1qTunnelMatcher,
      'all': CliCommand.guardedKeyword(
         'all',
         helpdesc="match on all inner C-VLANs for dot1q tunnel packets",
         guard=egressVlanXlateToDot1qAllSupported ),
      }

   handler = setVlanMappingDot1qTunnelEgressAll
   noOrDefaultHandler = noVlanMappingDot1qTunnelEgressAll

SwitchportModelet.addCommandClass( VlanMappingDot1qTunnelEgressAllCmd )

#-------------------------------------------------------------------------------
# The "encapsulation dot1q vlan VLAN_ID" command, in "config-if" mode.
#-------------------------------------------------------------------------------
nodeEncapsulation = CliCommand.guardedKeyword( 'encapsulation',
                       helpdesc='configure encapsulation',
                       guard=encapDot1qSupportedGuard )

def setVlanEncap( mode, args ):
   sic = getSwitchInputConfig( mode.intf.name )
   if not sic:
      mode.addError( invalidConfigErrStr( mode.intf.name ) )
      return

   if CliCommand.isNoOrDefaultCmd( args ):
      # Remove the native vlan mapping for both ingress and egress
      delNativeVlanMapping( sic )
      return

   if sic.enabled:
      mode.addWarning( 'Not a routed port. '
                       'Dot1q encapsulation will take effect only for routed ports' )

   # Add the native vlan mapping for both ingress and egress
   addNativeVlanMapping( sic, args[ 'VLAN_ID' ].id )

class EncapsulationDot1QVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'encapsulation dot1q vlan VLAN_ID'
   noOrDefaultSyntax = 'encapsulation dot1q vlan ...'
   data = {
      'encapsulation': nodeEncapsulation,
      'dot1q': '802.1q encapsulation',
      'vlan': 'VLAN tag',
      'VLAN_ID': vlanIdMatcher,
   }

   handler = setVlanEncap
   noOrDefaultHandler = handler

SwitchportModelet.addCommandClass( EncapsulationDot1QVlanidCmd )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] switchport vlan mapping" command
#-------------------------------------------------------------------------------
switchportForShowKw = CliMatcher.KeywordMatcher( 'switchport',
                                   helpdesc='Details on switchports' )

vlanForShowKw = CliCommand.guardedKeyword( 'vlan',
                                           'VLAN mapping information',
                                           vlanMappingSupportedGuard )

def showVlanMapping( mode, args ): # pylint: disable=inconsistent-return-statements
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   def createInterfaceVlanMappings( intf, sic, vlanXlateStatus,
         vlanMappingRequiredStatus ):
      def populateInterfaceVlanMappings( direction, singleMap, doubleMap,
                                         hwStatus, config ):
         def addToVlanMap( direction, singleMap, doubleMap,
                           xlateKey, xlateMapping, active, configured ):
            assert direction in [ 'in', 'out' ]
            assert not ( xlateKey.innerVid and xlateMapping.innerVid )
            if xlateMapping.tunnel and not xlateMapping.allInnerVid:
               if direction == 'in':
                  assert not ( xlateKey.innerVid or xlateMapping.innerVid )
               elif direction == 'out':
                  assert xlateKey.innerVid and not xlateMapping.innerVid

            mappedVlan = MappedVlan()
            mappedVlan.vlanId = xlateMapping.vlanId
            if xlateMapping.innerVid:
               mappedVlan.innerVlanId = xlateMapping.innerVid
            mappedVlan.dot1qTunnel = xlateMapping.tunnel
            mappedVlan.allInnerVid = xlateMapping.allInnerVid
            mappedVlan.deQinQXlate = xlateMapping.deQinQXlate
            mappedVlan.active = active
            mappedVlan.configured = configured

            if xlateKey.innerVid:
               mappings = doubleMap.setdefault(
                                xlateKey.outerVid, InnerTagMappings() ).mappings
               mappings.setdefault( xlateKey.innerVid, mappedVlan )
            else:
               singleMap.setdefault( xlateKey.outerVid, mappedVlan )

         # BEGIN populateIntfVlanMappings
         if hwStatus: # pylint: disable=too-many-nested-blocks
            # If PortChannel then check if ( xlateKey, xlateMapping ) entry
            # is present in all its active members. Otherwise show it as inactive.
            for ( xlateKey, xlateMapping ) in hwStatus.items():
               if PortChannelIntfId.isPortChannelIntfId( intf.name ) and \
                     vlanXlateStatusDir.usePortChannelMemberStatus and \
                     EbraCliLib.getLagMembersCallBack:
                  lagMembers = \
           list( EbraCliLib.getLagMembersCallBack( # pylint: disable-msg=not-callable
                        mode, intf.name, lacpOnly=False,
                        status='filterOnActive' ) )
                  active = bool( lagMembers )
                  for m in lagMembers:
                     # Skip PeerEthIntf while computing status for lag interfaces
                     if PeerEthIntfId.isPeerEthIntfId( m ):
                        continue
                     mVlanXlateStatus = vlanXlateStatusDir.vlanXlateStatus.get( m )
                     if mVlanXlateStatus:
                        if direction == 'in':
                           mHwStatus = mVlanXlateStatus.hwIngressVlanXlate
                        elif direction == 'out':
                           mHwStatus = mVlanXlateStatus.hwEgressVlanXlate
                        mXlateMapping = mHwStatus.get( xlateKey )
                        if mXlateMapping is None or mXlateMapping != xlateMapping:
                           active = False
                           break
                     else:
                        active = False
               else:
                  active = True

               addToVlanMap( direction, singleMap, doubleMap,
                  xlateKey, xlateMapping, active=active,
                  configured=( config is not None  and
                     ( config.get( xlateKey ) == \
                           xlateMapping ) ) )

         # This should be after hwStatus loop, so that hwStatus takes precedence
         if config:
            for ( xlateKey, xlateMapping ) in config.items():
               addToVlanMap( direction, singleMap, doubleMap,
                             xlateKey, xlateMapping, active=False,
                             configured=True )

      # BEGIN createInterfaceVlanMappings
      intfVlanMap = InterfaceVlanMappings()
      populateInterfaceVlanMappings( 'in', intfVlanMap.ingressVlanMappings,
                                     intfVlanMap.ingressDoubleTaggedMappings,
                                     vlanXlateStatus.hwIngressVlanXlate if
                                        vlanXlateStatus else None,
                                     sic.ingressVlanXlate if sic else None )

      populateInterfaceVlanMappings( 'out', intfVlanMap.egressVlanMappings,
                                     intfVlanMap.egressDoubleTaggedMappings,
                                     vlanXlateStatus.hwEgressVlanXlate if
                                        vlanXlateStatus else None,
                                     sic.egressVlanXlate if sic else None )

      if vlanMappingRequiredStatus is not None:
         intfVlanMap.ingressVlanMappingRequired = \
               vlanMappingRequiredStatus.get( 'vlanMappingIngress', False )
         intfVlanMap.egressVlanMappingRequired = \
               vlanMappingRequiredStatus.get( 'vlanMappingEgress', False )
      return intfVlanMap

   # BEGIN showVlanMapping
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   if intfs is None:
      return

   intfVlanMappingRequiredStatus = aggregateVlanMappingRequiredStatus()
   vlanMappings = VlanMappings()

   for intf in intfs:
      if PortChannelIntfId.isPortChannelIntfId( intf.name ) and \
            EbraCliLib.getLagMembersCallBack:
         # get active lagMembers for port channel
         # we cannot do Tac.Type( ""Lag::LagStatusFilter" ).filterOnActive here
         # as we cannot depend on Lag package
         lagMembers = \
           list( EbraCliLib.getLagMembersCallBack( # pylint: disable-msg=not-callable
               mode, intf.name, lacpOnly=False, status='filterOnActive' ) )
         if lagMembers:
            # pylint: disable-next=use-a-generator
            if all( [ intfVlanMappingRequiredStatus.get( m, {} )
               .get( VlanMappingDirection.vlanMappingIngress, False )
               for m in lagMembers ] ):
               intfVlanMappingRequiredStatus[ intf.name ] = \
                     { VlanMappingDirection.vlanMappingIngress: True }
            # pylint: disable-next=use-a-generator
            if all( [ intfVlanMappingRequiredStatus.get( m, {} )
               .get( VlanMappingDirection.vlanMappingEgress, False )
               for m in lagMembers ] ):
               intfVlanMappingRequiredStatus[ intf.name ] = \
                     { VlanMappingDirection.vlanMappingEgress: True }
         for p in lagMembers:
            intfVlanMappingRequiredStatus.pop( p, None )

   for intf in intfs:
      switchIntfConfig = cliConfig.switchIntfConfig.get( intf.name )
      vlanXlateStatus = vlanXlateStatusDir.vlanXlateStatus.get( intf.name )
      vlanMappingRequiredStatus = intfVlanMappingRequiredStatus.get( intf.name )
      if switchIntfConfig or vlanXlateStatus or vlanMappingRequiredStatus:
         vlanMappings.intfVlanMappings[ intf.name ] = \
               createInterfaceVlanMappings( intf, switchIntfConfig,
                     vlanXlateStatus, vlanMappingRequiredStatus )

   return vlanMappings

class ShowIntfSwitchportVlanMapping( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces switchport vlan mapping'
   data = {
      'switchport': switchportForShowKw,
      'vlan': vlanForShowKw,
      'mapping': 'Currently configured VLAN mappings',
   }
   handler = showVlanMapping
   cliModel = VlanMappings

BasicCli.addShowCommandClass( ShowIntfSwitchportVlanMapping )

def aggregateVlanMappingRequiredStatus():
   intfVlanMappingRequired = {}
   for vlanMappingRequiredStatus in \
         vlanMappingRequiredStatusDir.entityPtr.values():
      for intf, status in \
            vlanMappingRequiredStatus.intfVlanMappingRequiredStatus.items():
         intfVlanMappingRequired[ intf ] = (
               dict( list( status.vlanMappingRequired.items() ) ) )
   return intfVlanMappingRequired

# Common token used by both show interface encapsulation (vlan|dot1q vlan)
showVlanEncapKw = CliCommand.singleKeyword( 'encapsulation',
                                            'Show encapsulation information' )
#-------------------------------------------------------------------------------
# The "show interfaces [<name>] encapsulation dot1q vlan" command
#-------------------------------------------------------------------------------
def showEncapsulationDot1QVlan( mode, args ):
   """
   Helper method to populate EncapsulationDot1QVlans' IntfEncapsulations model

    "intfEncapsulations": {
        "Ethernet1/12": {
            "ingressActive": false,
            "vlanId": "N/A",
            "egressActive": false
        },
        "Ethernet1/1": {
            "ingressActive": false,
            "vlanId": "2631",
            "egressActive": false
        },
   }
   """
   encapsulations = EncapsulationDot1QVlans()
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   intfs = intfs or []
   for intf in intfs:
      vid = 0
      inActive = False
      outActive = False

      vlanXlateStatus = vlanXlateStatusDir.vlanXlateStatus.get( intf.name )
      # first look at what is actually programmed into the hardware
      if vlanXlateStatus:
         # values programmed into h/w take precedence over whats configured for show
         if vlanXlateStatus.hwIngressNativeVlanXlate.mappedVlanId > 0:
            vid = vlanXlateStatus.hwIngressNativeVlanXlate.mappedVlanId
         elif vlanXlateStatus.hwEgressNativeVlanXlate.mappedVlanId > 0:
            vid = vlanXlateStatus.hwEgressNativeVlanXlate.mappedVlanId

         if vid > 0:
            inActive = vid == vlanXlateStatus.hwIngressNativeVlanXlate.mappedVlanId
            outActive = vid == vlanXlateStatus.hwEgressNativeVlanXlate.mappedVlanId

      switchIntfConfig = cliConfig.switchIntfConfig.get( intf.name )
      # look at what is configured through the CLI if h/w is not programmed
      if switchIntfConfig and vid <= 0:
         vid = switchIntfConfig.ingressNativeVlanXlate
         # pylint: disable-next=superfluous-parens
         assert( vid == switchIntfConfig.egressNativeVlanXlate )

      # setup the encapsulation details in the model for this interface
      intfEncap = EncapsulationDot1QVlans.InterfaceEncapsulations()
      # we need to translate vlan ID into a string, do this only if its valid
      intfEncap.vlanId = str( vid ) if( vid > 0 ) else "N/A"
      intfEncap.ingressActive = inActive
      intfEncap.egressActive = outActive
      # add an entry for this interface into the encapsulation model
      encapsulations.intfEncapsulations[ intf.name ] = intfEncap
   return encapsulations

class ShowIntfEncapDot1QVlan( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces encapsulation dot1q vlan'
   data = {
      'encapsulation': showVlanEncapKw,
      'dot1q': '802.1q based encapsulation',
      'vlan': 'Show configured Encapsulation VLAN',
   }
   handler = showEncapsulationDot1QVlan
   cliModel = EncapsulationDot1QVlans

BasicCli.addShowCommandClass( ShowIntfEncapDot1QVlan )

def showEncapsulationVlan( mode, args ):
   """
   Helper method to populate EncapsulationVlans' FlexibleEncapsulations model

   "intfEncapsulations": {
       "Ethernet1.1": {
           "entries": [
               {
                   "status": "active",
                   "network": {
                       "specType": "client"
                   },
                   "client": {
                       "specType": "tagged",
                       "encap": {
                           "outer": {
                               "valueType": "tagged",
                               "tags": [
                                   { tag: 100 },
                               ]
                           },
                           "inner": {
                               "valueType": "tagged",
                               "tags": [
                                   { tag: 200 },
                               ]
                           }
                       }
                   }
               }
           ]
       }
   },
   """
   def subIntfConfigs( mode, args ):
      """
      Helper method to returns the list of subIntfConfig with encapsulations
      configured using flexible encapsulation configuration mode
      """
      # use dynamic import to avoid circular dependency
      from CliPlugin import SubIntfCli # pylint: disable=import-outside-toplevel
      intf = args.get( 'INTF' )
      mod = args.get( 'MOD' )
      intfs = ( { i.name
                  for i in IntfCli.Intf.getAll( mode, intf=intf, mod=mod,
                                                intfType=SubIntfCli.SubIntf,
                                                sort=False ) }
                if intf else None )
      for sic in subIntfConfigDir.intfConfig.values():
         if ( sic.dot1qEncapConfigMode ==
                 EncapConfigMode.subIntfDot1qEncapConfigFlexEncap and
              ( not intf or sic.intfId in intfs ) ):
            yield sic

   def populateValue( flexField ):
      """ Helper method to populate FlexEncapValue """
      value = None
      if flexField.type == FlexEncapFieldType.anyOrNone:
         # CAPI has it as optional, so lets return None
         return None
      elif flexField.type in ( FlexEncapFieldType.client, FlexEncapFieldType.dot1q,
                               FlexEncapFieldType.dot1ad,
                               FlexEncapFieldType.userDefinedTpid,
                               FlexEncapFieldType.untagged ):
         value = FlexEncapValue()
         value.valueType = flexField.type
         # Only populate the optional tags field if type is actually tagged
         if flexField.type in ( FlexEncapFieldType.dot1q,
                                FlexEncapFieldType.dot1ad,
                                FlexEncapFieldType.userDefinedTpid ):
            value.tpid = flexField.tpidValue
            for r in flexField.range.values():
               taggedValue = FlexEncapTaggedValue()
               taggedValue.tag = r.lower
               if Toggles.EbraToggleLib.toggleFlexEncapSubIntfMultiEncapEnabled():
                  taggedValue.endTag = r.upper
               else:
                  # We only support single tags
                  assert r.lower == r.upper
               value.tags.append( taggedValue )
         else:
            value.tags = None
      else:
         assert False, "Unexpected field type: {}". format( flexField.type )
      return value

   def populateTaggedSpec( flexSpec ):
      """ Helper method to populate FlexEncapSpec specType tagged """
      assert flexSpec.type == FlexEncapSpecType.tagged
      spec = FlexEncapSpec()
      spec.specType = 'tagged'
      tagged = FlexEncapTaggedSpec()
      # We must have an outer tag
      tagged.outer = populateValue( flexSpec.outer )
      tagged.inner = populateValue( flexSpec.inner )
      spec.tagged = tagged
      return spec

   def populateClientSpec( sicEntry ):
      """
      Helper method to populate client-side FlexEncapSpec.
      'specType client' is not valid on client side.
      """
      flexSpec = sicEntry.client
      # It's not possible to get anyOrNone here until we expand
      # the CLI to also display port connectors coming from
      # PwA
      if flexSpec.type == FlexEncapSpecType.tagged:
         return populateTaggedSpec( flexSpec )
      elif flexSpec.type in ( FlexEncapSpecType.untagged,
                              FlexEncapSpecType.unmatched ):
         spec = FlexEncapSpec()
         spec.specType = flexSpec.type
         return spec
      else:
         assert False, 'Invalid spec type for client'

   def populateNetworkSpec( sicEntry ):
      """
      Helper method to populate network-side FlexEncapSpec.
      """
      if sicEntry.client.type == FlexEncapSpecType.unmatched:
         # For the unmatched case, the network side is initialized with a spec type
         # of 'client'.  Displaying this would be confusing, and could be taken to
         # indicate 'client unmatched network client' is the config, which is not the
         # case.  No network portion is even allowed for the unmatched case.  So,
         # display nothing for the network side.
         return None

      flexSpec = sicEntry.network
      spec = None
      if flexSpec.type in ( FlexEncapSpecType.client,
                            FlexEncapSpecType.clientInner,
                            FlexEncapSpecType.untagged ):
         spec = FlexEncapSpec()
         spec.specType = flexSpec.type
      elif flexSpec.type == FlexEncapSpecType.anyOrNone:
         # return none, as the CAPI model has it as optional
         pass
      elif flexSpec.type == FlexEncapSpecType.tagged:
         spec = populateTaggedSpec( flexSpec )
      return spec

   def encapsulationStatus( clientId, sicEntry, sis, clientConflict,
                            networkConflict ):
      if sicEntry.hasOverlap():
         return 'overlap'

      if sicEntry.isAsymmetric():
         return 'asymmetric'

      if ( clientConflict and
           ( clientConflict.hasConflictForServiceIntf() or
             clientId in clientConflict.clientIdConflict ) ):
         return 'conflict'

      if ( networkConflict and
           ( networkConflict.hasConflictForServiceIntf() or
             clientId in networkConflict.clientIdConflict ) ):
         return 'conflict'

      # Subinterface status must have a status, be operationally up, and have
      # no encapsulation status issues.
      if ( sis and sis.operStatus == IntfOperStatus.intfOperUp ):
         return 'active'

      return 'inactive'

   # pylint: disable-next=unused-variable
   EntryState = TacLazyType( 'Ebra::FlexEncapEntry::EntryState' )
   encapsulations = EncapsulationVlans()
   sics = subIntfConfigs( mode, args )
   for sic in sics:
      intfName = sic.intfId
      sis = subIntfStatusDir.intfStatus.get( intfName )

      # Get conflict status collections.
      clientConflict = conflictStatusClient.serviceIntfConflict.get( intfName )
      networkConflict = conflictStatusNetwork.serviceIntfConflict.get( intfName )

      # Need to check the interface winner, because if Ebra is the winner, there will
      # not be a network response info entry.
      winner = serviceIntfTypeWinner.serviceIntfType.get( intfName )
      # pylint: disable-next=unused-variable
      ebraWinner = ( winner == ServiceIntfType.l2 or winner == ServiceIntfType.l3 )
      # Populate the encap model layer.
      modelFlexEncap = FlexibleEncapsulations()
      if sic.flexEncap is not None:
         for clientId in sorted( sic.flexEncap.entry ):
            # Determine where to check the entry state:  config or response.
            sicEntry = sic.flexEncap.entry[ clientId ]

            # Populate the encap entry model based on the entry being examined.
            modelEntry = FlexEncapEntry()
            modelEntry.clientId = clientId

            # Encapsulation status.
            modelEntry.status = encapsulationStatus(
               clientId, sicEntry, sis, clientConflict, networkConflict )

            # Populate the specs from the config entry.
            modelEntry.client = populateClientSpec( sicEntry )
            modelEntry.network = populateNetworkSpec( sicEntry )
            if clientId == FlexEncapConstants.defaultClientId:
               # Make the default clientId the first entry in the list.
               modelFlexEncap.entries.insert( 0, modelEntry )
            else:
               modelFlexEncap.entries.append( modelEntry )
         encapsulations.intfEncapsulations[ intfName ] = modelFlexEncap
   return encapsulations

class ShowIntfEncapVlan( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces encapsulation vlan'
   data = {
      'encapsulation': showVlanEncapKw,
      'vlan': 'VLAN encapsulation',
   }
   handler = showEncapsulationVlan
   cliModel = EncapsulationVlans

BasicCli.addShowCommandClass( ShowIntfEncapVlan )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] switchport configuration source" command
#-------------------------------------------------------------------------------
def showSwitchportConfigSources( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args

   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )

   switchportsConfigSources = BridgingCliModel.SwitchportsConfigSources()
   if not intfs:
      return switchportsConfigSources

   for intf in intfs:
      if intf.name in bridgingSwitchIntfConfig.switchIntfConfig:
         configSource = BridgingCliModel.ConfigSource()
         # Print eth as default to avoid confusion over our internals
         if bridgingSwitchIntfConfig.switchIntfConfig[ intf.name ].source == 'eth':
            configSource.source = 'default'
         else:
            configSource.source = \
                    bridgingSwitchIntfConfig.switchIntfConfig[ intf.name ].source
         cmds = CliSaveBlock.CommandSequence( None )
         addSwitchIntfConfigCmds( cmds,
                  bridgingSwitchIntfConfig.switchIntfConfig[ intf.name ],
                                  bridgingConfig, detail, False, cliConfig )
         for cmd in cmds.commands_:
            configSource.config.append( cmd )
         switchportsConfigSources.interfaces[ intf.name ] = configSource
   return switchportsConfigSources

class ShowIntfSwitchportConfigSource( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces switchport configuration source [ detail ]'
   data = {
      'switchport': switchportForShowKw,
      'configuration': 'Show the status of switchport\'s configuration',
      'source': 'Show the switchport\'s configuration source',
      'detail': 'Output default config commands for each source',
   }
   handler = showSwitchportConfigSources
   cliModel = "BridgingCliModel.SwitchportsConfigSources"

BasicCli.addShowCommandClass( ShowIntfSwitchportConfigSource )

#-------------------------------------------------------------------------------
# Associate the SwitchportModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( SwitchportModelet )


#-------------------------------------------------------------------------------
# The "vlan database" command, in "enable" mode.
#-------------------------------------------------------------------------------

def displayVlanDatabaseMsg( mode, args ):
   mode.addError( 'VLAN database mode is not supported.  Please configure VLANs '
                  'from config mode.' )

class VlanDatabaseCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan database'
   data = {
      'vlan': 'Configure VLAN parameters',
      'database': 'Configure VLAN database',
   }

   hidden = True
   handler = displayVlanDatabaseMsg

BasicCliModes.EnableMode.addCommandClass( VlanDatabaseCmd )

#-------------------------------------------------------------------------------
# The "[no] vlan dynamic range VLANS" command
#
# The full syntax of this command is:
#
#   [no|default] vlan dynamic range VLAN_SET
#-------------------------------------------------------------------------------

def setVlanDynamicRange( mode, args ):
   cliConfig.dynVlanRange = MultiRangeRule.multiRangeToCanonicalString(
         args[ 'VLAN_SET' ] )

def noVlanDynamicRange( mode, args ):
   cliConfig.dynVlanRange = '1-4094'

class VlanDynamicRangeVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan dynamic range VLAN_SET'
   noOrDefaultSyntax = 'vlan dynamic range ...'
   data = {
      'vlan': nodeVlan,
      'dynamic': 'Configure dynamic VLANs properties',
      'range': 'Specify range of VLAN IDs',
      'VLAN_SET': vlanSetMatcher,
   }

   allowCache = False
   handler = setVlanDynamicRange
   noOrDefaultHandler = noVlanDynamicRange

BasicCliModes.GlobalConfigMode.addCommandClass( VlanDynamicRangeVlansetCmd )

#-------------------------------------------------------------------------------
# The "[no] switchport phone vlan <vlan-id>" command
# The "[no] switchport phone trunk { tagged [ phone ] | untagged [ phone ] }" command
# The "[no] switchport default phone vlan <vlan-id>" command
# The "[no] switchport default phone cos <0-7>" command
# The '[no] switchport default phone trunk tagged [ phone ] | \
#     untagged [ phone ] " command
#
# The full syntax of this command is:
#
#  [no|default] switchport phone vlan <vlan-id>
#  [no|default] switchport phone trunk untagged | tagged
#  [no|default] switchport default phone vlan <vlan-id>
#  [no|default] switchport default phone cos <0-7>
#  [no|default] switchport default phone trunk tagged [ phone ] | \
#  untagged [ phone ]
#-------------------------------------------------------------------------------

defaultMatcher = CliMatcher.KeywordMatcher( 'default',
   helpdesc='Configure switchport default behavior' )
trunkNode = CliCommand.guardedKeyword( 'trunk',
   helpdesc='Set trunking characteristics of the interface',
   guard=bridgingSupportedGuard )

defaultCos = 5
noPhoneVlanId = 0
TagMode = Tac.Type( "Bridging::TagMode" )
tagModeMap = { 'none': 'notSet',
                'untagged' : 'phoneTrunkUntagged',
                'tagged' : 'phoneTrunkTagged' }

def setPhoneVlan( mode, args ):
   vlanId = args.get( 'VLAN_ID', Vlan( noPhoneVlanId ) )
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.phoneVlan = vlanId.id
   if vlanId.id != 0:
      if not vlanId.lookupConfig( mode ):
         mode.addWarning( 'VLAN id %d not found in current VLAN configuration' %
                          vlanId.id )

def setDefaultPhoneVlan( mode, args ):
   vlanId = args.get( 'VLAN_ID', Vlan( noPhoneVlanId ) )
   cliConfig.defaultPhoneVlan = vlanId.id

def setDefaultCos( mode, args ):
   cliConfig.defaultPhoneCos = args.get( 'COS', defaultCos )

def setDefaultTrustMode( mode, args ):
   trustMode = args[ 'MODE' ]
   cliConfig.defaultPhoneTrustMode = trustMode

def noDefaultTrustMode( mode, args ):
   trustMode = TrustMode.untrusted
   cliConfig.defaultPhoneTrustMode = trustMode

def setPhoneAclBypass( mode, args ):
   cliConfig.defaultPhoneAclBypass = True

def noPhoneAclBypass( mode, args ):
   cliConfig.defaultPhoneAclBypass = False

def noDefaultPhoneTrunkTagMode( mode, args ):
   cliConfig.defaultPhoneTrunkTagMode = TagMode.notSet

def setDefaultPhoneTrunkTagMode( mode, args ):
   if args.get( 'PHONE_KW1' ):
      cliConfig.defaultPhoneTrunkTagMode = TagMode.phoneVlanTagged
   elif 'tagged' in args:
      cliConfig.defaultPhoneTrunkTagMode = TagMode.phoneTrunkTagged
   elif args.get( 'PHONE_KW2' ):
      cliConfig.defaultPhoneTrunkTagMode = TagMode.phoneVlanUntagged
   else:
      cliConfig.defaultPhoneTrunkTagMode = TagMode.phoneTrunkUntagged

def noPhoneTrunkTagMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.phoneTrunkTagMode = TagMode.notSet

def setPhoneTrunkTagMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   if args.get( 'PHONE_KW1' ):
      sic.phoneTrunkTagMode = TagMode.phoneVlanTagged
   elif 'tagged' in args:
      sic.phoneTrunkTagMode = TagMode.phoneTrunkTagged
   elif args.get( 'PHONE_KW2' ):
      sic.phoneTrunkTagMode = TagMode.phoneVlanUntagged
   else:
      sic.phoneTrunkTagMode = TagMode.phoneTrunkUntagged

class SwitchportPhoneVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport phone vlan VLAN_ID'
   noOrDefaultSyntax = 'switchport phone vlan ...'
   data = {
      'switchport': switchportMatcher,
      'phone': phoneNode,
      'vlan': vlanMatcher,
      'VLAN_ID': vlanIdMatcher,
   }
   handler = setPhoneVlan
   noOrDefaultHandler = setPhoneVlan

SwitchportModelet.addCommandClass( SwitchportPhoneVlanVlanidCmd )

class SwitchportDefaultPhoneVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default phone vlan VLAN_ID'
   noOrDefaultSyntax = 'switchport default phone vlan ...'
   data = {
      'switchport': switchportMatcher,
      'default': defaultMatcher,
      'phone': phoneNode,
      'vlan': vlanMatcher,
      'VLAN_ID': vlanIdMatcher,
   }
   handler = setDefaultPhoneVlan
   noOrDefaultHandler = setDefaultPhoneVlan

BasicCliModes.GlobalConfigMode.addCommandClass( SwitchportDefaultPhoneVlanVlanidCmd )

class SwitchportDefaultPhoneCosCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default phone cos COS'
   noOrDefaultSyntax = 'switchport default phone cos ...'
   data = {
      'switchport': switchportMatcher,
      'default': defaultMatcher,
      'phone': phoneNode,
      'cos': 'Configure COS',
      'COS': CliMatcher.IntegerMatcher( 0, 7, helpdesc='COS range' ),
   }
   handler = setDefaultCos
   noOrDefaultHandler = setDefaultCos

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportDefaultPhoneCosCmd )

nodeTrust = CliCommand.guardedKeyword( 'trust',
      helpdesc='Configure trust mode',
      guard=phoneTrustModeGuard )

class SwitchportDefaultPhoneTrustModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default phone qos trust MODE'
   noOrDefaultSyntax = 'switchport default phone qos trust ...'
   data = {
      'switchport': switchportMatcher,
      'default': defaultMatcher,
      'phone': phoneNode,
      'qos': 'Configure QOS',
      'trust' : nodeTrust,
      'MODE': CliMatcher.EnumMatcher( {
        'cos': 'Set trust mode to CoS',
        'dscp': 'Set trust mode to DSCP',
      } )
   }
   handler = setDefaultTrustMode
   noOrDefaultHandler = noDefaultTrustMode

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportDefaultPhoneTrustModeCmd )

class SwitchportDefaultPhoneAclBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default phone access-list bypass'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': switchportMatcher,
      'default': defaultMatcher,
      'phone': phoneNode,
      'access-list': 'Port access-list configuration',
      'bypass': 'Bypass phone traffic from configured access-list'
   }
   handler = setPhoneAclBypass
   noOrDefaultHandler = noPhoneAclBypass

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportDefaultPhoneAclBypassCmd )

class SwitchportPhoneTrunkTaggedCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport phone trunk'\
         '( ( tagged [ PHONE_KW1 ] ) | ( untagged [ PHONE_KW2 ] ) )'
   noOrDefaultSyntax = 'switchport phone trunk ...'
   data = {
      'switchport': switchportMatcher,
      'phone': phoneNode,
      'trunk': trunkNode,
      'tagged': 'Configure the port to send tagged packets',
      'PHONE_KW1': CliMatcher.KeywordMatcher( 'phone',
                     helpdesc='Tagged packets on phone VLAN' ),
      'untagged': 'Configure the port to send untagged packets',
      'PHONE_KW2': CliMatcher.KeywordMatcher( 'phone',
                     helpdesc='Untagged packets on phone VLAN' ),
   }
   handler = setPhoneTrunkTagMode
   noOrDefaultHandler = noPhoneTrunkTagMode

SwitchportModelet.addCommandClass( SwitchportPhoneTrunkTaggedCmd )

class SwitchportDefaultPhoneTrunkUntaggedCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default phone trunk'\
         '( ( tagged [ PHONE_KW1 ] ) | ( untagged [ PHONE_KW2 ] ) )'
   noOrDefaultSyntax = 'switchport default phone trunk ...'
   data = {
      'switchport': switchportMatcher,
      'default': defaultMatcher,
      'phone': phoneNode,
      'trunk': trunkNode,
      'tagged': 'Configure the port to send tagged packets',
      'PHONE_KW1': CliMatcher.KeywordMatcher( 'phone',
                     helpdesc='Tagged packets on phone VLAN' ),
      'untagged': 'Configure the port to send untagged packets',
      'PHONE_KW2': CliMatcher.KeywordMatcher( 'phone',
                     helpdesc='Untagged packets on phone VLAN' ),
   }
   handler = setDefaultPhoneTrunkTagMode
   noOrDefaultHandler = noDefaultPhoneTrunkTagMode

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportDefaultPhoneTrunkUntaggedCmd )

# --------------------------------------------------------------------------
# [ no | default ] switchport vlan tag validation
# --------------------------------------------------------------------------
def vlanTagValidationGuard( mode, token ):
   if bridgingHwCapabilities.vlanTagValidationSupported:
      return None
   return CliParser.guardNotThisPlatform

def enableVlanTagValidation( mode, args ):
   cliConfig.vlanTagValidationEnabled = True

def disableVlanTagValidation( mode, args ):
   cliConfig.vlanTagValidationEnabled = False

tagMatcher = CliMatcher.KeywordMatcher( 'tag',
   helpdesc='Details on VLAN tag operation' )
validationMatcher = CliCommand.guardedKeyword( 'validation',
   helpdesc='Enable VLAN tag validation',
   guard=vlanTagValidationGuard )

class SwitchportVlanTagValidationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport vlan tag validation'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': switchportMatcher,
      'vlan': vlanMatcher,
      'tag': tagMatcher,
      'validation': validationMatcher,
   }
   handler = enableVlanTagValidation
   noOrDefaultHandler = disableVlanTagValidation

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportVlanTagValidationCmd )

# --------------------------------------------------------------------------
# [ no | default ] switchport ethernet llc validation
# --------------------------------------------------------------------------
def llcValidationGuard( mode, token ):
   if bridgingHwCapabilities.llcValidationSupported:
      return None
   return CliParser.guardNotThisPlatform

def enableLlcValidation( mode, args ):
   cliConfig.llcValidationEnabled = True

def disableLlcValidation( mode, args ):
   cliConfig.llcValidationEnabled = False

llcMatcher = CliMatcher.KeywordMatcher( 'llc',
   helpdesc='Details on Ethernet 802.2 LLC header operation' )

ethernetLlcMatcher = CliMatcher.KeywordMatcher( 'ethernet',
   helpdesc='Details on Ethernet 802.2 LLC header operation' )

llcValidationMatcher = CliCommand.guardedKeyword( 'validation',
   helpdesc='Enable Ethernet LLC header validation',
   guard=llcValidationGuard )

class SwitchportLlcValidationCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport ethernet llc validation'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': switchportMatcher,
      'ethernet': ethernetLlcMatcher,
      'llc': llcMatcher,
      'validation': llcValidationMatcher,
   }
   handler = enableLlcValidation
   noOrDefaultHandler = disableLlcValidation

BasicCliModes.GlobalConfigMode.addCommandClass(
      SwitchportLlcValidationCmd )

class ShowEtreeVlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show e-tree vlan'
   data = {
      'e-tree': 'E-Tree configuration information',
      'vlan': 'See VLAN E-Tree role configuration'
   }
   cliModel = VlanEtreeRolesModel

   @staticmethod
   def handler( mode, args ):
      model = VlanEtreeRolesModel()
      for vlanId, vlanConfig in cliConfig.vlanConfig.items():
         if vlanConfig.etreeRole == EtreeRole.etreeRoleLeaf:
            remoteLeafInstallMode = None
            if vlanConfig.etreeRemoteLeafInstallMode == \
                  EtreeRemoteLeafInstallMode.dontInstall:
               remoteLeafInstallMode = 'dontinstall'
            elif vlanConfig.etreeRemoteLeafInstallMode == \
                  EtreeRemoteLeafInstallMode.installDrop:
               remoteLeafInstallMode = 'installdrop'
            model.vlans[ vlanId ] = EtreeRoleModel( etreeRoleLeaf=True,
                  remoteLeafInstallMode=remoteLeafInstallMode )
      return model

BasicCli.addShowCommandClass( ShowEtreeVlanCmd )

# ------------------------------------------------------------------------------
# The "show split-horizon group [vlan ( <vid> | name <vlan_name>) ] [shgname]"
# command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show split-horizon group [vlan (<vlan_set> | name <vlan_name>) ] [<shg_name>]
# -------------------------------------------------------------------------------

def splitHorizonGroupSupportedGuard( mode, token ):
   if intfCapability.splitHorizonGroupSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

splitHorizonGroupMatcher = CliMatcher.PatternMatcher( pattern='.+',
         helpname='WORD',
         helpdesc='Split horizon group name',
         value=lambda mode, match: match[ : maxSplitHorizonGroupNameSize ] )

showSplitHorizonGroupStatusHook = CliExtensions.CliHook()
def doShowSplitHorizonGroups( mode, args ):
   model = SplitHorizonGroups()

   assert len( showSplitHorizonGroupStatusHook.extensions() ) <= 1
   for func in showSplitHorizonGroupStatusHook.extensions():
      model = func( mode, args )

   return model

class ShowSplitHorizonCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show split-horizon group '
              '[ vlan ( VLAN_SET | ( name VLAN_NAME ) ) ] '
              '[ SHGNAME ]' )
   data = {
      'split-horizon': CliCommand.guardedKeyword(
         'split-horizon',
         helpdesc='Show information about split horizon groups',
         guard=splitHorizonGroupSupportedGuard ),
      'group': 'Show information about split horizon groups',
      'vlan': 'Show groups in a VLAN',
      'name': 'Show groups in a VLAN by VLAN name',
      'VLAN_SET': vlanSetMatcher,
      'SHGNAME': splitHorizonGroupMatcher,
      'VLAN_NAME': vlanNameMatcher,
   }

   handler = doShowSplitHorizonGroups
   cliModel = SplitHorizonGroups

BasicCli.addShowCommandClass( ShowSplitHorizonCmd )

# ------------------------------------------------------------------------------
# The "[no] vlan tpid" command
# -------------------------------------------------------------------------------

def userDefinedTpidsSupportedGuard( mode, token ):
   if intfCapability.userDefinedTpidsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class VlanTpidCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan tpid TPID'
   noOrDefaultSyntax = syntax
   data = {
      'vlan': nodeVlan,
      'tpid': CliCommand.guardedKeyword( 'tpid',
                                         'TPID Registration',
                                         guard=userDefinedTpidsSupportedGuard ),
      'TPID': vlanTpidMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      tpid = args[ 'TPID' ]
      cliConfig.vlanTpid.add( tpid )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tpid = args[ 'TPID' ]
      cliConfig.vlanTpid.remove( tpid )


BasicCliModes.GlobalConfigMode.addCommandClass( VlanTpidCmd )

def Plugin( entityManager ):
   global cliConfig, bridgingConfig, vlanXlateStatusDir, bridgingSwitchIntfConfig
   global vlanFloodsetExpandedStatusDir
   global bridgingHwCapabilities, bridgingHwEnabled
   global intfCapability
   global topologyConfig, ethIntfStatusDir
   global topologyConfig
   global dynVlanDir
   global dynVlanIntfDir
   global dynAllowedVlanDir
   global dynBlockedVlanDir
   global peerVlanSet
   global vlanIntfConfigDir
   global vlanCountersCliReq, vlanCounterDir, vlanCounterSnapshotDir
   global responseReservedVlanId
   global requestPseudowireReservedVlanId
   global requestIpsecReservedVlanId
   global requestSandGreTunnelReservedVlanId
   global requestGreTunnelReservedVlanId
   global bridgingInputConfigDir
   global vlanMappingRequiredStatusDir
   global conflictStatusDir
   global conflictStatusClient
   global conflictStatusNetwork
   global serviceIntfTypeWinner
   global subIntfConfigDir
   global subIntfStatusDir

   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli",
                                  "Bridging::Input::CliConfig", "w" )
   mount = LazyMount.mount
   bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                           "Bridging::HwCapabilities", "r" )
   bridgingHwEnabled = mount( entityManager, "bridging/hwenabled",
                               "Bridging::HwEnabled", "r" )
   bridgingConfig = mount( entityManager, "bridging/config",
                           "Bridging::Config", "r" )
   vlanXlateStatusDir = mount( entityManager, "bridging/vlanxlate/status",
                           "Bridging::VlanXlateStatusDir", "r" )
   vlanFloodsetExpandedStatusDir = mount( entityManager,
                                          "bridging/vlan/floodsetExpandedVlanStatus",
                                          "Bridging::VlanFloodsetExpandedStatusDir",
                                          "r" )
   bridgingSwitchIntfConfig = mount( entityManager, "bridging/switchIntfConfig",
                                     "Bridging::SwitchIntfConfigDir", "r" )
   topologyConfig = mount( entityManager, "bridging/topology/config",
                           "Bridging::Topology::Config", "r" )
   ethIntfStatusDir = mount( entityManager, "interface/status/eth/intf",
                             "Interface::EthIntfStatusDir", "r" )
   dynAllowedVlanDir = mount( entityManager, "bridging/input/dynvlan/allowedvlan",
                               "Tac::Dir", "ri" )
   dynBlockedVlanDir = mount( entityManager, "bridging/input/dynBlockedVlan",
                              "Tac::Dir", "ri" )
   dynVlanIntfDir = mount( entityManager, "interface/input/dynvlanintf",
                       "Tac::Dir", "ri" )
   dynVlanDir = mount( entityManager, "bridging/input/dynvlan/vlan",
                       "Tac::Dir", "ri" )
   peerVlanSet = mount( entityManager, "bridging/input/peervlan",
                        "Bridging::Input::VlanIdSet", "r" )
   vlanIntfConfigDir = mount( entityManager, "interface/config/eth/vlan",
                              "Interface::VlanIntfConfigDir", "r" )
   vlanCountersCliReq = mount( entityManager, "bridging/vlan/clireq",
                               "Bridging::VlanCountersCliReq", "w" )
   responseReservedVlanId = mount( entityManager, "bridging/external/status",
                                   "EbraExt::ResponseReservedVlanId", "r" )
   requestPseudowireReservedVlanId = mount( entityManager,
                                    "bridging/external/requestVlanId/pseudowire",
                                    "EbraExt::RequestReservedVlanId", "r" )
   requestIpsecReservedVlanId = mount( entityManager,
                                    "bridging/external/requestVlanId/ipsec",
                                    "EbraExt::RequestReservedVlanId", "r" )
   requestGreTunnelReservedVlanId = mount( entityManager,
                                    "bridging/external/requestVlanId/gretunnel",
                                    "EbraExt::RequestReservedVlanId", "r" )
   requestSandGreTunnelReservedVlanId = mount( entityManager,
                                    "bridging/external/requestVlanId/sandgretunnel",
                                    "EbraExt::RequestReservedVlanId", "r" )
   bridgingInputConfigDir = mount( entityManager, "bridging/input/config",
                                   "Tac::Dir", "ri" )
   vlanMappingRequiredStatusDir = mount( entityManager,
                                         "bridging/vlanxlate/requiredstatus",
                                         "Tac::Dir", "ri" )
   conflictStatusDir = mount( entityManager,
                              "interface/external/vlanTag/conflictStatus",
                              "Tac::Dir", "ri" )
   conflictStatusClient = mount( entityManager,
                                 "interface/external/vlanTag/conflictStatus/ebra",
                                 "EbraExt::VlanTagConflictStatus", "r" )
   conflictStatusNetwork = mount(
      entityManager, "interface/external/vlanTag/conflictStatus/pseudowire",
      "EbraExt::VlanTagConflictStatus", "r" )
   serviceIntfTypeWinner = mount( entityManager,
                                  "interface/external/serviceType/winner",
                                  "EbraExt::ServiceIntfTypeWinner", "r" )
   subIntfConfigDir = mount( entityManager, "interface/config/subintf",
                             "Interface::SubIntfConfigDir", "r" )
   subIntfStatusDir = mount( entityManager, "interface/status/subintf",
                             "Interface::SubIntfStatusDir", "r" )
   vlanCounterDir = SmashLazyMount.mount( entityManager, "bridging/vlan/counter",
      "Smash::Bridging::VlanCounterDir", SmashLazyMount.mountInfo( 'reader' ) )
   vlanCounterSnapshotDir = SmashLazyMount.mount( entityManager,
      "bridging/vlan/counterSnapshot", "Smash::Bridging::VlanCounterSnapshotDir",
      SmashLazyMount.mountInfo( 'reader' ) )
   intfCapability = mount( entityManager, "interface/hardware/capability",
                           "Interface::Hardware::Capability", "r" )
   priority = 10 # arbitrarily chosen priority.
   IntfCli.Intf.registerDependentClass( SwitchIntfConfigCleaner, priority )
