# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Vlan interface type.  In particular, it provides
# the VlanIntf class.
#-------------------------------------------------------------------------------
import Tac
import CliParser
import re
import Ethernet, LazyMount
import ConfigMount
import SharedMem
import Smash
from IntfRangePlugin.VlanIntf import VlanAutoIntfType
from CliPlugin.BridgingCli import warnMacTableUnsupportedUFTModeHook
from CliPlugin.VlanCli import bridgingSupportedGuard, vlanInUseErrStr
from CliPlugin.IntfModel import InterfaceCounters
from CliPlugin.VlanIntfModel import VlanInterfaceCounters
from CliPlugin.VlanIntfModel import VlanInterfaceStatistics
from CliPlugin.VlanIntfModel import VlanInterfaceStatus
from CliPlugin.VlanIntfModel import VlanInterfaceCountersRate
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.VirtualIntfRule as VirtualIntfRule
from math import exp
import CliCommand

vlanIntfConfigDir = None
dynVlanIntfConfigDir = None
vlanIntfStatusDir = None
bridgingConfig = None
bridgingHwEnabled = None
brHwCapabilities = None
cliConfig = None
systemName = None
entityManager = None

countersDir = {}
countersCheckpointDir = {}
countersCheckpointSm = {}

#-------------------------------------------------------------------------------
# A subclass of the base IntfCli.Intf class for Vlan interfaces.
#-------------------------------------------------------------------------------
class VlanIntf( IntfCli.VirtualIntf ):

   #----------------------------------------------------------------------------
   # Creates a new VlanIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      m = re.match( r'Vlan(\d+)$', name )
      self.vlanId = int( m.group( 1 ) )
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.intfConfigDir = vlanIntfConfigDir
      self.dynIntfConfigDir = dynVlanIntfConfigDir
      self.intfStatuses = vlanIntfStatusDir.intfStatus
      self.intfStatus = None
      self.intfConfig = None
      self.dynIntfConfig = None
      self.intfCountersModel = None
      self.intfCountersDiscardsModel = None
      self.intfCountersRateModel = None

   #----------------------------------------------------------------------------
   # The rule for matching Vlan interface names.  When this pattern matches, it
   # returns an instance of the VlanIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   valueFunc = lambda mode, intf: VlanIntf( intf, mode )
   matcher = VirtualIntfRule.VirtualIntfMatcher( 'Vlan', 1, 4094,
                                                 value=valueFunc,
                                                 guard=bridgingSupportedGuard )

   #----------------------------------------------------------------------------
   # If a VLAN ID is in use as an internal vlan by a routed port,
   # VlanIntf creation for that VLAN ID is not supported.
   #----------------------------------------------------------------------------
   def intfCreationCurrentlySupported( self, mode ):
      vc = bridgingConfig.vlanConfig.get( self.vlanId )
      # Do not allow creation of SVIs in interactive sessions only.
      if vc and vc.internal and mode.session_.interactive_:
         mode.addError( vlanInUseErrStr( self.vlanId ) )
         return False
      return True

   #----------------------------------------------------------------------------
   # Creates the Interface::VlanIntfConfig object for this interface if it does
   # not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      ic = self.intfConfigDir.intfConfig.newMember( self.name )
      ic.source = 'cli'
      # Hook is populated on Trident4 platforms to warn of conflicting configuration
      warnMacTableUnsupportedUFTModeHook.notifyExtensions( self.mode_ )

   #----------------------------------------------------------------------------
   # Determines whether the Interface::VlanIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.intfStatuses.get( self.name )
      return ( self.intfStatus is not None ) # pylint: disable=superfluous-parens

   #----------------------------------------------------------------------------
   # Destroys the Interface::VlanIntfConfig object for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del self.intfConfigDir.intfConfig[ self.name ]
      self.destroySviMappingState()

   #----------------------------------------------------------------------------
   # Destroys the sviMapping state for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroySviMappingState( self ):
      sviMappingState = cliConfig.sviMapping.get( self.vlanId )
      if sviMappingState:
         sviMappingState.defaultSviMapping = True
      del cliConfig.sviMapping[ self.vlanId ]

   #----------------------------------------------------------------------------
   # Returns the VlanIntfConfig object for this interface.
   # VlanIntfConfig is cached to prevent an update from the EntityLog thread
   # in the middle of processing a "show interfaces" command from causing an
   # inconsistent snapshot to be displayed.
   # The caching is done only if not in a config session, since the above
   # scenario cannot happen when in a config session.
   #----------------------------------------------------------------------------
   def config( self ):
      config = self.intfConfigDir.intfConfig.get( self.name )
      if not self.mode_.session_.inConfigSession():
         self.intfConfig = config
      return config

   #----------------------------------------------------------------------------
   # VLAN interfaces can be requested by Agents and created by Ebra. The requested
   # VlanIntfConfig appears under interface/input/dynvlanintf and is 'r' mounted
   # since they are created by the requesting Agent.
   #
   # This dynConfig should only used by show commands and follow the
   #  intf.config() or intf.dynConfig()
   # pattern. Attempts to mutate the VlanIntfConfig created by CLI should only
   # do config()
   #----------------------------------------------------------------------------
   def dynConfig( self ):
      def _dynConfig():
         intfStatus = self.status()
         if intfStatus:
            source = intfStatus.source
            if source != 'cli':
               if source in self.dynIntfConfigDir.entityPtr and \
                  source in self.dynIntfConfigDir.entryState:
                  return self.dynIntfConfigDir[ source ].intfConfig.get( self.name )
         return None

      dynConfig = _dynConfig()
      if not self.mode_.session_.inConfigSession():
         self.dynIntfConfig = dynConfig
      return dynConfig

   #----------------------------------------------------------------------------
   # Returns the VlanIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      if not self.intfStatus:
         self.intfStatus = self.intfStatuses.get( self.name )
      return self.intfStatus

   #----------------------------------------------------------------------------
   # Static method returns True if only layer-3 counters are supported. When
   # True the current counters are at Smash path:
   #    interface/counter/svi/current, Smash::Interface::AllIntfCounterDir,
   # and checkpoint/snapshot are at Smash path
   #    interface/counter/svi/snapshot,
   #    Smash::Interface::AllIntfCounterSnapshotDir.
   # Otherwise, the current counters are at Smash path:
   #    interface/cli/counters/vlan, Interface::AllVlanIntfCounterDir,
   # and checkpoint counters are in Sysdb at
   #    interface/cli/counters/vlan, Interface::AllVlanIntfCounterDir
   #----------------------------------------------------------------------------
   @staticmethod
   def l3InterfaceCounters():
      return brHwCapabilities.vlanIntfL3CountersSupported

   #----------------------------------------------------------------------------
   # Returns the VlanInterfaceCounters object for this interface.
   #----------------------------------------------------------------------------
   def counter( self ):
      mountCounters( )
      return countersDir[ systemName ].counter.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the EthIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ): # pylint: disable=inconsistent-return-statements
      if self.counterCapable():
         mountCounters( )
         # we do not expect to come here for layer-3 counters
         assert not VlanIntf.l3InterfaceCounters()
         return countersCheckpointDir[ systemName ]

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      if VlanIntf.l3InterfaceCounters():
         # L3 interface counters not yet shown along with the status
         return
      if self.countersSupported():
         intfStatusModel.interfaceStatistics = self.getInterfaceStatistics()
         intfStatusModel.interfaceCounters = self.getInterfaceCounters()

   def getLastUpdateTimeStamp( self ):
      ckpt = self.getLatestCounterCheckpoint()
      if ckpt:
         statsUpdateTime = getattr( ckpt.rates, 'statsUpdateTime', None )
      else:
         statsUpdateTime = None
      if statsUpdateTime is not None:
         return statsUpdateTime
      else:
         return None

   def getIntfStatusModel( self ):
      return VlanInterfaceStatus( name=self.status().intfId )

   def getInterfaceCounters( self, notFoundString=None, checkpoint=None,
                             zeroOut=False ):
      if zeroOut:
         return None
      counter = self.counter()
      if counter is None:
         return None
      if checkpoint is None:
         ckpt = self.getLatestCounterCheckpoint()
      else:
         ckpt = checkpoint
      def stat( attr, supported=True ):
         if not supported:
            return None
         sysdbValue = getattr( counter.statistics, attr, None )
         if sysdbValue is None:
            sysdbValue = getattr( counter.vlanStatistics, attr, None )
         if sysdbValue is None:
            return notFoundString
         checkpointValue = 0
         if ckpt:
            checkpointValue = getattr( ckpt.statistics, attr, None )
            if checkpointValue is None:
               checkpointValue = getattr( ckpt.vlanStatistics, attr, 0 )
         return sysdbValue - checkpointValue

      interfaceCounters = VlanInterfaceCounters()

      # Packet counters.
      interfaceCounters.inUcastPkts = stat( "inUcastPkts",
            self.inUcastPktCountersSupported() )
      interfaceCounters.outUcastPkts = stat( "outUcastPkts",
            self.outUcastPktCountersSupported() )
      interfaceCounters.inMulticastPkts = stat( "inMulticastPkts",
            self.inMcastPktCountersSupported() )
      interfaceCounters.outMulticastPkts = stat( "outMulticastPkts",
            self.outMcastPktCountersSupported() )

      # Octet counters.
      interfaceCounters.inOctets = stat( "inOctets",
            self.inOctetCountersSupported() )
      interfaceCounters.outOctets = stat( "outOctets",
            self.outOctetCountersSupported() )
      interfaceCounters.inUcastOctets = stat( "inUcastOctets",
            self.inUcastOctetCountersSupported() )
      interfaceCounters.outUcastOctets = stat( "outUcastOctets",
            self.outUcastOctetCountersSupported() )
      interfaceCounters.inMulticastOctets = stat( "inMulticastOctets",
            self.inMcastOctetCountersSupported() )
      interfaceCounters.outMulticastOctets = stat( "outMulticastOctets",
            self.outMcastOctetCountersSupported() )

      lastStatsUpdateTime = self.getLastUpdateTimeStamp()
      if lastStatsUpdateTime:
         interfaceCounters.lastClear = lastStatsUpdateTime + Tac.utcNow() - Tac.now()
      else:
         interfaceCounters.lastClear = None

      interfaceCounters._name = self.name # pylint: disable=protected-access

      interfaceCounters.counterRefreshTime = Tac.utcNow()
      interfaceCounters.linkStatusChanges = self.status().operStatusChange.count - \
                                            getattr( ckpt, "linkStatusChanges", 0 )

      return interfaceCounters

   def updateL3InterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                       zeroOut=False ):
      #future taks to reuse getInterfaceCounters above
      intfCountersModel = InterfaceCounters(
            _name=self.status().intfId,
            _ingressCounters=self.ingressCountersSupported(),
            _egressCounters=self.egressCountersSupported(),
            _umCounters=( self.inMcastPktCountersSupported() or
                          self.outMcastPktCountersSupported() ),
            _bumCounters=False,
            _l3Counters=True )

      mountCounters()
      def stat( attr, supported=True ):
         if not supported:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )

         return currentValue - checkpointValue

      if not zeroOut and self.counter():
         intfCountersModel.inOctets = stat( 'inOctets',
               self.inOctetCountersSupported() )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts',
               self.inUcastPktCountersSupported() )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts',
               self.inMcastPktCountersSupported() )
         intfCountersModel.inBroadcastPkts = 0

         intfCountersModel.outOctets = stat( 'outOctets',
               self.outOctetCountersSupported() )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts',
               self.outUcastPktCountersSupported() )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts',
               self.outMcastPktCountersSupported() )
         intfCountersModel.outBroadcastPkts = 0
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.inBroadcastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0

      return intfCountersModel

   def updateVlanInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                         zeroOut=False ):
      self.intfCountersModel = VlanInterfaceCounters( _name=self.status().intfId )
      current = self.getInterfaceCounters( notFoundString=notFoundString,
                                           checkpoint=checkpoint, zeroOut=zeroOut )

      if not zeroOut and current:
         if self.inUcastPktCountersSupported():
            self.intfCountersModel.inUcastPkts = current.inUcastPkts

         if self.inMcastPktCountersSupported():
            self.intfCountersModel.inMulticastPkts = current.inMulticastPkts

         if self.outUcastPktCountersSupported():
            self.intfCountersModel.outUcastPkts = current.outUcastPkts

         if self.outMcastPktCountersSupported():
            self.intfCountersModel.outMulticastPkts = current.outMulticastPkts

         if self.inOctetCountersSupported():
            self.intfCountersModel.inOctets = current.inOctets

         if self.outOctetCountersSupported():
            self.intfCountersModel.outOctets = current.outOctets

         if self.inUcastOctetCountersSupported():
            self.intfCountersModel.inUcastOctets = current.inUcastOctets

         if self.outUcastOctetCountersSupported():
            self.intfCountersModel.outUcastOctets = current.outUcastOctets

         if self.inMcastOctetCountersSupported():
            self.intfCountersModel.inMulticastOctets = current.inMulticastOctets

         if self.outMcastOctetCountersSupported():
            self.intfCountersModel.outMulticastOctets = current.outMulticastOctets
      else:
         self.intfCountersModel.inUcastPkts = 0
         self.intfCountersModel.inOctets = 0
         self.intfCountersModel.inUcastOctets = 0
         self.intfCountersModel.inMulticastOctets = 0
         self.intfCountersModel.outUcastPkts = 0
         self.intfCountersModel.outOctets = 0
         self.intfCountersModel.outUcastOctets = 0
         self.intfCountersModel.outMulticastOctets = 0

      return self.intfCountersModel

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      if VlanIntf.l3InterfaceCounters():
         return self.updateL3InterfaceCountersModel( checkpoint=checkpoint,
               notFoundString=notFoundString, zeroOut=zeroOut )
      else:
         return self.updateVlanInterfaceCountersModel( checkpoint=checkpoint,
               notFoundString=notFoundString, zeroOut=zeroOut )

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None:
         # If no counters exist, do nothing.
         return None

      self.intfCountersRateModel = VlanInterfaceCountersRate(
                                                        _name=self.status().intfId )
      self.intfCountersRateModel.description = self.description()
      config = self.config() or self.dynConfig()
      self.intfCountersRateModel.interval = int( IntfCli.getActualLoadIntervalValue(
         config.loadInterval ) + 0.5 )

      self.intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      self.intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )

      return self.intfCountersRateModel

   def rateValue( self, attr ):
      # Calculate the rate counter value by exponentially decay the
      # previously saved value and subtract it from the current rate
      # value.
      counter = self.counter()
      if not counter:
         # If no counters exist, just return 0 for a rate.
         return 0.0

      ckpt = self.getLatestCounterCheckpoint()
      config = self.config() or self.dynConfig()
      loadInterval = IntfCli.getActualLoadIntervalValue(
         config.loadInterval )

      sysdbValue = getattr( counter.rates, attr )
      if ckpt:
         ckptValue = getattr( ckpt.rates, attr, 0 )
         ckptStatsUpdateTime = getattr( ckpt.rates, "statsUpdateTime", 0 )
      else:
         ckptValue = 0
         ckptStatsUpdateTime = 0

      # If an interface is error disabled, its counters would never be updated. For
      # an error disabled interface, If clear command is issued, the last time when
      # counters are updated will be less than the last time counter are cleared.
      # So Addin a guard for this case!
      if counter.rates.statsUpdateTime < ckptStatsUpdateTime:
         return 0.0
      if loadInterval == 0:
         return sysdbValue

      expFactor = - ( ( counter.rates.statsUpdateTime - ckptStatsUpdateTime ) /
                      float( loadInterval ) )
      decayedCkptValue = ckptValue * exp( expFactor )
      return max( 0.0, sysdbValue - decayedCkptValue )

   def getInterfaceStatistics( self ):
      if self.counter() is None or (
         not self.inOctetCountersSupported() and
         not self.outOctetCountersSupported() ):
         return None

      interfaceStatistics = VlanInterfaceStatistics()
      config = self.config() or self.dynConfig()
      interfaceStatistics.updateInterval = \
                     IntfCli.getActualLoadIntervalValue( config.loadInterval )
      interfaceStatistics.inBitsRate = self.rateValue( "inBitsRate" )
      interfaceStatistics.outBitsRate = self.rateValue( "outBitsRate" )
      return interfaceStatistics

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      # pylint: disable-next=consider-using-f-string
      return "address is %s (bia %s)" % ( self.addr(), self.burnedInAddr() )

   def addr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def burnedInAddr( self ):
      # VlanIntfStatus'es do not have a burnedInAddr attribute so fake it
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def hardware( self ):
      return "vlan"

   def bandwidth( self ):
      return 0


   #----------------------------------------------------------------------------
   # Returns an unsorted list of VlanIntf objects representing all the existing
   # Interface::VlanIntfStatus objects.
   #----------------------------------------------------------------------------

   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in vlanIntfStatusDir.intfStatus:
         intf = VlanIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def updateLastCounters( self, mode ):
      if VlanIntf.l3InterfaceCounters():
         IntfCli.Intf.updateLastCounters( self, mode )
         return
      # following is the special handling for VlanIntfCounter
      lastCounter = self.storeLastCounters( mode,
                                    className='Interface::VlanIntfCounter' )
      lastCounter.statistics = self.counter().statistics
      lastCounter.vlanStatistics = self.counter().vlanStatistics

   def getLatestCounterCheckpoint( self ):
      '''Get the latest counter checkpoint wheter global or for this session only'''

      ccGlobal = self.getCounterCheckpoint( 'Interface::VlanIntfCounter',
                                             sessionOnly=False )
      ccSession = self.getCounterCheckpoint( 'Interface::VlanIntfCounter',
                                             sessionOnly=True )
      if ccGlobal and ccSession:
         # both there,  compare timestamps
         if ccGlobal.rates.statsUpdateTime > ccSession.rates.statsUpdateTime:
            return ccGlobal
         else:
            return ccSession
      else:
         # at least one not there, return one if any that is there
         if ccGlobal:
            return ccGlobal
         else:
            return ccSession

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      if VlanIntf.l3InterfaceCounters():
         if not sessionOnly:
            # Counter agent manages the checkpoint
            mountCounters()
            x = countersCheckpointDir[ systemName ]
            if x is None:
               return None
            y = x.counterSnapshot.get( self.name )
            return y
         else:
            return IntfCli.Intf.getCounterCheckpoint( self,
                  className='Interface::IntfCounter', sessionOnly=sessionOnly )
      else:
         return IntfCli.Intf.getCounterCheckpoint( self, className, sessionOnly )

   #----------------------------------------------------------------------------
   # Hooks/support for "clear counters".
   #----------------------------------------------------------------------------
   def clearCounters( self, sessionOnly=False ):
      if VlanIntf.l3InterfaceCounters():
         if sessionOnly:
            # Use parent clearCounters
            IntfCli.Intf.clearCounters( self, sessionOnly=sessionOnly )
         else:
            # We have a hook to take care of global clear
            pass
         return

      ckpt = self.getCounterCheckpoint( 'Interface::VlanIntfCounter',
                                        sessionOnly=sessionOnly)
      if ckpt is None or self.counter() is None:
         return
      ckpt.statistics = self.counter().statistics
      ckpt.vlanStatistics = self.counter().vlanStatistics
      ckpt.rates = self.counter().rates
      ckpt.linkStatusChanges = self.status().operStatusChange.count

   # Ingress counter supported functions.
   def ingressCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               ( brHwCapabilities.vlanIntfInOctetCountersSupported or
                 brHwCapabilities.vlanIntfInUcastPktCountersSupported or
                 brHwCapabilities.vlanIntfInMcastPktCountersSupported ) )

   def inMcastPktCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               brHwCapabilities.vlanIntfInMcastPktCountersSupported )

   def inMcastOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               brHwCapabilities.vlanIntfInOctetCountersSupported and
               brHwCapabilities.vlanIntfMcastOctetCountersSupported )

   def inOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               brHwCapabilities.vlanIntfInOctetCountersSupported )

   def inUcastOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               brHwCapabilities.vlanIntfInOctetCountersSupported and
               brHwCapabilities.vlanIntfUcastOctetCountersSupported )

   def inUcastPktCountersSupported( self ):
      return ( bridgingHwEnabled.sviIngressCountersEnabled and
               brHwCapabilities.vlanIntfInUcastPktCountersSupported )

   # Egress counter supported functions.
   def egressCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               ( brHwCapabilities.vlanIntfOutOctetCountersSupported or
                 brHwCapabilities.vlanIntfOutUcastPktCountersSupported or
                 brHwCapabilities.vlanIntfOutMcastPktCountersSupported ) )

   def outMcastPktCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               brHwCapabilities.vlanIntfOutMcastPktCountersSupported )

   def outMcastOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               brHwCapabilities.vlanIntfOutOctetCountersSupported and
               brHwCapabilities.vlanIntfMcastOctetCountersSupported )

   def outOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               brHwCapabilities.vlanIntfOutOctetCountersSupported )

   def outUcastOctetCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               brHwCapabilities.vlanIntfOutOctetCountersSupported and
               brHwCapabilities.vlanIntfUcastOctetCountersSupported )

   def outUcastPktCountersSupported( self ):
      return ( bridgingHwEnabled.sviEgressCountersEnabled and
               brHwCapabilities.vlanIntfOutUcastPktCountersSupported )

   def counterCapable( self ):
      return ( brHwCapabilities.vlanIntfInPktCountersSupported or
               brHwCapabilities.vlanIntfOutPktCountersSupported  or
               brHwCapabilities.vlanIntfInUcastPktCountersSupported or
               brHwCapabilities.vlanIntfOutUcastPktCountersSupported or
               brHwCapabilities.vlanIntfInMcastPktCountersSupported or
               brHwCapabilities.vlanIntfOutMcastPktCountersSupported or
               brHwCapabilities.vlanIntfInOctetCountersSupported or
               brHwCapabilities.vlanIntfOutOctetCountersSupported or
               brHwCapabilities.vlanIntfUcastOctetCountersSupported or
               brHwCapabilities.vlanIntfMcastOctetCountersSupported )

   def countersSupported( self ):
      if self.counterCapable() and self.status() and self.counter():
         return True
      return False

   def countersRateSupported( self ):
      if VlanIntf.l3InterfaceCounters():
         return ( brHwCapabilities.vlanIntfCounterRatesSupported and
                  ( self.ingressCountersSupported() or
                    self.egressCountersSupported() ) )
      elif IntfCli.Intf.countersRateSupported( self ) and \
           ( self.inOctetCountersSupported() or self.outOctetCountersSupported() ):
         return True
      return False

   def countersDiscardSupported( self ):
      return False

   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      self.config().autoState = True
      self.destroySviMappingState()
      self.setMacAddr( None )

#--------------------------------------------------------------------------------
# Define VlanIntf Modelet to add the autostate cli command
#--------------------------------------------------------------------------------
class VlanIntfModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, VlanIntf )

IntfCli.IntfConfigMode.addModelet( VlanIntfModelet )
#-------------------------------------------------------------------------------
# The "[no] autostate" command, in "config-vlan-if" mode.
#
# The full syntax of this command is:
#
#   [no|default] autostate
#-------------------------------------------------------------------------------

def setAutoState( mode, args ):
   mode.intf.config().autoState = not CliCommand.isNoCmd( args )

class SetIntfLinkAutoStateCmd( CliCommand.CliCommandClass ):
   syntax = 'autostate'
   noOrDefaultSyntax = syntax
   data = {
         'autostate': 'Automatically manage the interface link state',
   }
   handler = setAutoState
   noOrDefaultHandler = setAutoState

VlanIntfModelet.addCommandClass( SetIntfLinkAutoStateCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#--------------------------------------------------------------------------------
VlanIntfModelet.addCommandClass( IntfCli.MacAddressRouterAddrCmd )

#-------------------------------------------------------------------------------
# Register the VlanIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( VlanIntf, VlanAutoIntfType,
                                  withIpSupport=True )

vlanPathMounted = {}
l3PathMounted = {}
mountCounters = None

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global vlanIntfConfigDir, vlanIntfStatusDir, bridgingConfig, cliConfig
   global dynVlanIntfConfigDir
   global bridgingHwEnabled, brHwCapabilities
   global mountCounters, entityManager

   entityManager = em

   vlanIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/vlan",
                                          "Interface::VlanIntfConfigDir", "w" )
   dynVlanIntfConfigDir = LazyMount.mount( entityManager,
                                           "interface/input/dynvlanintf",
                                           "Tac::Dir", "ri" )
   vlanIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/vlan",
                                        "Interface::VlanIntfStatusDir", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingHwEnabled = LazyMount.mount( entityManager, "bridging/hwenabled",
                                        "Bridging::HwEnabled", "r" )
   brHwCapabilities = LazyMount.mount( entityManager, "bridging/hwcapabilities",
                                       "Bridging::HwCapabilities", "r" )
   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli",
                                  "Bridging::Input::CliConfig", "w" )
   global systemName
   systemName = entityManager.sysname()

   def _mountL3Path( ):
      if l3PathMounted.get( systemName ) is not None:
         return
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      smi = Smash.mountInfo( 'reader' )
      smashedEntity = shmemEm.doMount( 'interface/counter/svi/current',
            'Smash::Interface::AllIntfCounterDir', smi )
      countersDir[ systemName ] = smashedEntity
      smashedEntity = shmemEm.doMount( 'interface/counter/svi/snapshot',
                                       'Smash::Interface::AllIntfCounterSnapshotDir',
                                       smi )
      countersCheckpointDir[ systemName ] = smashedEntity
      l3PathMounted[ systemName ] = True

   def _mountVlanPath( ):
      if vlanPathMounted.get( systemName ) is not None:
         return
      assert not Tac.activityManager.inExecTime.isCurrent
      mg = entityManager.mountGroup( threadSafe=True )
      countersCheckpointDir[ systemName ] = mg.mount( 'interface/cli/counters/vlan',
            'Interface::AllVlanIntfCounterDir', 'w' )
      vlanIntfCounterWriterStatus = mg.mount(
            'interface/vlanIntfCounter/writerStatus',
            'Interface::VlanIntfCounterWriterStatus', 'r' )
      mg.close( callback=None, blocking=True, msg="" )

      countersCheckpointSm[ systemName ] = Tac.newInstance(
            'Interface::VlanIntfCounterCheckpointSm',
            vlanIntfCounterWriterStatus,
            countersCheckpointDir[ systemName ] )

      smashEntityManager = SharedMem.entityManager( systemName,
                                                     entityManager.isLocalEm() )
      mountHelper = Tac.newInstance( 'Interface::VlanIntfCounterMountHelper',
                                     smashEntityManager )
      countersDir[ systemName ] = mountHelper.doMount( 'current' )

      vlanPathMounted[ systemName ] = True

   def _mountCounters( ):
      if VlanIntf.l3InterfaceCounters():
         _mountL3Path()
      else:
         _mountVlanPath()

   mountCounters = _mountCounters
