#!/usr/bin/env python3
# Copyright (c) 2009, 2010, 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
from ctypes import cdll
import re
import Arnet, LazyMount, Tracing
import ConfigMount, SmashLazyMount
import CliParser
import BasicCli
import AgentCommandRequest
import MultiRangeRule
import ReversibleSecretCli

from CliMode.VmTracer import VCenterMode
from CliMode.VmTracer import VShieldMode
import CliModel
import Tac

from CliPlugin import VlanCli
from CliPlugin.VlanModel import VlanIds
from CliPlugin.VmTracerModels import EsxHost
from CliPlugin.VmTracerModels import LogicalSwitch
from CliPlugin.VmTracerModels import McastRange
from CliPlugin.VmTracerModels import NetworkScope
from CliPlugin.VmTracerModels import NetworkScopes
from CliPlugin.VmTracerModels import StatusDetail
from CliPlugin.VmTracerModels import VCenterServiceStatus
from CliPlugin.VmTracerModels import VShieldServiceStatus
from CliPlugin.VmTracerModels import Vm
from CliPlugin.VmTracerModels import VmTracerAll
from CliPlugin.VmTracerModels import VmTracerDebugAll
from CliPlugin.VmTracerModels import VmTracerEsxDebug
from CliPlugin.VmTracerModels import VmTracerInterface
from CliPlugin.VmTracerModels import VmTracerInterfaces
from CliPlugin.VmTracerModels import VmTracerLagInterface
from CliPlugin.VmTracerModels import VmTracerSessionStatus
from CliPlugin.VmTracerModels import VmTracerSessionStatuses
from CliPlugin.VmTracerModels import VmTracerSwitches
from CliPlugin.VmTracerModels import VmTracerVm
from CliPlugin.VmTracerModels import VmTracerVmDebug
from CliPlugin.VmTracerModels import VmTracerVms
from CliPlugin.VmTracerModels import VmTracerVxlanSegmentRange
from CliPlugin.VmTracerModels import VniRange
from CliPlugin.VmTracerModels import WebServiceStatus
from IpLibConsts import DEFAULT_VRF
from io import StringIO

traceHandle = Tracing.Handle( 'VmTracerCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
mlib = cdll.LoadLibrary( "libCliPrint.so" )
configSessionDir = None
statusSessionDir = None
enableDir = None
allIntfStatusDir = None
bridgingStatus = None
cliStatusDir = {}
bridgingHwCapabilities = None

InvalidVni = Tac.Value( "Vxlan::VniExtOrNone" ).invalidVni
VxlanRole = Tac.Type( "VmTracer::PortGroupRoleType" ).vxlan
DefaultSourceIntf = Tac.Value( "Arnet::IntfId" )
DefaultPassword = ReversibleSecretCli.getDefaultSecret()

def vmTracerSupportedGuard( mode, token ):
   if bridgingHwCapabilities.vmTracerSupported:
      return None
   return CliParser.guardNotThisPlatform

def formatStr( output ):
   if output is None or output == '':
      return '--'
   return output

#-------------------------------------------------------------------------------
# The "vmtracer session <session-name>" command
#-------------------------------------------------------------------------------
class VCenterConfigMode( VCenterMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'VCenter configuration'

   def __init__( self, parent, session, sessionName ):
      VCenterMode.__init__( self, sessionName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.sessionName = sessionName
      self.vrf = DEFAULT_VRF 
      self.url = ''
      self.username = ''
      self.password = DefaultPassword
      self.autoVlan = True
      self.allowedVlans = '1-4094'
      self.sourceIntf = DefaultSourceIntf
      self.commit = True

      if len( configSessionDir.keys() ) > 10:
         self.addError( "Max. number of VmTracer sessions supported is 10." )
         self.commit = False
         return

      configDir = configSessionDir.get( self.sessionName )
      if configDir:
         viConfig = configDir.get( 'viConfig' )
         if viConfig:
            self.allowedVlans = viConfig.allowedVlans
            self.username = viConfig.username
            self.password = viConfig.password
            self.url = viConfig.url
            self.autoVlan = viConfig.autoVlan
            self.vrf = viConfig.vrf
            self.sourceIntf = viConfig.sourceIntf

   def setVrf( self, args ):
      self.vrf = args[ 'VRFNAME' ]

   def noVrf( self, args ):
      self.vrf = DEFAULT_VRF
   
   def setUrl( self, args ):
      self.url = args[ 'URL' ]

   def noUrl( self, args ):
      self.url = ''

   def setSourceIntf( self, args ):
      intf = args.get( 'INTF' )
      if intf:
         self.sourceIntf = intf.name
      else:
         self.sourceIntf = DefaultSourceIntf

   def noSourceIntf( self, args ):
      self.sourceIntf = DefaultSourceIntf

   def setPassword( self, args):
      self.password = args[ 'PASSWORD' ]

   def noPassword( self, args ):
      self.password = DefaultPassword 

   def setUsername( self, args ):
      self.username = args[ 'USERNAME' ]

   def noUsername( self, args ):
      self.username = ''

   def onExit( self ):
      self._commitSession( )

   def setTrunkAllowedVlan( self, args ):
      def allowedVlanStringHelper( oldString, arg2, arg3 ):
         return oldString

      s = VlanCli.vlanRangeOpResultString( args, 'VLANSET', 'VLANSETOP', False,
                                           allowedVlanStringHelper,
                                           self.allowedVlans, None, None )
      print( s )
      self.allowedVlans = s

   def noTrunkAllowedVlan( self, args ):
      self.allowedVlans = ""

   def defaultTrunkAllowedVlan( self, args ):
      self.setTrunkAllowedVlan( { 'all': None } )

   def disableAutoVlan( self, args ):
      self.autoVlan = False

   def enableAutoVlan( self, args ):
      self.autoVlan = True

   def _commitSession( self ):
      t1( '_commitSession', self.commit )
      if not self.commit:
         return

      if self.sessionName not in configSessionDir:
         t1( 'new config dir', self.sessionName )
         configSessionDir.newEntity( 'Tac::Dir', self.sessionName )
      configDir = configSessionDir[ self.sessionName ]

      if 'viConfig' not in configDir:
         t1( 'new viConfig' )
         configDir.newEntity( 'VmwareVI::ViConfig', 'viConfig' )
      viConfig = configDir[ 'viConfig' ]
      
      viConfig.url = self.url
      viConfig.password = self.password
      viConfig.username = self.username
      viConfig.vrf = self.vrf
      viConfig.autoVlan = self.autoVlan
      viConfig.allowedVlans = self.allowedVlans
      viConfig.sourceIntf = self.sourceIntf

      if self.sessionName not in enableDir:
         # Create a qualPath for the session agent
         enableDir.newEntity( 'Tac::Dir', self.sessionName )

def gotoVCenterMode( mode, args):
   sessionName = args[ 'SESSION' ]
   childMode = mode.childMode( VCenterConfigMode, sessionName=sessionName )
   mode.session_.gotoChildMode( childMode )

class VShieldConfigMode( VShieldMode, BasicCli.ConfigModeBase ):
   name = 'VShield configuration'

   def __init__( self, parent, session ):
      VShieldMode.__init__( self, parent.sessionName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.url = None
      self.username = None
      self.password = None
      self.sessionName = parent.sessionName
   
   def setUrl( self, args ):
      self.url = args[ 'URL' ]

   def noUrl( self, args ):
      self.url = ''

   def setUsername( self, args ):
      self.username = args[ 'USERNAME' ]

   def noUsername( self, args ):
      self.username = ''

   def setPassword( self, args):
      self.password = args[ 'PASSWORD' ]

   def noPassword( self, args ):
      self.password = DefaultPassword 

   def onExit( self ):
      t1( 'onExit', self.url, self.username )
      if not self.url and not self.username and not self.password:
         return

      if self.sessionName not in configSessionDir:
         t1( 'new config dir', self.sessionName )
         configSessionDir.newEntity( 'Tac::Dir', self.sessionName )
      configDir = configSessionDir[ self.sessionName ]

      if 'vsConfig' not in configDir:
         t1( 'new vsConfig' )
         configDir.newEntity( 'VmwareVI::VsConfig', 'vsConfig' )
      vsConfig = configDir[ 'vsConfig' ]

      if self.url is not None:
         vsConfig.url = self.url
      if self.username is not None:
         vsConfig.username = self.username
      if self.password is not None:
         vsConfig.password = self.password

def enterVShieldMode( mode, args ):
   childMode = mode.childMode( VShieldConfigMode )
   mode.session_.gotoChildMode( childMode )

#-------------------------------------------------------------------------------
# The "no vmtracer session <session-name> vxlan" command
#-------------------------------------------------------------------------------

def delVmTracerSessionVxlan( mode, args ):
   sessionName = args[ 'SESSION' ]
   if sessionName in configSessionDir:
      config = configSessionDir[ sessionName ]
      if 'vsConfig' in config:
         config.deleteEntity( 'vsConfig' )

#-------------------------------------------------------------------------------
# The "no vxlan" command
#-------------------------------------------------------------------------------

def delVxlan( mode, args ):
   args[ 'SESSION' ] = mode.sessionName
   delVmTracerSessionVxlan( mode, args )

#-------------------------------------------------------------------------------
# The "no vmtracer session <session-name>" command
#-------------------------------------------------------------------------------
def delVmTracerSession( mode, args ):
   sessionName = args[ 'SESSION' ]
   delVmTracerSessionVxlan( mode, args )
   if 'vxlan' in args:
      return

   if sessionName in configSessionDir:
      config = configSessionDir[ sessionName ]
      if 'viConfig' in config:
         config.deleteEntity( 'viConfig' )

   configSessionDir.deleteEntity( sessionName )
   enableDir.deleteEntity( sessionName )

#-------------------------------------------------------------------------------
# "show vmtracer session [ <WORD> ]"
# "show vmtracer session <WORD> ( vsm | vcenter ) [ detail ]"
#-------------------------------------------------------------------------------
def populateVCenterStatus( viConfig, status, sessionStat ):
   if viConfig is None or status is None:
      return
   if status.vCenterSessionStatus is None:
      return

   sessionState = status.vCenterSessionStatus.sessionState
   vrfStatus = status.vrfSessionStatus
   vCenterStat = WebServiceStatus( details=None,
                                   url=formatStr( viConfig.url ),
                                   username=formatStr( viConfig.username ),
                                   vrf=formatStr( viConfig.vrf ),
                                   sessionState=sessionState,
                                   vrfStatus=vrfStatus,
                                   sourceIntf=viConfig.sourceIntf )

   vlans = MultiRangeRule.listFromCanonicalString( viConfig.allowedVlans )
   allowedVlans = VlanIds( vlanIds=vlans )

   sessionStat.vCenterStatus = VCenterServiceStatus( serviceStatus=vCenterStat,
                                                     autoVlan=viConfig.autoVlan,
                                                     allowedVlans=allowedVlans )

def populateVShieldStatus( vsConfig, status, sessionStat, viConfig ):
   if vsConfig is None or status is None:
      return

   if status.vShieldSessionStatus is None:
      return

   sessionState = status.vShieldSessionStatus.sessionState
   vrfStatus = status.vrfSessionStatus
   vShieldStat = WebServiceStatus( details=None,
                                   url=formatStr( vsConfig.url ),
                                   username=formatStr( vsConfig.username ),
                                   vrf = formatStr( viConfig.vrf ),
                                   sessionState=sessionState,
                                   vrfStatus=vrfStatus,
                                   sourceIntf=viConfig.sourceIntf )

   sessionStat.vShieldStatus = VShieldServiceStatus( serviceStatus=vShieldStat )

def populateStatusDetail( sessionStatus, rpcStatus, statusDet ):
   statusDet.lastStateChangeTime = sessionStatus.lastStateChangeTime
   statusDet.lastMsgSent = rpcStatus.lastMsgSent
   statusDet.timeOfLastMsg = rpcStatus.timeOfLastMsg
   statusDet.responseTimeForLastMsg = rpcStatus.responseTimeForLastMsg
   statusDet.numSuccessfulMsg = rpcStatus.numSuccessfulMsg
   statusDet.lastSuccessfulMsg = formatStr( rpcStatus.lastSuccessfulMsg )
   statusDet.lastSuccessfulMsgTime = rpcStatus.lastSuccessfulMsgTime
   statusDet.numFailedMsg = rpcStatus.numFailedMsg
   statusDet.lastFailedMsg = formatStr( rpcStatus.lastFailedMsg )
   statusDet.lastFailedMsgTime = rpcStatus.lastFailedMsgTime
   statusDet.lastErrorCode = formatStr( rpcStatus.lastErrorCode )

def populateSession( sessionName, vCenterOrVsm, detail, model ):
   configDir = configSessionDir.get( sessionName )
   if configDir is None:
      return

   status = statusSessionDir.get( sessionName )
   viConfig = configDir.get( 'viConfig' )
   vsConfig = configDir.get( 'vsConfig' )
 
   # sessionStatus will be common to all models
   sessionStat = VmTracerSessionStatus()
   sessionStat.vShieldStatus = None
   sessionStat.vCenterStatus = None

   if vCenterOrVsm is None:
      populateVCenterStatus( viConfig, status, sessionStat )
      populateVShieldStatus( vsConfig, status, sessionStat, viConfig )
      model.sessionStatuses[ sessionName ] = sessionStat
      return
      
   if vCenterOrVsm == 'vcenter':
      populateVCenterStatus( viConfig, status, sessionStat )
      if detail is not None and status is not None and \
            status.vCenterSessionStatus is not None \
            and status.soapStatus is not None:
         statusDet = StatusDetail()
         populateStatusDetail( status.vCenterSessionStatus, \
               status.soapStatus, statusDet )
         sessionStat.vCenterStatus.serviceStatus.details = statusDet
      model.sessionStatuses[ sessionName ] = sessionStat

   if vCenterOrVsm == 'vsm' and vsConfig is not None:
      populateVShieldStatus( vsConfig, status, sessionStat, viConfig )
      if detail is not None and status is not None and \
            status.vShieldSessionStatus is not None and \
            status.restStatus is not None:
         statusDet = StatusDetail()
         populateStatusDetail( status.vShieldSessionStatus, \
               status.restStatus, statusDet )
         sessionStat.vShieldStatus.serviceStatus.details = statusDet
      model.sessionStatuses[ sessionName ] = sessionStat

# "show vmtracer session [ <WORD> ]"
def doShowVmTracerSessionWord( mode, args ):
   sessionName = args.get( 'SESSION' )
   model = VmTracerSessionStatuses()
   if sessionName is None:
      for sessionName in configSessionDir:
         populateSession( sessionName, None, None, model )
   else:
      populateSession( sessionName, None, None, model )
   return model

# "show vmtracer session SESSION ( vcenter | vsm ) [ detail ]"
def doShowVmTracerSessionWordVCenterVsmDetail( mode, args ):
   sessionName = args[ 'SESSION' ]
   detail = 'detail' in args
   model = VmTracerSessionStatuses()
   vsmOrVCenter = 'vcenter' if 'vcenter' in args else 'vsm'
   populateSession( sessionName, vsmOrVCenter, detail, model )
   return model

def checkfdbStatus( macAddress ):
   fdb = bridgingStatus.smashFdbStatus
   return any( host.address == macAddress for host in fdb.values() )

def getVlan( portGroup ):
   vlan = ''
   if portGroup:
      if portGroup.trunk:
         vlan = 'trunk'
      else:
         if portGroup.accessVlanId == '':
            vlan = 'native'
         else:
            vlan = str( portGroup.accessVlanId )
   return vlan

def isPortChannel( i ):
   return i.name.startswith( 'Po' )

def isPeer( i ):
   return i.name.startswith( 'Peer' )

def isLagMember( intf ):
   return bool( intf.lagName )

def lagMembers( lagIntf, intfs ):
   return toValues( intfs, lambda i: i.lagName == lagIntf.name )

def isConnected( sessionStatus ):
   return bool( sessionStatus and 'stateConnected' == sessionStatus.sessionState )

def isNotConnected( sessionStatus ):
   return bool( sessionStatus and 'stateConnected' != sessionStatus.sessionState )

# Factory for CAPI models used by VmTracer CLI to enable separation of model
# construction and transformation from Sysdb -> cli outputs
class CapiModelFactory:
   def makeVmTracerVms( self ):
      return VmTracerVms()

   def makeVmTracerVm( self ):
      return VmTracerVm()

   def makeVmTracerInterfaces( self, _renderMode ):
      return VmTracerInterfaces( _renderMode=_renderMode )

   def makeVmTracerLagInterface( self, members ):
      return VmTracerLagInterface( members=members )

   def makeVmTracerInterface( self, intf ):
      return VmTracerInterface( switchName=intf.switchName,
                                uplinkName=intf.uplinkName,
                                hostName=intf.hostName )

class VmTracerVmModelBuilder:
   def __init__( self, factory, sessionConfig, status ):
      self.factory = factory
      self.sessionConfig = sessionConfig
      self.status = status

      self.portGroup = None
      self.interface = None

   def portGroupIs( self, portGroup ):
      self.portGroup = portGroup

   def interfaceIs( self, interface ):
      self.interface = interface

   def build( self, vm ):
      def getDisplayIntf( intf, allIntfs ):
         if isPortChannel( intf ):
            members = lagMembers( intf, allIntfs )
            # get first non-peer member or return intf, if none found
            return next( ( i for i in members if not isPeer( i ) ), intf )

         return intf

      def checkCfg( key ):
         return self.sessionConfig and key in self.sessionConfig

      def vxlanFields( model ):
         model.logicalSwitch = None
         model.vni = None
         model.transportZone = None
         model.vtepIp = None
         model.transportVlan = None

         if pg and VxlanRole == pg.role.type:
            swi = pg.role.parent
            model.logicalSwitch = swi.switchName
            model.transportZone = swi.transportZone

            if swi.vni and swi.vni != InvalidVni:
               model.vni = swi.vni

            vtep = pg.role.vtep
            if vtep:
               vtepNic = vtep.vtepHostNic.get( self.interface.hostRef )
               if vtepNic:
                  model.vtepIp = vtepNic.ipAddr.address

               if vtep.portGroup.accessVlanId:
                  model.transportVlan = int( vtep.portGroup.accessVlanId )

      pg = self.portGroup

      model = self.factory.makeVmTracerVm()
      model.vmName = vm.vmName
      model.vNic = vm.vnicName
      model.macAddress = vm.getRawAttribute( "address" )
      model.portGroup = pg.pgName if pg else ''
      model.vlan = getVlan( pg )
      model.powerStatus = 'up' if vm.poweredOn else 'down'
      model.macStatus = 'learnt' if checkfdbStatus( vm.address ) else 'unlearnt'

      intf = getDisplayIntf( self.interface, self.status.intfStatus )
      model.host = intf.hostName
      model.switch = intf.switchName
      model.switchMtu = intf.switchMtu
      model.datacenter = intf.datacenter
      model.interface = self.interface.name
      model.physicalNic = self.interface.pnicKey
      model.inVmotion = vm.inVmotion
      model.failoverToleranceState = vm.ftStatus
      model.vmType = vm.vmType

      cfg = self.sessionConfig
      model.vCenterUrl = cfg[ 'viConfig' ].url if checkCfg( 'viConfig' ) else None
      model.nsxConnected = isConnected( self.status.vShieldSessionStatus )

      vxlanFields( model )

      return model

# converts a tacc collection into a generator expression and filter with a predicate
# also guards against Sysdb mount shifting under your feet via None checks
def toValues( tacDict, filterFn=lambda i: True ):
   arr = ( tacDict.get( k, None ) for k in tacDict )
   return ( i for i in arr if i is not None and filterFn( i ) )

def vmNameFilter( vmName ):
   if vmName:
      return lambda i: re.search( vmName, i.vmName )
   else:
      return lambda i: True

def macAddressFilter( macAddress ):
   if macAddress:
      return lambda i: Arnet.EthAddr( macAddress ).stringValue == i.address
   else:
      return lambda i: True

def makeVmTracerVms( builder, intf, session, filterFn ):
   vms = []

   for i in sorted( toValues( intf.portGroupSegment ), key=lambda i: i.name ):
      builder.portGroupIs( i.portGroup )

      for j in toValues( i.vmEntry, filterFn ):
         vms.append( builder.build( j ) )

   return vms

#-------------------------------------------------------------------------------
# The "show vmtracer vm detail"
# The "show vmtracer vm mac <mac>"
# The "show vmtracer vm name <vm-name>"
#-------------------------------------------------------------------------------
def doShowVm( mode, args, models=CapiModelFactory() ):
   vmName = args.get( 'VM_NAME' )
   macAddress = args.get( 'MAC' )
   detail = 'detail' in args
   nameFilter = vmNameFilter( vmName )
   macFilter = macAddressFilter( macAddress )

   filterFn = lambda i: nameFilter( i ) and macFilter( i )

   vms = models.makeVmTracerVms()
   # pylint: disable-next=protected-access
   vms._printDetail = bool( detail or vmName or macAddress )

   for status in toValues( statusSessionDir ):
      sessionConfig = configSessionDir.get( status.name or None )
      builder = VmTracerVmModelBuilder( models, sessionConfig, status )
      before = len( vms.vms )

      for intf in toValues( status.intfStatus, lambda i: not i.lagName ):
         builder.interfaceIs( intf )
         vms.vms.extend( makeVmTracerVms( builder, intf, status, filterFn ) )

      if isNotConnected( status.vCenterSessionStatus ) and before < len( vms.vms ):
         name = status.name
         syncTime = status.soapStatus.lastSuccessfulMsgTime
         vms.staleSessions[ name ] = syncTime

   return vms

#-------------------------------------------------------------------------------
# The "show vmtracer vm mac <mac> vnic counters" command
# The "show vmtracer vm name <name> vnic counters" command
#-------------------------------------------------------------------------------
def doShowVmVnicCounters( mode, args ):
   vmName = args.get( 'VM_NAME' )
   macAddress = args.get( 'MAC' )
   assert ( vmName is None ) ^ ( macAddress is None )

   # Mac and vmName filter might match more than one VM. VM name can easily collide 
   # and Mac can collide across session or even across different logical switches
   vms = {}
   for session in statusSessionDir: # pylint: disable=too-many-nested-blocks
      status = statusSessionDir[ session ]
      if status.vCenterSessionStatus.sessionState == 'stateConnected':
         for intf, intfStatus  in status.intfStatus.items():
            for segment in intfStatus.portGroupSegment.values():
               for vmEntry in segment.vmEntry.values():
                  # Match vmName or macAddress filter
                  if vmName and vmEntry.vmName == vmName or macAddress and \
                        vmEntry.address == Arnet.EthAddr( macAddress ).stringValue:
                     vmInfo = vms.setdefault( vmEntry.vmRef, {} )
                     vmInfo[ 'sessionName' ] = session
                     vmInfo.setdefault( 'intfNames', set() ).add( intf )
                     vmInfo.setdefault( 'vnicNames', [] ).append( vmEntry.vnicName )

   displayHeader = "True"
   for vmRef, vmInfo in vms.items():
      for intf in vmInfo[ 'intfNames' ]:
         for vnicName in vmInfo[ 'vnicNames' ]:
            sessionName = vmInfo[ 'sessionName' ]
            # Ensure the vmRef is in quotes so that whitespace can be handled
            # properly on the other side.
            cli = 'show vmtracer vm "' + vmRef + '" ' + intf + ' ' + displayHeader \
               + ' ' + vnicName + ' vnic counters'

            if displayHeader:
               displayHeader = "False"

            AgentCommandRequest.runSocketCommand( mode.entityManager,
                                                  "VmTracer-" + sessionName,
                                                  "ShowVmTracerVmCallback",
                                                  command=cli, keepalive=True,
                                                  timeout=90 )

#-------------------------------------------------------------------------------
# The "show vmtracer interface [ <Et1,Et2.. > ]" command
#-------------------------------------------------------------------------------
def doShowInterface( mode, args ):
   vmName = args.get( 'VMNAME' )
   disp = args.get( 'detail' )

   if ( intfs := args.get( 'INTFS' ) ) is None:
      intfs = getattr( allIntfStatusDir, 'intfStatus', allIntfStatusDir )

   factory = CapiModelFactory()
   vmFilterFn = vmNameFilter( vmName )
   model = factory.makeVmTracerInterfaces( disp )

   for name in intfs:
      sessions = ( i for i in toValues( statusSessionDir ) if name in i.intfStatus )
      potentialIntfs = ( i.intfStatus[ name ] for i in sessions )
      nonLagIntfs = [ i for i in potentialIntfs if not isLagMember( i ) ]

      if len( nonLagIntfs ):
         preferenceFilterFn = lambda i: ( i.hostName != "" or
                                          ( isPortChannel( i ) and
                                            i.portGroupSegment ) )
         preferred = list( filter( preferenceFilterFn, nonLagIntfs ) )
         intf = preferred.pop() if preferred else nonLagIntfs[ 0 ]
         session = intf.parent
         sessionConfig = configSessionDir.get( session.name or None )
         builder = VmTracerVmModelBuilder( factory, sessionConfig, session )
         builder.interfaceIs( intf )

         if not isPortChannel( intf ):
            capiIntf = factory.makeVmTracerInterface( intf )
         else:
            capiMembers = {}
            for i in lagMembers( intf, session.intfStatus ):
               capiMembers[ i.name ] = factory.makeVmTracerInterface( i )

            capiIntf = factory.makeVmTracerLagInterface( capiMembers )

         vmList = makeVmTracerVms( builder, intf, session, vmFilterFn )
         capiIntf.vms = { i.macAddress: i for i in vmList }

         model.intfs[ intf.name ] = capiIntf

   return model

#-------------------------------------------------------------------------------
# The "show vmtracer interface [ <Et1,Et2.. > ] vnic counters" command
#-------------------------------------------------------------------------------
def doShowInterfaceVnic( mode, args ):
   if ( intfs := args.get( 'INTFS' ) ) is None:
      intfs = getattr( allIntfStatusDir, 'intfStatus', allIntfStatusDir )

   sessionIntfs = {}
   for intfKey in Arnet.sortIntf( intfs ):
      for sessionName in statusSessionDir:
         status = statusSessionDir[ sessionName ]
         if status.vCenterSessionStatus.sessionState == 'stateConnected':
            if intfKey in status.intfStatus:
               if sessionName not in sessionIntfs:
                  sessionIntfs[ sessionName ] = []

               sessionIntfs[ sessionName ].append( intfKey )

   # pylint: disable-next=consider-using-dict-items,too-many-nested-blocks
   for sessionName in sessionIntfs:
      poOrIntfNames = sessionIntfs[ sessionName ]
      status = statusSessionDir[ sessionName ]
      cmds = []
      portChannelsHeading = {}
      for poOrIntf in poOrIntfNames:
         if poOrIntf.startswith( 'Port-Channel' ):
            sortedNames = sorted( status.intfStatus )
            for name in sortedNames:
               intfStatus = status.intfStatus[ name ]
               if intfStatus.lagName == poOrIntf:
                  if poOrIntf not in portChannelsHeading:
                     cmds.append( 'show vmtracer interface ' + name + ' ' + \
                        poOrIntf + ' vnic counters' )
                     portChannelsHeading[ poOrIntf ] = True
                  else:
                     cmds.append( 'show vmtracer interface ' + name + ' ' + \
                        'None' + ' vnic counters' )
         else:
            cmds.append( 'show vmtracer interface ' + poOrIntf + ' ' + 'None' + \
               ' vnic counters' )

      for cmd in cmds:
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               'VmTracer-' + sessionName ,
                                               'ShowVmTracerInterfaceCallback',
                                               command=cmd, keepalive=True,
                                               timeout=90 )

#-------------------------------------------------------------------------------
# The "show vmtracer interface host
#-------------------------------------------------------------------------------
def doShowHost( mode, args ):
 
   def printHostInfo( intfStatus ):
      if not intfStatus.hostName:
         print( "%s : --" % ( intfKey ) )
      else:
         # Strip excess blank spaces from cpuDescription
         cpuType = ''
         for words in intfStatus.cpuDescription.split():
            cpuType = cpuType + words + ' '

         if intfStatus.lagName:
            print( f" {intfStatus.name} : {intfStatus.hostName}" )
         else:
            print( f"{intfStatus.name} : {intfStatus.hostName}" )
         print( "   Manufacturer:        %s" % ( intfStatus.hostVendor ) )
         print( "   Model:               %s" % ( intfStatus.hostModel ) )
         print( "   CPU type:            %s" % ( cpuType ) )
         print( "   CPUs :               %s" % ( intfStatus.numCpus ) )
         print( "   CPU Cores:           %s" % ( intfStatus.numCores ) )
         print( "   NIC Manufacturer:    %s" % ( intfStatus.pnicVendor ) )
         print( "   NIC Model:           %s" % ( intfStatus.pnicModel ) )
         print( "   Service Tag:         %s" % ( intfStatus.hostServiceTag ) )
      print() 

   if ( intfs := args.get( 'INTFS' ) ) is None:
      intfs = getattr( allIntfStatusDir, 'intfStatus', allIntfStatusDir )

   for intfKey in Arnet.sortIntf( intfs ):
      intfStatus = None
      for sessionName in statusSessionDir:
         status = statusSessionDir[ sessionName ]
         tintfStatus = status.intfStatus.get( intfKey )
         if not tintfStatus:
            continue
         if not intfStatus or ( intfStatus and tintfStatus.hostName != "" ) or \
                ( intfKey.startswith('Po') and tintfStatus.portGroupSegment ):
            intfStatus = tintfStatus

      if intfStatus and not intfStatus.lagName:
         if not intfStatus.name.startswith( 'Po'):
            printHostInfo( intfStatus )
         else:
            print( "\n%-10s :" % ( intfKey ) )
            for status in status.intfStatus.values():
               if status.lagName == intfKey:
                  printHostInfo( status )

#-------------------------------------------------------------------------------
# The "show vmtracer all"'
#-------------------------------------------------------------------------------
# BUG48527: I don't want to make Agent depend on Cli, but I'd love
# have this function there.
def runEapiSocketCommand( modelClass, entityManager, dirName, 
                          commandType, command, timeout=120 ):
   output = StringIO()
   AgentCommandRequest.runSocketCommand( entityManager, dirName,
                                         commandType, command,
                                         timeout=timeout, stringBuff=output )
   outputValue = output.getvalue()
   output.close()

   if '' == outputValue:
      return None

   import yaml # pylint: disable=import-outside-toplevel
   parsedJson = yaml.safe_load( outputValue )
   return CliModel.unmarshalModel( modelClass, parsedJson )
            
def doShowAll( mode, args ):
   macAddress = args.get( 'MAC' )
   model = VmTracerAll()
   cmd = 'show vmtracer all' + ( 'mac %s' % macAddress if macAddress else '' )

   for session in statusSessionDir:
      switches = populateSessionModel( session, statusSessionDir[ session ],
                                       VmTracerSwitches, mode.entityManager,
                                       'ShowVmTracerAllCallback', cmd )
      model.sessions[ session ] = switches
   return model

#-------------------------------------------------------------------------------
# The "show vmtracer vxlan segment" command
#-------------------------------------------------------------------------------

def getControlPlane( segment ):
   controlPlane = LogicalSwitch.ControlPlane()
   controlPlane.mode = segment.controlPlaneMode

   if "unicast" != controlPlane.mode:
      controlPlane.mcastAddr = segment.multicastIpAddr
   else:
      controlPlane.mcastAddr = None

   return controlPlane

def populateVxlanSegment( segment ):
   sw = LogicalSwitch()
   sw.displayName = segment.displayName
   sw.vni = segment.vni
   sw.controlPlane = getControlPlane( segment )
   return sw

def doShowVmTracerVxlanSegmentWrapper( mode, args ):
   return doShowVmTracerVxlanSegment( mode, pool='pool' in args )

def doShowVmTracerVxlanSegment( mode, filterFn=lambda segment: True,
                                renderMode='summary', pool=None ):
   def getScope( vs, networkScopeId, scopes ):
      name = '--'
      if networkScopeId in vs.networkScope:
         name = vs.networkScope[ networkScopeId ].displayName
      if name not in scopes:
         scopes[ name ] = NetworkScope()
      return scopes[ name ]

   scopes = NetworkScopes( _renderMode=renderMode )
   if pool:
      scopes = NetworkScopes( _renderMode='segPool' )

   for sessionName in statusSessionDir:
      vs = statusSessionDir[ sessionName ].vs
      if pool:
         for networkScopeId, networkScope in vs.networkScope.items():
            displayName = networkScope.displayName
            desc = networkScope.desc
            nScope = NetworkScope()

            for segment in vs.vxlanSegment.values():
               if segment.networkScope == networkScopeId:
                  sw = populateVxlanSegment( segment )
                  nScope.switches[ segment.name ] = sw
            nScope.desc = desc
            scopes.scopes[ displayName ] = nScope
      
      else:
         for segment in filter( filterFn, vs.vxlanSegment.values() ):
            sw = populateVxlanSegment( segment )
            scope = getScope( vs, segment.networkScope, scopes.scopes )
            scope.switches[ segment.name ] = sw

   return scopes

def doShowLogicalSwitch( mode, args ):
   switchName = args.get( 'SWITCH_NAME' )
   if switchName is None:
      return doShowVmTracerVxlanSegment( mode )
   else:
      filterFn = lambda s: switchName == s.displayName
      return doShowVmTracerVxlanSegment( mode, filterFn, 'detail' )

#-------------------------------------------------------------------------------
#-------------------------------------------------------------------------------
def doShowVmTracerVxlanSegmentRange( mode, args ):
   model = VmTracerVxlanSegmentRange()
   for sessionName in statusSessionDir:
      status = statusSessionDir[ sessionName ]
      vs = status.vs

      if len( vs.vniRange ) == 0 and len( vs.multicastAddrRange ) == 0:
         continue

      # VNI Range and Mcast Range can be of different sized
      for vniRange in vs.vniRange.values():
         vniRangeModel = VniRange()
         vniRangeModel.begin = vniRange.begin
         vniRangeModel.end = vniRange.end
         model.vniRanges.append( vniRangeModel )
         
      for mcastRange in vs.multicastAddrRange.values():
         mcastRangeModel = McastRange()
         mcastRangeModel.begin = mcastRange.begin
         mcastRangeModel.end = mcastRange.end
         model.mcastRanges.append( mcastRangeModel )
         
   doShowVmTracerVxlanSegment( mode )
   return model

#-------------------------------------------------------------------------------
# The "show vmtracer vxlan segment pool <poolName>" command
#-------------------------------------------------------------------------------
def doShowVmTracerVxlanSegmentPoolWord( mode, args ):
   poolName = args[ 'POOL' ]
   for session in statusSessionDir:
      status = statusSessionDir[ session ]
      if status.vShieldSessionStatus.sessionState == 'stateConnected':
         cmd = 'show vmtracer vxlan segment pool ' +  poolName
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               "VmTracer-" + session,
                                               "VShieldCallback", command=cmd,
                                               timeout=90 )

#-------------------------------------------------------------------------------
# The "show vmtracer vxlan vm" command
#-------------------------------------------------------------------------------
def doShowVmTracerVxlanVm( mode, args ):
   for session in statusSessionDir:
      status = statusSessionDir[ session ]
      if status.vShieldSessionStatus.sessionState == 'stateConnected':
         cmd = 'show vmtracer vxlan vm'
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               "VmTracer-" + session,
            "VShieldCallback", command=cmd, timeout=90 )

#-------------------------------------------------------------------------------
# The "show vmtracer debug vm <vmname> [session <sessionname>]" command
#-------------------------------------------------------------------------------
# pylint: disable-next=inconsistent-return-statements
def populateSessionModel( session, status, cliModel, entMan, cmdType, cmd ):
   if status.sessSmInitialized:
      return runEapiSocketCommand( cliModel, entMan, 'VmTracer-'+session, cmdType,
                                   command=cmd, timeout=90 )

def doShowVmTracerDebug( mode, resultModel, cliModel, sessionName=None, 
                         esxName=None, vmName=None ):
   if esxName:
      cmd = 'show vmtracer debug esx %s' % ( esxName )
   elif vmName:
      cmd = 'show vmtracer debug vm %s' % ( vmName )
   else:
      cmd = 'show vmtracer debug all'
   if sessionName is None:
      for session in configSessionDir:
         if session in statusSessionDir:
            status = statusSessionDir[ session ]
            sessionModel = populateSessionModel( session, status, cliModel,
                                                 mode.entityManager,
                                                 'ShowVmTracerDebugCallback', cmd )
            if sessionModel and sessionModel.name:
               resultModel.sessions[ session ] = sessionModel
   elif ( sessionName in configSessionDir ) and \
         ( sessionName in statusSessionDir ):
      status = statusSessionDir[ sessionName ]
      sessionModel = populateSessionModel( sessionName, status, cliModel,
                                           mode.entityManager,
                                           'ShowVmTracerDebugCallback', cmd )
      if sessionModel and sessionModel.name:
         resultModel.sessions[ sessionName ] = sessionModel
   return resultModel

def doShowVmTracerVmDebug( mode, args ):
   return doShowVmTracerDebug( mode, VmTracerVmDebug(), Vm,
                               args.get( 'SESSION' ), vmName=args.get( 'VM_NAME' ) )

#------------------------------------------------------------------------------
# The "show vmtracer debug esx <esxname> [session <sessionname>]" command
#------------------------------------------------------------------------------
def doShowVmTracerEsxDebug( mode, args ):

   return doShowVmTracerDebug( mode, VmTracerEsxDebug(), EsxHost,
         args.get( 'SESSION' ), esxName=args.get( 'ESX_NAME' ) )

#-------------------------------------------------------------------------------
# The "show vmtracer debug [session <sessionname>]" command
#------------------------------------------------------------------------------
def doShowVmTracerAllDebug( mode, args ):
   return doShowVmTracerDebug( mode, VmTracerDebugAll(), 
         VmTracerDebugAll.VI, args.get( 'SESSION' ) )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global configSessionDir
   global statusSessionDir
   global enableDir
   global allIntfStatusDir
   global bridgingStatus
   global bridgingHwCapabilities

   configSessionDir = ConfigMount.mount( em, 'vmtracer/config/session',
                                         'Tac::Dir', 'wi' )
   statusSessionDir = LazyMount.mount( em, 'vmtracer/status/session',
                                      'Tac::Dir', 'ri' )
   enableDir = ConfigMount.mount( em, 'vmtracer/slice/enable', 'Tac::Dir', 'wi' )
   allIntfStatusDir = LazyMount.mount( em, 'interface/status/all',
                                       'Interface::AllIntfStatusDir', 'r' )
   bridgingStatus = SmashLazyMount.mount( em, 'bridging/status',
                                          'Smash::Bridging::Status',
                                          SmashLazyMount.mountInfo( 'reader' ) )
   bridgingHwCapabilities = LazyMount.mount( em, "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
