#!/usr/bin/env python3
# Copyright (c) 2012, 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=singleton-comparison
# pylint: disable=consider-using-f-string

import Ark
import Arnet
from CliPlugin import IntfCli
from CliPlugin.VlanModel import VlanIds 
import Tac
from ArnetModel import MacAddress  
from ArnetModel import Ip4Address
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import Int
from TableOutput import Format
from TableOutput import Headings
from TableOutput import TableFormatter
from Vlan import vlanSetToCanonicalString
from IntfModels import Interface
import TableOutput

MAX_HOSTNAME_LEN = 25
FORMAT_STR = "%(indent)s%(name)-25s%(value)-20s"
INDENT = '\t' #indent size for vmtracer debug output

VmKernelType = Tac.Type( "VmwareVI::VmType" ).vmKernel

# table constants and methods
normFmt = TableOutput.Format( justify='left' )
normFmt.noPadLeftIs( True )
headFmt = TableOutput.Format( isHeading=True, justify='left' )
headFmt.noPadLeftIs( True )

def rangedWidthFmt( minWidth, maxWidth, justify='left', wrap=True, isHeading=False ):
   fmt = Format( justify=justify, minWidth=minWidth, maxWidth=minWidth, wrap=wrap,
                 isHeading=isHeading )
   fmt.noPadLeftIs( True )
   return fmt

def fixedWidthFmt( width, isHeading=False ):
   return rangedWidthFmt( width, width, isHeading=isHeading )

def renderField( label, value, fill=' ', length=36 ):
   print( label.ljust( length, fill ), value )

def renderTimeStampField( label, utcTime, relative=True, fill=' ', length=36 ):
   timeStr = Ark.timestampToStr( utcTime, relative, Tac.utcNow() )
   print( label.ljust( length, fill ), timeStr )
   
class StatusDetail( Model ):
   lastStateChangeTime = Float( help="The time of last state change" )
   lastMsgSent = Str( help="The last message sent" )
   timeOfLastMsg = Float( help="The time of the last message" )
   responseTimeForLastMsg = Float( help="The time it took, in seconds, to send \
         response for the last message " )
   numSuccessfulMsg = Int( help="The number of successful messages" )
   lastSuccessfulMsg = Str( help="The last successful message" )
   lastSuccessfulMsgTime = Float( help="The time of the last \
         successful message" )
   numFailedMsg = Int( help="The number of failed messages" )
   lastFailedMsg = Str( help="The last failed message" )
   lastFailedMsgTime = Float( help="The time of the last \
         failed message" )
   lastErrorCode = Str( help="The last error code" )

   def printValues( self ):
      renderTimeStampField( 'lastStateChangeTime', self.lastStateChangeTime )
      renderField( 'lastMsgSent', self.lastMsgSent )
      renderTimeStampField( 'timeOfLastMsg', self.timeOfLastMsg )
      renderField( 'responseTimeForLastMsg', self.responseTimeForLastMsg )
      renderField( 'numSuccessfulMsg', self.numSuccessfulMsg )
      renderField( 'lastSuccessfulMsg', self.lastSuccessfulMsg )
      renderTimeStampField( 'lastSuccessfulMsgTime', self.lastSuccessfulMsgTime )
      renderField( 'numFailedMsg', self.numFailedMsg )
      renderField( 'lastFailedMsg', self.lastFailedMsg )
      renderTimeStampField( 'lastFailedMsgTime', self.lastFailedMsgTime )
      renderField( 'lastErrorCode', self.lastErrorCode )

class WebServiceStatus( Model ):
   url = Str( help='vCenter/vShield/NSX-V URL' )
   username = Str( help='vCenter/vShield username' )
   vrf = Str( help='vCenter/vShield VRF' )
   sessionState = Enum( values=Tac.Type( "VmwareVI::SessionState" ).attributes,
         help='Current vCenter/vShield session state' )
   vrfStatus = Enum( values=Tac.Type( "VmTracer::VrfSessionStatus" ).attributes,
         help='VRF state of current session' )
   sourceIntf = Interface( help='Local interface for vCenter/vShield connections' )

   details = Submodel( valueType=StatusDetail, help=
                       'Details of connection status associated with vShield/vCenter'
                       'that is only populated when requested by cli command',
               optional=True )

   def renderState( self ):
      vrfStateMap = { 'vrfStatusUnknown' : 'Unknown',
                      'vrfStatusError' : 'Initializing',
                      'vrfStatusWaiting' : 'Waiting for VRF ready',
                      'vrfStatusReady' : 'Connecting' }

      stateMap = { 'stateUnknown' : vrfStateMap[ self.vrfStatus ],
                   'stateDisconnected' : 'Disconnected',
                   'stateConnecting' : 'Connecting',
                   'stateConnected' : 'Connected' }

      renderField( 'sessionState', stateMap[ self.sessionState ] )

   def renderDetails( self ):
      if self.details != None:
         self.details.printValues()

class VShieldServiceStatus( Model ):
   serviceStatus = Submodel( valueType=WebServiceStatus,
                             help='vShield appliance service status' )

   def render( self ):
      status = self.serviceStatus

      renderField( 'vShield/NSX URL', status.url )
      renderField( 'username', status.username )
      renderField( 'VRF', status.vrf ) 
      renderField( 'Source Interface', status.sourceIntf )

      status.renderState()
      status.renderDetails()

class VCenterServiceStatus( Model ):
   serviceStatus = Submodel( valueType=WebServiceStatus,
                             help='vCenter appliance service status' )
   autoVlan = Bool( help='Whether auto VLAN provision is enabled' )
   allowedVlans = Submodel( valueType=VlanIds,
                            help='List of VLANs allowed in the vCenter session' )

   def render( self ):
      status = self.serviceStatus

      renderField( 'vCenter URL', status.url )
      renderField( 'username', status.username )
      renderField( 'VRF', status.vrf )
      renderField( 'autovlan', 'enabled' if self.autoVlan else 'disabled' )
      renderField( 'allowed-vlans',
                   vlanSetToCanonicalString( self.allowedVlans.vlanIds ) )
      renderField( 'Source Interface', status.sourceIntf )

      status.renderState()
      status.renderDetails()

class VmTracerSessionStatus( Model ):
   # each of these submodels may or may not be populated based on cli command
   vShieldStatus = Submodel( valueType=VShieldServiceStatus,
                             help='vShield manager status', optional=True )
   vCenterStatus = Submodel( valueType=VCenterServiceStatus, help='vCenter status',
                             optional=True )

class VmTracerSessionStatuses( Model ):
   sessionStatuses = Dict( valueType=VmTracerSessionStatus, help=
         'A mapping of session names to status information' )

   def render( self ):
      for sessionName, sessionDetail in self.sessionStatuses.items():
         renderField( 'Session', sessionName )

         if sessionDetail.vCenterStatus:
            sessionDetail.vCenterStatus.render()

         if sessionDetail.vShieldStatus:
            sessionDetail.vShieldStatus.render()

         print( '' )

class VniRange( Model ):
   begin = Int( help="Start of VNI range" )
   end = Int( help="End of VNI range" )

class McastRange( Model ):
   begin = Ip4Address( help="Start of Multicast Range" )
   end = Ip4Address( help="End of Multicast Range" )

class VmTracerVxlanSegmentRange( Model ):
   vniRanges = List( valueType=VniRange, help="Vni range" )
   mcastRanges = List( valueType=McastRange, help="Multicast range" )
   
   def render( self ):
      table = TableOutput.createTable( ( 'VNI Range', 
         'Multicast IP Range' ) )
      table.formatColumns( normFmt, normFmt )
      maxRows = max( len( self.vniRanges ), len( self.mcastRanges ) )
        
      # accounts for vni and mcast ranges of different sizes
      # fills in an empty cell of either with ''
      for row in range( maxRows ):
         vniR = ''
         mcastR = ''
         if row < len( self.vniRanges ) :
            vniR = str( self.vniRanges[ row ].begin ) + \
            ' - ' + str( self.vniRanges[ row ].end )
         if row < len( self.mcastRanges ):
            mcastR = str( self.mcastRanges[ row ].begin ) + \
            ' - ' + str( self.mcastRanges[ row ].end )
         table.newRow( vniR, mcastR )

      print( table.output() )

def renderNetworkScopesTable( self ):
   def controlPlaneAttr( cp ):
      if "unicast" == cp.mode:
         return "Unicast"
      else:
         return f"{cp.mode.capitalize()} ({cp.mcastAddr})"

   def getRowAttrs( sw ):
      return sw.displayName, sw.vni, controlPlaneAttr( sw.controlPlane )

   def getRows( scope ):
      switches = ( scope.switches[ k ] for k in sorted( scope.switches.keys() ) )
      return ( getRowAttrs( s ) for s in switches )

   table = TableFormatter()
   headings = ( 'Name', 'VNI', 'Control Plane', 'Transport Zone / Network Scope' )
   Headings( headings ).doApplyHeaders( table )

   nameFmt = Format( justify="left", minWidth=20, maxWidth=20 )
   vniFmt = Format( justify="left", minWidth=6, maxWidth=8 )
   controlPlaneFmt = Format( justify="left", minWidth=7, maxWidth=27 )
   scopeFmt = Format( justify="left", minWidth=20, maxWidth=20, wrap=True )

   table.formatColumns( nameFmt, vniFmt, controlPlaneFmt, scopeFmt )

   for k in sorted( self.scopes.keys() ):
      for r in getRows( self.scopes[ k ] ):
         table.newRow( *( r + ( k, ) ) )

   print( table.output() )

def renderNetworkScopesDetails( self ):
   def renderSwitch( scopeName, switch ):
      mcastAddr = switch.controlPlane.mcastAddr
      multicastIp = '--' if None == mcastAddr else str( mcastAddr )

      detailLines = [
         'LS Name  %s' % switch.displayName,
         '  Transport Zone   :     %s' % scopeName,
         '  VNI              :     %s' % switch.vni,
         '  Control Plane    :     %s' % switch.controlPlane.mode.capitalize(),
         '  Multicast IP     :     %s' % multicastIp ]

      print( '\n'.join( detailLines ) )

   for scopeName in sorted( self.scopes.keys() ):
      scope = self.scopes[ scopeName ]

      for switchName in sorted( scope.switches.keys() ):
         renderSwitch( scopeName, scope.switches[ switchName ] )

def renderNetworkScopesSegPool( self ):
   def formatStr( output ):
      if output is None or output == '':
         return '--'
      return output
   
   table = TableOutput.createTable( ( 'Name', 'Description', 'Segments' ) )
   table.formatColumns( normFmt, normFmt, normFmt )

   sortedKeys = sorted( self.scopes.keys() )
   for displayName in sortedKeys:
      scope = self.scopes[ displayName ]
      allSegs = ''
      if len( scope.switches ) == 0:
         allSegs = formatStr( allSegs )
      else:
         swList = list( scope.switches.values() )
         displayNameList = []
         for sw in swList:
            displayNameList.append( sw.displayName )
         displayNameList = sorted( displayNameList )
         allSegs = ', '.join( displayNameList )
      table.newRow( displayName, scope.desc, allSegs )

   print( table.output() )

class LogicalSwitch( Model ):
   class ControlPlane( Model ):
      mode = Enum( values=Tac.Type( "VmwareVI::ControlPlaneMode" ).attributes,
                   help="NSX mode" )
      mcastAddr = Ip4Address( help="VXLAN ARP broadcast address", optional=True )

   displayName = Str( help="Logical switch display name" )
   vni = Int( help="VXLAN VNI" )
   controlPlane = Submodel( valueType=ControlPlane,
                            help="Mode of VMWare control plane" )

class NetworkScope( Model ):
   switches = Dict( valueType=LogicalSwitch,
                    help="A mapping of logical switch ids to switches" )
   desc = Str( help="Description of the network scope", optional=True )

class NetworkScopes( Model ):
   _renderMode = Enum( values=[ 'detail', 'summary', 'segPool' ],
                       help="Internal rendering mode control" )

   scopes = Dict( valueType=NetworkScope,
                  help="A mapping of network scope display names to scopes" )

   def render( self ):
      if 'summary' == self._renderMode:
         renderNetworkScopesTable( self )
      elif 'segPool' == self._renderMode:
         renderNetworkScopesSegPool( self ) 
      else:
         renderNetworkScopesDetails( self )

class VmTracerVm( Model ):
   vmName = Str( help="Name of the VM" )
   vNic = Str( help="Name of the Vm Nic" )
   macAddress = MacAddress( help="MAC address of the VM" )
   portGroup = Str( help="Name of the port group to which the VM belongs" )
   vlan = Str( help="Portgroup vlan as 'native', 'trunk' or string representation "
               "of a vlan" )
   powerStatus = Enum( values=( "up", "down" ), help="Power status of the VM" )
   macStatus = Enum( values=( "learnt", "unlearnt" ), optional=True,
                     help="VM Mac address learn status" )
   host = Str ( help="Host name where VM resides" )
   switch = Str( help="Name of the vSwitch or DVS the VM is on" )
   switchMtu = Int( help="MTU of the vSwitch or DVS this VM is on", optional=True )
   datacenter = Str( help="Name of the Data Center where the VM resides" )
   interface = Str( help="Interface on which the VM is connected" )
   inVmotion = Bool( help="Flag indicating if a VM is being Vmotioned",
                     optional=True )
   failoverToleranceState = Enum( values=Tac.Type( "VmwareVI::FTStatus" ).attributes,
                                  optional=True,
                                  help="Failover tolerance state" )
   logicalSwitch = Str( help="Name of Vmware logical switch this vm is connected to",
                        optional=True )
   vni = Int( help="VNI of VXLAN that this VM is attached to", optional=True )
   vtepIp = Ip4Address( help="IP address of the VTEP used by this vm",
                        optional=True )
   transportVlan = Int( help="Transport VLAN for VXLAN VTEP", optional=True )
   vmType = Enum( values=Tac.Type( "VmwareVI::VmType" ).attributes,
                  help="Type of VM entry", optional=True )
   vCenterUrl = Str( help="Url of associated vCenter instance", optional=True )
   nsxConnected = Bool( help="Flag indicating if an NSX connection is active",
                        optional=True )
   physicalNic = Str( help="Name of the physical nic the VM is attached to",
                      optional=True )
   transportZone = Str( help="VXLAN Transport Zone of this VM", optional=True )

   def renderVtep( self, blank='--' ):
      return str( self.vtepIp ) if self.vtepIp else blank

   def renderStatus( self ):
      powerStatus = self.powerStatus.capitalize()

      if self.logicalSwitch or VmKernelType == self.vmType:
         return powerStatus
      else:
         macStatus = 'Up' if 'learnt' == self.macStatus else 'Down'
         return powerStatus + '/' + macStatus

   def renderState( self ):
      FTStatus = { 'ftActive': 'FT-A', 'ftStandby': 'FT-S' }
      ftState = self.failoverToleranceState or 'ftDisabled'

      if self.inVmotion:
         if 'ftDisabled' == ftState:
            return 'VMotion'
         else:
            return 'VMotion/' + FTStatus[ ftState ]
      else:
         if 'ftDisabled' == ftState:
            return '--'
         else:
            return FTStatus[ ftState ]

   def renderDetails( self ):
      # Header label spacing is len(longest field) + 2
      # Indent label spacing is len(longest field)
      # Longest field is 'Physical switchport' -> len() == 19

      headerFields = [
         ( 'VM Name', self.vmName ),
         ( 'Data Center', self.datacenter ),
         ( 'vCenter instance', self.vCenterUrl or '--' ),
         ( 'Host', self.host ),
         ( 'Status', self.renderStatus() ),
         ( 'vNIC', self.vNic )
      ]

      vsmBlank = '--' if self.nsxConnected else '--; no active VSM or NSX connection'

      indentFields = [
         ( 'vNIC MAC addr', self.macAddress ),
         ( 'vNIC Portgroup', self.portGroup ),
         ( 'vNIC VLAN', self.vlan ),
         ( 'vSwitch / vDS', self.switch ),
         ( 'vSwitch MTU', self.switchMtu or '--' ),
         ( 'Logical Switch', self.logicalSwitch or vsmBlank ),
         ( 'VNI', self.vni or '--' ),
         ( 'VTEP IP', self.renderVtep( vsmBlank ) ),
         ( 'Transport VLAN', self.transportVlan or '--' ),
         ( 'Host PNIC', self.physicalNic or '--' ),
         ( 'Physical switchport', IntfCli.Intf.getShortname( self.interface ) ),
         ( 'Transport Zone', self.transportZone or '--' )
         ]

      for i in headerFields:
         print( '%-22s : %s' % ( i[ 0 ], i[ 1 ] ) )

      for i in indentFields:
         print( '  %-20s : %s' % ( i[ 0 ], i[ 1 ] ) )

      print()

class VmTracerGenericInterface( Model ):
   vms = Dict( keyType=MacAddress,
               valueType=VmTracerVm,
               help="A mapping between VM mac and VM information" )

   def renderInterface( self, name ):
      def vmRow( vmRef, vm ):
         return ( vm.vmName, vm.vNic, vm.logicalSwitch or vm.vlan,
                  vm.renderStatus(), vm.renderVtep(), vm.renderState() )

      table = TableFormatter( indent=3 )
      headings = ( 'VM Name', 'VM Adapter', 'Logical Switch / VLAN',
                   'Status', 'VTEP IP', 'State' )

      Headings( headings ).doApplyHeaders( table )
      nameFmt = rangedWidthFmt( 20, 25, isHeading=True )
      adapterFmt = rangedWidthFmt( 20, 25 )
      vlanFmt = rangedWidthFmt( 21, 25 )
      statusFmt = fixedWidthFmt( 9 )
      stateFmt = fixedWidthFmt( 12 )
      vtepFmt = fixedWidthFmt( 15 )
      table.formatColumns( nameFmt, adapterFmt, vlanFmt, statusFmt,
                           vtepFmt, stateFmt )

      sortedVms = sorted( self.vms.items(), key=lambda i: i[ 0 ] )
      standardVms = ( ( k, v ) for ( k, v ) in sortedVms if None == v.logicalSwitch )
      vxlanVms = ( ( k, v ) for ( k, v ) in sortedVms if None != v.logicalSwitch )

      for i in standardVms:
         table.newRow( *vmRow( *i ) )

      for i in vxlanVms:
         table.newRow( *vmRow( *i ) )

      return '\n' + self.renderTitle( name ) + '\n' + table.output()

class VmTracerInterface( VmTracerGenericInterface ):
   hostName = Str( help="Host name where this interface is attached" )
   switchName = Str( help="Vswitch this interface is attached to" )
   uplinkName = Str( help="Uplink port this interface is attached to" )

   def renderTitle( self, name ):
      def field( value ):
         return '--' if value == "" else value

      return "%-10s : %s/%s/%s" % ( name,
                                    field( self.hostName ),
                                    field( self.switchName ),
                                    field( self.uplinkName ) )

class VmTracerLagInterface( VmTracerGenericInterface ):
   members = Dict( keyType=Interface,
                   valueType=VmTracerInterface,
                   help="A mapping between interface names and LAG members" )

   def renderTitle( self, name ):
      rows = [ "%-10s :" % ( name ) ]
      children = self.members.items()
      rows.extend( ' ' + v.renderTitle( k ) for ( k, v ) in children )
      return '\n'.join( rows )

class _Interfaces( Model ):
   def interfaces( self ):
      '''return the list of interfaces for this interface set'''
      raise NotImplementedError()

   def render( self ):
      intfs = self.interfaces()
      items = sorted( intfs, key=lambda item: Arnet.intfNameKey( item[ 0 ] ) )

      for ( k, v ) in items:
         print( v.renderInterface( k ) )

class VmTracerInterfaces( _Interfaces ):
   _renderMode = Enum( values=[ 'detail', 'summary' ],
                       help="Internal rendering mode control" )

   intfs = Dict( keyType=Interface,
                 valueType=VmTracerGenericInterface,
                 help="A mapping of interface names to interfaces" )

   def interfaces( self ):
      if 'detail' == self._renderMode:
         return self.intfs.items()
      else:
         return ( i for i in self.intfs.items() if 0 < len( i[ 1 ].vms ) )

   def renderMode( self ):
      return self._renderMode

class VmTracerSwitches( Model ):
   class SwitchInterfaces( _Interfaces ):
      intfs = Dict( valueType=VmTracerInterface,
                    help="A mapping of interface names to interfaces" )

      def interfaces( self ):
         return self.intfs.items()

   _renderMode = Enum( values=[ 'detail', 'summary' ],
                       help='Internal rendering mode control' )

   switches = Dict( valueType=SwitchInterfaces,
                    help="A mapping of switch names to switch interface sets" )

   def render( self ):
      if 'detail' == self._renderMode:
         for i in self.switches.values():
            for j in i.intfs.values():
               for k in j.vms.values():
                  k.renderDetails()
      else:
         for ( k, v ) in self.switches.items():
            print( 'Switch: %s' % ( k ) )
            v.render()

class VmTracerAll( Model ):
   __revision__ = 2
   sessions = Dict( valueType=VmTracerSwitches,
                    help="A mapping of session names to vCenter switches" )

   def degrade( self, dictRepr, revision ):
      def degradeVm( vm ):
         return {
            'adapter': vm.vNic,
            'vlan': vm.vlan,
            'powerStatus': vm.powerStatus.capitalize(),
            'fdbStatus': vm.macStatus,
            'state': vm.renderState()
         }

      def degradeIntf( intf ):
         return {
            'switchName': intf.switchName,
            'uplinkName': intf.uplinkName,
            'hostName': intf.hostName,
            'vms': { i.vmName: degradeVm( i ) for i
                     in intf.vms.values() }
         }

      def degradeSwitch( sw ):
         return {
            'interfaces': { k: degradeIntf( v ) for ( k, v )
                            in sw.intfs.items() }
         }

      def degradeSession( sess ):
         return {
            'switches': { k: degradeSwitch( v ) for ( k, v )
                          in sess.switches.items() }
         }

      if 1 == revision:
         oldSessions = self.sessions.values()
         dictRepr[ 'sessions' ] = [ degradeSession( i ) for i in oldSessions ]

      return dictRepr

   def render( self ):
      for i in self.sessions.values():
         i.render()

class VmTracerVms( Model ):
   __revision__ = 2
   _printDetail = Bool( help="Internal use only. Tells whether details of VMs "
                        " needs to be printed or not" )
   vms = List( valueType=VmTracerVm, help="List of VMs" )
   staleSessions = Dict( valueType=float,
                         help="A mapping of stale session names to their sync time" )

   def _oldestSession( self, sessions ):
      sortedTimes = sorted( sessions.items(), key=lambda i: i[ 1 ] )
      return [ i for i in sortedTimes if None != i ][ 0 ]

   def degrade( self, dictRepr, revision ):
      if 1 == revision:
         if 0 < len( dictRepr[ 'staleSessions' ] ):
            first = self._oldestSession( dictRepr[ 'staleSessions' ] )
            dictRepr[ 'sessionWithOldData' ] = first[ 0 ]
            dictRepr[ 'lastSyncTime' ] = first[ 1 ]
         else:
            dictRepr[ 'sessionWithOldData' ] = None
            dictRepr[ 'lastSyncTime' ] = None

         del dictRepr[ 'staleSessions' ]

         for i in dictRepr[ 'vms' ]:
            i[ 'macStatus' ] = i.get( 'macStatus' ) or 'unlearnt'

      return dictRepr

   def render( self ):
      if self._printDetail:
         for vm in self.vms:
            vm.renderDetails()
      else:
         table = TableFormatter()
         headings = ( 'VM Name', 'Esx Host', 'Interface',
                      'Logical Switch / VLAN', 'Status', 'VTEP IP' )
         Headings( headings ).doApplyHeaders( table )

         table.formatColumns( fixedWidthFmt( 20, isHeading=True ),
                              fixedWidthFmt( 40 ),
                              fixedWidthFmt( 10 ),
                              fixedWidthFmt( 21 ),
                              fixedWidthFmt( 9 ),
                              fixedWidthFmt( 15 ) )

         stdVms = ( i for i in self.vms if None == i.logicalSwitch )
         vxlanVms = ( i for i in self.vms if None != i.logicalSwitch )

         self.renderRows( table, stdVms )
         self.renderRows( table, vxlanVms )

         print( table.output() )

      if len( self.staleSessions ) > 0:
         first = self._oldestSession( self.staleSessions )
         print( "Warning: Session ( %s ) data might be stale" % first[ 0 ] )
         print( "Last sync was " + Ark.timestampToStr( first[ 1 ],
               False, Tac.utcNow() ) )

   def renderRows( self, table, vms ):
      for i in vms:
         table.newRow( i.vmName, i.host,
                       IntfCli.Intf.getShortname( i.interface ),
                       i.logicalSwitch or i.vlan,
                       i.renderStatus(),
                       i.renderVtep() )

# i needed dictionary to keep track of the order of attributes in
# object, as it is not kept in order in the __attributes__ dict
attrDict = { 'VmwareVI::PortGroup' : [ 'objType', 'name', 'pgName', 'vlanId', 
                                 'accessVlanId', 'switchRef', 'trunk', 
                                 'trunkAllowedVlans', 'sVlanId','eVlanId', 
                                 'active' ],
            'VmwareVI::DVS' :  [ 'objType', 'name', 'uuId', 'dvsName' ],
            'VmwareVI::IntfToEsxMap' : [ 'objType', 'name', 'lagName', 
                                 'cleanupIntfStatus', 'cleanupAfter', 
                                 'hostRef', 'vmnicKey', 'linkUp', 
                                 'nwInfoRspCnt', 'nwInfoReqCnt' ],
            'VmwareVI::Vnic' : [ 'objType', 'name', 'macAddress', 
                                 'ipAddrWithMask', 'portKey', 'dvsName', 
                                 'switchId', 'label', 'portGroup', 'vmotionPg',
                                 'active', 'configId', 'stats', 
                                 'counterRefreshTime', 'operStatus', 
                                 'linkStatus', 'mtu', 'vnicType' ],
            'VmwareVI::Pnic' : [ 'objType', 'name', 'pnicId', 'macAddress', 
                           'uplinkPortKey', 'switchId', 'dvsName', 'label', 
                           'uplinkName', 'deviceName', 'duplex', 'speed', 
                           'mtu', 'peerIntf', 'peerDevId', 'peerMgmtAddr', 
                           'peerSysName', 'vendor', 'model', 'active', 
                           'configId', 'stats', 'counterRefreshTime', 
                           'operStatus', 'linkStatus' ],
            'VmwareVI::Vm' : [ 'objType', 'name', 'vmName', 'hostRef',
                              'hostName', 'ftStatus', 'vmotionDst', 'inVmotion',
                              'vmotionFailed', 'powerState', 'vnic' ],
            'VmwareVI::EsxHost' : [ 'objType', 'name', 'hostId', 'hostName', 
                              'networkId', 'dcRef', 'parentRef', 
                              'cpuDescription', 'numCpus', 'numCores', 'vendor', 
                              'model', 'serviceTag', 'vswitch', 'vmname', 
                              'pnicId', 'pnic', 'portGroup', 'vnic' ],
            'VmwareVI::VI' : [ 'objType', 'connected', 'switchId', 'dvsName', 
                        'initialSync', 'dcRefTodcName', 'hostFolderTodcRef', 
                        'compResToHostFolder', 'modifiedVms', 'modifiedHosts', 
                        'newPort', 'shutDownPorts', 'portKey', 'linkUpEvent', 
                        'linkDownEvent', 'initialWaitForUpdate', 'portGroup', 
                        'dvs', 'vm', 'esxHost', 'intfToEsxMap' ] }

class ViObjects( Model ):
   objType = Str( help='obj type' )
   name = Str( help='obj key' )

   def render( self ):
      self.do_render( 0 )

   def do_render( self, level=0 ):
      indentLvl = INDENT * level
      # retrieve the attributes from the attrdict and render each attribute
      attrList = attrDict[ self.objType ]
      for attr in attrList:
         attrValue = getattr( self, attr )
         # extract and compare the attribute type
         attrType = self.__attributes__[ attr ] 
         if type( attrType ) is Dict: # pylint: disable=unidiomatic-typecheck
            if attrType.valueType is str:
               stringCollection = ', '.join( v for v in attrValue.values() )
               print( FORMAT_STR % { 'indent': indentLvl, 'name' : attr, 
                     'value' : stringCollection if stringCollection else '--' } )
            else:
               objList = '[' + ', '.join( v for v in attrValue ) + ']'
               print( FORMAT_STR % { 'indent': indentLvl, 
                     'name' : attr, 'value' : objList } )
               for obj in attrValue.values():
                  print()
                  obj.do_render( level + 1 )
         # pylint: disable-next=unidiomatic-typecheck
         elif type( attrType ) is Str and attrValue is None:
            print( FORMAT_STR % { 'indent': indentLvl, 
                  'name' : attr, 'value' : '--' } )
         else:
            print( FORMAT_STR % { 'indent': indentLvl, 'name' : attr, 
                  'value' : attrValue } )

class PortGroup( ViObjects ):
   pgName = Str( help="portGroupName", optional=True )
   vlanId = Int( help="vlanId" )
   accessVlanId = Str( help="accessVlanId", optional = True )
   switchRef = Str( help="switchRef", optional = True )
   trunk = Bool( help="is trunk on?", optional = True )
   trunkAllowedVlans = Str( help="allowed vlans on the trunk", optional=True )
   sVlanId = Int( help="svlanId" )
   eVlanId = Int( help="evlanId" )
   active = Bool( help="is active?" )

class DVS( ViObjects ):
   uuId = Str( help="universally unique id", optional=True )
   dvsName = Str( help="distributed virtual switch name", optional=True )

class IntfToEsxMap( ViObjects ):
   lagName = Str( help="name of link aggregation", optional=True )
   cleanupIntfStatus = Bool( help="cleanupIntfStatus" )
   cleanupAfter = Int( help="cleanupAfter" )
   hostRef = Str( help="host reference", optional=True )
   vmnicKey = Str( help="vmnic key", optional=True )
   linkUp = Bool( help="is link up?" )
   nwInfoRspCnt = Int( help="network info response count" )
   nwInfoReqCnt = Int( help="network info request count" )

class Vnic( ViObjects ):
   macAddress = Str( help="macAddress", optional=True )
   ipAddrWithMask = Str( help="ip address with mask", optional=True )
   portKey = Str( help="port key", optional=True )
   dvsName = Str( help="distributed virtual switch name", optional=True )
   switchId = Str( help="switch id", optional=True )
   label = Str( help="vnic label", optional=True )
   portGroup = Str( help="port group", optional=True )
   vmotionPg = Str( help="vmotion port group", optional=True )
   active = Bool( help="is active?" )
   configId = Int( help="config Id" )
   stats = Str( help="interface statistics", optional=True )
   counterRefreshTime = Float( help="counterRefreshTime" )
   operStatus = Enum( values=( "intfOperUp",
      "intfOperDown",
      "intfOperTesting",
      "intfOperUnknown",
      "intfOperDormant",
      "intfOperNotPresent",
      "intfOperLowerLayerDown" ),
      help="operational status")
   linkStatus = Enum( values=( "linkUnknown",
      "linkDown",
      "linkUp" ),
      help="link status" )
   mtu = Int( help="maximum transmission unit" )
   vnicType = Enum( values=( "virtualMachineNic", "vmKernelNic" ), 
         help="type" , optional=True )

class Pnic( ViObjects ):
   pnicId = Int( help="pnic id" )
   macAddress = Str( help="macAddress", optional=True )
   uplinkPortKey = Str( help="uplinkPortKey", optional=True )
   switchId = Str( help="switch Id", optional=True )
   dvsName = Str( help="distributed virtual switch name", optional=True )
   label = Str( help="label", optional=True )
   uplinkName = Str( help="uplinkName", optional=True )
   deviceName = Str( help="deviceName", optional=True )
   duplex  = Enum( values=( "duplexUnknown",
      "duplexHalf", 
      "duplexFull" ), help="duplex" )
   speed = Enum( values=( "speedUnknown",
      " speed10Mbps",
      "speed100Mbps",
      "speed1Gbps",
      "speed10Gbps",
      "speed40Gbps",
      "speed100Gbps" ), help="speed", optional=True )
   mtu = Int( help="maximum transmission unit" )
   peerIntf = Str( help="peer interface" , optional=True )
   peerDevId = Str( help="peer device id", optional=True )
   peerMgmtAddr = Str( help="peer management address", optional=True )
   peerSysName = Str( help="peer system name", optional=True )
   vendor = Str( help="vendor", optional=True )
   model = Str( help="model", optional=True )
   active = Bool( help="up or down" )
   configId = Int( help="config id" )
   stats = Str( help="stats", optional=True )
   counterRefreshTime = Float( help="counter refresh time" )
   operStatus = Enum( values=( "intfOperUp",
      "intfOperDown",
      "intfOperTesting",
      "intfOperUnknown",
      "intfOperDormant",
      "intfOperNotPresent",
      "intfOperLowerLayerDown"),
      help="operational status")
   linkStatus = Enum ( values=( "linkUnknown",
      "linkDown",
      "linkUp" ),
      help="link status" )

class Vm( ViObjects ):
   vmName = Str( help="vm name", optional=True )
   hostRef = Str( help="host reference", optional=True )
   hostName = Str( help="host name", optional=True )
   ftStatus = Enum( values=( "ftDisabled",
      "ftActive",
      "ftStandby" ), help="FTstatus", optional=True )
   vmotionDst = Str( help="vmotion dstributed switch", optional=True )
   inVmotion = Bool( help="vm in process of moving from one host to"
                     "other", optional=True )
   vmotionFailed = Bool( help="failed, success", optional=True )
   powerState = Bool( help="up, down", optional=True )
   vnic = Dict( valueType=Vnic, help="list of vnics on vm", optional=True )

class EsxHost( ViObjects ):
   hostId = Int( help="host id" )
   hostName = Str( help="host name", optional=True )
   networkId = Str( help="network id", optional=True )
   dcRef = Str( help="data center ref", optional=True )
   parentRef = Str( help="ref to parent", optional=True )
   cpuDescription = Str( help="cpu description", optional=True )
   numCpus = Int( help="number of cpus" )
   numCores = Int( help="number of cores" )
   vendor = Str( help="vendor", optional=True )
   model = Str( help="model", optional=True )
   serviceTag = Str( help="service tag", optional=True )
   vswitch = Dict( valueType=str, help="list of vswitches" )
   vmname = Str( help="name of vm for the Vm::Ptr", optional=True )
   pnicId = Int( help="pnic id" )
   pnic = Dict( valueType=Pnic, help="list of pnics attached to host" )
   portGroup = Dict( valueType=PortGroup, help="list of port group" )
   vnic = Dict( valueType=Vnic, help="list of vnics" )

class VmTracerDebugAll( Model ):
   __public__ = False
   class VI ( ViObjects ):
      connected = Bool( help="connected or not" )
      switchId = Str( help="name of switch", optional=True )
      dvsName = Str( help="name of distributed virtual switch",
                  optional=True )
      initialSync = Bool( help="initialSync" )
      dcRefTodcName = Dict(valueType=str, help ="number of dcRefs" )
      hostFolderTodcRef = Dict( valueType=str,
            help="number of hostFolderTodcNames" )
      compResToHostFolder = Dict( valueType=str,
            help="number of compResToHostFolders" )
      modifiedVms = Dict( valueType=str, help="list of modified vms",
            optional=True )
      modifiedHosts = Dict( valueType=str, help="number of modified hosts",
            optional=True )
      newPort = Dict( valueType=str, help="number of new ports", optional=True )
      shutDownPorts = Dict( valueType=str, help="number of down ports" )
      portKey = Str( help="port key", optional=True )
      linkUpEvent = Int( help="link up event #" )
      linkDownEvent = Int( help="link down event #" )
      initialWaitForUpdate = Bool( help="wait for updates ?" )
      portGroup = Dict( valueType=PortGroup, help="number of port groups" )
      dvs = Dict( valueType=DVS, help="number of distributed virtual switches" )
      vm = Dict( valueType=Vm, help="number of vms", optional=True )
      esxHost = Dict( valueType=EsxHost, help="number of esx", optional=True )
      intfToEsxMap = Dict( valueType=IntfToEsxMap,
            help="number of interfaces to esx mapping" )

   sessions = Dict( valueType=VI, help="Connections to vCenter instances",
         optional=True )
   
   def render( self ):
      for sessId, vi in self.sessions.items():
         print( ''.rjust( 60, '-' ) )
         print( FORMAT_STR % { 'indent': '', 'name' : 'session',
               'value' : sessId } )
         vi.render()

class VmTracerVmDebug( Model ):
   __public__ = False
   sessions = Dict( valueType=Vm, help="holds results for all sessions",
         optional=True )

   def render( self ):
      for sessId, vm in self.sessions.items():
         print( ''.rjust( 60, '-' ) )
         print( FORMAT_STR % { 'indent' : '', 'name' : 'session',
               'value' : sessId } )
         vm.render()

class VmTracerEsxDebug( Model ):
   __public__ = False
   sessions = Dict (valueType=EsxHost, help="holds results for all sessions",
         optional=True )
   def render( self ):
      for sessId, esx in self.sessions.items():
         print( ''.rjust( 60, '-' ) )
         print( FORMAT_STR % { 'indent' : '', 'name' : 'session', 
               'value' : sessId } )
         esx.render()
