# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Toggles.PseudowireAgentToggleLib import toggleVplsShowPwBriefEnabled
from BasicCli import addShowCommandClass
from BasicCliUtil import anyCaseRegex
from CliCommand import (
   CliExpression,
   Node,
   singleKeyword
)
from CliMatcher import (
   DynamicNameMatcher,
   IntegerMatcher,
   KeywordMatcher,
   PatternMatcher,
   EnumMatcher,
)
from CliParserCommon import (
   MatchResult,
   noMatch,
)
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.Vpls import vplsNodeForShow
from CliPlugin.VplsCli import getVplsConfigInstanceCollection
from PseudowireLib import (
   validPwNameRegex,
   vplsNameRegex,
)
from ShowCommand import ShowCliCommandClass

def anyCaseKwRegex( kwList ):
   return r'(' + '|'.join( [ anyCaseRegex( kw ) for kw in kwList ] ) + ')'

def excludeKeywords( regex, kwList ):
   return fr'^(?!{anyCaseKwRegex( kwList )}$){regex}$'

instanceNameRegex = excludeKeywords( vplsNameRegex,
                                     [ 'pseudowire', 'detail', 'vlan' ] )
instanceNameMatcher = DynamicNameMatcher( getVplsConfigInstanceCollection,
                                          helpdesc='VPLS instance name',
                                          pattern=instanceNameRegex )
instanceKwForShow = KeywordMatcher( 'instance',
                                    helpdesc='VPLS instance information' )

pseudowireGroupKwForShow = KeywordMatcher(
   'pseudowire', helpdesc='VPLS Pseudowire Group information' )
groupKwForShow = KeywordMatcher( 'group',
                                 helpdesc='VPLS pseudowire group information' )

def getAllVplsGroupNames():
   res = set()
   for instance in getVplsConfigInstanceCollection().values():
      res.update( instance.ldpPseudowireGroup )
      res.update( instance.bgpPseudowireGroup )

   return res

def getVplsInstanceGroupNames( mode, ctx ):
   if not ctx.sharedResult:
      return getAllVplsGroupNames()

   instanceName = ctx.sharedResult[ 'INSTANCE_NAME' ]
   instance = getVplsConfigInstanceCollection().get( instanceName )
   if not instance:
      return []

   return set( instance.ldpPseudowireGroup ) | set( instance.bgpPseudowireGroup )

groupNameMatcher = DynamicNameMatcher( getVplsInstanceGroupNames,
                                       'VPLS pseudowire group name',
                                       passContext=True,
                                       pattern=vplsNameRegex )

vlanKwForShow = KeywordMatcher( 'vlan', helpdesc='VLAN VPLS information' )
vlanIdMatcher = IntegerMatcher( 1, 4094, helpdesc='Identifier for a Virtual LAN' )

pseudowireKwForShow = KeywordMatcher( 'pseudowire',
                                      helpdesc='VPLS Pseudowire information' )

neighborKwForShow = KeywordMatcher( 'neighbor', helpdesc='VPLS neighbor' )
neighborAddressMatcher = IpAddrMatcher( 'Neighbor IPv4 address' )

pseudowireIdKwForShow = KeywordMatcher( 'id', helpdesc='LDP pseudowire ID' )
pseudowireIdMatcher = IntegerMatcher( 1, 0xffffffff, helpdesc='Pseudowire ID' )

def getVplsPseudowireLdpConfigNames( mode ):
   pseudowireNames = set()
   for instance in getVplsConfigInstanceCollection().values():
      for group in instance.ldpPseudowireGroup.values():
         pseudowireNames.update( group.pseudowire )
   return pseudowireNames

pseudowireConfigNamePattern = excludeKeywords( validPwNameRegex,
                                               [ 'neighbor', 'interface' ] )
pseudowireLdpConfigNameMatcher = DynamicNameMatcher(
   getVplsPseudowireLdpConfigNames,
   'VPLS pseudowire name',
   pattern=pseudowireConfigNamePattern )

interfaceKwForShow = KeywordMatcher( 'interface', helpdesc='Pseudowire interface' )

# pylint: disable-next=consider-using-f-string
pseudowireInterfaceNamePattern = r'^%s(\d+)(\.0)?$' % (
   anyCaseKwRegex( [ 'Pseudowire', 'Pw' ] ) )

class PseudowireInterfaceNameMatcher( PatternMatcher ):
   def __init__( self ):
      PatternMatcher.__init__( self,
                               pattern=pseudowireInterfaceNamePattern,
                               helpname='Pseudowire',
                               helpdesc='Pseudowire interface name',
                               rawResult=True )

   def match( self, mode, context, token ):
      result = PatternMatcher.match( self, mode, context, token )
      if result == noMatch:
         return result
      m = result.result
      # pylint: disable-next=consider-using-f-string
      pwName = 'Pseudowire%s' % ( m.group( 2 ) )
      return MatchResult( pwName, token )

class PseudowireInterfaceMatcher( CliExpression ):
   expression = 'INTERFACE_NAME | INTERFACE_INDEX'
   data = {
      'INTERFACE_NAME': PseudowireInterfaceNameMatcher(),
      'INTERFACE_INDEX': IntegerMatcher( 1, 0xffffffffff,
                                         helpdesc='Pseudowire interface index' ),
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'INTERFACE_NAME' in args:
         args[ 'INTERFACE' ] = args[ 'INTERFACE_NAME' ]
      elif 'INTERFACE_INDEX' in args:
         # pylint: disable-next=consider-using-f-string
         args[ 'INTERFACE' ] = 'Pseudowire%s' % ( args[ 'INTERFACE_INDEX' ] )

detailKwForShow = KeywordMatcher( 'detail',
                                  helpdesc='Display VPLS pseudowire details' )

class ShowVpls( ShowCliCommandClass ):
   syntax = (
      'show vpls ['
      '  ( vlan VLAN_ID ) | '
      '  ( [ ( INSTANCE_NAME [ pseudowire1 group GROUP_NAME ] ) |'
      '      ( pseudowire2 ( ( neighbor NEIGHBOR_ADDRESS ( id PSEUDOWIRE_ID ) |'
      '                         ( instance PW_INSTANCE_NAME'
      '                           group PW_GROUP_NAME ) ) |'
      '                      ( interface INTERFACE ) |'
      '                      PSEUDOWIRE_NAME ) ) ]'
      '    [ detail ] )'
      ']' )
   data = {
      'vpls': vplsNodeForShow,
      'INSTANCE_NAME': Node( instanceNameMatcher, storeSharedResult=True ),
      'pseudowire1': pseudowireGroupKwForShow,
      'group': groupKwForShow,
      'GROUP_NAME': groupNameMatcher,
      'pseudowire2': pseudowireKwForShow,
      'neighbor': neighborKwForShow,
      'NEIGHBOR_ADDRESS': neighborAddressMatcher,
      'id': pseudowireIdKwForShow,
      'instance': instanceKwForShow,
      'PW_INSTANCE_NAME': Node( instanceNameMatcher, storeSharedResult=True ),
      'PW_GROUP_NAME': groupNameMatcher,
      'PSEUDOWIRE_ID': pseudowireIdMatcher,
      'interface': interfaceKwForShow,
      'INTERFACE': PseudowireInterfaceMatcher,
      'PSEUDOWIRE_NAME': pseudowireLdpConfigNameMatcher,
      'detail': detailKwForShow,
      'vlan': vlanKwForShow,
      'VLAN_ID': vlanIdMatcher,
   }
   cliModel = 'VplsModel.Vpls'
   handler = 'VplsShowCli.showVplsHandler'

addShowCommandClass( ShowVpls )

pwTypeEnum = Node(
   matcher=EnumMatcher( {
      'tagged': 'Pseudowire type 4 (tagged)',
      'raw': 'Pseudowire type 5 (raw)',
   } ), maxMatches=1 )

class ShowVplsPseudowireBrief( ShowCliCommandClass ):
   syntax = (
      'show vpls'
      '[ INSTANCE_NAME | ( vlan VLAN ) ]'
      'pseudowire [ group GROUP ] brief'
      '( all'
      '| { ( neighbor NEIGHBOR )'
      '  | ( pseudowire-type PW_TYPE ) } )' )
   data = {
      'vpls': vplsNodeForShow,
      'INSTANCE_NAME': Node( instanceNameMatcher, storeSharedResult=True ),
      'vlan': vlanKwForShow,
      'VLAN': vlanIdMatcher,
      'pseudowire': 'VPLS pseudowire brief information',
      'group': groupKwForShow,
      'GROUP': groupNameMatcher,
      'brief': 'VPLS pseudowire brief information',
      'all': 'Display all information',
      'neighbor': singleKeyword( 'neighbor', 'Filter output based on LDP neighbor' ),
      'NEIGHBOR': Node( neighborAddressMatcher, maxMatches=1 ),
      'pseudowire-type': singleKeyword(
         'pseudowire-type', 'Filter output based on pseudowire type' ),
      'PW_TYPE': pwTypeEnum,
   }
   cliModel = 'VplsModel.Vpls' # cli model is a dummy
   handler = 'VplsShowCli.showVplsPseudowireBriefHandler'

if toggleVplsShowPwBriefEnabled():
   addShowCommandClass( ShowVplsPseudowireBrief )
